#ifndef __IMX135_H__
#define __IMX135_H__

#define IMX_NAME	"imx135"
#define IMX_ID	0x0135

#define IMX_SC_CMMN_CHIP_ID_H	0x0016
#define IMX_SC_CMMN_CHIP_ID_L	0x0017
#define IMX_FOCAL_LENGTH_NUM	369	/*3.69mm*/
#define IMX_FOCAL_LENGTH_DEM	100
#define IMX_F_NUMBER_DEFAULT_NUM	22
#define IMX_F_NUMBER_DEM	10

#define IMX_RES_WIDTH_MAX	4208
#define IMX_RES_HEIGHT_MAX	3120
#define IMX_BIN_FACTOR_MAX			4
/*
 * focal length bits definition:
 * bits 31-16: numerator, bits 15-0: denominator
 */
#define IMX_FOCAL_LENGTH_DEFAULT 0x1710064

/*
 * current f-number bits definition:
 * bits 31-16: numerator, bits 15-0: denominator
 */
#define IMX_F_NUMBER_DEFAULT 0x16000a

/*
 * f-number range bits definition:
 * bits 31-24: max f-number numerator
 * bits 23-16: max f-number denominator
 * bits 15-8: min f-number numerator
 * bits 7-0: min f-number denominator
 */
#define IMX_F_NUMBER_RANGE 0x160a160a

enum imx_tok_type {
	IMX_8BIT  = 0x0001,
	IMX_16BIT = 0x0002,
	IMX_TOK_TERM   = 0xf000,	/* terminating token for reg list */
	IMX_TOK_DELAY  = 0xfe00	/* delay token for reg list */
};

/**
 * struct imx_reg - MI sensor  register format
 * @type: type of the register
 * @reg: 16-bit offset to register
 * @val: 8/16/32-bit register value
 *
 * Define a structure for sensor register initialization values
 */
struct imx_reg {
	enum imx_tok_type type;
	u16 sreg;
	u32 val;	/* @set value for read/mod/write, @mask */
};

struct imx_resolution {
	u8 *desc;
	const struct imx_reg *regs;
	int res;
	int width;
	int height;
	int fps;
	unsigned short pixels_per_line;
	unsigned short lines_per_frame;
	u8 bin_factor_x;
	u8 bin_factor_y;
	bool used;
};
#define GROUPED_PARAMETER_HOLD_ENABLE  {IMX_8BIT, 0x0104, 0x1}
#define GROUPED_PARAMETER_HOLD_DISABLE  {IMX_8BIT, 0x0104, 0x0}

/********************** settings for imx from vendor*********************/

static struct imx_reg const imx135_SwReset[] = {
	{ IMX_8BIT,  0x0100 , 0x00 },
	{ IMX_8BIT,  0x0103 , 0x01 },
	{ IMX_TOK_DELAY, 0, 5},
	{ IMX_8BIT,  0x0103 , 0x00 },
	{ IMX_TOK_TERM, 0, 0},
};
static struct imx_reg const imx135_13M_25fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03},
	{IMX_8BIT, 0x0112, 0x0A},
	{IMX_8BIT, 0x0113, 0x0A},
	{IMX_8BIT, 0x0381, 0x01},
	{IMX_8BIT, 0x0383, 0x01},
	{IMX_8BIT, 0x0385, 0x01},
	{IMX_8BIT, 0x0387, 0x01},
	{IMX_8BIT, 0x0390, 0x00},
	{IMX_8BIT, 0x0391, 0x11},
	{IMX_8BIT, 0x0392, 0x00},
	{IMX_8BIT, 0x0401, 0x00},
	{IMX_8BIT, 0x0404, 0x00},
	{IMX_8BIT, 0x0405, 0x10},
	{IMX_8BIT, 0x4082, 0x01},
	{IMX_8BIT, 0x4083, 0x01},
	{IMX_8BIT, 0x4203, 0xFF},
	{IMX_8BIT, 0x7006, 0x04},
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x0C},
	{IMX_8BIT, 0x0341, 0x46},
	{IMX_8BIT, 0x0342, 0x11},
	{IMX_8BIT, 0x0343, 0xDC},
	{IMX_8BIT, 0x0344, 0x00},
	{IMX_8BIT, 0x0345, 0x00},
	{IMX_8BIT, 0x0346, 0x00},
	{IMX_8BIT, 0x0347, 0x00},
	{IMX_8BIT, 0x0348, 0x10},
	{IMX_8BIT, 0x0349, 0x6F},
	{IMX_8BIT, 0x034A, 0x0C},
	{IMX_8BIT, 0x034B, 0x2F},
	{IMX_8BIT, 0x034C, 0x10},
	{IMX_8BIT, 0x034D, 0x70},
	{IMX_8BIT, 0x034E, 0x0C},
	{IMX_8BIT, 0x034F, 0x30},
	{IMX_8BIT, 0x0350, 0x00},
	{IMX_8BIT, 0x0351, 0x00},
	{IMX_8BIT, 0x0352, 0x00},
	{IMX_8BIT, 0x0353, 0x00},
	{IMX_8BIT, 0x0354, 0x10},
	{IMX_8BIT, 0x0355, 0x70},
	{IMX_8BIT, 0x0356, 0x0C},
	{IMX_8BIT, 0x0357, 0x30},
	{IMX_8BIT, 0x301D, 0x30},
	{IMX_8BIT, 0x3310, 0x10},
	{IMX_8BIT, 0x3311, 0x70},
	{IMX_8BIT, 0x3312, 0x0C},
	{IMX_8BIT, 0x3313, 0x30},
	{IMX_8BIT, 0x331C, 0x00},
	{IMX_8BIT, 0x331D, 0x10},
	{IMX_8BIT, 0x4084, 0x00},
	{IMX_8BIT, 0x4085, 0x00},
	{IMX_8BIT, 0x4086, 0x00},
	{IMX_8BIT, 0x4087, 0x00},
	{IMX_8BIT, 0x4400, 0x00},
	/* Global Timing */
	{IMX_8BIT, 0x0830, 0x87},
	{IMX_8BIT, 0x0831, 0x3F},
	{IMX_8BIT, 0x0832, 0x67},
	{IMX_8BIT, 0x0833, 0x3F},
	{IMX_8BIT, 0x0834, 0x3F},
	{IMX_8BIT, 0x0835, 0x4F},
	{IMX_8BIT, 0x0836, 0xDF},
	{IMX_8BIT, 0x0837, 0x47},
	{IMX_8BIT, 0x0839, 0x1F},
	{IMX_8BIT, 0x083A, 0x17},
	{IMX_8BIT, 0x083B, 0x02},
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x0C},
	{IMX_8BIT, 0x0203, 0x42},
	{IMX_TOK_TERM, 0, 0},
};

static struct imx_reg const imx135_13M_3fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	/* Clock Setting */
	{IMX_8BIT, 0x011E, 0x13},
	{IMX_8BIT, 0x011F, 0x33},
	{IMX_8BIT, 0x0301, 0x05},
	{IMX_8BIT, 0x0303, 0x01},
	{IMX_8BIT, 0x0305, 0x0C},
	{IMX_8BIT, 0x0309, 0x05},
	{IMX_8BIT, 0x030B, 0x01},
	{IMX_8BIT, 0x030C, 0x02},
	{IMX_8BIT, 0x030D, 0x08},
	{IMX_8BIT, 0x030E, 0x01},
	{IMX_8BIT, 0x3A06, 0x11},
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03},
	{IMX_8BIT, 0x0112, 0x0A},
	{IMX_8BIT, 0x0113, 0x0A},
	{IMX_8BIT, 0x0381, 0x01},
	{IMX_8BIT, 0x0383, 0x01},
	{IMX_8BIT, 0x0385, 0x01},
	{IMX_8BIT, 0x0387, 0x01},
	{IMX_8BIT, 0x0390, 0x00},
	{IMX_8BIT, 0x0391, 0x11},
	{IMX_8BIT, 0x0392, 0x00},
	{IMX_8BIT, 0x0401, 0x00},
	{IMX_8BIT, 0x0404, 0x00},
	{IMX_8BIT, 0x0405, 0x10},
	{IMX_8BIT, 0x4082, 0x01},
	{IMX_8BIT, 0x4083, 0x01},
	{IMX_8BIT, 0x4203, 0xFF},
	{IMX_8BIT, 0x7006, 0x04},
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x20},
	{IMX_8BIT, 0x0341, 0x00},
	{IMX_8BIT, 0x0342, 0x30},
	{IMX_8BIT, 0x0343, 0x00},
	{IMX_8BIT, 0x0344, 0x00},
	{IMX_8BIT, 0x0345, 0x00},
	{IMX_8BIT, 0x0346, 0x00},
	{IMX_8BIT, 0x0347, 0x00},
	{IMX_8BIT, 0x0348, 0x10},
	{IMX_8BIT, 0x0349, 0x6F},
	{IMX_8BIT, 0x034A, 0x0C},
	{IMX_8BIT, 0x034B, 0x2F},
	{IMX_8BIT, 0x034C, 0x10},
	{IMX_8BIT, 0x034D, 0x70},
	{IMX_8BIT, 0x034E, 0x0C},
	{IMX_8BIT, 0x034F, 0x30},
	{IMX_8BIT, 0x0350, 0x00},
	{IMX_8BIT, 0x0351, 0x00},
	{IMX_8BIT, 0x0352, 0x00},
	{IMX_8BIT, 0x0353, 0x00},
	{IMX_8BIT, 0x0354, 0x10},
	{IMX_8BIT, 0x0355, 0x70},
	{IMX_8BIT, 0x0356, 0x0C},
	{IMX_8BIT, 0x0357, 0x30},
	{IMX_8BIT, 0x301D, 0x30},
	{IMX_8BIT, 0x3310, 0x10},
	{IMX_8BIT, 0x3311, 0x70},
	{IMX_8BIT, 0x3312, 0x0C},
	{IMX_8BIT, 0x3313, 0x30},
	{IMX_8BIT, 0x331C, 0x00},
	{IMX_8BIT, 0x331D, 0x10},
	{IMX_8BIT, 0x4084, 0x00},
	{IMX_8BIT, 0x4085, 0x00},
	{IMX_8BIT, 0x4086, 0x00},
	{IMX_8BIT, 0x4087, 0x00},
	{IMX_8BIT, 0x4400, 0x00},
	/* Global Timing */
	{IMX_8BIT, 0x0830, 0x7F},
	{IMX_8BIT, 0x0831, 0x37},
	{IMX_8BIT, 0x0832, 0x67},
	{IMX_8BIT, 0x0833, 0x3F},
	{IMX_8BIT, 0x0834, 0x3F},
	{IMX_8BIT, 0x0835, 0x47},
	{IMX_8BIT, 0x0836, 0xDF},
	{IMX_8BIT, 0x0837, 0x47},
	{IMX_8BIT, 0x0839, 0x1F},
	{IMX_8BIT, 0x083A, 0x17},
	{IMX_8BIT, 0x083B, 0x02},
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x0D},
	{IMX_8BIT, 0x0203, 0xFC},
	/* Gain Setting */
	{IMX_8BIT, 0x0205, 0x00},
	{IMX_8BIT, 0x020E, 0x01},
	{IMX_8BIT, 0x020F, 0x00},
	{IMX_8BIT, 0x0210, 0x01},
	{IMX_8BIT, 0x0211, 0x00},
	{IMX_8BIT, 0x0212, 0x01},
	{IMX_8BIT, 0x0213, 0x00},
	{IMX_8BIT, 0x0214, 0x01},
	{IMX_8BIT, 0x0215, 0x00},
	{IMX_TOK_TERM, 0, 0},
};

static struct imx_reg const imx135_13M_24fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	/* Clock Setting */
	{IMX_8BIT, 0x011E, 0x13}, /* exck_freq [15:8]integer item */
	{IMX_8BIT, 0x011F, 0x33}, /* exck_freq [7:0] decimal fraction item */
	{IMX_8BIT, 0x0301, 0x05}, /* vt_pix_clk_div */
	{IMX_8BIT, 0x0303, 0x01}, /* vt_sys_clk_div */
	{IMX_8BIT, 0x0305, 0x0C}, /* op_pre_pll_clk_div */
	{IMX_8BIT, 0x0309, 0x05}, /* op_pix_clk_div */
	{IMX_8BIT, 0x030B, 0x01}, /* op_sys_clk_div */
	{IMX_8BIT, 0x030C, 0x02}, /* op_pll_multiplier [10:8] */
	{IMX_8BIT, 0x030D, 0x38}, /* op_pll_multiplier [7:0] */
	{IMX_8BIT, 0x030E, 0x01}, /* pll_singledrive-enable 0:double,1:single */
	/*
	 * ckdiv_mode/src2div
	 * [4] mdbck clock setting,0:auto,1:manual
	 * [1:0] when ckdiv_mode is1, set same with opsys_div, only 1 or 2
	 */
	{IMX_8BIT, 0x3A06, 0x11},

	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03},
	{IMX_8BIT, 0x0112, 0x0A},
	{IMX_8BIT, 0x0113, 0x0A},
	{IMX_8BIT, 0x0381, 0x01},
	{IMX_8BIT, 0x0383, 0x01},
	{IMX_8BIT, 0x0385, 0x01},
	{IMX_8BIT, 0x0387, 0x01},
	{IMX_8BIT, 0x0390, 0x00},
	{IMX_8BIT, 0x0391, 0x11},
	{IMX_8BIT, 0x0392, 0x00},
	{IMX_8BIT, 0x0401, 0x00},
	{IMX_8BIT, 0x0404, 0x00},
	{IMX_8BIT, 0x0405, 0x10},
	{IMX_8BIT, 0x4082, 0x01},
	{IMX_8BIT, 0x4083, 0x01},
	{IMX_8BIT, 0x4203, 0xFF},
	{IMX_8BIT, 0x7006, 0x04},
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x0C},
	{IMX_8BIT, 0x0341, 0xF0},
	{IMX_8BIT, 0x0342, 0x11},
	{IMX_8BIT, 0x0343, 0xDC},
	{IMX_8BIT, 0x0344, 0x00},
	{IMX_8BIT, 0x0345, 0x00},
	{IMX_8BIT, 0x0346, 0x00},
	{IMX_8BIT, 0x0347, 0x00},
	{IMX_8BIT, 0x0348, 0x10},
	{IMX_8BIT, 0x0349, 0x6F},
	{IMX_8BIT, 0x034A, 0x0C},
	{IMX_8BIT, 0x034B, 0x2F},
	{IMX_8BIT, 0x034C, 0x10},
	{IMX_8BIT, 0x034D, 0x70},
	{IMX_8BIT, 0x034E, 0x0C},
	{IMX_8BIT, 0x034F, 0x30},
	{IMX_8BIT, 0x0350, 0x00},
	{IMX_8BIT, 0x0351, 0x00},
	{IMX_8BIT, 0x0352, 0x00},
	{IMX_8BIT, 0x0353, 0x00},
	{IMX_8BIT, 0x0354, 0x10},
	{IMX_8BIT, 0x0355, 0x70},
	{IMX_8BIT, 0x0356, 0x0C},
	{IMX_8BIT, 0x0357, 0x30},
	{IMX_8BIT, 0x301D, 0x30},
	{IMX_8BIT, 0x3310, 0x10},
	{IMX_8BIT, 0x3311, 0x70},
	{IMX_8BIT, 0x3312, 0x0C},
	{IMX_8BIT, 0x3313, 0x30},
	{IMX_8BIT, 0x331C, 0x00},
	{IMX_8BIT, 0x331D, 0x10},
	{IMX_8BIT, 0x4084, 0x00},
	{IMX_8BIT, 0x4085, 0x00},
	{IMX_8BIT, 0x4086, 0x00},
	{IMX_8BIT, 0x4087, 0x00},
	{IMX_8BIT, 0x4400, 0x00},
	/* Global Timing */
	{IMX_8BIT, 0x0830, 0x87},
	{IMX_8BIT, 0x0831, 0x3F},
	{IMX_8BIT, 0x0832, 0x67},
	{IMX_8BIT, 0x0833, 0x3F},
	{IMX_8BIT, 0x0834, 0x3F},
	{IMX_8BIT, 0x0835, 0x4F},
	{IMX_8BIT, 0x0836, 0xDF},
	{IMX_8BIT, 0x0837, 0x47},
	{IMX_8BIT, 0x0839, 0x1F},
	{IMX_8BIT, 0x083A, 0x17},
	{IMX_8BIT, 0x083B, 0x02},
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x0C},
	{IMX_8BIT, 0x0203, 0x42},
	{IMX_TOK_TERM, 0, 0},
};

static struct imx_reg const imx135_8M_3fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	/* Clock Setting */
	{ IMX_8BIT, 0x011E, 0x13},
	{ IMX_8BIT, 0x011F, 0x33},
	{ IMX_8BIT, 0x0301, 0x05},
	{ IMX_8BIT, 0x0303, 0x01},
	{ IMX_8BIT, 0x0305, 0x0C},
	{ IMX_8BIT, 0x0309, 0x05},
	{ IMX_8BIT, 0x030B, 0x01},
	{ IMX_8BIT, 0x030C, 0x02},
	{ IMX_8BIT, 0x030D, 0x32},
	{ IMX_8BIT, 0x030E, 0x01},
	{ IMX_8BIT, 0x3A06, 0x11},
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03 },
	{IMX_8BIT, 0x0112, 0x0A },
	{IMX_8BIT, 0x0113, 0x0A },
	{IMX_8BIT, 0x0381, 0x01 },
	{IMX_8BIT, 0x0383, 0x01 },
	{IMX_8BIT, 0x0385, 0x01 },
	{IMX_8BIT, 0x0387, 0x01 },
	{IMX_8BIT, 0x0390, 0x00 },
	{IMX_8BIT, 0x0391, 0x11 },
	{IMX_8BIT, 0x0392, 0x00 },
	{IMX_8BIT, 0x0401, 0x00 },
	{IMX_8BIT, 0x0404, 0x00 },
	{IMX_8BIT, 0x0405, 0x10 },
	{IMX_8BIT, 0x4082, 0x01 },
	{IMX_8BIT, 0x4083, 0x01 },
	{IMX_8BIT, 0x7006, 0x04 },
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x0A },
	{IMX_8BIT, 0x0341, 0x3C },
	{IMX_8BIT, 0x0342, 0x27 }, /* quick fix, causing low fps*/
	{IMX_8BIT, 0x0343, 0x10 }, /* to be furnished later */
	{IMX_8BIT, 0x0344, 0x01 },
	{IMX_8BIT, 0x0345, 0xd0 },
	{IMX_8BIT, 0x0346, 0x01 },
	{IMX_8BIT, 0x0347, 0x48 },
	{IMX_8BIT, 0x0348, 0x0e },
	{IMX_8BIT, 0x0349, 0x9f },
	{IMX_8BIT, 0x034A, 0x0a },
	{IMX_8BIT, 0x034B, 0xe7 },
	{IMX_8BIT, 0x034C, 0x0c },
	{IMX_8BIT, 0x034D, 0xd0 },
	{IMX_8BIT, 0x034E, 0x09 },
	{IMX_8BIT, 0x034F, 0xA0 },
	{IMX_8BIT, 0x0350, 0x00 },
	{IMX_8BIT, 0x0351, 0x00 },
	{IMX_8BIT, 0x0352, 0x00 },
	{IMX_8BIT, 0x0353, 0x00 },
	{IMX_8BIT, 0x0354, 0x0c },
	{IMX_8BIT, 0x0355, 0xd0 },
	{IMX_8BIT, 0x0356, 0x09 },
	{IMX_8BIT, 0x0357, 0xa0 },
	{IMX_8BIT, 0x301D, 0x30 },
	{IMX_8BIT, 0x3310, 0x0C },
	{IMX_8BIT, 0x3311, 0xD0 },
	{IMX_8BIT, 0x3312, 0x09 },
	{IMX_8BIT, 0x3313, 0xA0 },
	{IMX_8BIT, 0x331C, 0x00 },
	{IMX_8BIT, 0x331D, 0x10 },
	{IMX_8BIT, 0x4084, 0x00 },
	{IMX_8BIT, 0x4085, 0x00 },
	{IMX_8BIT, 0x4086, 0x00 },
	{IMX_8BIT, 0x4087, 0x00 },
	{IMX_8BIT, 0x4400, 0x00 },
	/* Global Timing Setting */
	{IMX_8BIT, 0x0830, 0x87 },
	{IMX_8BIT, 0x0831, 0x3F },
	{IMX_8BIT, 0x0832, 0x67 },
	{IMX_8BIT, 0x0833, 0x3F },
	{IMX_8BIT, 0x0834, 0x3F },
	{IMX_8BIT, 0x0835, 0x4f },
	{IMX_8BIT, 0x0836, 0xdf },
	{IMX_8BIT, 0x0837, 0x47 },
	{IMX_8BIT, 0x0839, 0x1F },
	{IMX_8BIT, 0x083A, 0x17 },
	{IMX_8BIT, 0x083B, 0x02 },
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x0a },
	{IMX_8BIT, 0x0203, 0x38 },
	/* Gain Setting */
	{IMX_8BIT, 0x0205, 0x00},
	{IMX_8BIT, 0x020E, 0x01},
	{IMX_8BIT, 0x020F, 0x00},
	{IMX_8BIT, 0x0210, 0x01},
	{IMX_8BIT, 0x0211, 0x00},
	{IMX_8BIT, 0x0212, 0x01},
	{IMX_8BIT, 0x0213, 0x00},
	{IMX_8BIT, 0x0214, 0x01},
	{IMX_8BIT, 0x0215, 0x00},
	/* HDR Setting */
	{IMX_8BIT, 0x0230, 0x00},
	{IMX_8BIT, 0x0231, 0x00},
	{IMX_8BIT, 0x0233, 0x00},
	{IMX_8BIT, 0x0234, 0x00},
	{IMX_8BIT, 0x0235, 0x40},
	{IMX_8BIT, 0x0238, 0x00},
	{IMX_8BIT, 0x0239, 0x04},
	{IMX_8BIT, 0x023B, 0x00},
	{IMX_8BIT, 0x023C, 0x01},
	{IMX_8BIT, 0x33B0, 0x04},
	{IMX_8BIT, 0x33B1, 0x00},
	{IMX_8BIT, 0x33B3, 0x00},
	{IMX_8BIT, 0x33B4, 0x01},
	{IMX_8BIT, 0x3800, 0x00},
	{IMX_TOK_TERM, 0, 0},
};

static struct imx_reg const imx135_5M_26fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03},
	{IMX_8BIT, 0x0112, 0x0A},
	{IMX_8BIT, 0x0113, 0x0A},
	{IMX_8BIT, 0x0381, 0x01},
	{IMX_8BIT, 0x0383, 0x01},
	{IMX_8BIT, 0x0385, 0x01},
	{IMX_8BIT, 0x0387, 0x01},
	{IMX_8BIT, 0x0390, 0x00},
	{IMX_8BIT, 0x0391, 0x11},
	{IMX_8BIT, 0x0392, 0x00},
	{IMX_8BIT, 0x0401, 0x02},
	{IMX_8BIT, 0x0404, 0x00},
	{IMX_8BIT, 0x0405, 0x19},
	{IMX_8BIT, 0x4082, 0x00},
	{IMX_8BIT, 0x4083, 0x00},
	{IMX_8BIT, 0x4203, 0xFF},
	{IMX_8BIT, 0x7006, 0x04},
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x0B},
	{IMX_8BIT, 0x0341, 0xEA},
	{IMX_8BIT, 0x0342, 0x11},
	{IMX_8BIT, 0x0343, 0xDC},
	{IMX_8BIT, 0x0344, 0x00},
	{IMX_8BIT, 0x0345, 0x5C},
	{IMX_8BIT, 0x0346, 0x00},
	{IMX_8BIT, 0x0347, 0x30},
	{IMX_8BIT, 0x0348, 0x10},
	{IMX_8BIT, 0x0349, 0x15},
	{IMX_8BIT, 0x034A, 0x0C},
	{IMX_8BIT, 0x034B, 0x03},
	{IMX_8BIT, 0x034C, 0x0A},
	{IMX_8BIT, 0x034D, 0x10},
	{IMX_8BIT, 0x034E, 0x07},
	{IMX_8BIT, 0x034F, 0x90},
	{IMX_8BIT, 0x0350, 0x00},
	{IMX_8BIT, 0x0351, 0x00},
	{IMX_8BIT, 0x0352, 0x00},
	{IMX_8BIT, 0x0353, 0x01},
	{IMX_8BIT, 0x0354, 0x0F},
	{IMX_8BIT, 0x0355, 0xBA},
	{IMX_8BIT, 0x0356, 0x0B},
	{IMX_8BIT, 0x0357, 0xD2},
	{IMX_8BIT, 0x301D, 0x30},
	{IMX_8BIT, 0x3310, 0x0A},
	{IMX_8BIT, 0x3311, 0x10},
	{IMX_8BIT, 0x3312, 0x07},
	{IMX_8BIT, 0x3313, 0x90},
	{IMX_8BIT, 0x331C, 0x03},
	{IMX_8BIT, 0x331D, 0xAE},
	{IMX_8BIT, 0x4084, 0x0A},
	{IMX_8BIT, 0x4085, 0x10},
	{IMX_8BIT, 0x4086, 0x07},
	{IMX_8BIT, 0x4087, 0x90},
	{IMX_8BIT, 0x4400, 0x00},
	/* Global Timing */
	{IMX_8BIT, 0x0830, 0x87},
	{IMX_8BIT, 0x0831, 0x3F},
	{IMX_8BIT, 0x0832, 0x67},
	{IMX_8BIT, 0x0833, 0x3F},
	{IMX_8BIT, 0x0834, 0x3F},
	{IMX_8BIT, 0x0835, 0x4F},
	{IMX_8BIT, 0x0836, 0xDF},
	{IMX_8BIT, 0x0837, 0x47},
	{IMX_8BIT, 0x0839, 0x1F},
	{IMX_8BIT, 0x083A, 0x17},
	{IMX_8BIT, 0x083B, 0x02},
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x0B},
	{IMX_8BIT, 0x0203, 0xE6},
	{IMX_TOK_TERM, 0, 0},
};

static struct imx_reg const imx135_1080p_dvs_30fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03},
	{IMX_8BIT, 0x0112, 0x0A},
	{IMX_8BIT, 0x0113, 0x0A},
	{IMX_8BIT, 0x0381, 0x01},
	{IMX_8BIT, 0x0383, 0x01},
	{IMX_8BIT, 0x0385, 0x01},
	{IMX_8BIT, 0x0387, 0x01},
	{IMX_8BIT, 0x0390, 0x00},
	{IMX_8BIT, 0x0391, 0x11},
	{IMX_8BIT, 0x0392, 0x00},
	{IMX_8BIT, 0x0401, 0x02},
	{IMX_8BIT, 0x0404, 0x00},
	{IMX_8BIT, 0x0405, 0x1C},
	{IMX_8BIT, 0x4082, 0x00},
	{IMX_8BIT, 0x4083, 0x00},
	{IMX_8BIT, 0x4203, 0xFF},
	{IMX_8BIT, 0x7006, 0x04},
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x0A},
	{IMX_8BIT, 0x0341, 0x5A},
	{IMX_8BIT, 0x0342, 0x11},
	{IMX_8BIT, 0x0343, 0xDC},
	{IMX_8BIT, 0x0344, 0x00},
	{IMX_8BIT, 0x0345, 0x3C},
	{IMX_8BIT, 0x0346, 0x01},
	{IMX_8BIT, 0x0347, 0x94},
	{IMX_8BIT, 0x0348, 0x10},
	{IMX_8BIT, 0x0349, 0x33},
	{IMX_8BIT, 0x034A, 0x0A},
	{IMX_8BIT, 0x034B, 0x9B},
	{IMX_8BIT, 0x034C, 0x09},
	{IMX_8BIT, 0x034D, 0x20},
	{IMX_8BIT, 0x034E, 0x05},
	{IMX_8BIT, 0x034F, 0x28},
	{IMX_8BIT, 0x0350, 0x00},
	{IMX_8BIT, 0x0351, 0x00},
	{IMX_8BIT, 0x0352, 0x00},
	{IMX_8BIT, 0x0353, 0x01},
	{IMX_8BIT, 0x0354, 0x0F},
	{IMX_8BIT, 0x0355, 0xF8},
	{IMX_8BIT, 0x0356, 0x09},
	{IMX_8BIT, 0x0357, 0x06},
	{IMX_8BIT, 0x301D, 0x30},
	{IMX_8BIT, 0x3310, 0x09},
	{IMX_8BIT, 0x3311, 0x20},
	{IMX_8BIT, 0x3312, 0x05},
	{IMX_8BIT, 0x3313, 0x28},
	{IMX_8BIT, 0x331C, 0x04},
	{IMX_8BIT, 0x331D, 0xE2},
	{IMX_8BIT, 0x4084, 0x09},
	{IMX_8BIT, 0x4085, 0x20},
	{IMX_8BIT, 0x4086, 0x05},
	{IMX_8BIT, 0x4087, 0x28},
	{IMX_8BIT, 0x4400, 0x00},
	/* Global Timing */
	{IMX_8BIT, 0x0830, 0x87},
	{IMX_8BIT, 0x0831, 0x3F},
	{IMX_8BIT, 0x0832, 0x67},
	{IMX_8BIT, 0x0833, 0x3F},
	{IMX_8BIT, 0x0834, 0x3F},
	{IMX_8BIT, 0x0835, 0x4F},
	{IMX_8BIT, 0x0836, 0xDF},
	{IMX_8BIT, 0x0837, 0x47},
	{IMX_8BIT, 0x0839, 0x1F},
	{IMX_8BIT, 0x083A, 0x17},
	{IMX_8BIT, 0x083B, 0x02},
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x0A},
	{IMX_8BIT, 0x0203, 0x56},
	{IMX_TOK_TERM, 0, 0},
};

static struct imx_reg const imx135_wide_preview_30fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03},
	{IMX_8BIT, 0x0112, 0x0A},
	{IMX_8BIT, 0x0113, 0x0A},
	{IMX_8BIT, 0x0381, 0x01},
	{IMX_8BIT, 0x0383, 0x01},
	{IMX_8BIT, 0x0385, 0x01},
	{IMX_8BIT, 0x0387, 0x01},
	{IMX_8BIT, 0x0390, 0x01},
	{IMX_8BIT, 0x0391, 0x22},
	{IMX_8BIT, 0x0392, 0x00},
	{IMX_8BIT, 0x0401, 0x02},
	{IMX_8BIT, 0x0404, 0x00},
	{IMX_8BIT, 0x0405, 0x1A},
	{IMX_8BIT, 0x4082, 0x00},
	{IMX_8BIT, 0x4083, 0x00},
	{IMX_8BIT, 0x4203, 0xFF},
	{IMX_8BIT, 0x7006, 0x04},
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x0A},
	{IMX_8BIT, 0x0341, 0x5A},
	{IMX_8BIT, 0x0342, 0x11},
	{IMX_8BIT, 0x0343, 0xDC},
	{IMX_8BIT, 0x0344, 0x00},
	{IMX_8BIT, 0x0345, 0x18},
	{IMX_8BIT, 0x0346, 0x01},
	{IMX_8BIT, 0x0347, 0x88},
	{IMX_8BIT, 0x0348, 0x10},
	{IMX_8BIT, 0x0349, 0x57},
	{IMX_8BIT, 0x034A, 0x0A},
	{IMX_8BIT, 0x034B, 0xAB},
	{IMX_8BIT, 0x034C, 0x05},
	{IMX_8BIT, 0x034D, 0x00},
	{IMX_8BIT, 0x034E, 0x02},
	{IMX_8BIT, 0x034F, 0xD0},
	{IMX_8BIT, 0x0350, 0x00},
	{IMX_8BIT, 0x0351, 0x00},
	{IMX_8BIT, 0x0352, 0x00},
	{IMX_8BIT, 0x0353, 0x00},
	{IMX_8BIT, 0x0354, 0x08},
	{IMX_8BIT, 0x0355, 0x20},
	{IMX_8BIT, 0x0356, 0x04},
	{IMX_8BIT, 0x0357, 0x92},
	{IMX_8BIT, 0x301D, 0x30},
	{IMX_8BIT, 0x3310, 0x05},
	{IMX_8BIT, 0x3311, 0x00},
	{IMX_8BIT, 0x3312, 0x02},
	{IMX_8BIT, 0x3313, 0xD0},
	{IMX_8BIT, 0x331C, 0x02},
	{IMX_8BIT, 0x331D, 0x18},
	{IMX_8BIT, 0x4084, 0x05},
	{IMX_8BIT, 0x4085, 0x00},
	{IMX_8BIT, 0x4086, 0x02},
	{IMX_8BIT, 0x4087, 0xD0},
	{IMX_8BIT, 0x4400, 0x00},
	/* Global Timing */
	{IMX_8BIT, 0x0830, 0x67},
	{IMX_8BIT, 0x0831, 0x27},
	{IMX_8BIT, 0x0832, 0x47},
	{IMX_8BIT, 0x0833, 0x27},
	{IMX_8BIT, 0x0834, 0x27},
	{IMX_8BIT, 0x0835, 0x1F},
	{IMX_8BIT, 0x0836, 0x87},
	{IMX_8BIT, 0x0837, 0x2F},
	{IMX_8BIT, 0x0839, 0x1F},
	{IMX_8BIT, 0x083A, 0x17},
	{IMX_8BIT, 0x083B, 0x02},
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x0A},
	{IMX_8BIT, 0x0203, 0x56},
	{IMX_TOK_TERM, 0, 0},
};

static struct imx_reg const imx135_D1_dvs_30fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03},
	{IMX_8BIT, 0x0112, 0x0A},
	{IMX_8BIT, 0x0113, 0x0A},
	{IMX_8BIT, 0x0381, 0x01},
	{IMX_8BIT, 0x0383, 0x01},
	{IMX_8BIT, 0x0385, 0x01},
	{IMX_8BIT, 0x0387, 0x01},
	{IMX_8BIT, 0x0390, 0x01},
	{IMX_8BIT, 0x0391, 0x44},
	{IMX_8BIT, 0x0392, 0x00},
	{IMX_8BIT, 0x0401, 0x02},
	{IMX_8BIT, 0x0404, 0x00},
	{IMX_8BIT, 0x0405, 0x11},
	{IMX_8BIT, 0x4082, 0x00},
	{IMX_8BIT, 0x4083, 0x00},
	{IMX_8BIT, 0x4203, 0xFF},
	{IMX_8BIT, 0x7006, 0x04},
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x0A},
	{IMX_8BIT, 0x0341, 0x5A},
	{IMX_8BIT, 0x0342, 0x11},
	{IMX_8BIT, 0x0343, 0xDC},
	{IMX_8BIT, 0x0344, 0x00},
	{IMX_8BIT, 0x0345, 0x70},
	{IMX_8BIT, 0x0346, 0x01},
	{IMX_8BIT, 0x0347, 0x18},
	{IMX_8BIT, 0x0348, 0x0F},
	{IMX_8BIT, 0x0349, 0xFF},
	{IMX_8BIT, 0x034A, 0x0B},
	{IMX_8BIT, 0x034B, 0x17},
	{IMX_8BIT, 0x034C, 0x03},
	{IMX_8BIT, 0x034D, 0xA8},
	{IMX_8BIT, 0x034E, 0x02},
	{IMX_8BIT, 0x034F, 0x5A},
	{IMX_8BIT, 0x0350, 0x00},
	{IMX_8BIT, 0x0351, 0x00},
	{IMX_8BIT, 0x0352, 0x00},
	{IMX_8BIT, 0x0353, 0x00},
	{IMX_8BIT, 0x0354, 0x03},
	{IMX_8BIT, 0x0355, 0xE4},
	{IMX_8BIT, 0x0356, 0x02},
	{IMX_8BIT, 0x0357, 0x80},
	{IMX_8BIT, 0x301D, 0x30},
	{IMX_8BIT, 0x3310, 0x03},
	{IMX_8BIT, 0x3311, 0xA8},
	{IMX_8BIT, 0x3312, 0x02},
	{IMX_8BIT, 0x3313, 0x5A},
	{IMX_8BIT, 0x331C, 0x02},
	{IMX_8BIT, 0x331D, 0x3A},
	{IMX_8BIT, 0x4084, 0x03},
	{IMX_8BIT, 0x4085, 0xA8},
	{IMX_8BIT, 0x4086, 0x02},
	{IMX_8BIT, 0x4087, 0x5A},
	{IMX_8BIT, 0x4400, 0x00},
	/* Global Timing */
	{IMX_8BIT, 0x0830, 0x67},
	{IMX_8BIT, 0x0831, 0x27},
	{IMX_8BIT, 0x0832, 0x47},
	{IMX_8BIT, 0x0833, 0x27},
	{IMX_8BIT, 0x0834, 0x27},
	{IMX_8BIT, 0x0835, 0x1F},
	{IMX_8BIT, 0x0836, 0x87},
	{IMX_8BIT, 0x0837, 0x2F},
	{IMX_8BIT, 0x0839, 0x1F},
	{IMX_8BIT, 0x083A, 0x17},
	{IMX_8BIT, 0x083B, 0x02},
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x0A},
	{IMX_8BIT, 0x0203, 0x56},
	{IMX_TOK_TERM, 0, 0},
};

static struct imx_reg const imx135_preview_30fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03},
	{IMX_8BIT, 0x0112, 0x0A},
	{IMX_8BIT, 0x0113, 0x0A},
	{IMX_8BIT, 0x0381, 0x01},
	{IMX_8BIT, 0x0383, 0x01},
	{IMX_8BIT, 0x0385, 0x01},
	{IMX_8BIT, 0x0387, 0x01},
	{IMX_8BIT, 0x0390, 0x01},
	{IMX_8BIT, 0x0391, 0x44},
	{IMX_8BIT, 0x0392, 0x00},
	{IMX_8BIT, 0x0401, 0x02},
	{IMX_8BIT, 0x0404, 0x00},
	{IMX_8BIT, 0x0405, 0x13},
	{IMX_8BIT, 0x4082, 0x00},
	{IMX_8BIT, 0x4083, 0x00},
	{IMX_8BIT, 0x4203, 0xFF},
	{IMX_8BIT, 0x7006, 0x04},
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x0A},
	{IMX_8BIT, 0x0341, 0x5A},
	{IMX_8BIT, 0x0342, 0x11},
	{IMX_8BIT, 0x0343, 0xDC},
	{IMX_8BIT, 0x0344, 0x00},
	{IMX_8BIT, 0x0345, 0x58},
	{IMX_8BIT, 0x0346, 0x00},
	{IMX_8BIT, 0x0347, 0x60},
	{IMX_8BIT, 0x0348, 0x10},
	{IMX_8BIT, 0x0349, 0x17},
	{IMX_8BIT, 0x034A, 0x0B},
	{IMX_8BIT, 0x034B, 0xCF},
	{IMX_8BIT, 0x034C, 0x03},
	{IMX_8BIT, 0x034D, 0x50},
	{IMX_8BIT, 0x034E, 0x02},
	{IMX_8BIT, 0x034F, 0x68},
	{IMX_8BIT, 0x0350, 0x00},
	{IMX_8BIT, 0x0351, 0x00},
	{IMX_8BIT, 0x0352, 0x00},
	{IMX_8BIT, 0x0353, 0x00},
	{IMX_8BIT, 0x0354, 0x03},
	{IMX_8BIT, 0x0355, 0xF0},
	{IMX_8BIT, 0x0356, 0x02},
	{IMX_8BIT, 0x0357, 0xDC},
	{IMX_8BIT, 0x301D, 0x30},
	{IMX_8BIT, 0x3310, 0x03},
	{IMX_8BIT, 0x3311, 0x50},
	{IMX_8BIT, 0x3312, 0x02},
	{IMX_8BIT, 0x3313, 0x68},
	{IMX_8BIT, 0x331C, 0x02},
	{IMX_8BIT, 0x331D, 0x1C},
	{IMX_8BIT, 0x4084, 0x03},
	{IMX_8BIT, 0x4085, 0x50},
	{IMX_8BIT, 0x4086, 0x02},
	{IMX_8BIT, 0x4087, 0x68},
	{IMX_8BIT, 0x4400, 0x00},
	/* Global Timing */
	{IMX_8BIT, 0x0830, 0x67},
	{IMX_8BIT, 0x0831, 0x27},
	{IMX_8BIT, 0x0832, 0x47},
	{IMX_8BIT, 0x0833, 0x27},
	{IMX_8BIT, 0x0834, 0x27},
	{IMX_8BIT, 0x0835, 0x1F},
	{IMX_8BIT, 0x0836, 0x87},
	{IMX_8BIT, 0x0837, 0x2F},
	{IMX_8BIT, 0x0839, 0x1F},
	{IMX_8BIT, 0x083A, 0x17},
	{IMX_8BIT, 0x083B, 0x02},
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x0A},
	{IMX_8BIT, 0x0203, 0x56},
	{IMX_TOK_TERM, 0, 0},
};

static struct imx_reg const imx135_VGA_dvs_30fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	/* Clock setting */
	{IMX_8BIT, 0x011E, 0x13},
	{IMX_8BIT, 0x011F, 0x33},
	{IMX_8BIT, 0x0301, 0x05},
	{IMX_8BIT, 0x0303, 0x01},
	{IMX_8BIT, 0x0305, 0x0C},
	{IMX_8BIT, 0x0309, 0x05},
	{IMX_8BIT, 0x030B, 0x01},
	{IMX_8BIT, 0x030C, 0x02},
	{IMX_8BIT, 0x030D, 0x32},
	{IMX_8BIT, 0x030E, 0x01},
	{IMX_8BIT, 0x3A06, 0x11},
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03},
	{IMX_8BIT, 0x0112, 0x0A},
	{IMX_8BIT, 0x0113, 0x0A},
	{IMX_8BIT, 0x0381, 0x01},
	{IMX_8BIT, 0x0383, 0x01},
	{IMX_8BIT, 0x0385, 0x01},
	{IMX_8BIT, 0x0387, 0x01},
	{IMX_8BIT, 0x0390, 0x01},
	{IMX_8BIT, 0x0391, 0x44},
	{IMX_8BIT, 0x0392, 0x00},
	{IMX_8BIT, 0x0401, 0x02},
	{IMX_8BIT, 0x0404, 0x00},
	{IMX_8BIT, 0x0405, 0x14},
	{IMX_8BIT, 0x4082, 0x00},
	{IMX_8BIT, 0x4083, 0x00},
	{IMX_8BIT, 0x4203, 0xFF},
	{IMX_8BIT, 0x7006, 0x04},
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x0A},
	{IMX_8BIT, 0x0341, 0x5A},
	{IMX_8BIT, 0x0342, 0x11},
	{IMX_8BIT, 0x0343, 0xDC},
	{IMX_8BIT, 0x0344, 0x00},
	{IMX_8BIT, 0x0345, 0x34},
	{IMX_8BIT, 0x0346, 0x00},
	{IMX_8BIT, 0x0347, 0x14},
	{IMX_8BIT, 0x0348, 0x10},
	{IMX_8BIT, 0x0349, 0x3B},
	{IMX_8BIT, 0x034A, 0x0C},
	{IMX_8BIT, 0x034B, 0x1B},
	{IMX_8BIT, 0x034C, 0x03},
	{IMX_8BIT, 0x034D, 0x34},
	{IMX_8BIT, 0x034E, 0x02},
	{IMX_8BIT, 0x034F, 0x68},
	{IMX_8BIT, 0x0350, 0x00},
	{IMX_8BIT, 0x0351, 0x00},
	{IMX_8BIT, 0x0352, 0x00},
	{IMX_8BIT, 0x0353, 0x00},
	{IMX_8BIT, 0x0354, 0x04},
	{IMX_8BIT, 0x0355, 0x02},
	{IMX_8BIT, 0x0356, 0x03},
	{IMX_8BIT, 0x0357, 0x02},
	{IMX_8BIT, 0x301D, 0x30},
	{IMX_8BIT, 0x3310, 0x03},
	{IMX_8BIT, 0x3311, 0x34},
	{IMX_8BIT, 0x3312, 0x02},
	{IMX_8BIT, 0x3313, 0x68},
	{IMX_8BIT, 0x331C, 0x02},
	{IMX_8BIT, 0x331D, 0x21},
	{IMX_8BIT, 0x4084, 0x03},
	{IMX_8BIT, 0x4085, 0x34},
	{IMX_8BIT, 0x4086, 0x02},
	{IMX_8BIT, 0x4087, 0x68},
	{IMX_8BIT, 0x4400, 0x00},
	/* Global Timing */
	{IMX_8BIT, 0x0830, 0x67},
	{IMX_8BIT, 0x0831, 0x27},
	{IMX_8BIT, 0x0832, 0x47},
	{IMX_8BIT, 0x0833, 0x27},
	{IMX_8BIT, 0x0834, 0x27},
	{IMX_8BIT, 0x0835, 0x1F},
	{IMX_8BIT, 0x0836, 0x87},
	{IMX_8BIT, 0x0837, 0x2F},
	{IMX_8BIT, 0x0839, 0x1F},
	{IMX_8BIT, 0x083A, 0x17},
	{IMX_8BIT, 0x083B, 0x02},
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x0A},
	{IMX_8BIT, 0x0203, 0x56},
	/* Gain Setting */
	{IMX_8BIT, 0x0205, 0x00},
	{IMX_8BIT, 0x020E, 0x01},
	{IMX_8BIT, 0x020F, 0x00},
	{IMX_8BIT, 0x0210, 0x01},
	{IMX_8BIT, 0x0211, 0x00},
	{IMX_8BIT, 0x0212, 0x01},
	{IMX_8BIT, 0x0213, 0x00},
	{IMX_8BIT, 0x0214, 0x01},
	{IMX_8BIT, 0x0215, 0x00},
	/* HDR Setting */
	{IMX_8BIT, 0x0230, 0x00},
	{IMX_8BIT, 0x0231, 0x00},
	{IMX_8BIT, 0x0233, 0x00},
	{IMX_8BIT, 0x0234, 0x00},
	{IMX_8BIT, 0x0235, 0x40},
	{IMX_8BIT, 0x0238, 0x00},
	{IMX_8BIT, 0x0239, 0x04},
	{IMX_8BIT, 0x023B, 0x00},
	{IMX_8BIT, 0x023C, 0x01},
	{IMX_8BIT, 0x33B0, 0x04},
	{IMX_8BIT, 0x33B1, 0x00},
	{IMX_8BIT, 0x33B3, 0x00},
	{IMX_8BIT, 0x33B4, 0x01},
	{IMX_8BIT, 0x3800, 0x00},
	{IMX_TOK_TERM, 0, 0},
};

static struct imx_reg const imx135_CIF_30fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03},
	{IMX_8BIT, 0x0112, 0x0A},
	{IMX_8BIT, 0x0113, 0x0A},
	{IMX_8BIT, 0x0381, 0x01},
	{IMX_8BIT, 0x0383, 0x01},
	{IMX_8BIT, 0x0385, 0x01},
	{IMX_8BIT, 0x0387, 0x01},
	{IMX_8BIT, 0x0390, 0x01},
	{IMX_8BIT, 0x0391, 0x44},
	{IMX_8BIT, 0x0392, 0x00},
	{IMX_8BIT, 0x0401, 0x02},
	{IMX_8BIT, 0x0404, 0x00},
	{IMX_8BIT, 0x0405, 0x23},
	{IMX_8BIT, 0x4082, 0x00},
	{IMX_8BIT, 0x4083, 0x00},
	{IMX_8BIT, 0x4203, 0xFF},
	{IMX_8BIT, 0x7006, 0x04},
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x0A},
	{IMX_8BIT, 0x0341, 0x5A},
	{IMX_8BIT, 0x0342, 0x11},
	{IMX_8BIT, 0x0343, 0xDC},
	{IMX_8BIT, 0x0344, 0x00},
	{IMX_8BIT, 0x0345, 0xA0},
	{IMX_8BIT, 0x0346, 0x00},
	{IMX_8BIT, 0x0347, 0x24},
	{IMX_8BIT, 0x0348, 0x0F},
	{IMX_8BIT, 0x0349, 0xCF},
	{IMX_8BIT, 0x034A, 0x0C},
	{IMX_8BIT, 0x034B, 0x0B},
	{IMX_8BIT, 0x034C, 0x01},
	{IMX_8BIT, 0x034D, 0xBC},
	{IMX_8BIT, 0x034E, 0x01},
	{IMX_8BIT, 0x034F, 0x5C},
	{IMX_8BIT, 0x0350, 0x00},
	{IMX_8BIT, 0x0351, 0x00},
	{IMX_8BIT, 0x0352, 0x00},
	{IMX_8BIT, 0x0353, 0x00},
	{IMX_8BIT, 0x0354, 0x03},
	{IMX_8BIT, 0x0355, 0xCC},
	{IMX_8BIT, 0x0356, 0x02},
	{IMX_8BIT, 0x0357, 0xFA},
	{IMX_8BIT, 0x301D, 0x30},
	{IMX_8BIT, 0x3310, 0x01},
	{IMX_8BIT, 0x3311, 0xBC},
	{IMX_8BIT, 0x3312, 0x01},
	{IMX_8BIT, 0x3313, 0x5C},
	{IMX_8BIT, 0x331C, 0x01},
	{IMX_8BIT, 0x331D, 0x7C},
	{IMX_8BIT, 0x4084, 0x01},
	{IMX_8BIT, 0x4085, 0xBC},
	{IMX_8BIT, 0x4086, 0x01},
	{IMX_8BIT, 0x4087, 0x5C},
	{IMX_8BIT, 0x4400, 0x00},
	/* Global Timing */
	{IMX_8BIT, 0x0830, 0x67},
	{IMX_8BIT, 0x0831, 0x27},
	{IMX_8BIT, 0x0832, 0x47},
	{IMX_8BIT, 0x0833, 0x27},
	{IMX_8BIT, 0x0834, 0x27},
	{IMX_8BIT, 0x0835, 0x1F},
	{IMX_8BIT, 0x0836, 0x87},
	{IMX_8BIT, 0x0837, 0x2F},
	{IMX_8BIT, 0x0839, 0x1F},
	{IMX_8BIT, 0x083A, 0x17},
	{IMX_8BIT, 0x083B, 0x02},
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x0A},
	{IMX_8BIT, 0x0203, 0x56},
	{IMX_TOK_TERM, 0, 0},
};

static struct imx_reg const imx135_QVGA_dvs_30fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03},
	{IMX_8BIT, 0x0112, 0x0A},
	{IMX_8BIT, 0x0113, 0x0A},
	{IMX_8BIT, 0x0381, 0x01},
	{IMX_8BIT, 0x0383, 0x01},
	{IMX_8BIT, 0x0385, 0x01},
	{IMX_8BIT, 0x0387, 0x01},
	{IMX_8BIT, 0x0390, 0x01},
	{IMX_8BIT, 0x0391, 0x44},
	{IMX_8BIT, 0x0392, 0x00},
	{IMX_8BIT, 0x0401, 0x02},
	{IMX_8BIT, 0x0404, 0x00},
	{IMX_8BIT, 0x0405, 0x28},
	{IMX_8BIT, 0x4082, 0x00},
	{IMX_8BIT, 0x4083, 0x00},
	{IMX_8BIT, 0x4203, 0xFF},
	{IMX_8BIT, 0x7006, 0x04},
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x0A},
	{IMX_8BIT, 0x0341, 0x5A},
	{IMX_8BIT, 0x0342, 0x11},
	{IMX_8BIT, 0x0343, 0xDC},
	{IMX_8BIT, 0x0344, 0x00},
	{IMX_8BIT, 0x0345, 0x40},
	{IMX_8BIT, 0x0346, 0x00},
	{IMX_8BIT, 0x0347, 0x14},
	{IMX_8BIT, 0x0348, 0x10},
	{IMX_8BIT, 0x0349, 0x2F},
	{IMX_8BIT, 0x034A, 0x0C},
	{IMX_8BIT, 0x034B, 0x1B},
	{IMX_8BIT, 0x034C, 0x01},
	{IMX_8BIT, 0x034D, 0x98},
	{IMX_8BIT, 0x034E, 0x01},
	{IMX_8BIT, 0x034F, 0x34},
	{IMX_8BIT, 0x0350, 0x00},
	{IMX_8BIT, 0x0351, 0x00},
	{IMX_8BIT, 0x0352, 0x00},
	{IMX_8BIT, 0x0353, 0x00},
	{IMX_8BIT, 0x0354, 0x03},
	{IMX_8BIT, 0x0355, 0xFC},
	{IMX_8BIT, 0x0356, 0x03},
	{IMX_8BIT, 0x0357, 0x02},
	{IMX_8BIT, 0x301D, 0x30},
	{IMX_8BIT, 0x3310, 0x01},
	{IMX_8BIT, 0x3311, 0x98},
	{IMX_8BIT, 0x3312, 0x01},
	{IMX_8BIT, 0x3313, 0x34},
	{IMX_8BIT, 0x331C, 0x01},
	{IMX_8BIT, 0x331D, 0x68},
	{IMX_8BIT, 0x4084, 0x01},
	{IMX_8BIT, 0x4085, 0x98},
	{IMX_8BIT, 0x4086, 0x01},
	{IMX_8BIT, 0x4087, 0x34},
	{IMX_8BIT, 0x4400, 0x00},
	/* Global Timing */
	{IMX_8BIT, 0x0830, 0x67},
	{IMX_8BIT, 0x0831, 0x27},
	{IMX_8BIT, 0x0832, 0x47},
	{IMX_8BIT, 0x0833, 0x27},
	{IMX_8BIT, 0x0834, 0x27},
	{IMX_8BIT, 0x0835, 0x1F},
	{IMX_8BIT, 0x0836, 0x87},
	{IMX_8BIT, 0x0837, 0x2F},
	{IMX_8BIT, 0x0839, 0x1F},
	{IMX_8BIT, 0x083A, 0x17},
	{IMX_8BIT, 0x083B, 0x02},
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x0A},
	{IMX_8BIT, 0x0203, 0x56},
	{IMX_TOK_TERM, 0, 0},
};

static struct imx_reg const imx135_QCIF_dvs_30fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03},
	{IMX_8BIT, 0x0112, 0x0A},
	{IMX_8BIT, 0x0113, 0x0A},
	{IMX_8BIT, 0x0381, 0x01},
	{IMX_8BIT, 0x0383, 0x01},
	{IMX_8BIT, 0x0385, 0x01},
	{IMX_8BIT, 0x0387, 0x01},
	{IMX_8BIT, 0x0390, 0x01},
	{IMX_8BIT, 0x0391, 0x44},
	{IMX_8BIT, 0x0392, 0x00},
	{IMX_8BIT, 0x0401, 0x02},
	{IMX_8BIT, 0x0404, 0x00},
	{IMX_8BIT, 0x0405, 0x46},
	{IMX_8BIT, 0x4082, 0x00},
	{IMX_8BIT, 0x4083, 0x00},
	{IMX_8BIT, 0x4203, 0xFF},
	{IMX_8BIT, 0x7006, 0x04},
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x0A},
	{IMX_8BIT, 0x0341, 0x5A},
	{IMX_8BIT, 0x0342, 0x11},
	{IMX_8BIT, 0x0343, 0xDC},
	{IMX_8BIT, 0x0344, 0x00},
	{IMX_8BIT, 0x0345, 0xD4},
	{IMX_8BIT, 0x0346, 0x00},
	{IMX_8BIT, 0x0347, 0x14},
	{IMX_8BIT, 0x0348, 0x0F},
	{IMX_8BIT, 0x0349, 0x9B},
	{IMX_8BIT, 0x034A, 0x0C},
	{IMX_8BIT, 0x034B, 0x1B},
	{IMX_8BIT, 0x034C, 0x00},
	{IMX_8BIT, 0x034D, 0xD8},
	{IMX_8BIT, 0x034E, 0x00},
	{IMX_8BIT, 0x034F, 0xB0},
	{IMX_8BIT, 0x0350, 0x00},
	{IMX_8BIT, 0x0351, 0x00},
	{IMX_8BIT, 0x0352, 0x00},
	{IMX_8BIT, 0x0353, 0x00},
	{IMX_8BIT, 0x0354, 0x03},
	{IMX_8BIT, 0x0355, 0xB2},
	{IMX_8BIT, 0x0356, 0x03},
	{IMX_8BIT, 0x0357, 0x02},
	{IMX_8BIT, 0x301D, 0x30},
	{IMX_8BIT, 0x3310, 0x00},
	{IMX_8BIT, 0x3311, 0xD8},
	{IMX_8BIT, 0x3312, 0x00},
	{IMX_8BIT, 0x3313, 0xB0},
	{IMX_8BIT, 0x331C, 0x00},
	{IMX_8BIT, 0x331D, 0xD4},
	{IMX_8BIT, 0x4084, 0x00},
	{IMX_8BIT, 0x4085, 0xD8},
	{IMX_8BIT, 0x4086, 0x00},
	{IMX_8BIT, 0x4087, 0xB0},
	{IMX_8BIT, 0x4400, 0x00},
	/* Global Timing */
	{IMX_8BIT, 0x0830, 0x67},
	{IMX_8BIT, 0x0831, 0x27},
	{IMX_8BIT, 0x0832, 0x47},
	{IMX_8BIT, 0x0833, 0x27},
	{IMX_8BIT, 0x0834, 0x27},
	{IMX_8BIT, 0x0835, 0x1F},
	{IMX_8BIT, 0x0836, 0x87},
	{IMX_8BIT, 0x0837, 0x2F},
	{IMX_8BIT, 0x0839, 0x1F},
	{IMX_8BIT, 0x083A, 0x17},
	{IMX_8BIT, 0x083B, 0x02},
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x0A},
	{IMX_8BIT, 0x0203, 0x56},
	{IMX_TOK_TERM, 0, 0},
};

static struct imx_reg const imx135_6M_30fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03},
	{IMX_8BIT, 0x0112, 0x0A},
	{IMX_8BIT, 0x0113, 0x0A},
	{IMX_8BIT, 0x0381, 0x01},
	{IMX_8BIT, 0x0383, 0x01},
	{IMX_8BIT, 0x0385, 0x01},
	{IMX_8BIT, 0x0387, 0x01},
	{IMX_8BIT, 0x0390, 0x00},
	{IMX_8BIT, 0x0391, 0x11},
	{IMX_8BIT, 0x0392, 0x00},
	{IMX_8BIT, 0x0401, 0x02},
	{IMX_8BIT, 0x0404, 0x00},
	{IMX_8BIT, 0x0405, 0x14},
	{IMX_8BIT, 0x4082, 0x00},
	{IMX_8BIT, 0x4083, 0x00},
	{IMX_8BIT, 0x4203, 0xFF},
	{IMX_8BIT, 0x7006, 0x04},
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x0A},
	{IMX_8BIT, 0x0341, 0x5A},
	{IMX_8BIT, 0x0342, 0x11},
	{IMX_8BIT, 0x0343, 0xDC},
	{IMX_8BIT, 0x0344, 0x00},
	{IMX_8BIT, 0x0345, 0x36},
	{IMX_8BIT, 0x0346, 0x01},
	{IMX_8BIT, 0x0347, 0x94},
	{IMX_8BIT, 0x0348, 0x10},
	{IMX_8BIT, 0x0349, 0x39},
	{IMX_8BIT, 0x034A, 0x0A},
	{IMX_8BIT, 0x034B, 0x9F},
	{IMX_8BIT, 0x034C, 0x0C},
	{IMX_8BIT, 0x034D, 0xD0},
	{IMX_8BIT, 0x034E, 0x07},
	{IMX_8BIT, 0x034F, 0x3C},
	{IMX_8BIT, 0x0350, 0x00},
	{IMX_8BIT, 0x0351, 0x00},
	{IMX_8BIT, 0x0352, 0x00},
	{IMX_8BIT, 0x0353, 0x00},
	{IMX_8BIT, 0x0354, 0x10},
	{IMX_8BIT, 0x0355, 0x04},
	{IMX_8BIT, 0x0356, 0x09},
	{IMX_8BIT, 0x0357, 0x0C},
	{IMX_8BIT, 0x301D, 0x30},
	{IMX_8BIT, 0x3310, 0x0C},
	{IMX_8BIT, 0x3311, 0xD0},
	{IMX_8BIT, 0x3312, 0x07},
	{IMX_8BIT, 0x3313, 0x3C},
	{IMX_8BIT, 0x331C, 0x02},
	{IMX_8BIT, 0x331D, 0xA0},
	{IMX_8BIT, 0x4084, 0x0C},
	{IMX_8BIT, 0x4085, 0xD0},
	{IMX_8BIT, 0x4086, 0x07},
	{IMX_8BIT, 0x4087, 0x3C},
	{IMX_8BIT, 0x4400, 0x00},
	/* Global Timing */
	{IMX_8BIT, 0x0830, 0x87},
	{IMX_8BIT, 0x0831, 0x3F},
	{IMX_8BIT, 0x0832, 0x67},
	{IMX_8BIT, 0x0833, 0x3F},
	{IMX_8BIT, 0x0834, 0x3F},
	{IMX_8BIT, 0x0835, 0x4F},
	{IMX_8BIT, 0x0836, 0xDF},
	{IMX_8BIT, 0x0837, 0x47},
	{IMX_8BIT, 0x0839, 0x1F},
	{IMX_8BIT, 0x083A, 0x17},
	{IMX_8BIT, 0x083B, 0x02},
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x0A},
	{IMX_8BIT, 0x0203, 0x56},
	{IMX_TOK_TERM, 0, 0},
};

static struct imx_reg const imx135_3M_30fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	/* Clock setting */
	{IMX_8BIT, 0x011E, 0x13},
	{IMX_8BIT, 0x011F, 0x33},
	{IMX_8BIT, 0x0301, 0x05},
	{IMX_8BIT, 0x0303, 0x01},
	{IMX_8BIT, 0x0305, 0x0B},
	{IMX_8BIT, 0x0309, 0x05},
	{IMX_8BIT, 0x030B, 0x02},
	{IMX_8BIT, 0x030C, 0x01},
	{IMX_8BIT, 0x030D, 0x00},
	{IMX_8BIT, 0x030E, 0x01},
	{IMX_8BIT, 0x3A06, 0x12},
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03},
	{IMX_8BIT, 0x0112, 0x0A},
	{IMX_8BIT, 0x0113, 0x0A},
	{IMX_8BIT, 0x0381, 0x01},
	{IMX_8BIT, 0x0383, 0x01},
	{IMX_8BIT, 0x0385, 0x01},
	{IMX_8BIT, 0x0387, 0x01},
	{IMX_8BIT, 0x0390, 0x01},
	{IMX_8BIT, 0x0391, 0x22},
	{IMX_8BIT, 0x0392, 0x00},
	{IMX_8BIT, 0x0401, 0x00},
	{IMX_8BIT, 0x0404, 0x00},
	{IMX_8BIT, 0x0405, 0x10},
	{IMX_8BIT, 0x4082, 0x01},
	{IMX_8BIT, 0x4083, 0x01},
	{IMX_8BIT, 0x4203, 0xFF},
	{IMX_8BIT, 0x7006, 0x04},
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x0A},
	{IMX_8BIT, 0x0341, 0x5A},
	{IMX_8BIT, 0x0342, 0x11},
	{IMX_8BIT, 0x0343, 0xDC},
	{IMX_8BIT, 0x0344, 0x00},
	{IMX_8BIT, 0x0345, 0x28},
	{IMX_8BIT, 0x0346, 0x00},
	{IMX_8BIT, 0x0347, 0x08},
	{IMX_8BIT, 0x0348, 0x10},
	{IMX_8BIT, 0x0349, 0x47},
	{IMX_8BIT, 0x034A, 0x0C},
	{IMX_8BIT, 0x034B, 0x27},
	{IMX_8BIT, 0x034C, 0x08},
	{IMX_8BIT, 0x034D, 0x10},
	{IMX_8BIT, 0x034E, 0x06},
	{IMX_8BIT, 0x034F, 0x10},
	{IMX_8BIT, 0x0350, 0x00},
	{IMX_8BIT, 0x0351, 0x00},
	{IMX_8BIT, 0x0352, 0x00},
	{IMX_8BIT, 0x0353, 0x00},
	{IMX_8BIT, 0x0354, 0x08},
	{IMX_8BIT, 0x0355, 0x10},
	{IMX_8BIT, 0x0356, 0x06},
	{IMX_8BIT, 0x0357, 0x10},
	{IMX_8BIT, 0x301D, 0x30},
	{IMX_8BIT, 0x3310, 0x08},
	{IMX_8BIT, 0x3311, 0x10},
	{IMX_8BIT, 0x3312, 0x06},
	{IMX_8BIT, 0x3313, 0x10},
	{IMX_8BIT, 0x331C, 0x00},
	{IMX_8BIT, 0x331D, 0xAA},
	{IMX_8BIT, 0x4084, 0x00},
	{IMX_8BIT, 0x4085, 0x00},
	{IMX_8BIT, 0x4086, 0x00},
	{IMX_8BIT, 0x4087, 0x00},
	{IMX_8BIT, 0x4400, 0x00},
	/* Global Timing */
	{IMX_8BIT, 0x0830, 0x5F},
	{IMX_8BIT, 0x0831, 0x0C},
	{IMX_8BIT, 0x0832, 0x3F},
	{IMX_8BIT, 0x0833, 0x1F},
	{IMX_8BIT, 0x0834, 0x1F},
	{IMX_8BIT, 0x0835, 0x17},
	{IMX_8BIT, 0x0836, 0x67},
	{IMX_8BIT, 0x0837, 0x27},
	{IMX_8BIT, 0x0839, 0x1F},
	{IMX_8BIT, 0x083A, 0x17},
	{IMX_8BIT, 0x083B, 0x02},
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x06},
	{IMX_8BIT, 0x0203, 0x22},
	/* Gain Setting */
	{IMX_8BIT, 0x0205, 0x00},
	{IMX_8BIT, 0x020E, 0x01},
	{IMX_8BIT, 0x020F, 0x00},
	{IMX_8BIT, 0x0210, 0x01},
	{IMX_8BIT, 0x0211, 0x00},
	{IMX_8BIT, 0x0212, 0x01},
	{IMX_8BIT, 0x0213, 0x00},
	{IMX_8BIT, 0x0214, 0x01},
	{IMX_8BIT, 0x0215, 0x00},
	/* HDR Setting */
	{IMX_8BIT, 0x0230, 0x00},
	{IMX_8BIT, 0x0231, 0x00},
	{IMX_8BIT, 0x0233, 0x00},
	{IMX_8BIT, 0x0234, 0x00},
	{IMX_8BIT, 0x0235, 0x40},
	{IMX_8BIT, 0x0238, 0x00},
	{IMX_8BIT, 0x0239, 0x04},
	{IMX_8BIT, 0x023B, 0x00},
	{IMX_8BIT, 0x023C, 0x01},
	{IMX_8BIT, 0x33B0, 0x04},
	{IMX_8BIT, 0x33B1, 0x00},
	{IMX_8BIT, 0x33B3, 0x00},
	{IMX_TOK_TERM, 0, 0},
};

static struct imx_reg const imx135_1080p_30fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	/* Clock setting */
	{IMX_8BIT, 0x011E, 0x13},
	{IMX_8BIT, 0x011F, 0x33},
	{IMX_8BIT, 0x0301, 0x05},
	{IMX_8BIT, 0x0303, 0x01},
	{IMX_8BIT, 0x0305, 0x0B},
	{IMX_8BIT, 0x0309, 0x05},
	{IMX_8BIT, 0x030B, 0x02},
	{IMX_8BIT, 0x030C, 0x01},
	{IMX_8BIT, 0x030D, 0x83},
	{IMX_8BIT, 0x030E, 0x01},
	{IMX_8BIT, 0x3A06, 0x12},
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03},
	{IMX_8BIT, 0x0112, 0x0A},
	{IMX_8BIT, 0x0113, 0x0A},
	{IMX_8BIT, 0x0381, 0x01},
	{IMX_8BIT, 0x0383, 0x01},
	{IMX_8BIT, 0x0385, 0x01},
	{IMX_8BIT, 0x0387, 0x01},
	{IMX_8BIT, 0x0390, 0x01},
	{IMX_8BIT, 0x0391, 0x22},
	{IMX_8BIT, 0x0392, 0x00},
	{IMX_8BIT, 0x0401, 0x02},
	{IMX_8BIT, 0x0404, 0x00},
	{IMX_8BIT, 0x0405, 0x11},
	{IMX_8BIT, 0x4082, 0x00},
	{IMX_8BIT, 0x4083, 0x00},
	{IMX_8BIT, 0x7006, 0x04},
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x04},
	{IMX_8BIT, 0x0341, 0xAC},
	{IMX_8BIT, 0x0342, 0x12},
	{IMX_8BIT, 0x0343, 0x58},
	{IMX_8BIT, 0x0344, 0x00},
	{IMX_8BIT, 0x0345, 0x2E},
	{IMX_8BIT, 0x0346, 0x01},
	{IMX_8BIT, 0x0347, 0x84},
	{IMX_8BIT, 0x0348, 0x10},
	{IMX_8BIT, 0x0349, 0x41},
	{IMX_8BIT, 0x034A, 0x0A},
	{IMX_8BIT, 0x034B, 0xAF},
	{IMX_8BIT, 0x034C, 0x07},
	{IMX_8BIT, 0x034D, 0x90},
	{IMX_8BIT, 0x034E, 0x04},
	{IMX_8BIT, 0x034F, 0x50},
	{IMX_8BIT, 0x0350, 0x00},
	{IMX_8BIT, 0x0351, 0x00},
	{IMX_8BIT, 0x0352, 0x00},
	{IMX_8BIT, 0x0353, 0x00},
	{IMX_8BIT, 0x0354, 0x08},
	{IMX_8BIT, 0x0355, 0x0A},
	{IMX_8BIT, 0x0356, 0x04},
	{IMX_8BIT, 0x0357, 0x96},
	{IMX_8BIT, 0x301D, 0x30},
	{IMX_8BIT, 0x3310, 0x07},
	{IMX_8BIT, 0x3311, 0x90},
	{IMX_8BIT, 0x3312, 0x04},
	{IMX_8BIT, 0x3313, 0x50},
	{IMX_8BIT, 0x331C, 0x01},
	{IMX_8BIT, 0x331D, 0x00},
	{IMX_8BIT, 0x4084, 0x07},
	{IMX_8BIT, 0x4085, 0x90},
	{IMX_8BIT, 0x4086, 0x04},
	{IMX_8BIT, 0x4087, 0x50},
	{IMX_8BIT, 0x4400, 0x00},
	/* Global Timing */
	{IMX_8BIT, 0x0830, 0x5F},
	{IMX_8BIT, 0x0831, 0x0C},
	{IMX_8BIT, 0x0832, 0x3F},
	{IMX_8BIT, 0x0833, 0x1F},
	{IMX_8BIT, 0x0834, 0x1F},
	{IMX_8BIT, 0x0835, 0x17},
	{IMX_8BIT, 0x0836, 0x67},
	{IMX_8BIT, 0x0837, 0x27},
	{IMX_8BIT, 0x0839, 0x1F},
	{IMX_8BIT, 0x083A, 0x17},
	{IMX_8BIT, 0x083B, 0x02},
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x04},
	{IMX_8BIT, 0x0203, 0xA8},
	/* Gain Setting */
	{IMX_8BIT, 0x0205, 0x00},
	{IMX_8BIT, 0x020E, 0x01},
	{IMX_8BIT, 0x020F, 0x00},
	{IMX_8BIT, 0x0210, 0x01},
	{IMX_8BIT, 0x0211, 0x00},
	{IMX_8BIT, 0x0212, 0x01},
	{IMX_8BIT, 0x0213, 0x00},
	{IMX_8BIT, 0x0214, 0x01},
	{IMX_8BIT, 0x0215, 0x00},
	/* HDR Setting */
	{IMX_8BIT, 0x0230, 0x00},
	{IMX_8BIT, 0x0231, 0x00},
	{IMX_8BIT, 0x0233, 0x00},
	{IMX_8BIT, 0x0234, 0x00},
	{IMX_8BIT, 0x0235, 0x40},
	{IMX_8BIT, 0x0238, 0x00},
	{IMX_8BIT, 0x0239, 0x04},
	{IMX_8BIT, 0x023B, 0x00},
	{IMX_8BIT, 0x023C, 0x01},
	{IMX_8BIT, 0x33B0, 0x04},
	{IMX_8BIT, 0x33B1, 0x00},
	{IMX_8BIT, 0x33B3, 0x00},
	{IMX_TOK_TERM, 0, 0},
};

static struct imx_reg const imx135_2M_30fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	{IMX_8BIT, 0x011E, 0x13},
	{IMX_8BIT, 0x011F, 0x33},
	{IMX_8BIT, 0x0301, 0x05},
	{IMX_8BIT, 0x0303, 0x01},
	{IMX_8BIT, 0x0305, 0x0C},
	{IMX_8BIT, 0x0309, 0x05},
	{IMX_8BIT, 0x030B, 0x02},
	{IMX_8BIT, 0x030C, 0x02},
	{IMX_8BIT, 0x030D, 0x38},
	{IMX_8BIT, 0x030E, 0x01},
	{IMX_8BIT, 0x3A06, 0x12},
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03},
	{IMX_8BIT, 0x0112, 0x0A},
	{IMX_8BIT, 0x0113, 0x0A},
	{IMX_8BIT, 0x0381, 0x01},
	{IMX_8BIT, 0x0383, 0x01},
	{IMX_8BIT, 0x0385, 0x01},
	{IMX_8BIT, 0x0387, 0x01},
	{IMX_8BIT, 0x0390, 0x01},
	{IMX_8BIT, 0x0391, 0x22},
	{IMX_8BIT, 0x0392, 0x00},
	{IMX_8BIT, 0x0401, 0x02},
	{IMX_8BIT, 0x0404, 0x00},
	{IMX_8BIT, 0x0405, 0x14},
	{IMX_8BIT, 0x4082, 0x00},
	{IMX_8BIT, 0x4083, 0x00},
	{IMX_8BIT, 0x4203, 0xFF},
	{IMX_8BIT, 0x7006, 0x04},
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x0C},
	{IMX_8BIT, 0x0341, 0x1C},
	{IMX_8BIT, 0x0342, 0x17},
	{IMX_8BIT, 0x0343, 0xD4},
	{IMX_8BIT, 0x0344, 0x00},
	{IMX_8BIT, 0x0345, 0x36},
	{IMX_8BIT, 0x0346, 0x00},
	{IMX_8BIT, 0x0347, 0x14},
	{IMX_8BIT, 0x0348, 0x10},
	{IMX_8BIT, 0x0349, 0x39},
	{IMX_8BIT, 0x034A, 0x0C},
	{IMX_8BIT, 0x034B, 0x1B},
	{IMX_8BIT, 0x034C, 0x06},
	{IMX_8BIT, 0x034D, 0x68},
	{IMX_8BIT, 0x034E, 0x04},
	{IMX_8BIT, 0x034F, 0xD0},
	{IMX_8BIT, 0x0350, 0x00},
	{IMX_8BIT, 0x0351, 0x00},
	{IMX_8BIT, 0x0352, 0x00},
	{IMX_8BIT, 0x0353, 0x00},
	{IMX_8BIT, 0x0354, 0x08},
	{IMX_8BIT, 0x0355, 0x02},
	{IMX_8BIT, 0x0356, 0x06},
	{IMX_8BIT, 0x0357, 0x04},
	{IMX_8BIT, 0x301D, 0x30},
	{IMX_8BIT, 0x3310, 0x06},
	{IMX_8BIT, 0x3311, 0x68},
	{IMX_8BIT, 0x3312, 0x04},
	{IMX_8BIT, 0x3313, 0xD0},
	{IMX_8BIT, 0x331C, 0x01},
	{IMX_8BIT, 0x331D, 0xC2},
	{IMX_8BIT, 0x4084, 0x06},
	{IMX_8BIT, 0x4085, 0x68},
	{IMX_8BIT, 0x4086, 0x04},
	{IMX_8BIT, 0x4087, 0xD0},
	{IMX_8BIT, 0x4400, 0x00},
	/* Global Timing */
	{IMX_8BIT, 0x0830, 0x5f},
	{IMX_8BIT, 0x0831, 0x0c},
	{IMX_8BIT, 0x0832, 0x3f},
	{IMX_8BIT, 0x0833, 0x1f},
	{IMX_8BIT, 0x0834, 0x1f},
	{IMX_8BIT, 0x0835, 0x17},
	{IMX_8BIT, 0x0836, 0x67},
	{IMX_8BIT, 0x0837, 0x27},
	{IMX_8BIT, 0x0839, 0x1F},
	{IMX_8BIT, 0x083A, 0x17},
	{IMX_8BIT, 0x083B, 0x02},
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x0c},
	{IMX_8BIT, 0x0203, 0x18},
	/* Gain Setting */
	{ IMX_8BIT, 0x0205, 0x00},
	{ IMX_8BIT, 0x020E, 0x01},
	{ IMX_8BIT, 0x020F, 0x00},
	{ IMX_8BIT, 0x0210, 0x01},
	{ IMX_8BIT, 0x0211, 0x00},
	{ IMX_8BIT, 0x0212, 0x01},
	{ IMX_8BIT, 0x0213, 0x00},
	{ IMX_8BIT, 0x0214, 0x01},
	{ IMX_8BIT, 0x0215, 0x00},
	/* HDR Setting */
	{ IMX_8BIT, 0x0230, 0x00},
	{ IMX_8BIT, 0x0231, 0x00},
	{ IMX_8BIT, 0x0233, 0x00},
	{ IMX_8BIT, 0x0234, 0x00},
	{ IMX_8BIT, 0x0235, 0x40},
	{ IMX_8BIT, 0x0238, 0x00},
	{ IMX_8BIT, 0x0239, 0x04},
	{ IMX_8BIT, 0x023B, 0x00},
	{ IMX_8BIT, 0x023C, 0x01},
	{ IMX_8BIT, 0x33B0, 0x04},
	{ IMX_8BIT, 0x33B1, 0x00},
	{ IMX_8BIT, 0x33B3, 0x00},
	{ IMX_8BIT, 0x33B4, 0x01},
	{ IMX_8BIT, 0x3800, 0x00},
	{IMX_TOK_TERM, 0, 0},
};

static struct imx_reg const imx135_1600x1200_30fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	{ IMX_8BIT, 0x011E, 0x13},
	{ IMX_8BIT, 0x011F, 0x33},
	{ IMX_8BIT, 0x0301, 0x05},
	{ IMX_8BIT, 0x0303, 0x01},
	{ IMX_8BIT, 0x0305, 0x0B},
	{ IMX_8BIT, 0x0309, 0x05},
	{ IMX_8BIT, 0x030B, 0x02},
	{ IMX_8BIT, 0x030C, 0x01},
	{ IMX_8BIT, 0x030D, 0x83},
	{ IMX_8BIT, 0x030E, 0x01},
	{ IMX_8BIT, 0x3A06, 0x12},
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03},
	{IMX_8BIT, 0x0112, 0x0A},
	{IMX_8BIT, 0x0113, 0x0A},
	{IMX_8BIT, 0x0381, 0x01},
	{IMX_8BIT, 0x0383, 0x01},
	{IMX_8BIT, 0x0385, 0x01},
	{IMX_8BIT, 0x0387, 0x01},
	{IMX_8BIT, 0x0390, 0x01},
	{IMX_8BIT, 0x0391, 0x22},
	{IMX_8BIT, 0x0392, 0x00},
	{IMX_8BIT, 0x0401, 0x02},
	{IMX_8BIT, 0x0404, 0x00},
	{IMX_8BIT, 0x0405, 0x14},
	{IMX_8BIT, 0x4082, 0x00},
	{IMX_8BIT, 0x4083, 0x00},
	{IMX_8BIT, 0x4203, 0xFF},
	{IMX_8BIT, 0x7006, 0x04},
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x08},
	{IMX_8BIT, 0x0341, 0xb8},
	{IMX_8BIT, 0x0342, 0x15},
	{IMX_8BIT, 0x0343, 0x18},
	{IMX_8BIT, 0x0344, 0x00},
	{IMX_8BIT, 0x0345, 0x40},
	{IMX_8BIT, 0x0346, 0x00},
	{IMX_8BIT, 0x0347, 0x20},
	{IMX_8BIT, 0x0348, 0x10},
	{IMX_8BIT, 0x0349, 0x2F},
	{IMX_8BIT, 0x034A, 0x0C},
	{IMX_8BIT, 0x034B, 0x13},
	{IMX_8BIT, 0x034C, 0x06},
	{IMX_8BIT, 0x034D, 0x60},
	{IMX_8BIT, 0x034E, 0x04},
	{IMX_8BIT, 0x034F, 0xC8},
	{IMX_8BIT, 0x0350, 0x00},
	{IMX_8BIT, 0x0351, 0x00},
	{IMX_8BIT, 0x0352, 0x00},
	{IMX_8BIT, 0x0353, 0x00},
	{IMX_8BIT, 0x0354, 0x07},
	{IMX_8BIT, 0x0355, 0xF8},
	{IMX_8BIT, 0x0356, 0x05},
	{IMX_8BIT, 0x0357, 0xFA},
	{IMX_8BIT, 0x301D, 0x30},
	{IMX_8BIT, 0x3310, 0x06},
	{IMX_8BIT, 0x3311, 0x60},
	{IMX_8BIT, 0x3312, 0x04},
	{IMX_8BIT, 0x3313, 0xC8},
	{IMX_8BIT, 0x331C, 0x01},
	{IMX_8BIT, 0x331D, 0x9A},
	{IMX_8BIT, 0x4084, 0x06},
	{IMX_8BIT, 0x4085, 0x60},
	{IMX_8BIT, 0x4086, 0x04},
	{IMX_8BIT, 0x4087, 0xC8},
	{IMX_8BIT, 0x4400, 0x00},
	/* Global Timing */
	{IMX_8BIT, 0x0830, 0x67},
	{IMX_8BIT, 0x0831, 0x27},
	{IMX_8BIT, 0x0832, 0x47},
	{IMX_8BIT, 0x0833, 0x27},
	{IMX_8BIT, 0x0834, 0x27},
	{IMX_8BIT, 0x0835, 0x1F},
	{IMX_8BIT, 0x0836, 0x87},
	{IMX_8BIT, 0x0837, 0x2F},
	{IMX_8BIT, 0x0839, 0x1F},
	{IMX_8BIT, 0x083A, 0x17},
	{IMX_8BIT, 0x083B, 0x02},
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x0A},
	{IMX_8BIT, 0x0203, 0x56},
	{IMX_TOK_TERM, 0, 0},
};

static struct imx_reg const imx135_720p_30fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	/* Clock setting */
	{IMX_8BIT, 0x011E, 0x13},
	{IMX_8BIT, 0x011F, 0x33},
	{IMX_8BIT, 0x0301, 0x05},
	{IMX_8BIT, 0x0303, 0x01},
	{IMX_8BIT, 0x0305, 0x0B},
	{IMX_8BIT, 0x0309, 0x05},
	{IMX_8BIT, 0x030B, 0x02},
	{IMX_8BIT, 0x030C, 0x01},
	{IMX_8BIT, 0x030D, 0x83},
	{IMX_8BIT, 0x030E, 0x01},
	{IMX_8BIT, 0x3A06, 0x12},
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03},
	{IMX_8BIT, 0x0112, 0x0A},
	{IMX_8BIT, 0x0113, 0x0A},
	{IMX_8BIT, 0x0381, 0x01},
	{IMX_8BIT, 0x0383, 0x01},
	{IMX_8BIT, 0x0385, 0x01},
	{IMX_8BIT, 0x0387, 0x01},
	{IMX_8BIT, 0x0390, 0x01},
	{IMX_8BIT, 0x0391, 0x22},
	{IMX_8BIT, 0x0392, 0x00},
	{IMX_8BIT, 0x0401, 0x02},
	{IMX_8BIT, 0x0404, 0x00},
	{IMX_8BIT, 0x0405, 0x19},
	{IMX_8BIT, 0x4082, 0x00},
	{IMX_8BIT, 0x4083, 0x00},
	{IMX_8BIT, 0x7006, 0x04},
	/* optimal function setting */
	{IMX_8BIT, 0x0700, 0x00},
	{IMX_8BIT, 0x3A63, 0x00},
	{IMX_8BIT, 0x4100, 0xF8},
	{IMX_8BIT, 0x4203, 0xFF},
	{IMX_8BIT, 0x4344, 0x00},
	{IMX_8BIT, 0x441C, 0x01},
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x04},
	{IMX_8BIT, 0x0341, 0xAC},
	{IMX_8BIT, 0x0342, 0x12},
	{IMX_8BIT, 0x0343, 0x58},
	{IMX_8BIT, 0x0344, 0x00},
	{IMX_8BIT, 0x0345, 0x4E},
	{IMX_8BIT, 0x0346, 0x01},
	{IMX_8BIT, 0x0347, 0x9C},
	{IMX_8BIT, 0x0348, 0x10},
	{IMX_8BIT, 0x0349, 0x21},
	{IMX_8BIT, 0x034A, 0x0A},
	{IMX_8BIT, 0x034B, 0x97},
	{IMX_8BIT, 0x034C, 0x05},
	{IMX_8BIT, 0x034D, 0x10},
	{IMX_8BIT, 0x034E, 0x02},
	{IMX_8BIT, 0x034F, 0xE0},
	{IMX_8BIT, 0x0350, 0x00},
	{IMX_8BIT, 0x0351, 0x00},
	{IMX_8BIT, 0x0352, 0x00},
	{IMX_8BIT, 0x0353, 0x00},
	{IMX_8BIT, 0x0354, 0x07},
	{IMX_8BIT, 0x0355, 0xEA},
	{IMX_8BIT, 0x0356, 0x04},
	{IMX_8BIT, 0x0357, 0x7E},
	{IMX_8BIT, 0x301D, 0x30},
	{IMX_8BIT, 0x3310, 0x05},
	{IMX_8BIT, 0x3311, 0x10},
	{IMX_8BIT, 0x3312, 0x02},
	{IMX_8BIT, 0x3313, 0xE0},
	{IMX_8BIT, 0x331C, 0x02},
	{IMX_8BIT, 0x331D, 0x26},
	{IMX_8BIT, 0x4084, 0x05},
	{IMX_8BIT, 0x4085, 0x10},
	{IMX_8BIT, 0x4086, 0x02},
	{IMX_8BIT, 0x4087, 0xE0},
	{IMX_8BIT, 0x4400, 0x00},
	/* Global Timing */
	{IMX_8BIT, 0x0830, 0x5F},
	{IMX_8BIT, 0x0831, 0x0C},
	{IMX_8BIT, 0x0832, 0x3F},
	{IMX_8BIT, 0x0833, 0x1F},
	{IMX_8BIT, 0x0834, 0x1F},
	{IMX_8BIT, 0x0835, 0x17},
	{IMX_8BIT, 0x0836, 0x67},
	{IMX_8BIT, 0x0837, 0x27},
	{IMX_8BIT, 0x0839, 0x1F},
	{IMX_8BIT, 0x083A, 0x17},
	{IMX_8BIT, 0x083B, 0x02},
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x0C},
	{IMX_8BIT, 0x0203, 0x18},
	/* Gain Setting */
	{IMX_8BIT, 0x0205, 0x00},
	{IMX_8BIT, 0x020E, 0x01},
	{IMX_8BIT, 0x020F, 0x00},
	{IMX_8BIT, 0x0210, 0x01},
	{IMX_8BIT, 0x0211, 0x00},
	{IMX_8BIT, 0x0212, 0x01},
	{IMX_8BIT, 0x0213, 0x00},
	{IMX_8BIT, 0x0214, 0x01},
	{IMX_8BIT, 0x0215, 0x00},
	/* HDR Setting */
	{IMX_8BIT, 0x0230, 0x00},
	{IMX_8BIT, 0x0231, 0x00},
	{IMX_8BIT, 0x0233, 0x00},
	{IMX_8BIT, 0x0234, 0x00},
	{IMX_8BIT, 0x0235, 0x40},
	{IMX_8BIT, 0x0238, 0x00},
	{IMX_8BIT, 0x0239, 0x04},
	{IMX_8BIT, 0x023B, 0x00},
	{IMX_8BIT, 0x023C, 0x01},
	{IMX_8BIT, 0x33B0, 0x04},
	{IMX_8BIT, 0x33B1, 0x00},
	{IMX_8BIT, 0x33B3, 0x00},
	{IMX_TOK_TERM, 0, 0},
};

static struct imx_reg const imx135_1M_4_3_30fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	{IMX_8BIT, 0x011E, 0x13},
	{IMX_8BIT, 0x011F, 0x33},
	{IMX_8BIT, 0x0301, 0x05},
	{IMX_8BIT, 0x0303, 0x01},
	{IMX_8BIT, 0x0305, 0x0B},
	{IMX_8BIT, 0x0309, 0x05},
	{IMX_8BIT, 0x030B, 0x02},
	{IMX_8BIT, 0x030C, 0x01},
	{IMX_8BIT, 0x030D, 0x83},
	{IMX_8BIT, 0x030E, 0x01},
	{IMX_8BIT, 0x3A06, 0x12},
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03},
	{IMX_8BIT, 0x0112, 0x0A},
	{IMX_8BIT, 0x0113, 0x0A},
	{IMX_8BIT, 0x0381, 0x01},
	{IMX_8BIT, 0x0383, 0x01},
	{IMX_8BIT, 0x0385, 0x01},
	{IMX_8BIT, 0x0387, 0x01},
	{IMX_8BIT, 0x0390, 0x01},
	{IMX_8BIT, 0x0391, 0x22},
	{IMX_8BIT, 0x0392, 0x00},
	{IMX_8BIT, 0x0401, 0x02},
	{IMX_8BIT, 0x0404, 0x00},
	{IMX_8BIT, 0x0405, 0x1F},
	{IMX_8BIT, 0x4082, 0x00},
	{IMX_8BIT, 0x4083, 0x00},
	{IMX_8BIT, 0x7006, 0x04},
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x07},
	{IMX_8BIT, 0x0341, 0x08},
	{IMX_8BIT, 0x0342, 0x12},
	{IMX_8BIT, 0x0343, 0x58},
	{IMX_8BIT, 0x0344, 0x00},
	{IMX_8BIT, 0x0345, 0x58},
	{IMX_8BIT, 0x0346, 0x00},
	{IMX_8BIT, 0x0347, 0x28},
	{IMX_8BIT, 0x0348, 0x10},
	{IMX_8BIT, 0x0349, 0x17},
	{IMX_8BIT, 0x034A, 0x0C},
	{IMX_8BIT, 0x034B, 0x07},
	{IMX_8BIT, 0x034C, 0x04},
	{IMX_8BIT, 0x034D, 0x10},
	{IMX_8BIT, 0x034E, 0x03},
	{IMX_8BIT, 0x034F, 0x10},
	{IMX_8BIT, 0x0350, 0x00},
	{IMX_8BIT, 0x0351, 0x00},
	{IMX_8BIT, 0x0352, 0x00},
	{IMX_8BIT, 0x0353, 0x00},
	{IMX_8BIT, 0x0354, 0x07},
	{IMX_8BIT, 0x0355, 0xE0},
	{IMX_8BIT, 0x0356, 0x05},
	{IMX_8BIT, 0x0357, 0xF0},
	{IMX_8BIT, 0x301D, 0x30},
	{IMX_8BIT, 0x3310, 0x04},
	{IMX_8BIT, 0x3311, 0x10},
	{IMX_8BIT, 0x3312, 0x03},
	{IMX_8BIT, 0x3313, 0x10},
	{IMX_8BIT, 0x331C, 0x02},
	{IMX_8BIT, 0x331D, 0x4E},
	{IMX_8BIT, 0x4084, 0x04},
	{IMX_8BIT, 0x4085, 0x10},
	{IMX_8BIT, 0x4086, 0x03},
	{IMX_8BIT, 0x4087, 0x10},
	{IMX_8BIT, 0x4400, 0x00},
	/* Global Timing */
	{IMX_8BIT, 0x0830, 0x5F},
	{IMX_8BIT, 0x0831, 0x0C},
	{IMX_8BIT, 0x0832, 0x3F},
	{IMX_8BIT, 0x0833, 0x1F},
	{IMX_8BIT, 0x0834, 0x1F},
	{IMX_8BIT, 0x0835, 0x17},
	{IMX_8BIT, 0x0836, 0x67},
	{IMX_8BIT, 0x0837, 0x27},
	{IMX_8BIT, 0x0839, 0x1F},
	{IMX_8BIT, 0x083A, 0x17},
	{IMX_8BIT, 0x083B, 0x02},
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x06},
	{IMX_8BIT, 0x0203, 0x02},
	/* Gain Setting */
	{IMX_8BIT, 0x0205, 0x00},
	{IMX_8BIT, 0x020E, 0x01},
	{IMX_8BIT, 0x020F, 0x00},
	{IMX_8BIT, 0x0210, 0x01},
	{IMX_8BIT, 0x0211, 0x00},
	{IMX_8BIT, 0x0212, 0x01},
	{IMX_8BIT, 0x0213, 0x00},
	{IMX_8BIT, 0x0214, 0x01},
	{IMX_8BIT, 0x0215, 0x00},
	/* HDR Setting */
	{IMX_8BIT, 0x0230, 0x00},
	{IMX_8BIT, 0x0231, 0x00},
	{IMX_8BIT, 0x0233, 0x00},
	{IMX_8BIT, 0x0234, 0x00},
	{IMX_8BIT, 0x0235, 0x40},
	{IMX_8BIT, 0x0238, 0x00},
	{IMX_8BIT, 0x0239, 0x04},
	{IMX_8BIT, 0x023B, 0x00},
	{IMX_8BIT, 0x023C, 0x01},
	{IMX_8BIT, 0x33B0, 0x04},
	{IMX_8BIT, 0x33B1, 0x00},
	{IMX_8BIT, 0x33B3, 0x00},
	{IMX_TOK_TERM, 0, 0},
};

static struct imx_reg const imx135_D1_30fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03},
	{IMX_8BIT, 0x0112, 0x0A},
	{IMX_8BIT, 0x0113, 0x0A},
	{IMX_8BIT, 0x0381, 0x01},
	{IMX_8BIT, 0x0383, 0x01},
	{IMX_8BIT, 0x0385, 0x01},
	{IMX_8BIT, 0x0387, 0x01},
	{IMX_8BIT, 0x0390, 0x01},
	{IMX_8BIT, 0x0391, 0x44},
	{IMX_8BIT, 0x0392, 0x00},
	{IMX_8BIT, 0x0401, 0x02},
	{IMX_8BIT, 0x0404, 0x00},
	{IMX_8BIT, 0x0405, 0x16},
	{IMX_8BIT, 0x4082, 0x00},
	{IMX_8BIT, 0x4083, 0x00},
	{IMX_8BIT, 0x4203, 0xFF},
	{IMX_8BIT, 0x7006, 0x04},
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x0A},
	{IMX_8BIT, 0x0341, 0x5A},
	{IMX_8BIT, 0x0342, 0x11},
	{IMX_8BIT, 0x0343, 0xDC},
	{IMX_8BIT, 0x0344, 0x00},
	{IMX_8BIT, 0x0345, 0x50},
	{IMX_8BIT, 0x0346, 0x00},
	{IMX_8BIT, 0x0347, 0xC4},
	{IMX_8BIT, 0x0348, 0x10},
	{IMX_8BIT, 0x0349, 0x1F},
	{IMX_8BIT, 0x034A, 0x0B},
	{IMX_8BIT, 0x034B, 0x6B},
	{IMX_8BIT, 0x034C, 0x02},
	{IMX_8BIT, 0x034D, 0xE0},
	{IMX_8BIT, 0x034E, 0x01},
	{IMX_8BIT, 0x034F, 0xF0},
	{IMX_8BIT, 0x0350, 0x00},
	{IMX_8BIT, 0x0351, 0x00},
	{IMX_8BIT, 0x0352, 0x00},
	{IMX_8BIT, 0x0353, 0x00},
	{IMX_8BIT, 0x0354, 0x03},
	{IMX_8BIT, 0x0355, 0xF4},
	{IMX_8BIT, 0x0356, 0x02},
	{IMX_8BIT, 0x0357, 0xAA},
	{IMX_8BIT, 0x301D, 0x30},
	{IMX_8BIT, 0x3310, 0x02},
	{IMX_8BIT, 0x3311, 0xE0},
	{IMX_8BIT, 0x3312, 0x01},
	{IMX_8BIT, 0x3313, 0xF0},
	{IMX_8BIT, 0x331C, 0x01},
	{IMX_8BIT, 0x331D, 0xFE},
	{IMX_8BIT, 0x4084, 0x02},
	{IMX_8BIT, 0x4085, 0xE0},
	{IMX_8BIT, 0x4086, 0x01},
	{IMX_8BIT, 0x4087, 0xF0},
	{IMX_8BIT, 0x4400, 0x00},
	/* Global Timing */
	{IMX_8BIT, 0x0830, 0x67},
	{IMX_8BIT, 0x0831, 0x27},
	{IMX_8BIT, 0x0832, 0x47},
	{IMX_8BIT, 0x0833, 0x27},
	{IMX_8BIT, 0x0834, 0x27},
	{IMX_8BIT, 0x0835, 0x1F},
	{IMX_8BIT, 0x0836, 0x87},
	{IMX_8BIT, 0x0837, 0x2F},
	{IMX_8BIT, 0x0839, 0x1F},
	{IMX_8BIT, 0x083A, 0x17},
	{IMX_8BIT, 0x083B, 0x02},
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x0A},
	{IMX_8BIT, 0x0203, 0x56},
	{IMX_TOK_TERM, 0, 0},
};

static struct imx_reg const imx135_VGA_30fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	{IMX_8BIT, 0x011E, 0x13},
	{IMX_8BIT, 0x011F, 0x33},
	{IMX_8BIT, 0x0301, 0x05},
	{IMX_8BIT, 0x0303, 0x01},
	{IMX_8BIT, 0x0305, 0x0B},
	{IMX_8BIT, 0x0309, 0x05},
	{IMX_8BIT, 0x030B, 0x02},
	{IMX_8BIT, 0x030C, 0x01},
	{IMX_8BIT, 0x030D, 0x83},
	{IMX_8BIT, 0x030E, 0x01},
	{IMX_8BIT, 0x3A06, 0x12},
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03},
	{IMX_8BIT, 0x0112, 0x0A},
	{IMX_8BIT, 0x0113, 0x0A},
	{IMX_8BIT, 0x0381, 0x01},
	{IMX_8BIT, 0x0383, 0x01},
	{IMX_8BIT, 0x0385, 0x01},
	{IMX_8BIT, 0x0387, 0x01},
	{IMX_8BIT, 0x0390, 0x01},
	{IMX_8BIT, 0x0391, 0x44},
	{IMX_8BIT, 0x0392, 0x00},
	{IMX_8BIT, 0x0401, 0x02},
	{IMX_8BIT, 0x0404, 0x00},
	{IMX_8BIT, 0x0405, 0x19},
	{IMX_8BIT, 0x4082, 0x00},
	{IMX_8BIT, 0x4083, 0x00},
	{IMX_8BIT, 0x7006, 0x04},
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x03},
	{IMX_8BIT, 0x0341, 0x1E},
	{IMX_8BIT, 0x0342, 0x11},
	{IMX_8BIT, 0x0343, 0xDC},
	{IMX_8BIT, 0x0344, 0x00},
	{IMX_8BIT, 0x0345, 0x34},
	{IMX_8BIT, 0x0346, 0x00},
	{IMX_8BIT, 0x0347, 0x08},
	{IMX_8BIT, 0x0348, 0x10},
	{IMX_8BIT, 0x0349, 0x3B},
	{IMX_8BIT, 0x034A, 0x0C},
	{IMX_8BIT, 0x034B, 0x27},
	{IMX_8BIT, 0x034C, 0x02},
	{IMX_8BIT, 0x034D, 0x90},
	{IMX_8BIT, 0x034E, 0x01},
	{IMX_8BIT, 0x034F, 0xF0},
	{IMX_8BIT, 0x0350, 0x00},
	{IMX_8BIT, 0x0351, 0x00},
	{IMX_8BIT, 0x0352, 0x00},
	{IMX_8BIT, 0x0353, 0x00},
	{IMX_8BIT, 0x0354, 0x04},
	{IMX_8BIT, 0x0355, 0x02},
	{IMX_8BIT, 0x0356, 0x03},
	{IMX_8BIT, 0x0357, 0x08},
	{IMX_8BIT, 0x301D, 0x30},
	{IMX_8BIT, 0x3310, 0x02},
	{IMX_8BIT, 0x3311, 0x90},
	{IMX_8BIT, 0x3312, 0x01},
	{IMX_8BIT, 0x3313, 0xF0},
	{IMX_8BIT, 0x331C, 0x01},
	{IMX_8BIT, 0x331D, 0xDC},
	{IMX_8BIT, 0x4084, 0x02},
	{IMX_8BIT, 0x4085, 0x90},
	{IMX_8BIT, 0x4086, 0x01},
	{IMX_8BIT, 0x4087, 0xF0},
	{IMX_8BIT, 0x4400, 0x00},
	/* Global Timing */
	{IMX_8BIT, 0x0830, 0x5F},
	{IMX_8BIT, 0x0831, 0x0C},
	{IMX_8BIT, 0x0832, 0x3F},
	{IMX_8BIT, 0x0833, 0x1F},
	{IMX_8BIT, 0x0834, 0x1F},
	{IMX_8BIT, 0x0835, 0x17},
	{IMX_8BIT, 0x0836, 0x67},
	{IMX_8BIT, 0x0837, 0x27},
	{IMX_8BIT, 0x0839, 0x1F},
	{IMX_8BIT, 0x083A, 0x17},
	{IMX_8BIT, 0x083B, 0x02},
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x03},
	{IMX_8BIT, 0x0203, 0x1A},
	/* Gain Setting */
	{ IMX_8BIT, 0x0205, 0x00},
	{ IMX_8BIT, 0x020E, 0x01},
	{ IMX_8BIT, 0x020F, 0x00},
	{ IMX_8BIT, 0x0210, 0x01},
	{ IMX_8BIT, 0x0211, 0x00},
	{ IMX_8BIT, 0x0212, 0x01},
	{ IMX_8BIT, 0x0213, 0x00},
	{ IMX_8BIT, 0x0214, 0x01},
	{ IMX_8BIT, 0x0215, 0x00},
	/* HDR Setting */
	{ IMX_8BIT, 0x0230, 0x00},
	{ IMX_8BIT, 0x0231, 0x00},
	{ IMX_8BIT, 0x0233, 0x00},
	{ IMX_8BIT, 0x0234, 0x00},
	{ IMX_8BIT, 0x0235, 0x40},
	{ IMX_8BIT, 0x0238, 0x00},
	{ IMX_8BIT, 0x0239, 0x04},
	{ IMX_8BIT, 0x023B, 0x00},
	{ IMX_8BIT, 0x023C, 0x01},
	{ IMX_8BIT, 0x33B0, 0x04},
	{ IMX_8BIT, 0x33B1, 0x00},
	{ IMX_8BIT, 0x33B3, 0x00},
	{IMX_TOK_TERM, 0, 0},
};

static struct imx_reg const imx135_CIF_dvs_30fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03},
	{IMX_8BIT, 0x0112, 0x0A},
	{IMX_8BIT, 0x0113, 0x0A},
	{IMX_8BIT, 0x0381, 0x01},
	{IMX_8BIT, 0x0383, 0x01},
	{IMX_8BIT, 0x0385, 0x01},
	{IMX_8BIT, 0x0387, 0x01},
	{IMX_8BIT, 0x0390, 0x01},
	{IMX_8BIT, 0x0391, 0x44},
	{IMX_8BIT, 0x0392, 0x00},
	{IMX_8BIT, 0x0401, 0x02},
	{IMX_8BIT, 0x0404, 0x00},
	{IMX_8BIT, 0x0405, 0x29},
	{IMX_8BIT, 0x4082, 0x00},
	{IMX_8BIT, 0x4083, 0x00},
	{IMX_8BIT, 0x4203, 0xFF},
	{IMX_8BIT, 0x7006, 0x04},
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x0A},
	{IMX_8BIT, 0x0341, 0x5A},
	{IMX_8BIT, 0x0342, 0x11},
	{IMX_8BIT, 0x0343, 0xDC},
	{IMX_8BIT, 0x0344, 0x00},
	{IMX_8BIT, 0x0345, 0xD8},
	{IMX_8BIT, 0x0346, 0x00},
	{IMX_8BIT, 0x0347, 0x00},
	{IMX_8BIT, 0x0348, 0x0F},
	{IMX_8BIT, 0x0349, 0x97},
	{IMX_8BIT, 0x034A, 0x0C},
	{IMX_8BIT, 0x034B, 0x2F},
	{IMX_8BIT, 0x034C, 0x01},
	{IMX_8BIT, 0x034D, 0x70},
	{IMX_8BIT, 0x034E, 0x01},
	{IMX_8BIT, 0x034F, 0x30},
	{IMX_8BIT, 0x0350, 0x00},
	{IMX_8BIT, 0x0351, 0x00},
	{IMX_8BIT, 0x0352, 0x00},
	{IMX_8BIT, 0x0353, 0x00},
	{IMX_8BIT, 0x0354, 0x03},
	{IMX_8BIT, 0x0355, 0xB0},
	{IMX_8BIT, 0x0356, 0x03},
	{IMX_8BIT, 0x0357, 0x0C},
	{IMX_8BIT, 0x301D, 0x30},
	{IMX_8BIT, 0x3310, 0x01},
	{IMX_8BIT, 0x3311, 0x70},
	{IMX_8BIT, 0x3312, 0x01},
	{IMX_8BIT, 0x3313, 0x30},
	{IMX_8BIT, 0x331C, 0x01},
	{IMX_8BIT, 0x331D, 0x4A},
	{IMX_8BIT, 0x4084, 0x01},
	{IMX_8BIT, 0x4085, 0x70},
	{IMX_8BIT, 0x4086, 0x01},
	{IMX_8BIT, 0x4087, 0x30},
	{IMX_8BIT, 0x4400, 0x00},
	/* Global Timing */
	{IMX_8BIT, 0x0830, 0x67},
	{IMX_8BIT, 0x0831, 0x27},
	{IMX_8BIT, 0x0832, 0x47},
	{IMX_8BIT, 0x0833, 0x27},
	{IMX_8BIT, 0x0834, 0x27},
	{IMX_8BIT, 0x0835, 0x1F},
	{IMX_8BIT, 0x0836, 0x87},
	{IMX_8BIT, 0x0837, 0x2F},
	{IMX_8BIT, 0x0839, 0x1F},
	{IMX_8BIT, 0x083A, 0x17},
	{IMX_8BIT, 0x083B, 0x02},
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x0A},
	{IMX_8BIT, 0x0203, 0x56},
	{IMX_TOK_TERM, 0, 0},
};

static struct imx_reg const imx135_QVGA_30fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	{IMX_8BIT, 0x011E, 0x13},
	{IMX_8BIT, 0x011F, 0x33},
	{IMX_8BIT, 0x0301, 0x05},
	{IMX_8BIT, 0x0303, 0x01},
	{IMX_8BIT, 0x0305, 0x0B},
	{IMX_8BIT, 0x0309, 0x05},
	{IMX_8BIT, 0x030B, 0x02},
	{IMX_8BIT, 0x030C, 0x01},
	{IMX_8BIT, 0x030D, 0x83},
	{IMX_8BIT, 0x030E, 0x01},
	{IMX_8BIT, 0x3A06, 0x12},
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03},
	{IMX_8BIT, 0x0112, 0x0A},
	{IMX_8BIT, 0x0113, 0x0A},
	{IMX_8BIT, 0x0381, 0x01},
	{IMX_8BIT, 0x0383, 0x01},
	{IMX_8BIT, 0x0385, 0x01},
	{IMX_8BIT, 0x0387, 0x01},
	{IMX_8BIT, 0x0390, 0x01},
	{IMX_8BIT, 0x0391, 0x44},
	{IMX_8BIT, 0x0392, 0x00},
	{IMX_8BIT, 0x0401, 0x02},
	{IMX_8BIT, 0x0404, 0x00},
	{IMX_8BIT, 0x0405, 0x28},
	{IMX_8BIT, 0x4082, 0x00},
	{IMX_8BIT, 0x4083, 0x00},
	{IMX_8BIT, 0x4203, 0x00},
	{IMX_8BIT, 0x7006, 0x04},
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x0A},
	{IMX_8BIT, 0x0341, 0x5A},
	{IMX_8BIT, 0x0342, 0x11},
	{IMX_8BIT, 0x0343, 0xDC},
	{IMX_8BIT, 0x0344, 0x00},
	{IMX_8BIT, 0x0345, 0x40},
	{IMX_8BIT, 0x0346, 0x00},
	{IMX_8BIT, 0x0347, 0x14},
	{IMX_8BIT, 0x0348, 0x10},
	{IMX_8BIT, 0x0349, 0x2F},
	{IMX_8BIT, 0x034A, 0x0C},
	{IMX_8BIT, 0x034B, 0x1B},
	{IMX_8BIT, 0x034C, 0x01},
	{IMX_8BIT, 0x034D, 0x50},
	{IMX_8BIT, 0x034E, 0x01},
	{IMX_8BIT, 0x034F, 0x00},
	{IMX_8BIT, 0x0350, 0x00},
	{IMX_8BIT, 0x0351, 0x00},
	{IMX_8BIT, 0x0352, 0x00},
	{IMX_8BIT, 0x0353, 0x00},
	{IMX_8BIT, 0x0354, 0x03},
	{IMX_8BIT, 0x0355, 0xFC},
	{IMX_8BIT, 0x0356, 0x03},
	{IMX_8BIT, 0x0357, 0x02},
	{IMX_8BIT, 0x301D, 0x30},
	{IMX_8BIT, 0x3310, 0x01},
	{IMX_8BIT, 0x3311, 0x98},
	{IMX_8BIT, 0x3312, 0x01},
	{IMX_8BIT, 0x3313, 0x34},
	{IMX_8BIT, 0x331C, 0x01},
	{IMX_8BIT, 0x331D, 0x68},
	{IMX_8BIT, 0x4084, 0x01},
	{IMX_8BIT, 0x4085, 0x98},
	{IMX_8BIT, 0x4086, 0x01},
	{IMX_8BIT, 0x4087, 0x34},
	{IMX_8BIT, 0x4400, 0x00},
	/* Global Timing */
	{IMX_8BIT, 0x0830, 0x5F},
	{IMX_8BIT, 0x0831, 0x0C},
	{IMX_8BIT, 0x0832, 0x3F},
	{IMX_8BIT, 0x0833, 0x1F},
	{IMX_8BIT, 0x0834, 0x1F},
	{IMX_8BIT, 0x0835, 0x17},
	{IMX_8BIT, 0x0836, 0x67},
	{IMX_8BIT, 0x0837, 0x27},
	{IMX_8BIT, 0x0839, 0x1F},
	{IMX_8BIT, 0x083A, 0x17},
	{IMX_8BIT, 0x083B, 0x02},
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x07},
	{IMX_8BIT, 0x0203, 0xA4},
	/* Gain Setting */
	{ IMX_8BIT, 0x0205, 0x00},
	{ IMX_8BIT, 0x020E, 0x01},
	{ IMX_8BIT, 0x020F, 0x00},
	{ IMX_8BIT, 0x0210, 0x01},
	{ IMX_8BIT, 0x0211, 0x00},
	{ IMX_8BIT, 0x0212, 0x01},
	{ IMX_8BIT, 0x0213, 0x00},
	{ IMX_8BIT, 0x0214, 0x01},
	{ IMX_8BIT, 0x0215, 0x00},
	/* HDR Setting */
	{ IMX_8BIT, 0x0230, 0x00},
	{ IMX_8BIT, 0x0231, 0x00},
	{ IMX_8BIT, 0x0233, 0x00},
	{ IMX_8BIT, 0x0234, 0x00},
	{ IMX_8BIT, 0x0235, 0x40},
	{ IMX_8BIT, 0x0238, 0x00},
	{ IMX_8BIT, 0x0239, 0x04},
	{ IMX_8BIT, 0x023B, 0x00},
	{ IMX_8BIT, 0x023C, 0x01},
	{ IMX_8BIT, 0x33B0, 0x04},
	{ IMX_8BIT, 0x33B1, 0x00},
	{ IMX_8BIT, 0x33B3, 0x00},
	{IMX_TOK_TERM, 0, 0},
};

static struct imx_reg const imx135_QCIF_30fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03},
	{IMX_8BIT, 0x0112, 0x0A},
	{IMX_8BIT, 0x0113, 0x0A},
	{IMX_8BIT, 0x0381, 0x01},
	{IMX_8BIT, 0x0383, 0x01},
	{IMX_8BIT, 0x0385, 0x01},
	{IMX_8BIT, 0x0387, 0x01},
	{IMX_8BIT, 0x0390, 0x01},
	{IMX_8BIT, 0x0391, 0x44},
	{IMX_8BIT, 0x0392, 0x00},
	{IMX_8BIT, 0x0401, 0x02},
	{IMX_8BIT, 0x0404, 0x00},
	{IMX_8BIT, 0x0405, 0x4E},
	{IMX_8BIT, 0x4082, 0x00},
	{IMX_8BIT, 0x4083, 0x00},
	{IMX_8BIT, 0x4203, 0xFF},
	{IMX_8BIT, 0x7006, 0x04},
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x0A},
	{IMX_8BIT, 0x0341, 0x5A},
	{IMX_8BIT, 0x0342, 0x11},
	{IMX_8BIT, 0x0343, 0xDC},
	{IMX_8BIT, 0x0344, 0x00},
	{IMX_8BIT, 0x0345, 0xE8},
	{IMX_8BIT, 0x0346, 0x00},
	{IMX_8BIT, 0x0347, 0x00},
	{IMX_8BIT, 0x0348, 0x0F},
	{IMX_8BIT, 0x0349, 0x87},
	{IMX_8BIT, 0x034A, 0x0C},
	{IMX_8BIT, 0x034B, 0x2F},
	{IMX_8BIT, 0x034C, 0x00},
	{IMX_8BIT, 0x034D, 0xC0},
	{IMX_8BIT, 0x034E, 0x00},
	{IMX_8BIT, 0x034F, 0xA0},
	{IMX_8BIT, 0x0350, 0x00},
	{IMX_8BIT, 0x0351, 0x00},
	{IMX_8BIT, 0x0352, 0x00},
	{IMX_8BIT, 0x0353, 0x00},
	{IMX_8BIT, 0x0354, 0x03},
	{IMX_8BIT, 0x0355, 0xA8},
	{IMX_8BIT, 0x0356, 0x03},
	{IMX_8BIT, 0x0357, 0x0C},
	{IMX_8BIT, 0x301D, 0x30},
	{IMX_8BIT, 0x3310, 0x00},
	{IMX_8BIT, 0x3311, 0xC0},
	{IMX_8BIT, 0x3312, 0x00},
	{IMX_8BIT, 0x3313, 0xA0},
	{IMX_8BIT, 0x331C, 0x00},
	{IMX_8BIT, 0x331D, 0xC0},
	{IMX_8BIT, 0x4084, 0x00},
	{IMX_8BIT, 0x4085, 0xC0},
	{IMX_8BIT, 0x4086, 0x00},
	{IMX_8BIT, 0x4087, 0xA0},
	{IMX_8BIT, 0x4400, 0x00},
	/* Global Timing */
	{IMX_8BIT, 0x0830, 0x67},
	{IMX_8BIT, 0x0831, 0x27},
	{IMX_8BIT, 0x0832, 0x47},
	{IMX_8BIT, 0x0833, 0x27},
	{IMX_8BIT, 0x0834, 0x27},
	{IMX_8BIT, 0x0835, 0x1F},
	{IMX_8BIT, 0x0836, 0x87},
	{IMX_8BIT, 0x0837, 0x2F},
	{IMX_8BIT, 0x0839, 0x1F},
	{IMX_8BIT, 0x083A, 0x17},
	{IMX_8BIT, 0x083B, 0x02},
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x0A},
	{IMX_8BIT, 0x0203, 0x56},
	{IMX_TOK_TERM, 0, 0},
};

static struct imx_reg const imx_init_settings_bak[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	/* Basic Config */
	{IMX_8BIT, 0x0101, 0x00},
	{IMX_8BIT, 0x0105, 0x01},
	{IMX_8BIT, 0x0110, 0x00},
	{IMX_8BIT, 0x0220, 0x01},
	{IMX_8BIT, 0x3302, 0x11},
	{IMX_8BIT, 0x3833, 0x20},
	{IMX_8BIT, 0x3893, 0x00},
	{IMX_8BIT, 0x3906, 0x08},
	{IMX_8BIT, 0x3907, 0x01},
	{IMX_8BIT, 0x391B, 0x01},
	{IMX_8BIT, 0x3C09, 0x01},
	{IMX_8BIT, 0x600A, 0x00},
	/* Analog Sensor Optimization */
	{IMX_8BIT, 0x320A, 0x01},
	{IMX_8BIT, 0x320D, 0x10},
	{IMX_8BIT, 0x3216, 0x2E},
	{IMX_8BIT, 0x322C, 0x02},
	{IMX_8BIT, 0x3409, 0x0C},
	{IMX_8BIT, 0x340C, 0x2D},
	{IMX_8BIT, 0x3411, 0x39},
	{IMX_8BIT, 0x3414, 0x1E},
	{IMX_8BIT, 0x3427, 0x04},
	{IMX_8BIT, 0x3480, 0x1E},
	{IMX_8BIT, 0x3484, 0x1E},
	{IMX_8BIT, 0x3488, 0x1E},
	{IMX_8BIT, 0x348C, 0x1E},
	{IMX_8BIT, 0x3490, 0x1E},
	{IMX_8BIT, 0x3494, 0x1E},
	{IMX_8BIT, 0x3511, 0x8F},
	{IMX_8BIT, 0x364F, 0x2D},
	GROUPED_PARAMETER_HOLD_DISABLE,
	{IMX_TOK_TERM, 0, 0},
};

/********************** settings for imx - reference *********************/
static struct imx_reg const imx_init_settings[] = {
	/* basic settings */
	GROUPED_PARAMETER_HOLD_ENABLE,
	{ IMX_8BIT, 0x0220, 0x01},
	{ IMX_8BIT, 0x3008, 0xB0},
	{ IMX_8BIT, 0x320A, 0x01},
	{ IMX_8BIT, 0x320D, 0x10},
	{ IMX_8BIT, 0x3216, 0x2E},
	{ IMX_8BIT, 0x3230, 0x0A},
	{ IMX_8BIT, 0x3228, 0x05},
	{ IMX_8BIT, 0x3229, 0x02},
	{ IMX_8BIT, 0x322C, 0x02},
	{ IMX_8BIT, 0x3302, 0x10},
	{ IMX_8BIT, 0x3390, 0x45},
	{ IMX_8BIT, 0x3409, 0x0C},
	{ IMX_8BIT, 0x340B, 0xF5},
	{ IMX_8BIT, 0x340C, 0x2D},
	{ IMX_8BIT, 0x3412, 0x41},
	{ IMX_8BIT, 0x3413, 0xAD},
	{ IMX_8BIT, 0x3414, 0x1E},
	{ IMX_8BIT, 0x3427, 0x04},
	{ IMX_8BIT, 0x3480, 0x1E},
	{ IMX_8BIT, 0x3484, 0x1E},
	{ IMX_8BIT, 0x3488, 0x1E},
	{ IMX_8BIT, 0x348C, 0x1E},
	{ IMX_8BIT, 0x3490, 0x1E},
	{ IMX_8BIT, 0x3494, 0x1E},
	{ IMX_8BIT, 0x349C, 0x38},
	{ IMX_8BIT, 0x34A3, 0x38},
	{ IMX_8BIT, 0x3511, 0x8F},
	{ IMX_8BIT, 0x3518, 0x00},
	{ IMX_8BIT, 0x3519, 0x94},
	{ IMX_8BIT, 0x3833, 0x20},
	{ IMX_8BIT, 0x3893, 0x01},
	{ IMX_8BIT, 0x38C2, 0x08},
	{ IMX_8BIT, 0x38C3, 0x08},
	{ IMX_8BIT, 0x3C09, 0x01},
	{ IMX_8BIT, 0x4000, 0x0E},
	{ IMX_8BIT, 0x4300, 0x00},
	{ IMX_8BIT, 0x4316, 0x12},
	{ IMX_8BIT, 0x4317, 0x22},
	{ IMX_8BIT, 0x4318, 0x00},
	{ IMX_8BIT, 0x4319, 0x00},
	{ IMX_8BIT, 0x431A, 0x00},
	{ IMX_8BIT, 0x4324, 0x03},
	{ IMX_8BIT, 0x4325, 0x20},
	{ IMX_8BIT, 0x4326, 0x03},
	{ IMX_8BIT, 0x4327, 0x84},
	{ IMX_8BIT, 0x4328, 0x03},
	{ IMX_8BIT, 0x4329, 0x20},
	{ IMX_8BIT, 0x432A, 0x03},
	{ IMX_8BIT, 0x432B, 0x84},
	{ IMX_8BIT, 0x432C, 0x01},
	{ IMX_8BIT, 0x4401, 0x3F},
	{ IMX_8BIT, 0x4402, 0xFF},
	{ IMX_8BIT, 0x4412, 0x3F},
	{ IMX_8BIT, 0x4413, 0xFF},
	{ IMX_8BIT, 0x441D, 0x28},
	{ IMX_8BIT, 0x4444, 0x00},
	{ IMX_8BIT, 0x4445, 0x00},
	{ IMX_8BIT, 0x4446, 0x3F},
	{ IMX_8BIT, 0x4447, 0xFF},
	{ IMX_8BIT, 0x4452, 0x00},
	{ IMX_8BIT, 0x4453, 0xA0},
	{ IMX_8BIT, 0x4454, 0x08},
	{ IMX_8BIT, 0x4455, 0x00},
	{ IMX_8BIT, 0x4458, 0x18},
	{ IMX_8BIT, 0x4459, 0x18},
	{ IMX_8BIT, 0x445A, 0x3F},
	{ IMX_8BIT, 0x445B, 0x3A},
	{ IMX_8BIT, 0x4462, 0x00},
	{ IMX_8BIT, 0x4463, 0x00},
	{ IMX_8BIT, 0x4464, 0x00},
	{ IMX_8BIT, 0x4465, 0x00},
	{ IMX_8BIT, 0x446E, 0x01},
	{ IMX_8BIT, 0x4500, 0x1F},
	{ IMX_8BIT, 0x600a, 0x00},
	{ IMX_8BIT, 0x380a, 0x00},
	{ IMX_8BIT, 0x380b, 0x00},
	{ IMX_8BIT, 0x4103, 0x00},
	{ IMX_8BIT, 0x4243, 0x9a},
	{ IMX_8BIT, 0x4330, 0x01},
	{ IMX_8BIT, 0x4331, 0x90},
	{ IMX_8BIT, 0x4332, 0x02},
	{ IMX_8BIT, 0x4333, 0x58},
	{ IMX_8BIT, 0x4334, 0x03},
	{ IMX_8BIT, 0x4335, 0x20},
	{ IMX_8BIT, 0x4336, 0x03},
	{ IMX_8BIT, 0x4337, 0x84},
	{ IMX_8BIT, 0x433C, 0x01},
	{ IMX_8BIT, 0x4340, 0x02},
	{ IMX_8BIT, 0x4341, 0x58},
	{ IMX_8BIT, 0x4342, 0x03},
	{ IMX_8BIT, 0x4343, 0x52},
	{ IMX_8BIT, 0x4364, 0x0b},
	{ IMX_8BIT, 0x4368, 0x00},
	{ IMX_8BIT, 0x4369, 0x0f},
	{ IMX_8BIT, 0x436a, 0x03},
	{ IMX_8BIT, 0x436b, 0xa8},
	{ IMX_8BIT, 0x436c, 0x00},
	{ IMX_8BIT, 0x436d, 0x00},
	{ IMX_8BIT, 0x436e, 0x00},
	{ IMX_8BIT, 0x436f, 0x06},
	{ IMX_8BIT, 0x4281, 0x21},
	{ IMX_8BIT, 0x4282, 0x18},
	{ IMX_8BIT, 0x4283, 0x04},
	{ IMX_8BIT, 0x4284, 0x08},
	{ IMX_8BIT, 0x4287, 0x7f},
	{ IMX_8BIT, 0x4288, 0x08},
	{ IMX_8BIT, 0x428b, 0x7f},
	{ IMX_8BIT, 0x428c, 0x08},
	{ IMX_8BIT, 0x428f, 0x7f},
	{ IMX_8BIT, 0x4297, 0x00},
	{ IMX_8BIT, 0x4298, 0x7E},
	{ IMX_8BIT, 0x4299, 0x7E},
	{ IMX_8BIT, 0x429A, 0x7E},
	{ IMX_8BIT, 0x42A4, 0xFB},
	{ IMX_8BIT, 0x42A5, 0x7E},
	{ IMX_8BIT, 0x42A6, 0xDF},
	{ IMX_8BIT, 0x42A7, 0xB7},
	{ IMX_8BIT, 0x42AF, 0x03},
	{ IMX_8BIT, 0x4207, 0x03},
	{ IMX_8BIT, 0x4216, 0x08},
	{ IMX_8BIT, 0x4217, 0x08},
	{ IMX_8BIT, 0x4218, 0x00},
	{ IMX_8BIT, 0x421B, 0x20},
	{ IMX_8BIT, 0x421F, 0x04},
	{ IMX_8BIT, 0x4222, 0x02},
	{ IMX_8BIT, 0x4223, 0x22},
	{ IMX_8BIT, 0x422E, 0x54},
	{ IMX_8BIT, 0x422F, 0xFB},
	{ IMX_8BIT, 0x4230, 0xFF},
	{ IMX_8BIT, 0x4231, 0xFE},
	{ IMX_8BIT, 0x4232, 0xFF},
	{ IMX_8BIT, 0x4235, 0x58},
	{ IMX_8BIT, 0x4236, 0xF7},
	{ IMX_8BIT, 0x4237, 0xFD},
	{ IMX_8BIT, 0x4239, 0x4E},
	{ IMX_8BIT, 0x423A, 0xFC},
	{ IMX_8BIT, 0x423B, 0xFD},
	{ IMX_8BIT, 0x4300, 0x00},
	{ IMX_8BIT, 0x4316, 0x12},
	{ IMX_8BIT, 0x4317, 0x22},
	{ IMX_8BIT, 0x4318, 0x00},
	{ IMX_8BIT, 0x4319, 0x00},
	{ IMX_8BIT, 0x431A, 0x00},
	{ IMX_8BIT, 0x4324, 0x03},
	{ IMX_8BIT, 0x4325, 0x20},
	{ IMX_8BIT, 0x4326, 0x03},
	{ IMX_8BIT, 0x4327, 0x84},
	{ IMX_8BIT, 0x4328, 0x03},
	{ IMX_8BIT, 0x4329, 0x20},
	{ IMX_8BIT, 0x432A, 0x03},
	{ IMX_8BIT, 0x432B, 0x20},
	{ IMX_8BIT, 0x432C, 0x01},
	{ IMX_8BIT, 0x432D, 0x01},
	{ IMX_8BIT, 0x4338, 0x02},
	{ IMX_8BIT, 0x4339, 0x00},
	{ IMX_8BIT, 0x433A, 0x00},
	{ IMX_8BIT, 0x433B, 0x02},
	{ IMX_8BIT, 0x435A, 0x03},
	{ IMX_8BIT, 0x435B, 0x84},
	{ IMX_8BIT, 0x435E, 0x01},
	{ IMX_8BIT, 0x435F, 0xFF},
	{ IMX_8BIT, 0x4360, 0x01},
	{ IMX_8BIT, 0x4361, 0xF4},
	{ IMX_8BIT, 0x4362, 0x03},
	{ IMX_8BIT, 0x4363, 0x84},
	{ IMX_8BIT, 0x437B, 0x01},
	{ IMX_8BIT, 0x4401, 0x3F},
	{ IMX_8BIT, 0x4402, 0xFF},
	{ IMX_8BIT, 0x4404, 0x13},
	{ IMX_8BIT, 0x4405, 0x26},
	{ IMX_8BIT, 0x4406, 0x07},
	{ IMX_8BIT, 0x4408, 0x20},
	{ IMX_8BIT, 0x4409, 0xE5},
	{ IMX_8BIT, 0x440A, 0xFB},
	{ IMX_8BIT, 0x440C, 0xF6},
	{ IMX_8BIT, 0x440D, 0xEA},
	{ IMX_8BIT, 0x440E, 0x20},
	{ IMX_8BIT, 0x4410, 0x00},
	{ IMX_8BIT, 0x4411, 0x00},
	{ IMX_8BIT, 0x4412, 0x3F},
	{ IMX_8BIT, 0x4413, 0xFF},
	{ IMX_8BIT, 0x4414, 0x1F},
	{ IMX_8BIT, 0x4415, 0xFF},
	{ IMX_8BIT, 0x4416, 0x20},
	{ IMX_8BIT, 0x4417, 0x00},
	{ IMX_8BIT, 0x4418, 0x1F},
	{ IMX_8BIT, 0x4419, 0xFF},
	{ IMX_8BIT, 0x441A, 0x20},
	{ IMX_8BIT, 0x441B, 0x00},
	{ IMX_8BIT, 0x441D, 0x40},
	{ IMX_8BIT, 0x441E, 0x1E},
	{ IMX_8BIT, 0x441F, 0x38},
	{ IMX_8BIT, 0x4420, 0x01},
	{ IMX_8BIT, 0x4444, 0x00},
	{ IMX_8BIT, 0x4445, 0x00},
	{ IMX_8BIT, 0x4446, 0x1D},
	{ IMX_8BIT, 0x4447, 0xF9},
	{ IMX_8BIT, 0x4452, 0x00},
	{ IMX_8BIT, 0x4453, 0xA0},
	{ IMX_8BIT, 0x4454, 0x08},
	{ IMX_8BIT, 0x4455, 0x00},
	{ IMX_8BIT, 0x4456, 0x0F},
	{ IMX_8BIT, 0x4457, 0xFF},
	{ IMX_8BIT, 0x4458, 0x18},
	{ IMX_8BIT, 0x4459, 0x18},
	{ IMX_8BIT, 0x445A, 0x3F},
	{ IMX_8BIT, 0x445B, 0x3A},
	{ IMX_8BIT, 0x445C, 0x00},
	{ IMX_8BIT, 0x445D, 0x28},
	{ IMX_8BIT, 0x445E, 0x01},
	{ IMX_8BIT, 0x445F, 0x90},
	{ IMX_8BIT, 0x4460, 0x00},
	{ IMX_8BIT, 0x4461, 0x60},
	{ IMX_8BIT, 0x4462, 0x00},
	{ IMX_8BIT, 0x4463, 0x00},
	{ IMX_8BIT, 0x4464, 0x00},
	{ IMX_8BIT, 0x4465, 0x00},
	{ IMX_8BIT, 0x446C, 0x00},
	{ IMX_8BIT, 0x446D, 0x00},
	{ IMX_8BIT, 0x446E, 0x00},
	{ IMX_8BIT, 0x452A, 0x02},
	{ IMX_8BIT, 0x0712, 0x01},
	{ IMX_8BIT, 0x0713, 0x00},
	{ IMX_8BIT, 0x0714, 0x01},
	{ IMX_8BIT, 0x0715, 0x00},
	{ IMX_8BIT, 0x0716, 0x01},
	{ IMX_8BIT, 0x0717, 0x00},
	{ IMX_8BIT, 0x0718, 0x01},
	{ IMX_8BIT, 0x0719, 0x00},
	{ IMX_8BIT, 0x4500, 0x1F },

	/*clock*/
	{ IMX_8BIT, 0x011E, 0x13},
	{ IMX_8BIT, 0x011F, 0x33},
	{ IMX_8BIT, 0x0301, 0x05},
	{ IMX_8BIT, 0x0303, 0x01},
	{ IMX_8BIT, 0x0305, 0x0B},
	{ IMX_8BIT, 0x0309, 0x05},
	{ IMX_8BIT, 0x030B, 0x02},
	{ IMX_8BIT, 0x030C, 0x01},
	{ IMX_8BIT, 0x030D, 0x83},
	{ IMX_8BIT, 0x030E, 0x01},
	{ IMX_8BIT, 0x3A06, 0x12},

	/* Gain Setting */
	{ IMX_8BIT, 0x0205, 0x00},
	{ IMX_8BIT, 0x020E, 0x01},
	{ IMX_8BIT, 0x020F, 0x00},
	{ IMX_8BIT, 0x0210, 0x02},
	{ IMX_8BIT, 0x0211, 0x00},
	{ IMX_8BIT, 0x0212, 0x02},
	{ IMX_8BIT, 0x0213, 0x00},
	{ IMX_8BIT, 0x0214, 0x01},
	{ IMX_8BIT, 0x0215, 0x00},
	/* HDR Setting */
	{ IMX_8BIT, 0x0230, 0x00},
	{ IMX_8BIT, 0x0231, 0x00},
	{ IMX_8BIT, 0x0233, 0x00},
	{ IMX_8BIT, 0x0234, 0x00},
	{ IMX_8BIT, 0x0235, 0x40},
	{ IMX_8BIT, 0x0238, 0x00},
	{ IMX_8BIT, 0x0239, 0x04},
	{ IMX_8BIT, 0x023B, 0x00},
	{ IMX_8BIT, 0x023C, 0x01},
	{ IMX_8BIT, 0x33B0, 0x04},
	{ IMX_8BIT, 0x33B1, 0x00},
	{ IMX_8BIT, 0x33B3, 0x00},
	{ IMX_8BIT, 0x33B4, 0x01},
	{ IMX_8BIT, 0x3800, 0x00},
	GROUPED_PARAMETER_HOLD_DISABLE,
	{ IMX_TOK_TERM, 0, 0}
};

/*****************************STILL********************************/

/* Capture1: H : 656 V : 496 */
static const struct imx_reg imx135_VGA_STILL_74fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03 },
	{IMX_8BIT, 0x0112, 0x0A },
	{IMX_8BIT, 0x0113, 0x0A },
	{IMX_8BIT, 0x0381, 0x01 },
	{IMX_8BIT, 0x0383, 0x01 },
	{IMX_8BIT, 0x0385, 0x01 },
	{IMX_8BIT, 0x0387, 0x01 },
	{IMX_8BIT, 0x0390, 0x01 },
	{IMX_8BIT, 0x0391, 0x44 },
	{IMX_8BIT, 0x0392, 0x00 },
	{IMX_8BIT, 0x0401, 0x02 },
	{IMX_8BIT, 0x0404, 0x00 },
	{IMX_8BIT, 0x0405, 0x19 },
	{IMX_8BIT, 0x4082, 0x00 },
	{IMX_8BIT, 0x4083, 0x00 },
	{IMX_8BIT, 0x7006, 0x04 },
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x03 },
	{IMX_8BIT, 0x0341, 0x1E },
	{IMX_8BIT, 0x0342, 0x11 },
	{IMX_8BIT, 0x0343, 0xDC },
	{IMX_8BIT, 0x0344, 0x00 },
	{IMX_8BIT, 0x0345, 0x34 },
	{IMX_8BIT, 0x0346, 0x00 },
	{IMX_8BIT, 0x0347, 0x08 },
	{IMX_8BIT, 0x0348, 0x10 },
	{IMX_8BIT, 0x0349, 0x3B },
	{IMX_8BIT, 0x034A, 0x0C },
	{IMX_8BIT, 0x034B, 0x27 },
	{IMX_8BIT, 0x034C, 0x02 },
	{IMX_8BIT, 0x034D, 0x90 },
	{IMX_8BIT, 0x034E, 0x01 },
	{IMX_8BIT, 0x034F, 0xF0 },
	{IMX_8BIT, 0x0350, 0x00 },
	{IMX_8BIT, 0x0351, 0x00 },
	{IMX_8BIT, 0x0352, 0x00 },
	{IMX_8BIT, 0x0353, 0x00 },
	{IMX_8BIT, 0x0354, 0x04 },
	{IMX_8BIT, 0x0355, 0x02 },
	{IMX_8BIT, 0x0356, 0x03 },
	{IMX_8BIT, 0x0357, 0x08 },
	{IMX_8BIT, 0x301D, 0x30 },
	{IMX_8BIT, 0x3310, 0x02 },
	{IMX_8BIT, 0x3311, 0x90 },
	{IMX_8BIT, 0x3312, 0x01 },
	{IMX_8BIT, 0x3313, 0xF0 },
	{IMX_8BIT, 0x331C, 0x01 },
	{IMX_8BIT, 0x331D, 0xDC },
	{IMX_8BIT, 0x4084, 0x02 },
	{IMX_8BIT, 0x4085, 0x90 },
	{IMX_8BIT, 0x4086, 0x01 },
	{IMX_8BIT, 0x4087, 0xF0 },
	{IMX_8BIT, 0x4400, 0x00 },
	/* Global Timing Setting */
	{IMX_8BIT, 0x0830, 0x5F },
	{IMX_8BIT, 0x0831, 0x1F },
	{IMX_8BIT, 0x0832, 0x3F },
	{IMX_8BIT, 0x0833, 0x1F },
	{IMX_8BIT, 0x0834, 0x1F },
	{IMX_8BIT, 0x0835, 0x17 },
	{IMX_8BIT, 0x0836, 0x67 },
	{IMX_8BIT, 0x0837, 0x27 },
	{IMX_8BIT, 0x0839, 0x1F },
	{IMX_8BIT, 0x083A, 0x17 },
	{IMX_8BIT, 0x083B, 0x02 },
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x03 },
	{IMX_8BIT, 0x0203, 0x1A },
	{ IMX_TOK_TERM, 0, 0}
};

/* Capture2: H : 1936 V : 1104 */
static const struct imx_reg imx135_1080P_STILL_48fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03 },
	{IMX_8BIT, 0x0112, 0x0A },
	{IMX_8BIT, 0x0113, 0x0A },
	{IMX_8BIT, 0x0381, 0x01 },
	{IMX_8BIT, 0x0383, 0x01 },
	{IMX_8BIT, 0x0385, 0x01 },
	{IMX_8BIT, 0x0387, 0x01 },
	{IMX_8BIT, 0x0390, 0x01 },
	{IMX_8BIT, 0x0391, 0x22 },
	{IMX_8BIT, 0x0392, 0x00 },
	{IMX_8BIT, 0x0401, 0x02 },
	{IMX_8BIT, 0x0404, 0x00 },
	{IMX_8BIT, 0x0405, 0x11 },
	{IMX_8BIT, 0x4082, 0x00 },
	{IMX_8BIT, 0x4083, 0x00 },
	{IMX_8BIT, 0x7006, 0x04 },
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x04 },
	{IMX_8BIT, 0x0341, 0xAC },
	{IMX_8BIT, 0x0342, 0x12 },
	{IMX_8BIT, 0x0343, 0x58 },
	{IMX_8BIT, 0x0344, 0x00 },
	{IMX_8BIT, 0x0345, 0x2E },
	{IMX_8BIT, 0x0346, 0x01 },
	{IMX_8BIT, 0x0347, 0x84 },
	{IMX_8BIT, 0x0348, 0x10 },
	{IMX_8BIT, 0x0349, 0x41 },
	{IMX_8BIT, 0x034A, 0x0A },
	{IMX_8BIT, 0x034B, 0xAF },
	{IMX_8BIT, 0x034C, 0x07 },
	{IMX_8BIT, 0x034D, 0x90 },
	{IMX_8BIT, 0x034E, 0x04 },
	{IMX_8BIT, 0x034F, 0x50 },
	{IMX_8BIT, 0x0350, 0x00 },
	{IMX_8BIT, 0x0351, 0x00 },
	{IMX_8BIT, 0x0352, 0x00 },
	{IMX_8BIT, 0x0353, 0x00 },
	{IMX_8BIT, 0x0354, 0x08 },
	{IMX_8BIT, 0x0355, 0x0A },
	{IMX_8BIT, 0x0356, 0x04 },
	{IMX_8BIT, 0x0357, 0x96 },
	{IMX_8BIT, 0x301D, 0x30 },
	{IMX_8BIT, 0x3310, 0x07 },
	{IMX_8BIT, 0x3311, 0x90 },
	{IMX_8BIT, 0x3312, 0x04 },
	{IMX_8BIT, 0x3313, 0x50 },
	{IMX_8BIT, 0x331C, 0x01 },
	{IMX_8BIT, 0x331D, 0x00 },
	{IMX_8BIT, 0x4084, 0x07 },
	{IMX_8BIT, 0x4085, 0x90 },
	{IMX_8BIT, 0x4086, 0x04 },
	{IMX_8BIT, 0x4087, 0x50 },
	{IMX_8BIT, 0x4400, 0x00 },
	/* Global Timing Setting */
	{IMX_8BIT, 0x0830, 0x5F },
	{IMX_8BIT, 0x0831, 0x1F },
	{IMX_8BIT, 0x0832, 0x3F },
	{IMX_8BIT, 0x0833, 0x1F },
	{IMX_8BIT, 0x0834, 0x1F },
	{IMX_8BIT, 0x0835, 0x17 },
	{IMX_8BIT, 0x0836, 0x67 },
	{IMX_8BIT, 0x0837, 0x27 },
	{IMX_8BIT, 0x0839, 0x1F },
	{IMX_8BIT, 0x083A, 0x17 },
	{IMX_8BIT, 0x083B, 0x02 },
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x04 },
	{IMX_8BIT, 0x0203, 0xA8 },
	{ IMX_TOK_TERM, 0, 0}
};

/* Capture3: H : 1040 V : 784 */
static const struct imx_reg imx135_1M_STILL_37fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03 },
	{IMX_8BIT, 0x0112, 0x0A },
	{IMX_8BIT, 0x0113, 0x0A },
	{IMX_8BIT, 0x0381, 0x01 },
	{IMX_8BIT, 0x0383, 0x01 },
	{IMX_8BIT, 0x0385, 0x01 },
	{IMX_8BIT, 0x0387, 0x01 },
	{IMX_8BIT, 0x0390, 0x01 },
	{IMX_8BIT, 0x0391, 0x22 },
	{IMX_8BIT, 0x0392, 0x00 },
	{IMX_8BIT, 0x0401, 0x02 },
	{IMX_8BIT, 0x0404, 0x00 },
	{IMX_8BIT, 0x0405, 0x1F },
	{IMX_8BIT, 0x4082, 0x00 },
	{IMX_8BIT, 0x4083, 0x00 },
	{IMX_8BIT, 0x7006, 0x04 },
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x06 },
	{IMX_8BIT, 0x0341, 0x06 },
	{IMX_8BIT, 0x0342, 0x12 },
	{IMX_8BIT, 0x0343, 0x58 },
	{IMX_8BIT, 0x0344, 0x00 },
	{IMX_8BIT, 0x0345, 0x58 },
	{IMX_8BIT, 0x0346, 0x00 },
	{IMX_8BIT, 0x0347, 0x28 },
	{IMX_8BIT, 0x0348, 0x10 },
	{IMX_8BIT, 0x0349, 0x17 },
	{IMX_8BIT, 0x034A, 0x0C },
	{IMX_8BIT, 0x034B, 0x07 },
	{IMX_8BIT, 0x034C, 0x04 },
	{IMX_8BIT, 0x034D, 0x10 },
	{IMX_8BIT, 0x034E, 0x03 },
	{IMX_8BIT, 0x034F, 0x10 },
	{IMX_8BIT, 0x0350, 0x00 },
	{IMX_8BIT, 0x0351, 0x00 },
	{IMX_8BIT, 0x0352, 0x00 },
	{IMX_8BIT, 0x0353, 0x00 },
	{IMX_8BIT, 0x0354, 0x07 },
	{IMX_8BIT, 0x0355, 0xE0 },
	{IMX_8BIT, 0x0356, 0x05 },
	{IMX_8BIT, 0x0357, 0xF0 },
	{IMX_8BIT, 0x301D, 0x30 },
	{IMX_8BIT, 0x3310, 0x04 },
	{IMX_8BIT, 0x3311, 0x10 },
	{IMX_8BIT, 0x3312, 0x03 },
	{IMX_8BIT, 0x3313, 0x10 },
	{IMX_8BIT, 0x331C, 0x02 },
	{IMX_8BIT, 0x331D, 0x4E },
	{IMX_8BIT, 0x4084, 0x04 },
	{IMX_8BIT, 0x4085, 0x10 },
	{IMX_8BIT, 0x4086, 0x03 },
	{IMX_8BIT, 0x4087, 0x10 },
	{IMX_8BIT, 0x4400, 0x00 },
	/* Global Timing Setting */
	{IMX_8BIT, 0x0830, 0x5F },
	{IMX_8BIT, 0x0831, 0x1F },
	{IMX_8BIT, 0x0832, 0x3F },
	{IMX_8BIT, 0x0833, 0x1F },
	{IMX_8BIT, 0x0834, 0x1F },
	{IMX_8BIT, 0x0835, 0x17 },
	{IMX_8BIT, 0x0836, 0x67 },
	{IMX_8BIT, 0x0837, 0x27 },
	{IMX_8BIT, 0x0839, 0x1F },
	{IMX_8BIT, 0x083A, 0x17 },
	{IMX_8BIT, 0x083B, 0x02 },
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x06 },
	{IMX_8BIT, 0x0203, 0x02 },
	{IMX_TOK_TERM, 0, 0}
};

/* Capture4: H : 1640 V : 1232 */
static const struct imx_reg imx135_2M_STILL_37fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03 },
	{IMX_8BIT, 0x0112, 0x0A },
	{IMX_8BIT, 0x0113, 0x0A },
	{IMX_8BIT, 0x0381, 0x01 },
	{IMX_8BIT, 0x0383, 0x01 },
	{IMX_8BIT, 0x0385, 0x01 },
	{IMX_8BIT, 0x0387, 0x01 },
	{IMX_8BIT, 0x0390, 0x01 },
	{IMX_8BIT, 0x0391, 0x22 },
	{IMX_8BIT, 0x0392, 0x00 },
	{IMX_8BIT, 0x0401, 0x02 },
	{IMX_8BIT, 0x0404, 0x00 },
	{IMX_8BIT, 0x0405, 0x14 },
	{IMX_8BIT, 0x4082, 0x00 },
	{IMX_8BIT, 0x4083, 0x00 },
	{IMX_8BIT, 0x7006, 0x04 },
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x06 },
	{IMX_8BIT, 0x0341, 0x1A },
	{IMX_8BIT, 0x0342, 0x12 },
	{IMX_8BIT, 0x0343, 0x58 },
	{IMX_8BIT, 0x0344, 0x00 },
	{IMX_8BIT, 0x0345, 0x36 },
	{IMX_8BIT, 0x0346, 0x00 },
	{IMX_8BIT, 0x0347, 0x14 },
	{IMX_8BIT, 0x0348, 0x10 },
	{IMX_8BIT, 0x0349, 0x39 },
	{IMX_8BIT, 0x034A, 0x0C },
	{IMX_8BIT, 0x034B, 0x1B },
	{IMX_8BIT, 0x034C, 0x06 },
	{IMX_8BIT, 0x034D, 0x68 },
	{IMX_8BIT, 0x034E, 0x04 },
	{IMX_8BIT, 0x034F, 0xD0 },
	{IMX_8BIT, 0x0350, 0x00 },
	{IMX_8BIT, 0x0351, 0x00 },
	{IMX_8BIT, 0x0352, 0x00 },
	{IMX_8BIT, 0x0353, 0x00 },
	{IMX_8BIT, 0x0354, 0x08 },
	{IMX_8BIT, 0x0355, 0x02 },
	{IMX_8BIT, 0x0356, 0x06 },
	{IMX_8BIT, 0x0357, 0x04 },
	{IMX_8BIT, 0x301D, 0x30 },
	{IMX_8BIT, 0x3310, 0x06 },
	{IMX_8BIT, 0x3311, 0x68 },
	{IMX_8BIT, 0x3312, 0x04 },
	{IMX_8BIT, 0x3313, 0xD0 },
	{IMX_8BIT, 0x331C, 0x01 },
	{IMX_8BIT, 0x331D, 0xC2 },
	{IMX_8BIT, 0x4084, 0x06 },
	{IMX_8BIT, 0x4085, 0x68 },
	{IMX_8BIT, 0x4086, 0x04 },
	{IMX_8BIT, 0x4087, 0xD0 },
	{IMX_8BIT, 0x4400, 0x00 },
	/* Global Timing Setting */
	{IMX_8BIT, 0x0830, 0x5F },
	{IMX_8BIT, 0x0831, 0x1F },
	{IMX_8BIT, 0x0832, 0x3F },
	{IMX_8BIT, 0x0833, 0x1F },
	{IMX_8BIT, 0x0834, 0x1F },
	{IMX_8BIT, 0x0835, 0x17 },
	{IMX_8BIT, 0x0836, 0x67 },
	{IMX_8BIT, 0x0837, 0x27 },
	{IMX_8BIT, 0x0839, 0x1F },
	{IMX_8BIT, 0x083A, 0x17 },
	{IMX_8BIT, 0x083B, 0x02 },
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x06 },
	{IMX_8BIT, 0x0203, 0x16 },
	{IMX_TOK_TERM, 0, 0}
};

/* Capture5: H : 2064 V : 1552 */
static const struct imx_reg imx135_3M_STILL_37fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03 },
	{IMX_8BIT, 0x0112, 0x0A },
	{IMX_8BIT, 0x0113, 0x0A },
	{IMX_8BIT, 0x0381, 0x01 },
	{IMX_8BIT, 0x0383, 0x01 },
	{IMX_8BIT, 0x0385, 0x01 },
	{IMX_8BIT, 0x0387, 0x01 },
	{IMX_8BIT, 0x0390, 0x01 },
	{IMX_8BIT, 0x0391, 0x22 },
	{IMX_8BIT, 0x0392, 0x00 },
	{IMX_8BIT, 0x0401, 0x00 },
	{IMX_8BIT, 0x0404, 0x00 },
	{IMX_8BIT, 0x0405, 0x10 },
	{IMX_8BIT, 0x4082, 0x01 },
	{IMX_8BIT, 0x4083, 0x01 },
	{IMX_8BIT, 0x7006, 0x04 },
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x06 },
	{IMX_8BIT, 0x0341, 0x26 },
	{IMX_8BIT, 0x0342, 0x12 },
	{IMX_8BIT, 0x0343, 0x58 },
	{IMX_8BIT, 0x0344, 0x00 },
	{IMX_8BIT, 0x0345, 0x28 },
	{IMX_8BIT, 0x0346, 0x00 },
	{IMX_8BIT, 0x0347, 0x08 },
	{IMX_8BIT, 0x0348, 0x10 },
	{IMX_8BIT, 0x0349, 0x47 },
	{IMX_8BIT, 0x034A, 0x0C },
	{IMX_8BIT, 0x034B, 0x27 },
	{IMX_8BIT, 0x034C, 0x08 },
	{IMX_8BIT, 0x034D, 0x10 },
	{IMX_8BIT, 0x034E, 0x06 },
	{IMX_8BIT, 0x034F, 0x10 },
	{IMX_8BIT, 0x0350, 0x00 },
	{IMX_8BIT, 0x0351, 0x00 },
	{IMX_8BIT, 0x0352, 0x00 },
	{IMX_8BIT, 0x0353, 0x00 },
	{IMX_8BIT, 0x0354, 0x08 },
	{IMX_8BIT, 0x0355, 0x10 },
	{IMX_8BIT, 0x0356, 0x06 },
	{IMX_8BIT, 0x0357, 0x10 },
	{IMX_8BIT, 0x301D, 0x30 },
	{IMX_8BIT, 0x3310, 0x08 },
	{IMX_8BIT, 0x3311, 0x10 },
	{IMX_8BIT, 0x3312, 0x06 },
	{IMX_8BIT, 0x3313, 0x10 },
	{IMX_8BIT, 0x331C, 0x00 },
	{IMX_8BIT, 0x331D, 0xAA },
	{IMX_8BIT, 0x4084, 0x00 },
	{IMX_8BIT, 0x4085, 0x00 },
	{IMX_8BIT, 0x4086, 0x00 },
	{IMX_8BIT, 0x4087, 0x00 },
	{IMX_8BIT, 0x4400, 0x00 },
	/* Global Timing Setting */
	{IMX_8BIT, 0x0830, 0x5F },
	{IMX_8BIT, 0x0831, 0x1F },
	{IMX_8BIT, 0x0832, 0x3F },
	{IMX_8BIT, 0x0833, 0x1F },
	{IMX_8BIT, 0x0834, 0x1F },
	{IMX_8BIT, 0x0835, 0x17 },
	{IMX_8BIT, 0x0836, 0x67 },
	{IMX_8BIT, 0x0837, 0x27 },
	{IMX_8BIT, 0x0839, 0x1F },
	{IMX_8BIT, 0x083A, 0x17 },
	{IMX_8BIT, 0x083B, 0x02 },
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x06 },
	{IMX_8BIT, 0x0203, 0x22 },
	{IMX_TOK_TERM, 0, 0}
};

/* Capture6: H : 2576 V : 1936 */
static const struct imx_reg imx135_5M_STILL_19fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	{ IMX_8BIT, 0x011E, 0x13},
	{ IMX_8BIT, 0x011F, 0x33},
	{ IMX_8BIT, 0x0301, 0x05},
	{ IMX_8BIT, 0x0303, 0x01},
	{ IMX_8BIT, 0x0305, 0x0C},
	{ IMX_8BIT, 0x0309, 0x05},
	{ IMX_8BIT, 0x030B, 0x01},
	{ IMX_8BIT, 0x030C, 0x02},
	{ IMX_8BIT, 0x030D, 0x32},
	{ IMX_8BIT, 0x030E, 0x01},
	{ IMX_8BIT, 0x3A06, 0x11},
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03 },
	{IMX_8BIT, 0x0112, 0x0A },
	{IMX_8BIT, 0x0113, 0x0A },
	{IMX_8BIT, 0x0381, 0x01 },
	{IMX_8BIT, 0x0383, 0x01 },
	{IMX_8BIT, 0x0385, 0x01 },
	{IMX_8BIT, 0x0387, 0x01 },
	{IMX_8BIT, 0x0390, 0x00 },
	{IMX_8BIT, 0x0391, 0x11 },
	{IMX_8BIT, 0x0392, 0x00 },
	{IMX_8BIT, 0x0401, 0x02 },
	{IMX_8BIT, 0x0404, 0x00 },
	{IMX_8BIT, 0x0405, 0x19 },
	{IMX_8BIT, 0x4082, 0x00 },
	{IMX_8BIT, 0x4083, 0x00 },
	{IMX_8BIT, 0x7006, 0x04 },
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x0B },
	{IMX_8BIT, 0x0341, 0xEA },
	{IMX_8BIT, 0x0342, 0x12 },
	{IMX_8BIT, 0x0343, 0x58 },
	{IMX_8BIT, 0x0344, 0x00 },
	{IMX_8BIT, 0x0345, 0x5C },
	{IMX_8BIT, 0x0346, 0x00 },
	{IMX_8BIT, 0x0347, 0x30 },
	{IMX_8BIT, 0x0348, 0x10 },
	{IMX_8BIT, 0x0349, 0x15 },
	{IMX_8BIT, 0x034A, 0x0C },
	{IMX_8BIT, 0x034B, 0x03 },
	{IMX_8BIT, 0x034C, 0x0A },
	{IMX_8BIT, 0x034D, 0x10 },
	{IMX_8BIT, 0x034E, 0x07 },
	{IMX_8BIT, 0x034F, 0x90 },
	{IMX_8BIT, 0x0350, 0x00 },
	{IMX_8BIT, 0x0351, 0x00 },
	{IMX_8BIT, 0x0352, 0x00 },
	{IMX_8BIT, 0x0353, 0x01 },
	{IMX_8BIT, 0x0354, 0x0F },
	{IMX_8BIT, 0x0355, 0xBA },
	{IMX_8BIT, 0x0356, 0x0B },
	{IMX_8BIT, 0x0357, 0xD2 },
	{IMX_8BIT, 0x301D, 0x30 },
	{IMX_8BIT, 0x3310, 0x0A },
	{IMX_8BIT, 0x3311, 0x10 },
	{IMX_8BIT, 0x3312, 0x07 },
	{IMX_8BIT, 0x3313, 0x90 },
	{IMX_8BIT, 0x331C, 0x03 },
	{IMX_8BIT, 0x331D, 0xAE },
	{IMX_8BIT, 0x4084, 0x0A },
	{IMX_8BIT, 0x4085, 0x10 },
	{IMX_8BIT, 0x4086, 0x07 },
	{IMX_8BIT, 0x4087, 0x90 },
	{IMX_8BIT, 0x4400, 0x00 },
	/* Global Timing Setting */
	{IMX_8BIT, 0x0830, 0x77 },
	{IMX_8BIT, 0x0831, 0x2F },
	{IMX_8BIT, 0x0832, 0x4F },
	{IMX_8BIT, 0x0833, 0x2F },
	{IMX_8BIT, 0x0834, 0x2F },
	{IMX_8BIT, 0x0835, 0x37 },
	{IMX_8BIT, 0x0836, 0xA7 },
	{IMX_8BIT, 0x0837, 0x37 },
	{IMX_8BIT, 0x0839, 0x1F },
	{IMX_8BIT, 0x083A, 0x17 },
	{IMX_8BIT, 0x083B, 0x02 },
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x0B },
	{IMX_8BIT, 0x0203, 0xE6 },
	{IMX_TOK_TERM, 0, 0}
};

/* Capture7: H : 3264 V : 1836 */
static const struct imx_reg imx135_6M_STILL_20fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	{ IMX_8BIT, 0x011E, 0x13},
	{ IMX_8BIT, 0x011F, 0x33},
	{ IMX_8BIT, 0x0301, 0x05},
	{ IMX_8BIT, 0x0303, 0x01},
	{ IMX_8BIT, 0x0305, 0x0C},
	{ IMX_8BIT, 0x0309, 0x05},
	{ IMX_8BIT, 0x030B, 0x01},
	{ IMX_8BIT, 0x030C, 0x02},
	{ IMX_8BIT, 0x030D, 0x32},
	{ IMX_8BIT, 0x030E, 0x01},
	{ IMX_8BIT, 0x3A06, 0x11},
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03 },
	{IMX_8BIT, 0x0112, 0x0A },
	{IMX_8BIT, 0x0113, 0x0A },
	{IMX_8BIT, 0x0381, 0x01 },
	{IMX_8BIT, 0x0383, 0x01 },
	{IMX_8BIT, 0x0385, 0x01 },
	{IMX_8BIT, 0x0387, 0x01 },
	{IMX_8BIT, 0x0390, 0x00 },
	{IMX_8BIT, 0x0391, 0x11 },
	{IMX_8BIT, 0x0392, 0x00 },
	{IMX_8BIT, 0x0401, 0x02 },
	{IMX_8BIT, 0x0404, 0x00 },
	{IMX_8BIT, 0x0405, 0x14 },
	{IMX_8BIT, 0x4082, 0x00 },
	{IMX_8BIT, 0x4083, 0x00 },
	{IMX_8BIT, 0x4203, 0xFF },
	{IMX_8BIT, 0x7006, 0x04 },
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x0b }, /* 0c */
	{IMX_8BIT, 0x0341, 0xea }, /* c0 */
	{IMX_8BIT, 0x0342, 0x27 }, /* 27 */
	{IMX_8BIT, 0x0343, 0x10 }, /* 10 */
	{IMX_8BIT, 0x0344, 0x00 },
	{IMX_8BIT, 0x0345, 0x36 },
	{IMX_8BIT, 0x0346, 0x01 },
	{IMX_8BIT, 0x0347, 0x94 },
	{IMX_8BIT, 0x0348, 0x10 },
	{IMX_8BIT, 0x0349, 0x39 },
	{IMX_8BIT, 0x034A, 0x0a },
	{IMX_8BIT, 0x034B, 0x9f },
	{IMX_8BIT, 0x034C, 0x0C },
	{IMX_8BIT, 0x034D, 0xd0 },
	{IMX_8BIT, 0x034E, 0x07 },
	{IMX_8BIT, 0x034F, 0x3c },
	{IMX_8BIT, 0x0350, 0x00 }, /* crop x */
	{IMX_8BIT, 0x0351, 0x00 },
	{IMX_8BIT, 0x0352, 0x00 }, /* crop y */
	{IMX_8BIT, 0x0353, 0x00 },
	{IMX_8BIT, 0x0354, 0x10 }, /* crop size */
	{IMX_8BIT, 0x0355, 0x04 },
	{IMX_8BIT, 0x0356, 0x09 }, /* crop size */
	{IMX_8BIT, 0x0357, 0x0c },
	{IMX_8BIT, 0x301D, 0x30 },
	{IMX_8BIT, 0x3310, 0x0C },
	{IMX_8BIT, 0x3311, 0xd0 },
	{IMX_8BIT, 0x3312, 0x07 },
	{IMX_8BIT, 0x3313, 0x3C },
	{IMX_8BIT, 0x331C, 0x02 },
	{IMX_8BIT, 0x331D, 0xa0 },
	{IMX_8BIT, 0x4084, 0x0C },
	{IMX_8BIT, 0x4085, 0xd0 },
	{IMX_8BIT, 0x4086, 0x07 },
	{IMX_8BIT, 0x4087, 0x3C },
	{IMX_8BIT, 0x4400, 0x00 },
	{IMX_8BIT, 0x0830, 0x87 },
	{IMX_8BIT, 0x0831, 0x3F },
	{IMX_8BIT, 0x0832, 0x67 },
	{IMX_8BIT, 0x0833, 0x3F },
	{IMX_8BIT, 0x0834, 0x3F },
	{IMX_8BIT, 0x0835, 0x4F },
	{IMX_8BIT, 0x0836, 0xDF },
	{IMX_8BIT, 0x0837, 0x47 },
	{IMX_8BIT, 0x0839, 0x1F },
	{IMX_8BIT, 0x083A, 0x17 },
	{IMX_8BIT, 0x083B, 0x02 },
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x0A },
	{IMX_8BIT, 0x0203, 0x56 },
	{IMX_TOK_TERM, 0, 0}
};

/* Capture8: H : 3280 V : 2464 */
static const struct imx_reg imx135_8M_STILL_15fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	{ IMX_8BIT, 0x011E, 0x13},
	{ IMX_8BIT, 0x011F, 0x33},
	{ IMX_8BIT, 0x0301, 0x05},
	{ IMX_8BIT, 0x0303, 0x01},
	{ IMX_8BIT, 0x0305, 0x0C},
	{ IMX_8BIT, 0x0309, 0x05},
	{ IMX_8BIT, 0x030B, 0x01},
	{ IMX_8BIT, 0x030C, 0x02},
	{ IMX_8BIT, 0x030D, 0x32},
	{ IMX_8BIT, 0x030E, 0x01},
	{ IMX_8BIT, 0x3A06, 0x11},
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03 },
	{IMX_8BIT, 0x0112, 0x0A },
	{IMX_8BIT, 0x0113, 0x0A },
	{IMX_8BIT, 0x0381, 0x01 },
	{IMX_8BIT, 0x0383, 0x01 },
	{IMX_8BIT, 0x0385, 0x01 },
	{IMX_8BIT, 0x0387, 0x01 },
	{IMX_8BIT, 0x0390, 0x00 },
	{IMX_8BIT, 0x0391, 0x11 },
	{IMX_8BIT, 0x0392, 0x00 },
	{IMX_8BIT, 0x0401, 0x00 },
	{IMX_8BIT, 0x0404, 0x00 },
	{IMX_8BIT, 0x0405, 0x10 },
	{IMX_8BIT, 0x4082, 0x01 },
	{IMX_8BIT, 0x4083, 0x01 },
	{IMX_8BIT, 0x7006, 0x04 },
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x0a },
	{IMX_8BIT, 0x0341, 0x3c },
	{IMX_8BIT, 0x0342, 0x11 },
	{IMX_8BIT, 0x0343, 0xdc },
	{IMX_8BIT, 0x0344, 0x01 },
	{IMX_8BIT, 0x0345, 0xd0 },
	{IMX_8BIT, 0x0346, 0x01 },
	{IMX_8BIT, 0x0347, 0x48 },
	{IMX_8BIT, 0x0348, 0x0e },
	{IMX_8BIT, 0x0349, 0x9f },
	{IMX_8BIT, 0x034A, 0x0a },
	{IMX_8BIT, 0x034B, 0xe7 },
	{IMX_8BIT, 0x034C, 0x0c },
	{IMX_8BIT, 0x034D, 0xd0 },
	{IMX_8BIT, 0x034E, 0x09 },
	{IMX_8BIT, 0x034F, 0xA0 },
	{IMX_8BIT, 0x0350, 0x00 },
	{IMX_8BIT, 0x0351, 0x00 },
	{IMX_8BIT, 0x0352, 0x00 },
	{IMX_8BIT, 0x0353, 0x00 },
	{IMX_8BIT, 0x0354, 0x0c },
	{IMX_8BIT, 0x0355, 0xd0 },
	{IMX_8BIT, 0x0356, 0x09 },
	{IMX_8BIT, 0x0357, 0xa0 },
	{IMX_8BIT, 0x301D, 0x30 },
	{IMX_8BIT, 0x3310, 0x0C },
	{IMX_8BIT, 0x3311, 0xD0 },
	{IMX_8BIT, 0x3312, 0x09 },
	{IMX_8BIT, 0x3313, 0xA0 },
	{IMX_8BIT, 0x331C, 0x00 },
	{IMX_8BIT, 0x331D, 0x10 },
	{IMX_8BIT, 0x4084, 0x00 },
	{IMX_8BIT, 0x4085, 0x00 },
	{IMX_8BIT, 0x4086, 0x00 },
	{IMX_8BIT, 0x4087, 0x00 },
	{IMX_8BIT, 0x4400, 0x00 },
	/* Global Timing Setting */
	{IMX_8BIT, 0x0830, 0x87 },
	{IMX_8BIT, 0x0831, 0x3F },
	{IMX_8BIT, 0x0832, 0x67 },
	{IMX_8BIT, 0x0833, 0x3F },
	{IMX_8BIT, 0x0834, 0x3F },
	{IMX_8BIT, 0x0835, 0x4f },
	{IMX_8BIT, 0x0836, 0xdf },
	{IMX_8BIT, 0x0837, 0x47 },
	{IMX_8BIT, 0x0839, 0x1F },
	{IMX_8BIT, 0x083A, 0x17 },
	{IMX_8BIT, 0x083B, 0x02 },
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x0a },
	{IMX_8BIT, 0x0203, 0x38 },
	{IMX_TOK_TERM, 0, 0}
};

/* Capture9: H : 4208 V : 3120 */
static const struct imx_reg imx135_13M_STILL_7fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03 },
	{IMX_8BIT, 0x0112, 0x0A },
	{IMX_8BIT, 0x0113, 0x0A },
	{IMX_8BIT, 0x0381, 0x01 },
	{IMX_8BIT, 0x0383, 0x01 },
	{IMX_8BIT, 0x0385, 0x01 },
	{IMX_8BIT, 0x0387, 0x01 },
	{IMX_8BIT, 0x0390, 0x00 },
	{IMX_8BIT, 0x0391, 0x11 },
	{IMX_8BIT, 0x0392, 0x00 },
	{IMX_8BIT, 0x0401, 0x00 },
	{IMX_8BIT, 0x0404, 0x00 },
	{IMX_8BIT, 0x0405, 0x10 },
	{IMX_8BIT, 0x4082, 0x01 },
	{IMX_8BIT, 0x4083, 0x01 },
	{IMX_8BIT, 0x7006, 0x04 },
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x0E },
	{IMX_8BIT, 0x0341, 0xD8 },
	{IMX_8BIT, 0x0342, 0x27 },
	{IMX_8BIT, 0x0343, 0x10 },
	{IMX_8BIT, 0x0344, 0x00 },
	{IMX_8BIT, 0x0345, 0x00 },
	{IMX_8BIT, 0x0346, 0x00 },
	{IMX_8BIT, 0x0347, 0x00 },
	{IMX_8BIT, 0x0348, 0x10 },
	{IMX_8BIT, 0x0349, 0x6F },
	{IMX_8BIT, 0x034A, 0x0C },
	{IMX_8BIT, 0x034B, 0x2F },
	{IMX_8BIT, 0x034C, 0x10 },
	{IMX_8BIT, 0x034D, 0x70 },
	{IMX_8BIT, 0x034E, 0x0C },
	{IMX_8BIT, 0x034F, 0x30 },
	{IMX_8BIT, 0x0350, 0x00 },
	{IMX_8BIT, 0x0351, 0x00 },
	{IMX_8BIT, 0x0352, 0x00 },
	{IMX_8BIT, 0x0353, 0x00 },
	{IMX_8BIT, 0x0354, 0x10 },
	{IMX_8BIT, 0x0355, 0x70 },
	{IMX_8BIT, 0x0356, 0x0C },
	{IMX_8BIT, 0x0357, 0x30 },
	{IMX_8BIT, 0x301D, 0x30 },
	{IMX_8BIT, 0x3310, 0x10 },
	{IMX_8BIT, 0x3311, 0x70 },
	{IMX_8BIT, 0x3312, 0x0C },
	{IMX_8BIT, 0x3313, 0x30 },
	{IMX_8BIT, 0x331C, 0x00 },
	{IMX_8BIT, 0x331D, 0x10 },
	{IMX_8BIT, 0x4084, 0x00 },
	{IMX_8BIT, 0x4085, 0x00 },
	{IMX_8BIT, 0x4086, 0x00 },
	{IMX_8BIT, 0x4087, 0x00 },
	{IMX_8BIT, 0x4400, 0x00 },
	/* Global Timing Setting */
	{IMX_8BIT, 0x0830, 0x77 },
	{IMX_8BIT, 0x0831, 0x2F },
	{IMX_8BIT, 0x0832, 0x4F },
	{IMX_8BIT, 0x0833, 0x2F },
	{IMX_8BIT, 0x0834, 0x2F },
	{IMX_8BIT, 0x0835, 0x37 },
	{IMX_8BIT, 0x0836, 0xA7 },
	{IMX_8BIT, 0x0837, 0x37 },
	{IMX_8BIT, 0x0839, 0x1F },
	{IMX_8BIT, 0x083A, 0x17 },
	{IMX_8BIT, 0x083B, 0x02 },
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x0E },
	{IMX_8BIT, 0x0203, 0xD4 },
	{IMX_TOK_TERM, 0, 0}
};

/*****************************IMX135 PREVIEW********************************/

/* Preview1: H : 848 V : 616 */
static struct imx_reg const imx135_PREVIEW_848x616_30fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	{ IMX_8BIT, 0x011E, 0x13},
	{ IMX_8BIT, 0x011F, 0x33},
	{ IMX_8BIT, 0x0301, 0x05},
	{ IMX_8BIT, 0x0303, 0x01},
	{ IMX_8BIT, 0x0305, 0x0B},
	{ IMX_8BIT, 0x0309, 0x05},
	{ IMX_8BIT, 0x030B, 0x02},
	{ IMX_8BIT, 0x030C, 0x01},
	{ IMX_8BIT, 0x030D, 0x83},
	{ IMX_8BIT, 0x030E, 0x01},
	{ IMX_8BIT, 0x3A06, 0x12},
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03 },
	{IMX_8BIT, 0x0112, 0x0A },
	{IMX_8BIT, 0x0113, 0x0A },
	{IMX_8BIT, 0x0381, 0x01 },
	{IMX_8BIT, 0x0383, 0x01 },
	{IMX_8BIT, 0x0385, 0x01 },
	{IMX_8BIT, 0x0387, 0x01 },
	{IMX_8BIT, 0x0390, 0x01 },
	{IMX_8BIT, 0x0391, 0x44 },
	{IMX_8BIT, 0x0392, 0x00 },
	{IMX_8BIT, 0x0401, 0x02 },
	{IMX_8BIT, 0x0404, 0x00 },
	{IMX_8BIT, 0x0405, 0x13 },
	{IMX_8BIT, 0x4082, 0x00 },
	{IMX_8BIT, 0x4083, 0x00 },
	{IMX_8BIT, 0x7006, 0x04 },
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x08 },
	{IMX_8BIT, 0x0341, 0xB8 },
	{IMX_8BIT, 0x0342, 0x15 },
	{IMX_8BIT, 0x0343, 0x18 },
	{IMX_8BIT, 0x0344, 0x00 },
	{IMX_8BIT, 0x0345, 0x58 },
	{IMX_8BIT, 0x0346, 0x00 },
	{IMX_8BIT, 0x0347, 0x60 },
	{IMX_8BIT, 0x0348, 0x10 },
	{IMX_8BIT, 0x0349, 0x17 },
	{IMX_8BIT, 0x034A, 0x0B },
	{IMX_8BIT, 0x034B, 0xCF },
	{IMX_8BIT, 0x034C, 0x03 },
	{IMX_8BIT, 0x034D, 0x50 },
	{IMX_8BIT, 0x034E, 0x02 },
	{IMX_8BIT, 0x034F, 0x68 },
	{IMX_8BIT, 0x0350, 0x00 },
	{IMX_8BIT, 0x0351, 0x00 },
	{IMX_8BIT, 0x0352, 0x00 },
	{IMX_8BIT, 0x0353, 0x00 },
	{IMX_8BIT, 0x0354, 0x03 },
	{IMX_8BIT, 0x0355, 0xF0 },
	{IMX_8BIT, 0x0356, 0x02 },
	{IMX_8BIT, 0x0357, 0xDC },
	{IMX_8BIT, 0x301D, 0x30 },
	{IMX_8BIT, 0x3310, 0x03 },
	{IMX_8BIT, 0x3311, 0x50 },
	{IMX_8BIT, 0x3312, 0x02 },
	{IMX_8BIT, 0x3313, 0x68 },
	{IMX_8BIT, 0x331C, 0x02 },
	{IMX_8BIT, 0x331D, 0x1C },
	{IMX_8BIT, 0x4084, 0x03 },
	{IMX_8BIT, 0x4085, 0x50 },
	{IMX_8BIT, 0x4086, 0x02 },
	{IMX_8BIT, 0x4087, 0x68 },
	{IMX_8BIT, 0x4400, 0x00 },
	/* Global Timing Setting */
	{IMX_8BIT, 0x0830, 0x5F },
	{IMX_8BIT, 0x0831, 0x0C },
	{IMX_8BIT, 0x0832, 0x3F },
	{IMX_8BIT, 0x0833, 0x1F },
	{IMX_8BIT, 0x0834, 0x1F },
	{IMX_8BIT, 0x0835, 0x17 },
	{IMX_8BIT, 0x0836, 0x67 },
	{IMX_8BIT, 0x0837, 0x27 },
	{IMX_8BIT, 0x0839, 0x1F },
	{IMX_8BIT, 0x083A, 0x17 },
	{IMX_8BIT, 0x083B, 0x02 },
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x07 },
	{IMX_8BIT, 0x0203, 0xA4 },
	/* Gain Setting */
	{ IMX_8BIT, 0x0205, 0x00},
	{ IMX_8BIT, 0x020E, 0x01},
	{ IMX_8BIT, 0x020F, 0x00},
	{ IMX_8BIT, 0x0210, 0x02},
	{ IMX_8BIT, 0x0211, 0x00},
	{ IMX_8BIT, 0x0212, 0x02},
	{ IMX_8BIT, 0x0213, 0x00},
	{ IMX_8BIT, 0x0214, 0x01},
	{ IMX_8BIT, 0x0215, 0x00},
	{IMX_TOK_TERM, 0, 0}
};
/* Preview2: H : 1040 V : 592 */
static struct imx_reg const imx135_1M_16_9_30fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	{ IMX_8BIT, 0x011E, 0x13},
	{ IMX_8BIT, 0x011F, 0x33},
	{ IMX_8BIT, 0x0301, 0x05},
	{ IMX_8BIT, 0x0303, 0x01},
	{ IMX_8BIT, 0x0305, 0x0B},
	{ IMX_8BIT, 0x0309, 0x05},
	{ IMX_8BIT, 0x030B, 0x02},
	{ IMX_8BIT, 0x030C, 0x01},
	{ IMX_8BIT, 0x030D, 0x83},
	{ IMX_8BIT, 0x030E, 0x01},
	{ IMX_8BIT, 0x3A06, 0x12},
	/*mode setting*/
	{IMX_8BIT, 0x0108, 0x03},
	{IMX_8BIT, 0x0112, 0x0A},
	{IMX_8BIT, 0x0113, 0x0A},
	{IMX_8BIT, 0x0381, 0x01},
	{IMX_8BIT, 0x0383, 0x01},
	{IMX_8BIT, 0x0385, 0x01},
	{IMX_8BIT, 0x0387, 0x01},

	{IMX_8BIT, 0x0390, 0x01}, /* Binning 0x22 : 1/2 ox44: 1/4 */
	{IMX_8BIT, 0x0391, 0x44},
	{IMX_8BIT, 0x0392, 0x00},
	{IMX_8BIT, 0x0401, 0x00}, /* SCALING */
	{IMX_8BIT, 0x0404, 0x00},
	{IMX_8BIT, 0x0405, 0x00},
	{IMX_8BIT, 0x4082, 0x00},
	{IMX_8BIT, 0x4083, 0x00},
	/*{IMX_8BIT, 0x4203, 0xFF},*/
	{IMX_8BIT, 0x7006, 0x04},
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x0A}, /* VTS HTS*/
	{IMX_8BIT, 0x0341, 0x5A},
	{IMX_8BIT, 0x0342, 0x11},
	{IMX_8BIT, 0x0343, 0xDC},
	{IMX_8BIT, 0x0344, 0x00}, /* xo-y1 */
	{IMX_8BIT, 0x0345, 0x18},
	{IMX_8BIT, 0x0346, 0x01},
	{IMX_8BIT, 0x0347, 0x78},
	{IMX_8BIT, 0x0348, 0x10},
	{IMX_8BIT, 0x0349, 0x57},
	{IMX_8BIT, 0x034A, 0x0A},
	{IMX_8BIT, 0x034B, 0xB7},

	{IMX_8BIT, 0x034C, 0x04}, /* out */
	{IMX_8BIT, 0x034D, 0x10},
	{IMX_8BIT, 0x034E, 0x02},
	{IMX_8BIT, 0x034F, 0x50},

	{IMX_8BIT, 0x0350, 0x00},
	{IMX_8BIT, 0x0351, 0x00},
	{IMX_8BIT, 0x0352, 0x00},
	{IMX_8BIT, 0x0353, 0x00},
	{IMX_8BIT, 0x0354, 0x04}, /* Dig Crop */
	{IMX_8BIT, 0x0355, 0x10},
	{IMX_8BIT, 0x0356, 0x02},
	{IMX_8BIT, 0x0357, 0x50},

	{IMX_8BIT, 0x301D, 0x30},
	{IMX_8BIT, 0x3310, 0x04},
	{IMX_8BIT, 0x3311, 0x10},
	{IMX_8BIT, 0x3312, 0x02},
	{IMX_8BIT, 0x3313, 0x50},
	{IMX_8BIT, 0x331C, 0x02},
	{IMX_8BIT, 0x331D, 0x3F},
	{IMX_8BIT, 0x4084, 0x00},
	{IMX_8BIT, 0x4085, 0x00},
	{IMX_8BIT, 0x4086, 0x00},
	{IMX_8BIT, 0x4087, 0x00},
	{IMX_8BIT, 0x4400, 0x00},
	/* Global Timing */
	{IMX_8BIT, 0x0830, 0x5F},
	{IMX_8BIT, 0x0831, 0x0C},
	{IMX_8BIT, 0x0832, 0x3F},
	{IMX_8BIT, 0x0833, 0x1F},
	{IMX_8BIT, 0x0834, 0x1F},
	{IMX_8BIT, 0x0835, 0x17},
	{IMX_8BIT, 0x0836, 0x67},
	{IMX_8BIT, 0x0837, 0x27},
	{IMX_8BIT, 0x0839, 0x1F},
	{IMX_8BIT, 0x083A, 0x17},
	{IMX_8BIT, 0x083B, 0x02},
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x06},
	{IMX_8BIT, 0x0203, 0x02},
	/* Gain Setting */
	{IMX_8BIT, 0x0205, 0x00},
	{IMX_8BIT, 0x020E, 0x01},
	{IMX_8BIT, 0x020F, 0x00},
	{IMX_8BIT, 0x0210, 0x01},
	{IMX_8BIT, 0x0211, 0x00},
	{IMX_8BIT, 0x0212, 0x01},
	{IMX_8BIT, 0x0213, 0x00},
	{IMX_8BIT, 0x0214, 0x01},
	{IMX_8BIT, 0x0215, 0x00},
	/* HDR Setting */
	{IMX_8BIT, 0x0230, 0x00},
	{IMX_8BIT, 0x0231, 0x00},
	{IMX_8BIT, 0x0233, 0x00},
	{IMX_8BIT, 0x0234, 0x00},
	{IMX_8BIT, 0x0235, 0x40},
	{IMX_8BIT, 0x0238, 0x00},
	{IMX_8BIT, 0x0239, 0x04},
	{IMX_8BIT, 0x023B, 0x00},
	{IMX_8BIT, 0x023C, 0x01},
	{IMX_8BIT, 0x33B0, 0x04},
	{IMX_8BIT, 0x33B1, 0x00},
	{IMX_8BIT, 0x33B3, 0x00},
	{IMX_8BIT, 0x33B4, 0x01},
	{IMX_8BIT, 0x3800, 0x00},
	{IMX_TOK_TERM, 0, 0},
};

/* Preview3: H : 1280 V : 720 */
static struct imx_reg const imx135_PREVIEW_WIDE_PREVIEW_30fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	{ IMX_8BIT, 0x011E, 0x13},
	{ IMX_8BIT, 0x011F, 0x33},
	{ IMX_8BIT, 0x0301, 0x05},
	{ IMX_8BIT, 0x0303, 0x01},
	{ IMX_8BIT, 0x0305, 0x0C},
	{ IMX_8BIT, 0x0309, 0x05},
	{ IMX_8BIT, 0x030B, 0x01},
	{ IMX_8BIT, 0x030C, 0x02},
	{ IMX_8BIT, 0x030D, 0x32},
	{ IMX_8BIT, 0x030E, 0x01},
	{ IMX_8BIT, 0x3A06, 0x11},
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03 },
	{IMX_8BIT, 0x0112, 0x0A },
	{IMX_8BIT, 0x0113, 0x0A },
	{IMX_8BIT, 0x0381, 0x01 },
	{IMX_8BIT, 0x0383, 0x01 },
	{IMX_8BIT, 0x0385, 0x01 },
	{IMX_8BIT, 0x0387, 0x01 },
	{IMX_8BIT, 0x0390, 0x01 },
	{IMX_8BIT, 0x0391, 0x22 },
	{IMX_8BIT, 0x0392, 0x00 },
	{IMX_8BIT, 0x0401, 0x02 },
	{IMX_8BIT, 0x0404, 0x00 },
	{IMX_8BIT, 0x0405, 0x1A },
	{IMX_8BIT, 0x4082, 0x00 },
	{IMX_8BIT, 0x4083, 0x00 },
	{IMX_8BIT, 0x7006, 0x04 },
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x07 },
	{IMX_8BIT, 0x0341, 0xA8 },
	{IMX_8BIT, 0x0342, 0x11 },
	{IMX_8BIT, 0x0343, 0xDC },
	{IMX_8BIT, 0x0344, 0x00 },
	{IMX_8BIT, 0x0345, 0x18 },
	{IMX_8BIT, 0x0346, 0x01 },
	{IMX_8BIT, 0x0347, 0x88 },
	{IMX_8BIT, 0x0348, 0x10 },
	{IMX_8BIT, 0x0349, 0x57 },
	{IMX_8BIT, 0x034A, 0x0A },
	{IMX_8BIT, 0x034B, 0xAB },
	{IMX_8BIT, 0x034C, 0x05 },
	{IMX_8BIT, 0x034D, 0x00 },
	{IMX_8BIT, 0x034E, 0x02 },
	{IMX_8BIT, 0x034F, 0xD0 },
	{IMX_8BIT, 0x0350, 0x00 },
	{IMX_8BIT, 0x0351, 0x00 },
	{IMX_8BIT, 0x0352, 0x00 },
	{IMX_8BIT, 0x0353, 0x00 },
	{IMX_8BIT, 0x0354, 0x08 },
	{IMX_8BIT, 0x0355, 0x20 },
	{IMX_8BIT, 0x0356, 0x04 },
	{IMX_8BIT, 0x0357, 0x92 },
	{IMX_8BIT, 0x301D, 0x30 },
	{IMX_8BIT, 0x3310, 0x05 },
	{IMX_8BIT, 0x3311, 0x00 },
	{IMX_8BIT, 0x3312, 0x02 },
	{IMX_8BIT, 0x3313, 0xD0 },
	{IMX_8BIT, 0x331C, 0x02 },
	{IMX_8BIT, 0x331D, 0x18 },
	{IMX_8BIT, 0x4084, 0x05 },
	{IMX_8BIT, 0x4085, 0x00 },
	{IMX_8BIT, 0x4086, 0x02 },
	{IMX_8BIT, 0x4087, 0xD0 },
	{IMX_8BIT, 0x4400, 0x00 },
	/* Global Timing Setting */
	{IMX_8BIT, 0x0830, 0x5F },
	{IMX_8BIT, 0x0831, 0x1F },
	{IMX_8BIT, 0x0832, 0x3F },
	{IMX_8BIT, 0x0833, 0x1F },
	{IMX_8BIT, 0x0834, 0x1F },
	{IMX_8BIT, 0x0835, 0x17 },
	{IMX_8BIT, 0x0836, 0x67 },
	{IMX_8BIT, 0x0837, 0x27 },
	{IMX_8BIT, 0x0839, 0x1F },
	{IMX_8BIT, 0x083A, 0x17 },
	{IMX_8BIT, 0x083B, 0x02 },
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x07 },
	{IMX_8BIT, 0x0203, 0xA4 },
	{IMX_TOK_TERM, 0, 0}
};

/* Preview4: H : 1632 V : 1224 */
static struct imx_reg const imx135_PREVIEW_1632x1224_30fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	{ IMX_8BIT, 0x011E, 0x13},
	{ IMX_8BIT, 0x011F, 0x33},
	{ IMX_8BIT, 0x0301, 0x05},
	{ IMX_8BIT, 0x0303, 0x01},
	{ IMX_8BIT, 0x0305, 0x0B},
	{ IMX_8BIT, 0x0309, 0x05},
	{ IMX_8BIT, 0x030B, 0x02},
	{ IMX_8BIT, 0x030C, 0x01},
	{ IMX_8BIT, 0x030D, 0x83},
	{ IMX_8BIT, 0x030E, 0x01},
	{ IMX_8BIT, 0x3A06, 0x12},
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03 },
	{IMX_8BIT, 0x0112, 0x0A },
	{IMX_8BIT, 0x0113, 0x0A },
	{IMX_8BIT, 0x0381, 0x01 },
	{IMX_8BIT, 0x0383, 0x01 },
	{IMX_8BIT, 0x0385, 0x01 },
	{IMX_8BIT, 0x0387, 0x01 },
	{IMX_8BIT, 0x0390, 0x01 },
	{IMX_8BIT, 0x0391, 0x22 },
	{IMX_8BIT, 0x0392, 0x00 },
	{IMX_8BIT, 0x0401, 0x02 },
	{IMX_8BIT, 0x0404, 0x00 },
	{IMX_8BIT, 0x0405, 0x14 },
	{IMX_8BIT, 0x4082, 0x00 },
	{IMX_8BIT, 0x4083, 0x00 },
	{IMX_8BIT, 0x7006, 0x04 },
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x08 },
	{IMX_8BIT, 0x0341, 0xb8 },
	{IMX_8BIT, 0x0342, 0x15 },
	{IMX_8BIT, 0x0343, 0x18 },

	{IMX_8BIT, 0x0344, 0x00 },
	{IMX_8BIT, 0x0345, 0x40 },
	{IMX_8BIT, 0x0346, 0x00 },
	{IMX_8BIT, 0x0347, 0x20 },
	{IMX_8BIT, 0x0348, 0x10 },
	{IMX_8BIT, 0x0349, 0x2F },
	{IMX_8BIT, 0x034A, 0x0C },
	{IMX_8BIT, 0x034B, 0x13 },

	{IMX_8BIT, 0x034C, 0x06 },
	{IMX_8BIT, 0x034D, 0x60 },
	{IMX_8BIT, 0x034E, 0x04 },
	{IMX_8BIT, 0x034F, 0xC8 },

	{IMX_8BIT, 0x0350, 0x00 },
	{IMX_8BIT, 0x0351, 0x00 },
	{IMX_8BIT, 0x0352, 0x00 },
	{IMX_8BIT, 0x0353, 0x00 },

	{IMX_8BIT, 0x0354, 0x07 },
	{IMX_8BIT, 0x0355, 0xF8 },
	{IMX_8BIT, 0x0356, 0x05 },
	{IMX_8BIT, 0x0357, 0xFA },

	{IMX_8BIT, 0x301D, 0x30 },

	{IMX_8BIT, 0x3310, 0x06 },
	{IMX_8BIT, 0x3311, 0x60 },
	{IMX_8BIT, 0x3312, 0x04 },
	{IMX_8BIT, 0x3313, 0xC8 },

	{IMX_8BIT, 0x331C, 0x01 },
	{IMX_8BIT, 0x331D, 0x9A },
	{IMX_8BIT, 0x4084, 0x06 },
	{IMX_8BIT, 0x4085, 0x60 },
	{IMX_8BIT, 0x4086, 0x04 },
	{IMX_8BIT, 0x4087, 0xC8 },
	{IMX_8BIT, 0x4400, 0x00 },
	/* Global Timing Setting */
	{IMX_8BIT, 0x0830, 0x5F },
	{IMX_8BIT, 0x0831, 0x0C },
	{IMX_8BIT, 0x0832, 0x3F },
	{IMX_8BIT, 0x0833, 0x1F },
	{IMX_8BIT, 0x0834, 0x1F },
	{IMX_8BIT, 0x0835, 0x17 },
	{IMX_8BIT, 0x0836, 0x67 },
	{IMX_8BIT, 0x0837, 0x27 },
	{IMX_8BIT, 0x0839, 0x1F },
	{IMX_8BIT, 0x083A, 0x17 },
	{IMX_8BIT, 0x083B, 0x02 },
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x07 },
	{IMX_8BIT, 0x0203, 0xA4 },
	/* Gain Setting */
	{ IMX_8BIT, 0x0205, 0x00},
	{ IMX_8BIT, 0x020E, 0x01},
	{ IMX_8BIT, 0x020F, 0x00},
	{ IMX_8BIT, 0x0210, 0x02},
	{ IMX_8BIT, 0x0211, 0x00},
	{ IMX_8BIT, 0x0212, 0x02},
	{ IMX_8BIT, 0x0213, 0x00},
	{ IMX_8BIT, 0x0214, 0x01},
	{ IMX_8BIT, 0x0215, 0x00},
	{IMX_TOK_TERM, 0, 0}
};


/***************** IMX135 VIDEO ***************************************/

/* Video1: H : 216 V : 176 */
static const struct imx_reg imx135_QCIF_strong_dvs_30fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03 },
	{IMX_8BIT, 0x0112, 0x0A },
	{IMX_8BIT, 0x0113, 0x0A },
	{IMX_8BIT, 0x0381, 0x01 },
	{IMX_8BIT, 0x0383, 0x01 },
	{IMX_8BIT, 0x0385, 0x01 },
	{IMX_8BIT, 0x0387, 0x01 },
	{IMX_8BIT, 0x0390, 0x01 },
	{IMX_8BIT, 0x0391, 0x44 },
	{IMX_8BIT, 0x0392, 0x00 },
	{IMX_8BIT, 0x0401, 0x02 },
	{IMX_8BIT, 0x0404, 0x00 },
	{IMX_8BIT, 0x0405, 0x46 },
	{IMX_8BIT, 0x4082, 0x00 },
	{IMX_8BIT, 0x4083, 0x00 },
	{IMX_8BIT, 0x7006, 0x04 },
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x07 },
	{IMX_8BIT, 0x0341, 0xA8 },
	{IMX_8BIT, 0x0342, 0x11 },
	{IMX_8BIT, 0x0343, 0xDC },
	{IMX_8BIT, 0x0344, 0x00 },
	{IMX_8BIT, 0x0345, 0xD4 },
	{IMX_8BIT, 0x0346, 0x00 },
	{IMX_8BIT, 0x0347, 0x14 },
	{IMX_8BIT, 0x0348, 0x0F },
	{IMX_8BIT, 0x0349, 0x9B },
	{IMX_8BIT, 0x034A, 0x0C },
	{IMX_8BIT, 0x034B, 0x1B },
	{IMX_8BIT, 0x034C, 0x00 },
	{IMX_8BIT, 0x034D, 0xD8 },
	{IMX_8BIT, 0x034E, 0x00 },
	{IMX_8BIT, 0x034F, 0xB0 },
	{IMX_8BIT, 0x0350, 0x00 },
	{IMX_8BIT, 0x0351, 0x00 },
	{IMX_8BIT, 0x0352, 0x00 },
	{IMX_8BIT, 0x0353, 0x00 },
	{IMX_8BIT, 0x0354, 0x03 },
	{IMX_8BIT, 0x0355, 0xB2 },
	{IMX_8BIT, 0x0356, 0x03 },
	{IMX_8BIT, 0x0357, 0x02 },
	{IMX_8BIT, 0x301D, 0x30 },
	{IMX_8BIT, 0x3310, 0x00 },
	{IMX_8BIT, 0x3311, 0xD8 },
	{IMX_8BIT, 0x3312, 0x00 },
	{IMX_8BIT, 0x3313, 0xB0 },
	{IMX_8BIT, 0x331C, 0x00 },
	{IMX_8BIT, 0x331D, 0xD4 },
	{IMX_8BIT, 0x4084, 0x00 },
	{IMX_8BIT, 0x4085, 0xD8 },
	{IMX_8BIT, 0x4086, 0x00 },
	{IMX_8BIT, 0x4087, 0xB0 },
	{IMX_8BIT, 0x4400, 0x00 },
	/* Global Timing Setting */
	{IMX_8BIT, 0x0830, 0x5F },
	{IMX_8BIT, 0x0831, 0x1F },
	{IMX_8BIT, 0x0832, 0x3F },
	{IMX_8BIT, 0x0833, 0x1F },
	{IMX_8BIT, 0x0834, 0x1F },
	{IMX_8BIT, 0x0835, 0x17 },
	{IMX_8BIT, 0x0836, 0x67 },
	{IMX_8BIT, 0x0837, 0x27 },
	{IMX_8BIT, 0x0839, 0x1F },
	{IMX_8BIT, 0x083A, 0x17 },
	{IMX_8BIT, 0x083B, 0x02 },
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x07 },
	{IMX_8BIT, 0x0203, 0xA4 },
	{IMX_TOK_TERM, 0, 0}
};

/* Video2: H : 408 V : 308 */
static const struct imx_reg imx135_QVGA_strong_dvs_30fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03 },
	{IMX_8BIT, 0x0112, 0x0A },
	{IMX_8BIT, 0x0113, 0x0A },
	{IMX_8BIT, 0x0381, 0x01 },
	{IMX_8BIT, 0x0383, 0x01 },
	{IMX_8BIT, 0x0385, 0x01 },
	{IMX_8BIT, 0x0387, 0x01 },
	{IMX_8BIT, 0x0390, 0x01 },
	{IMX_8BIT, 0x0391, 0x44 },
	{IMX_8BIT, 0x0392, 0x00 },
	{IMX_8BIT, 0x0401, 0x02 },
	{IMX_8BIT, 0x0404, 0x00 },
	{IMX_8BIT, 0x0405, 0x28 },
	{IMX_8BIT, 0x4082, 0x00 },
	{IMX_8BIT, 0x4083, 0x00 },
	{IMX_8BIT, 0x7006, 0x04 },
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x07 },
	{IMX_8BIT, 0x0341, 0xA8 },
	{IMX_8BIT, 0x0342, 0x11 },
	{IMX_8BIT, 0x0343, 0xDC },
	{IMX_8BIT, 0x0344, 0x00 },
	{IMX_8BIT, 0x0345, 0x40 },
	{IMX_8BIT, 0x0346, 0x00 },
	{IMX_8BIT, 0x0347, 0x14 },
	{IMX_8BIT, 0x0348, 0x10 },
	{IMX_8BIT, 0x0349, 0x2F },
	{IMX_8BIT, 0x034A, 0x0C },
	{IMX_8BIT, 0x034B, 0x1B },
	{IMX_8BIT, 0x034C, 0x01 },
	{IMX_8BIT, 0x034D, 0x98 },
	{IMX_8BIT, 0x034E, 0x01 },
	{IMX_8BIT, 0x034F, 0x34 },
	{IMX_8BIT, 0x0350, 0x00 },
	{IMX_8BIT, 0x0351, 0x00 },
	{IMX_8BIT, 0x0352, 0x00 },
	{IMX_8BIT, 0x0353, 0x00 },
	{IMX_8BIT, 0x0354, 0x03 },
	{IMX_8BIT, 0x0355, 0xFC },
	{IMX_8BIT, 0x0356, 0x03 },
	{IMX_8BIT, 0x0357, 0x02 },
	{IMX_8BIT, 0x301D, 0x30 },
	{IMX_8BIT, 0x3310, 0x01 },
	{IMX_8BIT, 0x3311, 0x98 },
	{IMX_8BIT, 0x3312, 0x01 },
	{IMX_8BIT, 0x3313, 0x34 },
	{IMX_8BIT, 0x331C, 0x01 },
	{IMX_8BIT, 0x331D, 0x68 },
	{IMX_8BIT, 0x4084, 0x01 },
	{IMX_8BIT, 0x4085, 0x98 },
	{IMX_8BIT, 0x4086, 0x01 },
	{IMX_8BIT, 0x4087, 0x34 },
	{IMX_8BIT, 0x4400, 0x00 },
	/* Global Timing Setting */
	{IMX_8BIT, 0x0830, 0x5F },
	{IMX_8BIT, 0x0831, 0x1F },
	{IMX_8BIT, 0x0832, 0x3F },
	{IMX_8BIT, 0x0833, 0x1F },
	{IMX_8BIT, 0x0834, 0x1F },
	{IMX_8BIT, 0x0835, 0x17 },
	{IMX_8BIT, 0x0836, 0x67 },
	{IMX_8BIT, 0x0837, 0x27 },
	{IMX_8BIT, 0x0839, 0x1F },
	{IMX_8BIT, 0x083A, 0x17 },
	{IMX_8BIT, 0x083B, 0x02 },
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x07 },
	{IMX_8BIT, 0x0203, 0xA4 },
	{IMX_TOK_TERM, 0, 0}
};

/* Video3: H : 820 V : 616 */
static const struct imx_reg imx135_VGA_strong_dvs_30fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03 },
	{IMX_8BIT, 0x0112, 0x0A },
	{IMX_8BIT, 0x0113, 0x0A },
	{IMX_8BIT, 0x0381, 0x01 },
	{IMX_8BIT, 0x0383, 0x01 },
	{IMX_8BIT, 0x0385, 0x01 },
	{IMX_8BIT, 0x0387, 0x01 },
	{IMX_8BIT, 0x0390, 0x01 },
	{IMX_8BIT, 0x0391, 0x44 },
	{IMX_8BIT, 0x0392, 0x00 },
	{IMX_8BIT, 0x0401, 0x02 },
	{IMX_8BIT, 0x0404, 0x00 },
	{IMX_8BIT, 0x0405, 0x14 },
	{IMX_8BIT, 0x4082, 0x00 },
	{IMX_8BIT, 0x4083, 0x00 },
	{IMX_8BIT, 0x7006, 0x04 },
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x07 },
	{IMX_8BIT, 0x0341, 0xA8 },
	{IMX_8BIT, 0x0342, 0x11 },
	{IMX_8BIT, 0x0343, 0xDC },
	{IMX_8BIT, 0x0344, 0x00 },
	{IMX_8BIT, 0x0345, 0x34 },
	{IMX_8BIT, 0x0346, 0x00 },
	{IMX_8BIT, 0x0347, 0x14 },
	{IMX_8BIT, 0x0348, 0x10 },
	{IMX_8BIT, 0x0349, 0x3B },
	{IMX_8BIT, 0x034A, 0x0C },
	{IMX_8BIT, 0x034B, 0x1B },
	{IMX_8BIT, 0x034C, 0x03 },
	{IMX_8BIT, 0x034D, 0x34 },
	{IMX_8BIT, 0x034E, 0x02 },
	{IMX_8BIT, 0x034F, 0x68 },
	{IMX_8BIT, 0x0350, 0x00 },
	{IMX_8BIT, 0x0351, 0x00 },
	{IMX_8BIT, 0x0352, 0x00 },
	{IMX_8BIT, 0x0353, 0x00 },
	{IMX_8BIT, 0x0354, 0x04 },
	{IMX_8BIT, 0x0355, 0x02 },
	{IMX_8BIT, 0x0356, 0x03 },
	{IMX_8BIT, 0x0357, 0x02 },
	{IMX_8BIT, 0x301D, 0x30 },
	{IMX_8BIT, 0x3310, 0x03 },
	{IMX_8BIT, 0x3311, 0x34 },
	{IMX_8BIT, 0x3312, 0x02 },
	{IMX_8BIT, 0x3313, 0x68 },
	{IMX_8BIT, 0x331C, 0x02 },
	{IMX_8BIT, 0x331D, 0x21 },
	{IMX_8BIT, 0x4084, 0x03 },
	{IMX_8BIT, 0x4085, 0x34 },
	{IMX_8BIT, 0x4086, 0x02 },
	{IMX_8BIT, 0x4087, 0x68 },
	{IMX_8BIT, 0x4400, 0x00 },
	/* Global Timing Setting */
	{IMX_8BIT, 0x0830, 0x5F },
	{IMX_8BIT, 0x0831, 0x1F },
	{IMX_8BIT, 0x0832, 0x3F },
	{IMX_8BIT, 0x0833, 0x1F },
	{IMX_8BIT, 0x0834, 0x1F },
	{IMX_8BIT, 0x0835, 0x17 },
	{IMX_8BIT, 0x0836, 0x67 },
	{IMX_8BIT, 0x0837, 0x27 },
	{IMX_8BIT, 0x0839, 0x1F },
	{IMX_8BIT, 0x083A, 0x17 },
	{IMX_8BIT, 0x083B, 0x02 },
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x07 },
	{IMX_8BIT, 0x0203, 0xA4 },
	{IMX_TOK_TERM, 0, 0}
};

/* Video4: H : 1640 V : 1024 */
static const struct imx_reg imx135_WVGA_strong_dvs_30fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03 },
	{IMX_8BIT, 0x0112, 0x0A },
	{IMX_8BIT, 0x0113, 0x0A },
	{IMX_8BIT, 0x0381, 0x01 },
	{IMX_8BIT, 0x0383, 0x01 },
	{IMX_8BIT, 0x0385, 0x01 },
	{IMX_8BIT, 0x0387, 0x01 },
	{IMX_8BIT, 0x0390, 0x01 },
	{IMX_8BIT, 0x0391, 0x22 },
	{IMX_8BIT, 0x0392, 0x00 },
	{IMX_8BIT, 0x0401, 0x02 },
	{IMX_8BIT, 0x0404, 0x00 },
	{IMX_8BIT, 0x0405, 0x14 },
	{IMX_8BIT, 0x4082, 0x00 },
	{IMX_8BIT, 0x4083, 0x00 },
	{IMX_8BIT, 0x7006, 0x04 },
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x07 },
	{IMX_8BIT, 0x0341, 0xA8 },
	{IMX_8BIT, 0x0342, 0x11 },
	{IMX_8BIT, 0x0343, 0xDC },
	{IMX_8BIT, 0x0344, 0x00 },
	{IMX_8BIT, 0x0345, 0x36 },
	{IMX_8BIT, 0x0346, 0x01 },
	{IMX_8BIT, 0x0347, 0x18 },
	{IMX_8BIT, 0x0348, 0x10 },
	{IMX_8BIT, 0x0349, 0x39 },
	{IMX_8BIT, 0x034A, 0x0B },
	{IMX_8BIT, 0x034B, 0x17 },
	{IMX_8BIT, 0x034C, 0x06 },
	{IMX_8BIT, 0x034D, 0x68 },
	{IMX_8BIT, 0x034E, 0x04 },
	{IMX_8BIT, 0x034F, 0x00 },
	{IMX_8BIT, 0x0350, 0x00 },
	{IMX_8BIT, 0x0351, 0x00 },
	{IMX_8BIT, 0x0352, 0x00 },
	{IMX_8BIT, 0x0353, 0x00 },
	{IMX_8BIT, 0x0354, 0x08 },
	{IMX_8BIT, 0x0355, 0x02 },
	{IMX_8BIT, 0x0356, 0x05 },
	{IMX_8BIT, 0x0357, 0x00 },
	{IMX_8BIT, 0x301D, 0x30 },
	{IMX_8BIT, 0x3310, 0x06 },
	{IMX_8BIT, 0x3311, 0x68 },
	{IMX_8BIT, 0x3312, 0x04 },
	{IMX_8BIT, 0x3313, 0x00 },
	{IMX_8BIT, 0x331C, 0x01 },
	{IMX_8BIT, 0x331D, 0xBD },
	{IMX_8BIT, 0x4084, 0x06 },
	{IMX_8BIT, 0x4085, 0x68 },
	{IMX_8BIT, 0x4086, 0x04 },
	{IMX_8BIT, 0x4087, 0x00 },
	{IMX_8BIT, 0x4400, 0x00 },
	/* Global Timing Setting */
	{IMX_8BIT, 0x0830, 0x5F },
	{IMX_8BIT, 0x0831, 0x1F },
	{IMX_8BIT, 0x0832, 0x3F },
	{IMX_8BIT, 0x0833, 0x1F },
	{IMX_8BIT, 0x0834, 0x1F },
	{IMX_8BIT, 0x0835, 0x17 },
	{IMX_8BIT, 0x0836, 0x67 },
	{IMX_8BIT, 0x0837, 0x27 },
	{IMX_8BIT, 0x0839, 0x1F },
	{IMX_8BIT, 0x083A, 0x17 },
	{IMX_8BIT, 0x083B, 0x02 },
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x07 },
	{IMX_8BIT, 0x0203, 0xA4 },
	{IMX_TOK_TERM, 0, 0}
};

/* Video5: H : 1568 V : 880 */
static const struct imx_reg imx135_720p_strong_dvs_30fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03 },
	{IMX_8BIT, 0x0112, 0x0A },
	{IMX_8BIT, 0x0113, 0x0A },
	{IMX_8BIT, 0x0381, 0x01 },
	{IMX_8BIT, 0x0383, 0x01 },
	{IMX_8BIT, 0x0385, 0x01 },
	{IMX_8BIT, 0x0387, 0x01 },
	{IMX_8BIT, 0x0390, 0x01 },
	{IMX_8BIT, 0x0391, 0x22 },
	{IMX_8BIT, 0x0392, 0x00 },
	{IMX_8BIT, 0x0401, 0x02 },
	{IMX_8BIT, 0x0404, 0x00 },
	{IMX_8BIT, 0x0405, 0x15 },
	{IMX_8BIT, 0x4082, 0x00 },
	{IMX_8BIT, 0x4083, 0x00 },
	{IMX_8BIT, 0x7006, 0x04 },
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x07 },
	{IMX_8BIT, 0x0341, 0xA8 },
	{IMX_8BIT, 0x0342, 0x11 },
	{IMX_8BIT, 0x0343, 0xDC },
	{IMX_8BIT, 0x0344, 0x00 },
	{IMX_8BIT, 0x0345, 0x2E },
	{IMX_8BIT, 0x0346, 0x01 },
	{IMX_8BIT, 0x0347, 0x94 },
	{IMX_8BIT, 0x0348, 0x10 },
	{IMX_8BIT, 0x0349, 0x41 },
	{IMX_8BIT, 0x034A, 0x0A },
	{IMX_8BIT, 0x034B, 0x9B },
	{IMX_8BIT, 0x034C, 0x06 },
	{IMX_8BIT, 0x034D, 0x20 },
	{IMX_8BIT, 0x034E, 0x03 },
	{IMX_8BIT, 0x034F, 0x70 },
	{IMX_8BIT, 0x0350, 0x00 },
	{IMX_8BIT, 0x0351, 0x00 },
	{IMX_8BIT, 0x0352, 0x00 },
	{IMX_8BIT, 0x0353, 0x00 },
	{IMX_8BIT, 0x0354, 0x08 },
	{IMX_8BIT, 0x0355, 0x0A },
	{IMX_8BIT, 0x0356, 0x04 },
	{IMX_8BIT, 0x0357, 0x84 },
	{IMX_8BIT, 0x301D, 0x30 },
	{IMX_8BIT, 0x3310, 0x06 },
	{IMX_8BIT, 0x3311, 0x20 },
	{IMX_8BIT, 0x3312, 0x03 },
	{IMX_8BIT, 0x3313, 0x70 },
	{IMX_8BIT, 0x331C, 0x01 },
	{IMX_8BIT, 0x331D, 0xD6 },
	{IMX_8BIT, 0x4084, 0x06 },
	{IMX_8BIT, 0x4085, 0x20 },
	{IMX_8BIT, 0x4086, 0x03 },
	{IMX_8BIT, 0x4087, 0x70 },
	{IMX_8BIT, 0x4400, 0x00 },
	/* Global Timing Setting */
	{IMX_8BIT, 0x0830, 0x5F },
	{IMX_8BIT, 0x0831, 0x1F },
	{IMX_8BIT, 0x0832, 0x3F },
	{IMX_8BIT, 0x0833, 0x1F },
	{IMX_8BIT, 0x0834, 0x1F },
	{IMX_8BIT, 0x0835, 0x17 },
	{IMX_8BIT, 0x0836, 0x67 },
	{IMX_8BIT, 0x0837, 0x27 },
	{IMX_8BIT, 0x0839, 0x1F },
	{IMX_8BIT, 0x083A, 0x17 },
	{IMX_8BIT, 0x083B, 0x02 },
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x07 },
	{IMX_8BIT, 0x0203, 0xA4 },
	{IMX_TOK_TERM, 0, 0}
};

/* Video6: H : 2336V : 1308 */
static const struct imx_reg imx135_1080p_strong_dvs_30fps[] = {
	GROUPED_PARAMETER_HOLD_ENABLE,
	/* Mode setting */
	{IMX_8BIT, 0x0108, 0x03 },
	{IMX_8BIT, 0x0112, 0x0A },
	{IMX_8BIT, 0x0113, 0x0A },
	{IMX_8BIT, 0x0381, 0x01 },
	{IMX_8BIT, 0x0383, 0x01 },
	{IMX_8BIT, 0x0385, 0x01 },
	{IMX_8BIT, 0x0387, 0x01 },
	{IMX_8BIT, 0x0390, 0x00 },
	{IMX_8BIT, 0x0391, 0x11 },
	{IMX_8BIT, 0x0392, 0x00 },
	{IMX_8BIT, 0x0401, 0x00 },
	{IMX_8BIT, 0x0404, 0x00 },
	{IMX_8BIT, 0x0405, 0x10 },
	{IMX_8BIT, 0x4082, 0x01 },
	{IMX_8BIT, 0x4083, 0x01 },
	{IMX_8BIT, 0x7006, 0x04 },
	/* Size setting */
	{IMX_8BIT, 0x0340, 0x07 },
	{IMX_8BIT, 0x0341, 0xA8 },
	{IMX_8BIT, 0x0342, 0x27 },
	{IMX_8BIT, 0x0343, 0x10 },
	{IMX_8BIT, 0x0344, 0x03 },
	{IMX_8BIT, 0x0345, 0xA8 },
	{IMX_8BIT, 0x0346, 0x03 },
	{IMX_8BIT, 0x0347, 0x8C },
	{IMX_8BIT, 0x0348, 0x0C },
	{IMX_8BIT, 0x0349, 0xC7 },
	{IMX_8BIT, 0x034A, 0x08 },
	{IMX_8BIT, 0x034B, 0xA7 },
	{IMX_8BIT, 0x034C, 0x09 },
	{IMX_8BIT, 0x034D, 0x20 },
	{IMX_8BIT, 0x034E, 0x05 },
	{IMX_8BIT, 0x034F, 0x1C },
	{IMX_8BIT, 0x0350, 0x00 },
	{IMX_8BIT, 0x0351, 0x00 },
	{IMX_8BIT, 0x0352, 0x00 },
	{IMX_8BIT, 0x0353, 0x00 },
	{IMX_8BIT, 0x0354, 0x09 },
	{IMX_8BIT, 0x0355, 0x20 },
	{IMX_8BIT, 0x0356, 0x05 },
	{IMX_8BIT, 0x0357, 0x1C },
	{IMX_8BIT, 0x301D, 0x30 },
	{IMX_8BIT, 0x3310, 0x09 },
	{IMX_8BIT, 0x3311, 0x20 },
	{IMX_8BIT, 0x3312, 0x05 },
	{IMX_8BIT, 0x3313, 0x1C },
	{IMX_8BIT, 0x331C, 0x00 },
	{IMX_8BIT, 0x331D, 0xB4 },
	{IMX_8BIT, 0x4084, 0x00 },
	{IMX_8BIT, 0x4085, 0x00 },
	{IMX_8BIT, 0x4086, 0x00 },
	{IMX_8BIT, 0x4087, 0x00 },
	{IMX_8BIT, 0x4400, 0x00 },
	/* Global Timing Setting */
	{IMX_8BIT, 0x0830, 0x77 },
	{IMX_8BIT, 0x0831, 0x2F },
	{IMX_8BIT, 0x0832, 0x4F },
	{IMX_8BIT, 0x0833, 0x2F },
	{IMX_8BIT, 0x0834, 0x2F },
	{IMX_8BIT, 0x0835, 0x37 },
	{IMX_8BIT, 0x0836, 0xA7 },
	{IMX_8BIT, 0x0837, 0x37 },
	{IMX_8BIT, 0x0839, 0x1F },
	{IMX_8BIT, 0x083A, 0x17 },
	{IMX_8BIT, 0x083B, 0x02 },
	/* Integration Time Setting */
	{IMX_8BIT, 0x0202, 0x07 },
	{IMX_8BIT, 0x0203, 0xA4 },
	{IMX_TOK_TERM, 0, 0}
};

static const struct imx_reg imx135_2336x1320_30fps[] = {
	/* Video */
	/* H : 2336 */
	/* V : 1320 */
	/* clock setting */
	{ IMX_8BIT, 0x011E, 0x13 },
	{ IMX_8BIT, 0x011F, 0x33 },
	{ IMX_8BIT, 0x0301, 0x05 },
	{ IMX_8BIT, 0x0303, 0x01 },
	{ IMX_8BIT, 0x0305, 0x0B },
	{ IMX_8BIT, 0x0309, 0x05 },
	{ IMX_8BIT, 0x030B, 0x01 },
	{ IMX_8BIT, 0x030C, 0x01 },
	{ IMX_8BIT, 0x030D, 0x91 },
	{ IMX_8BIT, 0x030E, 0x01 },
	{ IMX_8BIT, 0x3A06, 0x11 },
	/* mode setting */
	{ IMX_8BIT, 0x0108, 0x03 },
	{ IMX_8BIT, 0x0112, 0x0A },
	{ IMX_8BIT, 0x0113, 0x0A },
	{ IMX_8BIT, 0x0381, 0x01 },
	{ IMX_8BIT, 0x0383, 0x01 },
	{ IMX_8BIT, 0x0385, 0x01 },
	{ IMX_8BIT, 0x0387, 0x01 },
	{ IMX_8BIT, 0x0390, 0x00 },
	{ IMX_8BIT, 0x0391, 0x11 },
	{ IMX_8BIT, 0x0392, 0x00 },
	{ IMX_8BIT, 0x0401, 0x02 },
	{ IMX_8BIT, 0x0404, 0x00 },
	{ IMX_8BIT, 0x0405, 0x1C },
	{ IMX_8BIT, 0x4082, 0x00 },
	{ IMX_8BIT, 0x4083, 0x00 },
	{ IMX_8BIT, 0x7006, 0x04 },
	/* function setting */
	{ IMX_8BIT, 0x0700, 0x00 },
	{ IMX_8BIT, 0x3A63, 0x00 },
	{ IMX_8BIT, 0x4100, 0xF8 },
	{ IMX_8BIT, 0x4203, 0xFF },
	{ IMX_8BIT, 0x4344, 0x00 },
	{ IMX_8BIT, 0x441C, 0x01 },
	/* size setting */
	{ IMX_8BIT, 0x0340, 0x09 },
	{ IMX_8BIT, 0x0341, 0x1E },
	{ IMX_8BIT, 0x0342, 0x11 },
	{ IMX_8BIT, 0x0343, 0xDC },
	{ IMX_8BIT, 0x0344, 0x00 },
	{ IMX_8BIT, 0x0345, 0x3C },
	{ IMX_8BIT, 0x0346, 0x01 },
	{ IMX_8BIT, 0x0347, 0x94 },
	{ IMX_8BIT, 0x0348, 0x10 },
	{ IMX_8BIT, 0x0349, 0x33 },
	{ IMX_8BIT, 0x034A, 0x0A },
	{ IMX_8BIT, 0x034B, 0x9B },
	{ IMX_8BIT, 0x034C, 0x09 },
	{ IMX_8BIT, 0x034D, 0x20 },
	{ IMX_8BIT, 0x034E, 0x05 },
	{ IMX_8BIT, 0x034F, 0x28 },
	{ IMX_8BIT, 0x0350, 0x00 },
	{ IMX_8BIT, 0x0351, 0x00 },
	{ IMX_8BIT, 0x0352, 0x00 },
	{ IMX_8BIT, 0x0353, 0x00 },
	{ IMX_8BIT, 0x0354, 0x0F },
	{ IMX_8BIT, 0x0355, 0xF8 },
	{ IMX_8BIT, 0x0356, 0x09 },
	{ IMX_8BIT, 0x0357, 0x06 },
	{ IMX_8BIT, 0x301D, 0x30 },
	{ IMX_8BIT, 0x3310, 0x09 },
	{ IMX_8BIT, 0x3311, 0x20 },
	{ IMX_8BIT, 0x3312, 0x05 },
	{ IMX_8BIT, 0x3313, 0x28 },
	{ IMX_8BIT, 0x331C, 0x04 },
	{ IMX_8BIT, 0x331D, 0xE2 },
	{ IMX_8BIT, 0x4084, 0x09 },
	{ IMX_8BIT, 0x4085, 0x20 },
	{ IMX_8BIT, 0x4086, 0x05 },
	{ IMX_8BIT, 0x4087, 0x28 },
	{ IMX_8BIT, 0x4400, 0x00 },
	/* global timing setting */
	{ IMX_8BIT, 0x0830, 0x77 },
	/* { IMX_8BIT, { 0x0831 }, 0x2F }, */
	{ IMX_8BIT, 0x0831, 0x19 },
	{ IMX_8BIT, 0x0832, 0x4F },
	{ IMX_8BIT, 0x0833, 0x37 },
	{ IMX_8BIT, 0x0834, 0x2F },
	{ IMX_8BIT, 0x0835, 0x37 },
	{ IMX_8BIT, 0x0836, 0xAF },
	{ IMX_8BIT, 0x0837, 0x37 },
	{ IMX_8BIT, 0x0839, 0x1F },
	{ IMX_8BIT, 0x083A, 0x17 },
	{ IMX_8BIT, 0x083B, 0x02 },
	/* integration timing setting */
	{ IMX_8BIT, 0x0202, 0x09 },
	{ IMX_8BIT, 0x0203, 0x1A },
	/* gain setting */
	{ IMX_8BIT, 0x0205, 0x00 },
	{ IMX_8BIT, 0x020E, 0x01 },
	{ IMX_8BIT, 0x020F, 0x00 },
	{ IMX_8BIT, 0x0210, 0x01 },
	{ IMX_8BIT, 0x0211, 0x00 },
	{ IMX_8BIT, 0x0212, 0x01 },
	{ IMX_8BIT, 0x0213, 0x00 },
	{ IMX_8BIT, 0x0214, 0x01 },
	{ IMX_8BIT, 0x0215, 0x00 },
	/* HDR setting */
	{ IMX_8BIT, 0x0230, 0x00 },
	{ IMX_8BIT, 0x0231, 0x00 },
	{ IMX_8BIT, 0x0233, 0x00 },
	{ IMX_8BIT, 0x0234, 0x00 },
	{ IMX_8BIT, 0x0235, 0x40 },
	{ IMX_8BIT, 0x0238, 0x00 },
	{ IMX_8BIT, 0x0239, 0x04 },
	{ IMX_8BIT, 0x023B, 0x00 },
	{ IMX_8BIT, 0x023C, 0x01 },
	{ IMX_8BIT, 0x33B0, 0x04 },
	{ IMX_8BIT, 0x33B1, 0x00 },
	{ IMX_8BIT, 0x33B3, 0x00 },
	{ IMX_8BIT, 0x33B4, 0x01 },
	{ IMX_8BIT, 0x3800, 0x00 },

	{ IMX_TOK_TERM, 0, 0}
};

struct imx_resolution imx_res_preview[] = {

	{
		.desc = "imx135_PREVIEW_848x616_30fps",
		.regs = imx135_PREVIEW_848x616_30fps,
		.width = 848,
		.height = 616,
		.fps = 30,
		.pixels_per_line = 4572,
		.lines_per_frame = 1960,
		.bin_factor_x = 2,
		.bin_factor_y = 2,
		.used = 0,
	},
	{
		.desc = "imx135_PREVIEW_WIDE_PREVIEW_30fps",
		.regs = imx135_PREVIEW_WIDE_PREVIEW_30fps,
		.width = 1280,
		.height = 720,
		.fps = 30,
		.pixels_per_line = 4572,
		.lines_per_frame = 1960,
		.bin_factor_x = 2,
		.bin_factor_y = 2,
		.used = 0,
	},
	{
		.desc = "imx135_720p_30fps",
		.regs = imx135_720p_30fps,
		.width = 1296,
		.height = 736,
		.fps = 30,
		.pixels_per_line = 4572,
		.lines_per_frame = 2650,
		.bin_factor_x = 1,
		.bin_factor_y = 1,
		.used = 0,
	},
	{
		.desc = "imx135_1M_16_9_30fps",
		.regs = imx135_1M_16_9_30fps,
		.width = 1040,
		.height = 592,
		.fps = 30,
		.pixels_per_line = 4572,
		.lines_per_frame = 2650,
		.bin_factor_x = 2,
		.bin_factor_y = 2,
		.used = 0,
	},
	{
		.desc = "imx135_1M_4_3_30fps",
		.regs = imx135_1M_4_3_30fps,
		.width = 1640,
		.height = 1232,
		.fps = 30,
		.pixels_per_line = 4572,
		.lines_per_frame = 2650,
		.bin_factor_x = 1,
		.bin_factor_y = 1,
		.used = 0,
	},
	{
		.desc = "imx135_1080p_30fps",
		.regs = imx135_1080p_30fps,
		.width = 1936,
		.height = 1104,
		.fps = 30,
		.pixels_per_line = 4572,
		.lines_per_frame = 2650,
		.bin_factor_x = 1,
		.bin_factor_y = 1,
		.used = 0,
	},
	{
		.desc = "imx135_PREVIEW_1632x1224_30fps",
		.regs = imx135_PREVIEW_1632x1224_30fps,
		.width = 1632,
		.height = 1224,
		.fps = 30,
		.pixels_per_line = 4572,
		.lines_per_frame = 1960,
		.bin_factor_x = 2,
		.bin_factor_y = 2,
		.used = 0,
	},
	{
		.desc = "imx135_2M_30fps",
		.regs = imx135_2M_30fps,
		.width = 1640,
		.height = 1232,
		.fps = 30,
		.pixels_per_line = 4572, /* consistent with regs arrays */
		.lines_per_frame = 2650, /* consistent with regs arrays */
		.bin_factor_x = 1,
		.bin_factor_y = 1,
		.used = 0,
	},
	{
		.desc = "imx135_5M_19fps",
		.regs = imx135_5M_STILL_19fps,
		.width = 2576,
		.height = 1936,
		.fps = 19,
		.pixels_per_line = 4572, /* consistent with regs arrays */
		.lines_per_frame = 3050, /* consistent with regs arrays */
		.bin_factor_x = 0,
		.bin_factor_y = 0,
		.used = 0,
	},
	{
		.desc = "imx135_6M_STILL_20fps",
		.regs = imx135_6M_STILL_20fps,
		.width = 3280,
		.height = 1852,
		.fps = 20,
		.pixels_per_line = 4696, /* consistent with regs arrays */
		.lines_per_frame = 2318, /* consistent with regs arrays */
		.bin_factor_x = 0,
		.bin_factor_y = 0,
		.used = 0,
	},
	{
		.desc = "imx135_8M_15fps",
		.regs = imx135_8M_STILL_15fps,
		.width = 3280,
		.height = 2464,
		.fps = 15,
		.pixels_per_line = 4572, /* consistent with regs arrays */
		.lines_per_frame = 2620, /* consistent with regs arrays */
		.bin_factor_x = 0,
		.bin_factor_y = 0,
		.used = 0,
	},
	{
		.desc = "imx135_13M_24fps",
		.regs = imx135_13M_24fps,
		.width = 4208,
		.height = 3120,
		.fps = 24,
		.pixels_per_line = 4572,
		.lines_per_frame = 3310,
		.bin_factor_x = 0,
		.bin_factor_y = 0,
		.used = 0,
	},
};

struct imx_resolution imx_res_still[] = {
	{
		.desc = "imx135_QCIF_30fps",
		.regs = imx135_QCIF_30fps,
		.width = 192,
		.height = 160,
		.fps = 30,
		.pixels_per_line = 4572, /* consistent with regs arrays */
		.lines_per_frame = 2650, /* consistent with regs arrays */
		.bin_factor_x = 2,
		.bin_factor_y = 2,
		.used = 0,
	},
	{
		.desc = "imx135_QVGA_30fps",
		.regs = imx135_QVGA_30fps,
		.width = 336,
		.height = 256,
		.fps = 30,
		.pixels_per_line = 4572, /* consistent with regs arrays */
		.lines_per_frame = 2650, /* consistent with regs arrays */
		.bin_factor_x = 2,
		.bin_factor_y = 2,
		.used = 0,
	},
	{
		.desc = "imx135_CIF_30fps",
		.regs = imx135_CIF_30fps,
		.width = 368,
		.height = 304,
		.fps = 30,
		.pixels_per_line = 4572, /* consistent with regs arrays */
		.lines_per_frame = 2650, /* consistent with regs arrays */
		.bin_factor_x = 2,
		.bin_factor_y = 2,
		.used = 0,
	},
	{
		.desc = "imx135_VGA_30fps",
		.regs = imx135_VGA_30fps,
		.width = 656,
		.height = 496,
		.fps = 30,
		.pixels_per_line = 4572, /* consistent with regs arrays */
		.lines_per_frame = 2650, /* consistent with regs arrays */
		.bin_factor_x = 2,
		.bin_factor_y = 2,
		.used = 0,
	},
	{
		.desc = "imx135_D1_30fps",
		.regs = imx135_D1_30fps,
		.width = 736,
		.height = 496,
		.fps = 30,
		.pixels_per_line = 4572, /* consistent with regs arrays */
		.lines_per_frame = 2650, /* consistent with regs arrays */
		.bin_factor_x = 2,
		.bin_factor_y = 2,
		.used = 0,
	},
	{
		.desc = "imx135_VGA_dvs_30fps",
		.regs = imx135_VGA_dvs_30fps,
		.width = 820,
		.height = 616,
		.fps = 30,
		.pixels_per_line = 4572,
		.lines_per_frame = 2650,
		.bin_factor_x = 2,
		.bin_factor_y = 2,
		.used = 0,
	},
	{
		.desc = "imx135_1M_16_9_30fps",
		.regs = imx135_1M_16_9_30fps,
		.width = 1040,
		.height = 592,
		.fps = 30,
		.pixels_per_line = 4572, /* consistent with regs arrays */
		.lines_per_frame = 2650, /* consistent with regs arrays */
		.bin_factor_x = 2,
		.bin_factor_y = 2,
		.used = 0,
	},
	{
		.desc = "imx135_1M_4_3_30fps",
		.regs = imx135_1M_4_3_30fps,
		.width = 1040,
		.height = 784,
		.fps = 30,
		.pixels_per_line = 4572, /* consistent with regs arrays */
		.lines_per_frame = 2650, /* consistent with regs arrays */
		.bin_factor_x = 1,
		.bin_factor_y = 1,
		.used = 0,
	},
	{
		.desc = "imx135_720p_30fps",
		.regs = imx135_720p_30fps,
		.width = 1296,
		.height = 736,
		.fps = 30,
		.pixels_per_line = 4572, /* consistent with regs arrays */
		.lines_per_frame = 2650, /* consistent with regs arrays */
		.bin_factor_x = 1,
		.bin_factor_y = 1,
		.used = 0,
	},
	{
		.desc = "imx135_1600x1200_30fps",
		.regs = imx135_1600x1200_30fps,
		.width = 1632,
		.height = 1224,
		.fps = 30,
		.pixels_per_line = 4572, /* consistent with regs arrays */
		.lines_per_frame = 2650, /* consistent with regs arrays */
		.bin_factor_x = 1,
		.bin_factor_y = 1,
		.used = 0,
	},
	{
		.desc = "imx135_1080p_30fps",
		.regs = imx135_1080p_30fps,
		.width = 1936,
		.height = 1104,
		.fps = 30,
		.pixels_per_line = 4572, /* consistent with regs arrays */
		.lines_per_frame = 2650, /* consistent with regs arrays */
		.bin_factor_x = 1,
		.bin_factor_y = 1,
		.used = 0,
	},
	{
		.desc = "imx135_3M_30fps",
		.regs = imx135_3M_30fps,
		.width = 2064,
		.height = 1552,
		.fps = 30,
		.pixels_per_line = 4572, /* consistent with regs arrays */
		.lines_per_frame = 2650, /* consistent with regs arrays */
		.bin_factor_x = 1,
		.bin_factor_y = 1,
		.used = 0,
	},
	{
		.desc = "imx135_5M_26fps",
		.regs = imx135_5M_STILL_19fps,
		.width = 2576	,
		.height = 1936,
		.fps = 26,
		.pixels_per_line = 4572, /* consistent with regs arrays */
		.lines_per_frame = 3050, /* consistent with regs arrays */
		.bin_factor_x = 0,
		.bin_factor_y = 0,
		.used = 0,
	},
	{
		.desc = "imx135_6M_STILL_20fps",
		.regs = imx135_6M_STILL_20fps,
		.width = 3280,
		.height = 1852,
		.fps = 20,
		.pixels_per_line = 4696, /* consistent with regs arrays */
		.lines_per_frame = 2318, /* consistent with regs arrays */
		.bin_factor_x = 0,
		.bin_factor_y = 0,
		.used = 0,
	},
	{
		.desc = "imx135_8M_3fps",
		.regs = imx135_8M_3fps,
		.width = 3280,
		.height = 2464,
		.fps = 3,
		.pixels_per_line = 10000, /* consistent with regs arrays */
		.lines_per_frame = 2620, /* consistent with regs arrays */
		.bin_factor_x = 0,
		.bin_factor_y = 0,
		.used = 0,
	},
	{
		.desc = "imx135_13M_3fps",
		.regs = imx135_13M_3fps,
		.width = 4208,
		.height = 3120,
		.fps = 3,
		.pixels_per_line = 0x3000,
		.lines_per_frame = 0x2000,
		.bin_factor_x = 0,
		.bin_factor_y = 0,
		.used = 0,
	},
};

struct imx_resolution imx_res_video[] = {
	{
		.desc = "imx135_2336x1320_30fps",
		.regs = imx135_2336x1320_30fps,
		.width = 2336,
		.height = 1320,
		.fps = 26,
		.pixels_per_line = 4572, /* consistent with regs arrays */
		.lines_per_frame = 2334, /* consistent with regs arrays */
		.bin_factor_x = 0,
		.bin_factor_y = 0,
		.used = 0,
	},
};

struct imx_resolution imx_res_video_nodvs[] = {
	{
		.desc = "imx135_QCIF_30fps",
		.regs = imx135_QCIF_30fps,
		.width = 192,
		.height = 160,
		.fps = 30,
		.pixels_per_line = 4572, /* consistent with regs arrays */
		.lines_per_frame = 2650, /* consistent with regs arrays */
		.bin_factor_x = 2,
		.bin_factor_y = 2,
		.used = 0,
	},
	{
		.desc = "imx135_QVGA_30fps",
		.regs = imx135_QVGA_30fps,
		.width = 336,
		.height = 256,
		.fps = 30,
		.pixels_per_line = 4572, /* consistent with regs arrays */
		.lines_per_frame = 2650, /* consistent with regs arrays */
		.bin_factor_x = 2,
		.bin_factor_y = 2,
		.used = 0,
	},
	{
		.desc = "imx135_CIF_30fps",
		.regs = imx135_CIF_30fps,
		.width = 368,
		.height = 304,
		.fps = 30,
		.pixels_per_line = 4572, /* consistent with regs arrays */
		.lines_per_frame = 2650, /* consistent with regs arrays */
		.bin_factor_x = 2,
		.bin_factor_y = 2,
		.used = 0,
	},
	{
		.desc = "imx135_VGA_30fps",
		.regs = imx135_VGA_30fps,
		.width = 656,
		.height = 496,
		.fps = 30,
		.pixels_per_line = 4572, /* consistent with regs arrays */
		.lines_per_frame = 2650, /* consistent with regs arrays */
		.bin_factor_x = 2,
		.bin_factor_y = 2,
		.used = 0,
	},
	{
		.desc = "imx135_D1_30fps",
		.regs = imx135_D1_30fps,
		.width = 736,
		.height = 496,
		.fps = 30,
		.pixels_per_line = 4572, /* consistent with regs arrays */
		.lines_per_frame = 2650, /* consistent with regs arrays */
		.bin_factor_x = 2,
		.bin_factor_y = 2,
		.used = 0,
	},
	{
		.desc = "imx135_720p_30fps",
		.regs = imx135_720p_30fps,
		.width = 1296,
		.height = 736,
		.fps = 30,
		.pixels_per_line = 4572, /* consistent with regs arrays */
		.lines_per_frame = 2650, /* consistent with regs arrays */
		.bin_factor_x = 1,
		.bin_factor_y = 1,
		.used = 0,
	},
	{
		.desc = "imx135_1080p_30fps",
		.regs = imx135_1080p_30fps,
		.width = 1936,
		.height = 1104,
		.fps = 30,
		.pixels_per_line = 4572, /* consistent with regs arrays */
		.lines_per_frame = 2650, /* consistent with regs arrays */
		.bin_factor_x = 1,
		.bin_factor_y = 1,
		.used = 0,
	},
};

#define N_RES_PREVIEW (ARRAY_SIZE(imx_res_preview))
#define N_RES_STILL (ARRAY_SIZE(imx_res_still))
#define N_RES_VIDEO (ARRAY_SIZE(imx_res_video))

struct imx_resolution *imx_res = imx_res_preview;
static int N_RES = N_RES_PREVIEW;


#endif

