/*
 * STM32L051 driver <-> HAL interface.
 *
 * Copyright (c) 2016 Intel Corporation. All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License version
 * 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 */

#ifndef __STM32L051_H__
#define __STM32L051_H__

#ifdef __KERNEL__
#include <linux/ioctl.h>
#include <linux/types.h>
#endif

#pragma pack(push, 4)

/**
 * name of the device in /dev
 */
#define STM32L051_DEVICE_FILE_NAME "stm32l051"

/**
 * Sensor type
 * This enum represents the sensors supported by the devices.
 * @STM32L051_SENSOR_TYPE_ACCELOROMETER:
 * @STM32L051_SENSOR_TYPE_GYROSCOPE:
 * @STM32L051_SENSOR_TYPE_DEPTH_CAMERA:
 * @STM32L051_SENSOR_TYPE_MOTION_CAMERA:
 * @STM32L051_SENSOR_TYPE_EXTERNAL:
 */
enum stm32l051_sensor_type {
	STM32L051_SENSOR_TYPE_ACCELOROMETER,
	STM32L051_SENSOR_TYPE_GYROSCOPE,
	STM32L051_SENSOR_TYPE_DEPTH_CAMERA,
	STM32L051_SENSOR_TYPE_MOTION_CAMERA,
	STM32L051_SENSOR_TYPE_EXTERNAL
};

/**
 * Accelerometer bandwidth
 * @125 HZ
 * @250 HZ
 */
enum stm32l051_acc_bw {
	STM32L051_ACC_BW_125_HZ = 125,
	STM32L051_ACC_BW_250_HZ = 250,
	STM32L051_ACC_BW_500_HZ = 500,
	STM32L051_ACC_BW_1000_HZ = 1000,
	STM32L051_ACC_BW_2000_HZ = 2000,
	STM32L051_ACC_BW_INVALID = 0xFFFF
};

/**
 * Accelerometer range
 * @4
 * @8
 * @16
 */
enum stm32l051_acc_range {
	STM32L051_ACC_RANGE_4 = 4,
	STM32L051_ACC_RANGE_8 = 8,
	STM32L051_ACC_RANGE_16 = 16,
	STM32L051_ACC_RANGE_INVALID = 0xFFFF
};

/**
 * Gyroscope bandwidth
 * @116 HZ
 * @230 HZ
 */
enum stm32l051_gyro_bw {
	STM32L051_GYRO_BW_100_HZ = 100,
	STM32L051_GYRO_BW_200_HZ = 200,
	STM32L051_GYRO_BW_400_HZ = 400,
	STM32L051_GYRO_BW_1000_HZ = 1000,
	STM32L051_GYRO_BW_2000_HZ = 2000,
	STM32L051_GYRO_BW_INVALID = 0xFFFF
};

/**
 * Gyroscope range
 * @1000
 * @2000
 */
enum stm32l051_gyro_range {
	STM32L051_GYRO_RANGE_1000 = 1000,
	STM32L051_GYRO_RANGE_2000 = 2000,
	STM32L051_GYRO_RANGE_INVALID = 0xFFFF,
};

/**
 * IMU State
 * @Stopped
 * @Paused
 * @Running
 */
enum stm32l051_state {
	STM32L051_STOP,
	STM32L051_PAUSE,
	STM32L051_START
};

/**
 * Sensors Configurations
 * The following structs represent configuration of a specific sensor.
 */

/**
 * external sensor configuration
 * @enabled: Sensor is enabled / disabled
 * @falling edge: True means falling, False means rising.
 */
struct stm32l051_external_sensor_cfg {
	__u32 falling_edge; /* 1: Falling, 0: rising */
};

/**
 * vectorial sensor configuration (accelerometer, gyroscope).
 * @range
 * @sampling rate
 */
struct stm32l051_vec_sensor_cfg {
	__u32 range;
	__u32 bandwidth;
};

/**
 * Sensor vectorial data
 * @x:
 * @y:
 * @z:
 */
struct stm32l051_sensor_vec_data {
	__s16 x;
	__s16 y;
	__s16 z;
};

/**
 * Sensor data
 * @handle: The sensor handle (@see get sensor list)
 * @hw_time_stamp:
 * @system time stamp: (not used).
 * @handle (identifies the sensor, @see stm32l051_sensor_t).
 * @hw frame id:
 * @sequence number: (not used).
 * @sensor data:
 */
struct stm32l051_sensor_data {
	__u64 hw_time_stamp;		/* set by hw */
	__u64 system_time_stamp;	/* set by driver */
	__u32 handle;			/* source id */
	__u32 hw_frame_id;		/* set by hw */
	__u32 sequence_num;		/* driver numbering */
	union {
		struct stm32l051_sensor_vec_data acc_data;
		struct stm32l051_sensor_vec_data gyro_data;
	};
};

/**
 * Sensor info
 * @name:
 * @type: @see enum stm32l051_sensor_type.
 * @handle: The sensor handle (@see get sensor list)
 * @enabled: (1 = enabled, 0 = disabled).
 * @type specific sensor configuration.
 */
 #define STM32L051_SENSOR_NAME_MAX_LENGTH 32
struct stm32l051_sensor_t {
	__u32 handle;
	char name[STM32L051_SENSOR_NAME_MAX_LENGTH];
	enum stm32l051_sensor_type type;
	__u32 enabled;
	union {
		struct stm32l051_external_sensor_cfg external_sensor_cfg;
		struct stm32l051_vec_sensor_cfg vec_sensor_cfg;
	} configuration;
};

/**
 * IOCTLS
 * The following structs define the IOCTL data structures.
 */

/**
 * external sensor configuration IOCTL data.
 * STM32L051_IOCTL_SET_EXT_SENSOR_CONFIG
 * @handle: The sensor handle (@see get sensor list)
 * @enabled: (1 = enabled, 0 = disabled).
 * @configuration:
 */
struct stm32l051_external_sensor_cfg_cmd {
	__u32 handle;
	__u32 enabled;
	struct stm32l051_external_sensor_cfg configuration;
};

/**
 * HW Sync configuration IOCTL data.
 * STM32L051_IOCTL_HW_SYNC_CONFIG
 * @enabled: (1 = enabled, 0 = disabled).
 */
struct stm32l051_hw_sync_cfg_cmd {
	__u32 enabled;
};

/**
 * vectorial sensor configuration IOCTL data.
 * STM32L051_IOCTL_SET_ACCL_CONFIG
 * STM32L051_IOCTL_SET_GYRO_CONFIG
 * @handle: The sensor handle (@see get sensor list)
 * @enabled: (1 = enabled, 0 = disabled).
 * @configuration:
 */
struct stm32l051_vec_sensor_cfg_cmd {
	__u32 handle;
	__u32 enabled;
	struct stm32l051_vec_sensor_cfg configuration;
};

/**
 * vectorial sensor configuration IOCTL data.
 * STM32L051_IOCTL_SET_DEPTH_CAM_CONFIG
 * STM32L051_IOCTL_SET_MOTION_CAM_CONFIG
 * @handle: The sensor handle (@see get sensor list)
 * @enabled: (1 = enabled, 0 = disabled).
 * @configuration:
 */
struct stm32l051_camera_sensor_cfg_cmd {
	__u32 handle;
	__u32 enabled;
};

#define MAX_LTR_FIFO_LEN 0x7F
/**
 * LTR (latency tolerance) configuration IOCTL data.
 * STM32L051_IOCTL_SET_LTR_CONFIG
 * The fields below define the time of the most old entry in the HW FIFO
 * to trigger LTR threshold reached interrupt.
 * @Timestamp FIFO: (range 0::MAX_LTR_FIFO_LEN)
 * @IMU FIFO: (range 0::MAX_LTR_FIFO_LEN)
 */
struct stm32l051_ltr_cfg_cmd {
	__u32 ts_fifo_length;  /* range: 0::MAX_LTR_FIFO_LEN */
	__u32 imu_fifo_length; /* range: 0::MAX_LTR_FIFO_LEN */
};

/**
 * Time stamp configuration IOCTL data.
 * STM32L051_IOCTL_SET_TIMESTAMP_CONFIG
 * @reset_value: Time stamp to start from.
 * @prescaler: any value between 1 - 65536
 * prescaler is used to lower the frequency.
 *
 */
#define MIN_TIMESTAMP_PRESCALER 1
#define MAX_TIMESTAMP_PRESCALER 65536
struct stm32l051_timestamp_cfg_cmd {
	__u32 reset_value;
	__u32 prescaler;
};




/**
 * Get UC timestamp IOCTL data.
 * STM32L051_IOCTL_GET_TIMESTAMP
 *
 */
struct stm32l051_uc_timestamp {
	__u64 uc_time;
	__u64 sys_time;
    __u32 call_time;
};


/**
 * Device info IOCTL data.
 * This IOCTL retrieves the common device data.
 * For sensor specific, see stm32l051_sensors_list_cmd.
 * time stamp reset value (@see timestamp config).
 * timestamp prescaler (@see timestamp config).
 * @driver version.
 * @FW version.
 * @num sensors.
 * @hw sync enabled.
 * @TS FIFO LTR (@see stm32l051_ltr_cfg_cmd).
 * @IMU FIFO LTR (@see stm32l051_ltr_cfg_cmd).
 */
struct stm32l051_device_info_cmd {
	__u32 timestamp_reset_value;
	__u32 timestamp_prescaler;
	__u32 driver_version;
	__u32 fw_version;
	__u32 num_sensors;
	__u32 hw_sync_enabled;
	__u32 ts_fifo_length;
	__u32 imu_fifo_length;
	enum stm32l051_state state;
};

/**
 * sensors list IOCTL data.
 * @num sensors
 * @array of stm32l051_sensor_t (@see num_sensors in stm32l051_device_info_cmd).
 */
struct stm32l051_sensors_list_cmd {
	__u32 num_sensors;
	struct stm32l051_sensor_t sensors[1];
};

/**
 * Set State IOCTL data.
 */
struct stm32l051_set_state_cmd {
	enum stm32l051_state state;
};

/**
 * FW Update IOCTL data.
 * @image length
 * @image
 */
struct stm32l051_fw_update_cmd {
	__u32 length;
	char buffer[1];
};

/**
 * EEprom access data.
 * @address in EEPROM.
 * @data (length must be STM32L051_EEPROM_BLOCK_LENGTH)
 */
#define STM32L051_EEPROM_BLOCK_LENGTH (252)
struct stm32l051_eeprom_access_cmd {
	__u16 address;
	unsigned char buffer[STM32L051_EEPROM_BLOCK_LENGTH];
};

/**
 * This enum is translated to bit mask of errors (bit number).
 */
enum stm32l051_error_t {
	STM32L051_ERROR_DRV_FIFO_FULL = 0x1,
	STM32L051_ERROR_FW_ERROR = 0x2,
	STM32L051_ERROR_FRAME_LOSS_DETECTED = 0x4,
	STM32L051_ERROR_I2C_TIMEOUT = 0x8,
	STM32L051_ERROR_I2C_REMOTE_IO = 0x10,
	STM32L051_ERROR_I2C_UNSPECIFIED_FAILURE = 0x20,
	STM32L051_ERROR_FW_UPDATE_FAILED_WRITE = 0x40,
	STM32L051_ERROR_FW_UPDATE_FAILED_FW_CORRUPTED = 0x80,
};

/**
 * Get device error data.
 * @drv_error_mask: Bit mask of driver errors (@see enum stm32l051_error_t).
 *					this field is clear on read.
 */
struct stm32l051_get_device_error_cmd {
	__u32 drv_error_mask;
};

/*
 * IOCTL definitions
 */

/* Use 'u' as magic number */
#define STM32L051_IOCTL_MAGIC 'u'

/**
 * IOCTL INIT IMU error codes.
 * @FW Corrupted: This error is returned if FW boot loader failed
 *     (e.g. due to checksum error).
 * @Failed to power up
 * @Device already intialized (can't intialize the device when it is already intialized).
 * @Driver internal error (such as memory allocation failure, etc...)
 */
enum stm32l051_init_cmd_status {
	STM32L051_INIT_STATUS_SUCCESS,
	STM32L051_INIT_STATUS_FW_CORRUPTED,
	STM32L051_INIT_STATUS_FAILED_TO_POWER_UP,
	STM32L051_INIT_STATUS_DEVICE_ALREADY_INITIALIZED,
	STM32L051_INIT_STATUS_INTERNAL_ERROR,
};

/**
 * Initializes the IMU.
 * This IOCTL MUST be called before any other IOCTL.
 * This IOCTL powers up the device and checks FW validity.
 * The IOCTL returns 0 for success.
 * @see stm32l051_init_cmd_status for error codes.
 */
#define STM32L051_IOCTL_INIT \
	_IOW(STM32L051_IOCTL_MAGIC, 0, enum stm32l051_init_cmd_status *)

/**
 * Uninitializes the IMU.
 * there is no data to this IOCTL (int is used as a dummy).
 */
#define STM32L051_IOCTL_UNINIT _IOW(STM32L051_IOCTL_MAGIC, 1, int)

/**
 * Returns the device info
 */
#define STM32L051_IOCTL_GET_DEVICE_INFO \
	_IOR(STM32L051_IOCTL_MAGIC, 2, struct stm32l051_device_info_cmd *)

/**
 * Returns the sensors list
 * The IOCTL shall fail if the number of sensors (implies length of sensors
 * array) is smaller than num sensors (see get device info).
 * This IOCTL returns the sensors list.
 * A sensor entry contains the module ID.
 */
#define STM32L051_IOCTL_GET_SENSORS_LIST \
	_IOR(STM32L051_IOCTL_MAGIC, 3, struct stm32l051_sensors_list_cmd *)

/**
 * Set device state
 */
#define STM32L051_IOCTL_SET_STATE \
	_IOW(STM32L051_IOCTL_MAGIC, 4, struct stm32l051_set_state_cmd)

/**
 * Configure Accelerator
 */
#define STM32L051_IOCTL_SET_ACCL_CONFIG \
_IOW(STM32L051_IOCTL_MAGIC, 5, struct stm32l051_vec_sensor_cfg_cmd *)

/**
 * Configure Gyroscope
 */
#define STM32L051_IOCTL_SET_GYRO_CONFIG \
	_IOW(STM32L051_IOCTL_MAGIC, 6, struct stm32l051_vec_sensor_cfg_cmd *)

/**
 * Configure external sensor
 */
#define STM32L051_IOCTL_SET_EXT_SENSOR_CONFIG \
	_IOW(STM32L051_IOCTL_MAGIC, 7, \
	struct stm32l051_external_sensor_cfg_cmd *)

/**
 * Configure IMU HW Sync
 */
#define STM32L051_IOCTL_HW_SYNC_CONFIG \
	_IOW(STM32L051_IOCTL_MAGIC, 8, struct stm32l051_hw_sync_cfg_cmd)

/**
 * Configure LTR (latency tolerance)
 */
#define STM32L051_IOCTL_SET_LTR_CONFIG \
	_IOW(STM32L051_IOCTL_MAGIC, 9, struct stm32l051_ltr_cfg *)

/**
 * Configure time stamp
 */
#define STM32L051_IOCTL_SET_TIMESTAMP_CONFIG \
	_IOW(STM32L051_IOCTL_MAGIC, 10, struct stm32l051_timestamp_cfg_cmd *)

/**
 * FW Update
 */
#define STM32L051_IOCTL_UPDATE_FIRMWARE \
	_IOW(STM32L051_IOCTL_MAGIC, 11, struct stm32l051_fw_update_cmd *)

/**
 * Read EEPROM
 */
#define STM32L051_IOCTL_READ_EEPROM \
	_IOR(STM32L051_IOCTL_MAGIC, 12, struct stm32l051_eeprom_access_cmd *)

/**
 * Write EEPROM
 */
#define STM32L051_IOCTL_WRITE_EEPROM \
	_IOW(STM32L051_IOCTL_MAGIC, 13, struct stm32l051_eeprom_access_cmd *)

/**
 * Get device error
 */
#define STM32L051_IOCTL_GET_DEVICE_ERROR \
	_IOR(STM32L051_IOCTL_MAGIC, 14, struct stm32l051_get_device_error_cmd *)

/**
 * Configure Depth Camera
 */
#define STM32L051_IOCTL_SET_DEPTH_CAM_CONFIG \
	_IOW(STM32L051_IOCTL_MAGIC, 15, struct stm32l051_camera_sensor_cfg_cmd *)

/**
 * Configure Motion Camera
 */
#define STM32L051_IOCTL_SET_MOTION_CAM_CONFIG \
	_IOW(STM32L051_IOCTL_MAGIC, 16, struct stm32l051_camera_sensor_cfg_cmd *)

/**
 * reads the hardware cycle count
 */
#define STM32L051_IOCTL_GET_TIMESTAMP \
	_IOR(STM32L051_IOCTL_MAGIC, 17, struct stm32l051_uc_timestamp *)

#define STM32L051_IOCTL_MAX_NUM 17

/**
 Flows
 This section defines basic flows of the stm32l051 driver

 Operational Flow
 1. Open the device.
 2. Init.
 3. Get device Info (optional)
 4. Get sensors list
 5. Config (sensor, LTR, time stamp, etc...)
 6. Set state to start.
 7. set state to pause.
 8. Config
 9. Set state to start.
 10. Set state to stop.
 11. Uninit.
 12. Close the device.

 FW Update
 1. Open the device.
 2. Init.
 3. Update FW (using write).
 4. Get device info to verify that the FW version is as expected.

 Fix corrupted FW
 1. Open the device.
 2. Init, error will be corrupted FW.
 3. Update FW (using write).
 4. Get device info to verify that the FW is not corrupted and version is as
	expected.
 */

#pragma pack(pop)

#endif /* __STM32L051_H__ */
