/*
 * Motion Module timestamping uc driver
 *
 * Copyright (c) 2014 Intel Corporation. All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License version
 * 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 */
#include <linux/module.h>
#include <linux/init.h>
#include <linux/i2c.h>
#include <linux/input.h>
#include <linux/workqueue.h>
#include <linux/mutex.h>
#include <linux/delay.h>
#include <linux/gpio.h>
#include <linux/slab.h>
#include <linux/mutex.h>
#include <linux/regulator/consumer.h>
#include <linux/vlv2_plat_clock.h>
#include <linux/atomisp_gmin_platform.h>

#include <linux/acpi.h>
#include <linux/io.h>


#define TANSTAMP_NAME                 "tanstamp"

struct tanstamp {
	struct i2c_client *client;
	u16 chip_id;
	struct input_dev *input;
	struct regulator *v1p8_reg;
};

static int tanstamp_i2c_write_read(struct i2c_client *client,u16 write_len, u8 *write_data, u16 read_len , u8 *read_data)
{
	int ret;
	struct i2c_msg msg[2];

	int count = 0;

	if (!client->adapter) {
		dev_err(&client->dev, "%s error, no client->adapter\n",
			__func__);
		return -ENODEV;
	}

	memset(msg, 0 , sizeof(msg));

	if (write_len > 0 && write_data!=NULL)
	{

		msg[count].addr = client->addr;
		msg[count].flags = 0;
		msg[count].len = write_len;
		msg[count].buf = write_data;
		count++;
	}

	if (read_len > 0 && read_data!=NULL)
	{

		msg[count].addr = client->addr;
		msg[count].flags = I2C_M_RD;
		msg[count].len = read_len;
		msg[count].buf = read_data;
		count++;
	}


	ret = i2c_transfer(client->adapter, msg, count);
	if (ret!= count)
	{
		
		return -EIO;
	}
	return 0 ;
}





/*
 * Something of a hack. The CHT RVP board drives camera 1.8v from an
 * external regulator instead of the PMIC just like ECS E7 board, see the
 * comments above.
 */
enum { V1P8_GPIO_UNSET = -2, V1P8_GPIO_NONE = -1 };
static int v1p8_gpio = V1P8_GPIO_UNSET;

static int power_ctrl(struct i2c_client *client, bool flag)
{
	struct tanstamp *data = i2c_get_clientdata(client);
	int ret = -1;

	if (v1p8_gpio == V1P8_GPIO_UNSET) {
		v1p8_gpio = gmin_get_var_int(NULL, "V1P8GPIO", V1P8_GPIO_NONE);
		if (v1p8_gpio != V1P8_GPIO_NONE) {
			pr_info("atomisp_gmin_platform: 1.8v power on GPIO %d\n",
				v1p8_gpio);
			ret = gpio_request(v1p8_gpio, "camera_v1p8");
			if (!ret)
				ret = gpio_direction_output(v1p8_gpio, 0);
			if (ret)
				pr_err("V1P8 GPIO initialization failed\n");
		}
	}


	if (flag)
	{
		if (v1p8_gpio != V1P8_GPIO_NONE)
			gpio_set_value(v1p8_gpio, flag);
		regulator_enable(data->v1p8_reg);
		usleep_range(5000, 6000);
	}
	else
	{

		if (v1p8_gpio != V1P8_GPIO_NONE)
			gpio_set_value(v1p8_gpio, flag);
		regulator_disable(data->v1p8_reg);

		usleep_range(5000, 6000);
	}

	return 0;
}

static int reset_gpio = -1;


static int platclk_index = -1;

static u8 tanstamp_clkindex_dsm_uuid[] __initdata = "154FE31C-DD42-41BD-8E87-45E79DE34CC5";

static int tanstamp_acpi_get_clock_index(struct i2c_client *client)
{
	u8 uuid[16];
	union acpi_object *obj;
	union acpi_object *item;
	struct acpi_device *adev;
	acpi_handle handle;

	acpi_str_to_uuid(tanstamp_clkindex_dsm_uuid, uuid);

	if (!client)
		return -ENODEV;

	handle = ACPI_HANDLE(&client->dev);
	if (!handle || acpi_bus_get_device(handle, &adev))
		return -ENODEV;

	obj = acpi_evaluate_dsm_typed(handle, uuid, 0, 0, NULL,
				      ACPI_TYPE_INTEGER);
	if (!obj) {
		dev_err(&client->dev, "device _DSM execution failed\n");
		return -ENODEV;
	}

	platclk_index = (int)(obj->integer.value);
	dev_info(&client->dev, "Obtained platform clock index: %d\n", platclk_index);

	ACPI_FREE(obj);

	return 0;
}

static int gpio_ctrl(struct i2c_client *client, bool flag)
{
	if (reset_gpio < 0)
		return -ENODEV;

	gpio_set_value(reset_gpio, flag);
	usleep_range(5000, 6000);
	return 0;
}

static int power_up(struct i2c_client *client)
{
	int ret;

	ret = gpio_ctrl(client, 1);

	msleep(5);

	vlv2_plat_set_clock_freq(platclk_index, 0);
	vlv2_plat_configure_clock(platclk_index, 1);

	msleep(5);

	/* power control */
	ret = power_ctrl(client, 1);

	msleep(50);

	return 0;

}

static int power_down(struct i2c_client *client)
{
	int ret = 0;

	ret =  power_ctrl(client, 0);

	vlv2_plat_configure_clock(platclk_index, 0);

	return ret;

}




static int tanstamp_detect(struct i2c_client *client)
{
	struct tanstamp *data = i2c_get_clientdata(client);
	u8 buf[10];
	u16 chip_id = 0;
	int ret = 0;
	buf[0]=0x12;
	ret = tanstamp_i2c_write_read(client,1,buf, 2,(u8*)&chip_id);
	if (ret) {
		
		return -ENODEV;
	}
	data->chip_id = chip_id;
	return ret;
}





static ssize_t tanstamp_chip_id_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct i2c_client *client = to_i2c_client(dev);
	struct tanstamp *data = i2c_get_clientdata(client);

	return sprintf(buf, "%x\n", data->chip_id);
}


static ssize_t tanstamp_current_time_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct input_dev *input = to_input_dev(dev);
	struct tanstamp *data = input_get_drvdata(input);

	u8 inBuffer[1];
	u32 current_time = 0;
	int ret;

	inBuffer[0]=0x15;

	ret = tanstamp_i2c_write_read(data->client,1,inBuffer, sizeof(current_time),(u8*)&current_time);

	if (ret) {
		
		return -ENODEV;
	}

	return sprintf(buf, "%u\n", current_time);
}




#pragma pack(push, 1)
typedef struct __timestamp_frame
{
  uint16_t source_id:3;
  uint16_t frame_number:12;
  uint16_t imu_sync:1;
  uint32_t timestamp;
}timestamp_frame ;
#pragma pack(pop)


static ssize_t tanstamp_fifo_frame_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct input_dev *input = to_input_dev(dev);
	struct tanstamp *data = input_get_drvdata(input);


	timestamp_frame tmpBuffer[128];
	u8 inBuffer[1];
	u8 outBuffer[1] = {0};
	int ret = 0;
	int i, len;
	u32 fifo_size;
//	static int tal=0;
//	static int tal_frame=0;


	// get current fifo-size
	inBuffer[0]=0x14;

	ret = tanstamp_i2c_write_read(data->client,1,inBuffer, sizeof(outBuffer),outBuffer);

	if (ret) {
		
		return -ENODEV;
	}

	i = outBuffer[0];

//	
//	

	fifo_size = (i * sizeof(timestamp_frame));

	if (fifo_size > sizeof(tmpBuffer))
	{
		fifo_size = sizeof(tmpBuffer);
	}


	if (fifo_size==0)
	{
		return -EIO;
	}

	inBuffer[0]=0x13;

	ret = tanstamp_i2c_write_read(data->client,1,inBuffer, fifo_size ,(u8*)tmpBuffer);

	if (ret) {
		
		return -ENODEV;
	}

	for (i=0;i<(fifo_size/sizeof(timestamp_frame));i++)
	{
		if (tmpBuffer[i].source_id==0)
			continue;

#if 0
		if (tmpBuffer[i].source_id==1)
		{
			tal++;
			if (tal==4)
			{
				tal=0;

				len = sprintf(buf, "%u %u %u %u ", 4 , tal_frame++ , 0 , tmpBuffer[i].timestamp);
				buf+=len;
				ret+=len;


			}
		}
#endif
		len = sprintf(buf, "%u %u %u %u ", tmpBuffer[i].source_id, tmpBuffer[i].frame_number , tmpBuffer[i].imu_sync , tmpBuffer[i].timestamp);
		buf+=len;
		ret+=len;

	}

	return ret;
}



static ssize_t tanstamp_enable_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct input_dev *input = to_input_dev(dev);
	struct tanstamp *data = input_get_drvdata(input);

	u8 inBuffer[1];
	u8 outBuffer[1] = {0};
	int ret = 0;
	int enabled;

	inBuffer[0]=0x10;

	ret = tanstamp_i2c_write_read(data->client,1,inBuffer, sizeof(outBuffer),outBuffer);

	if (ret) {
		
		return -ENODEV;
	}

	enabled = outBuffer[0];

	ret = sprintf(buf, "%d\n", enabled);
	return ret;
}


static ssize_t tanstamp_enable_store(struct device *dev,
		struct device_attribute *attr,
		const char *buf, size_t count)
{
	unsigned long value;
	struct input_dev *input = to_input_dev(dev);
	struct tanstamp *data = input_get_drvdata(input);
	u8 inBuffer[2];
	int ret = 0;

	ret = kstrtoul(buf, 10, &value);
	if (ret)
		return ret;

	inBuffer[0]=0x10;
	inBuffer[1]=(u8)value;

	ret = tanstamp_i2c_write_read(data->client,2,inBuffer, 0,0);

	if (ret) {
		
		return -ENODEV;
	}

	return count;
}

static ssize_t tanstamp_set_power(struct device *dev,
		struct device_attribute *attr,
		const char *buf, size_t count)
{
	int enable=0;
	struct input_dev *input = to_input_dev(dev);
	struct tanstamp *data = input_get_drvdata(input);

	sscanf(buf, "%d", &enable);

	if (enable)
		power_up(data->client);
	else
		power_down(data->client);


	return count;
}




static ssize_t tanstamp_enable_by_read(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct input_dev *input = to_input_dev(dev);
	struct tanstamp *data = input_get_drvdata(input);
	u8 inBuffer[2];
	int ret = 0;

	inBuffer[0]=0x10;
	inBuffer[1]=1;

	ret = tanstamp_i2c_write_read(data->client,2,inBuffer, 0,0);

	if (ret) {
		
		return -ENODEV;
	}

	ret = sprintf(buf, "%d\n", 1);
	return ret;
}

static ssize_t tanstamp_disable_by_read(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct input_dev *input = to_input_dev(dev);
	struct tanstamp *data = input_get_drvdata(input);
	u8 inBuffer[2];
	int ret = 0;

	inBuffer[0]=0x10;
	inBuffer[1]=0;

	ret = tanstamp_i2c_write_read(data->client,2,inBuffer, 0,0);

	if (ret) {
		
		return -ENODEV;
	}

	ret = sprintf(buf, "%d\n", 0);
	return ret;
}



static DEVICE_ATTR(chip_id, S_IRUGO,
		tanstamp_chip_id_show, NULL);

static DEVICE_ATTR(enable, S_IRUGO|S_IWUSR|S_IRGRP|S_IROTH,
		tanstamp_enable_show, tanstamp_enable_store);

static DEVICE_ATTR(enable_by_read, S_IRUGO,
		tanstamp_enable_by_read,NULL);

static DEVICE_ATTR(disable_by_read, S_IRUGO,
		tanstamp_disable_by_read,NULL);

static DEVICE_ATTR(spower, S_IWUSR|S_IRUGO,
		NULL, tanstamp_set_power);

static DEVICE_ATTR(current_time, S_IRUGO,
		tanstamp_current_time_show, NULL);

static DEVICE_ATTR(fifo_frame, S_IRUGO,
		tanstamp_fifo_frame_show, NULL);


static struct attribute *tanstamp_attributes[] = {
	&dev_attr_chip_id.attr,
	&dev_attr_enable.attr,
	&dev_attr_enable_by_read.attr,
	&dev_attr_disable_by_read.attr,
	&dev_attr_spower.attr,
	&dev_attr_current_time.attr,
	&dev_attr_fifo_frame.attr,
	NULL
};

static struct attribute_group tanstamp_attribute_group = {
	.attrs = tanstamp_attributes
};


static int tanstamp_acpi_gpio_probe(struct i2c_client *client)
{
	const struct acpi_device_id *id;
	struct device *dev;
	struct gpio_desc *gpio;
	int ret;

	if (!client)
		return -EINVAL;

	dev = &client->dev;
	if (!ACPI_HANDLE(dev))
		return -ENODEV;

	id = acpi_match_device(dev->driver->acpi_match_table, dev);
	if (!id)
		return -ENODEV;

	/* data ready gpio interrupt pin */
	gpio = devm_gpiod_get_index(dev, "uc_reset", 0, GPIOD_OUT_LOW);
	if (IS_ERR(gpio)) {
		dev_err(dev, "acpi gpio get index failed\n");
		return PTR_ERR(gpio);
	}

	/* Set pin as LOW by default */
	ret = gpiod_direction_output(gpio, 0);
	if (ret) {
		dev_err(dev, "acpi gpio direction output failed\n");
		devm_gpiod_put(dev, gpio);
		return ret;
	}

	reset_gpio = desc_to_gpio(gpio);

	dev_dbg(dev, "GPIO resource, no:%d\n", reset_gpio);

	return 0;
}

static int tanstamp_probe(struct i2c_client *client,
		const struct i2c_device_id *id)
{
	int err = 0;
	struct tanstamp *data;
	struct input_dev *dev;
	struct regulator *v1p8_reg;

	//
	v1p8_reg = devm_regulator_get(&client->dev, "V1P8SX");
	if (IS_ERR(v1p8_reg)) {
		dev_err(&client->dev, "Failure to retrieve V1P8SX regulator\n");
		return PTR_ERR(v1p8_reg);
	}

	err = tanstamp_acpi_get_clock_index(client);
	if (err)
		return err;

	err = tanstamp_acpi_gpio_probe(client);
	if (err)
		return err;

	if (!i2c_check_functionality(client->adapter, I2C_FUNC_I2C)) {
		err = -EIO;
		goto exit;
	}
	data = kzalloc(sizeof(struct tanstamp), GFP_KERNEL);
	if (!data) {
		err = -ENOMEM;
		goto exit;
	}

	data->v1p8_reg = v1p8_reg;

	i2c_set_clientdata(client, data);
	data->client = client;

	power_up(client);

	/* read and check chip id */
	if (tanstamp_detect(client) < 0) {
		err = -EINVAL;
		goto kfree_exit;
	}

	dev = input_allocate_device();
	if (!dev)
	{
		err = -EINVAL;
		goto kfree_exit;
	}

	/* only value events reported */
	dev->name = TANSTAMP_NAME;
	dev->id.bustype = BUS_I2C;

	input_set_drvdata(dev, data);
	err = input_register_device(dev);
	if (err < 0)
		goto err_register_input_device;
	data->input = dev;


	err = sysfs_create_group(&data->input->dev.kobj,
			&tanstamp_attribute_group);
	if (err < 0)
		goto error_sysfs;

	power_down(client);

	dev_notice(&client->dev, "tanstamp driver probe successfully");

	return 0;


error_sysfs:
	input_unregister_device(data->input);

err_register_input_device:
	input_free_device(dev);

kfree_exit:
	kfree(data);
exit:
	power_down(client);

	return err;
}


static int tanstamp_remove(struct i2c_client *client)
{
	//
	struct tanstamp *data = i2c_get_clientdata(client);

//	power_down(client);

	if (reset_gpio)
		gpio_free(reset_gpio);
	reset_gpio=0;

	sysfs_remove_group(&data->input->dev.kobj, &tanstamp_attribute_group);
	input_unregister_device(data->input);

	kfree(data);

	return 0;
}

void tanstamp_shutdown(struct i2c_client *client)
{
	//
	//struct tanstamp *data = i2c_get_clientdata(client);
//	power_down(client);
}

 
static const struct i2c_device_id tanstamp_id[] = {
	{ TANSTAMP_NAME, 0 },
	{ }
};

MODULE_DEVICE_TABLE(i2c, tanstamp_id);


static struct acpi_device_id tanstamp_acpi_match[] = {
	{ "INT35BB" },
	{},
};

MODULE_DEVICE_TABLE(acpi, tanstamp_acpi_match);


static struct i2c_driver tanstamp_driver = {
	.driver = {
		.owner  = THIS_MODULE,
		.name   = TANSTAMP_NAME,
		.acpi_match_table = ACPI_PTR(tanstamp_acpi_match),
	},
	.class = I2C_CLASS_HWMON,
	.id_table   = tanstamp_id,
	.probe      = tanstamp_probe,
	.remove     = tanstamp_remove,
	.shutdown   = tanstamp_shutdown,
};



static int __init tanstamp_init(void)
{
	return i2c_add_driver(&tanstamp_driver);
}

static void __exit tanstamp_exit(void)
{
	i2c_del_driver(&tanstamp_driver);
}

MODULE_AUTHOR("Tal Shustak <tal.shustak@intel.com>");
MODULE_DESCRIPTION("Motion Module timestamping UC DRIVER");
MODULE_LICENSE("GPL v2");

module_init(tanstamp_init);
module_exit(tanstamp_exit);
