/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as 
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "launcher-recent-source.h"

#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n-lib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <clutk/clutk.h>
#include <clutter/clutter.h>
#include <clutter-gtk/clutter-gtk.h>

G_DEFINE_TYPE (LauncherRecentSource, launcher_recent_source, G_TYPE_OBJECT);

#define LAUNCHER_RECENT_SOURCE_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  LAUNCHER_TYPE_RECENT_SOURCE, \
  LauncherRecentSourcePrivate))

struct _LauncherRecentSourcePrivate
{
  NlShell       *shell;
  NlPixbufCache *pixbuf_cache;
  GtkRecentManager    *manager;
  GtkWidget           *menu;
  
  ClutterActor *vbox;
  ClutterActor *text;
  ClutterActor *iconview;
  ClutterActor *clean_button;

  gint n_items;
};

enum
{
  PROP_0,
  PROP_SHELL
};

/* Forwards */
static gboolean reload_recent            (LauncherRecentSource *self);
static gboolean on_clean_button_released (ClutterActor         *actor,
                                          ClutterButtonEvent   *event,
                                          LauncherRecentSource *self);
static gboolean on_clean_button_entered  (ClutterActor         *actor,
                                          ClutterCrossingEvent *event,
                                          LauncherRecentSource *self);
static gboolean on_clean_button_left     (ClutterActor         *actor,
                                          ClutterCrossingEvent *event,
                                          LauncherRecentSource *self);

/* GObject stuff */
static void
launcher_recent_source_set_property (GObject      *object, 
                                     guint         prop_id,
                                     const GValue *value, 
                                     GParamSpec   *pspec)
{
  LauncherRecentSourcePrivate *priv;
  
  g_return_if_fail (LAUNCHER_IS_RECENT_SOURCE (object));
  priv = LAUNCHER_RECENT_SOURCE_GET_PRIVATE (object);

  switch (prop_id) 
    {
    case PROP_SHELL:
      priv->shell = g_value_get_pointer (value);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
launcher_recent_source_get_property (GObject      *object, 
                                     guint         prop_id,
                                     GValue       *value, 
                                     GParamSpec   *pspec)
{
  LauncherRecentSourcePrivate *priv;
  
  g_return_if_fail (LAUNCHER_IS_RECENT_SOURCE (object));
  priv = LAUNCHER_RECENT_SOURCE_GET_PRIVATE (object);

  switch (prop_id) 
    {
    case PROP_SHELL:
      g_value_set_pointer (value, priv->shell);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}
static void
launcher_recent_source_finalize (GObject *object)
{
  LauncherRecentSourcePrivate *priv;

  priv = LAUNCHER_RECENT_SOURCE_GET_PRIVATE (object);

  if (priv->pixbuf_cache)
    {
      g_object_unref (priv->pixbuf_cache);
      priv->pixbuf_cache = NULL;
    }

  if (priv->menu)
    {
      gtk_widget_destroy (priv->menu);
      priv->menu = NULL;
    }

  if (priv->vbox)
    {
      g_object_unref (priv->vbox);
      priv->vbox = NULL;
    }

  G_OBJECT_CLASS (launcher_recent_source_parent_class)->finalize (object);
}

static void
launcher_recent_source_constructed (GObject *object)
{
  LauncherRecentSourcePrivate *priv;
      
  priv = LAUNCHER_RECENT_SOURCE_GET_PRIVATE (object);

  /* Build the view */
  priv->vbox = ctk_vbox_new (6);
  g_object_ref_sink (priv->vbox);

  priv->text = ctk_text_new (_("<big><b>Recent Documents</b></big>"));
  ctk_text_set_alignment (CTK_TEXT (priv->text), PANGO_ALIGN_CENTER);
  ctk_box_pack (CTK_BOX (priv->vbox), priv->text, FALSE, FALSE);
  
  priv->iconview = ctk_icon_view_new ();
  ctk_box_pack (CTK_BOX (priv->vbox), priv->iconview, FALSE, FALSE);

  priv->clean_button = ctk_text_new (_("<small><b><u>Clear Recent Documents</u></b></small>"));
  ctk_text_set_alignment (CTK_TEXT (priv->clean_button), PANGO_ALIGN_RIGHT);
  clutter_actor_set_opacity (priv->clean_button, 0);
  ctk_box_pack (CTK_BOX (priv->vbox), priv->clean_button, FALSE, FALSE);
  clutter_actor_set_reactive (priv->clean_button, TRUE);
  g_signal_connect (priv->clean_button, "button-release-event",
                    G_CALLBACK (on_clean_button_released), object);
  g_signal_connect (priv->clean_button, "enter-event",
                    G_CALLBACK (on_clean_button_entered), object);
  g_signal_connect (priv->clean_button, "leave-event",
                    G_CALLBACK (on_clean_button_left), object);

  nl_shell_add_places_source (priv->shell, priv->vbox);
  g_idle_add ((GSourceFunc)reload_recent, object);
}

static void
launcher_recent_source_class_init (LauncherRecentSourceClass *klass)
{
  GObjectClass *obj_class = G_OBJECT_CLASS (klass);
  GParamSpec   *pspec;
    
  obj_class->finalize     = launcher_recent_source_finalize;
  obj_class->constructed  = launcher_recent_source_constructed;
  obj_class->set_property = launcher_recent_source_set_property;
  obj_class->get_property = launcher_recent_source_get_property;

  pspec = g_param_spec_pointer ("shell", "shell", "shell",  
                                G_PARAM_READWRITE | G_PARAM_CONSTRUCT);
  g_object_class_install_property (obj_class, PROP_SHELL, pspec);
  
  g_type_class_add_private (obj_class, sizeof (LauncherRecentSourcePrivate));
}

static void
launcher_recent_source_init (LauncherRecentSource *source)
{
  LauncherRecentSourcePrivate *priv;
  
  priv = source->priv = LAUNCHER_RECENT_SOURCE_GET_PRIVATE (source);

  priv->n_items = 0;

  priv->pixbuf_cache = nl_pixbuf_cache_get_default ();
  priv->manager = gtk_recent_manager_get_default ();
  g_signal_connect_swapped (priv->manager, "changed",
                            G_CALLBACK (reload_recent), source);
  
  priv->menu = gtk_recent_chooser_menu_new_for_manager (priv->manager);
  gtk_recent_chooser_set_sort_type (GTK_RECENT_CHOOSER (priv->menu),
                                    GTK_RECENT_SORT_MRU);
}

/*
 * Public methods
 */
GObject *
launcher_recent_source_new (NlShell *shell)

{
  GObject *recent_source;

  g_return_val_if_fail (NL_IS_SHELL (shell), NULL);
 
  recent_source = g_object_new (LAUNCHER_TYPE_RECENT_SOURCE,
                                 "shell", shell,
                                 NULL);

  return recent_source;
}

/*
 * Private methods
 */
static void
on_tile_clicked (ClutterActor *actor)
{
  GtkRecentInfo *info;

  info = g_object_get_data (G_OBJECT (actor), "gtk-recent-info");
  if (info)
    {
      GError *error = NULL;

      g_app_info_launch_default_for_uri (gtk_recent_info_get_uri (info),
                                         NULL,
                                         &error);
      if (error)
        {
          g_warning ("Unable to launcher %s: %s",
                     gtk_recent_info_get_uri (info),
                     error->message);
          g_error_free (error);
        }
    }
}

static void
show_menu (ClutterActor         *actor, 
           guint32               event_time,
           LauncherRecentSource *source)
{
  GtkWidget *menu, *item;

  g_return_if_fail (source);

  menu = gtk_menu_new ();

  item = gtk_image_menu_item_new_from_stock (GTK_STOCK_OPEN, NULL);
  gtk_widget_show (item);
  g_signal_connect_swapped (item, "activate",
                            G_CALLBACK (on_tile_clicked), actor);
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
  
  gtk_menu_popup (GTK_MENU (menu),
                  NULL, NULL,
                  NULL, NULL,
                  3, event_time);
}

static GdkPixbuf *
try_get_thumbnail_for_uri (const gchar *uri, NlPixbufCache *cache)
{
  GdkPixbuf *ret;
  gchar     *md5;
  gchar     *file;
  gchar     *path;

  md5 = g_compute_checksum_for_data (G_CHECKSUM_MD5,
                                     (const guchar *)uri,
                                     strlen (uri));
  file = g_strconcat (md5, ".png", NULL);
  g_free (md5);

  path = g_build_filename (g_get_home_dir (),
                           ".thumbnails",
                           "normal",
                           file,
                           NULL);

  if (g_file_test (path, G_FILE_TEST_EXISTS))
    ret = nl_pixbuf_cache_icon_for_name (cache, path, 128);
  
  g_free (path);
  return ret;
}

static NlIconTile *
make_tile_for_info (LauncherRecentSource *source, GtkRecentInfo *info)
{
  ClutterActor *tile;
  GdkPixbuf    *pixbuf;
  CtkText      *text;

  pixbuf = try_get_thumbnail_for_uri (gtk_recent_info_get_uri (info),
                                      source->priv->pixbuf_cache);
  if (!GDK_IS_PIXBUF (pixbuf))
    pixbuf = gtk_recent_info_get_icon (info, 64);
    
  tile = nl_icon_tile_new (gtk_recent_info_get_display_name (info), 
                           gtk_recent_info_get_description (info),
                           pixbuf);
  text = ctk_button_get_text (CTK_BUTTON (tile));
  clutter_text_set_line_wrap (CLUTTER_TEXT (text), TRUE);
  clutter_text_set_line_wrap_mode (CLUTTER_TEXT (text), PANGO_WRAP_WORD_CHAR);

  gtk_recent_info_ref (info);
  g_object_set_data (G_OBJECT (tile), "gtk-recent-info", info);

  g_signal_connect (tile, "clicked", 
                    G_CALLBACK (on_tile_clicked), NULL);
  g_signal_connect (tile, "show-context-menu",
                    G_CALLBACK (show_menu), source);
  if (pixbuf)
    g_object_unref (pixbuf);
  
  return NL_ICON_TILE (tile);
}

static void
remove_child (ClutterActor *child, ClutterContainer *container)
{
  GtkRecentInfo *info;

  info = g_object_get_data (G_OBJECT (child), "gtk-recent-info");
  gtk_recent_info_unref (info);

  clutter_container_remove_actor (container, child);
}

static gboolean
reload_recent (LauncherRecentSource *source)
{
  LauncherRecentSourcePrivate *priv;
  GList *items, *i;
  GList *children;
  gint n = 0;
  
  g_return_val_if_fail (LAUNCHER_IS_RECENT_SOURCE (source), FALSE);
  priv = source->priv;

  children = clutter_container_get_children(CLUTTER_CONTAINER (priv->iconview));
  g_list_foreach (children, (GFunc)remove_child, priv->iconview);
  g_list_free (children);

  items = gtk_recent_chooser_get_items (GTK_RECENT_CHOOSER (priv->menu));
  
  for (i = items; i && n < 10; i = i->next)
    {
      GtkRecentInfo *info = i->data;
    
      clutter_container_add_actor (CLUTTER_CONTAINER (priv->iconview), 
                                   CLUTTER_ACTOR (make_tile_for_info (source,
                                                  info)));
  
      n++;
    }
  g_list_foreach (items, (GFunc)gtk_recent_info_unref, NULL);  
  g_list_free (items);

  clutter_actor_animate (priv->clean_button, CLUTTER_EASE_IN_SINE, 200,
                         "opacity", n ? 180 : 0,
                         NULL);

  priv->n_items = n;
 
  return FALSE;  
}


static gboolean
on_clean_button_released (ClutterActor         *actor,
                          ClutterButtonEvent   *event,
                          LauncherRecentSource *self)
{
  if (event->button == 1)
    {
      GError *error = NULL;
      gint    n_purged;

      g_return_val_if_fail (LAUNCHER_IS_RECENT_SOURCE (self), FALSE);

      n_purged = gtk_recent_manager_purge_items (self->priv->manager, &error);

      if (error)
        {
          g_warning ("Unable to clear recent document list: %s",
                     error->message);
          g_error_free (error);
        }
      else
        {
          g_debug ("Cleared recent document list (%d items)", n_purged);
          self->priv->n_items = 0;
          clutter_actor_set_opacity (actor, 0);
        }
    }
  return FALSE;
}

static gboolean
on_clean_button_entered  (ClutterActor         *actor,
                          ClutterCrossingEvent *event,
                          LauncherRecentSource *self)
{
  if (self->priv->n_items)
    clutter_actor_animate (actor, CLUTTER_EASE_IN_SINE, 200,
                           "opacity", 255,
                           NULL);
  return FALSE;
}

static gboolean
on_clean_button_left     (ClutterActor         *actor,
                          ClutterCrossingEvent *event,
                          LauncherRecentSource *self)
{
  clutter_actor_animate (actor, CLUTTER_EASE_IN_SINE, 200,
                         "opacity", self->priv->n_items ? 180 : 0,
                         NULL);
  return FALSE;
}
