/* This file is part of the KDE libraries
   Copyright (C) 1999 Ian Zepp (icszepp@islc.net)
   Copyright (C) 2000 Rik Hemsley (rikkus) <rik@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License version 2 as published by the Free Software Foundation.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/
#ifndef KSTRINGHANDLER_H
#define KSTRINGHANDLER_H

#include <kdecore_export.h>

class QChar;
class QRegExp;
class QString;
class QStringList;

/**
 * This class contains utility functions for handling strings.
 *
 * This class is @em not a substitute for the QString class. What
 * I tried to do with this class is provide an easy way to
 * cut/slice/splice words inside sentences in whatever order desired.
 * While the main focus of this class are words (ie characters
 * separated by spaces/tabs), the two core functions here ( split()
 * and join() ) will function given any char to use as a separator.
 * This will make it easy to redefine what a 'word' means in the
 * future if needed.
 *
 * I freely stole some of the function names from python. I also think
 * some of these were influenced by mIRC (yes, believe it if you will, I
 * used to write a LOT of scripts in mIRC).
 *
 * The ranges are a fairly powerful way of getting/stripping words from
 * a string. These ranges function, for the large part, as they would in
 * python. See the word(const QString&, int) and remword(const QString&, int) functions for more detail.
 *
 * This class contains no data members of it own. All strings are cut
 * on the fly and returned as new qstrings/qstringlists.
 *
 * @short Namespace for manipulating words and sentences in strings
 * @author Ian Zepp <icszepp@islc.net>
 * @see KShell
 */
namespace KStringHandler
{

    /** Capitalizes each word in the string
      * "hello there" becomes "Hello There"        (string)
      * @param text the text to capitalize
      * @return the resulting string
      */
    KDECORE_EXPORT QString capwords( const QString &text );

    /** Capitalizes each word in the list
      * [hello, there] becomes [Hello, There]    (list)
      * @param list the list to capitalize
      * @return the resulting list
      */
    KDECORE_EXPORT QStringList capwords( const QStringList &list );

    /** Substitute characters at the beginning of a string by "...".
     * @param str is the string to modify
     * @param maxlen is the maximum length the modified string will have
     * If the original string is shorter than "maxlen", it is returned verbatim
     * @return the modified string
     */
    KDECORE_EXPORT QString lsqueeze( const QString & str, int maxlen = 40 );

    /** Substitute characters at the middle of a string by "...".
     * @param str is the string to modify
     * @param maxlen is the maximum length the modified string will have
     * If the original string is shorter than "maxlen", it is returned verbatim
     * @return the modified string
     */
    KDECORE_EXPORT QString csqueeze( const QString & str, int maxlen = 40 );

    /** Substitute characters at the end of a string by "...".
     * @param str is the string to modify
     * @param maxlen is the maximum length the modified string will have
     * If the original string is shorter than "maxlen", it is returned verbatim
     * @return the modified string
     */
    KDECORE_EXPORT QString rsqueeze( const QString & str, int maxlen = 40 );

    /**
     * Split a QString into a QStringList in a similar fashion to the static
     * QStringList function in Qt, except you can specify a maximum number
     * of tokens. If max is specified (!= 0) then only that number of tokens
     * will be extracted. The final token will be the remainder of the string.
     *
     * Example:
     * \code
     * perlSplit("__", "some__string__for__you__here", 4)
     * QStringList contains: "some", "string", "for", "you__here"
     * \endcode
     *
     * @param sep is the string to use to delimit s.
     * @param s is the input string
     * @param max is the maximum number of extractions to perform, or 0.
     * @return A QStringList containing tokens extracted from s.
     */
    KDECORE_EXPORT QStringList perlSplit( const QString & sep,
                                          const QString & s,
                                          int max = 0 );

    /**
     * Split a QString into a QStringList in a similar fashion to the static
     * QStringList function in Qt, except you can specify a maximum number
     * of tokens. If max is specified (!= 0) then only that number of tokens
     * will be extracted. The final token will be the remainder of the string.
     *
     * Example:
     * \code
     * perlSplit(' ', "kparts reaches the parts other parts can't", 3)
     * QStringList contains: "kparts", "reaches", "the parts other parts can't"
     * \endcode
     *
     * @param sep is the character to use to delimit s.
     * @param s is the input string
     * @param max is the maximum number of extractions to perform, or 0.
     * @return A QStringList containing tokens extracted from s.
     */
    KDECORE_EXPORT QStringList perlSplit( const QChar & sep,
                                          const QString & s,
                                          int max = 0 );

    /**
     * Split a QString into a QStringList in a similar fashion to the static
     * QStringList function in Qt, except you can specify a maximum number
     * of tokens. If max is specified (!= 0) then only that number of tokens
     * will be extracted. The final token will be the remainder of the string.
     *
     * Example:
     * \code
     * perlSplit(QRegExp("[! ]"), "Split me up ! I'm bored ! OK ?", 3)
     * QStringList contains: "Split", "me", "up ! I'm bored ! OK ?"
     * \endcode
     *
     * @param sep is the regular expression to use to delimit s.
     * @param s is the input string
     * @param max is the maximum number of extractions to perform, or 0.
     * @return A QStringList containing tokens extracted from s.
     */
    KDECORE_EXPORT QStringList perlSplit( const QRegExp & sep,
                                          const QString & s,
                                          int max = 0);

    /**
     * This method auto-detects URLs in strings, and adds HTML markup to them
     * so that richtext or HTML-enabled widgets will display the URL correctly.
     * @param text the string which may contain URLs
     * @return the resulting text
     */
    KDECORE_EXPORT QString tagUrls( const QString& text );

    /**
      Obscure string by using a simple symmetric encryption. Applying the
      function to a string obscured by this function will result in the original
      string.

      The function can be used to obscure passwords stored to configuration
      files. Note that this won't give you any more security than preventing
      that the password is directly copied and pasted.

      @param str string to be obscured
      @return obscured string
    */
    KDECORE_EXPORT QString obscure( const QString &str );


    /**
      Guess whether a string is UTF8 encoded.

      @param str the string to check
      @return true if UTF8. If false, the string is probably in Local8Bit.
     */
    KDECORE_EXPORT bool isUtf8( const char *str );

    /**
      Construct QString from a c string, guessing whether it is UTF8- or
      Local8Bit-encoded.

      @param str the input string
      @return the (hopefully correctly guessed) QString representation of @p str
      @see KEncodingDetector

     */
    KDECORE_EXPORT QString from8Bit( const char *str );
}
#endif
