/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

#include <string.h>
#include <glib/gi18n.h>

#include "kz-navi.h"
#include "kazehakase.h"
#include "bookmarks/kz-bookmark.h"
#include "bookmarks/kz-bookmark-file.h"

static void dispose      (GObject      *object);

static void cb_navi_load_completed (KzBookmark *bookmark, gpointer data);
static void cb_navi_load_error     (KzBookmark *bookmark, gpointer data);

G_DEFINE_TYPE(KzNavi, kz_navi, G_TYPE_OBJECT)

static void
kz_navi_class_init (KzNaviClass *klass)
{
	GObjectClass *object_class;

	kz_navi_parent_class = g_type_class_peek_parent(klass);
	object_class = (GObjectClass *)klass;

	object_class->dispose = dispose;
}

static void
kz_navi_init (KzNavi *navi)
{
}

KzNavi *
kz_navi_new (void)
{
	return kz_navi_new_with_info(NULL, NULL, NULL, NULL);
}

KzNavi *
kz_navi_new_with_info (const gchar *name, const gchar *uri,
                       const gchar *title, const gchar *type)
{
	KzNavi *navi;

	navi = g_object_new(KZ_TYPE_NAVI, NULL);

	navi->name = g_strdup(name);
	navi->uri = g_strdup(uri);
	navi->title = g_strdup(title);
	navi->type = g_strdup(type);
	
	return navi;
}

static void
dispose (GObject *object)
{
	KzNavi *navi = KZ_NAVI(object);

	g_free(navi->name);
	g_free(navi->uri);
	g_free(navi->title);
	g_free(navi->type);
	
	navi->name = NULL;
	navi->uri = NULL;
	navi->title = NULL;
	navi->type = NULL;
	
	if (G_OBJECT_CLASS (kz_navi_parent_class)->dispose)
		(*G_OBJECT_CLASS (kz_navi_parent_class)->dispose) (object);
}

static void 
disconnect_signals (KzBookmark *bookmark)
{
	g_signal_handlers_disconnect_by_func(bookmark,
					     G_CALLBACK(cb_navi_load_completed),
					     NULL);
	g_signal_handlers_disconnect_by_func(bookmark,
					     G_CALLBACK(cb_navi_load_completed),
					     NULL);
}

static void
cb_navi_load_completed (KzBookmark *bookmark, gpointer data)
{
	const gchar *title;

	title = kz_bookmark_get_document_title(bookmark);

	if(title == NULL || *title == '\0')
	{
		kz_bookmark_set_title(bookmark, "RSS");
	}
	else
	{
		kz_bookmark_set_title(bookmark, title);
	}

	disconnect_signals(bookmark);
}

static void
cb_navi_load_error (KzBookmark *bookmark, gpointer data)
{
	disconnect_signals(bookmark);
}

void
kz_navi_add_bookmark (KzNavi *navi)
{
	KzBookmark *folder, *bookmark, *sibling = NULL;
	KzBookmark *file;
	GList *bars;
	const gchar *title, *uri;

	g_return_if_fail(navi);
	g_return_if_fail(navi->uri);

	bars = kz_bookmark_get_children(KZ_GET_BAR_BOOKMARK);
	g_return_if_fail(bars);
	
	folder = bars->data;
	g_list_free(bars);
	
	if (!kz_bookmark_is_folder(folder))
	{
		sibling = folder;
		folder = kz_bookmark_get_parent(folder);
		g_return_if_fail(KZ_IS_BOOKMARK(folder));
	}

	title = navi->title ? navi->title : _("Feed");
	uri   = navi->uri;

	bookmark = KZ_BOOKMARK(kz_bookmark_file_new(uri, title, NULL));
	/* FIXME! show dialog */
	if (sibling)
		kz_bookmark_insert_before(folder, bookmark, sibling);
	else
		kz_bookmark_append(folder, bookmark);
	
	if (KZ_IS_BOOKMARK_FILE(folder))
		file = folder;
	else
		file = kz_bookmark_get_parent_file(folder);
	
	kz_bookmark_file_save(KZ_BOOKMARK_FILE(file));

	/* connect to load_completed signal for setting bookmark title in RSS file */
	g_signal_connect (bookmark, "load_completed", 
			  G_CALLBACK (cb_navi_load_completed), NULL);
	g_signal_connect (bookmark, "error", 
			  G_CALLBACK (cb_navi_load_error), NULL);
	kz_bookmark_file_load_start(KZ_BOOKMARK_FILE(bookmark));

	g_object_unref(G_OBJECT(bookmark));
}
