#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2011-2012 Canonical, Ltd.
#
# Authors:
#  Ugo Riboni <ugo.riboni@canonical.com>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 3.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

class ClientTracker(object):
   def __init__(self):
      self.last_client_id = 0
      self.clients = {}
      self.last_call_id = 0
      self.calls = {}      

   def next_client_id(self):
      client_id = self.last_client_id
      self.last_client_id += 1
      return client_id

   def next_call_id(self):
      call_id = self.last_call_id
      self.last_call_id += 1
      return call_id

   def register_client(self, client):
      client_id = self.next_client_id()
      self.clients[client_id] = {
         'id': client_id,
         'client': client,
         'proxy': None,
         'interfaces': {},
         'calls': []
      }
      return client_id

   def unregister_client(self, client_id):
      # Need to do this explicily otherwise the object will keep holding the bus name
      # until garbage collected
      self.clients[client_id]['proxy'].remove_from_connection()

      # Error out any outstanding calls for this client, then remove them from the main list
      for call_id in self.clients[client_id]['calls']:
         _, _, errback = self.calls[call_id]
         errback(Exception("Connection was closed before the peer returned a result"))
         self.calls.pop(call_id, None)

      del self.clients[client_id]
      print self.clients

   def register_on_bus(self, client_id, createProxy):
      client = self.clients.get(client_id, None)
      if client is None:
         print "Something went wrong: client id does not seem to exist: %d" % client_id
         return False

      if not client.get('proxy', None) is None:
         print "Each connected client can only register once on the bus."
         return False

      client['proxy'] = createProxy()
      return True

   def register_method(self, client_id, interface, method, in_sig, out_sig):
      client = self.clients.get(client_id, None)
      if client is None:
         print "The client with id %d is not registered." % client_id
         return False

      if client['proxy'] is None:
         print "No proxy was set. You probably need to successfully call register_on_bus first."
         return False

      iface = client['interfaces'].setdefault(interface, {})    
      iface[method] = (in_sig, out_sig)
      client['proxy'].register_method(interface, method, in_sig, out_sig)
      return True
 
   def client_with_method(self, interface, method):
      clients = [client for id, client in self.clients.items()
                 if interface in client['interfaces']
                 and method in client['interfaces'][interface]]
      if len(clients) == 0:
         print "Internal error. Could not find any client with the requested method."
         return -1, None
      if len(clients) > 1:
         print ("Internal error. More than one client supplies this method."
                "It shoulnd't be possible. Returning first found.")
      return clients[0]['id'], clients[0]['client']       

   def defer_call_return(self, client_id, callback, errback):
      call_id = self.next_call_id()
      self.calls[call_id] = (client_id, callback, errback)
      # We keep a list of the calls also on the client, so that when it disconnects we can
      # go and call the errback on each of them and remove them, without having to iterate
      # the entire list.
      self.clients[client_id]['calls'].append(call_id)
      return call_id

   def unqueue_deferred_call(self, call_id):
      client_id, callback, errback = self.calls.pop(call_id, (-1, None, None))
      if client_id != -1:
         # Get the client and remove this call id from the client's call list too
         client = self.clients.get(client_id, {'calls':[]})
         client['calls'][:] = [call_id for call_id in client['calls'] if call_id != call_id]

      return callback, errback

# Export "singleton" instance from module
tracker = ClientTracker()