/*
 * indicator-network
 * Copyright 2010-2012 Canonical Ltd.
 *
 * Authors:
 * Antti Kaijanmäki <antti.kaijanmaki@canonical.com>
 * Kalle Valo       <kalle.valo@canonical.com>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/** @todo This class contains circular reference to Manager as manager is the
          creator of this class. The code should be refactored in a way that 
	  there is no circular reference and there is no possibility that that
	  the Manager object would go away under us. Maybe with weak references. */

#include "service.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <unistd.h>
#include <glib/gi18n.h>
#include <libdbusmenu-glib/server.h>
#include <libdbusmenu-glib/client.h>

#include "manager.h"

#include "network-menu.h"
#include "dbus-shared-names.h"
#include "marshal.h"

static void handle_event (DbusmenuMenuitem * mi, const gchar * name, 
                          GVariant * value, guint timestamp);

void
network_strength_changed(gpointer object,
			 gint strength,
			 gpointer user_data);

void
network_state_changed(gpointer object,
		      gint state,
		      gpointer user_data);

#define NETWORK_AGENT LIBEXECDIR"/indicator-network-agent"

G_DEFINE_TYPE(Service, service, DBUSMENU_TYPE_MENUITEM)

#define GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), TYPE_SERVICE, ServicePrivate))

typedef struct _ServicePrivate ServicePrivate;

struct _ServicePrivate {
  Manager *network_service;
  
  gchar                  *identifier;
  AndroidNetworkType      type; 
  AndroidNetworkState     state;
  gint                    strength;
  gchar                  *name;
  AndroidNetworkSecurity  security;
};

enum {
  STATE_CHANGED,
  STRENGTH_UPDATED,
  ERROR,
  LAST_SIGNAL,
};

static guint signals[LAST_SIGNAL] = { 0 };

static void
service_dispose(GObject *object)
{
  Service *self = SERVICE(object);
  ServicePrivate *priv = GET_PRIVATE(self);

  G_OBJECT_CLASS (service_parent_class)->dispose (object);
}

static void service_finalize(GObject *object)
{
  Service *self = SERVICE(object);
  ServicePrivate *priv = GET_PRIVATE(self);

  g_free(priv->identifier);
  g_free(priv->name);

  G_OBJECT_CLASS (service_parent_class)->finalize (object);
}

static void service_class_init(ServiceClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  g_type_class_add_private (klass, sizeof (ServicePrivate));

  object_class->dispose = service_dispose;
  object_class->finalize = service_finalize;

  signals[STATE_CHANGED] = g_signal_new("state-changed",
					G_TYPE_FROM_CLASS(klass),
					G_SIGNAL_RUN_LAST,
					0, NULL, NULL,
					_marshal_VOID__VOID,
					G_TYPE_NONE, 0);

  signals[STRENGTH_UPDATED] = g_signal_new("strength-updated",
					G_TYPE_FROM_CLASS(klass),
					G_SIGNAL_RUN_LAST,
					0, NULL, NULL,
					_marshal_VOID__VOID,
					G_TYPE_NONE, 0);

  DbusmenuMenuitemClass *mclass = DBUSMENU_MENUITEM_CLASS(klass);
  mclass->handle_event = handle_event;

}

static void service_init(Service *self)
{
}

static const gchar *get_disconnected_icon(Service *self)
{
  ServicePrivate *priv = GET_PRIVATE(self);

  g_return_val_if_fail(priv != NULL, NULL);

  switch (priv->type) {
  case ANDROID_NETWORK_TYPE_WIFI:
    return ICON_NOTIFICATION_WIFI_DISCONNECTED;
  case ANDROID_NETWORK_TYPE_MOBILE:
    return ICON_NOTIFICATION_CELLULAR_DISCONNECTED;
  default:
    return ICON_NOTIFICATION_WIRED_DISCONNECTED;
  }
}

static void service_connect_cb(GObject *object, GAsyncResult *res,
				  gpointer user_data)
{
#if 0
  Service *self = user_data;
  ServicePrivate *priv = GET_PRIVATE(self);
  const gchar *summary, *msg, *icon;
  GError *error = NULL;

  g_return_if_fail(priv != NULL);

  void_service_connect_finish(priv->service, res, &error);

  if (error == NULL)
    /* method call had no errors */
    return;

  /* FIXME: don't show errors if passphrase is requested through agent */

  g_warning("[%p] failed to connect service %s: %s",
	    self, void_service_get_path(priv->service), error->message);

  summary = void_service_get_name(priv->service);
  msg = _("Connection Failed");
  icon = get_disconnected_icon(self);

  manager_notify(priv->network_service, summary, msg, icon);

  g_error_free(error);
#endif
}

static void service_connect(Service *self)
{
#if 0
  ServicePrivate *priv = GET_PRIVATE(self);

  g_return_if_fail(priv != NULL);

  g_debug("[%p] connect", self);

  void_service_connect(priv->service, service_connect_cb, self);
#endif
}

static void service_disconnect_cb(GObject *object, GAsyncResult *res,
				  gpointer user_data)
{
#if 0
  Service *self = SERVICE(user_data);
  ServicePrivate *priv = GET_PRIVATE(self);
  const gchar *summary, *msg, *icon;
  GError *error = NULL;

  g_return_if_fail(priv != NULL);

  /////// !!!!!
  // void_service_disconnect_finish(priv->service, res, &error);

  if (error == NULL)
    /* method call had no errors */
    return;

  g_warning("[%p] failed to disconnect service %s: %s",
	    self, void_service_get_path(priv->service), error->message);

  summary = void_service_get_name(priv->service);
  msg = _("Disconnection Failed");
  icon = get_disconnected_icon(self);

  manager_notify(priv->network_service, summary, msg, icon);

  g_error_free(error);
#endif
}

static void service_disconnect(Service *self)
{
#if 0
  ServicePrivate *priv = GET_PRIVATE(self);

  g_return_if_fail(priv != NULL);
  g_return_if_fail(priv->service != NULL);

  g_debug("[%p] disconnect", self);

  void_service_disconnect(priv->service, service_disconnect_cb, self);
#endif
}

static void service_item_activated(DbusmenuMenuitem *mi, guint timestamp,
				   gpointer user_data)
{
}

static void handle_event (DbusmenuMenuitem * mi, const gchar * name,
                          GVariant * value, guint timestamp)
{

  Service *self = SERVICE(mi);
  ServicePrivate *priv = GET_PRIVATE(self);
  AndroidService *android_service = NULL;

  g_return_if_fail(priv != NULL);

  if (g_strcmp0(name, "indicator-network-service-activated") != 0)
    return;

  g_debug("service activated: %s", priv->identifier);

  android_service = manager_get_android_service(priv->network_service);

  switch (priv->state) {
  case ANDROID_NETWORK_STATE_DISCONNECTED:
    android_service_enable_network_with_identifier(android_service,
						   priv->identifier);
    break;
  case ANDROID_NETWORK_STATE_CONNECTED:
  case ANDROID_NETWORK_STATE_CONNECTING:
    android_service_disconnect_with_identifier(android_service,
					       priv->identifier);
    break;
  default:
    break;
  }

}

static void service_update_label(Service *self)
{
  ServicePrivate *priv = GET_PRIVATE(self);

  g_return_if_fail(priv);

  dbusmenu_menuitem_property_set(DBUSMENU_MENUITEM(self),
				 DBUSMENU_MENUITEM_PROP_LABEL,
				 priv->name);
}

static void service_update_checkmark(Service *self)
{
  ServicePrivate *priv = GET_PRIVATE(self);
  gint state;
  AndroidService *android_service;

  g_return_if_fail(priv);

  switch (priv->state) {
  case ANDROID_NETWORK_STATE_CONNECTED:
  case ANDROID_NETWORK_STATE_CONNECTING:
    state = DBUSMENU_MENUITEM_TOGGLE_STATE_CHECKED;
    break;
  default:    
    state = DBUSMENU_MENUITEM_TOGGLE_STATE_UNCHECKED;
  }

  dbusmenu_menuitem_property_set_int(DBUSMENU_MENUITEM(self),
				     DBUSMENU_MENUITEM_PROP_TOGGLE_STATE,
				     state);
}

static void show_connect_error(Service *self)
{
#if 0
  ServicePrivate *priv = GET_PRIVATE(self);
  const gchar *error_type = "general-connect-error";
  ServiceSecurity security;
  ServiceState state;
  ServiceType type;

  g_return_if_fail(priv != NULL);

  type = void_service_get_service_type(priv->service);
  security = void_service_get_security(priv->service);
  state = void_service_get_state(priv->service);

  if (type == SERVICE_TYPE_WIFI) {
    if (security == SERVICE_SECURITY_NONE) {
      error_type = "wifi-open-connect-error";
    }
    else if (security == SERVICE_SECURITY_WEP) {
      if (state == SERVICE_STATE_CONFIGURATION)
	error_type = "wifi-wep-dhcp-error";
      else
	error_type = "wifi-wep-connect-error";
    }
    else if (security == SERVICE_SECURITY_PSK) {
      if (state == SERVICE_STATE_CONFIGURATION)
	error_type = "wifi-wpa-dhcp-error";
      else
	error_type = "wifi-wpa-connect-error";
    }
  }

  /* FIXME: show error dialogs */
#endif
}

#if 0
static void state_updated(VoidService *service, GParamSpec *pspec,
			  gpointer user_data)
{

  Service *self = SERVICE(user_data);
  ServicePrivate *priv = GET_PRIVATE(self);
  ServiceState new_state;
  ServiceType type;
  const gchar *icon;

  g_return_if_fail(priv != NULL);

  new_state = void_service_get_state(priv->service);

  g_debug("[%p] %s -> %s", self,
	  service_state2str(priv->state),
	  service_state2str(new_state));

  /* only show state transitions */
  if (priv->state == new_state)
    return;

  type = void_service_get_service_type(priv->service);

  switch (new_state) {
  case SERVICE_STATE_READY:
    if (type == SERVICE_TYPE_WIFI)
      icon = ICON_NOTIFICATION_WIFI_FULL;
    else if (type == SERVICE_TYPE_ETHERNET)
      icon = ICON_NOTIFICATION_WIRED_CONNECTED;
    else if (type == SERVICE_TYPE_CELLULAR)
      icon = ICON_NOTIFICATION_CELLULAR_CONNECTED;
    else
      icon = NULL;
    manager_notify(priv->network_service,
		   void_service_get_name(priv->service),
		   _("Connection Established"), icon);
    break;
  case SERVICE_STATE_IDLE:
    icon = get_disconnected_icon(self);
    manager_notify(priv->network_service,
		   void_service_get_name(priv->service),
		   _("Disconnected"), icon);
    break;
  case SERVICE_STATE_FAILURE:
    icon = get_disconnected_icon(self);
    manager_notify(priv->network_service,
		   void_service_get_name(priv->service),
		   _("Connection Failed"), icon);
    break;
  case SERVICE_STATE_ASSOCIATION:
  case SERVICE_STATE_CONFIGURATION:
  default:
    /* do nothing */
    break;
  }

  if ((priv->state == SERVICE_STATE_ASSOCIATION ||
       priv->state == SERVICE_STATE_CONFIGURATION) &&
      new_state ==SERVICE_STATE_FAILURE) {
    show_connect_error(self);
  }

  priv->state = new_state;
  service_update_label(self);
  service_update_checkmark(self);

  g_signal_emit(self, signals[STATE_CHANGED], 0);
}
#endif

static void service_update_icon(Service *self)
{
  ServicePrivate *priv = GET_PRIVATE(self);
  const gchar *icon;

  g_return_if_fail(priv != NULL);

  /// @todo does not work with mobile, but we don't use the Service classes for mobile anyway
  icon = manager_wifi_icon_name(priv->strength);

  dbusmenu_menuitem_property_set(DBUSMENU_MENUITEM(self),
				 DBUSMENU_MENUITEM_PROP_ICON_NAME,
				 icon);
}

#if 0
static void strength_updated(VoidService *service, GParamSpec *pspec,
			 gpointer user_data)
{
  Service *self = SERVICE(user_data);

  service_update_icon(self);

  g_signal_emit(self, signals[STRENGTH_UPDATED], 0);
}
#endif 

static void setup_required(Service *self)
{
#if 0
  ServicePrivate *priv = GET_PRIVATE(self);
  ServiceType type;
  gchar *argv[3];
  gchar *cmd = BINDIR "/indicator-network-mobile-wizard";

  g_return_if_fail(priv);

  g_debug("[%p] setup required", self);

  type = void_service_get_service_type(priv->service);

  /* mobile wizard is only for cellular */
  if (type != SERVICE_TYPE_CELLULAR)
    return;

  argv[0] = cmd;
  argv[1] = g_strdup(void_service_get_path(priv->service));
  argv[2] = 0;

  /*
   * FIXME: for the prototype just start the script, but with the proper UI
   * we use dbus
   */
  g_debug("%s(): starting %s %s", __func__, argv[0], argv[1]);
  g_spawn_async(NULL, argv, NULL, 0, NULL, NULL, NULL, NULL);

  g_free(argv[1]);
#endif
}

#if 0
static void name_updated(VoidService *service, GParamSpec *pspec,
			 gpointer user_data)
{

  Service *self = SERVICE(user_data);
  ServicePrivate *priv = GET_PRIVATE(self);

  g_return_if_fail(priv != NULL);

  g_debug("[%p] name %s", self, void_service_get_name(priv->service));

  service_update_label(self);
}
#endif

Service *service_new(AndroidServiceNetwork *network,
		     Manager *ns)
{
  ServicePrivate *priv;
  Service *self;
  gint mode;

  g_return_val_if_fail(ns != NULL, NULL);

  self = g_object_new(TYPE_SERVICE, NULL);
  priv = GET_PRIVATE(self);

  g_return_val_if_fail(self != NULL, NULL);
  g_return_val_if_fail(priv != NULL, NULL);

  priv->network_service = ns;

  priv->identifier = g_strdup(network->identifier);
  priv->type       = network->type;
  priv->state      = android_service_network_get_state(network);
  priv->strength   = android_service_network_get_strength(network);
  priv->name       = g_strdup(network->ssid);
  priv->security   = network->security;
  
  /* FIXME: follow signals from priv->service */

  // g_debug("[%p] service new %s", self, priv->network->ssid);

  dbusmenu_menuitem_property_set(DBUSMENU_MENUITEM(self),
				 DBUSMENU_MENUITEM_PROP_TYPE,
				 SERVICE_MENUITEM_NAME);
  dbusmenu_menuitem_property_set(DBUSMENU_MENUITEM(self),
				 DBUSMENU_MENUITEM_PROP_TOGGLE_TYPE,
				 DBUSMENU_MENUITEM_TOGGLE_CHECK);

  if (priv->security == ANDROID_NETWORK_SECURITY_NONE)
    mode = SERVICE_MENUITEM_PROP_OPEN;
  else
    mode = SERVICE_MENUITEM_PROP_PROTECTED;

  dbusmenu_menuitem_property_set_int(DBUSMENU_MENUITEM(self),
				     SERVICE_MENUITEM_PROP_PROTECTION_MODE,
				     mode);

  service_update_label(self);
  service_update_checkmark(self);
  service_update_icon(self);


  g_signal_connect(G_OBJECT(self),
		   DBUSMENU_MENUITEM_SIGNAL_ITEM_ACTIVATED,
		   G_CALLBACK(service_item_activated), self);

  g_signal_connect(network,
		   "strength-changed",
		   G_CALLBACK(network_strength_changed),
		   self);

  g_signal_connect(network,
		   "state-changed",
		   G_CALLBACK(network_state_changed),
		   self);

#if 0
  g_signal_connect(priv->service, "notify::state", G_CALLBACK(state_updated),
		   self);
  g_signal_connect(priv->service, "notify::name", G_CALLBACK(name_updated),
		   self);
  if (void_service_get_setup_required(priv->service))
    setup_required(self);
#endif

  return self;
}

const gchar *
service_get_identifier(Service *self)
{
  ServicePrivate *priv = GET_PRIVATE(self);

  return priv->identifier;
}

AndroidNetworkSecurity
service_get_security(Service *self)
{
  ServicePrivate *priv = GET_PRIVATE(self);

  return priv->security;
}

/* service_get_type() existed already, use "service_type" instead */
AndroidNetworkType service_get_service_type(Service *self)
{
  ServicePrivate *priv = GET_PRIVATE(self);

  return priv->type;
}

AndroidNetworkState service_get_state(Service *self)
{
  ServicePrivate *priv = GET_PRIVATE(self);

  return priv->state;
}

gint service_get_strength(Service *self)
{
  ServicePrivate *priv = GET_PRIVATE(self);

  g_return_val_if_fail(IS_SERVICE(self), 0);
  g_return_val_if_fail(priv != NULL, 0);

  return priv->strength;
}

const gchar *service_get_name(Service *self)
{
  ServicePrivate *priv = GET_PRIVATE(self);

  g_return_val_if_fail(IS_SERVICE(self), NULL);
  g_return_val_if_fail(priv != NULL, NULL);

  return priv->name;
}

void
network_strength_changed(gpointer object,
			 gint strength,
			 gpointer user_data) 
{
  Service *self        = SERVICE(user_data);
  ServicePrivate *priv = GET_PRIVATE(self);

  g_debug("%s %d", __func__, strength);
  priv->strength = strength;

  service_update_icon(self);
}

void
network_state_changed(gpointer object,
		      gint state,
		      gpointer user_data) 
{
  Service *self        = SERVICE(user_data);
  ServicePrivate *priv = GET_PRIVATE(self);

  g_debug("%s %d", __func__, state);
  priv->state = state;

  service_update_label(self);
  service_update_checkmark(self);
}

