# -*- encoding: utf-8 -*-
#
# Copyright 2009 Canonical Ltd.
#
# Written by:
#     Gustavo Niemeyer <gustavo.niemeyer@canonical.com>
#     Sidnei da Silva <sidnei.da.silva@canonical.com>
#
# This file is part of the Image Store Proxy.
#
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
import commands
import tempfile
import hashlib
import datetime
import logging
import copy
import time
import sys
import os

from twisted.web import server, resource, static
from twisted.internet import reactor

try:
    import json
except ImportError:
    import simplejson as json

from imagestore.lib import bpickle


# DISCLAIMER: This is an UGLY HACK to test the UI logic.  This code will be
#             TRASHED and replaced by a descent implementation at the earliest
#             chance.


SUPPORTED_API_VERSIONS = ["2009-10-01"]

def sha256(path, save=True):
    log("Computing sha256 for path %s" % path)
    sha256Path = path + ".sha256"
    if os.path.exists(sha256Path):
        sha256File = open(sha256Path, "r")
        checksum = sha256File.read()
        sha256File.close()
        return checksum
    sha256 = hashlib.sha256()
    handle = open(path, "r")
    try:
        while True:
            data = handle.read(1<<20)
            if not data:
                break
            sha256.update(data)
    finally:
        handle.close()

    checksum = sha256.hexdigest()
    if save:
        sha256File = open(sha256Path, "w")
        sha256File.write(checksum)
        sha256File.close()
    return checksum

FILES = [
    ]


ALFRESCO_IMAGE = {
    "uri": "http://imagestore.canonical.com/api/images/b8b4680a-f218-4e85-8671-002705e38e34",
    "title": u"Alfresco Server (áéíóú)",
    "version": "1.0",
    "summary": "Open source content management system",
    "size-in-mb": 640,
    "description-html":
        "<p>Open source enterprise content management system including "
        "document management, web content management, SharePoint "
        "alternative and content repository.</p>"
        "<ul><li>This<li>is<li>html!</ul>",
    "provider": {
        "uri": "http://www.alfresco.com",
        "title": "Alfresco"
        },
    "files": FILES,
    }

WEB_SERVER_IMAGE = {
    "uri": "http://imagestore.canonical.com/api/images/fd82787c-f346-4426-aa2b-c75b7d5544a7",
    "title": "Ubuntu Apache Web Server",
    "version": "1.5",
    "summary": "Basic apache-based appliance",
    "tags": ["ubuntu", "apache", "web", "server"],
    "size-in-mb": 520,
    "description-html":
        "Basic appliance based on Apache.",
    "provider": {
        "uri": "http://www.canonical.com",
        "title": "Canonical"
        },
    "files": FILES,
    }

TOMCAT_IMAGE_14 = {
    "uri": "http://imagestore.canonical.com/api/images/f2644bd5-d598-471c-9e14-d910b1a47c49",
    "title": "Tomcat",
    "version": "1.4",
    "summary": "Java application server",
    "size-in-mb": 700,
    "description-html":
        "Robust Java application server backed by the Apache Foundation.",
    "provider": {
        "uri": "http://www.apache.org",
        "title": "Apache Foundation"
        },
    "files": FILES,
    }

TOMCAT_IMAGE_15 = {
    "uri": "http://imagestore.canonical.com/api/images/f2644bd5-d598-471c-9e14-d910b1a47c50",
    "title": "Tomcat",
    "version": "1.5",
    "summary": "Java application server",
    "size-in-mb": 700,
    "description-html":
        "Robust Java application server backed by the Apache Foundation.",
    "provider": {
        "uri": "http://www.apache.org",
        "title": "Apache Foundation"
        },
    "files": FILES,
    }

DJANGO_IMAGE_09 = {
    "uri": "http://imagestore.canonical.com/api/images/0893b94e-436c-49d2-bce8-07c189128b15",
    "title": "Django",
    "version": "0.9",
    "summary": "Django web framework",
    "description-html":
        "Python-based web framework.",
    "files": FILES,
    }


DJANGO_IMAGE_10 = {
    "uri": "http://imagestore.canonical.com/api/images/0893b94e-436c-49d2-bce8-07c189128b16",
    "title": "Django",
    "version": "1.0",
    "summary": "Django web framework",
    "description-html":
        "Python-based web framework.",
    "files": FILES,
    }

KARMIC_IMAGE = {
    "uri": "http://imagestore.canonical.com/api/images/619447e9-b746-43d7-91c5-4a49efe40a64",
    "title": "Ubuntu Karmic",
    "version": "1.0",
    "size-in-mb": 310,
    "summary": "Basic Ubuntu Karmic image",
    "description-html":
        "Yet another excellent Ubuntu image for your pleasure.",
    "files": FILES,
    }


IMAGES = [
    TOMCAT_IMAGE_14,
    ALFRESCO_IMAGE,
    WEB_SERVER_IMAGE,
    DJANGO_IMAGE_09,
    KARMIC_IMAGE,
    ]

UPGRADE_IMAGES = [
    TOMCAT_IMAGE_15,
    DJANGO_IMAGE_10
    ]

UPGRADE_SECTION = {"title": "Upgrades",
                   "summary": "The following recent versions of appliances you have installed are available:",
                   "image-uris": [],
                   }

DASHBOARD_SECTIONS = [
    {"title": "Highlights",
     "summary": "Fantastic images brought by Ubuntu and Eucalyptus to you:",
     "image-uris": [
         ALFRESCO_IMAGE["uri"],
         WEB_SERVER_IMAGE["uri"],
         ],
    },
    {"title": "New",
     "summary": "Appliances recently made available:",
     "image-uris": [
         DJANGO_IMAGE_09["uri"],
         ALFRESCO_IMAGE["uri"],
         ],
    },
    ]

SEARCH_SECTIONS = [
    {"title": "Search results",
     "summary": "The following results were found:",
     "image-uris": [],
    },
    ]


def log(*args):
    print datetime.datetime.now(), " ".join(str(x) for x in args)


def signImage(image):
    image["signature-fields"] = ["title", "version", "files",
                                 "signature-fields"]
    image["signature"] = ""

    signatureDict = {}
    for key in image["signature-fields"]:
        signatureDict[key] = image[key]

    # Transform all strings to unicode.
    signatureDict = json.loads(json.dumps(signatureDict))

    signaturePayload = bpickle.dumps(signatureDict)
    signaturePayloadFD, signaturePayloadPath = tempfile.mkstemp()
    try:
        os.write(signaturePayloadFD, signaturePayload)
        os.fsync(signaturePayloadFD)

        signaturePath = tempfile.mktemp()
        try:
            command = ("gpg --batch --no-default-keyring --secret-keyring %s "
                       "--keyring %s --armor --output %s --detach-sign %s"
                       % ("./fake-secring.gpg", "./fake-pubring.gpg",
                          signaturePath, signaturePayloadPath))
            status, output = commands.getstatusoutput(command)
            if status != 0:
                logging.error("GPG failed signing image:\n%s" % (output,))

            signatureFile = open(signaturePath)
            try:
                image["signature"] = signatureFile.read()
            finally:
                signatureFile.close()

        finally:
            if os.path.isfile(signaturePath):
                os.unlink(signaturePath)
    finally:
        os.close(signaturePayloadFD)
        os.unlink(signaturePayloadPath)


def buildImageAnswer(image, request):
    p = request.URLPath()
    p.path = "/images/"
    p.fragment = p.query = ""
    serverRoot = str(p)
    image = copy.deepcopy(image)
    for imageFile in image["files"]:
        imageFile["url"] = serverRoot + imageFile["url"]
    signImage(image)
    return image


def buildAnswer(request, sections=None, image_uris=None):
    answer = {}
    answer["images"] = images = []
    done = set()
    if image_uris is not None:
        for image in IMAGES:
            image_uri = image["uri"]
            if image_uri not in done and image_uri in image_uris:
                done.add(image_uri)
                images.append(buildImageAnswer(image, request))
    if sections is not None:
        answer["sections"] = sections
        for section in sections:
            for image_uri in section["image-uris"]:
                if image_uri not in done:
                    done.add(image_uri)
                    for image in IMAGES:
                        if image["uri"] == image_uri:
                            images.append(buildImageAnswer(image, request))
                            break
                    for image in UPGRADE_IMAGES:
                        if image["uri"] == image_uri:
                            images.append(buildImageAnswer(image, request))
                            break
    return answer


class DashboardResource(resource.Resource):

    isLeaf = True

    def render_POST(self, request):
        log("Loading dashboard")
        sections = DASHBOARD_SECTIONS[:]
        upgrade_section = UPGRADE_SECTION.copy()
        upgrade_uris = []
        for upgrade_uri in request.args.get("upgrade-uri", ()):
            for image in UPGRADE_IMAGES:
                if image["uri"] == upgrade_uri:
                    upgrade_uris.append(upgrade_uri)
                    break
        if upgrade_uris:
            upgrade_section["image-uris"] = upgrade_uris
            sections.insert(0, upgrade_section)
        return json.dumps(buildAnswer(request, sections))


class UpgradesResource(resource.Resource):

    isLeaf = True

    def render_POST(self, request):
        image_uris = request.args.get("image-uri", [])
        log("Loading upgrades for %d URIs" % len(image_uris))
        upgrade_image_uris = []
        for image_uri in image_uris:
            if image_uri == DJANGO_IMAGE_09["uri"]:
                upgrade_image_uris.append(DJANGO_IMAGE_10["uri"])
            elif image_uri == TOMCAT_IMAGE_14["uri"]:
                upgrade_image_uris.append(TOMCAT_IMAGE_15["uri"])
        return json.dumps(buildAnswer(request, image_uris=upgrade_image_uris))


class SearchResource(resource.Resource):

    isLeaf = True

    def render_GET(self, request):
        params = dict(request.args)
        search_param = params.get("q", [None])[0]
        log("Searching for", search_param)

        sections = copy.deepcopy(SEARCH_SECTIONS)
        found_image_uris = sections[0]["image-uris"]

        if search_param:
            search_param = search_param.lower().decode("UTF-8")
            for image in IMAGES:
                if (search_param in image["title"].lower() or
                    search_param in image["summary"].lower() or
                    search_param in image["description-html"].lower()):
                    found_image_uris.append(image["uri"])

        if not found_image_uris:
            sections[0]["summary"] = "No results were found."

        time.sleep(1) # Simulate the handling of this request.

        return json.dumps(buildAnswer(request, sections))


class Root(resource.Resource):

    prefix = ["api"]

    isLeaf = False

    def __init__(self, imagesPath):
        self.imagesPath = imagesPath
        resource.Resource.__init__(self)

    def getChild(self, path, request):
        if path == "images":
            return static.File(self.imagesPath)

        request.prepath.pop()
        request.postpath.insert(0, path)

        if request.postpath[:len(self.prefix)] == self.prefix:
            del request.postpath[:len(self.prefix)]
            request.prepath.extend(self.prefix)
            return CloudRoot()
        return NotFound()


class NotFound(resource.Resource):

    isLeaf = False

    def render_GET(self, request):
        pass


class ImageResource(resource.Resource):

    def __init__(self, uri):
        resource.Resource.__init__(self)
        self._uri = uri

    def render_GET(self, request):
        assert False, "Implement this."


class ImagesResource(resource.Resource):

    isLeaf = False

    def getChild(self, name, request):
        return ImageResource(name)


class CloudRoot(resource.Resource):

    def __init__(self):
        resource.Resource.__init__(self)
        self.putChild("dashboard", DashboardResource())
        self.putChild("upgrades", UpgradesResource())
        self.putChild("search", SearchResource())
        self.putChild("images", ImagesResource())


if __name__ == "__main__":
    if len(sys.argv) < 2:
        raise ValueError("A directory to serve images from is required")
    basePath = os.path.normpath(os.path.abspath(sys.argv[1]))
    for kind in ("kernel", "ramdisk", "image"):
        for extension in ("", ".gz", ".tar.gz"):
            filePath = os.path.join(basePath, kind) + extension
            if os.path.isfile(filePath):
                imageFile = {
                    "size-in-bytes": os.path.getsize(filePath),
                    "kind": kind,
                    "sha256": sha256(filePath),
                    "url": kind + extension,
                    }
                FILES.append(imageFile)

    site = server.Site(Root(sys.argv[1]))
    reactor.listenTCP(52781, site)
    log("Ready to serve requests")
    reactor.run()
