#include <config.h>

#include "hrn.h"
#include "hrn-tile-frame.h"
#include "hrn-texture-cache.h"

enum {
    PROP_0,
    PROP_TEXTURE,
    PROP_LABEL
};

enum {
    PRIMARY,
    SECONDARY,
    LAST_SIGNAL,
};

struct _HrnTileFramePrivate {
    ClutterActor *group;
    ClutterActor *texture;

    ClutterActor *overlay_group;
    ClutterActor *overlay;
    ClutterActor *play_icon;
    ClutterActor *label;
};

#define GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), HRN_TYPE_TILE_FRAME, HrnTileFramePrivate))
G_DEFINE_TYPE (HrnTileFrame, hrn_tile_frame, NBTK_TYPE_BIN);
static guint32 signals[LAST_SIGNAL] = {0,};

static void
hrn_tile_frame_finalize (GObject *object)
{
    G_OBJECT_CLASS (hrn_tile_frame_parent_class)->finalize (object);
}

static void
hrn_tile_frame_dispose (GObject *object)
{
    HrnTileFrame *self = (HrnTileFrame *) object;
    HrnTileFramePrivate *priv = self->priv;

    if (priv->texture) {
        g_object_unref (priv->texture);
        priv->texture = NULL;
    }

    G_OBJECT_CLASS (hrn_tile_frame_parent_class)->dispose (object);
}

static gboolean
texture_released_cb (ClutterActor       *actor,
                     ClutterButtonEvent *event,
                     HrnTileFrame       *frame)
{
    if (event->button != 1) {
        return FALSE;
    }

    g_signal_emit (frame, signals[PRIMARY], 0);

    return TRUE;
}

static void
hrn_tile_frame_set_property (GObject      *object,
                             guint         prop_id,
                             const GValue *value,
                             GParamSpec   *pspec)
{
    HrnTileFrame *self = (HrnTileFrame *) object;
    HrnTileFramePrivate *priv = self->priv;
    const char *label;

    switch (prop_id) {
    case PROP_TEXTURE:
        priv->texture = g_value_dup_object (value);

        clutter_actor_set_reactive (priv->texture, TRUE);
        g_signal_connect (priv->texture, "button-release-event",
                          G_CALLBACK (texture_released_cb), self);
        clutter_container_add_actor ((ClutterContainer *) priv->group,
                                     priv->texture);
        clutter_actor_lower (priv->texture, priv->overlay_group);
        break;

    case PROP_LABEL:
        label = g_value_get_string (value);
        nbtk_label_set_text ((NbtkLabel *) priv->label, label);
        break;

    default:
        break;
    }
}

static void
hrn_tile_frame_get_property (GObject    *object,
                             guint       prop_id,
                             GValue     *value,
                             GParamSpec *pspec)
{
    switch (prop_id) {

    default:
        break;
    }
}

static void
hrn_tile_frame_allocate (ClutterActor          *actor,
                         const ClutterActorBox *box,
                         ClutterAllocationFlags flags)
{
    HrnTileFrame *frame = (HrnTileFrame *) actor;
    HrnTileFramePrivate *priv = frame->priv;
    ClutterActorBox child_box;

    CLUTTER_ACTOR_CLASS (hrn_tile_frame_parent_class)->allocate (actor, box,
                                                                 flags);
    child_box.x1 = 5;
    child_box.y1 = 2;
    child_box.x2 = box->x2 - box->x1 - 5;
    child_box.y2 = box->y2 - box->y1 - 7;

    if (priv->texture) {
        if (clutter_actor_is_rotated (priv->texture)) {
            ClutterActorBox tmp_box;

            tmp_box.x1 = 5;
            tmp_box.x2 = box->y2 - box->y1 - 5;
            tmp_box.y1 = 2;
            tmp_box.y2 = box->x2 - box->x1 - 8;

            clutter_actor_allocate (priv->texture, &tmp_box, flags);
        } else {
            clutter_actor_allocate (priv->texture, &child_box, flags);
        }
    }

    child_box.y1 = child_box.y2 - 28;
    clutter_actor_allocate (priv->overlay_group, &child_box, flags);

    child_box.x1 = 0;
    child_box.y1 = 0;
    child_box.x2 = box->x2 - box->x1 - 10;
    child_box.y2 = 28;
    clutter_actor_allocate (priv->overlay, &child_box, flags);

    child_box.y1 = 3;
    child_box.x1 = child_box.x2 - 25;
    child_box.y2 = 25;
    child_box.x2 = child_box.x1 + 22;
    clutter_actor_allocate (priv->play_icon, &child_box, flags);
}

static gboolean
hrn_tile_frame_enter (ClutterActor         *actor,
                      ClutterCrossingEvent *event)
{
    HrnTileFrame *frame = (HrnTileFrame *) actor;
    HrnTileFramePrivate *priv = frame->priv;

    nbtk_widget_set_style_pseudo_class ((NbtkWidget *) actor, "hover");
    clutter_actor_animate (priv->overlay_group,
                           CLUTTER_EASE_IN_OUT_CUBIC, 500,
                           "opacity", 0xff,
                           NULL);
    return FALSE;
}

static gboolean
hrn_tile_frame_leave (ClutterActor         *actor,
                      ClutterCrossingEvent *event)
{
    HrnTileFrame *frame = (HrnTileFrame *) actor;
    HrnTileFramePrivate *priv = frame->priv;

    nbtk_widget_set_style_pseudo_class ((NbtkWidget *) actor, NULL);
    clutter_actor_animate (priv->overlay_group,
                           CLUTTER_EASE_IN_OUT_CUBIC, 500,
                           "opacity", 0x00,
                           NULL);

    return FALSE;
}

static void
hrn_tile_frame_class_init (HrnTileFrameClass *klass)
{
    GObjectClass *o_class = (GObjectClass *) klass;
    ClutterActorClass *a_class = (ClutterActorClass *) klass;

    o_class->dispose = hrn_tile_frame_dispose;
    o_class->finalize = hrn_tile_frame_finalize;
    o_class->set_property = hrn_tile_frame_set_property;
    o_class->get_property = hrn_tile_frame_get_property;

    a_class->allocate = hrn_tile_frame_allocate;
    a_class->enter_event = hrn_tile_frame_enter;
    a_class->leave_event = hrn_tile_frame_leave;

    g_type_class_add_private (klass, sizeof (HrnTileFramePrivate));

    g_object_class_install_property (o_class, PROP_TEXTURE,
                                     g_param_spec_object ("texture", "", "",
                                                          CLUTTER_TYPE_ACTOR,
                                                          G_PARAM_WRITABLE |
                                                          G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_LABEL,
                                     g_param_spec_string ("label", "", "", "",
                                                          G_PARAM_WRITABLE |
                                                          G_PARAM_STATIC_STRINGS));

    signals[PRIMARY] = g_signal_new ("primary-action",
                                     G_TYPE_FROM_CLASS (klass),
                                     G_SIGNAL_RUN_FIRST |
                                     G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                     g_cclosure_marshal_VOID__VOID,
                                     G_TYPE_NONE, 0);
    signals[SECONDARY] = g_signal_new ("secondary-action",
                                       G_TYPE_FROM_CLASS (klass),
                                       G_SIGNAL_RUN_FIRST |
                                       G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                       g_cclosure_marshal_VOID__VOID,
                                       G_TYPE_NONE, 0);
}

static gboolean
overlay_released_cb (ClutterActor       *actor,
                     ClutterButtonEvent *event,
                     HrnTileFrame       *frame)
{
    if (event->button != 1) {
        return FALSE;
    }

    g_signal_emit (frame, signals[SECONDARY], 0);
    return TRUE;
}

static void
hrn_tile_frame_init (HrnTileFrame *self)
{
    HrnTileFramePrivate *priv = GET_PRIVATE (self);
    HrnTextureCache *cache = hrn_texture_cache_get_default ();
    ClutterColor black = { 0x00, 0x00, 0x00, 0xaa };

    self->priv = priv;
    priv->group = clutter_group_new ();

    clutter_actor_set_reactive ((ClutterActor *) self, TRUE);

    clutter_container_add_actor ((ClutterContainer *) self, priv->group);
    clutter_actor_set_size (priv->group, THUMBNAIL_WIDTH, THUMBNAIL_HEIGHT);
    clutter_actor_set_position (priv->group, 0, 0);

    priv->overlay_group = clutter_group_new ();
    clutter_actor_set_reactive (priv->overlay_group, TRUE);
    clutter_actor_set_opacity (priv->overlay_group, 0x00);
    clutter_container_add_actor ((ClutterContainer *) priv->group,
                                 priv->overlay_group);
    g_signal_connect (priv->overlay_group, "button-release-event",
                      G_CALLBACK (overlay_released_cb), self);

    priv->overlay = clutter_rectangle_new_with_color (&black);
    clutter_container_add_actor ((ClutterContainer *) priv->overlay_group,
                                 priv->overlay);

    priv->label = (ClutterActor*) nbtk_label_new ("");
    clutter_actor_set_name (priv->label, "tile-overlay-label");

#if 0
    height = clutter_actor_get_height (priv->label);
    clutter_actor_set_position (priv->label, 3.0, (28.0 - height) / 2.0);
#else
    clutter_actor_set_position (priv->label, 3.0, 5.0);
#endif
    clutter_container_add_actor ((ClutterContainer *) priv->overlay_group,
                                 priv->label);

    priv->play_icon = hrn_texture_cache_get_texture
        (cache, THEMEDIR "/Hover-frame-play.png");
    clutter_container_add_actor ((ClutterContainer *) priv->overlay_group,
                                 priv->play_icon);
}

