/*
 * Hornsey - Moblin Media Player.
 * Copyright © 2009 Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#define Y_OFFSET    50

#include <clutter/clutter.h>
#include <string.h>
#include <math.h>

#include <bickley/bkl-db.h>

#include "hrn.h"
#include "hrn-view.h"
#include "hrn-switcher.h"


static void hrn_view_pause (void);
static void hrn_view_go (void);
static void hrn_view_refilter             (HrnView *view);
typedef struct _HrnViewActorData           HrnViewActorData;
static void hrn_view_dispose              (GObject            *object);
static void hrn_view_finalize             (GObject            *object);
static void hrn_view_finalize             (GObject            *object);
static void hrn_view_set_property         (GObject            *object,
                                           guint               prop_id,
                                           const GValue       *value,
                                           GParamSpec         *pspec);
static void hrn_view_get_property         (GObject            *object,
                                           guint               prop_id,
                                           GValue             *value,
                                           GParamSpec         *pspec);
static void clutter_container_iface_init  (ClutterContainerIface *iface);
static void hrn_view_real_add             (ClutterContainer   *container,
                                           ClutterActor       *actor);
static void hrn_view_real_remove          (ClutterContainer   *container,
                                           ClutterActor       *actor);
static void hrn_view_real_foreach         (ClutterContainer   *container,
                                           ClutterCallback     callback,
                                           gpointer            user_data);
static void hrn_view_real_raise           (ClutterContainer   *container,
                                           ClutterActor       *actor,
                                           ClutterActor       *sibling);
static void hrn_view_real_lower           (ClutterContainer   *container,
                                           ClutterActor       *actor,
                                           ClutterActor       *sibling);
static void hrn_view_real_sort_depth_order(ClutterContainer   *container);
static void hrn_view_free_actor_data      (gpointer data);
static void hrn_view_paint                (ClutterActor       *actor);
static void hrn_view_pick                 (ClutterActor       *actor,
                                           const ClutterColor *color);
static void hrn_view_get_preferred_width  (ClutterActor       *self,
                                           gfloat              for_height,
                                           gfloat             *min_width_p,
                                           gfloat             *natural_width_p);
static void hrn_view_get_preferred_height (ClutterActor       *self,
                                           gfloat              for_width,
                                           gfloat             *min_height_p,
                                           gfloat             *natural_height_p);
static void hrn_view_allocate             (ClutterActor            *self,
                                           const ClutterActorBox  *box,
                                           ClutterAllocationFlags flags);
void        source_item_added             (HrnSource              *source,
                                           BklItem                *item,
                                           gpointer                userdata);
void        source_item_removed           (HrnSource              *source,
                                           BklItem                *item,
                                           gpointer                userdata);
void        source_item_changed           (HrnSource              *source,
                                           BklItem                *item,
                                           gpointer                userdata);


G_DEFINE_TYPE_WITH_CODE (HrnView, hrn_view,
                         CLUTTER_TYPE_ACTOR,
                         G_IMPLEMENT_INTERFACE (CLUTTER_TYPE_CONTAINER,
                                                clutter_container_iface_init));

#define HRN_VIEW_GET_PRIVATE(obj) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((obj), HRN_TYPE_VIEW, \
                                HrnViewPrivate))

struct _HrnViewPrivate
{
  gfloat        for_height, for_width;
  gfloat        pref_width, pref_height;
  gfloat        alloc_width, alloc_height;
  ClutterActor *group;


  gchar *sort;
  gint   center_item;       /* should be updated from center pos and not vice
                               versa */

  gboolean compute_pos;     /* set when the pos should be updated from the item
                               and not
                             * the other way around
                             */

  gdouble center_pos;
  gdouble target_center_pos;

  gdouble height;

  ClutterAllocationFlags flags;
  GHashTable            *hash_table;
  GList                 *list;

  gint labels_visible;

  gboolean first_of_batch;
  gfloat   a_current_sum, a_wrap;

  HrnSource *current_source;
  guint32    item_added_id;
  guint32    item_removed_id;
  guint32    item_changed_id;

  GPtrArray *results;

  ClutterActor *lasso;
  gint          lx, ly;
  GHashTable   *selection; /* what would currently be added/removed by
                              the current lasso */
  GHashTable *selected;

  gboolean zoom_lock; /* FIXME: This should really be refactored into
                         g_signal_handler_block, but the code is too complicated
                         to do that at the moment */
};


enum
{
  PROP_0
};


/* XXX: these shuld be elsewhere */
void hrn_zoom_slider_update ();
void hrn_loc_slider_update ();

struct _HrnViewActorData
{
  gint num;
  gint accum_pos;
  gint cells; /* the number of cells needed for item when
               * expanded.
               */
  gfloat xmod; /* the position of the first item */
};


static void
hrn_view_map (ClutterActor *self)
{
  HrnViewPrivate *priv = HRN_VIEW (self)->priv;

  CLUTTER_ACTOR_CLASS (hrn_view_parent_class)->map (self);

  if (priv->group)
    clutter_actor_map (CLUTTER_ACTOR (priv->group));
  if (priv->lasso)
    clutter_actor_map (CLUTTER_ACTOR (priv->lasso));
}

static void
hrn_view_unmap (ClutterActor *self)
{
  HrnViewPrivate *priv = HRN_VIEW (self)->priv;

  CLUTTER_ACTOR_CLASS (hrn_view_parent_class)->unmap (self);

  if (priv->group)
    clutter_actor_unmap (CLUTTER_ACTOR (priv->group));
  if (priv->lasso)
    clutter_actor_map (CLUTTER_ACTOR (priv->lasso));
}



static void
hrn_view_class_init (HrnViewClass *klass)
{
  GObjectClass      *gobject_class = (GObjectClass *) klass;
  ClutterActorClass *actor_class   = (ClutterActorClass *) klass;

  gobject_class->dispose  = hrn_view_dispose;
  gobject_class->finalize = hrn_view_finalize;

  gobject_class->set_property = hrn_view_set_property;
  gobject_class->get_property = hrn_view_get_property;

  actor_class->paint                = hrn_view_paint;
  actor_class->pick                 = hrn_view_pick;
  actor_class->map                  = hrn_view_map;
  actor_class->unmap                = hrn_view_unmap;
  actor_class->get_preferred_width  = hrn_view_get_preferred_width;
  actor_class->get_preferred_height = hrn_view_get_preferred_height;
  actor_class->allocate             = hrn_view_allocate;

  g_type_class_add_private (klass, sizeof (HrnViewPrivate));
}



static void
clutter_container_iface_init (ClutterContainerIface *iface)
{
  iface->add              = hrn_view_real_add;
  iface->remove           = hrn_view_real_remove;
  iface->foreach          = hrn_view_real_foreach;
  iface->raise            = hrn_view_real_raise;
  iface->lower            = hrn_view_real_lower;
  iface->sort_depth_order = hrn_view_real_sort_depth_order;
}

static guint dirty = 0;
static gboolean
idle_repos (gpointer view)
{
  hrn_view_reposition (view);
  dirty = 0;
  return FALSE;
}
static void
hrn_view_queue_reposition (HrnView *view)
{
  if (!dirty)
    dirty = g_idle_add (idle_repos, view);
}

static gboolean remove_from_selection = FALSE;

gboolean
hrn_view_has_selected (void)
{
  HrnViewPrivate *priv;

  priv = HRN_VIEW_GET_PRIVATE (hrn_view);

  return g_hash_table_size (priv->selected) > 0;
}

gboolean
hrn_view_is_selected (const gchar *uri)
{
  HrnViewPrivate *priv;

  priv = HRN_VIEW_GET_PRIVATE (hrn_view);

  if (remove_from_selection)
    {
      if (g_hash_table_lookup (priv->selected, uri))
        {
          if (!g_hash_table_lookup (priv->selection, uri))
            return TRUE;
        }
      else
        {
          if (g_hash_table_lookup (priv->selection, uri))
            return TRUE;
        }
    }
  else
    {
      if (g_hash_table_lookup (priv->selection, uri) ||
          g_hash_table_lookup (priv->selected, uri))
        return TRUE;
    }

  return FALSE;
}


gint
hrn_view_labels_visible (HrnView *self)
{
  HrnViewPrivate *priv;

  priv = HRN_VIEW_GET_PRIVATE (self);
  return priv->labels_visible;
}

static void
hrn_view_init (HrnView *self)
{
  HrnViewPrivate *priv;

  self->priv = priv = HRN_VIEW_GET_PRIVATE (self);

  /* do not unref in the hashtable, the reference is for now kept by the list
   * (double bookkeeping sucks)
   */
  priv->hash_table
    = g_hash_table_new_full (g_direct_hash,
                             g_direct_equal,
                             NULL,
                             hrn_view_free_actor_data);

  priv->group = clutter_group_new ();
  clutter_actor_set_parent (priv->group, CLUTTER_ACTOR (self));
  self->search    = NULL;
  priv->selection = g_hash_table_new_full (g_str_hash, g_str_equal,
                                           g_free, NULL);
  priv->selected = g_hash_table_new_full (g_str_hash, g_str_equal,
                                          g_free, NULL);

  priv->zoom_lock = FALSE;
}

static void
hrn_view_dispose (GObject *object)
{
  HrnView        *self = (HrnView *) object;
  HrnViewPrivate *priv;

  priv = self->priv;

  /* Destroy all of the children. This will cause them to be removed
     from the container and unparented */
  clutter_container_foreach (CLUTTER_CONTAINER (object),
                             (ClutterCallback) clutter_actor_destroy,
                             NULL);
  if (priv->group)
    {
      clutter_actor_destroy (priv->group);
      priv->group = NULL;
    }

  G_OBJECT_CLASS (hrn_view_parent_class)->dispose (object);
}

static void
hrn_view_finalize (GObject *object)
{
  HrnView        *self = (HrnView *) object;
  HrnViewPrivate *priv = self->priv;

  g_hash_table_destroy (priv->hash_table);

  G_OBJECT_CLASS (hrn_view_parent_class)->finalize (object);
}


static void
hrn_view_set_property (GObject *object, guint prop_id, const GValue *value,
                       GParamSpec   *pspec)
{
  switch (prop_id)
    {
      default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}

static void
hrn_view_get_property (GObject *object, guint prop_id, GValue     *value,
                       GParamSpec *pspec)
{
  switch (prop_id)
    {
      default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}


static void
hrn_view_free_actor_data (gpointer data)
{
  g_slice_free (HrnViewActorData, data);
}

static gboolean
handle_scroll (ClutterActor *actor, ClutterEvent *event, gpointer userdata)
{
  switch (event->scroll.direction)
    {
      case CLUTTER_SCROLL_UP:
        hrn_view_scroll (-0.1);
        break;

      case CLUTTER_SCROLL_DOWN:
        hrn_view_scroll (0.1);
        break;

      default:
        break;
    }
  return TRUE;
}


static gboolean
intersects (gint min, gint max, gint minb, gint maxb)
{
  if (minb <= max && minb >= min)
    return TRUE;
  if (min <= maxb && min >= minb)
    return TRUE;
  return FALSE;
}

void
hrn_cluster_update_selections (HrnView *view)
{
  HrnViewPrivate *priv = HRN_VIEW (view)->priv;
  GList          *iter;

  for (iter = priv->list; iter; iter = iter->next)
    {
      HrnCluster *cluster = iter->data;
      /* these calls could be moved into hrn-cluster */
      hrn_cluster_set_selected (cluster, hrn_cluster_get_selected (cluster));
    }
}

void
hrn_view_set_selected (HrnView *view, const gchar *uri, BklItem     *item,
                       gboolean state)
{
  HrnViewPrivate *priv = HRN_VIEW (view)->priv;

  if (state)
    {
      g_hash_table_insert (priv->selected, g_strdup (uri), item);
    }
  else
    {
      g_hash_table_remove (priv->selected, uri);
    }
}

GList *
hrn_view_get_selected (HrnView *view)

{
  HrnViewPrivate *priv = HRN_VIEW (view)->priv;

  return g_hash_table_get_values (priv->selected);
}

static guint select_capture = 0;
static gboolean
view_capture (ClutterActor *actor, ClutterEvent *event, gpointer view)
{
  gfloat          x, y;
  HrnViewPrivate *priv = HRN_VIEW (view)->priv;

  switch (clutter_event_type (event))
    {
      case CLUTTER_BUTTON_RELEASE:
        g_signal_handler_disconnect (actor, select_capture);
        select_capture = 0;

#if 1
        clutter_actor_animate (priv->lasso, CLUTTER_LINEAR, 250,
                               "opacity", 0,
                               /*                           "signal::completed",
                                  destroy_when_done, priv,*/
                               NULL);
#endif

        {
          ClutterModifierType state = event->button.modifier_state;
          GHashTableIter      iter;
          gpointer            key, value;

          g_hash_table_iter_init (&iter, priv->selection);
          while (g_hash_table_iter_next (&iter, &key, &value))
            {
              if (state & CLUTTER_CONTROL_MASK)
                {
                  if (g_hash_table_lookup (priv->selected, key))
                    {
                      hrn_view_set_selected (view, key, value, FALSE);
                    }
                  else
                    {
                      hrn_view_set_selected (view, key, value, TRUE);
                    }
                }
              else
                {
                  hrn_view_set_selected (view, key, value, TRUE);
                }
            }
        }
        hrn_cluster_update_selections (view);

        g_hash_table_remove_all (priv->selection);

        HRN_VIEW (view)->dirty_frames = TRUE;
        break;

      case CLUTTER_MOTION:

        x = event->motion.x;
        y = event->motion.y;

        clutter_actor_transform_stage_point (view, x, y, &x, &y);

        gint mx = MIN (x, priv->lx);
        gint my = MIN (y, priv->ly);
        gint mw = MAX (x, priv->lx) - mx;
        gint mh = MAX (y, priv->ly) - my;

        clutter_actor_set_position (priv->lasso,
                                    mx - 5 / hrn_get_zoom (), my - 5 /
                                    hrn_get_zoom ());
        clutter_actor_set_size (priv->lasso,
                                mw + 10 / hrn_get_zoom (), mh + 10 /
                                hrn_get_zoom ());


        g_hash_table_remove_all (priv->selection);

        if (hrn_get_zoom () < 1.5)
          {
            gint   no;
            GList *j = priv->list;
            for (no = 0; j; no++, j = j->next)
              {
                gint cx = hrn_view_compute_x (no, 0, hrn_get_zoom ());
                gint cy =
                  hrn_view_compute_y (no, 0,
                                      hrn_get_zoom ()) - priv->center_pos +
                  Y_OFFSET / hrn_get_zoom ();

                if (intersects (mx, mx + mw, cx, cx + DIM / 4 * 0.8) &&
                    intersects (my, my + mh, cy, cy + DIM / 4 * 0.8))
                  {
                    HrnCluster *cluster = j->data;
                    GList      *i;

                    for (i = cluster->results; i; i = i->next)
                      {
                        g_hash_table_insert (priv->selection,
                                             g_strdup (bkl_item_get_uri (i->
                                                                         data)),
                                             i->data);
                      }
                  }
              }
          }
        else
          {
            gint no;
            for (no = 0; no < hrn_view_get_count (); no++)
              {
                gint cx = hrn_view_compute_x (0, no, hrn_get_zoom ());
                gint cy =
                  hrn_view_compute_y (0, no,
                                      hrn_get_zoom ()) - priv->center_pos +
                  Y_OFFSET / hrn_get_zoom ();

                if (intersects (mx, mx + mw, cx, cx + DIM / 4 * 0.8) &&
                    intersects (my, my + mh, cy, cy + DIM / 4 * 0.8))
                  {
                    g_hash_table_insert (priv->selection,
                                         g_strdup (bkl_item_get_uri (
                                                     hrn_view_item_to_bkl_item (
                                                       no))),
                                         hrn_view_item_to_bkl_item (no));
                  }
              }
          }
        if (clutter_event_get_state (event) & CLUTTER_CONTROL_MASK)
          remove_from_selection = TRUE;
        else
          remove_from_selection = FALSE;

        break;

      default:
        break;
    }
  return TRUE;
}


static ClutterTexture *
select_tex (void)
{
  static ClutterTexture *ret = NULL;

  if (ret)
    return ret;
  ret =
    CLUTTER_TEXTURE (clutter_texture_new_from_file (PKGDATADIR "hrn-lasso.png",
                                                    NULL));
  return ret;
}

static gboolean
view_pressed (ClutterActor *actor, ClutterEvent *event, gpointer data)
{
  gfloat              x, y;
  HrnViewPrivate     *priv  = HRN_VIEW (actor)->priv;
  ClutterModifierType state = event->button.modifier_state;

  return TRUE;

  if (!((state & CLUTTER_SHIFT_MASK) ||
        (state & CLUTTER_CONTROL_MASK)))
    {
      g_hash_table_remove_all (priv->selected);
    }
  if (state & CLUTTER_CONTROL_MASK)
    {
      remove_from_selection = TRUE;
    }
  else
    {
      remove_from_selection = TRUE;
    }

  if (priv->lasso == NULL)
    priv->lasso = hrn_texture_frame_new (select_tex (), 5, 5, 5, 5);
  clutter_actor_set_opacity (priv->lasso, 255);

  hrn_texture_frame_set_scale_invariant (HRN_TEXTURE_FRAME (priv->lasso), TRUE);
  hrn_texture_frame_set_draw_middle (HRN_TEXTURE_FRAME (priv->lasso), TRUE);

  x = event->button.x;
  y = event->button.y;

  clutter_actor_transform_stage_point (actor, x, y, &x, &y);


  clutter_actor_set_position (priv->lasso,
                              x - 5 / hrn_get_zoom (), y - 5 / hrn_get_zoom ());
  clutter_actor_set_size (priv->lasso, 10 / hrn_get_zoom (), 10 / hrn_get_zoom ());
  clutter_actor_set_parent (priv->lasso, actor);

  priv->lx = x;
  priv->ly = y;

  select_capture = g_signal_connect (
    clutter_stage_get_default (), "captured-event",
    G_CALLBACK (view_capture), actor);

  {
    ClutterEvent cevent = *event;
    cevent.type = CLUTTER_MOTION;
    clutter_actor_event (actor, &cevent, FALSE);
    view_capture (clutter_stage_get_default (), &cevent, actor);
  }

  return TRUE;
}

ClutterActor *
hrn_view_new (void)
{
  ClutterActor *self = g_object_new (HRN_TYPE_VIEW, NULL);

  clutter_actor_set_scale (self, INITIAL_SCALE, INITIAL_SCALE);
  clutter_actor_set_reactive (self, TRUE);

  g_signal_connect (self, "button-press-event", G_CALLBACK (view_pressed), NULL);
  g_signal_connect (self, "scroll-event", G_CALLBACK (handle_scroll), NULL);

  {
    gfloat w, h;
    clutter_actor_get_size (self, &w, &h);
  }

  return self;
}

static void
hrn_view_real_add (ClutterContainer *container, ClutterActor     *actor)
{
  HrnViewPrivate   *priv;
  HrnViewActorData *data;

  g_return_if_fail (HRN_IS_VIEW (container));

  priv = HRN_VIEW (container)->priv;

  g_object_ref (actor);

  clutter_actor_set_parent (actor, CLUTTER_ACTOR (container));

  data = g_slice_alloc0 (sizeof (HrnViewActorData));

  priv->list = g_list_append (priv->list, actor);

  data->cells = hrn_cluster_get_cells (actor);

  g_hash_table_insert (priv->hash_table, actor, data);

  g_signal_emit_by_name (container, "actor-added", actor);

  g_object_unref (actor);

  hrn_view_queue_reposition (HRN_VIEW (container));
}


static void
hrn_view_real_remove (ClutterContainer *container, ClutterActor     *actor)
{
  HrnView        *layout = HRN_VIEW (container);
  HrnViewPrivate *priv   = layout->priv;

  g_object_ref (actor);

  if (g_hash_table_remove (priv->hash_table, actor))
    {
      clutter_actor_unparent (actor);


      g_signal_emit_by_name (container, "actor-removed", actor);

      if (CLUTTER_ACTOR_IS_VISIBLE (CLUTTER_ACTOR (layout)))
        clutter_actor_queue_redraw (CLUTTER_ACTOR (layout));
    }
  priv->list = g_list_remove (priv->list, actor);

  g_object_unref (actor);

  if (!dirty)
    dirty = g_idle_add (idle_repos, layout);
}

static void
hrn_view_real_foreach (ClutterContainer *container, ClutterCallback callback,
                       gpointer user_data)
{
  HrnView        *layout = HRN_VIEW (container);
  HrnViewPrivate *priv   = layout->priv;

  g_list_foreach (priv->list, (GFunc) callback, user_data);
}

static void
hrn_view_real_raise (ClutterContainer *container, ClutterActor     *actor,
                     ClutterActor     *sibling)
{
  /* STUB */
}

static void
hrn_view_real_lower (ClutterContainer *container, ClutterActor     *actor,
                     ClutterActor     *sibling)
{
  /* STUB */
}

static void
hrn_view_real_sort_depth_order (ClutterContainer *container)
{
  /* STUB */
}


static void
hrn_view_paint (ClutterActor *actor)
{
  HrnView        *layout = (HrnView *) actor;
  HrnViewPrivate *priv   = layout->priv;
  GList          *child_item;

  for (child_item = priv->list;
       child_item != NULL;
       child_item = child_item->next)
    {
      ClutterActor *child = child_item->data;

      g_assert (child != NULL);

      if (CLUTTER_ACTOR_IS_VISIBLE (child))
        clutter_actor_paint (child);
    }

  if (CLUTTER_ACTOR_IS_VISIBLE (priv->group))
    clutter_actor_paint (priv->group);

  if (priv->lasso)
    clutter_actor_paint (priv->lasso);
}

static void
hrn_view_pick (ClutterActor *actor, const ClutterColor *color)
{
  CLUTTER_ACTOR_CLASS (hrn_view_parent_class)
  ->pick (actor, color);
  /* we do not paint our background even when we are reactive
   * to save on picking draw calls
   */
  if (CLUTTER_ACTOR_IS_VISIBLE (actor))
    hrn_view_paint (actor);
}

static void
hrn_view_get_preferred_width (ClutterActor *self, gfloat for_height,
                              gfloat       *min_width_p,
                              gfloat       *natural_width_p)
{
  HrnView        *layout = (HrnView *) self;
  HrnViewPrivate *priv   = layout->priv;
  gfloat          natural_width;

  natural_width = 1024 * 2;
  if (min_width_p)
    *min_width_p = natural_width;
  if (natural_width_p)
    *natural_width_p = natural_width;

  priv->pref_width = natural_width;
}

static void
hrn_view_get_preferred_height (ClutterActor *self, gfloat for_width,
                               gfloat       *min_height_p,
                               gfloat       *natural_height_p)
{
  HrnView        *layout = (HrnView *) self;
  HrnViewPrivate *priv   = layout->priv;
  gfloat          natural_height;

  natural_height = 600 * 2;

  priv->for_width   = for_width;
  priv->pref_height = natural_height;

  if (min_height_p)
    *min_height_p = natural_height;
  if (natural_height_p)
    *natural_height_p = natural_height;
}

gfloat
compute_static_x (gint cluster_no, gint accumpos, gint power_of_two)
{
  gint i   = 1;
  gint dim = DIM;
  gint no  = cluster_no;

  if (power_of_two == 1)
    {
      no           = accumpos;
      power_of_two = 3;
      dim         /= 4;
    }
  else if (power_of_two == 0)
    {
      no           = accumpos;
      power_of_two = 2;
      dim         /= 4;
    }
  else if (power_of_two == -1)
    {
      no   = accumpos;
      dim *= 0.16;
      return rint ((no % 3) * dim);  /* not a power of two, the code should
                                        be refactored/renamed to better reflect
                                        a more flexible nature
                                      */
    }
  else if (power_of_two == -2)
    {
      no           = accumpos;
      power_of_two = 0;
      dim         /= 4;
    }
  i <<= power_of_two;
  return rint ((no % i) * dim);
}

gfloat
compute_static_y (gint cluster_no, gint accumpos, gint power_of_two)
{
  gint i   = 1;
  gint dim = DIM * HRN_ROW_SPACING;
  gint no  = cluster_no;

  if (power_of_two == 1)
    {
      no           = accumpos;
      power_of_two = 3;
      dim         /= 4;
    }
  else if (power_of_two == 0)
    {
      no           = accumpos;
      power_of_two = 2;
      dim         /= 4;
    }
  else if (power_of_two == -1)
    {
      no   = accumpos;
      dim *= 0.16;
      return rint ((no / 3) * dim);
    }
  else if (power_of_two == -2)
    {
      no           = accumpos;
      power_of_two = 0;
      dim         /= 4;
    }
  i <<= power_of_two;

  return rint ((no / i) * dim);
}


#define HRN_SCALE_A    0.25
#define HRN_SCALE_B    0.5
#define HRN_SCALE_C    1.0
#define HRN_SCALE_D    2.0
#define HRN_SCALE_E    4.0
#define HRN_SCALE_F    8.0
#define HRN_SCALE_G    16.0

gfloat
hrn_view_compute_x (gint num, gint accum_pos, gdouble scale)
{
  gdouble progress = 0.5;

  if (scale < HRN_SCALE_A)
    {
      return compute_static_x (num, accum_pos, 4) + HRN_X_OFFSET / scale;
    }
  else if (scale < HRN_SCALE_B)
    {
      progress = (scale - HRN_SCALE_A) / (HRN_SCALE_B - HRN_SCALE_A);
      return LIN (compute_static_x (num, accum_pos, 4),
                  compute_static_x (num, accum_pos, 3)) + HRN_X_OFFSET / scale;
    }
  else if (scale < HRN_SCALE_C)
    {
      progress = (scale - HRN_SCALE_B) / (HRN_SCALE_C - HRN_SCALE_B);
      return LIN (compute_static_x (num, accum_pos, 3),
                  compute_static_x (num, accum_pos, 2)) + HRN_X_OFFSET / scale;
    }
  else if (scale < HRN_SCALE_D)
    {
      progress = (scale - HRN_SCALE_C) / (HRN_SCALE_D - HRN_SCALE_C);
      return LIN (compute_static_x (num, accum_pos, 2),
                  compute_static_x (num, accum_pos, 1)) + HRN_X_OFFSET / scale;
    }
  else if (scale < HRN_SCALE_E)
    {
      progress = (scale - HRN_SCALE_D) / (HRN_SCALE_E - HRN_SCALE_D);
      return LIN (compute_static_x (num, accum_pos, 1),
                  compute_static_x (num, accum_pos, 0)) + HRN_X_OFFSET / scale;
    }
  else if (scale < HRN_SCALE_F)
    {
      progress = (scale - HRN_SCALE_E) / (HRN_SCALE_F - HRN_SCALE_E);
      return LIN (compute_static_x (num, accum_pos, 0),
                  compute_static_x (num, accum_pos, -1)) + HRN_X_OFFSET / scale;
    }
  else if (scale < HRN_SCALE_G)
    {
      progress = (scale - HRN_SCALE_F) / (HRN_SCALE_G - HRN_SCALE_F);
      return LIN (compute_static_x (num, accum_pos, -1),
                  compute_static_x (num, accum_pos, -1)) + HRN_X_OFFSET / scale;
    }
  else
    {
      return compute_static_x (num, accum_pos, -1) + HRN_X_OFFSET / scale;
    }
  return 0;
}

gfloat
hrn_view_compute_y (gint num, gint accum_pos, gdouble scale)
{
  gdouble progress = 0.5;

  if (scale < HRN_SCALE_A)
    {
      return compute_static_y (num, accum_pos, 4);
    }
  else if (scale < HRN_SCALE_B)
    {
      progress = (scale - HRN_SCALE_A) / (HRN_SCALE_B - HRN_SCALE_A);
      return LIN (compute_static_y (num, accum_pos, 4),
                  compute_static_y (num, accum_pos, 3));
    }
  else if (scale < HRN_SCALE_C)
    {
      progress = (scale - HRN_SCALE_B) / (HRN_SCALE_C - HRN_SCALE_B);
      return LIN (compute_static_y (num, accum_pos, 3),
                  compute_static_y (num, accum_pos, 2));
    }
  else if (scale < HRN_SCALE_D)
    {
      progress = (scale - HRN_SCALE_C) / (HRN_SCALE_D - HRN_SCALE_C);
      return LIN (compute_static_y (num, accum_pos, 2),
                  compute_static_y (num, accum_pos, 1));
    }
  else if (scale < HRN_SCALE_E)
    {
      progress = (scale - HRN_SCALE_D) / (HRN_SCALE_E - HRN_SCALE_D);
      return LIN (compute_static_y (num, accum_pos, 1),
                  compute_static_y (num, accum_pos, 0));
    }
  else if (scale < HRN_SCALE_F)
    {
      progress = (scale - HRN_SCALE_E) / (HRN_SCALE_F - HRN_SCALE_E);
      return LIN (compute_static_y (num, accum_pos, 0),
                  compute_static_y (num, accum_pos, -1));
    }
  else if (scale < HRN_SCALE_G)
    {
      progress = (scale - HRN_SCALE_F) / (HRN_SCALE_G - HRN_SCALE_F);
      return LIN (compute_static_y (num, accum_pos, -1),
                  compute_static_y (num, accum_pos, -1));
    }
  else
    {
      return compute_static_y (num, accum_pos, -1);
    }

#undef A
#undef B
#undef C
#undef D
#undef E
#undef F
#undef G
  return 0;
}

/* this calls reposition and indicates that we desire
 * centering
 */
gint
hrn_view_center (ClutterActor *self, gint item_no)
{
  HrnView        *layout = (HrnView *) self;
  HrnViewPrivate *priv   = layout->priv;

  priv->center_item = item_no;
  priv->compute_pos = TRUE;
  hrn_view_queue_reposition (layout);
  return 0;
}

static void
hrn_view_allocate (ClutterActor *self, const ClutterActorBox *box,
                   ClutterAllocationFlags flags)
{
  HrnView        *layout = (HrnView *) self;
  HrnViewPrivate *priv   = layout->priv;

  gint            i;

  GList          *iter;

  /* chain up to set actor->allocation */
  CLUTTER_ACTOR_CLASS (hrn_view_parent_class)
  ->allocate (self, box, flags);

  gdouble scale;
  clutter_actor_get_scale (self, &scale, NULL);
  scale              = hrn_actor_get_abs_scale (self);
  priv->alloc_width  = box->x2 - box->x1;
  priv->alloc_height = box->y2 - box->y1;
  priv->flags        = flags;

  clutter_actor_get_preferred_size (self, NULL, NULL, NULL, NULL);
  clutter_actor_allocate_preferred_size (CLUTTER_ACTOR (priv->group),
                                         flags);

  if (priv->lasso)
    clutter_actor_allocate_preferred_size (CLUTTER_ACTOR (priv->lasso),
                                           flags);


  priv->a_wrap = priv->alloc_width;

  for (i = 0, iter = priv->list; iter; i++, iter = iter->next)
    {
      ClutterActor *child = iter->data;
      clutter_actor_allocate_preferred_size (CLUTTER_ACTOR (child), flags);
    }
}

static void
hrn_view_show_labels (HrnView *view, gint state)
{
  HrnViewPrivate *priv = HRN_VIEW_GET_PRIVATE (view);

  if (priv->labels_visible != state)
    {
      GList *iter;
      priv->labels_visible = state;
      for (iter = priv->list; iter; iter = iter->next)
        {
          hrn_cluster_show_labels (iter->data, state);
        }
    }
}

/* return the currently focused item
 * according to scrolling
 */
gint
hrn_view_get_focused (HrnView *self)
{
  HrnViewPrivate   *priv = self->priv;
  HrnViewActorData *prev = NULL;

  gint              i;

  GList            *iter;

  gdouble           scale;

  scale = hrn_actor_get_abs_scale (CLUTTER_ACTOR (self));

  for (i = 0, iter = priv->list; iter; i++, iter = iter->next)
    {
      ClutterActor     *child = iter->data;
      HrnViewActorData *data;

      data = g_hash_table_lookup (priv->hash_table, child);

      {
        gfloat y = hrn_view_compute_y (data->num, data->accum_pos, scale) -
                   priv->target_center_pos;


        if (y > 0)
          {
            if (scale <= 1.0)
              {
                return data->accum_pos;
              }

            if (prev)
              {
                gint i;
                for (i = prev->accum_pos; i < data->accum_pos; i++)
                  {
                    gfloat y2 =
                      hrn_view_compute_y (prev->num, i,
                                          scale) - priv->target_center_pos;
                    if (y2 > 0)
                      return i;
                  }
              }
            return data->accum_pos;
          }
      }
      prev = data;
    }
  return 0;
}



void
hrn_view_reposition (HrnView *self)
{
  HrnViewPrivate *priv = self->priv;

  gint            i;

  gint            accum_pos = 0;

  GList          *iter;

  gdouble         scale;

  if (dirty)
    {
      g_source_remove (dirty);
      dirty = 0;
    }

  clutter_actor_get_scale (CLUTTER_ACTOR (self), &scale, NULL);
  scale = hrn_actor_get_abs_scale (CLUTTER_ACTOR (self));

  if (fabs (scale - 0.5) < 0.0001)
    {
      hrn_view_show_labels (self, ZOOM_CLUSTERS);
    }
  else if (fabs (scale - 1.0) < 0.0001)
    {
      hrn_view_show_labels (self, ZOOM_CLUSTERS_GROUPED);
    }
  else if (fabs (scale - 2.0) < 0.0001)
    {
      hrn_view_show_labels (self, ZOOM_ITEMS_8);
    }
  else if (fabs (scale - 4.0) < 0.0001)
    {
      hrn_view_show_labels (self, ZOOM_ITEMS_4);
    }
  else if (fabs (scale - 8.0) < 0.0001)
    {
      hrn_view_show_labels (self, ZOOM_ITEMS_3);
    }
  else if (fabs (scale - 16.0) < 0.0001)
    {
    }
  else
    {
      hrn_view_show_labels (self, ZOOM_NONE);
    }

  /* Make sure we have calculated the preferred size */
  /* what does this do? */
  clutter_actor_get_preferred_size (CLUTTER_ACTOR (
                                      self), NULL, NULL, NULL, NULL);

  {
    gboolean foundit = FALSE;

    for (i = 0, iter = priv->list; iter; i++, iter = iter->next)
      {
        ClutterActor     *child = iter->data;
        HrnViewActorData *data;

        data      = g_hash_table_lookup (priv->hash_table, child);
        data->num = i;

        data->accum_pos = accum_pos;
        accum_pos      += data->cells;
        g_object_set_data (G_OBJECT (child), "accumpos",
                           GINT_TO_POINTER (data->accum_pos));


        if (accum_pos >= priv->center_item && !foundit)
          {
            if (priv->compute_pos)
              {
                gint y;
                y =
                  hrn_view_compute_y (data->num, priv->center_item,
                                      scale);
                priv->target_center_pos = y;
                priv->center_pos        = y;
                priv->compute_pos       = FALSE;
              }

            foundit = TRUE;
          }

        priv->height = hrn_view_compute_y (data->num, accum_pos, scale);
      }
  }

  for (i = 0, iter = priv->list; iter; i++, iter = iter->next)
    {
      ClutterActor     *child = iter->data;
      ClutterActorBox   child_box;
      gfloat            natural_a;
      gfloat            natural_b;
      HrnViewActorData *data;



      data = g_hash_table_lookup (priv->hash_table, child);

#if 1
      /* each child will get as much space as they require, the
       * clusters adapt themselves to the surrounding grid
       */
      clutter_actor_get_preferred_size (CLUTTER_ACTOR (child),
                                        NULL, NULL,
                                        &natural_a, &natural_b);
#endif


      {
        gfloat y =
          hrn_view_compute_y (data->num, data->accum_pos,
                              scale) - priv->center_pos + Y_OFFSET / scale;

#if 0
#define BUG_THRESHOLD    30000

        if (y < -BUG_THRESHOLD)
          y = -BUG_THRESHOLD;
        else if (y > BUG_THRESHOLD)
          y = BUG_THRESHOLD;
#endif
        child_box.y1 = y;
      }
      child_box.x1 = hrn_view_compute_x (data->num, data->accum_pos, scale);
      clutter_actor_set_position (child, child_box.x1, child_box.y1);

      /* clamp the vertical position to avoid overflow (both on
       * stage as well as in clutter functions
       */

      child_box.x2 = child_box.x1 + natural_a;
      child_box.y2 = child_box.y1 + natural_b;

      hrn_switcher_adapt (HRN_SWITCHER (child));
      clutter_actor_queue_redraw (child);
    }
  /*clutter_actor_set_y (self, -priv->center_pos);*/
  hrn_loc_slider_update ();
  hrn_store_state ();
}



/****** hardcoded logic that should be incorporated into the cluster view
 * container **********/

gint
hrn_view_find_uri (const gchar *uri)
{
  GList *children;
  GList *iter;
  gint res = -1;
  gint i   = 0;

  children = clutter_container_get_children (CLUTTER_CONTAINER (hrn_view));

  iter = children;
  while (iter && res == -1)
    {
      GList *items = HRN_CLUSTER (iter->data)->results;
      GList *iter2;

      iter2 = items;
      while (iter2 && res == -1)
        {
          BklItem *item = iter2->data;

          if (!strcmp (bkl_item_get_uri (item), uri))
            res = i;
          i++;
          iter2 = iter2->next;
        }
      iter = iter->next;
    }

  g_list_free (children);
  return res;
}


BklItem *
hrn_view_item_to_bkl_item (gint no)
{
  GList *children;
  GList *iter;
  BklItem *res = NULL;
  gint i       = 0;

  children = clutter_container_get_children (CLUTTER_CONTAINER (hrn_view));

  iter = children;
  while (iter && !res)
    {
      GList *uris = HRN_CLUSTER (iter->data)->results;
      GList *iter2;

      iter2 = uris;
      while (iter2 && !res)
        {
          if (no <= i)
            {
              res = iter2->data;
            }
          i++;
          iter2 = iter2->next;
        }
      iter = iter->next;
    }

  g_list_free (children);
  return res;
}

gint hrn_total_items;

#define safe_strcmp(a, b)    (a == NULL ? 1 : \
                              b == NULL ? -1 : \
                              strcmp (a, b))



static gint
hrn_sort_title (gconstpointer a, gconstpointer b)
{
  BklItem **itemA = (void*) a;
  BklItem **itemB = (void*) b;
  BklItemType typeA;
  BklItemType typeB;
  const gchar *titleA;
  const gchar *titleB;

  typeA = bkl_item_get_item_type (*itemA);
  typeB = bkl_item_get_item_type (*itemB);

  switch (typeA)
    {
      case BKL_ITEM_TYPE_IMAGE:
        titleA = bkl_item_image_get_title (BKL_ITEM_IMAGE (*itemA));
        break;

      case BKL_ITEM_TYPE_AUDIO:
        titleA = bkl_item_audio_get_title (BKL_ITEM_AUDIO (*itemA));
        break;

      case BKL_ITEM_TYPE_VIDEO:
        titleA = bkl_item_video_get_title (BKL_ITEM_VIDEO (*itemA));
        break;

      default:
        titleA = NULL;
        break;
    }
  if (!titleA)
    titleA = strrchr (bkl_item_get_uri (*itemA), '/') + 1;

  switch (typeB)
    {
      case BKL_ITEM_TYPE_IMAGE:
        titleB = bkl_item_image_get_title (BKL_ITEM_IMAGE (*itemB));
        break;

      case BKL_ITEM_TYPE_AUDIO:
        titleB = bkl_item_audio_get_title (BKL_ITEM_AUDIO (*itemB));
        break;

      case BKL_ITEM_TYPE_VIDEO:
        titleB = bkl_item_video_get_title (BKL_ITEM_VIDEO (*itemB));
        break;

      default:
        titleB = NULL;
        break;
    }
  if (!titleB)
    titleB = strrchr (bkl_item_get_uri (*itemB), '/') + 1;
  return safe_strcmp (titleA, titleB);
}


static gint
hrn_sort_creator (gconstpointer a, gconstpointer b)
{
  BklItem **itemA = (void*) a;
  BklItem **itemB = (void*) b;
  BklItemType typeA;
  BklItemType typeB;
  const gchar *titleA;
  const gchar *titleB;

  typeA = bkl_item_get_item_type (*itemA);
  typeB = bkl_item_get_item_type (*itemB);

  switch (typeA)
    {
      case BKL_ITEM_TYPE_IMAGE:
        titleA = bkl_item_image_get_author (BKL_ITEM_IMAGE (*itemA));
        break;

      case BKL_ITEM_TYPE_AUDIO:
        titleA = bkl_item_audio_get_composer (BKL_ITEM_AUDIO (*itemA));
        break;

      case BKL_ITEM_TYPE_VIDEO:
        titleA = bkl_item_video_get_director (BKL_ITEM_VIDEO (*itemA));
        break;

      default:
        titleA = NULL;
        break;
    }
  if (!titleA)
    titleA = strrchr (bkl_item_get_uri (*itemA), '/') + 1;

  switch (typeB)
    {
      case BKL_ITEM_TYPE_IMAGE:
        titleB = bkl_item_image_get_author (BKL_ITEM_IMAGE (*itemB));
        break;

      case BKL_ITEM_TYPE_AUDIO:
        titleB = bkl_item_audio_get_composer (BKL_ITEM_AUDIO (*itemB));
        break;

      case BKL_ITEM_TYPE_VIDEO:
        titleB = bkl_item_video_get_director (BKL_ITEM_VIDEO (*itemB));
        break;

      default:
        titleB = NULL;
        break;
    }
  if (!titleB)
    titleB = strrchr (bkl_item_get_uri (*itemB), '/') + 1;
  return safe_strcmp (titleA, titleB);
}


static gint
hrn_sort_modification_time (gconstpointer a, gconstpointer b)
{
  BklItem **itemA = (void*) a;
  BklItem **itemB = (void*) b;

  return bkl_item_get_modification_time (*itemA) -
         bkl_item_get_modification_time (*itemB);
}


static gint
hrn_sort_default (gconstpointer a, gconstpointer b)
{
  BklItem **itemA = (void*) a;
  BklItem **itemB = (void*) b;
  BklItemType typeA;
  BklItemType typeB;

  typeA = bkl_item_get_item_type (*itemA);
  typeB = bkl_item_get_item_type (*itemB);

  /* toplevel sort into item type */
  if (typeA != typeB)
    {
      gint ascore;
      gint bscore;
      switch (typeA)
        {
          case BKL_ITEM_TYPE_IMAGE: ascore = 0; break;

          case BKL_ITEM_TYPE_VIDEO: ascore = 1; break;

          case BKL_ITEM_TYPE_AUDIO: ascore = 2; break;

          default: ascore = 0; break;
        }
      switch (typeB)
        {
          case BKL_ITEM_TYPE_IMAGE: bscore = 0; break;

          case BKL_ITEM_TYPE_VIDEO: bscore = 1; break;

          case BKL_ITEM_TYPE_AUDIO: bscore = 2; break;

          default: bscore = 0; break;
        }
      return ascore - bscore;
    }

  switch (typeA)
    {
      case BKL_ITEM_TYPE_IMAGE:
      {
        gint res;


        /* sort by directory */

        gchar *metaA = hrn_item_uri_parent_to_meta (*itemA);
        gchar *metaB = hrn_item_uri_parent_to_meta (*itemB);

        res = safe_strcmp (metaA, metaB);
        if (metaA)
          g_free (metaA);
        if (metaB)
          g_free (metaB);
        if (res != 0)
          return res;

        /* then time */

        res = safe_strcmp (bkl_item_image_get_time (BKL_ITEM_IMAGE (*itemA)),
                           bkl_item_image_get_time (BKL_ITEM_IMAGE (*itemB)));
        if (res != 0)
          return res;
      }

      break;

      case BKL_ITEM_TYPE_AUDIO:

        /* sort by artist */
      {
        gchar *artist1 = NULL;
        gchar *artist2 = NULL;
        gint res;

        GPtrArray *artists;
        artists = bkl_item_audio_get_artists (BKL_ITEM_AUDIO (*itemA));
        if (artists)
          artist1 = g_strdup (artists->pdata[0]);
        else
          artist1 = hrn_item_uri_parent_to_meta (BKL_ITEM (*itemA));

        artists = bkl_item_audio_get_artists (BKL_ITEM_AUDIO (*itemB));
        if (artists)
          artist2 = g_strdup (artists->pdata[0]);
        else
          artist2 = hrn_item_uri_parent_to_meta (BKL_ITEM (*itemB));

        res = safe_strcmp (artist1, artist2);
        g_free (artist1);
        g_free (artist2);

        if (res != 0)
          return res;
      }
        /* sort by album */
        {
          gint res;
          res = safe_strcmp (bkl_item_audio_get_album (BKL_ITEM_AUDIO (*itemA)),
                             bkl_item_audio_get_album (BKL_ITEM_AUDIO (*itemB)));
          if (res != 0)
            return res;
        }
        /* XXX: sort by track */
        /* sort by title */
        return safe_strcmp (bkl_item_audio_get_title (BKL_ITEM_AUDIO (*itemA)),
                            bkl_item_audio_get_title (BKL_ITEM_AUDIO (*itemB)));
        break;

      case BKL_ITEM_TYPE_VIDEO:

        /* sort by metadata (hopefully series or similar, falling back to dir */
      {
        gint res;
        gchar *metaA = hrn_item_get_meta (*itemA);
        gchar *metaB = hrn_item_get_meta (*itemB);

        res = safe_strcmp (metaA, metaB);
        if (metaA)
          g_free (metaA);
        if (metaB)
          g_free (metaB);
        if (res != 0)
          return res;
      }

        /* sort by title */
        return safe_strcmp (bkl_item_video_get_title (BKL_ITEM_VIDEO (*itemA)),
                            bkl_item_video_get_title (BKL_ITEM_VIDEO (*itemB)));
        break;

      case BKL_ITEM_TYPE_BROKEN:
      case BKL_ITEM_TYPE_PLAYLIST:
        return 0;

      case BKL_ITEM_TYPE_NONE:
        break;
    }

  return bkl_item_get_item_type (*itemA) - bkl_item_get_item_type (*itemB);
}

static gint (*sort_func)(gconstpointer a, gconstpointer b) = hrn_sort_default;

const gchar *
hrn_view_get_sort (HrnView *view)
{
  HrnViewPrivate *priv = HRN_VIEW_GET_PRIVATE (view);

  if (priv->sort)
    return priv->sort;
  return "default";
}

void
hrn_view_set_sort (HrnView *view, const gchar *sortfunc)
{
  HrnViewPrivate *priv = HRN_VIEW_GET_PRIVATE (view);

  if (priv->sort)
    {
      g_free (priv->sort);
      priv->sort = NULL;
    }
  if (sortfunc)
    priv->sort = g_strdup (sortfunc);
  if (!sortfunc ||
      g_str_equal (sort_func, "default"))
    {
      sort_func = hrn_sort_default;
    }
  else if (g_str_equal (sortfunc, "title"))
    sort_func = hrn_sort_title;
  else if (g_str_equal (sortfunc, "modification time"))
    sort_func = hrn_sort_modification_time;
  else if (g_str_equal (sortfunc, "creator"))
    sort_func = hrn_sort_creator;
  else
    {
      sort_func = hrn_sort_default;
      g_warning ("unknown sort func\n");
    }

  g_ptr_array_sort (priv->results, sort_func);
  hrn_view_refilter (HRN_VIEW (hrn_view));
  hrn_view_reposition (HRN_VIEW (hrn_view));
}

#if 0
static gboolean
search_match (BklItem *item, const gchar *search_string)
{
  const gchar *uri = bkl_item_get_uri (item);

  if (strstr (uri, search_string))
    return TRUE;
  return FALSE;
}
#endif

HrnSource *
hrn_view_get_source (HrnView *view)
{
  HrnViewPrivate *priv = HRN_VIEW_GET_PRIVATE (view);

  return priv->current_source;
}


void
hrn_view_set_source (HrnView *view, HrnSource   *source)
{
  HrnViewPrivate *priv = HRN_VIEW_GET_PRIVATE (view);

  if (source == NULL)
    {
      return;
    }

  if (priv->current_source)
    {
      if (priv->current_source == source)
        {
          /* return; */
        }

      /* Disconnect the old signals */
      g_signal_handler_disconnect (priv->current_source, priv->item_added_id);
      g_signal_handler_disconnect (priv->current_source, priv->item_removed_id);
      g_signal_handler_disconnect (priv->current_source, priv->item_changed_id);

      g_object_unref (priv->current_source);
      priv->current_source = NULL;
    }

  priv->current_source = g_object_ref (source);
  priv->item_added_id  = g_signal_connect (source, "item-added",
                                           G_CALLBACK (source_item_added), view);
  priv->item_removed_id = g_signal_connect (source, "item-removed",
                                            G_CALLBACK (source_item_removed),
                                            view);
  priv->item_changed_id = g_signal_connect (source, "item-changed",
                                            G_CALLBACK (source_item_changed),
                                            view);




  if (hrn_toolbar)
    {
      hrn_toolbar_set_pinned ((HrnToolbar *) hrn_toolbar, FALSE);
    }
}

static void
hrn_view_refilter (HrnView *view)
{
  HrnViewPrivate *priv = view->priv;
  GList *children, *iter;

  children = clutter_container_get_children (CLUTTER_CONTAINER (view));
  for (iter = children; iter; iter = iter->next)
    {
      clutter_actor_destroy (iter->data);
    }
  g_list_free (children);


  if (priv->current_source == NULL)
    {
      /* FIXME: There must be a better way to do this, what is this?
       * through refactoring this entire call should be possible
       * to remove
       */
      hrn_view_set_source (view, hrn_get_local_source ());
    }
  else
    {
      /* free old results */
      if (priv->results)
        {
          g_ptr_array_foreach (priv->results, (GFunc)(g_object_unref), NULL);
          g_ptr_array_free (priv->results, TRUE);
        }
      /* re-retrieve the results and sort them */
      priv->results = hrn_source_query (priv->current_source, view->search);
      g_ptr_array_sort (priv->results, sort_func);
    }

  if (priv->results == NULL)
    {
      return;
    }

  /* cluster the items */
  {
    gint categories = 0;
    gint catitems   = 0;
    gchar *prevcat  = NULL;
    gint i;
    GList *accumulate = NULL;

    hrn_total_items = 0;

    gboolean category_full = FALSE;

    for (i = 0; i < priv->results->len; i++)
      {
        BklItem *item = priv->results->pdata[i];

        if (((bkl_item_get_item_type (item) & view->bkl_item_type_mask)
             /*&& (view->search == NULL ||
                 view->search[0]=='\0' ||
                 search_match (item, view->search))*/)
            )
          {
            gchar *category;

            if (bkl_item_get_item_type (item) == BKL_ITEM_TYPE_IMAGE)
              category = hrn_item_uri_parent_to_meta (item);
            else
              category = hrn_item_get_meta (item);

            if (prevcat && strcmp (prevcat, category))
              {
                g_free (prevcat);
                prevcat       = category;
                category_full = TRUE;
              }
            else
              {
                if (!prevcat)
                  prevcat = category;
              }

            if (category_full)
              {
                if (accumulate != NULL)
                  {
                    ClutterActor *cluster;

                    /*if (g_list_length (accumulate)>2)
                       {*/
                    cluster = hrn_make_cluster (accumulate);
                    clutter_container_add_actor (CLUTTER_CONTAINER (
                                                   view), cluster);
                    /*}
                       else
                       categories --;*/
                    accumulate = NULL;
                    catitems   = 0;
                  }
                category_full = FALSE;
                categories++;
                if (categories > HRN_MAX_CLUSTERS)
                  goto done;
              }

            hrn_total_items++;
            catitems++;
            accumulate = g_list_append (accumulate, item);
          }
      }

    if (accumulate) /* duplicated code */
      {
        ClutterActor *cluster;
        cluster = hrn_make_cluster (accumulate);
        clutter_container_add_actor (CLUTTER_CONTAINER (view), cluster);
      }
done:
    if (prevcat)
      g_free (prevcat);
  }
}

guint
hrn_view_get_filter (HrnView *view)
{
  return view->bkl_item_type_mask;
}

void
hrn_view_set_filter (HrnView *view, guint bkl_item_type_mask)
{
  view->bkl_item_type_mask = bkl_item_type_mask;
  hrn_view_center (CLUTTER_ACTOR (view), 0);
  hrn_view_refilter (view);
  hrn_view_reposition (view);
}


const gchar *
hrn_view_get_search (HrnView *view)
{
  return view->search;
}

void
hrn_view_set_search (HrnView *view, const gchar *search)
{
  if (view->search)
    g_free (view->search);
  view->search = NULL;
  if (search)
    view->search = g_strdup (search);

  hrn_view_refilter (view);

  hrn_view_center (CLUTTER_ACTOR (view), 0);
  hrn_view_reposition (view);

  if (hrn_toolbar)
    {
      hrn_toolbar_set_pinned ((HrnToolbar *) hrn_toolbar, FALSE);
    }
}

static gboolean
idle_preloader (gpointer data)
{
  GList *children = clutter_container_get_children (data);
  GList *iter;
  gint i;

  for (iter = children, i = 0; iter && i < 8; iter = iter->next, i++)
    {
      hrn_cluster_preload (iter->data);
    }
  return FALSE;
}

ClutterActor *
hrn_view_build ()
{
  ClutterActor     *view;

  view = hrn_view_new ();

  if (0) g_timeout_add (500, idle_preloader, view);
  return view;
}

static gdouble scale           = INITIAL_SCALE;
static gdouble hrn_target_zoom = INITIAL_SCALE;
static gint hrn_target_loc     = 10;

void hrn_sidebar_unpin ();
void hrn_sidebar_pin ();


gdouble
hrn_get_scale (void)
{
  return scale;
}

gdouble
hrn_get_zoom (void)
{
  return hrn_target_zoom;
}

extern gint desired_pos;

void
hrn_set_theatre (void)
{
  hrn_set_zoom (8.0);
}

void
hrn_set_zoom (gdouble zoom)
{
  HrnViewPrivate *priv = HRN_VIEW (hrn_view)->priv;

  /* (hrn_target_zoom contains the old value in these checks) */

  if (priv->zoom_lock)
    {
      return;
    }

  if (zoom != hrn_target_zoom || zoom != scale)
    {
      hrn_view_go ();
    }

  priv->zoom_lock = TRUE;

  if (hrn_target_zoom < 1.0 && zoom >= 1.0)
  /* going from leftmost to a mid state */
    {
      hrn_sidebar_unpin ();
    }
  else if (hrn_target_zoom >= 1.0 && zoom < 1.0)
  /* going to the left most state */
    {
      /* hrn_sidebar_wake (); */
    }


  if (zoom > 15)
    {
      hrn_controls_can_has_fade = TRUE;
    }
  else
    {
      hrn_controls_can_has_fade = FALSE;
    }

#define THEATRE_THRESHOLD 7.0

  if (hrn_target_zoom < THEATRE_THRESHOLD && zoom > THEATRE_THRESHOLD)
  /* going to theatre mode */
    {
      hrn_theatre_show ((HrnTheatre *) hrn_theatre);
      hrn_toolbar_set_mode ((HrnToolbar *) hrn_toolbar,
                            HRN_TOOLBAR_MODE_THEATRE);
    }
  else if (hrn_target_zoom >= THEATRE_THRESHOLD && zoom < THEATRE_THRESHOLD)
  /* going from theatre mode */
    {
      hrn_theatre_hide ((HrnTheatre *) hrn_theatre);
      hrn_toolbar_set_mode ((HrnToolbar *) hrn_toolbar,
                            HRN_TOOLBAR_MODE_VIEW);
    }

  hrn_target_zoom = zoom;

  if (hrn_in_preload == FALSE)
    {
      if (ABS (zoom - 0.5) < 0.01) nbtk_adjustment_set_value (zoom_adjustment, 0.0);
      else if (ABS (zoom - 1.0) < 0.01) nbtk_adjustment_set_value (zoom_adjustment,
                                                                   0.25);
      else if (ABS (zoom - 2.0) < 0.01) nbtk_adjustment_set_value (zoom_adjustment,
                                                                   0.5);
      else if (ABS (zoom - 4.0) < 0.01) nbtk_adjustment_set_value (zoom_adjustment,
                                                                   0.75);
      else
        {
          nbtk_adjustment_set_value (zoom_adjustment, 1.0);
        }
    }

  if (desired_pos != 0)
    hrn_view_set_pos (desired_pos);
  desired_pos = 0; /*cancels out the one set by set_pos */

  priv->zoom_lock = FALSE;
}


void
hrn_set_zoom_hard (gdouble zoom)
{
  hrn_set_zoom (zoom);
  scale = zoom;
  hrn_view_reposition (HRN_VIEW (hrn_view));
}

gdouble
hrn_view_get_loc (void)
{
  HrnViewPrivate *priv;

  if (!hrn_view)
    return 0;
  priv = HRN_VIEW_GET_PRIVATE (hrn_view);
  if (priv->height == 0)
    return 0;
  return priv->target_center_pos / (priv->height * 1.0);
}

void
hrn_view_set_loc (gdouble loc)
{
  HrnViewPrivate *priv = HRN_VIEW_GET_PRIVATE (hrn_view);

  hrn_view_go ();

  priv->target_center_pos = priv->height * loc;

  if (clutter_get_pointer_grab ())
    priv->center_pos = priv->target_center_pos;
  hrn_view_reposition (HRN_VIEW (hrn_view));
}


gint
hrn_view_center_pos (void)
{
  HrnViewPrivate *priv = HRN_VIEW_GET_PRIVATE (hrn_view);

  return priv->target_center_pos;
}


void
hrn_view_set_pos (gint pos)
{
  hrn_view_go ();
  hrn_target_loc = pos;
  hrn_view_center (hrn_view, pos);
  hrn_view_reposition (HRN_VIEW (hrn_view));
  hrn_loc_slider_update ();
  desired_pos = pos;
}

gint
hrn_view_get_count (void)
{
  return hrn_total_items;
}

gint
hrn_view_get_pos (void)
{
  return hrn_target_loc;
}

const gchar *
hrn_view_get_uri (void)
{
  return "NYI";
}

void
hrn_view_scroll (gdouble stage_percentage)
{
  HrnViewPrivate *priv = HRN_VIEW_GET_PRIVATE (hrn_view);

  priv->target_center_pos = priv->target_center_pos +
                            clutter_actor_get_height (clutter_stage_get_default ())
                            * stage_percentage / scale;

  if (priv->target_center_pos < 0)
    priv->target_center_pos = 0;
  /* XXX: also handle going off on the other side */

  hrn_view_reposition (HRN_VIEW (hrn_view));
  clutter_actor_queue_redraw (hrn_view);
  desired_pos = hrn_view_get_focused (HRN_VIEW (hrn_view));  /* update which
                                                                item
                                                              * is considered
                                                              *focused
                                                              */
}


static gboolean paused = FALSE;

static gboolean
transition_iterate (ClutterTimeline *timeline, gint frame_no, gpointer actor)
{
  static gdouble prev_scale = 123;
  static gdouble prev_pos   = 123;
  gboolean did_something    = FALSE;
  HrnViewPrivate *priv      = HRN_VIEW_GET_PRIVATE (actor);
  guint delta;
  gint i;

  delta = clutter_timeline_get_delta (timeline);


  if (paused)
    return TRUE;

  for (i = 0; i < delta; i++)
    {
      if (fabs (hrn_target_zoom - scale) > 0.01)
        {
          if (hrn_target_zoom > scale)
            {
              if (scale * HRN_ZOOM_RATE < hrn_target_zoom)
                scale *= HRN_ZOOM_RATE;
              else
                scale = hrn_target_zoom;
            }
          else
            {
              if (scale / HRN_ZOOM_RATE > hrn_target_zoom)
                scale /= HRN_ZOOM_RATE;
              else
                scale = hrn_target_zoom;
            }
          did_something = TRUE;
        }
      else if (hrn_target_zoom != scale)
        {
          scale         = hrn_target_zoom;
          did_something = TRUE;
        }
      else
        break;
    }

  for (i = 0; i < delta; i++)
    {
      if (fabs (priv->target_center_pos - priv->center_pos) > 1)
        {
          #define RATE    0.012

          priv->center_pos = priv->target_center_pos * (RATE) +
                             priv->center_pos * (1.0 - RATE);
          did_something = TRUE;
        }
      else if (priv->target_center_pos != priv->center_pos)
        priv->center_pos = priv->target_center_pos;
      else
        break;
    }

  if (prev_scale != scale ||
      prev_pos != priv->center_pos)
    {
      clutter_actor_set_scale (actor, scale, scale);

      if (prev_scale != scale)
        priv->compute_pos = TRUE;
      hrn_view_reposition (actor);
      prev_scale = scale;
      prev_pos   = priv->center_pos;

      did_something = TRUE;
    }

  if (!did_something)
    hrn_view_pause ();
  return TRUE;
}

static ClutterTimeline *timeline;
static void
hrn_view_go (void)
{
  if (timeline == NULL)
    hrn_view_start ();
  clutter_timeline_start (timeline);
  paused = FALSE;
}

static void
hrn_view_pause (void)
{
  paused = TRUE;
  clutter_timeline_pause (timeline);
}


void
hrn_view_start (void)
{
  if (timeline)
    return;
  timeline = clutter_timeline_new (3000);
  clutter_timeline_set_loop (timeline, TRUE);
  g_signal_connect (timeline, "new-frame", G_CALLBACK (
                      transition_iterate), hrn_view);
}

static guint changed_items_handler = 0;
static gboolean
changed_items (gpointer data)
{
  hrn_view_refilter (HRN_VIEW (hrn_view));
  hrn_view_reposition (HRN_VIEW (hrn_view));
  changed_items_handler = 0;
  return FALSE;
}

gboolean
removed_items (gpointer data)
{
  hrn_view_refilter (HRN_VIEW (hrn_view));
  hrn_view_reposition (HRN_VIEW (hrn_view));
  changed_items_handler = 0;
  return FALSE;
}

void
source_item_added (HrnSource *source, BklItem   *item, gpointer userdata)
{
  HrnView *view        = HRN_VIEW (userdata);
  HrnViewPrivate *priv = HRN_VIEW_GET_PRIVATE (view);

  if (source == priv->current_source)
    {
      g_ptr_array_add (priv->results, g_object_ref (item));

      if (!changed_items_handler)
        changed_items_handler = g_timeout_add (4000, changed_items, NULL);
    }
}

void
source_item_removed (HrnSource *source, BklItem   *item, gpointer userdata)
{
  HrnView *view        = HRN_VIEW (userdata);
  HrnViewPrivate *priv = HRN_VIEW_GET_PRIVATE (view);

  if (source == priv->current_source)
    {
      g_ptr_array_remove (priv->results, item);
      g_object_unref (item);

      removed_items (NULL);
    }
}

void
source_item_changed (HrnSource *source, BklItem   *item, gpointer userdata)
{
  HrnView *view        = HRN_VIEW (userdata);
  HrnViewPrivate *priv = HRN_VIEW_GET_PRIVATE (view);

  if (source == priv->current_source)
    {
      if (!changed_items_handler)
        changed_items_handler = g_timeout_add (4000, changed_items, NULL);
    }
}

#define MAX_TEXTS    128

gdouble
hrn_get_pinch (void)
{
  gdouble zoom;

  gdouble pinch = 0.0;

  clutter_actor_get_scale (hrn_view, &zoom, NULL);
  if (zoom < 1.0)
    pinch = (1.0 - zoom) * 2 * 20;
  return pinch;
}
