/*
 * Hornsey - Moblin Media Player.
 * Copyright © 2007, 2008, 2009 Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glib.h>
#include <glib-object.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <string.h>

#include "hrn-texture-cache.h"

#include "hrn.h"
/*#include "hrn-private.h"
 */
G_DEFINE_TYPE (HrnTextureCache, hrn_texture_cache, G_TYPE_OBJECT)

#define TEXTURE_CACHE_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), HRN_TYPE_TEXTURE_CACHE, HrnTextureCachePrivate))

typedef struct _HrnTextureCachePrivate HrnTextureCachePrivate;

struct _HrnTextureCachePrivate
{
  GHashTable *cache;
  gint        max_size;
  gint        sum_size;
  gint        unused;
};

typedef struct HrnCacheEntry
{
  HrnTextureCache *cache;       /* back pointer since we use the entry as
                                   a closure as well */
  ClutterActor *original;
  gint          clone_count; /* current count of clones */
} HrnCacheEntry;


enum
{
  PROP_0,
};

enum
{
  LOADED,
  ERROR_LOADING,

  LAST_SIGNAL
};

static HrnTextureCache*__cache_singleton = NULL;

static void
hrn_texture_cache_set_property (GObject *object, guint prop_id,
                                const GValue *value,
                                GParamSpec   *pspec)
{
  switch (prop_id)
    {
      default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}

static void
hrn_texture_cache_get_property (GObject *object, guint prop_id,
                                GValue     *value,
                                GParamSpec *pspec)
{
  switch (prop_id)
    {
      default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}

static void
hrn_texture_cache_finalize (GObject *object)
{
  HrnTextureCachePrivate *priv = TEXTURE_CACHE_PRIVATE (object);

  if (priv->cache)
    {
      g_hash_table_unref (priv->cache);
      priv->cache = NULL;
    }

  G_OBJECT_CLASS (hrn_texture_cache_parent_class)->finalize (object);
}

static void
hrn_texture_cache_class_init (HrnTextureCacheClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  g_type_class_add_private (klass, sizeof (HrnTextureCachePrivate));

  object_class->get_property = hrn_texture_cache_get_property;
  object_class->set_property = hrn_texture_cache_set_property;
  object_class->finalize     = hrn_texture_cache_finalize;
}

static void
hrn_texture_cache_init (HrnTextureCache *self)
{
  HrnTextureCachePrivate *priv = TEXTURE_CACHE_PRIVATE (self);

  priv->cache = g_hash_table_new_full (g_str_hash, g_str_equal, g_free,
                                       g_free);
  priv->max_size = HRN_CACHE_MAX_SIZE;
  priv->sum_size = 0;
}

HrnTextureCache*
hrn_texture_cache_get_default (void)
{
  if (G_UNLIKELY (__cache_singleton == NULL))
    __cache_singleton = g_object_new (HRN_TYPE_TEXTURE_CACHE, NULL);

  return __cache_singleton;
}


gint
hrn_texture_cache_get_size (HrnTextureCache *self)
{
  HrnTextureCachePrivate *priv = TEXTURE_CACHE_PRIVATE (self);

  return priv->sum_size; 
}

static void
texture_loaded (ClutterActor *actor, gpointer data)
{
  /*clutter_actor_animate (actor, CLUTTER_LINEAR, 100, "opacity", 255, NULL);
     clutter_actor_animate (clutter_stage_get_default (), CLUTTER_LINEAR, 200,
     "opacity", 255, NULL);*/
  clutter_actor_queue_redraw (clutter_stage_get_default ());
}

static void
sync_clone (ClutterActor *original, gint width, gint height, gpointer clone)
{
  if (CLUTTER_IS_CLONE (clone))
    {
      clutter_clone_set_source (clone, NULL);
      clutter_clone_set_source (clone, original); /* we're listening
                                                   * for this signal
                                                   * where the clone
                                                   * is used
                                                   */
    }
  else
    {
      g_debug ("expected %p to be a clone is it freed?\n", clone);
    }
}


void
on_clone_finalized (gpointer data, GObject *where_the_object_was)
{
  HrnCacheEntry *entry = (HrnCacheEntry*) data;

  entry->clone_count--;

  if (entry->clone_count < 1)
    {
      HrnTextureCachePrivate *priv;
      priv = TEXTURE_CACHE_PRIVATE (entry->cache);
      priv->unused++;
      g_signal_handlers_disconnect_matched (entry->original,
                                            G_SIGNAL_MATCH_DATA,
                                            0,
                                            0,
                                            0,
                                            sync_clone,
                                            where_the_object_was);
    }
}

void
hrn_texture_cache_evict_items (HrnTextureCache *self, gint count)
{
  HrnTextureCachePrivate *priv = TEXTURE_CACHE_PRIVATE (self);

  while (count > 0 && priv->unused > 0)
    {
      {
        gint           item = g_random_int_range (0, priv->unused);
        GHashTableIter iter;
        gint           i = 0;
        gpointer       key, value;
        HrnCacheEntry *nuke_entry = NULL;
        const gchar   *nuke_path  = NULL;

        g_hash_table_iter_init (&iter, priv->cache);
        while (g_hash_table_iter_next (&iter, &key, &value))
          {
            HrnCacheEntry *entry = value;
            if (entry->clone_count == 0)
              {
                if (i++ > item)
                  {
                    nuke_entry = entry;
                    nuke_path  = key;
                    break;
                  }
              }
          }
        if (nuke_entry)
          {
            /*g_assert (G_OBJECT(nuke_entry->original)->ref_count == 1);  not
               valid with anim */
            clutter_actor_destroy (nuke_entry->original);
            {
              gfloat w, h;
              clutter_actor_get_size (nuke_entry->original, &w, &h);
              priv->sum_size -= w * h * 4;
            }
            priv->unused--;
            g_hash_table_remove (priv->cache, nuke_path);
          }
      }
      count--;
    }
}

/* NOTE: you should unref the returned texture when not needed */
ClutterActor*
hrn_texture_cache_get_texture (HrnTextureCache *self,
                               const gchar     *path_or_uri,
                               gboolean         load_async)
{
  GFile                  *file;
  gchar                  *path;
  HrnTextureCachePrivate *priv;
  ClutterActor           *original;
  ClutterActor           *clone;
  HrnCacheEntry          *entry;

  /* XXX: originally planned to use clones here, but it seems the square
   * texture works well on its own when it is not forced to be square
   */
  return hrn_texture_cache_get_sq_texture (self, path_or_uri, load_async);

  if (!path_or_uri)
    return NULL;
  file = g_file_new_for_commandline_arg (path_or_uri);
  /* XXX: should only accept uris */
  path = g_file_get_path (file);
  g_object_unref (file);

  if (self == NULL)
    self = hrn_texture_cache_get_default ();

  priv = TEXTURE_CACHE_PRIVATE (self);

  entry = g_hash_table_lookup (priv->cache, path);

  if (entry)
    {
      if (entry->clone_count == 0)
        {
          priv->unused--; /* we're going to use it */
        }
    }
  else
    {
      original = clutter_texture_new ();
      clutter_texture_set_filter_quality (CLUTTER_TEXTURE (original),
                                          CLUTTER_TEXTURE_QUALITY_MEDIUM);
#if HRN_LOAD_SIZE_ASYNC
      clutter_texture_set_load_async (CLUTTER_TEXTURE (original), TRUE);
#else
      clutter_texture_set_load_data_async (CLUTTER_TEXTURE (original), TRUE);
#endif
      clutter_texture_set_from_file (CLUTTER_TEXTURE (original), path, NULL);

      /*clutter_actor_set_opacity (original, 0x0);*/
      if (!original)
        {
          g_free (path);
          return NULL;
        }

      /*clutter_group_add (CLUTTER_GROUP (clutter_stage_get_default()),
          original);
         clutter_actor_set_position (original, -1000, -1000);*/

      if (load_async)
        {
          g_signal_connect (original, "load-finished",
                            G_CALLBACK (texture_loaded), NULL);
        }
      entry           = g_malloc0 (sizeof (HrnCacheEntry));
      entry->cache    = self;
      entry->original = original;
      g_object_ref_sink (original);

      g_hash_table_insert (priv->cache, g_strdup (path), entry);

      {
        gfloat w, h;
        clutter_actor_get_size (original, &w, &h);
        priv->sum_size += w * h * 4;
      }
    }

  g_free (path);

  original = entry->original;
  entry->clone_count++;
  clone = clutter_clone_new (original);
  g_object_weak_ref (G_OBJECT (clone), on_clone_finalized, entry);

  g_signal_connect (original, "size-change", G_CALLBACK (sync_clone), clone);


  if (hrn_texture_cache_get_size (self) > priv->max_size)
    hrn_texture_cache_evict_items (self, 1);


  return clone;
}




/* NOTE: you should unref the returned texture when not needed */
ClutterActor*
hrn_texture_cache_get_sq_texture (HrnTextureCache *self,
                                  const gchar     *path_or_uri,
                                  gboolean         load_async)
{
  GFile                  *file;
  gchar                  *path;
  HrnTextureCachePrivate *priv;
  ClutterActor           *original;
  ClutterActor           *clone;
  HrnCacheEntry          *entry;

  if (!path_or_uri)
    return NULL;
  file = g_file_new_for_commandline_arg (path_or_uri);
  /* XXX: should only accept uris */
  path = g_file_get_path (file);
  g_object_unref (file);


  if (self == NULL)
    self = hrn_texture_cache_get_default ();

  priv = TEXTURE_CACHE_PRIVATE (self);

  entry = g_hash_table_lookup (priv->cache, path);

  if (entry)
    {
      if (entry->clone_count == 0)
        {
          priv->unused--; /* we're going to use it */
        }
    }
  else
    {
      original = clutter_texture_new ();
      clutter_texture_set_filter_quality (CLUTTER_TEXTURE (original),
                                          CLUTTER_TEXTURE_QUALITY_MEDIUM);
#if HRN_LOAD_SIZE_ASYNC
      clutter_texture_set_load_async (CLUTTER_TEXTURE (original), TRUE);
#else
      clutter_texture_set_load_data_async (CLUTTER_TEXTURE (original), TRUE);
#endif
      clutter_texture_set_from_file (CLUTTER_TEXTURE (original), path, NULL);

      /*clutter_actor_set_opacity (original, 0x0);*/
      if (!original)
        {
          g_free (path);
          return NULL;
        }

      if (load_async)
        {
          g_signal_connect (original, "load-finished",
                            G_CALLBACK (texture_loaded), NULL);
        }
      entry           = g_malloc0 (sizeof (HrnCacheEntry));
      entry->cache    = self;
      entry->original = original;
      g_object_ref_sink (original);

      g_hash_table_insert (priv->cache, g_strdup (path), entry);

      {
        gfloat w, h;
        clutter_actor_get_size (original, &w, &h);
        priv->sum_size += w * h * 4;
      }
    }

  g_free (path);

  original = entry->original;
  entry->clone_count++;

  clone = hrn_square_clone_new (CLUTTER_TEXTURE (original));

  g_object_weak_ref (G_OBJECT (clone), on_clone_finalized, entry);
  return clone;

  g_signal_connect (original, "size-change", G_CALLBACK (sync_clone), clone);

  return clone;
}


ClutterActor *
hrn_texture_load (const gchar *path)
{
  ClutterActor *ret;

  ret = hrn_texture_cache_get_texture (NULL, path, TRUE);
  if (!ret)
    {
      ret = clutter_rectangle_new ();
      g_debug ("not able to load image %s", path);
    }
  return ret;
}

ClutterActor *
hrn_texture_load_sq (const gchar *path)
{
  ClutterActor *ret;

  ret = hrn_texture_cache_get_sq_texture (NULL, path, TRUE);
  if (!ret)
    {
      ret = clutter_rectangle_new ();
      g_debug ("not able to load image %s", path);
    }
  return ret;
}

