/*
 * Hornsey - Moblin Media Player.
 * Copyright © 2007, 2008, 2009 Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <clutter/clutter.h>
#include <math.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include "hrn.h"
#include "hrn-switcher.h"


/* XXX: this causes unneeded overhead.. */
#define HRN_SWITCHER_MAX_STATES    14

G_DEFINE_TYPE (HrnSwitcher, hrn_switcher, CLUTTER_TYPE_ACTOR);

#define HRN_SWITCHER_GET_PRIVATE(obj)                 \
  (G_TYPE_INSTANCE_GET_PRIVATE ((obj), \
                                HRN_TYPE_SWITCHER, \
                                HrnSwitcherPrivate))

static gboolean
hrn_actor_is_off_stage (HrnSwitcher *switcher, ClutterActor *actor);

gdouble
hrn_actor_get_abs_scale (ClutterActor *actor)
{
  ClutterActor *iter  = actor;
  gdouble       scale = 1.0;

  while (iter)
    {
      gdouble tscale;
      clutter_actor_get_scale (iter, &tscale, NULL);
      scale *= tscale;
      iter   = clutter_actor_get_parent (iter);
    }
  return scale;
}

/* XXX: maybe the StateInfo should be expanded into it's own object,
 *      that registers with the switcher
 */
typedef struct StateInfo
{
  ClutterActor *actor;
  gdouble       min_scale;
  gdouble       max_scale;
  ClutterActor * (*constructor)(HrnSwitcher * switcher,
                                guint stateno,
                                gpointer userdata);
  gpointer constructor_data;

  /* this should probably be a vfunc on the class instead */

  void (*adapt)(HrnSwitcher *switcher, guint state_no, ClutterActor *child,
                gdouble position, gpointer userdata);
  gpointer adapt_data;
  gdouble  currprog;
} StateInfo;

struct _HrnSwitcherPrivate
{
  StateInfo states[HRN_SWITCHER_MAX_STATES];
  gint      state_count;
  gint      state;      /* the current state we're in, gets set to
                         * state 0 when the first state is added
                         */
  /* this is a work around for some state instability that
   * otherwize occurs
   */
  gint off_stage_count;
};

static GObject *
hrn_switcher_constructor (GType type, guint n_params,
                          GObjectConstructParam *params);
static void
hrn_switcher_dispose (GObject *object);
static void
hrn_switcher_paint (ClutterActor *actor);
static void
hrn_switcher_pick (ClutterActor *actor, const ClutterColor   *color);
static void
hrn_switcher_allocate (ClutterActor *self, const ClutterActorBox *box,
                       ClutterAllocationFlags flags);

static void
hrn_switcher_map (ClutterActor *self)
{
  HrnSwitcher        *switcher = HRN_SWITCHER (self);
  HrnSwitcherPrivate *priv     = HRN_SWITCHER_GET_PRIVATE (switcher);
  gint                i;

  CLUTTER_ACTOR_CLASS (hrn_switcher_parent_class)->map (self);

  for (i = 0; i < priv->state_count; i++)
    {
      if (priv->states[i].actor)
        {
          clutter_actor_map (priv->states[i].actor);
        }
    }
}

static void
hrn_switcher_unmap (ClutterActor *self)
{
  HrnSwitcher        *switcher = HRN_SWITCHER (self);
  HrnSwitcherPrivate *priv     = HRN_SWITCHER_GET_PRIVATE (switcher);
  gint                i;

  CLUTTER_ACTOR_CLASS (hrn_switcher_parent_class)->unmap (self);

  for (i = 0; i < priv->state_count; i++)
    {
      if (priv->states[i].actor)
        {
          clutter_actor_unmap (priv->states[i].actor);
        }
    }
}

static void
hrn_switcher_realize (ClutterActor *self)
{
  HrnSwitcher        *switcher = HRN_SWITCHER (self);
  HrnSwitcherPrivate *priv     = HRN_SWITCHER_GET_PRIVATE (switcher);
  gint                i;

  for (i = 0; i < priv->state_count; i++)
    {
      if (priv->states[i].actor)
        {
          clutter_actor_realize (priv->states[i].actor);
        }
    }
}

static void
hrn_switcher_unrealize (ClutterActor *self)
{
  HrnSwitcher        *switcher = HRN_SWITCHER (self);
  HrnSwitcherPrivate *priv     = HRN_SWITCHER_GET_PRIVATE (switcher);
  gint                i;


  for (i = 0; i < priv->state_count; i++)
    {
      if (priv->states[i].actor)
        {
          clutter_actor_unrealize (priv->states[i].actor);
        }
    }
  CLUTTER_ACTOR_CLASS (hrn_switcher_parent_class)->unrealize (self);
}

static void
hrn_switcher_class_init (HrnSwitcherClass *klass)
{
  GObjectClass      *gobject_class = G_OBJECT_CLASS (klass);
  ClutterActorClass *actor_class   = CLUTTER_ACTOR_CLASS (klass);

  gobject_class->dispose     = hrn_switcher_dispose;
  gobject_class->constructor = hrn_switcher_constructor;
  actor_class->paint         = hrn_switcher_paint;
  actor_class->pick          = hrn_switcher_pick;
  actor_class->map           = hrn_switcher_map;
  actor_class->unmap         = hrn_switcher_unmap;
  actor_class->realize       = hrn_switcher_realize;
  actor_class->unrealize     = hrn_switcher_unrealize;
  actor_class->allocate      = hrn_switcher_allocate;

  g_type_class_add_private (gobject_class, sizeof (HrnSwitcherPrivate));
}

static void
hrn_switcher_init (HrnSwitcher *self)
{
  HrnSwitcherPrivate *priv = HRN_SWITCHER_GET_PRIVATE (self);

  priv->state_count = 0;
  memset (priv, 0, sizeof (priv));
  priv->state = -1;
}

static GObject *
hrn_switcher_constructor (GType type, guint n_params,
                          GObjectConstructParam *params)
{
  GObject      *object;
  ClutterActor *unused_state;
  ClutterColor  red = { 0xff, 0x00, 0x00, 0x55 };
  HrnSwitcher  *switcher;

  object = G_OBJECT_CLASS (hrn_switcher_parent_class)->constructor (
    type, n_params, params);

  switcher = HRN_SWITCHER (object);

  unused_state = clutter_rectangle_new_with_color (&red);

  clutter_actor_set_size (unused_state, DIM, DIM);
  hrn_switcher_add_state (switcher, unused_state, 0.001, 0.01);
  clutter_actor_set_reactive (CLUTTER_ACTOR (object), TRUE);

  return object;
}

static void
hrn_switcher_allocate (ClutterActor *self, const ClutterActorBox *box,
                       ClutterAllocationFlags flags)
{
  HrnSwitcherPrivate *priv = HRN_SWITCHER_GET_PRIVATE (self);
  ClutterActorClass  *parent_class;

  parent_class = CLUTTER_ACTOR_CLASS (hrn_switcher_parent_class);

  parent_class->allocate (self, box, flags);

  hrn_switcher_adapt (HRN_SWITCHER (self));

  if (priv->states[priv->state].actor)
    clutter_actor_allocate_preferred_size (priv->states[priv->state].actor,
                                           flags);
}

void
hrn_switcher_set_state_no (HrnSwitcher *switcher, gint state_no)
{
  HrnSwitcherPrivate *priv = HRN_SWITCHER_GET_PRIVATE (switcher);

  g_assert (switcher);
  if (!(state_no < priv->state_count &&
        state_no >= 0))
    return;
  g_assert (state_no < priv->state_count &&
            state_no >= 0);

  /* do not switch if we're already in the desired state */
  if (priv->state == state_no)
    return;

  clutter_actor_queue_relayout (CLUTTER_ACTOR (switcher));

  priv->state = state_no;

  /* the requested state does not exist, so we should create it,
   *
   * A callback based design with constructors for the states when
   * creating the switcher should be added.
   *
   * XXX: ideally using an asynchronous load, keeping the previous
   * state around and showing it rescaled during the loading.
   */
  if (!priv->states[priv->state].actor)
    {
      if (priv->states[priv->state].constructor)
        {
          ClutterActor *child;
          child = priv->states[priv->state].constructor (
            switcher,
            priv->state,
            priv->states[priv->
                         state].
            constructor_data);

          if (clutter_actor_get_parent (child) == NULL)
            {
              /* the child might have opted to hook in earlier, allowing
               * computations for its own position
               */
              clutter_actor_set_parent (child, CLUTTER_ACTOR (switcher));
              g_object_ref (child);
            }
          priv->states[priv->state].actor = child;
        }
      if (!priv->states[priv->state].actor)
        g_debug ("no actor set for state %i", priv->state);
    }

#if 0 && HRN_PURGE_UNUSED
  /* purge unusued actors in switcher when the previous state
   * is a custom one (not the dummy rectangle),
   *
   * could probably be extended to only apply to offscreen states
   * to preserve the actors needed to zoom out.
   */
  if (state_no == 0)
    {
      gint i;
      for (i = 1; i < priv->state_count; i++)
        {
          if (priv->states[i].actor != NULL &&
              priv->states[i].constructor != NULL)
            {
              clutter_actor_destroy (priv->states[i].actor);
              priv->states[i].actor = NULL;
            }
        }
    }
  else if (state_no < 6 && priv->state_count > 10)
    {
      gint i;
      for (i = 6; i < priv->state_count; i++)
        {
          if (priv->states[i].actor != NULL &&
              priv->states[i].constructor != NULL)
            {
              clutter_actor_destroy (priv->states[i].actor);
              priv->states[i].actor = NULL;
            }
        }
    }
#endif
}

/**
 * hrn_switcher_add_state_full:
 * @self: a #HrnSwitcher
 * @child: a child (of a scale/scaled to a scale) corresponding to other actors.
 * @min_scale: from which depth this state should be used. (-1 for previous)
 * @max_scale: until which scale should this state be used
 *
 * Adds a new child as a state for the switcher, this is the state to be
 *displayed
 * from a given depth.
 */
void
hrn_switcher_add_state_full (
  HrnSwitcher *self, ClutterActor                  *child, gdouble min_scale,
  gdouble max_scale, ClutterActor * (*constructor)(HrnSwitcher * switcher,
                                                   guint
                                                   stateno,
                                                   gpointer
                                                   userdata),
  gpointer constructor_data,
  void (*adapt)(HrnSwitcher *switcher, guint state_no, ClutterActor *child,
                gdouble position,
                gpointer userdata), gpointer adapt_data
  )
{
  HrnSwitcherPrivate *priv = HRN_SWITCHER_GET_PRIVATE (self);

  g_assert (self);

  g_assert (priv->state_count + 1 < HRN_SWITCHER_MAX_STATES);

  if (min_scale <= -0.9)
    min_scale = priv->states[priv->state_count - 1].max_scale;

  priv->states[priv->state_count].actor            = child;
  priv->states[priv->state_count].min_scale        = min_scale;
  priv->states[priv->state_count].max_scale        = max_scale;
  priv->states[priv->state_count].constructor      = constructor;
  priv->states[priv->state_count].constructor_data = constructor_data;
  priv->states[priv->state_count].adapt            = adapt;
  priv->states[priv->state_count].adapt_data       = adapt_data;
  priv->states[priv->state_count].currprog         = -42; /* used to check for
                                                             change, -42 never
                                                             occurs */

  /* XXX: at the moment only supporting insertion of states
   * at incrementing depths.
   */

  if (priv->state_count > 0)
    g_assert (min_scale > priv->states[priv->state_count - 1].min_scale);

  priv->state_count++;

  if (priv->state_count == 1)
    {
      /* Set our initial state when the first state is added */
      hrn_switcher_set_state_no (self, 0);
    }

  if (child)
    {
      if (clutter_actor_get_parent (child) == NULL)
        {
          clutter_actor_set_parent (child,
                                    CLUTTER_ACTOR (self));
          g_object_ref_sink (child);
        }
    }
}


/**
 * hrn_switcher_add_state:
 * @self: a #HrnSwitcher
 * @child: a child (of a scale/scaled to a scale) corresponding to other actors.
 * @min_scale: from which depth this state should be used. (-1 for previous max
 *scale)
 * @max_scale: until which scale should this state be used
 *
 * Adds a new child as a state for the switcher, this is the state to be
 *displayed
 * from a given depth.
 */
void
hrn_switcher_add_state (HrnSwitcher *self, ClutterActor  *child,
                        gdouble min_scale,
                        gdouble max_scale)
{
  return hrn_switcher_add_state_full (self, child, min_scale, max_scale,
                                      NULL, NULL,
                                      NULL, NULL);
}

void
hrn_actor_get_geom (ClutterActor *actor, gfloat *ox, gfloat *oy, gfloat *owidth,
                    gfloat *oheight, gdouble *oscalex,
                    gdouble *oscaley)
{
  gfloat        x, y;
  gfloat        width, height;
  gdouble       ascalex = 1, ascaley = 1;
  gdouble       scalex, scaley;
  ClutterActor *parent = NULL;

  x      = clutter_actor_get_x (actor);
  y      = clutter_actor_get_y (actor);
  width  = clutter_actor_get_width (actor);
  height = clutter_actor_get_height (actor);

  clutter_actor_get_scale (actor, &scalex, &scaley);
  ascalex *= scalex;
  ascaley *= scaley;
  width   *= scalex;
  height  *= scaley;

  parent = clutter_actor_get_parent (actor);
  while (parent)
    {
      gfloat ax, ay;
      clutter_actor_get_anchor_point (parent, &ax, &ay);
      clutter_actor_get_scale (parent, &scalex, &scaley);
      x       -= ax;
      y       -= ay;
      x       *= scalex;
      y       *= scaley;
      ascalex *= scalex;
      ascaley *= scaley;
      x       += clutter_actor_get_x (parent);
      y       += clutter_actor_get_y (parent);
      width   *= scalex;
      height  *= scaley;
      parent   = clutter_actor_get_parent (parent);
    }

#if 0  /* these results are not good when trying to determine state
          they are more expensive than the above as well
        */
  {
    gfloat w, h;
    gint   nx, ny;
    clutter_actor_get_transformed_size (actor, &w, &h);
    clutter_actor_get_transformed_position (actor, &nx, &ny);
    x      = nx;
    y      = ny;
    width  = w;
    height = h;
  }
#endif

  if (ox) *ox = x;
  if (oy) *oy = y;
  if (owidth) *owidth = width;
  if (oheight) *oheight = height;
  if (oscalex) *oscalex = ascalex;
  if (oscaley) *oscaley = ascaley;
}

static gboolean
hrn_actor_is_off_stage (HrnSwitcher *switcher, ClutterActor *actor)
{
  gfloat              x, y;
  gfloat              width, height;
  gdouble             scalex, scaley;
  HrnSwitcherPrivate *priv = HRN_SWITCHER_GET_PRIVATE (switcher);

  /*gint sw = CLUTTER_STAGE_WIDTH ();*/
  gint sh = CLUTTER_STAGE_HEIGHT ();

  hrn_actor_get_geom (actor, &x, &y, &width, &height, &scalex, &scaley);

  if (
    (gint)(y + height) < -sh * (HRN_OFF_STAGE_FUDGE) ||
    /*(gint)(x + width)     < -sw *  (0.0) ||
                       x > sw + sw * (0.0) ||*/
    y > sh + sh * (HRN_OFF_STAGE_FUDGE))
    {
      if (priv->off_stage_count <= 0)
        return TRUE;
      priv->off_stage_count--;
      return FALSE;
    }
  else
    {
      priv->off_stage_count = HRN_OFF_STAGE_COUNT;
    }
  return FALSE;
}

gint
hrn_switcher_get_state (HrnSwitcher *switcher)
{
  HrnSwitcherPrivate *priv = HRN_SWITCHER_GET_PRIVATE (switcher);

  hrn_switcher_adapt (switcher);
  return priv->state;
}

/* adapt the state to the current depth the actor is drawn at */
void
hrn_switcher_adapt (HrnSwitcher *switcher)
{
  HrnSwitcherPrivate *priv      = HRN_SWITCHER_GET_PRIVATE (switcher);
  gint                use_state = 1;
  gint                i;

  gdouble             scale;

  if (priv->state_count == 0)
    return;

  scale = hrn_actor_get_abs_scale (CLUTTER_ACTOR (switcher));

  /* scaling to take the size including the items expanded into account */
  if (HRN_IS_CLUSTER (switcher))
    {
      gint cells = hrn_cluster_get_cells (CLUTTER_ACTOR (switcher));
      if (scale < 1.5)
        {
          clutter_actor_set_size (priv->states[0].actor, DIM, DIM);
        }
      else if (scale < 3.5)
        {
          clutter_actor_set_size (priv->states[0].actor, DIM, DIM / 4 *
                                  ((cells / 8) + 1));
        }
      else if (scale < 7.5)
        {
          clutter_actor_set_size (priv->states[0].actor, DIM, DIM / 4 *
                                  ((cells / 4) + 1));
        }
      else if (scale < 15.5)
        {
          clutter_actor_set_size (priv->states[0].actor, DIM, DIM / 5.7 *
                                  ((cells / 3) + 1));
        }
    }

  /* if the already displayed state is off screen, use state 0 (simplest
   * form)
   */
  if (hrn_actor_is_off_stage (switcher, priv->states[priv->state].actor))
    {
      /* if we're a cluster then we set a different size for the 0
       * state to make it appear in context
       */

      hrn_switcher_set_state_no (switcher, 0);
      return;
    }

  for (i = 0; i < priv->state_count; i++)
    {
      if (scale >= priv->states[i].min_scale)
        {
          use_state = i;
        }
      if (scale <= priv->states[i].min_scale)
        {
          hrn_switcher_set_state_no (switcher, use_state);

          if (priv->states[use_state].adapt)
            {
              gfloat progress;
              progress = (scale - priv->states[use_state].min_scale) * 1.0 /
                         (priv->states[use_state].max_scale -
                          priv->states[use_state].min_scale);
              if (progress > 1.0)
                progress = 1.0;
              else if (progress < 0.0)
                progress = 0.0;

              if (priv->states[use_state].currprog != progress)
                {
                  priv->states[use_state].currprog = progress;
                  priv->states[use_state].adapt (
                    switcher,
                    use_state,
                    priv->states[use_state].actor,
                    progress,
                    priv->states[use_state].
                    adapt_data);
                }
            }
          return;
        }
    }

  hrn_switcher_set_state_no (switcher, use_state);
  if (priv->states[use_state].adapt)
    priv->states[use_state].adapt (switcher,
                                   use_state, priv->states[use_state].actor,
                                   1.0, priv->states[use_state].adapt_data);
}


ClutterActor *
hrn_switcher_get_current (HrnSwitcher *switcher)
{
  HrnSwitcherPrivate *priv;

  hrn_switcher_adapt (switcher);
  if (!HRN_IS_SWITCHER (switcher))
    return NULL;


  priv = HRN_SWITCHER_GET_PRIVATE (switcher);

  if (priv->state == 0)
    return NULL;

  return priv->states[priv->state].actor;
}

static void
hrn_switcher_paint (ClutterActor *actor)
{
  HrnSwitcher        *switcher = HRN_SWITCHER (actor);
  HrnSwitcherPrivate *priv     = HRN_SWITCHER_GET_PRIVATE (switcher);
  ClutterActor       *child;

  /* bail if the default, non used state is to be displayed,
   * this saves resource */

  if (priv->state == 0 && !HRN_DEBUG_OFF_STAGE)
    return;

  child = priv->states[priv->state].actor;

  if (child != NULL)
    {
      clutter_actor_paint (child);
    }
}

static void
hrn_switcher_pick (ClutterActor *actor, const ClutterColor   *color)
{
  /* Chain up so we get a bounding box pained (if we are reactive) */
  CLUTTER_ACTOR_CLASS (hrn_switcher_parent_class)->pick (actor, color);

  hrn_switcher_paint (actor);
}

static void
hrn_switcher_dispose (GObject *object)
{
  HrnSwitcher        *switcher = HRN_SWITCHER (object);
  HrnSwitcherPrivate *priv     = HRN_SWITCHER_GET_PRIVATE (switcher);
  gint                i;

  for (i = 0; i < priv->state_count; i++)
    {
      if (priv->states[i].actor)
        {
          clutter_actor_destroy (priv->states[i].actor);
        }
      priv->states[i].actor = NULL;
    }
  G_OBJECT_CLASS (hrn_switcher_parent_class)->dispose (object);
}

HrnSwitcher *
hrn_switcher_new (gint width, gint height)
{
  HrnSwitcher *switcher = g_object_new (HRN_TYPE_SWITCHER, NULL);

  clutter_actor_set_size (CLUTTER_ACTOR (switcher), width, height);

  return switcher;
}
