/*
 * Hornsey - Moblin Media Player.
 * Copyright © 2009 Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */


#define ROW_OFFSET    5
#define ROW_HEIGHT    40

#include <clutter/clutter.h>
#include <math.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include "hrn.h"
#include "hrn-queue.h"

#define    QUEUE_TIP_HEIGHT    47  /* used to determine drops on tip */


G_DEFINE_TYPE (HrnQueue, hrn_queue, NBTK_TYPE_WIDGET);

#define HRN_QUEUE_GET_PRIVATE(obj)                 \
  (G_TYPE_INSTANCE_GET_PRIVATE ((obj), \
                                HRN_TYPE_QUEUE, \
                                HrnQueuePrivate))

struct _HrnQueuePrivate
{
  BrQueue *br_queue;

  NbtkWidget     *toptable;
  NbtkWidget     *header;
  NbtkWidget     *table;
  NbtkWidget     *view;
  NbtkWidget     *scrollview;
  ClutterGroup   *children;
  NbtkWidget     *playpause;
  NbtkWidget     *expanded;
  NbtkWidget     *timebar;
  NbtkAdjustment *progress;

  NbtkWidget *highlight;

  NbtkWidget     *audio_timebar;
  NbtkAdjustment *audio_progress;

  guint32 visual_progress_id;
  guint32 audio_progress_id;
  guint32 play_pause_id;

  gint i;

  NbtkWidget *name;     /* we store the string in the actor, and
                         * don't keep it ourselves
                         */

  BklItem    *active_visual;
  NbtkWidget *active_title;

  /* keep audio in a separate playqueue,
   * that can proceed in paralell with images
   */
  BklItem    *active_audio;
  NbtkWidget *active_audio_title;

  /* FIXME: Queue needs to be a GQueue */
  GList *queue;

  gint n;
};

typedef struct QueueItem
{
  HrnQueue     *queue;
  BklItem      *item;
  ClutterActor *actor;
}
QueueItem;

static GObject * hrn_queue_constructor (GType                  type,
                                        guint                  n_params,
                                        GObjectConstructParam *params);
static void hrn_queue_dispose          (GObject               *object);
static void hrn_queue_allocate         (ClutterActor          *self,
                                        const ClutterActorBox *box,
                                        ClutterAllocationFlags flags);
static void hrn_queue_reposition       (HrnQueue              *queue);
static void hrn_queue_paint            (ClutterActor          *actor);
static void hrn_queue_pick             (ClutterActor          *actor,
                                        const ClutterColor    *color);
static void add_item                   (HrnQueue              *queue,
                                        QueueItem             *queue_item);
static void hrn_queue_map              (ClutterActor          *self)
{
  HrnQueuePrivate *priv = HRN_QUEUE (self)->priv;

  CLUTTER_ACTOR_CLASS (hrn_queue_parent_class)->map (self);

  if (priv->toptable)
    clutter_actor_map (CLUTTER_ACTOR (priv->toptable));
}

static void
hrn_queue_unmap (ClutterActor *self)
{
  HrnQueuePrivate *priv = HRN_QUEUE (self)->priv;

  CLUTTER_ACTOR_CLASS (hrn_queue_parent_class)->unmap (self);

  if (priv->toptable)
    clutter_actor_unmap (CLUTTER_ACTOR (priv->toptable));
}

static void
hrn_queue_class_init (HrnQueueClass *klass)
{
  GObjectClass      *gobject_class = G_OBJECT_CLASS (klass);
  ClutterActorClass *actor_class   = CLUTTER_ACTOR_CLASS (klass);

  gobject_class->dispose      = hrn_queue_dispose;
  gobject_class->constructor  = hrn_queue_constructor;
  actor_class->allocate       = hrn_queue_allocate;
  actor_class->paint          = hrn_queue_paint;
  actor_class->pick           = hrn_queue_pick;
  actor_class->map            = hrn_queue_map;
  actor_class->unmap          = hrn_queue_unmap;

  g_type_class_add_private (gobject_class, sizeof (HrnQueuePrivate));
}

static void
hrn_queue_init (HrnQueue *self)
{
  self->priv = HRN_QUEUE_GET_PRIVATE (self);
  memset (self->priv, 0, sizeof (self->priv));
}

static gboolean
hide_actor (ClutterAnimation *anim, ClutterActor     *scrollview)
{
  clutter_actor_hide (CLUTTER_ACTOR (scrollview));
  return FALSE;
}

static gboolean
expanded_clicked_cb (NbtkButton *button, gpointer queue)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);

  if (nbtk_button_get_checked (button))
    {
      clutter_actor_animate (queue, CLUTTER_EASE_IN_OUT_CUBIC, 350,
                             "y", -300.0,
                             "height", 362.0,
                             NULL);
      clutter_actor_show (CLUTTER_ACTOR (priv->scrollview));
    }
  else
    {
      clutter_actor_animate (queue, CLUTTER_EASE_IN_OUT_CUBIC, 350,
                             "y", 0.0,
                             "height", 60.0,
                             "signal::completed", hide_actor, priv->scrollview,
                             NULL);
    }
  return TRUE;
}

static gboolean
playpause_clicked_cb (NbtkButton *button, HrnQueue     *queue)
{
  hrn_queue_set_playing (queue, nbtk_button_get_checked (button));
  return TRUE;
}


static gboolean
active_drop (ClutterActor *actor, ClutterActor *dropped_on, gint x, gint y,
             gpointer queue)
{
  BklItem *active = hrn_queue_get_active_visual (queue);

  if (active)
    {
      if (!dropped_on)
        {
          /* FIXME: Just want to play next item */
#if 0
          hrn_theatre_set_visual_playing (HRN_THEATRE (hrn_theatre), FALSE);
          hrn_queue_set_active_visual (queue, NULL);
          hrn_queue_visual_peek (queue);
#endif
          return TRUE;
        }
      else
        return hrn_item_drop (actor, dropped_on, x, y, active);
    }
  return FALSE;
}

static gboolean
active_audio_drop (ClutterActor *actor, ClutterActor *dropped_on, gint x,
                   gint y,
                   gpointer queue)
{
  BklItem *active_audio = hrn_queue_get_active_audio (queue);

  if (active_audio)
    {
      if (!dropped_on)
        {
          /* hrn_theatre_set_audio_playing (HRN_THEATRE (hrn_theatre), FALSE);
           */
          hrn_queue_set_active_audio (queue, NULL, "");
          return TRUE;
        }
      else
        return hrn_item_drop (actor, dropped_on, x, y, active_audio);
    }
  return FALSE;
}


static gboolean
hrn_item_can_drop_active_audio (ClutterActor *actor, ClutterActor *dropped_on,
                                gint x, gint y,
                                gpointer queue)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);

  if (!priv->active_audio)
    return FALSE;
  return hrn_item_can_drop (actor, dropped_on, x, y, queue);
}


static gboolean
hrn_item_can_drop_active_visual (ClutterActor *actor, ClutterActor *dropped_on,
                                 gint x, gint y,
                                 gpointer queue)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);

  if (!priv->active_visual)
    return FALSE;
  return hrn_item_can_drop (actor, dropped_on, x, y, queue);
}

static void
empty_queue (HrnQueue *queue)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);

  br_queue_clear (priv->br_queue);
  hrn_popup_close ();
}

static gint shuffle_sort (gconstpointer a,
                          gconstpointer b)
{
  return g_random_int_range (0, 52)<26;
}

static void
randomize_queue (HrnQueue *queue)
{
  gint i;
  gint l = hrn_queue_get_length (queue);

  GList *iter, *list = NULL;

  /* create a copy of all the items in the queue */
  for (i = 0; i < l; i++)
    {
      BklItem *item = hrn_queue_get (queue, i);
      list = g_list_prepend (list, item);
      g_object_ref (item);
    }

  empty_queue (queue);

  /* we shuffle the list by sorting it*/
  for (i = 0; i < 3; i++)
     list = g_list_sort (list, shuffle_sort);

  /* reinsert the queue items by appending all */
  for (iter = list; iter; iter = iter->next)
    {
      hrn_queue_append (queue, iter->data);
      g_object_unref (iter->data);
    }

  g_list_free (list);

  hrn_popup_close ();
}

#define QUEUE_WIDTH         300
#define QUEUE_WIDTH2        310 /* including padding */
#define QUEUE_ITEM_WIDTH    QUEUE_WIDTH - 23 /* minus interior padding to avoid
                                                horizontal scroll */

static gboolean
queue_release (ClutterActor *actor, ClutterEvent *event, gpointer userdata)
{
  if (event->button.button == 3)
    {
      gpointer actions[] = {
        N_ ("Empty"),   empty_queue,
        N_ ("Shuffle"), randomize_queue,
        NULL
      };
      hrn_popup_actor (event->button.x, event->button.y,
                       hrn_popup_actions (actions, actor));
      return TRUE;
    }
  return FALSE;
}

static void
visual_progress_notify (NbtkAdjustment *adjustment, GParamSpec     *pspec,
                        HrnQueue       *queue)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);
  double           pos  = nbtk_adjustment_get_value (adjustment);

  br_queue_set_position (priv->br_queue, BR_QUEUE_VISUAL_TYPE, pos);
}

static void
audio_progress_notify (NbtkAdjustment *adjustment, GParamSpec     *pspec,
                       HrnQueue       *queue)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);
  double           pos  = nbtk_adjustment_get_value (adjustment);

  br_queue_set_position (priv->br_queue, BR_QUEUE_AUDIO_TYPE, pos);
}

static GObject *
hrn_queue_constructor (GType type, guint n_params,
                       GObjectConstructParam *params)
{
  HrnQueuePrivate *priv;
  GObject         *object;
  HrnQueue        *queue;
  NbtkAdjustment  *vadj;

  object = G_OBJECT_CLASS (hrn_queue_parent_class)->constructor (
    type, n_params, params);

  queue = HRN_QUEUE (object);
  priv  = HRN_QUEUE_GET_PRIVATE (queue);

  priv->toptable = nbtk_table_new ();
  priv->header   = nbtk_table_new ();
  nbtk_widget_set_style_class_name (priv->toptable, "HrnQueue");
  nbtk_widget_set_style_class_name (priv->header, "HrnQueueHeader");
  clutter_actor_set_parent (CLUTTER_ACTOR (priv->toptable),
                            CLUTTER_ACTOR (queue));
  g_object_set_data (object, "HRN_DROP_MASK",
                     GINT_TO_POINTER (HRN_DROP_MASK_QUEUE));

  priv->queue     = NULL;
  priv->playpause = nbtk_button_new ();

  nbtk_button_set_toggle_mode (NBTK_BUTTON (priv->playpause), TRUE);
  nbtk_widget_set_style_class_name (priv->playpause, "HrnPlayPause");

  clutter_actor_set_size (CLUTTER_ACTOR (priv->playpause), 37, 45);
  priv->play_pause_id = g_signal_connect (priv->playpause, "clicked",
                                          G_CALLBACK (playpause_clicked_cb),
                                          queue);

  priv->expanded = nbtk_button_new ();

  nbtk_button_set_toggle_mode (NBTK_BUTTON (priv->expanded), TRUE);
  nbtk_widget_set_style_class_name (priv->expanded, "HrnQueueExpanded");

  clutter_actor_set_size (CLUTTER_ACTOR (priv->expanded), 42, 42);
  g_signal_connect (priv->expanded, "clicked", G_CALLBACK (
                      expanded_clicked_cb), queue);
  g_signal_connect (object, "button-release-event", G_CALLBACK (
                      queue_release), queue);



  priv->active_title       = nbtk_label_new ("");
  priv->active_audio_title = nbtk_label_new ("");

  hrn_actor_make_draggable (
    CLUTTER_ACTOR (priv->active_audio_title),
    CLUTTER_ACTOR (priv->active_audio_title),
    HRN_DROP_MASK_QUEUE,
    NULL,
    G_CALLBACK (hrn_item_can_drop_active_audio),
    G_CALLBACK (active_audio_drop),
    queue);

  hrn_actor_make_draggable (
    CLUTTER_ACTOR (priv->active_title),
    CLUTTER_ACTOR (priv->active_title),
    HRN_DROP_MASK_QUEUE,
    NULL,
    G_CALLBACK (hrn_item_can_drop_active_visual),
    G_CALLBACK (active_drop),
    queue);

  priv->name = nbtk_label_new ("unnamed");
  clutter_actor_set_width (CLUTTER_ACTOR (priv->name), QUEUE_WIDTH - 84);
  clutter_actor_set_width (CLUTTER_ACTOR (priv->active_title), QUEUE_WIDTH - 84);
  clutter_actor_set_width (CLUTTER_ACTOR (
                             priv->active_audio_title), QUEUE_WIDTH - 84);

  priv->progress           = nbtk_adjustment_new (0.0, 0.0, 1.0, 0.0, 0.0, 0.0);
  priv->visual_progress_id = g_signal_connect
                         (priv->progress, "notify::value",
                         G_CALLBACK (visual_progress_notify), queue);

  priv->timebar = NBTK_WIDGET (nbtk_scroll_bar_new (priv->progress));

  priv->audio_progress    = nbtk_adjustment_new (0.0, 0.0, 1.0, 0.0, 0.0, 0.0);
  priv->audio_progress_id = g_signal_connect
                         (priv->audio_progress, "notify::value",
                         G_CALLBACK (audio_progress_notify), queue);

  priv->audio_timebar = NBTK_WIDGET (nbtk_scroll_bar_new (priv->audio_progress));

  nbtk_widget_set_style_class_name (priv->timebar, "HrnProgressBar");
  nbtk_widget_set_style_class_name (priv->audio_timebar, "HrnProgressBar");

  priv->scrollview = (NbtkWidget *) nbtk_scroll_view_new ();

  nbtk_widget_set_style_class_name (priv->scrollview, "HrnQueueView");


  clutter_actor_set_size (CLUTTER_ACTOR (priv->timebar), QUEUE_WIDTH, 20);
  clutter_actor_set_size (CLUTTER_ACTOR (priv->audio_timebar), QUEUE_WIDTH, 20);

  nbtk_table_add_actor_with_properties (NBTK_TABLE (priv->toptable),
                                        CLUTTER_ACTOR (priv->header), 0, 0,
                                        "row-span", 1,
                                        "col-span", 1,
                                        "x-align", 0.0,
                                        "y-align", 0.0,
                                        "x-fill", TRUE,
                                        "x-expand", TRUE,
                                        "y-fill", TRUE,
                                        "y-expand", FALSE,
                                        NULL);

  nbtk_table_add_actor_with_properties (NBTK_TABLE (priv->header),
                                        CLUTTER_ACTOR (priv->expanded), 0, 0,
                                        "row-span", 3,
                                        "col-span", 1,
                                        "x-align", 0.0,
                                        "y-align", 0.0,
                                        "x-fill", FALSE,
                                        "x-expand", FALSE,
                                        "y-fill", FALSE,
                                        "y-expand", FALSE,
                                        NULL);


  nbtk_table_add_actor_with_properties (NBTK_TABLE (priv->header),
                                        CLUTTER_ACTOR (priv->playpause), 0, 2,
                                        "row-span", 3,
                                        "col-span", 1,
                                        "x-align", 0.0,
                                        "y-align", 0.0,
                                        "y-fill", FALSE,
                                        "y-expand", FALSE,
                                        "x-fill", FALSE,
                                        "x-expand", FALSE,
                                        NULL);

  nbtk_table_add_actor_with_properties (NBTK_TABLE (priv->header),
                                        CLUTTER_ACTOR (
                                          priv->active_title), 1, 1,
                                        "row-span", 1,
                                        "col-span", 1,
                                        "x-fill", TRUE,
                                        "x-expand", TRUE,
                                        "y-fill", FALSE,
                                        "y-expand", FALSE,
                                        "x-align", 0.0,
                                        "y-align", 0.0,
                                        NULL);

  /* clutter_actor_set_width (CLUTTER_ACTOR (priv->active_title), 200);
     clutter_actor_set_width (CLUTTER_ACTOR (priv->active_audio_title), 200); */
  nbtk_table_add_actor_with_properties (NBTK_TABLE (priv->header),
                                        CLUTTER_ACTOR (
                                          priv->active_audio_title), 2, 1,
                                        "row-span", 1,
                                        "col-span", 1,
                                        "x-fill", TRUE,
                                        "x-expand", TRUE,
                                        "y-fill", FALSE,
                                        "y-expand", FALSE,
                                        "x-align", 0.0,
                                        "y-align", 0.0,
                                        NULL);


  nbtk_table_add_actor_with_properties (NBTK_TABLE (priv->header),
                                        CLUTTER_ACTOR (priv->timebar), 3, 0,
                                        "row-span", 1,
                                        "col-span", 3,
                                        "x-fill", TRUE,
                                        "y-fill", FALSE,
                                        "x-expand", TRUE,
                                        "y-expand", FALSE,
                                        "x-align", 0.0,
                                        "y-align", 0.0,
                                        NULL);

  nbtk_table_add_actor_with_properties (NBTK_TABLE (priv->header),
                                        CLUTTER_ACTOR (
                                          priv->audio_timebar), 4, 0,
                                        "row-span", 1,
                                        "col-span", 3,
                                        "x-fill", TRUE,
                                        "y-fill", FALSE,
                                        "x-expand", TRUE,
                                        "y-expand", FALSE,
                                        "x-align", 0.0,
                                        "y-align", 0.0,
                                        NULL);

  nbtk_table_add_actor_with_properties (NBTK_TABLE (priv->toptable),
                                        CLUTTER_ACTOR (priv->scrollview), 2, 0,
                                        "row-span", 1,
                                        "col-span", 3,
                                        "x-fill", TRUE,
                                        "x-expand", TRUE,
                                        "y-expand", TRUE,
                                        "y-fill", TRUE,
                                        "x-align", 0.0,
                                        "y-align", 0.0,
                                        NULL);

  clutter_actor_set_reactive (CLUTTER_ACTOR (object), TRUE);

  priv->view = (NbtkWidget *) nbtk_viewport_new ();
  clutter_container_add (CLUTTER_CONTAINER (priv->scrollview),
                         CLUTTER_ACTOR (priv->view), NULL);
  nbtk_scrollable_get_adjustments (NBTK_SCROLLABLE (priv->view), NULL, &vadj);
  g_object_set (vadj,
                "step-increment", (double) ROW_HEIGHT,
                "page-increment", (double) ROW_HEIGHT,
                NULL);

  priv->children = CLUTTER_GROUP (hrn_queue_group_new ());
  clutter_container_add (CLUTTER_CONTAINER (priv->view),
                         CLUTTER_ACTOR (priv->children), NULL);
  priv->highlight = nbtk_label_new ("");
  clutter_actor_set_size (CLUTTER_ACTOR (priv->highlight), QUEUE_ITEM_WIDTH, 45);
  clutter_actor_set_anchor_point_from_gravity (CLUTTER_ACTOR (
                                                 priv->highlight),
                                               CLUTTER_GRAVITY_WEST);
  nbtk_widget_set_style_class_name (priv->highlight, "HrnQueueHighlight");
  clutter_container_add (CLUTTER_CONTAINER (priv->children),
                         CLUTTER_ACTOR (priv->highlight), NULL);
  clutter_actor_hide (CLUTTER_ACTOR (priv->highlight));

  /* hack to sync up size */
  {
    clutter_actor_set_size (CLUTTER_ACTOR (priv->header), QUEUE_WIDTH, 60);
    clutter_actor_set_size (CLUTTER_ACTOR (object), QUEUE_WIDTH2, 350);
  }

  hrn_queue_hide_video_progress (queue);
  hrn_queue_hide_audio_progress (queue);

  return object;
}

static void
hrn_queue_allocate (ClutterActor *self, const ClutterActorBox *box,
                    ClutterAllocationFlags flags)
{
  HrnQueuePrivate   *priv = HRN_QUEUE_GET_PRIVATE (self);
  ClutterActorClass *parent_class;
  ClutterActorBox    actor_box;

  parent_class = CLUTTER_ACTOR_CLASS (hrn_queue_parent_class);

  parent_class->allocate (self, box, flags);
  actor_box    = *box;
  actor_box.y2 = box->y2 - box->y1;
  actor_box.y1 = 0;
  clutter_actor_allocate (CLUTTER_ACTOR (priv->toptable), &actor_box, flags);
}

static void
hrn_queue_dispose (GObject *object)
{
  G_OBJECT_CLASS (hrn_queue_parent_class)->dispose (object);
}

static void
hrn_queue_paint (ClutterActor *actor)
{
  HrnQueue        *queue = HRN_QUEUE (actor);
  HrnQueuePrivate *priv  = HRN_QUEUE_GET_PRIVATE (queue);

  /*CLUTTER_ACTOR_CLASS (hrn_queue_parent_class)->paint (actor);*/

  clutter_actor_paint (CLUTTER_ACTOR (priv->toptable));
}

#include <cogl/cogl.h>

static void
hrn_queue_pick (ClutterActor *actor, const ClutterColor   *color)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (actor);
  gfloat           width;
  gfloat           height;

  clutter_actor_get_size (CLUTTER_ACTOR (priv->toptable), &width, &height);

#if CLUTTER_MINOR_VERSION < 9
  cogl_color (color);
#else
  cogl_set_source_color4ub (color->red,
                            color->green,
                            color->blue,
                            color->alpha);
#endif
  cogl_rectangle (0, 0, width, height);

  hrn_queue_paint (actor);
}

static void
brq_uri_added (BrQueue *brq, const char *uri, int position, HrnQueue   *queue)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);
  BklItem         *item;

  item = hrn_get_item_for_uri (uri);
  if (item)
    {
      QueueItem *queue_item = g_new0 (QueueItem, 1);

      queue_item->item  = g_object_ref (item);
      queue_item->queue = queue;

      priv->queue = g_list_insert (priv->queue, queue_item, position);

      add_item (queue, queue_item);

      if (!nbtk_button_get_checked (NBTK_BUTTON (priv->expanded)))
        {
          nbtk_button_set_checked (NBTK_BUTTON (priv->expanded), TRUE);
          expanded_clicked_cb (NBTK_BUTTON (priv->expanded), queue);
        }

      /* If the item is an image, then we want to start the slideshow again */
      if (bkl_item_get_item_type (item) == BKL_ITEM_TYPE_IMAGE) {
        hrn_theatre_set_image_slideshow ((HrnTheatre *) hrn_theatre, TRUE);
      }
    }
}

static void
brq_uri_removed (BrQueue *brq, const char *uri, int position, HrnQueue   *queue)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);
  GList           *queue_list;
  QueueItem       *queue_item;

  queue_list = g_list_nth (priv->queue, position);
  if (queue_list == NULL)
    {
      return;
    }

  queue_item = queue_list->data;

  priv->queue = g_list_remove_link (priv->queue, queue_list);
  g_list_free (queue_list);

  g_object_unref (queue_item->item);
  clutter_actor_destroy (queue_item->actor);
  g_free (queue_item);

  hrn_queue_reposition (queue);
}

static void
brq_position_changed (BrQueue *brq, BrQueueType type, double position,
                      HrnQueue   *queue)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);

  switch (type)
    {
      case BR_QUEUE_VISUAL_TYPE:
        g_signal_handler_block (priv->progress, priv->visual_progress_id);
        nbtk_adjustment_set_value (priv->progress, position);
        g_signal_handler_unblock (priv->progress, priv->visual_progress_id);
        break;

      case BR_QUEUE_AUDIO_TYPE:
        g_signal_handler_block (priv->audio_progress, priv->audio_progress_id);
        nbtk_adjustment_set_value (priv->audio_progress, position);
        g_signal_handler_unblock (priv->audio_progress, priv->audio_progress_id);
        break;
    }
}

static void
brq_now_playing_changed (BrQueue     *brq,
                         const char  *uri, 
                         BrQueueType  type,
                         HrnQueue *queue)
{
  BklItem *item;

  item = hrn_get_item_for_uri (uri);

  /* Reset the position of the appropriate slider to 0 */
  brq_position_changed (brq, type, 0.0, queue);

  if (uri == NULL || uri[0]==0) /* finished playback of this media type */
    {
      switch (type)
        {
           case BR_QUEUE_VISUAL_TYPE:
             hrn_queue_hide_video_progress (queue);
             hrn_queue_set_active_visual (queue, NULL, "");
             break;
           case BR_QUEUE_AUDIO_TYPE:
             hrn_queue_hide_audio_progress (queue);
             hrn_queue_set_active_audio (queue, NULL, "");
             break;
        }
      return;
    }

  switch (type)
    {
      gboolean is_video;
      case BR_QUEUE_VISUAL_TYPE:
        is_video = FALSE;
        if (item)
          {
            if (bkl_item_get_item_type (item) == BKL_ITEM_TYPE_VIDEO)
              is_video = TRUE;
          }
        else
          {
            gchar *mimetype = hrn_resolve_mimetype (uri);
            if (g_str_has_prefix (mimetype, "video"))
              is_video = TRUE;
            g_free (mimetype);
          }
        
        hrn_queue_set_active_visual (queue, item, uri);

        if (is_video)
          {
            hrn_queue_show_video_progress (queue);
            hrn_queue_hide_audio_progress (queue);
          }
        else
          {
            hrn_queue_hide_video_progress (queue);
            hrn_queue_hide_audio_progress (queue);
          }
        break;

      case BR_QUEUE_AUDIO_TYPE:
        hrn_queue_set_active_audio (queue, item, uri);
        hrn_queue_show_audio_progress (queue);
        hrn_queue_hide_video_progress (queue);
        break;
    }
}

static void
brq_playing_changed (BrQueue  *brq,
                     gboolean  playing,
                     HrnQueue *queue)
{
  HrnQueuePrivate *priv  = HRN_QUEUE_GET_PRIVATE (queue);

  g_signal_handler_block (priv->playpause, priv->play_pause_id);
  nbtk_button_set_checked ((NbtkButton *) priv->playpause, playing);
  g_signal_handler_unblock (priv->playpause, priv->play_pause_id);
}

static void
set_name_reply (BrQueue *brq, char    *name, GError  *error, gpointer data)
{
  HrnQueue        *queue = (HrnQueue *) data;
  HrnQueuePrivate *priv  = HRN_QUEUE_GET_PRIVATE (queue);

  if (error != NULL)
    {
      g_warning ("Error getting queue name: %s", error->message);
      return;
    }

  nbtk_label_set_text ((NbtkLabel *) priv->name, name);
}

HrnQueue *
hrn_queue_new (BrQueue *br_queue)
{
  HrnQueue        *queue = g_object_new (HRN_TYPE_QUEUE, NULL);
  HrnQueuePrivate *priv  = HRN_QUEUE_GET_PRIVATE (queue);

  priv->br_queue = g_object_ref (br_queue);
  g_signal_connect (priv->br_queue, "uri-added",
                    G_CALLBACK (brq_uri_added), queue);
  g_signal_connect (priv->br_queue, "uri-removed",
                    G_CALLBACK (brq_uri_removed), queue);
  g_signal_connect (priv->br_queue, "now-playing-changed",
                    G_CALLBACK (brq_now_playing_changed), queue);
  g_signal_connect (priv->br_queue, "position-changed",
                    G_CALLBACK (brq_position_changed), queue);
  g_signal_connect (priv->br_queue, "playing-changed",
                    G_CALLBACK (brq_playing_changed), queue);

  /* Set the name of the queue from Bognor */
  br_queue_get_name (br_queue, set_name_reply, queue);

  return queue;
}

static void
list_uris_reply (BrQueue *brq,
                 char   **uris,
                 GError  *error,
                 gpointer data)
{
  HrnQueue        *queue = (HrnQueue *) data;
  HrnQueuePrivate *priv  = HRN_QUEUE_GET_PRIVATE (queue);
  int              i;

  if (error != NULL)
    {
      g_warning ("Error listing queue uris: %s", error->message);
      return;
    }

  for (i = 0; uris[i]; i++)
    {
      BklItem *item = hrn_get_item_for_uri (uris[i]);
      if (item)
        {
          QueueItem *queue_item = g_new0 (QueueItem, 1);

          queue_item->item  = g_object_ref (item);
          queue_item->queue = queue;

          priv->queue = g_list_append (priv->queue, queue_item);

          add_item (queue, queue_item);

          if (!nbtk_button_get_checked (NBTK_BUTTON (priv->expanded)))
            {
              nbtk_button_set_checked (NBTK_BUTTON (priv->expanded), TRUE);
              expanded_clicked_cb (NBTK_BUTTON (priv->expanded), queue);
            }
        }
    }
}

static void
get_now_playing_reply (BrQueue *brq,
                       char    *audio_uri,
                       char    *visual_uri,
                       GError  *error,
                       gpointer data)
{
    HrnQueue *queue = (HrnQueue *) data;

    if (error != NULL) {
        g_warning ("Error getting now playing: %s", error->message);
        return;
    }

    brq_now_playing_changed (brq, audio_uri, BR_QUEUE_AUDIO_TYPE, queue);
    brq_now_playing_changed (brq, visual_uri, BR_QUEUE_VISUAL_TYPE, queue);

#if 0
    if (visual_uri) {
        BklItem *item;

        item = hrn_get_item_for_uri (visual_uri);
        if (item) {
            hrn_queue_play_now (queue, item);
            br_queue_stop (brq);
        }
    }
#endif
}

static void
get_playing_reply (BrQueue *brq,
                   gboolean playing,
                   GError  *error,
                   gpointer data)
{
  HrnQueue *queue = (HrnQueue *) data;

  if (error != NULL) {
    g_warning ("Error getting initial play state: %s", error->message);
    return;
  }

  brq_playing_changed (brq, playing, queue);
}

void
hrn_queue_populate (HrnQueue *queue)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);

  br_queue_list_uris (priv->br_queue, list_uris_reply, queue);
  br_queue_get_now_playing (priv->br_queue, get_now_playing_reply, queue);
  br_queue_get_playing (priv->br_queue, get_playing_reply, queue);
}

void
hrn_queue_set_name (HrnQueue *queue, const gchar *name)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);

  nbtk_label_set_text (NBTK_LABEL (priv->name), name);
  nbtk_label_set_text (NBTK_LABEL (priv->name), " ");
}

const gchar *
hrn_queue_get_name (HrnQueue *queue)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);

  return clutter_text_get_text (CLUTTER_TEXT (priv->name));
}

gboolean
hrn_item_can_drop (ClutterActor *actor, ClutterActor *dropped_on, gint x,
                   gint y,
                   gpointer item)
{
  ClutterActor    *queue = dropped_on;
  gint             no    = 0;
  HrnQueuePrivate *priv;

  while (queue && !HRN_IS_QUEUE (queue))
    queue = clutter_actor_get_parent (queue);

  if (!queue || !item)
    return FALSE;

  priv = HRN_QUEUE_GET_PRIVATE (queue);

  {
    gfloat         xu, yu;
    gdouble        foo;
    NbtkScrollBar *scrollbar = NBTK_SCROLL_BAR (
      nbtk_scroll_view_get_vscroll_bar (NBTK_SCROLL_VIEW (priv->scrollview)));
    foo = nbtk_adjustment_get_value (nbtk_scroll_bar_get_adjustment (scrollbar));

    clutter_actor_transform_stage_point (CLUTTER_ACTOR (queue),
                                         x, y, &xu, &yu);

    no = ((floor (yu) + (gint) foo) - ROW_OFFSET) / ROW_HEIGHT - 1;

    if (no == -1)
      return TRUE;

    if (no > g_list_length (priv->queue) - (G_IS_OBJECT (item) ? 0 : 1))
      no = g_list_length (priv->queue) - (G_IS_OBJECT (item) ? 0 : 1);

    clutter_actor_set_y (CLUTTER_ACTOR (priv->highlight), no * ROW_HEIGHT);

    clutter_actor_show (CLUTTER_ACTOR (priv->highlight));
    clutter_actor_set_opacity (CLUTTER_ACTOR (priv->highlight), 0xff);

    clutter_actor_animate (CLUTTER_ACTOR (
                             priv->highlight), CLUTTER_EASE_IN_OUT_CUBIC, 2000,
                           "opacity", 0,
                           "signal::completed", hide_actor, priv->highlight,
                           NULL);
  }

  return TRUE;
}


gboolean
hrn_item_drop (ClutterActor *actor, ClutterActor *dropped_on, gint x, gint y,
               gpointer item)
{
  ClutterActor    *queue = dropped_on;
  gint             no    = 0;
  HrnQueuePrivate *priv;

  while (queue && !HRN_IS_QUEUE (queue))
    queue = clutter_actor_get_parent (queue);

  if (!queue || !item)
    return FALSE;

  priv = HRN_QUEUE_GET_PRIVATE (queue);

  {
    gfloat         xu, yu;
    gdouble        foo;
    NbtkScrollBar *scrollbar = NBTK_SCROLL_BAR (
      nbtk_scroll_view_get_vscroll_bar (NBTK_SCROLL_VIEW (priv->scrollview)));
    foo = nbtk_adjustment_get_value (nbtk_scroll_bar_get_adjustment (scrollbar));

    clutter_actor_transform_stage_point (CLUTTER_ACTOR (queue),
                                         x, y, &xu, &yu);

    no = ((floor (yu) + (gint) foo) - ROW_OFFSET) / ROW_HEIGHT - 1;
    if (yu < QUEUE_TIP_HEIGHT)
      no = -1;
  }

  g_assert (HRN_IS_QUEUE (queue));

  if (hrn_view_has_selected () &&
      hrn_view_is_selected (bkl_item_get_uri (item)))
    {
      GList *s, *selected = hrn_view_get_selected (HRN_VIEW (hrn_view));

      for (s = selected; s; s = s->next)
        {
          hrn_queue_insert (HRN_QUEUE (queue), no, s->data);
          no++;
        }
      g_list_free (selected);
      /* there is a selection */
    }
  else
    {
      hrn_queue_insert (HRN_QUEUE (queue), no, item);
    }
  return TRUE;
}


gboolean
hrn_cluster_dropped (ClutterActor *actor, ClutterActor *dropped_on, gint x,
                     gint y,
                     gpointer item)
{
  ClutterActor    *queue = dropped_on;
  gint             no    = 0;
  HrnQueuePrivate *priv;

  while (queue && !HRN_IS_QUEUE (queue))
    queue = clutter_actor_get_parent (queue);
  if (!queue)
    {
      return FALSE;
    }

  priv = HRN_QUEUE_GET_PRIVATE (queue);

  {
    gfloat         xu, yu;
    gdouble        foo;
    NbtkScrollBar *scrollbar = NBTK_SCROLL_BAR (
      nbtk_scroll_view_get_vscroll_bar (NBTK_SCROLL_VIEW (priv->scrollview)));
    foo = nbtk_adjustment_get_value (nbtk_scroll_bar_get_adjustment (scrollbar));

    clutter_actor_transform_stage_point (CLUTTER_ACTOR (queue),
                                         xu, yu, &xu, &yu);

    no = ((floor (yu) + (gint) foo) - ROW_OFFSET) / ROW_HEIGHT - 1;
    if (yu < QUEUE_TIP_HEIGHT)
      no = -1;
  }

  g_assert (HRN_IS_QUEUE (queue));
  {
    GList *iter;
    for (iter = HRN_CLUSTER (item)->results; iter; iter = iter->next)
      hrn_queue_insert (HRN_QUEUE (queue), no++, iter->data);
  }
  return TRUE;
}

static gboolean
hrn_item_queue_start_drag (ClutterActor *actor, gint x, gint y,
                           QueueItem    *queue_item)
{
  clutter_actor_hide (actor);
  hrn_queue_reposition (queue_item->queue);
  return TRUE;
}

static gboolean
from_queue_drop (ClutterActor *actor, ClutterActor *dropped_on, gint x, gint y,
                 QueueItem    *queue_item)
{
  gint             no   = 0;
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue_item->queue);

  if (dropped_on == NULL)
    {
      GList *iter;
      for (iter = priv->queue; iter; iter = iter->next)
        {
          if (iter->data == queue_item)
            break;
          no++;
        }
      hrn_queue_remove (queue_item->queue, no);
      return TRUE;
    }

  {
    gfloat         xu, yu;
    gdouble        foo;
    NbtkScrollBar *scrollbar = NBTK_SCROLL_BAR (
      nbtk_scroll_view_get_vscroll_bar (NBTK_SCROLL_VIEW (priv->scrollview)));
    foo = nbtk_adjustment_get_value (nbtk_scroll_bar_get_adjustment (scrollbar));

    clutter_actor_transform_stage_point (CLUTTER_ACTOR (queue_item->queue),
                                         x, y, &xu, &yu);

    no = 0;
    {
      GList *iter;
      for (iter = priv->queue; iter; iter = iter->next)
        {
          if (iter->data == queue_item)
            break;
          no++;
        }
    }

    {
      HrnQueue        *queue = HRN_QUEUE (dropped_on);
      HrnQueuePrivate *qp    = HRN_QUEUE_GET_PRIVATE (queue);
      BklItem         *item  = queue_item->item;

      g_object_ref (item);
      hrn_queue_remove (queue_item->queue, no);

      no = ((floor (yu) + (gint) foo) - ROW_OFFSET) / ROW_HEIGHT - 1;
      if (yu < QUEUE_TIP_HEIGHT)
        no = -1;

      if (no == -1)
        {
          br_queue_stop (qp->br_queue);
          br_queue_play_uri (qp->br_queue, bkl_item_get_uri (item),
                             bkl_item_get_mimetype (item));
        }
      else
        {
          hrn_queue_insert (queue, no, item);
        }

      g_object_unref (item);
    }
  }
  return TRUE;
}


static void
hrn_queue_reposition (HrnQueue *queue)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);
  GList           *iter;
  gint             no = 0;

  for (iter = priv->queue; iter; iter = iter->next)
    {
      QueueItem *item = iter->data;
      if (CLUTTER_ACTOR_IS_VISIBLE (item->actor))
        {
          gfloat x = 7;
          gfloat y = no * ROW_HEIGHT + ROW_OFFSET;
          if (clutter_actor_get_x (item->actor) == 0 &&
              clutter_actor_get_y (item->actor) == 0)

            {
              clutter_actor_set_position (CLUTTER_ACTOR (item->actor), x, y);
            }
          else
            {
              clutter_actor_animate (CLUTTER_ACTOR (item->actor),
                                     CLUTTER_LINEAR, 400,
                                     "x", x, "y", y,
                                     NULL);
            }
          clutter_actor_animate (CLUTTER_ACTOR (item->actor),
                                 CLUTTER_LINEAR, 400,
                                 "opacity", 0xff,
                                 NULL);
          no++;
        }
    }
}

static void
add_item (HrnQueue *queue, QueueItem *queue_item)
{
  HrnQueuePrivate *priv          = HRN_QUEUE_GET_PRIVATE (queue);
  BklItem         *item          = queue_item->item;
  ClutterColor     backing_color = { 0xdd, 0xdd, 0xdd, 0x00 };
  ClutterActor    *backing       = clutter_rectangle_new ();
  ClutterActor    *group         = clutter_group_new ();
  ClutterActor    *actor         = hrn_make_element (item, TRUE);
  gchar           *title         = hrn_item_get_title (item);
  gchar           *meta          = hrn_item_get_meta (item);
  NbtkWidget      *text          = nbtk_label_new (title);
  NbtkWidget      *meta_text     = nbtk_label_new (meta);
  gfloat           w, h;
  gdouble          scale;

  nbtk_widget_set_style_class_name (text, "HrnQueueItemLabel");
  nbtk_widget_set_style_class_name (meta_text, "HrnQueueItemMetaLabel");


  clutter_rectangle_set_color (CLUTTER_RECTANGLE (backing), &backing_color);
  clutter_actor_set_opacity (backing, 0xbb);
  g_free (title);

  clutter_actor_get_size (actor, &w, &h);
  clutter_actor_set_size (group, QUEUE_ITEM_WIDTH, 36);
  scale = 33.0 / h;
  clutter_actor_set_size (backing, QUEUE_ITEM_WIDTH, 36);
  clutter_actor_set_scale (actor, scale, scale);

  clutter_group_add (CLUTTER_GROUP (group), backing);
  clutter_group_add (CLUTTER_GROUP (group), actor);
  clutter_group_add (CLUTTER_GROUP (group), text);

  clutter_actor_set_width (CLUTTER_ACTOR (text), QUEUE_ITEM_WIDTH - 42);
  clutter_actor_set_position (CLUTTER_ACTOR (text), 42, 6);

  clutter_group_add (CLUTTER_GROUP (group), meta_text);
  clutter_actor_set_width (CLUTTER_ACTOR (meta_text), QUEUE_ITEM_WIDTH - 42);
  clutter_actor_set_position (CLUTTER_ACTOR (meta_text), 42, 6 +
                              clutter_actor_get_height (CLUTTER_ACTOR (text)));
  clutter_actor_set_position (actor, 2, 0);

  clutter_container_add_actor (CLUTTER_CONTAINER (
                                 priv->children), (void*) group);

  clutter_actor_set_opacity (group, 0);

  hrn_actor_make_draggable (
    CLUTTER_ACTOR (group),
    group,
    HRN_DROP_MASK_QUEUE,
    G_CALLBACK (hrn_item_queue_start_drag),
    G_CALLBACK (hrn_item_can_drop),
    G_CALLBACK (from_queue_drop),
    queue_item);

  priv->n++;
  g_object_ref (item);
  queue_item->actor = group;

  clutter_actor_queue_relayout (CLUTTER_ACTOR (queue));
  hrn_queue_reposition (queue);
}


void
hrn_queue_insert (HrnQueue *queue, gint pos, BklItem     *item)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);

  br_queue_insert_uri (priv->br_queue, bkl_item_get_uri (item),
                       bkl_item_get_mimetype (item), pos);
}


void
hrn_queue_append (HrnQueue *queue, BklItem  *item)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);

  br_queue_add_uri (priv->br_queue, bkl_item_get_uri (item),
                    bkl_item_get_mimetype (item));
}

void
hrn_queue_prepend_list (HrnQueue *queue, GList       *items)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);
  GList           *iter;

  for (iter = g_list_last (items); iter; iter = iter->prev)
    {
      BklItem *item = iter->data;

      br_queue_insert_uri (priv->br_queue, bkl_item_get_uri (item),
                           bkl_item_get_mimetype (item), 0);
    }

  if (!nbtk_button_get_checked (NBTK_BUTTON (priv->expanded)))
    {
      nbtk_button_set_checked (NBTK_BUTTON (priv->expanded), TRUE);
      expanded_clicked_cb (NBTK_BUTTON (priv->expanded), queue);
    }
}




gint
hrn_queue_get_length (HrnQueue *queue)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);

  return g_list_length (priv->queue);
}



void
hrn_queue_remove (HrnQueue *queue, gint pos)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);

  br_queue_remove (priv->br_queue, pos);
}

BklItem *
hrn_queue_get (HrnQueue *queue, gint pos)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);
  QueueItem       *item;

  item = g_list_nth_data (priv->queue, pos);
  if (item)
    {
      return item->item;
    }
  else
    {
      return NULL;
    }
}

BklItem     *
hrn_queue_proceed (HrnQueue *queue, gint item_type_mask)
{
  gint len = hrn_queue_get_length (queue);

  if (len)
    {
      gint i;
      for (i = 0; i < len; i++)
        {
          BklItem *ret = hrn_queue_get (queue, i);
          if (bkl_item_get_item_type (ret) & item_type_mask)
            {
              hrn_queue_remove (queue, i);
              return ret;
            }
        }
    }
  return NULL;
}

gboolean
hrn_queue_get_playing (HrnQueue *queue)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);

  return nbtk_button_get_checked (NBTK_BUTTON (priv->playpause));
}

void
hrn_queue_set_playing (HrnQueue *queue, gboolean playing)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);
  QueueItem  *item;

  if (priv->queue) {
      item = priv->queue->data;

      if (item) {
          BklItem    *bi;
          const char *mimetype;

          bi = item->item;
          mimetype = bkl_item_get_mimetype (bi);
          if (mimetype)
              {
                  if (g_str_has_prefix (mimetype, "video/") ||
                      g_str_has_prefix (mimetype, "image/"))
                      {
                          hrn_set_theatre ();
                      }
              }
      }
  }

  if (playing)
    {
      /* We want to turn the slideshow back on when the queue is started */
      /* FIXME: Stop using global variables... */
      hrn_theatre_set_image_slideshow ((HrnTheatre *) hrn_theatre, TRUE);
      br_queue_play (priv->br_queue);
    }
  else
    {
      br_queue_stop (priv->br_queue);
    }
}


void
hrn_queue_set_active_visual (HrnQueue    *queue,
                             BklItem     *item,
                             const gchar *uri)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);

  if (priv->active_visual)
    g_object_unref (priv->active_visual);
  priv->active_visual = NULL;

  if (item)
    {
      gchar *title;
      priv->active_visual = g_object_ref (item);
      title               = hrn_item_get_title (item);
      nbtk_label_set_text (NBTK_LABEL (priv->active_title), title);
      g_free (title);
    }
  else
    {
      if (uri)
        {
          GFile *file;
          gchar *basename;
          file = g_file_new_for_uri (uri);
          basename = g_file_get_basename (file);
          nbtk_label_set_text (NBTK_LABEL (priv->active_title), basename);
          if (basename)
            g_free (basename);
          g_object_unref (file);
        }
      else
        {
          nbtk_label_set_text (NBTK_LABEL (priv->active_title), "");
        }
    }
}

BklItem     *
hrn_queue_get_active_visual (HrnQueue *queue)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);

  return priv->active_visual;
}

void
hrn_queue_set_active_audio (HrnQueue *queue,
                               BklItem     *item,
                               const gchar *uri)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);

  if (priv->active_audio)
    g_object_unref (priv->active_audio);
  priv->active_audio = NULL;

  if (item)
    {
      gchar *title;
      priv->active_audio = g_object_ref (item);
      title              = hrn_item_get_title (item);
      nbtk_label_set_text (NBTK_LABEL (priv->active_audio_title), title);
      g_free (title);
    }
  else
    {
      if (uri)
        {
          GFile *file;
          gchar *basename;
          file = g_file_new_for_uri (uri);
          basename = g_file_get_basename (file);
          nbtk_label_set_text (NBTK_LABEL (priv->active_audio_title), basename);
          if (basename)
            g_free (basename);
          g_object_unref (file);
        }
      else
        {
          nbtk_label_set_text (NBTK_LABEL (priv->active_audio_title), "");
        }
    }
}

BklItem     *
hrn_queue_get_active_audio (HrnQueue *queue)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);

  return priv->active_audio;
}


void
hrn_queue_hide_video_progress (HrnQueue *queue)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);

  if (priv->timebar)
    {
      clutter_actor_set_height (CLUTTER_ACTOR (priv->timebar), 0);
      clutter_actor_set_opacity (CLUTTER_ACTOR (priv->timebar), 0);
    }
}

void
hrn_queue_show_video_progress (HrnQueue *queue)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);

  clutter_actor_set_opacity (CLUTTER_ACTOR (priv->timebar), 255);
  clutter_actor_set_height (CLUTTER_ACTOR (priv->timebar), 20);
}


void
hrn_queue_hide_audio_progress (HrnQueue *queue)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);

  if (priv->timebar)
    {
      clutter_actor_set_height (CLUTTER_ACTOR (priv->audio_timebar), 0);
      clutter_actor_set_opacity (CLUTTER_ACTOR (priv->audio_timebar), 0);
      clutter_actor_set_opacity (CLUTTER_ACTOR (priv->active_audio_title), 0);
    }
}

void
hrn_queue_show_audio_progress (HrnQueue *queue)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);

  clutter_actor_set_opacity (CLUTTER_ACTOR (priv->audio_timebar), 255);
  clutter_actor_set_opacity (CLUTTER_ACTOR (priv->active_audio_title), 255);
  clutter_actor_set_height (CLUTTER_ACTOR (priv->audio_timebar), 20);
}

void
hrn_queue_audio_finished (HrnQueue *queue)
{
  hrn_queue_set_active_audio (queue, NULL, "");
  hrn_queue_hide_audio_progress (queue);
}


/* on top of the audio queue the video queue exist, it pauses the audio
 * queue on demand
 */
void
hrn_queue_video_finished (HrnQueue *queue)
{
  hrn_queue_set_active_visual (queue, NULL, "");
  hrn_queue_hide_video_progress (queue);
}

void
hrn_queue_image_finished (HrnQueue *queue)
{
  hrn_queue_set_active_visual (queue, NULL, "");
}

static gchar *hrn_guess_mimetype (const gchar *uri)
{
  GFile *file;
  GFileInfo *info;
  gchar *mimetype = "unknown/unknown";

  file = g_file_new_for_commandline_arg (uri);
  info = g_file_query_info (file, "standard::*", G_FILE_QUERY_INFO_NONE,
                            NULL, NULL);
  if (info)
    mimetype = (void*)g_file_info_get_content_type (info);

  if (!info || 
      !(g_str_has_prefix (mimetype, "image") ||
        g_str_has_prefix (mimetype, "video") ||
        g_str_has_prefix (mimetype, "audio")))
    {
      if (g_str_has_suffix (uri, "png")||
          g_str_has_suffix (uri, "PNG")||
          g_str_has_suffix (uri, "jpg")||
          g_str_has_suffix (uri, "JPG"))
        {
          mimetype = "image/foo";
        }
      else if (g_str_has_suffix (uri, "ogg")||
               g_str_has_suffix (uri, "OGG")||
               g_str_has_suffix (uri, "mp3")||
               g_str_has_suffix (uri, "MP3"))
        {
          mimetype = "audio/foo";
        }
      else if (g_str_has_suffix (uri, "mpg")||
               g_str_has_suffix (uri, "MPG")||
               g_str_has_suffix (uri, "asf")||
               g_str_has_suffix (uri, "avi")||
               g_str_has_suffix (uri, "AVI")||
               g_str_has_suffix (uri, "ogv")||
               g_str_has_suffix (uri, "OGV")||
               g_str_has_suffix (uri, "MP4")||
               g_str_has_suffix (uri, "mp4"))
        {
          mimetype = "video/foo";
        }
    }
  mimetype = g_strdup (mimetype);
 
  if (info)
    g_object_unref (info);
  g_object_unref (file);

  return mimetype;
}

/* do an best effort to retrieve a mimetype for uri, either through bickley or gio,
 * ultimately falling back to extension based detection.
 */
gchar *hrn_resolve_mimetype (const gchar *uri)
{
  gchar *mimetype;
  BklItem *item = hrn_get_item_for_uri (uri);

  if (item)
    {
      mimetype = g_strdup (bkl_item_get_mimetype (item));
    }
  else
    mimetype = hrn_guess_mimetype (uri);

  return mimetype;
}

void
hrn_queue_play_uri_now (HrnQueue    *queue,
                        const gchar *uri)
{
  GFile *file;
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);
  gchar *mimetype = NULL;

  file = g_file_new_for_commandline_arg (uri);

  mimetype = hrn_resolve_mimetype (uri);
  br_queue_play_uri (priv->br_queue, g_file_get_uri (file), mimetype);

  g_free (mimetype);
  g_object_unref (file);
}

void
hrn_queue_play_now (HrnQueue *queue,
                    BklItem  *item)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);

  br_queue_play_uri (priv->br_queue, bkl_item_get_uri (item),
                     bkl_item_get_mimetype (item));
}
