/*
 * Hornsey - Moblin Media Player.
 * Copyright © 2009 Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */


#define ROW_OFFSET    5
#define ROW_HEIGHT    40

#include <clutter/clutter.h>
#include <math.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include "hrn.h"
#include "hrn-controls.h"

#define    QUEUE_TIP_HEIGHT    47  /* used to determine drops on tip */

G_DEFINE_TYPE (HrnControls, hrn_controls, NBTK_TYPE_WIDGET);

#define HRN_CONTROLS_GET_PRIVATE(obj)                 \
  (G_TYPE_INSTANCE_GET_PRIVATE ((obj), \
                                HRN_TYPE_CONTROLS, \
                                HrnControlsPrivate))

struct _HrnControlsPrivate
{
  BrQueue *br_queue;

  NbtkWidget     *header;
  NbtkWidget     *view;
  NbtkWidget     *scrollview;
  NbtkWidget     *ffw;
  NbtkWidget     *playpause;
  NbtkWidget     *leave;
  NbtkWidget     *timebar;
  NbtkAdjustment *progress;

  NbtkWidget *highlight;

  NbtkWidget     *audio_timebar;
  NbtkAdjustment *audio_progress;

  guint32 visual_progress_id;
  guint32 audio_progress_id;
  guint32 play_pause_id;

  gint i;

  NbtkWidget *name;     /* we store the string in the actor, and
                         * don't keep it ourselves
                         */

  BklItem    *active_visual;
  NbtkWidget *active_title;

  /* keep audio in a separate playqueue,
   * that can proceed in paralell with images
   */
  BklItem    *active_audio;
  NbtkWidget *active_audio_title;

  /* FIXME: Queue needs to be a GQueue */
  GList *queue;

  gint n;
};

typedef struct QueueItem
{
  HrnControls     *queue;
  BklItem      *item;
  ClutterActor *actor;
}
QueueItem;

static GObject * hrn_controls_constructor (GType                  type,
                                        guint                  n_params,
                                        GObjectConstructParam *params);
static void hrn_controls_dispose          (GObject               *object);
static void hrn_controls_allocate         (ClutterActor          *self,
                                        const ClutterActorBox *box,
                                        ClutterAllocationFlags flags);
static void hrn_controls_paint            (ClutterActor          *actor);
static void hrn_controls_pick             (ClutterActor          *actor,
                                        const ClutterColor    *color);
static void hrn_controls_map              (ClutterActor          *self)
{
  HrnControlsPrivate *priv = HRN_CONTROLS (self)->priv;

  CLUTTER_ACTOR_CLASS (hrn_controls_parent_class)->map (self);

  if (priv->header)
    clutter_actor_map (CLUTTER_ACTOR (priv->header));
}

static void
hrn_controls_unmap (ClutterActor *self)
{
  HrnControlsPrivate *priv = HRN_CONTROLS (self)->priv;

  CLUTTER_ACTOR_CLASS (hrn_controls_parent_class)->unmap (self);

  if (priv->header)
    clutter_actor_unmap (CLUTTER_ACTOR (priv->header));
}

static void
hrn_controls_class_init (HrnControlsClass *klass)
{
  GObjectClass      *gobject_class = G_OBJECT_CLASS (klass);
  ClutterActorClass *actor_class   = CLUTTER_ACTOR_CLASS (klass);

  gobject_class->dispose      = hrn_controls_dispose;
  gobject_class->constructor  = hrn_controls_constructor;
  actor_class->allocate       = hrn_controls_allocate;
  actor_class->paint          = hrn_controls_paint;
  actor_class->pick           = hrn_controls_pick;
  actor_class->map            = hrn_controls_map;
  actor_class->unmap          = hrn_controls_unmap;

  g_type_class_add_private (gobject_class, sizeof (HrnControlsPrivate));
}

static void
hrn_controls_init (HrnControls *self)
{
  self->priv = HRN_CONTROLS_GET_PRIVATE (self);
  memset (self->priv, 0, sizeof (self->priv));
}

static gboolean
leave_clicked_cb (NbtkButton *button, gpointer queue)
{
  hrn_set_zoom (hrn_get_zoom () / 2.0);
  return TRUE;
}

static gboolean
playpause_clicked_cb (NbtkButton  *button,
                      HrnControls *queue)
{
  hrn_controls_set_playing (queue, !nbtk_button_get_checked (button));
  return TRUE;
}


static gboolean
ffw_clicked_cb (NbtkButton   *button,
                HrnControls  *controls)
{
  HrnControlsPrivate *priv = HRN_CONTROLS (controls)->priv;
  br_queue_next (priv->br_queue);
  return TRUE;
}

#define QUEUE_WIDTH         300
#define QUEUE_WIDTH2        310 /* including padding */
#define QUEUE_ITEM_WIDTH    QUEUE_WIDTH - 23 /* minus interior padding to avoid
                                                horizontal scroll */

static void
visual_progress_notify (NbtkAdjustment *adjustment, GParamSpec     *pspec,
                        HrnControls       *queue)
{
  HrnControlsPrivate *priv = HRN_CONTROLS_GET_PRIVATE (queue);
  double              pos  = nbtk_adjustment_get_value (adjustment);

  br_queue_set_position (priv->br_queue, BR_QUEUE_VISUAL_TYPE, pos);
}

static void
audio_progress_notify (NbtkAdjustment *adjustment, GParamSpec     *pspec,
                       HrnControls       *queue)
{
  HrnControlsPrivate *priv = HRN_CONTROLS_GET_PRIVATE (queue);
  double              pos  = nbtk_adjustment_get_value (adjustment);

  br_queue_set_position (priv->br_queue, BR_QUEUE_AUDIO_TYPE, pos);
}


static gboolean
controls_enter (ClutterActor *actor, ClutterEvent *event, gpointer userdata)
{
  hrn_controls_can_has_fade = FALSE;
  return FALSE;
}

static gboolean
controls_leave (ClutterActor *actor, ClutterEvent *event, gpointer userdata)
{
  hrn_controls_can_has_fade = TRUE;
  return FALSE;
}

static GObject *
hrn_controls_constructor (GType type, guint n_params,
                       GObjectConstructParam *params)
{
  HrnControlsPrivate *priv;
  GObject            *object;
  HrnControls        *queue;

  object = G_OBJECT_CLASS (hrn_controls_parent_class)->constructor (
    type, n_params, params);

  queue = HRN_CONTROLS (object);
  priv  = HRN_CONTROLS_GET_PRIVATE (queue);

  priv->header   = nbtk_table_new ();
  nbtk_widget_set_style_class_name (priv->header, "HrnControls");
  clutter_actor_set_parent (CLUTTER_ACTOR (priv->header),
                            CLUTTER_ACTOR (queue));
  g_object_set_data (object, "HRN_DROP_MASK",
                     GINT_TO_POINTER (HRN_DROP_MASK_QUEUE));

  priv->queue     = NULL;
  priv->ffw       = nbtk_button_new ();
  priv->playpause = nbtk_button_new ();

  nbtk_button_set_toggle_mode (NBTK_BUTTON (priv->playpause), TRUE);
  nbtk_button_set_checked (NBTK_BUTTON (priv->playpause), TRUE);
  nbtk_widget_set_style_class_name (priv->playpause, "HrnControlsPlayPause");
  nbtk_widget_set_style_class_name (priv->ffw, "HrnControlsFfw");

  clutter_actor_set_size (CLUTTER_ACTOR (priv->playpause), 60, 60);
  clutter_actor_set_size (CLUTTER_ACTOR (priv->ffw), 60, 60);

  g_signal_connect (priv->ffw, "clicked", G_CALLBACK (ffw_clicked_cb), queue);

  priv->play_pause_id = g_signal_connect (priv->playpause, "clicked",
                                          G_CALLBACK (playpause_clicked_cb),
                                          queue);

  priv->leave = nbtk_button_new ();

  nbtk_widget_set_style_class_name (priv->leave, "HrnControlsLeave");

  clutter_actor_set_size (CLUTTER_ACTOR (priv->leave), 60, 60);
  g_signal_connect (priv->leave, "clicked", G_CALLBACK (
                      leave_clicked_cb), queue);

  priv->active_title       = nbtk_label_new ("");
  priv->active_audio_title = nbtk_label_new ("");

  priv->name = nbtk_label_new ("unnamed");

  priv->progress           = nbtk_adjustment_new (0.0, 0.0, 1.0, 0.0, 0.0, 0.0);
  priv->visual_progress_id = g_signal_connect
                         (priv->progress, "notify::value",
                         G_CALLBACK (visual_progress_notify), queue);

  priv->timebar = NBTK_WIDGET (nbtk_scroll_bar_new (priv->progress));

  priv->audio_progress    = nbtk_adjustment_new (0.0, 0.0, 1.0, 0.0, 0.0, 0.0);
  priv->audio_progress_id = g_signal_connect
                         (priv->audio_progress, "notify::value",
                         G_CALLBACK (audio_progress_notify), queue);

  priv->audio_timebar = NBTK_WIDGET (nbtk_scroll_bar_new (priv->audio_progress));

  nbtk_widget_set_style_class_name (priv->timebar, "HrnProgressBar");
  nbtk_widget_set_style_class_name (priv->audio_timebar, "HrnProgressBar");

  priv->scrollview = (NbtkWidget *) nbtk_scroll_view_new ();

  nbtk_widget_set_style_class_name (priv->scrollview, "HrnControlsView");

  nbtk_table_add_actor_with_properties (NBTK_TABLE (priv->header),
                                        CLUTTER_ACTOR (priv->leave), 0, 0,
                                        "row-span", 5,
                                        "col-span", 1,
                                        "x-align", 0.0,
                                        "y-align", 0.0,
                                        "x-fill", TRUE,
                                        "x-expand", FALSE,
                                        "y-fill", TRUE,
                                        "y-expand", TRUE,
                                        NULL);


  nbtk_table_add_actor_with_properties (NBTK_TABLE (priv->header),
                                        CLUTTER_ACTOR (priv->ffw), 0, 2,
                                        "row-span", 5,
                                        "col-span", 1,
                                        "x-align", 0.0,
                                        "y-align", 0.0,
                                        "y-fill", TRUE,
                                        "y-expand", TRUE,
                                        "x-fill", TRUE,
                                        "x-expand", FALSE,
                                        NULL);

  nbtk_table_add_actor_with_properties (NBTK_TABLE (priv->header),
                                        CLUTTER_ACTOR (priv->playpause), 0, 3,
                                        "row-span", 5,
                                        "col-span", 1,
                                        "x-align", 0.0,
                                        "y-align", 0.0,
                                        "y-fill", TRUE,
                                        "y-expand", TRUE,
                                        "x-fill", TRUE,
                                        "x-expand", FALSE,
                                        NULL);

  nbtk_table_add_actor_with_properties (NBTK_TABLE (priv->header),
                                        CLUTTER_ACTOR (
                                          priv->active_title), 1, 1,
                                        "row-span", 1,
                                        "col-span", 1,
                                        "x-fill", TRUE,
                                        "x-expand", TRUE,
                                        "y-fill", FALSE,
                                        "y-expand", FALSE,
                                        "x-align", 0.0,
                                        "y-align", 0.0,
                                        NULL);

  nbtk_table_add_actor_with_properties (NBTK_TABLE (priv->header),
                                        CLUTTER_ACTOR (
                                          priv->active_audio_title), 2, 1,
                                        "row-span", 1,
                                        "col-span", 1,
                                        "x-fill", TRUE,
                                        "x-expand", TRUE,
                                        "y-fill", FALSE,
                                        "y-expand", FALSE,
                                        "x-align", 0.0,
                                        "y-align", 0.0,
                                        NULL);

  nbtk_table_add_actor_with_properties (NBTK_TABLE (priv->header),
                                        CLUTTER_ACTOR (priv->timebar), 3, 1,
                                        "row-span", 1,
                                        "col-span", 1,
                                        "x-fill", TRUE,
                                        "y-fill", FALSE,
                                        "x-expand", TRUE,
                                        "y-expand", FALSE,
                                        "x-align", 0.0,
                                        "y-align", 0.0,
                                        NULL);

  nbtk_table_add_actor_with_properties (NBTK_TABLE (priv->header),
                                        CLUTTER_ACTOR (
                                          priv->audio_timebar), 4, 1,
                                        "row-span", 1,
                                        "col-span", 1,
                                        "x-fill", TRUE,
                                        "y-fill", FALSE,
                                        "x-expand", TRUE,
                                        "y-expand", FALSE,
                                        "x-align", 0.0,
                                        "y-align", 0.0,
                                        NULL);

  clutter_actor_set_reactive (CLUTTER_ACTOR (object), TRUE);
  g_signal_connect (object, "enter-event", G_CALLBACK (controls_enter),
                    NULL);
  g_signal_connect (object, "leave-event", G_CALLBACK (controls_leave),
                    NULL);

  priv->view = (NbtkWidget *) nbtk_viewport_new ();
  clutter_container_add (CLUTTER_CONTAINER (priv->scrollview),
                         CLUTTER_ACTOR (priv->view), NULL);


  hrn_controls_hide_video_progress (queue);
  hrn_controls_hide_audio_progress (queue);
  
  clutter_actor_set_size (CLUTTER_ACTOR (object), clutter_actor_get_width(clutter_stage_get_default()), 60);
  return object;
}

static void
hrn_controls_allocate (ClutterActor *self, const ClutterActorBox *box,
                    ClutterAllocationFlags flags)
{
  HrnControlsPrivate   *priv = HRN_CONTROLS_GET_PRIVATE (self);
  ClutterActorClass *parent_class;
  ClutterActorBox    actor_box;

  parent_class = CLUTTER_ACTOR_CLASS (hrn_controls_parent_class);

  parent_class->allocate (self, box, flags);
  actor_box    = *box;
  actor_box.y2 = box->y2 - box->y1;
  actor_box.y1 = 0;

  clutter_actor_allocate (CLUTTER_ACTOR (priv->header), &actor_box, flags);
}

static void
hrn_controls_dispose (GObject *object)
{
  G_OBJECT_CLASS (hrn_controls_parent_class)->dispose (object);
}

static void
hrn_controls_paint (ClutterActor *actor)
{
  HrnControls        *queue = HRN_CONTROLS (actor);
  HrnControlsPrivate *priv  = HRN_CONTROLS_GET_PRIVATE (queue);

  /*CLUTTER_ACTOR_CLASS (hrn_controls_parent_class)->paint (actor);*/

  clutter_actor_paint (CLUTTER_ACTOR (priv->header));
}

#include <cogl/cogl.h>

static void
hrn_controls_pick (ClutterActor *actor, const ClutterColor   *color)
{
  HrnControlsPrivate *priv = HRN_CONTROLS_GET_PRIVATE (actor);
  gfloat           width;
  gfloat           height;

  clutter_actor_get_size (CLUTTER_ACTOR (priv->header), &width, &height);

  cogl_set_source_color4ub (color->red,
                            color->green,
                            color->blue,
                            color->alpha);
  cogl_rectangle (0, 0, width, height);

  hrn_controls_paint (actor);
}

static void
brq_position_changed (BrQueue *brq, BrQueueType type, double position,
                      HrnControls   *queue)
{
  HrnControlsPrivate *priv = HRN_CONTROLS_GET_PRIVATE (queue);

  switch (type)
    {
      case BR_QUEUE_VISUAL_TYPE:
        g_signal_handler_block (priv->progress, priv->visual_progress_id);
        nbtk_adjustment_set_value (priv->progress, position);
        g_signal_handler_unblock (priv->progress, priv->visual_progress_id);
        break;

      case BR_QUEUE_AUDIO_TYPE:
        g_signal_handler_block (priv->audio_progress, priv->audio_progress_id);
        nbtk_adjustment_set_value (priv->audio_progress, position);
        g_signal_handler_unblock (priv->audio_progress, priv->audio_progress_id);
        break;
    }
}

static void
brq_now_playing_changed (BrQueue     *brq,
                         const char  *uri, 
                         BrQueueType  type,
                         HrnControls *queue)
{
  BklItem *item;

  item = hrn_get_item_for_uri (uri);

  /* Reset the position of the appropriate slider to 0 */
  brq_position_changed (brq, type, 0.0, queue);

  if (uri == NULL || uri[0]==0) /* finished playback of this media type */
    {
      switch (type)
        {
           case BR_QUEUE_VISUAL_TYPE:
             hrn_controls_hide_video_progress (queue);
             hrn_controls_set_active_visual (queue, NULL, "");
             break;
           case BR_QUEUE_AUDIO_TYPE:
             hrn_controls_hide_audio_progress (queue);
             hrn_controls_set_active_audio (queue, NULL, "");
             break;
        }
      return;
    }

  switch (type)
    {
      gboolean is_video;
      case BR_QUEUE_VISUAL_TYPE:
        is_video = FALSE;
        if (item)
          {
            if (bkl_item_get_item_type (item) == BKL_ITEM_TYPE_VIDEO)
              is_video = TRUE;
          }
        else
          {
            gchar *mimetype = hrn_resolve_mimetype (uri);
            if (g_str_has_prefix (mimetype, "video"))
              is_video = TRUE;
            g_free (mimetype);
          }
        
        hrn_controls_set_active_visual (queue, item, uri);

        if (is_video)
          {
            hrn_controls_show_video_progress (queue);
            hrn_controls_hide_audio_progress (queue);
          }
        else
          {
            hrn_controls_hide_video_progress (queue);
            hrn_controls_hide_audio_progress (queue);
          }
        break;

      case BR_QUEUE_AUDIO_TYPE:
        hrn_controls_set_active_audio (queue, item, uri);
        hrn_controls_show_audio_progress (queue);
        hrn_controls_hide_video_progress (queue);
        break;
    }
}

static void
brq_playing_changed (BrQueue  *brq,
                     gboolean  playing,
                     HrnControls *queue)
{
  HrnControlsPrivate *priv  = HRN_CONTROLS_GET_PRIVATE (queue);

  g_signal_handler_block (priv->playpause, priv->play_pause_id);
  nbtk_button_set_checked ((NbtkButton *) priv->playpause, !playing);
  g_signal_handler_unblock (priv->playpause, priv->play_pause_id);
}

static void
set_name_reply (BrQueue *brq, char    *name, GError  *error, gpointer data)
{
  HrnControls        *queue = (HrnControls *) data;
  HrnControlsPrivate *priv  = HRN_CONTROLS_GET_PRIVATE (queue);

  if (error != NULL)
    {
      g_warning ("Error getting queue name: %s", error->message);
      return;
    }

  nbtk_label_set_text ((NbtkLabel *) priv->name, name);
}

HrnControls *
hrn_controls_new (BrQueue *br_queue)
{
  HrnControls        *queue = g_object_new (HRN_TYPE_CONTROLS, NULL);
  HrnControlsPrivate *priv  = HRN_CONTROLS_GET_PRIVATE (queue);

  priv->br_queue = g_object_ref (br_queue);
  g_signal_connect (priv->br_queue, "now-playing-changed",
                    G_CALLBACK (brq_now_playing_changed), queue);
  g_signal_connect (priv->br_queue, "position-changed",
                    G_CALLBACK (brq_position_changed), queue);
  g_signal_connect (priv->br_queue, "playing-changed",
                    G_CALLBACK (brq_playing_changed), queue);

  /* Set the name of the queue from Bognor */
  br_queue_get_name (br_queue, set_name_reply, queue);

  return queue;
}

static void
get_now_playing_reply (BrQueue *brq,
                       char    *audio_uri,
                       char    *visual_uri,
                       GError  *error,
                       gpointer data)
{
    HrnControls *queue = (HrnControls *) data;

    if (error != NULL) {
        g_warning ("Error getting now playing: %s", error->message);
        return;
    }

    brq_now_playing_changed (brq, audio_uri, BR_QUEUE_AUDIO_TYPE, queue);
    brq_now_playing_changed (brq, visual_uri, BR_QUEUE_VISUAL_TYPE, queue);

#if 0
    if (visual_uri) {
        BklItem *item;

        item = hrn_get_item_for_uri (visual_uri);
        if (item) {
            hrn_controls_play_now (queue, item);
            br_queue_stop (brq);
        }
    }
#endif
}

static void
get_playing_reply (BrQueue *brq,
                   gboolean playing,
                   GError  *error,
                   gpointer data)
{
  HrnControls *queue = (HrnControls *) data;

  if (error != NULL) {
    g_warning ("Error getting initial play state: %s", error->message);
    return;
  }

  brq_playing_changed (brq, playing, queue);
}

void
hrn_controls_populate (HrnControls *queue)
{
  HrnControlsPrivate *priv = HRN_CONTROLS_GET_PRIVATE (queue);

  br_queue_get_now_playing (priv->br_queue, get_now_playing_reply, queue);
  br_queue_get_playing (priv->br_queue, get_playing_reply, queue);
}

void
hrn_controls_set_name (HrnControls *queue, const gchar *name)
{
  HrnControlsPrivate *priv = HRN_CONTROLS_GET_PRIVATE (queue);

  nbtk_label_set_text (NBTK_LABEL (priv->name), name);
  nbtk_label_set_text (NBTK_LABEL (priv->name), " ");
}

const gchar *
hrn_controls_get_name (HrnControls *queue)
{
  HrnControlsPrivate *priv = HRN_CONTROLS_GET_PRIVATE (queue);

  return clutter_text_get_text (CLUTTER_TEXT (priv->name));
}



gint
hrn_controls_get_length (HrnControls *queue)
{
  HrnControlsPrivate *priv = HRN_CONTROLS_GET_PRIVATE (queue);

  return g_list_length (priv->queue);
}



void
hrn_controls_remove (HrnControls *queue, gint pos)
{
  HrnControlsPrivate *priv = HRN_CONTROLS_GET_PRIVATE (queue);

  br_queue_remove (priv->br_queue, pos);
}

BklItem *
hrn_controls_get (HrnControls *queue, gint pos)
{
  HrnControlsPrivate *priv = HRN_CONTROLS_GET_PRIVATE (queue);
  QueueItem       *item;

  item = g_list_nth_data (priv->queue, pos);
  if (item)
    {
      return item->item;
    }
  else
    {
      return NULL;
    }
}

BklItem     *
hrn_controls_proceed (HrnControls *queue, gint item_type_mask)
{
  gint len = hrn_controls_get_length (queue);

  if (len)
    {
      gint i;
      for (i = 0; i < len; i++)
        {
          BklItem *ret = hrn_controls_get (queue, i);
          if (bkl_item_get_item_type (ret) & item_type_mask)
            {
              hrn_controls_remove (queue, i);
              return ret;
            }
        }
    }
  return NULL;
}

gboolean
hrn_controls_get_playing (HrnControls *queue)
{
  HrnControlsPrivate *priv = HRN_CONTROLS_GET_PRIVATE (queue);

  return nbtk_button_get_checked (NBTK_BUTTON (priv->playpause));
}

void
hrn_controls_set_playing (HrnControls *queue, gboolean playing)
{
  HrnControlsPrivate *priv = HRN_CONTROLS_GET_PRIVATE (queue);
  GList           *l;

  /* Check the queue for visual items, to see whether we need to go to
     theatre mode */
  for (l = priv->queue; l; l = l->next)
    {
      QueueItem  *item = l->data;
      BklItem    *bi   = item->item;
      const char *mimetype;

      if (bi == NULL)
        {
          continue;
        }

      mimetype = bkl_item_get_mimetype (bi);
      if (mimetype == NULL)
        {
          continue;
        }

      if (g_str_has_prefix (mimetype, "video/") ||
          g_str_has_prefix (mimetype, "image/"))
        {
          hrn_set_theatre ();
          break;
        }
    }

  if (playing)
    {
      /* We want to turn the slideshow back on when the queue is started */
      /* FIXME: Stop using global variables... */
      hrn_theatre_set_image_slideshow ((HrnTheatre *) hrn_theatre, TRUE);
      br_queue_play (priv->br_queue);
    }
  else
    {
      br_queue_stop (priv->br_queue);
    }
}


void
hrn_controls_set_active_visual (HrnControls *queue,
                                BklItem     *item,
                                const gchar *uri)
{
  HrnControlsPrivate *priv = HRN_CONTROLS_GET_PRIVATE (queue);

  if (priv->active_visual)
    g_object_unref (priv->active_visual);
  priv->active_visual = NULL;

  if (item)
    {
      gchar *title;
      priv->active_visual = g_object_ref (item);
      title               = hrn_item_get_title (item);
      nbtk_label_set_text (NBTK_LABEL (priv->active_title), title);
      g_free (title);
    }
  else
    {
      if (uri)
        {
          GFile *file;
          gchar *basename;
          file = g_file_new_for_uri (uri);
          basename = g_file_get_basename (file);
          nbtk_label_set_text (NBTK_LABEL (priv->active_title), basename);
          if (basename)
            g_free (basename);
          g_object_unref (file);
        }
      else
        {
          nbtk_label_set_text (NBTK_LABEL (priv->active_title), "");
        }
    }
}

BklItem     *
hrn_controls_get_active_visual (HrnControls *queue)
{
  HrnControlsPrivate *priv = HRN_CONTROLS_GET_PRIVATE (queue);

  return priv->active_visual;
}


void
hrn_controls_set_active_audio (HrnControls *queue,
                               BklItem     *item,
                               const gchar *uri)
{
  HrnControlsPrivate *priv = HRN_CONTROLS_GET_PRIVATE (queue);

  if (priv->active_audio)
    g_object_unref (priv->active_audio);
  priv->active_audio = NULL;

  if (item)
    {
      gchar *title;
      priv->active_audio = g_object_ref (item);
      title              = hrn_item_get_title (item);
      nbtk_label_set_text (NBTK_LABEL (priv->active_audio_title), title);
      g_free (title);
    }
  else
    {
      if (uri)
        {
          GFile *file;
          gchar *basename;
          file = g_file_new_for_uri (uri);
          basename = g_file_get_basename (file);
          nbtk_label_set_text (NBTK_LABEL (priv->active_audio_title), basename);
          if (basename)
            g_free (basename);
          g_object_unref (file);
        }
      else
        {
          nbtk_label_set_text (NBTK_LABEL (priv->active_audio_title), "");
        }
    }
}

BklItem     *
hrn_controls_get_active_audio (HrnControls *queue)
{
  HrnControlsPrivate *priv = HRN_CONTROLS_GET_PRIVATE (queue);

  return priv->active_audio;
}


void
hrn_controls_hide_video_progress (HrnControls *queue)
{
  HrnControlsPrivate *priv = HRN_CONTROLS_GET_PRIVATE (queue);

  if (priv->timebar)
    {
      clutter_actor_set_height (CLUTTER_ACTOR (priv->timebar), 0);
      clutter_actor_set_opacity (CLUTTER_ACTOR (priv->timebar), 0);
    }
}

void
hrn_controls_show_video_progress (HrnControls *queue)
{
  HrnControlsPrivate *priv = HRN_CONTROLS_GET_PRIVATE (queue);

  clutter_actor_set_opacity (CLUTTER_ACTOR (priv->timebar), 255);
  clutter_actor_set_height (CLUTTER_ACTOR (priv->timebar), 20);
}


void
hrn_controls_hide_audio_progress (HrnControls *queue)
{
  HrnControlsPrivate *priv = HRN_CONTROLS_GET_PRIVATE (queue);

  if (priv->timebar)
    {
      clutter_actor_set_height (CLUTTER_ACTOR (priv->audio_timebar), 0);
      clutter_actor_set_opacity (CLUTTER_ACTOR (priv->audio_timebar), 0);
      clutter_actor_set_opacity (CLUTTER_ACTOR (priv->active_audio_title), 0);
    }
}

void
hrn_controls_show_audio_progress (HrnControls *queue)
{
  HrnControlsPrivate *priv = HRN_CONTROLS_GET_PRIVATE (queue);

  clutter_actor_set_opacity (CLUTTER_ACTOR (priv->audio_timebar), 255);
  clutter_actor_set_opacity (CLUTTER_ACTOR (priv->active_audio_title), 255);
  clutter_actor_set_height (CLUTTER_ACTOR (priv->audio_timebar), 20);
}

void
hrn_controls_audio_finished (HrnControls *queue)
{
  hrn_controls_set_active_audio (queue, NULL, "");
  hrn_controls_hide_audio_progress (queue);
}

/* on top of the audio queue the video queue exist, it pauses the audio
 * queue on demand
 */
void
hrn_controls_video_finished (HrnControls *queue)
{
  hrn_controls_set_active_visual (queue, NULL, "");
  hrn_controls_hide_video_progress (queue);
}

void
hrn_controls_image_finished (HrnControls *queue)
{
  hrn_controls_set_active_visual (queue, NULL, "");
}

void
hrn_controls_play_uri_now (HrnControls    *queue,
                        const gchar *uri)
{
  GFile *file;
  HrnControlsPrivate *priv = HRN_CONTROLS_GET_PRIVATE (queue);
  gchar *mimetype = NULL;

  file = g_file_new_for_commandline_arg (uri);

  mimetype = hrn_resolve_mimetype (uri);
  br_queue_play_uri (priv->br_queue, g_file_get_uri (file), mimetype);

  g_free (mimetype);
  g_object_unref (file);
}

void
hrn_controls_play_now (HrnControls *queue,
                       BklItem  *item)
{
  HrnControlsPrivate *priv = HRN_CONTROLS_GET_PRIVATE (queue);

  br_queue_play_uri (priv->br_queue, bkl_item_get_uri (item),
                     bkl_item_get_mimetype (item));
}
