/*
 * Hornsey - Moblin Media Player.
 * Copyright © 2009 Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <glib.h>
#include <string.h>
#include "hrn.h"
#include "math.h"
#include "hrn-cluster.h"
#include <glib/gprintf.h>

G_DEFINE_TYPE (HrnCluster, hrn_cluster, HRN_TYPE_SWITCHER);

#define HRN_CLUSTER_GET_PRIVATE(obj)                 \
  (G_TYPE_INSTANCE_GET_PRIVATE ((obj),               \
                                HRN_TYPE_CLUSTER,    \
                                HrnClusterPrivate))

/* evil hack, a direct cogl handle for the fronts of items in the stack */
static CoglHandle front_material = 0;
static CoglHandle front_handle   = 0;

struct _HrnClusterPrivate
{
  ClutterActor *frame_bg;
  ClutterActor *frame;
  GList        *items; /* have items be in a list, show clones of the
                        * actors in this list.
                        *
                        * This shold probably be implemented similarily to the
                        *texture
                        * cache with weak references.
                        */
  gint cells;
  BklItemType  type_mask;  /* A bitmask of the types of items that have
                              been added/observed for this cluster */
};
gchar *     hrn_cluster_get_title (HrnCluster *cluster);

static GObject *
            hrn_cluster_constructor (GType type, guint n_params,
                         GObjectConstructParam *params);
static void
            hrn_cluster_dispose (GObject *object);

#if HRN_USE_NORMAL_LABELS
static void ensure_labels (HrnCluster *cluster, gint no);
#endif

static void
hrn_cluster_allocate (ClutterActor *self, const ClutterActorBox *box,
                      ClutterAllocationFlags flags)
{
  ClutterActorClass *parent_class;

  parent_class = CLUTTER_ACTOR_CLASS (hrn_cluster_parent_class);

  parent_class->allocate (self, box, flags);

  if (hrn_switcher_get_state (HRN_SWITCHER (self)) > 0)
    {
      switch (hrn_view_labels_visible (hrn_item_get_view (self)))
        {
          case ZOOM_ITEMS_8:
            ensure_labels (HRN_CLUSTER (self), 1);
            break;

          case ZOOM_ITEMS_4:
          case ZOOM_ITEMS_3:
            ensure_labels (HRN_CLUSTER (self), 2);
            ensure_labels (HRN_CLUSTER (self), 2);
            break;
        }
    }

  if (CLUTTER_ACTOR (HRN_CLUSTER (self)->priv->frame))
    clutter_actor_allocate_preferred_size (CLUTTER_ACTOR (HRN_CLUSTER (self)->
                                                          priv->frame), flags);


  if (CLUTTER_ACTOR (HRN_CLUSTER (self)->priv->frame_bg))
    clutter_actor_allocate_preferred_size (CLUTTER_ACTOR (HRN_CLUSTER (
                                                            self)->
                                                          priv->frame_bg),
                                           flags);

#if HRN_USE_NORMAL_LABELS
  if (HRN_CLUSTER (self)->title)
    {
      clutter_actor_allocate_preferred_size (CLUTTER_ACTOR (HRN_CLUSTER (self)
                                                            ->title), flags);
      clutter_actor_allocate_preferred_size (CLUTTER_ACTOR (HRN_CLUSTER (self)
                                                            ->meta), flags);
    }
  if (HRN_CLUSTER (self)->title2)
    {
      clutter_actor_allocate_preferred_size (CLUTTER_ACTOR (HRN_CLUSTER (self)
                                                            ->title2), flags);
      clutter_actor_allocate_preferred_size (CLUTTER_ACTOR (HRN_CLUSTER (self)
                                                            ->meta2), flags);
    }
#endif
}

gchar *hrn_cluster_get_meta (HrnCluster *cluster);

static gboolean in_pick = FALSE;

static void
hrn_cluster_pre_paint (ClutterActor *actor)
{
  HrnCluster *cluster;

  cluster = HRN_CLUSTER (actor);

  switch (hrn_view_labels_visible (hrn_item_get_view (actor)))
    {
      case ZOOM_CLUSTERS_GROUPED:
      case ZOOM_CLUSTERS:
        break;

      default: return;
    }

  if (!in_pick && cluster->priv->frame_bg)
    clutter_actor_paint (cluster->priv->frame_bg);
}

static void
hrn_cluster_post_paint (ClutterActor *actor)
{
  HrnCluster *cluster;

  cluster = HRN_CLUSTER (actor);

  switch (hrn_view_labels_visible (hrn_item_get_view (actor)))
    {
      case ZOOM_CLUSTERS_GROUPED:
      case ZOOM_CLUSTERS:
        break;

      default: return;
    }
  if (!in_pick && cluster->priv->frame)
    clutter_actor_paint (cluster->priv->frame);
}

static void
hrn_cluster_paint (ClutterActor *actor)
{
  ClutterActorClass *parent_class;
  HrnCluster        *cluster;
  HrnClusterPrivate *priv;
  gdouble            scale;
  gint               items;
  gint               i;

  cluster = HRN_CLUSTER (actor);
  priv = HRN_CLUSTER_GET_PRIVATE (cluster);
  items   = priv->cells;

  if (items > 7)
    items = 7;
  scale = hrn_get_scale ();
  if (!in_pick)
    hrn_cluster_pre_paint (actor);

  cogl_push_matrix ();

  if (hrn_switcher_get_state (HRN_SWITCHER (cluster)) == 1 && !in_pick)
    {
      gfloat factor = (1.0 - scale) * 2;
      cogl_translate (0.0, (-10.0 * items) * factor, 0);
      cogl_set_source (front_material);

      for (i = items - 1; i >= 0; i--)
        {
          cogl_rectangle_with_texture_coords (
            DIM * 0.044,  DIM * 0.975 + 10 * i - 10,
            DIM * 0.88, DIM * 0.975 + 10 * i + 10,

            0.0, 0.0,
            1.0, 1.0);
        }
    }
  parent_class = CLUTTER_ACTOR_CLASS (hrn_cluster_parent_class);

  parent_class->paint (actor);

  cogl_pop_matrix ();
  if (!in_pick)
    hrn_cluster_post_paint (actor);

  if (hrn_switcher_get_state (HRN_SWITCHER (actor)) > 0
#if HRN_HIDE_LABELS_ON_GRAB
      && clutter_get_pointer_grab () == NULL
#endif
      )
    {
#if HRN_USE_NORMAL_LABELS
      if (hrn_view_labels_visible (hrn_item_get_view (actor)) ==
          ZOOM_CLUSTERS_GROUPED)
        {
          ensure_labels (HRN_CLUSTER (cluster), 2);
          clutter_actor_paint (CLUTTER_ACTOR (cluster->title2));
          clutter_actor_paint (CLUTTER_ACTOR (cluster->meta2));
        }
      else if (hrn_view_labels_visible (hrn_item_get_view (actor)) ==
               ZOOM_CLUSTERS)
        {
          ensure_labels (HRN_CLUSTER (cluster), 1);
          clutter_actor_paint (CLUTTER_ACTOR (cluster->title));
          clutter_actor_paint (CLUTTER_ACTOR (cluster->meta));
        }
#endif
    }
}

static void
hrn_cluster_pick (ClutterActor *actor, const ClutterColor *color)
{
  in_pick++;
  hrn_cluster_paint (actor);
  in_pick--;
}

static void
hrn_cluster_map (ClutterActor *self)
{
  HrnCluster        *cluster = HRN_CLUSTER (self);
  HrnClusterPrivate *priv    = cluster->priv;

  CLUTTER_ACTOR_CLASS (hrn_cluster_parent_class)->map (self);

  if (cluster->title)
    clutter_actor_map (CLUTTER_ACTOR (cluster->title));
  if (cluster->meta)
    clutter_actor_map (CLUTTER_ACTOR (cluster->meta));
  if (cluster->title2)
    clutter_actor_map (CLUTTER_ACTOR (cluster->title2));
  if (cluster->meta2)
    clutter_actor_map (CLUTTER_ACTOR (cluster->meta2));
  if (priv->frame)
    clutter_actor_map (priv->frame);
  if (priv->frame_bg)
    clutter_actor_map (priv->frame_bg);
}

static void
hrn_cluster_unmap (ClutterActor *self)
{
  HrnCluster        *cluster = HRN_CLUSTER (self);
  HrnClusterPrivate *priv    = cluster->priv;

  CLUTTER_ACTOR_CLASS (hrn_cluster_parent_class)->unmap (self);

  if (cluster->title)
    clutter_actor_unmap (CLUTTER_ACTOR (cluster->title));
  if (cluster->meta)
    clutter_actor_unmap (CLUTTER_ACTOR (cluster->meta));
  if (cluster->title2)
    clutter_actor_unmap (CLUTTER_ACTOR (cluster->title2));
  if (cluster->meta2)
    clutter_actor_unmap (CLUTTER_ACTOR (cluster->meta2));
  if (priv->frame)
    clutter_actor_unmap (priv->frame);
  if (priv->frame_bg)
    clutter_actor_unmap (priv->frame_bg);
}


static void
hrn_cluster_class_init (HrnClusterClass *klass)
{
  GObjectClass      *gobject_class = G_OBJECT_CLASS (klass);
  ClutterActorClass *actor_class   = CLUTTER_ACTOR_CLASS (klass);

  gobject_class->dispose     = hrn_cluster_dispose;
  gobject_class->constructor = hrn_cluster_constructor;
  actor_class->paint         = hrn_cluster_paint;
  actor_class->pick          = hrn_cluster_pick;
  actor_class->allocate      = hrn_cluster_allocate;
  actor_class->map           = hrn_cluster_map;
  actor_class->unmap         = hrn_cluster_unmap;

  g_type_class_add_private (gobject_class, sizeof (HrnClusterPrivate));
}

static void
hrn_cluster_init (HrnCluster *self)
{
  HrnClusterPrivate *priv = HRN_CLUSTER_GET_PRIVATE (self);

  memset (priv, 0, sizeof (priv));
  self->priv = priv;
}

gchar *
hrn_cluster_get_title (HrnCluster *cluster)
{
  BklItem     *item = NULL;
  gchar        buf[256];
  const gchar *title = NULL;

  if (cluster->results)
    {
      item = cluster->results->data;
    }
  else
    return g_strdup ("foo");

  switch (bkl_item_get_item_type (item))
    {
      case BKL_ITEM_TYPE_VIDEO:
        title = bkl_item_video_get_series_name (BKL_ITEM_VIDEO (item));
        if (!title)
          title = bkl_item_video_get_director (BKL_ITEM_VIDEO (item));
        if (!title)
          title = bkl_item_video_get_title (BKL_ITEM_VIDEO (item));
        if (!title)
          {
            if (bkl_item_video_get_duration (BKL_ITEM_VIDEO (item)))
              {
                g_snprintf (buf, 255, "%2i:%02i",
                            bkl_item_video_get_duration (BKL_ITEM_VIDEO (
                                                           item)) / 60,
                            bkl_item_video_get_duration (BKL_ITEM_VIDEO (
                                                           item)) % 60);
                title = buf;
              }
          }
        break;

      case BKL_ITEM_TYPE_AUDIO:
        if (!title)
          title = bkl_item_audio_get_album (BKL_ITEM_AUDIO (item));
        if (!title)
          {
            GPtrArray *artists =
              bkl_item_audio_get_artists (BKL_ITEM_AUDIO (item));
            if (artists)
              title = artists->pdata[0];
          }
        if (!title)
          title = bkl_item_audio_get_composer (BKL_ITEM_AUDIO (item));
        break;

      case BKL_ITEM_TYPE_IMAGE:
      /*title = bkl_item_image_get_time (BKL_ITEM_IMAGE (item));
         if (!title)
         title = bkl_item_image_get_time_original (BKL_ITEM_IMAGE (item));
         if (!title)
         title = bkl_item_image_get_time_digitized (BKL_ITEM_IMAGE (item));
         break;*/

      default:
        break;
    }
  if (title)
    {
      return g_strdup (title);
    }
  return hrn_item_uri_parent_to_meta (item);
}

/* this should perhaps do different things for image/video/audio */
gchar *
hrn_cluster_get_meta (HrnCluster *cluster)
{
  BklItem     *item = NULL;
  const gchar *meta = NULL;

  if (cluster->results)
    {
      item = cluster->results->data;
    }
  else
    return g_strdup ("-=-");

  switch (bkl_item_get_item_type (item))
    {
      case BKL_ITEM_TYPE_VIDEO:
        meta = bkl_item_video_get_director (BKL_ITEM_VIDEO (item));
        if (!meta)
          meta = bkl_item_video_get_title (BKL_ITEM_VIDEO (item));
        break;

      case BKL_ITEM_TYPE_AUDIO:
        if (!meta)
          {
            GPtrArray *artists =
              bkl_item_audio_get_artists (BKL_ITEM_AUDIO (item));
            if (artists)
              meta = artists->pdata[0];
          }
        if (!meta)
          meta = bkl_item_audio_get_composer (BKL_ITEM_AUDIO (item));
        break;

      case BKL_ITEM_TYPE_IMAGE:
        meta = bkl_item_image_get_time (BKL_ITEM_IMAGE (item));
        /* FIXME: use location */
        if (!meta)
          meta = bkl_item_image_get_time_original (BKL_ITEM_IMAGE (item));
        if (!meta)
          meta = bkl_item_image_get_time_digitized (BKL_ITEM_IMAGE (item));
        break;

      default:
        break;
    }
  if (meta)
    {
      return g_strdup (meta);
    }
  return bkl_item_uri_to_title (item);
}



#if HRN_USE_NORMAL_LABELS

static void
ensure_labels (HrnCluster *cluster, gint no)
{
  if (cluster->title && no == 1)
    return;
  if (cluster->title2 && no > 1)
    return;

  if (hrn_switcher_get_state (HRN_SWITCHER (cluster)) > 3)
    return;

  if (no == 1)
    {
      cluster->title = hrn_button_new ();
      cluster->meta  = hrn_button_new ();


      {
        gchar *buf = hrn_cluster_get_title (cluster);
        nbtk_button_set_label (NBTK_BUTTON (cluster->title), buf);
        g_free (buf);

        buf = hrn_cluster_get_meta ((cluster));
        nbtk_button_set_label (NBTK_BUTTON (cluster->meta), buf);
        g_free (buf);

        nbtk_bin_set_alignment (NBTK_BIN (cluster->title), 0.0, 0.0);
        nbtk_bin_set_alignment (NBTK_BIN (cluster->meta), 0.0, 0.0);
      }



      clutter_actor_set_scale (CLUTTER_ACTOR (cluster->title), 2.0, 2.0);
      clutter_actor_set_scale (CLUTTER_ACTOR (cluster->meta), 2.0, 2.0);
      nbtk_widget_set_style_class_name (cluster->title, "HrnClusterTitle");
      nbtk_widget_set_style_class_name (cluster->meta, "HrnClusterMeta");

      clutter_actor_set_width (CLUTTER_ACTOR (cluster->title), DIM / 2 * 0.9);
      clutter_actor_set_width (CLUTTER_ACTOR (cluster->meta), DIM / 2 * 0.9);

      clutter_actor_set_position (CLUTTER_ACTOR (cluster->title), 0, (gint) DIM);
      clutter_actor_set_position (CLUTTER_ACTOR (
                                    cluster->meta), 0, (gint) DIM +
                                  clutter_actor_get_height (CLUTTER_ACTOR (
                                                              cluster->title))
                                  * 2.2);

      clutter_actor_set_parent (CLUTTER_ACTOR (cluster->title),
                                CLUTTER_ACTOR (cluster));
      clutter_actor_set_parent (CLUTTER_ACTOR (cluster->meta),
                                CLUTTER_ACTOR (cluster));

      /*clutter_actor_set_reactive(CLUTTER_ACTOR (HRN_CLUSTER(cluster)->meta),
         TRUE);*/


      /* we allocate here as well, since these actors might have been added on
       * demand for the paint */
      clutter_actor_allocate_preferred_size (CLUTTER_ACTOR (HRN_CLUSTER (
                                                              cluster)->title),
                                             TRUE);
      clutter_actor_allocate_preferred_size (CLUTTER_ACTOR (HRN_CLUSTER (
                                                              cluster)->meta),
                                             TRUE);
    }
  else
    {
      cluster->title2 = hrn_button_new ();
      cluster->meta2  = hrn_button_new ();

      {
        gchar *buf = hrn_cluster_get_title (cluster);
        nbtk_button_set_label (NBTK_BUTTON (cluster->title2), buf);
        g_free (buf);

        buf = hrn_cluster_get_meta ((cluster));
        nbtk_button_set_label (NBTK_BUTTON (cluster->meta2), buf);
        g_free (buf);

        nbtk_bin_set_alignment (NBTK_BIN (cluster->title2), 0.0, 0.0);
        nbtk_bin_set_alignment (NBTK_BIN (cluster->meta2), 0.0, 0.0);
      }



      clutter_actor_set_scale (CLUTTER_ACTOR (cluster->title2), 1.0, 1.0);
      clutter_actor_set_scale (CLUTTER_ACTOR (cluster->meta2), 1.0, 1.0);
      nbtk_widget_set_style_class_name (cluster->title2,
                                        "HrnClusterGroupedTitle");
      nbtk_widget_set_style_class_name (cluster->meta2, "HrnClusterGroupedMeta");

      clutter_actor_set_width (CLUTTER_ACTOR (cluster->title2), DIM * 0.9);
      clutter_actor_set_width (CLUTTER_ACTOR (cluster->meta2), DIM * 0.9);

      clutter_actor_set_position (CLUTTER_ACTOR (
                                    cluster->title2), 0, (gint) DIM);
      clutter_actor_set_position (CLUTTER_ACTOR (
                                    cluster->meta2), 0, (gint) DIM +
                                  clutter_actor_get_height (CLUTTER_ACTOR (
                                                              cluster->title2))
                                  * 1.1);

      clutter_actor_set_parent (CLUTTER_ACTOR (cluster->title2),
                                CLUTTER_ACTOR (cluster));
      clutter_actor_set_parent (CLUTTER_ACTOR (cluster->meta2),
                                CLUTTER_ACTOR (cluster));

      /*clutter_actor_set_reactive(CLUTTER_ACTOR (HRN_CLUSTER(cluster)->meta),
         TRUE);*/


      /* we allocate here as well, since these actors might have been added on
       * demand for the paint */

      clutter_actor_allocate_preferred_size (CLUTTER_ACTOR (HRN_CLUSTER (
                                                              cluster)->title2),
                                             TRUE);
      clutter_actor_allocate_preferred_size (CLUTTER_ACTOR (HRN_CLUSTER (
                                                              cluster)->meta2),
                                             TRUE);
    }
}
#endif


static ClutterTexture *
frame_tex_zoom_image (void)
{
  static ClutterTexture *ret = NULL;

  if (ret)
    return ret;
  ret =
    CLUTTER_TEXTURE (clutter_texture_new_from_file (PKGDATADIR
                                                    "hrn-cluster-image-zoom.png",
                                                    NULL));
  return ret;
}

static ClutterTexture *
frame_tex_zoom_video (void)
{
  static ClutterTexture *ret = NULL;

  if (ret)
    return ret;
  ret =
    CLUTTER_TEXTURE (clutter_texture_new_from_file (PKGDATADIR
                                                    "hrn-cluster-movie-zoom.png",
                                                    NULL));
  return ret;
}

static ClutterTexture *
frame_tex_zoom_audio (void)
{
  static ClutterTexture *ret = NULL;

  if (ret)
    return ret;
  ret =
    CLUTTER_TEXTURE (clutter_texture_new_from_file (PKGDATADIR
                                                    "hrn-cluster-music-zoom.png",
                                                    NULL));
  return ret;
}


static ClutterTexture *
frame_tex_play_image (void)
{
  static ClutterTexture *ret = NULL;

  if (ret)
    return ret;
  ret =
    CLUTTER_TEXTURE (clutter_texture_new_from_file (PKGDATADIR
                                                    "hrn-cluster-image-quick-play.png",
                                                    NULL));
  return ret;
}

static ClutterTexture *
frame_tex_play_video (void)
{
  static ClutterTexture *ret = NULL;

  if (ret)
    return ret;
  ret =
    CLUTTER_TEXTURE (clutter_texture_new_from_file (PKGDATADIR
                                                    "hrn-cluster-movie-quick-play.png",
                                                    NULL));
  return ret;
}

static ClutterTexture *
frame_tex_play_audio (void)
{
  static ClutterTexture *ret = NULL;

  if (ret)
    return ret;
  ret =
    CLUTTER_TEXTURE (clutter_texture_new_from_file (PKGDATADIR
                                                    "hrn-cluster-music-quick-play.png",
                                                    NULL));
  return ret;
}

static ClutterTexture *
frame_tex3 (void)
{
  static ClutterTexture *ret = NULL;

  if (ret)
    return ret;
  ret =
    CLUTTER_TEXTURE (clutter_texture_new_from_file (PKGDATADIR
                                                    "hrn-item-selected.png",
                                                    NULL));
  return ret;
}




static ClutterTexture *
frame_selected_bg (void)
{
  static ClutterTexture *ret = NULL;

  if (ret)
    return ret;
  ret =
    CLUTTER_TEXTURE (clutter_texture_new_from_file (PKGDATADIR
                                                    "hrn-bg-item-selected.png",
                                                    NULL));
  return ret;
}


static ClutterTexture *
frame_selected_hover_bg (void)
{
  static ClutterTexture *ret = NULL;

  if (ret)
    return ret;
  ret =
    CLUTTER_TEXTURE (clutter_texture_new_from_file (PKGDATADIR
                                                    "hrn-bg-item-selected-hover.png",
                                                    NULL));
  return ret;
}

#define MODE_NORMAL      0
#define MODE_ZOOM        1
#define MODE_SELECTED    2
#define MODE_PLAY        4

static void
cluster_set_frame (HrnCluster *cluster, gint no)
{
  HrnTextureFrame *frame    = HRN_TEXTURE_FRAME (cluster->priv->frame);
  HrnTextureFrame *frame_bg = HRN_TEXTURE_FRAME (cluster->priv->frame_bg);

  switch (no)
    {
      case MODE_ZOOM:
	switch (cluster->priv->type_mask)
	  {
	    case BKL_ITEM_TYPE_IMAGE:
              g_object_set (frame, "parent-texture",
                            frame_tex_zoom_image (), NULL);
	      break;
	    case BKL_ITEM_TYPE_AUDIO:
              g_object_set (frame, "parent-texture",
                            frame_tex_zoom_audio (), NULL);
	      break;
            default:
	    case BKL_ITEM_TYPE_VIDEO:
              g_object_set (frame, "parent-texture",
                            frame_tex_zoom_video (), NULL);
	      break;
	  }
        g_object_set (frame_bg, "parent-texture", frame_selected_bg (), NULL);
        break;

      case MODE_PLAY:
	switch (cluster->priv->type_mask)
	  {
	    case BKL_ITEM_TYPE_IMAGE:
              g_object_set (frame, "parent-texture",
                            frame_tex_play_image (), NULL);
	      break;
	    case BKL_ITEM_TYPE_AUDIO:
              g_object_set (frame, "parent-texture",
                            frame_tex_play_audio (), NULL);
	      break;
            default:
	    case BKL_ITEM_TYPE_VIDEO:
              g_object_set (frame, "parent-texture",
                            frame_tex_play_video (), NULL);
	      break;
	  }
        g_object_set (frame_bg, "parent-texture", frame_selected_bg (), NULL);
        break;

      case MODE_SELECTED:
        g_object_set (frame, "parent-texture", frame_tex3 (), NULL);
        g_object_set (frame_bg, "parent-texture",
                      frame_selected_hover_bg (), NULL);
        break;

      case MODE_SELECTED | MODE_ZOOM:
      case MODE_SELECTED | MODE_PLAY:
        g_object_set (frame, "parent-texture", frame_tex3 (), NULL);
        g_object_set (frame_bg, "parent-texture", frame_selected_bg (), NULL);
        break;


      case MODE_NORMAL:
      default:
        g_object_set (frame, "parent-texture", NULL, NULL);
        g_object_set (frame_bg, "parent-texture", NULL, NULL);
    }
  hrn_texture_frame_set_draw_middle (frame_bg, TRUE);
  hrn_texture_frame_set_draw_middle (frame, TRUE); /* this should be set only
                                                  for the modes that need it */
  clutter_actor_queue_redraw (CLUTTER_ACTOR (frame));
  clutter_actor_queue_redraw (CLUTTER_ACTOR (frame_bg));
}


static gboolean cluster_enter (ClutterActor *actor, ClutterEvent *event,
                               gpointer userdata);
static gboolean cluster_leave (ClutterActor *actor, ClutterEvent *event,
                               gpointer userdata);

static GObject *
hrn_cluster_constructor (GType type, guint n_params,
                         GObjectConstructParam *params)
{
  GObject    *object;
  HrnCluster *cluster;

  object = G_OBJECT_CLASS (hrn_cluster_parent_class)->constructor (
    type, n_params, params);

  cluster = HRN_CLUSTER (object);

  if (front_handle == 0)
    {
      front_handle = cogl_texture_new_from_file (
        PKGDATADIR "hrn-stack-front.png",
        COGL_TEXTURE_NONE,
        COGL_PIXEL_FORMAT_ANY,
        NULL);
      front_material = cogl_material_new ();
      cogl_material_set_color4ub (front_material, 255, 255, 255, 255);
      cogl_material_set_layer (front_material, 0, front_handle);
    }

  {
    ClutterActor *frame = hrn_texture_frame_new (NULL, 8, 64, 8, 64);
    cluster->priv->frame = frame;
    hrn_texture_frame_set_draw_middle (HRN_TEXTURE_FRAME (frame), TRUE);
    hrn_texture_frame_set_scale_invariant (HRN_TEXTURE_FRAME (frame), TRUE);
    clutter_actor_set_parent (frame, CLUTTER_ACTOR (object));
    clutter_actor_set_size (frame, DIM, DIM);
  }

  {
    ClutterActor *frame_bg = hrn_texture_frame_new (NULL, 8, 64, 8, 64);
    cluster->priv->frame_bg = frame_bg;
    hrn_texture_frame_set_draw_middle (HRN_TEXTURE_FRAME (frame_bg), TRUE);
    hrn_texture_frame_set_scale_invariant (HRN_TEXTURE_FRAME (frame_bg), TRUE);
    clutter_actor_set_parent (frame_bg, CLUTTER_ACTOR (object));
    clutter_actor_set_size (frame_bg, DIM, DIM);
  }

  g_signal_connect (object, "enter-event", G_CALLBACK (cluster_enter), object);
  g_signal_connect (object, "leave-event", G_CALLBACK (cluster_leave), object);

  return object;
}

static void
hrn_cluster_dispose (GObject *object)
{
  HrnCluster        *cluster = HRN_CLUSTER (object);
  HrnClusterPrivate *priv    = HRN_CLUSTER_GET_PRIVATE (object);

#if HRN_USE_NORMAL_LABELS
  if (cluster->title)
    clutter_actor_destroy (CLUTTER_ACTOR (cluster->title));
  if (cluster->meta)
    clutter_actor_destroy (CLUTTER_ACTOR (cluster->meta));
  cluster->title = cluster->meta = NULL;
  if (cluster->title2)
    clutter_actor_destroy (CLUTTER_ACTOR (cluster->title2));
  if (cluster->meta2)
    clutter_actor_destroy (CLUTTER_ACTOR (cluster->meta2));
  cluster->title2 = cluster->meta2 = NULL;
#endif
  if (priv->frame)
    clutter_actor_destroy (CLUTTER_ACTOR (priv->frame));
  priv->frame = NULL;
  if (priv->frame_bg)
    clutter_actor_destroy (CLUTTER_ACTOR (priv->frame_bg));
  priv->frame_bg = NULL;

  if (cluster->results)
    {
      g_list_free (cluster->results);
      cluster->results = NULL;
    }

  G_OBJECT_CLASS (hrn_cluster_parent_class)->dispose (object);
}

static gboolean
playbutton_clicked (NbtkButton *button, ClutterActor *event,
                    HrnCluster   *cluster)
{
  HrnQueue   *queue = master_queue;
  BklItem    *item;
  const char *mimetype;

  if (cluster->results == NULL)
    {
      return TRUE;
    }

  hrn_queue_prepend_list (queue,
                          HRN_CLUSTER (cluster)->results);

/* Check if the first item added was an image or video
   even though this is also done in hrn_queue_set_playing we need to do this
   here because by the time hrn_queue_set_playing is called the async calls
   that added the items to the queue will not have returned, so
   hrn_queue_set_playing will not know that there is anything in the queue

   FIXME: It might be useful for Bognor to have some way in the BognorPlayer
   interface to tell Hornsey to switch to theatre mode. */
  item     = cluster->results->data;
  mimetype = bkl_item_get_mimetype (item);

  if (mimetype)
    {
      if (g_str_has_prefix (mimetype, "video/") ||
          g_str_has_prefix (mimetype, "image/"))
        {
          hrn_set_theatre ();
        }
    }

  hrn_queue_set_playing (queue, TRUE);

  return TRUE;
}

static gboolean
playgroup_enter (ClutterActor *actor, ClutterEvent *event, gpointer userdata)
{
  clutter_actor_animate (userdata, CLUTTER_EASE_IN_CUBIC, 600,
                         "opacity", 255,
                         NULL);
  return FALSE;
}

static gboolean
playgroup_leave (ClutterActor *actor, ClutterEvent *event, gpointer userdata)
{
  clutter_actor_animate (userdata, CLUTTER_EASE_OUT_CUBIC, 600,
                         "opacity", 0,
                         NULL);
  return FALSE;
}


static gboolean
cluster_release (ClutterActor *actor, ClutterEvent *event, gpointer userdata)
{
  hrn_view_set_pos (GPOINTER_TO_INT (g_object_get_data (userdata, "accumpos")));

  hrn_set_zoom (hrn_get_zoom () * 2.0);
  return TRUE;
}

void
hrn_cluster_set_selected (HrnCluster *cluster, gboolean state)
{
  if (cluster->selected != state)
    {
      cluster->selected = state;
      if (cluster->selected)
        cluster_set_frame (cluster, MODE_SELECTED);
      else
        cluster_set_frame (cluster, MODE_NORMAL);
    }
}


gboolean
hrn_cluster_get_selected (HrnCluster *cluster)
{
  GList *iter;

  for (iter = cluster->results; iter; iter = iter->next)
    {
      BklItem *item = iter->data;
      if (hrn_view_is_selected (bkl_item_get_uri (item)))
        return TRUE;
    }
  return FALSE;
}


static gboolean
playbutton_enter (ClutterActor *actor, ClutterEvent *event, gpointer item)
{
  if (HRN_CLUSTER (item)->selected)
    cluster_set_frame (item, MODE_SELECTED | MODE_PLAY);
  else
    cluster_set_frame (item, MODE_PLAY);
  return TRUE;
}

static gboolean
playbutton_leave (ClutterActor *actor, ClutterEvent *event, gpointer item)
{
  if (hrn_cluster_get_selected (item))
    cluster_set_frame (item, MODE_SELECTED | MODE_ZOOM);
  else
    cluster_set_frame (item, MODE_ZOOM);
  return FALSE;
}


static void
make_cluster_play (HrnSwitcher *switcher, ClutterActor *group,
                   gboolean clustered)
{
  ClutterActor *foo = clutter_rectangle_new ();

  clutter_group_add (CLUTTER_GROUP (group), foo);
  ClutterColor blue = { 0, 0, 0xff, 0x90 };
  clutter_rectangle_set_color (CLUTTER_RECTANGLE (foo), &blue);

  clutter_actor_set_size (foo, DIM, DIM);
  clutter_actor_set_position (foo, DIM * 0.0, DIM * 0.0);
  clutter_actor_set_opacity (foo, 0x0);
  hrn_actor_make_draggable (
    CLUTTER_ACTOR (group), foo,
    HRN_DROP_MASK_QUEUE | HRN_DROP_MASK_SIDEBAR,
    NULL,
    G_CALLBACK (hrn_item_can_drop),
    G_CALLBACK (hrn_cluster_dropped),
    switcher);
  g_signal_connect (foo, "button-release-event", G_CALLBACK (
                      cluster_release), switcher);

  if (1)
    {
      ClutterActor *play_button = CLUTTER_ACTOR (hrn_button_new ());
      clutter_actor_set_name (play_button, "quick-play");
      clutter_group_add (CLUTTER_GROUP (group), play_button);
      clutter_actor_set_opacity (play_button, 0x00);

      clutter_actor_set_anchor_point_from_gravity (play_button,
                                                   CLUTTER_GRAVITY_SOUTH_EAST);

      if (clustered)
        {
          clutter_actor_set_size (play_button, DIM / 4, DIM / 4);
        }
      else
        {
          clutter_actor_set_size (play_button, DIM / 3, DIM / 3);
        }

      clutter_actor_set_position (play_button, DIM, DIM);

      g_signal_connect (play_button, "button-release-event",
                        G_CALLBACK (playbutton_clicked), switcher);

      g_signal_connect (group, "enter-event", G_CALLBACK (
                          playgroup_enter), play_button);
      g_signal_connect (group, "leave-event", G_CALLBACK (
                          playgroup_leave), play_button);


      g_signal_connect (play_button, "enter-event",
                        G_CALLBACK (playbutton_enter),
                        switcher);
      g_signal_connect (play_button, "leave-event",
                        G_CALLBACK (playbutton_leave),
                        switcher);

      hrn_actor_make_draggable (
        CLUTTER_ACTOR (group), play_button,
        HRN_DROP_MASK_QUEUE | HRN_DROP_MASK_SIDEBAR,
        NULL,
        G_CALLBACK (hrn_item_can_drop),
        G_CALLBACK (hrn_cluster_dropped),
        switcher);
    }
}

static ClutterActor *
cluster_alone (HrnSwitcher *switcher, guint state_no, gpointer userdata)
{
  ClutterActor *group = clutter_group_new ();
  ClutterActor *child1;

  /* we should pass the extra information to make the item lift/pinch
   * for the stack here.
   */
  child1 = hrn_make_item (g_list_nth_data (userdata, 0), TRUE);

  clutter_actor_set_position (child1, 0.04 * DIM, 0.14 * DIM);
  clutter_actor_set_scale (child1, 0.86, 0.86);
  clutter_group_add (CLUTTER_GROUP (group), child1);
  clutter_actor_set_reactive (group, TRUE);

  make_cluster_play (switcher, group, FALSE);

  hrn_switcher_adapt (HRN_SWITCHER (child1));

  return group;
}

#define DEPTH_SPREAD     (-100)
#define DEPTH2_SPREAD    (-1)

static gfloat
clustered_depth (guint no)
{
  if (no == 0)
    {
      return 0;
    }
  return DEPTH_SPREAD * no;
}

static gfloat
clustered_scale (guint no)
{
  gint   i;
  gfloat ret = 0.6;

  if (no == 0)
    {
      return 0.55;
    }
  for (i = 0; i < no; i++)
    ret *= 0.9;

  return ret;
}

static gfloat
clustered_x (guint no)
{
  gint   i;
  gfloat ret = DIM * 0.12;

  if (no == 0)
    return DIM * 0.5 - (clustered_scale (no) * DIM / 2);

  for (i = 0; i < no; i++)
    ret *= 1.0;

  if (no % 2 == 0)
    return DIM * 0.5 - (clustered_scale (no) * DIM / 2) + ret;
  else
    return DIM * 0.5 - (clustered_scale (no) * DIM / 2) - ret;
}

static gfloat
clustered_y (guint no)
{
  gint   i;

  gfloat ret = DIM * 0.36;

  if (no == 0)
    {
      return ret;
    }

  for (i = 0; i < no; i++)
    ret *= 0.6;

  if (no % 2 == 0)
    ret += DIM * 0.10;

  return ret;
}

#if 0
static gfloat
clustered_depth_hover (guint no)
{
  if (no == 0)
    {
      return 0;
    }
  return DEPTH_SPREAD * no;
}
#endif

static GList *
list_last_and_count (GList *list,
                     int   *count)
{
  GList *l;
  int c;

  for (l = list, c = 0; l;) {
    c++;

    if (l->next == NULL) {
      break;
    } else {
      l = l->next;
    }
  }

  *count = c;
  return l;
}

static gboolean
cluster_enter (ClutterActor *actor, ClutterEvent *event, gpointer userdata)
{
  /* geometry for states stored as a triplet, x, y, scale */
  GList *children;
  GList *children2;
  GList *c;

  gdouble hoverdata[HRN_CLUSTER_COUNT + 1][HRN_CLUSTER_COUNT][3] =
  {
    /* 0 */
    { { 0,                        }, },
    /* 1 */
    { { 0.04 * DIM, 0.14 * DIM, 0.86 }, },
    /* 2 */
    {
      { 0.2 * DIM, 0.3 * DIM, 0.7 },
      { 0.04 * DIM, 0.1 * DIM, 0.7 },
    },
    /* 3 */
    {
      { 0.20 * DIM, 0.40 * DIM, 0.60 },
      { 0.02 * DIM, 0.12 * DIM, 0.50 },
      { 0.40 * DIM, 0.12 * DIM, 0.50 },
    },
    /* 4 */
    {
      { 0.03 * DIM, 0.13 * DIM, 0.43 },
      { 0.47 * DIM, 0.13 * DIM, 0.43 },
      { 0.03 * DIM, 0.57 * DIM, 0.43 },
      { 0.47 * DIM, 0.57 * DIM, 0.43 },
    },
    /* 5 */
    {
      { 0.25 * DIM, 0.35 * DIM, 0.50 },
      { 0.03 * DIM, 0.13 * DIM, 0.43 },
      { 0.47 * DIM, 0.13 * DIM, 0.43 },
      { 0.03 * DIM, 0.57 * DIM, 0.43 },
      { 0.47 * DIM, 0.57 * DIM, 0.43 },
    },
#if 0  /* these layout are kept for now, since they might
          be useful if we can have more items visible */
    /* 6 */
    {
      { 0.25 * DIM, 0.25 * DIM, 0.50 },

      { 0.15 * DIM, 0.00 * DIM, 0.33 },
      { 0.48 * DIM, 0.00 * DIM, 0.33 },

      { 0.00 * DIM, 0.33 * DIM, 0.33 },
      { 0.66 * DIM, 0.33 * DIM, 0.33 },
      { 0.33 * DIM, 0.66 * DIM, 0.33 },
    },
    /* 7 */
    {
      { 0.25 * DIM, 0.25 * DIM, 0.50 },

      { 0.15 * DIM, 0.00 * DIM, 0.33 },
      { 0.48 * DIM, 0.00 * DIM, 0.33 },
      { 0.00 * DIM, 0.33 * DIM, 0.33 },
      { 0.66 * DIM, 0.33 * DIM, 0.33 },
      { 0.15 * DIM, 0.66 * DIM, 0.33 },
      { 0.48 * DIM, 0.66 * DIM, 0.33 },
    },
    /* 8 */
    {
      { 0.25 * DIM, 0.25 * DIM, 0.50 },
      { 0.15 * DIM, 0.00 * DIM, 0.33 },
      { 0.48 * DIM, 0.00 * DIM, 0.33 },
      { 0.00 * DIM, 0.33 * DIM, 0.33 },
      { 0.66 * DIM, 0.33 * DIM, 0.33 },
      { 0.00 * DIM, 0.66 * DIM, 0.33 },
      { 0.33 * DIM, 0.66 * DIM, 0.33 },
      { 0.66 * DIM, 0.66 * DIM, 0.33 },
    },
    /* 9 */
    {
      { 0.25 * DIM, 0.25 * DIM, 0.50 },
      { 0.00 * DIM, 0.00 * DIM, 0.33 },
      { 0.33 * DIM, 0.00 * DIM, 0.33 },
      { 0.66 * DIM, 0.00 * DIM, 0.33 },
      { 0.00 * DIM, 0.33 * DIM, 0.33 },
      { 0.66 * DIM, 0.33 * DIM, 0.33 },
      { 0.00 * DIM, 0.66 * DIM, 0.33 },
      { 0.33 * DIM, 0.66 * DIM, 0.33 },
      { 0.66 * DIM, 0.66 * DIM, 0.33 },
    },
#endif
  };

  if (hrn_switcher_get_state (userdata) == 3 ||
      hrn_switcher_get_state (userdata) <= 1)
    {
      if (HRN_CLUSTER (actor)->selected)
        cluster_set_frame (HRN_CLUSTER (actor), MODE_SELECTED | MODE_ZOOM);
      else
        cluster_set_frame (HRN_CLUSTER (actor), MODE_ZOOM);
    }

  if (hrn_switcher_get_state (userdata) != 3)
    return FALSE;

  children2 =
    clutter_container_get_children (CLUTTER_CONTAINER (hrn_switcher_get_current (
                                                         userdata)));
  children =
    clutter_container_get_children (CLUTTER_CONTAINER (children2->data));

  ClutterActor *child;

  gint          i, num;

  c = list_last_and_count (children, &num);
  for (i = 0; c; c = c->prev, i++)
    {
      child = c->data;
      if (child)
        {
          clutter_actor_animate (child, CLUTTER_EASE_IN_OUT_QUINT, 800,
                                 "x", (gdouble)(hoverdata[num][i][0] + 11),
                                 "y", (gdouble)(hoverdata[num][i][1] - 13),
                                 "scale-x", hoverdata[num][i][2],
                                 "scale-y", hoverdata[num][i][2],
                                 NULL);
        }
    }

  g_list_free (children);
  g_list_free (children2);

  /*hrn_cluster_preload (userdata);*/


  return FALSE;
}


static gboolean
cluster_leave (ClutterActor *actor, ClutterEvent *event, gpointer userdata)
{
  GList        *children;
  GList        *children2;
  GList        *c;
  HrnCluster   *cluster = HRN_CLUSTER (actor);
  ClutterActor *child;

  if (cluster->selected)
    cluster_set_frame (cluster, MODE_SELECTED);
  else
    cluster_set_frame (cluster, MODE_NORMAL);

  if (hrn_switcher_get_state (userdata) != 3)
    return FALSE;

  gint i;

  children2 =
    clutter_container_get_children (CLUTTER_CONTAINER (hrn_switcher_get_current (
                                                         userdata)));
  children =
    clutter_container_get_children (CLUTTER_CONTAINER (children2->data));

  for (c = g_list_last (children), i = 0; c; c = c->prev, i++)
    {
      child = c->data;

      clutter_actor_animate (child, CLUTTER_EASE_IN_OUT_QUINT, 800,
                             "x", (gdouble) clustered_x (i),
                             "y", (gdouble) clustered_y (i),
                             "scale-x", clustered_scale (i),
                             "scale-y", clustered_scale (i),

                             NULL);
    }
  g_list_free (children);
  g_list_free (children2);
  return FALSE;
}


static ClutterActor *
cluster_grouped (HrnSwitcher *switcher, guint state_no, gpointer userdata)
{
  ClutterActor *group  = clutter_group_new ();
  ClutterActor *group2 = clutter_group_new ();
  ClutterActor *child[HRN_CLUSTER_COUNT] = { NULL, };
  gint          i;

  clutter_group_add (CLUTTER_GROUP (group), group2);

  for (i = 0; i < HRN_CLUSTER_COUNT; i++)
    {
      if (g_list_nth_data (userdata, i))
        {
          child[i] = hrn_make_item (g_list_nth_data (userdata, i), TRUE);
          clutter_actor_set_scale (child[i], 0.78, 0.78);
          if (0) clutter_actor_set_depth (child[i], clustered_depth (i));
          clutter_actor_set_scale (child[i], clustered_scale (
                                     i), clustered_scale (i));
          clutter_actor_set_position (child[i], clustered_x (i),
                                      clustered_y (i));
        }
    }

  for (i = HRN_CLUSTER_COUNT - 1; i >= 0; i--)
    {
      if (child[i])
        {
          clutter_group_add (CLUTTER_GROUP (group2), child[i]);
          hrn_switcher_adapt (HRN_SWITCHER (child[i]));
        }
    }

  clutter_actor_set_reactive (group, TRUE);

  make_cluster_play (switcher, group, TRUE);


  /* starting the preload here (or perhaps after a timeout)
   * could be good (shoudl be uninstalled on continued user interaction?)
   */
  if (0) hrn_cluster_preload (HRN_CLUSTER (switcher));

  return group;
}


void
hrn_cluster_preload (HrnCluster *cluster)
{
  gint   i;
  GList *iter = cluster->results;

  for (iter = cluster->results, i = 0; iter;
       iter = iter->next, i++)
    {
      BklItem     *item  = iter->data;
      const gchar *thumb = bkl_item_extended_get_thumbnail (
        (BklItemExtended *) item);
      if (thumb)
        {
          ClutterActor *foo = hrn_texture_load (thumb);
          g_object_unref (foo);
        }

      if (i > 32)  /* maximum preload 32 items per cluster */
        {
          return;
        }
    }
}


static ClutterActor *
cluster_grid8lim (HrnSwitcher *switcher, guint state_no, gpointer userdata)
{
  ClutterActor *group = clutter_group_new ();
  gint          i;
  gint          j       = 0;
  gint          line    = 0;
  gint          modulus =
    GPOINTER_TO_INT (g_object_get_data (G_OBJECT (switcher), "accumpos")) % 8;
  GList *cells, *c;

  cells = (GList *) g_list_last (userdata);

  for (c = cells, i = 0; c && i < HRN_CLUSTER_TRANS_START; c = c->prev, i++)
    {
      ClutterActor *child;

      if (i > HRN_CLUSTER_TRANS_START)
        {
          ClutterColor white = { 0xff, 0xff, 0xff, 0x33 };
          child = clutter_rectangle_new ();
          clutter_rectangle_set_color (CLUTTER_RECTANGLE (child), &white);
          clutter_actor_set_size (child, DIM * 0.5, DIM * 0.5);
        }
      else
        {
          child = hrn_make_item (c->data, FALSE);
        }
      clutter_actor_set_scale (child, 0.22, 0.22);
      clutter_actor_set_position (child, DIM * 0.25 * j,
                                  rint (DIM * 0.25 * line * HRN_ROW_SPACING));
      j++;

      if (++modulus >= 8)
        {
          j -= 8;
          line++;
          modulus = 0;
        }

      clutter_group_add (CLUTTER_GROUP (group), child);
      hrn_switcher_adapt (HRN_SWITCHER (child));
    }

  return group;
}


static ClutterActor *
cluster_grid8 (HrnSwitcher *switcher, guint state_no, gpointer userdata)
{
  ClutterActor *group = clutter_group_new ();
  gint          j       = 0;
  gint          line    = 0;
  gint          modulus =
    GPOINTER_TO_INT (g_object_get_data (G_OBJECT (switcher), "accumpos")) % 8;
  GList *cells, *c;

  g_assert (HRN_IS_SWITCHER (switcher));

  cells = (GList *) userdata;

  for (c = g_list_last (cells); c; c = c->prev)
    {
      ClutterActor *child;

      child = hrn_make_item (c->data, FALSE);
      clutter_actor_set_scale (child, 0.22, 0.22);
      clutter_actor_set_position (child, DIM * 0.25 * j,
                                  rint (DIM * 0.25 * line * HRN_ROW_SPACING));
      j++;

      if (++modulus >= 8)
        {
          j -= 8;
          line++;
          modulus = 0;
        }

      clutter_group_add (CLUTTER_GROUP (group), child);
      hrn_switcher_adapt (HRN_SWITCHER (child));
    }

  return group;
}

static ClutterActor *
cluster_grid4 (HrnSwitcher *switcher, guint state_no, gpointer userdata)
{
  ClutterActor  *group = clutter_group_new ();
  gint           j       = 0;
  gint           line    = 0;
  gint           modulus =
    GPOINTER_TO_INT (g_object_get_data (G_OBJECT (switcher), "accumpos")) % 4;
  GList *cells, *c;

  cells = (GList *) g_list_last (userdata);
  for (c = cells; c; c = c->prev)
    {
      ClutterActor *child;

      child = hrn_make_item (c->data, FALSE);
      clutter_actor_set_scale (child, 0.22, 0.22);
      clutter_actor_set_position (child, DIM * 0.25 * j,
                                  rint (DIM * 0.25 * line * HRN_ROW_SPACING));
      j++;

      if (++modulus >= 4)
        {
          j -= 4;
          line++;
          modulus = 0;
        }

      clutter_group_add (CLUTTER_GROUP (group), child);
      hrn_switcher_adapt (HRN_SWITCHER (child));
    }

  return group;
}

static ClutterActor *
cluster_grid3 (HrnSwitcher *switcher, guint state_no, gpointer userdata)
{
  ClutterActor  *group = clutter_group_new ();
  gint           j       = 0;
  gint           line    = 0;
  gint           modulus =
    GPOINTER_TO_INT (g_object_get_data (G_OBJECT (switcher), "accumpos")) % 3;
  GList *cells, *c;

  cells = (GList *) g_list_last (userdata);
  for (c = cells; c; c = c->prev)
    {
      ClutterActor *child;

      child = hrn_make_item (c->data, FALSE);
      clutter_actor_set_scale (child, 0.15, 0.15);
      clutter_actor_set_position (child, DIM * 0.16 * j,
                                  rint (DIM * 0.16 * HRN_ROW_SPACING * line));
      j++;

      if (++modulus >= 3)
        {
          j -= 3;
          line++;
          modulus = 0;
        }

      clutter_group_add (CLUTTER_GROUP (group), child);
      hrn_switcher_adapt (HRN_SWITCHER (child));
    }

  return group;
}

static void
cluster_alone_to_grouped_adapt (HrnSwitcher *switcher, guint state_no,
                                ClutterActor *statechild, gdouble progress,
                                gpointer user_data)
{
  GList *parent, *children, *c;
  ClutterActor *child;
  gint          i;

  parent = clutter_container_get_children (CLUTTER_CONTAINER (statechild));
  children = clutter_container_get_children (CLUTTER_CONTAINER (parent->data));
  g_list_free (parent);

  for (c = g_list_last (children), i = 0; c; c = c->prev, i++)
    {
      child = c->data;
      if (child)
        {
          clutter_actor_set_position (child,
                                      LIN (0.0,
                                           clustered_x (i)),
                                      LIN (0, clustered_y (i)));
          clutter_actor_set_scale (child,
                                   LIN (1.0,
                                        clustered_scale (i)),
                                   LIN (1.0, clustered_scale (i)));

          if (i > 0) clutter_actor_set_opacity (child, LIN (0.0, 255.0));

          hrn_switcher_adapt (HRN_SWITCHER (child));
        }
    }
  g_list_free (children);
}

#define PW    1.0

static void
cluster_grouped_to_eightlim_adapt (HrnSwitcher *switcher, guint state_no,
                                   ClutterActor *child_, gdouble progress,
                                   gpointer user_data)
{
  GList        *children, *c;
  ClutterActor *child;
  gint          i;
  gint          j     = 0;
  gint          line  = 0;
  gint          num   = hrn_cluster_get_cells (CLUTTER_ACTOR (switcher));
  gint          accum =
    GPOINTER_TO_INT (g_object_get_data (G_OBJECT (switcher), "accumpos"));
  gint          modulus = accum % 8;

  children = clutter_container_get_children (CLUTTER_CONTAINER (child_));

  if (num > HRN_CLUSTER_TRANS_START)
    num = HRN_CLUSTER_TRANS_START;

  progress /= 2;

  progress = pow (progress, PW);

  for (c = g_list_last (children), i = 0; c && i < num; c = c->prev, i++)
    {
      child = c->data;
      clutter_actor_set_position (child, LIN (clustered_x (i), DIM * 0.25 * j),
                                  LIN (clustered_y (i), DIM * 0.25 * line *
                                       HRN_ROW_SPACING));
      clutter_actor_set_scale (child,
                               LIN (clustered_scale (i),
                                    0.22), LIN (clustered_scale (i), 0.22));
      if (0) clutter_actor_set_depth (child,
                                      LIN (clustered_depth (i), i *
                                           DEPTH2_SPREAD));
      if (i >= 5) clutter_actor_set_opacity (child, LIN (0.0, 255.0));

      hrn_switcher_adapt (HRN_SWITCHER (child));
      j++;

      if (++modulus >= 8)
        {
          j -= 8;
          line++;
          modulus = 0;
        }
    }


  g_list_free (children);
}


static void
cluster_eightlim_to_eight_adapt (HrnSwitcher *switcher, guint state_no,
                                 ClutterActor *child_, gdouble progress,
                                 gpointer user_data)
{
  GList        *children, *c;
  ClutterActor *child;
  gint          i;
  gint          j     = 0;
  gint          line  = 0;
  gint          num   = hrn_cluster_get_cells (CLUTTER_ACTOR (switcher));
  gint          accum =
    GPOINTER_TO_INT (g_object_get_data (G_OBJECT (switcher), "accumpos"));
  gint          modulus = accum % 8;

  children = clutter_container_get_children (CLUTTER_CONTAINER (child_));

  progress /= 2;
  progress += 0.5;

  progress = pow (progress, PW);

  for (c = g_list_last (children), i = 0; c && i < num; c = c->prev, i++)
    {
      child = c->data;
      clutter_actor_set_position (child, LIN (clustered_x (i), DIM * 0.25 * j),
                                  LIN (clustered_y (i), DIM * 0.25 * line *
                                       HRN_ROW_SPACING));
      clutter_actor_set_scale (child,
                               LIN (clustered_scale (i),
                                    0.22), LIN (clustered_scale (i), 0.22));
      if (0) clutter_actor_set_depth (child,
                                      LIN (clustered_depth (i), i *
                                           DEPTH2_SPREAD));
      if (i >= 5) clutter_actor_set_opacity (child, LIN (128.0, 255.0));
      if (i >= 5) clutter_actor_set_opacity (child, LIN (0.0, 255.0));

      hrn_switcher_adapt (HRN_SWITCHER (child));

      j++;

      if (++modulus >= 8)
        {
          j -= 8;
          line++;
          modulus = 0;
        }
    }


  g_list_free (children);
}

/* static buffers, holding the coordinates being transformed, we
 * should handle the case of more than 8192 items in a cluster
 * more gracefully.
 */
#define MAX_ITEMS 8192

static gfloat xs[MAX_ITEMS];
static gint   ys[MAX_ITEMS];

static void
cluster_eight_to_four_adapt (HrnSwitcher *switcher, guint state_no,
                             ClutterActor *child_, gdouble progress,
                             gpointer user_data)
{
  GList        *children, *c;
  ClutterActor *child;
  gint          i;
  gint          j       = 0;
  gint          line    = 0;
  gint          num     = hrn_cluster_get_cells (CLUTTER_ACTOR (switcher));
  gint          modulus =
    GPOINTER_TO_INT (g_object_get_data (G_OBJECT (switcher), "accumpos")) % 8;
  gboolean      leftstart = FALSE;

  if (num >= MAX_ITEMS)
    return;

  children = clutter_container_get_children (CLUTTER_CONTAINER (child_));

  if (modulus < 4)
    leftstart = TRUE;

  j    = 0;
  line = 0;
  for (i = 0; i < num; i++)
    {
      xs[i] = j;
      ys[i] = line;
      j++;

      if (++modulus >= 8)
        {
          j -= 8;
          line++;
          modulus = 0;
        }
    }

  modulus =
    GPOINTER_TO_INT (g_object_get_data (G_OBJECT (switcher), "accumpos")) % 4;
  j    = 0;
  line = 0;

  for (c = g_list_last (children), i = 0; c && i < num; c = c->prev, i++)
    {
      child = c->data;

      clutter_actor_set_position (child,
                                  LIN (DIM * 0.25 * xs[i], DIM * 0.25 * j),
                                  LIN (DIM * 0.25 * ys[i] * HRN_ROW_SPACING,
                                       DIM * 0.25 * line * HRN_ROW_SPACING));

      j++;

      if (++modulus >= 4)
        {
          j -= 4;
          line++;
          modulus = 0;
        }
      hrn_switcher_adapt (HRN_SWITCHER (child));
    }

  g_list_free (children);
}



static void
cluster_four_to_three_adapt (HrnSwitcher *switcher, guint state_no,
                             ClutterActor *child_, gdouble progress,
                             gpointer user_data)
{
  GList        *children, *c;
  ClutterActor *child;
  gint          i;
  gint          j       = 0;
  gint          line    = 0;
  gint          num     = hrn_cluster_get_cells (CLUTTER_ACTOR (switcher));
  gint          modulus =
    GPOINTER_TO_INT (g_object_get_data (G_OBJECT (switcher), "accumpos")) % 4;
  gboolean      leftstart = FALSE;

  if (num >= MAX_ITEMS)
    return;

  children = clutter_container_get_children (CLUTTER_CONTAINER (child_));


  if (modulus < 4)
    leftstart = TRUE;

  j    = 0;
  line = 0;
  for (i = 0; i < num; i++)
    {
      xs[i] = j;
      ys[i] = line;
      j++;

      if (++modulus >= 4)
        {
          j -= 4;
          line++;
          modulus = 0;
        }
    }

  modulus =
    GPOINTER_TO_INT (g_object_get_data (G_OBJECT (switcher), "accumpos")) % 3;
  j    = 0;
  line = 0;
  for (c = g_list_last (children), i = 0; c && i < num; c = c->prev, i++)
    {
      child = c->data;

      clutter_actor_set_position (child,
                                  LIN (DIM * 0.25 * xs[i], DIM * 0.16 * j),
                                  LIN (DIM * 0.25 * ys[i] * HRN_ROW_SPACING,
                                       DIM * 0.16 * line * HRN_ROW_SPACING));
      clutter_actor_set_scale (child, LIN (0.23, 0.15), LIN (0.23, 0.15));

      hrn_switcher_adapt (HRN_SWITCHER (child));
      j++;

      if (++modulus >= 3)
        {
          j -= 3;
          line++;
          modulus = 0;
        }
    }

  g_list_free (children);
}


static void
cluster_three_to_three_adapt (HrnSwitcher *switcher, guint state_no,
                              ClutterActor *child_, gdouble progress,
                              gpointer user_data)
{
  GList        *children, *c;
  ClutterActor *child;
  gint          i;
  gint          j       = 0;
  gint          line    = 0;
  gint          num     = hrn_cluster_get_cells (CLUTTER_ACTOR (switcher));
  gint          modulus =
    GPOINTER_TO_INT (g_object_get_data (G_OBJECT (switcher), "accumpos")) % 3;
  gboolean      leftstart = FALSE;

  children = clutter_container_get_children (CLUTTER_CONTAINER (child_));

  if (modulus < 3)
    leftstart = TRUE;

  j    = 0;
  line = 0;
  for (i = 0; i < num; i++)
    {
      xs[i] = j;
      ys[i] = line;
      j++;

      if (++modulus >= 3)
        {
          j -= 3;
          line++;
          modulus = 0;
        }
    }

  modulus =
    GPOINTER_TO_INT (g_object_get_data (G_OBJECT (switcher), "accumpos")) % 3;
  j    = 0;
  line = 0;

  for (c = g_list_last (children), i = 0; c && i < num; c = c->prev, i++)
    {
      child = c->data;

      clutter_actor_set_position (child,
                                  LIN (DIM * 0.16 * xs[i], DIM * 0.16 * j),
                                  LIN (DIM * 0.16 * ys[i] * HRN_ROW_SPACING,
                                       DIM * 0.16 * line * HRN_ROW_SPACING));
      clutter_actor_set_scale (child, 0.15, 0.15);

      hrn_switcher_adapt (HRN_SWITCHER (child));
      j++;

      if (++modulus >= 3)
        {
          j -= 3;
          line++;
          modulus = 0;
        }
    }

  g_list_free (children);
}


static void
hrn_cluster_set_cells (ClutterActor *cluster, gint cells)
{
  HrnClusterPrivate *priv = HRN_CLUSTER_GET_PRIVATE (cluster);

  priv->cells = cells;
}

ClutterActor *
hrn_make_cluster (GList *results)
{
  HrnSwitcher *switcher = g_object_new (HRN_TYPE_CLUSTER, NULL);
  HrnClusterPrivate *priv = HRN_CLUSTER (switcher)->priv;
  GList *iter;

  priv->type_mask = 0;
  for (iter = results; iter; iter=iter->next)
    {
      priv->type_mask |= bkl_item_get_item_type (iter->data);
    }

  clutter_actor_set_size (CLUTTER_ACTOR (switcher), DIM / 4, DIM / 4);

  HRN_CLUSTER (switcher)->results = results;

  hrn_switcher_add_state_full (switcher, NULL, -1, 0.6,
                               cluster_alone, results,
                               NULL, NULL);

  hrn_switcher_add_state_full (switcher, NULL, -1, 1.00,
                               cluster_grouped, results,
                               cluster_alone_to_grouped_adapt, NULL);

/* all the adapt calls below should be made into reusing the same state actor,
 * to avoid object creation/destroying churn (unless such churn might actually
 * be desirable for caching scalability)
 */

  hrn_switcher_add_state_full (switcher, NULL, -1, 1.05,
                               cluster_grouped, results,
                               NULL, NULL);

  hrn_switcher_add_state_full (switcher, NULL, -1, 1.7,
                               cluster_grid8lim, results,
                               cluster_grouped_to_eightlim_adapt, NULL);

  hrn_switcher_add_state_full (switcher, NULL, -1, 2.0,
                               cluster_grid8, results,
                               cluster_eightlim_to_eight_adapt, NULL);

  hrn_switcher_add_state_full (switcher, NULL, -1, 4.0,
                               cluster_grid8, results,
                               cluster_eight_to_four_adapt, NULL);
  hrn_switcher_add_state_full (switcher, NULL, -1, 7.8,
                               cluster_grid4, results,
                               cluster_four_to_three_adapt, NULL);


  hrn_switcher_add_state_full (switcher, NULL, -1, 40.01,
                               cluster_grid3, results,
                               cluster_three_to_three_adapt, NULL);

  hrn_switcher_add_state_full (switcher, NULL, -1, 42.01,
                               cluster_grid3, results,
                               cluster_three_to_three_adapt, NULL);

  hrn_cluster_set_cells (CLUTTER_ACTOR (switcher), g_list_length (results));

  return CLUTTER_ACTOR (switcher);
}

gint
hrn_cluster_get_cells (ClutterActor *cluster)
{
  HrnClusterPrivate *priv = HRN_CLUSTER_GET_PRIVATE (cluster);

  return priv->cells;
}


void
hrn_cluster_adapt (HrnCluster *cluster)
{
  hrn_switcher_adapt (HRN_SWITCHER (cluster));
}

void
hrn_cluster_show_labels (HrnCluster *cluster, gint state)
{
  /* bail if not visible */
  /* iterate through items and set the */
}
