/*
 * Hornsey - Moblin Media Player.
 * Copyright © 2009 Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <config.h>

#include <clutter/clutter.h>
#include <glib/gi18n.h>
#include <glib/gprintf.h>
#include <gio/gio.h>
#include <stdlib.h>
#include <string.h>

#include <dbus/dbus-protocol.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-bindings.h>

#include <bickley/bkl-utils.h>
#include <bickley/bkl-source-client.h>
#include <bickley/bkl-source-manager-client.h>

#include <bognor/br-iface-player.h>
#include <bognor/br-queue.h>

#define SN_API_NOT_YET_FROZEN
#include <libsn/sn.h>
#include <clutter/x11/clutter-x11.h>

#include "hrn.h"
#include "hrn-state-manager.h"

#include "math.h"

#include <unique/unique.h>

#define HORNSEY_DBUS_SERVICE    "org.moblin.Hornsey"
#define HORNSEY_DBUS_PATH       "/org/moblin/Hornsey"

/* copied raise function from moblin-web-browser */
gboolean
hrn_raise (void)
{
  Window              xid;
  Display            *display;
  ClutterStage       *stage;
  XClientMessageEvent xclient;

  stage = CLUTTER_STAGE (clutter_stage_get_default ());

  display = clutter_x11_get_default_display ();
  xid     = clutter_x11_get_stage_window (stage);

  /* Saw how to do this in gtk (gdk/x11/gdkwindow-x11.c:gdk_window_focus ()) */
  memset (&xclient, 0, sizeof (xclient));
  xclient.type         = ClientMessage;
  xclient.window       = xid;
  xclient.message_type = XInternAtom (display, "_NET_ACTIVE_WINDOW", False);
  xclient.format       = 32;
  xclient.data.l[0]    = 1;
  xclient.data.l[1]    = 0L;
  xclient.data.l[2]    = None;
  xclient.data.l[3]    = 0;
  xclient.data.l[4]    = 0;

  XSendEvent (display, clutter_x11_get_root_window (), False,
              SubstructureRedirectMask | SubstructureNotifyMask,
              (XEvent *) &xclient);

  return TRUE;
}


static UniqueResponse
hrn_message_received_cb (UniqueApp *app, gint command,
                         UniqueMessageData *message_data, guint time_,
                         gpointer user_data)
{
  gchar *uri;

  switch (command)
    {
      case UNIQUE_ACTIVATE:
        hrn_raise ();
        return UNIQUE_RESPONSE_OK;

      case UNIQUE_OPEN:
        uri = unique_message_data_get_text (message_data);
        if (uri && (uri[0] != '\0'))
          {
            BklItem *item;

            item = hrn_get_item_for_uri (uri);
            if (item)
              {
                hrn_set_theatre ();

                if (bkl_item_get_item_type (item) == BKL_ITEM_TYPE_IMAGE) {
                  /* We disable the slideshow so that the image will not
                     expire */
                  hrn_theatre_set_image_slideshow ((HrnTheatre *) hrn_theatre,
                                                   FALSE);
                }

                hrn_play_now (item);
              }
            else
              {
                hrn_set_theatre ();
                hrn_play_uri_now (uri);
              }
          }

        g_free (uri);
        hrn_raise ();

        return UNIQUE_RESPONSE_OK;

      default:
        g_print ("uh?! %i\n", command);
        break;
    }
  return UNIQUE_RESPONSE_PASSTHROUGH;
}

/* these are just ClutterGroups setting up a fixed hierachy
 * of containers to organize "toplevel" ui elements in. The
 * indentation is intentional and shows the toplevel
 */

ClutterActor *hrn_controls     = NULL;
ClutterActor *hrn_toolbar      = NULL;
ClutterActor *hrn_search_entry = NULL;

#if 0
ClutterActor *hrn_theatre_toolbar = NULL;
ClutterActor *hrn_normal_toolbar  = NULL;
#endif

ClutterActor *zoom_indicator     = NULL;
ClutterActor *hrn_sidebar        = NULL;
ClutterActor *hrn_scrollbar      = NULL;
ClutterActor *loc_indicator      = NULL;
ClutterActor *hrn_queues         = NULL;
ClutterActor *hrn_controls_frame = NULL;

ClutterActor *hrn_content_area     = NULL;
ClutterActor *hrn_theatre_parent   = NULL;    /* might be unneeded */
ClutterActor *hrn_theatre          = NULL;
ClutterActor *hrn_theatre_underlay = NULL;
ClutterActor *hrn_theatre_lowlight = NULL;    /* background for drag and drop.
                                               */
#if 0
ClutterActor *hrn_theatre_backdrop = NULL; / \ *
background of the theatre. *\ / * /
ClutterActor * hrn_audio_meta = NULL;  /* group containing meta data
                                        * about currently playing audio
                                        */
#endif
ClutterActor *hrn_view_parent = NULL;
ClutterActor *hrn_view        = NULL;

GKeyFile     *pinned_searches = NULL;

HrnQueue     *master_queue;           /* The main queue actor...FIXME: Need some
                                         sort of
                                         queue manager system */
ClutterActor *master_controls;        /* The controls is another client that
                                         also controls the master queue */

HrnSource              *local_source      = NULL;
GList                  *transient_sources = NULL; /* Contains HrnSource */

BklSourceManagerClient *manager     = NULL;
BrQueue                *local_queue = NULL;

HrnStateManager *state_manager;

BrQueue *hrn_get_default_queue (void)
{
  return local_queue;
}

/* This is so we know when all the sources are loaded */
static int   source_count   = 0;
static int   source_replies = 0;

static guint controls_hide_handler_id = 0;
static int   zoom_level               = -1;

static void
fade_controls_sanity (ClutterAnimation *animation,
                      ClutterActor     *controls)
{
  if (!hrn_controls_can_has_fade)
    clutter_actor_set_opacity (master_controls, 0xff);
  else
    clutter_actor_hide (master_controls);
}

#if 0
static gboolean
fade_out2 (gpointer foo)
{
  if (!hrn_controls_can_has_fade)
    return FALSE;
  clutter_actor_animate (master_controls, CLUTTER_EASE_IN_CUBIC,
                         CONTROLS_OUT_DURATION3,
                         "opacity", 0,
                         "signal::completed", fade_controls_sanity,
                         master_controls,
                         NULL);
  return FALSE;
}
#endif

static gboolean
fade_out (gpointer foo, gpointer *bar)
{
  if (!hrn_controls_can_has_fade
      || master_controls == NULL ||
      clutter_actor_get_paint_visibility (master_controls) == FALSE)
    return FALSE;
  clutter_stage_hide_cursor (CLUTTER_STAGE (clutter_stage_get_default ()));
  return FALSE;
}

static gboolean
controls_hide_handler (gpointer foo)
{
  if (!hrn_controls_can_has_fade || master_controls == NULL ||
      clutter_actor_get_paint_visibility (master_controls) == FALSE)
    {
      controls_hide_handler_id = 0;
      return FALSE;
    }
  clutter_actor_animate (master_controls, CLUTTER_LINEAR, CONTROLS_OUT_DURATION,
                         "opacity", 0,
                         "signal::completed", fade_out, NULL,
                         "signal::completed", fade_controls_sanity, NULL,
                         NULL);
  controls_hide_handler_id = 0;

  return FALSE;
}


gboolean hrn_controls_can_has_fade = FALSE;

void
hrn_controls_wake (void)
{
  if (controls_hide_handler_id)
    {
      g_source_remove (controls_hide_handler_id);
      controls_hide_handler_id = 0;
    }
  clutter_actor_show (master_controls);
  clutter_stage_show_cursor (CLUTTER_STAGE (clutter_stage_get_default ()));

  if (!hrn_controls_can_has_fade)
    {
      return;
    }

  clutter_actor_animate (master_controls, CLUTTER_LINEAR, CONTROLS_IN_DURATION,
                         "opacity", 255,
                         NULL);
  controls_hide_handler_id =
    g_timeout_add (HRN_CONTROLS_TIMEOUT, controls_hide_handler,
                   NULL);
}

static gboolean
controls_capture_cb (ClutterActor *actor, ClutterEvent *event, gpointer data)
{
  switch (event->any.type)
    {
      case CLUTTER_LEAVE:
        if (0)
          {
            ClutterActor *actor = event->motion.source;

            g_print ("leave.. %p %p ",
                     (event->motion.source),
                     (event->crossing.related));

            if (event->crossing.related)
              g_print ("leave.. %s %s|",
                       G_OBJECT_TYPE_NAME (event->motion.source),
                       G_OBJECT_TYPE_NAME (event->crossing.related));
            while (actor)
              {
                g_print ("%s<", G_OBJECT_TYPE_NAME (actor));
                actor = clutter_actor_get_parent (actor);
              }
            g_print ("\n");
          }
        break;

      case CLUTTER_KEY_PRESS:
      case CLUTTER_KEY_RELEASE:

        if (event->key.keyval == CLUTTER_q &&
            clutter_event_get_state (event) & CLUTTER_CONTROL_MASK)
          {
            hrn_quit ();
          }

      /* some keybindings should not wake the controls */
        if (event->key.keyval == CLUTTER_Shift_L ||
            event->key.keyval == CLUTTER_Shift_R ||
            event->key.keyval == CLUTTER_space)
          return FALSE;
      case CLUTTER_MOTION:
#if 0
       /* debug code used when picking goes awry */
          {
            ClutterActor *actor = event->motion.source;

            while (actor)
              {
                g_print ("%s<", G_OBJECT_TYPE_NAME (actor));
                actor = clutter_actor_get_parent (actor);
              }
            g_print ("\n");
          }
#endif

      case CLUTTER_BUTTON_PRESS:
      case CLUTTER_BUTTON_RELEASE:
      case CLUTTER_SCROLL:
        hrn_controls_wake ();

      default:
        if (0) g_print ("%i %s %p\n", event->type,
                        G_OBJECT_TYPE_NAME (clutter_event_get_source (event)),
                        clutter_event_get_source (event));
        break;
    }
  return FALSE;
}

void
zoom_changed (NbtkAdjustment *adjustment, GParamSpec     *pspec, gpointer data)
{
  gdouble current = nbtk_adjustment_get_value (adjustment);

  current = (current * 4) + 0.5;

  if (zoom_level == (int) current)
    {
      return;
    }

  zoom_level = current;

  switch ((gint)(current))
    {
      case 0: hrn_set_zoom (0.5); break;
      case 1: hrn_set_zoom (1.0); break;
      case 2: hrn_set_zoom (2.0); break;
      case 3: hrn_set_zoom (4.0); break;
      default: hrn_set_zoom (8.0); 
               hrn_controls_wake ();
    }

  hrn_store_state ();

  hrn_toolbar_set_pinned (HRN_TOOLBAR (hrn_toolbar), FALSE);
}

static NbtkAdjustment *loc_adjustment = NULL;

void
hrn_loc_slider_update ()
{
  nbtk_adjustment_set_value (loc_adjustment, hrn_view_get_loc ());
}

gint desired_pos = 0;

static void
loc_changed (NbtkAdjustment *adjustment, gpointer view)
{
  gdouble value = nbtk_adjustment_get_value (adjustment);

  if (!(value >= 0.0 && value <= 1.0))
    {
      value = 0.0;
      nbtk_adjustment_set_value (adjustment, value);
    }

  hrn_view_set_loc (value);

  desired_pos = hrn_view_get_focused (HRN_VIEW (hrn_view));

}




void sidebar_init (void); /*XXX: refactor into something sane */

static gboolean
theatre_key_press_cb (ClutterActor *actor, ClutterEvent *event,
                      gpointer userdata)
{
  switch (clutter_event_get_key_symbol ((ClutterEvent *) event))
    {
      case CLUTTER_space:
        hrn_queue_set_playing (master_queue,
                               !hrn_queue_get_playing (master_queue));
        break;

      default:
        break;
    }

  return FALSE;
}

static void
hrn_build_ui (void)
{
  ClutterActor *stage = clutter_stage_get_default ();

  clutter_group_add (stage, hrn_content_area = clutter_group_new ());
  clutter_group_add (CLUTTER_GROUP (
                       hrn_content_area), hrn_view_parent = clutter_group_new ());
  clutter_group_add (CLUTTER_GROUP (
                       hrn_content_area), hrn_theatre_lowlight =
                       clutter_group_new ());
  clutter_group_add (CLUTTER_GROUP (
                       hrn_content_area), hrn_theatre_parent =
                       clutter_group_new ());
  clutter_group_add (CLUTTER_GROUP (
                       hrn_theatre_parent), hrn_theatre_underlay =
                       clutter_group_new ());
  clutter_group_add (CLUTTER_GROUP (
                       hrn_theatre_parent), hrn_theatre =
                       CLUTTER_ACTOR (hrn_theatre_new (state_manager)));
  clutter_group_add (stage, hrn_controls = clutter_group_new ());

  /* Connect to this here to trap the Space keypress because the theatre
     doesn't know (or care) whether its playing or not */
  g_signal_connect (hrn_theatre, "key-press-event",
                    G_CALLBACK (theatre_key_press_cb), NULL);


  clutter_group_add (CLUTTER_GROUP (
                       hrn_controls), hrn_controls_frame = clutter_group_new ());
  clutter_group_add (CLUTTER_GROUP (
                       hrn_controls), hrn_scrollbar = clutter_group_new ());
  clutter_group_add (CLUTTER_GROUP (hrn_controls), hrn_sidebar =
                       CLUTTER_ACTOR (hrn_sidebar_new ()));
  clutter_group_add (CLUTTER_GROUP (
                       hrn_controls), hrn_queues = clutter_group_new ());

  clutter_group_add (CLUTTER_GROUP (hrn_controls), hrn_toolbar =
                       (ClutterActor *) hrn_toolbar_new ());

  {
    ClutterActor *bg    = clutter_rectangle_new ();
    ClutterColor  black = { 0x00, 0x00, 0x00, 0xff };

    clutter_group_add (hrn_theatre_lowlight, bg);
    clutter_actor_hide (hrn_theatre_lowlight);
    clutter_actor_set_opacity (hrn_theatre_lowlight, 0);
    clutter_rectangle_set_color (CLUTTER_RECTANGLE (bg), &black);
    clutter_actor_set_position (bg, -400, -400);
    clutter_actor_set_size (bg, 4096, 4096);    /* hard coded size that should
                                                   guarantee
                                                   the rectangle covering up
                                                   even with huge
                                                   stages */
  }

#define set_dummy_asset_full(group, file, x, y, opacity) \
  { ClutterActor *img = clutter_texture_new_from_file (file, NULL); \
    clutter_group_add (CLUTTER_GROUP (group), img); \
    clutter_actor_set_position (img, x, y); \
    clutter_actor_set_opacity (img, opacity); }

#define set_dummy_asset(group, file, x, y) \
  set_dummy_asset_full (group, file, x, y, 255)

  clutter_actor_set_y (hrn_sidebar, 62);

  clutter_actor_set_position (hrn_queues, clutter_actor_get_width (stage) - 335,
                              clutter_actor_get_height (stage) - 60);

  if (1)
    {
      HrnQueue *queue = hrn_queue_new (local_queue);

      /* FIXME: We need some sort of queue manager here */
      master_queue = queue;

      clutter_group_add (hrn_queues, queue);

    }


  if (1)
    {
      HrnControls *controls = hrn_controls_new (local_queue);

      master_controls = CLUTTER_ACTOR (controls);
      clutter_group_add (CLUTTER_GROUP (hrn_theatre_parent), master_controls);
      clutter_actor_set_position (master_controls, 0, clutter_actor_get_height(stage)-60);
      clutter_actor_hide (hrn_theatre_parent);
      clutter_actor_hide (master_controls);
    }

  {
    NbtkWidget *scroll;
    loc_adjustment = nbtk_adjustment_new (0.0, 0.0, 1.0, 0.2, 0.00, 0.0);

    scroll = NBTK_WIDGET (nbtk_scroll_bar_new (NULL));
    clutter_group_add (CLUTTER_GROUP (hrn_scrollbar), scroll);
    clutter_actor_set_size (CLUTTER_ACTOR (scroll),
                            clutter_actor_get_height (stage) - 62, 22);
    clutter_actor_set_rotation (CLUTTER_ACTOR (
                                  scroll), CLUTTER_Z_AXIS, 90, 0, 0, 0);
    clutter_actor_set_position (CLUTTER_ACTOR (scroll),
                                clutter_actor_get_width (stage) + 2, 62);

    nbtk_scroll_bar_set_adjustment (NBTK_SCROLL_BAR (scroll), loc_adjustment);
    g_signal_connect (loc_adjustment, "notify::value", G_CALLBACK (
                        loc_changed), hrn_view);
  }


  g_signal_connect (stage, "captured-event", G_CALLBACK (
                      controls_capture_cb), NULL);

  sidebar_init ();

  clutter_actor_hide (hrn_theatre);
}

#define CLUSTERS    10



static gboolean
key_pressed (ClutterActor *actor, ClutterEvent *event, gpointer data)
{
  switch (clutter_event_get_key_symbol ((ClutterEvent *) event))
    {
      case CLUTTER_Return:
        break;

      case CLUTTER_Page_Up:
        hrn_view_scroll (-1.0);
        break;

      case CLUTTER_Page_Down:
        hrn_view_scroll (1.0);
        break;

      case CLUTTER_Up:
        hrn_view_scroll (-0.5);
        break;

      case CLUTTER_Down:
        hrn_view_scroll (0.5);
        break;

      case CLUTTER_Escape:
        if (hrn_get_zoom () > 0.5)
           hrn_set_zoom (hrn_get_zoom () / 2.0);
        break;

      case CLUTTER_Right:
        /*target_scale *= 1.4142;*/
        if (hrn_theatre_get_active (HRN_THEATRE (hrn_theatre)))
          {
          }
        else
          {
            hrn_set_zoom (hrn_get_zoom () * 2.0);
          }
        break;

      case CLUTTER_Left:
        if (hrn_theatre_get_active (HRN_THEATRE (hrn_theatre)))
          {
            /* this goes to the context/thumbbar item left of
             * the current one
             */
          }
        else
          {
            if (hrn_get_zoom () > 0.5)
              hrn_set_zoom (hrn_get_zoom () / 2.0);
          }
        break;

      default:
        break;
    }
  return FALSE;
}

static gchar *load_path = NULL;

static gboolean
auto_zoomer (gpointer userdata)
{
  static gfloat target_zoom = 0.0;

  if (hrn_get_scale () != target_zoom && target_zoom > 0.0)
    {
    }
  else
    {
      if (target_zoom == 8.0)
        {
          target_zoom = 0.5;
        }
      else
        {
          target_zoom = 8.0;
        }
      hrn_texture_cache_evict_items (hrn_texture_cache_get_default (), 1000);
      hrn_set_zoom (target_zoom);
    }
  return TRUE;
}

gboolean hrn_in_preload = FALSE;
static gboolean preload_done = FALSE;

static gboolean
hrn_preload_data (gpointer userdata)
{
  static gint iteration = 0;

  iteration ++;
  /* Do test renders at specific zoom levels, this forces
   * hornsey to have all the needed data for the transitions
   * cached.
   */
  switch (iteration)
    {
      case 1: hrn_set_zoom_hard (0.75); break;
      case 2: hrn_set_zoom_hard (1.25); break;
      case 3: hrn_set_zoom_hard (1.88); break;
      case 4: hrn_set_zoom_hard (2.0); break;
      case 5: hrn_set_zoom_hard (2.1); break;
      case 6: hrn_view_set_loc (0.4); hrn_set_zoom_hard (1.25); break;
      case 7: hrn_set_zoom_hard (1.88); break;
      case 8: hrn_set_zoom_hard (2.0); break;
      case 9: hrn_set_zoom_hard (1.0); break;
      default:
         hrn_view_set_loc (0.0);
	 hrn_set_zoom_hard (1.0);   /* this is the default starting zoom level */
	 preload_done = TRUE;
	 return FALSE;
    }
  clutter_actor_paint (hrn_view); /* this is needed to force a full
                                     pregeneration cycle */
  return TRUE;
}

static void
check_all_sources_ready (void)
{
  if (source_replies == source_count &&
      local_source != NULL)
    {
      hrn_state_manager_ready (state_manager);

      if (hrn_state_manager_get_force_visual_mode (state_manager)) {
         hrn_set_theatre ();
      }

      hrn_view_start ();

      /* maybe the following two calls should be removed */
      hrn_view_set_source (HRN_VIEW (hrn_view), local_source);
      hrn_view_set_filter (HRN_VIEW (hrn_view), BKL_ITEM_TYPE_IMAGE |
                           BKL_ITEM_TYPE_VIDEO |
                           BKL_ITEM_TYPE_AUDIO);

      if (load_path)
        {
          BklItem *item;

          g_debug ("Hornsey commandline URI: %s", load_path);

          item = hrn_get_item_for_uri (load_path);
          hrn_set_theatre ();
          if (item)
            {
              hrn_play_now (item);
            }
          else
            {
              hrn_play_uri_now (load_path);
            }
        }
      else if (!hrn_state_manager_get_force_visual_mode (state_manager))
        { /* If were not launching into playback mode assume
             browsing and need for preloading */
	   ClutterActor *block;
           gchar *path = g_strdup_printf ("%s/.hrn-queries", g_get_home_dir ());
           gboolean has_queries_file = g_file_test (path, G_FILE_TEST_EXISTS);
           g_free (path);


           block = clutter_rectangle_new ();
           hrn_in_preload = TRUE;
           if (has_queries_file)
             hrn_retrieve_query ("hrn-last-state");
           clutter_actor_set_size (block, 4000, 4000);
           clutter_group_add (hrn_content_area, block);

           g_idle_add (hrn_preload_data, NULL);
           while (preload_done == FALSE)
             g_main_context_iteration (NULL, FALSE);

           clutter_actor_destroy (block);
           hrn_in_preload = FALSE;


           {
             gchar *path;
             path = g_strdup_printf ("%s/.hrn-queries", g_get_home_dir ());
             if (has_queries_file)
               hrn_retrieve_query ("hrn-last-state");
             else
               hrn_set_zoom_hard (1.0);
             g_free (path);
           }
        }
      {
        SnLauncheeContext *context;

        context = g_object_get_data (G_OBJECT (clutter_stage_get_default()),
                                     "sn-context");
      if (context)
        {
          sn_launchee_context_complete (context);
          sn_launchee_context_unref (context);
        }
      }

      hrn_queue_populate (master_queue);
      hrn_controls_populate (HRN_CONTROLS (master_controls));

      if (hrn_auto_zoom)
        g_timeout_add (50, auto_zoomer, NULL);

#if 0
      if (load_path == NULL) {
          /* The state needs to be restored in an idle because of a bug in NBTK
             that the handle position of the zoombar gets set before the zoombar
             has an allocation, so the position is always set to 0.0 no matter
             what the actual Adjustment says:

             http://bugzilla.moblin.org/show_bug.cgi?id=3691 */
          g_idle_add (restore_state_idle, NULL);
      }
#endif
    }
}

static void
source_ready_cb (BklSourceClient *source_client, gpointer userdata)
{
  HrnSource *source;

  source_replies++;

  source = hrn_source_new (source_client);
  if (source)
    {
      transient_sources = g_list_prepend (transient_sources,
                                          source);
      hrn_sidebar_update_sources (HRN_SIDEBAR (hrn_sidebar));
    }
  else
    {
      g_warning ("Could not create source for '%s'",
                 bkl_source_client_get_path (source_client));
    }

  /* HrnSource takes its own reference, we want to unref whether
     hrn_source_new() succeeded or not */
  g_object_unref (source_client);

  check_all_sources_ready ();
}

static void
source_added (BklSourceManagerClient *client,
              const char             *object_path,
              gpointer                userdata)
{
  BklSourceClient *source;

  source = bkl_source_client_new (object_path);
  g_signal_connect (source, "ready", G_CALLBACK (source_ready_cb), NULL);
}

static void
source_removed (BklSourceManagerClient *client,
                const char             *object_path,
                gpointer                userdata)
{
  GList *s;

  for (s = transient_sources; s; s = s->next)
    {
      HrnSource *source = s->data;

      if (g_str_equal (object_path,
                       hrn_source_get_object_path (source)))
        {
          if (hrn_view_get_source (HRN_VIEW (hrn_view)) == source)
            {
              hrn_view_set_source (HRN_VIEW (hrn_view), hrn_get_local_source ());
              hrn_view_set_search (HRN_VIEW (hrn_view), "");
              hrn_set_zoom_hard (0.5);
              hrn_view_set_filter (HRN_VIEW (hrn_view), BKL_ITEM_TYPE_IMAGE|
                                                        BKL_ITEM_TYPE_AUDIO|
                                                        BKL_ITEM_TYPE_VIDEO);
            }
          g_object_unref (source);
          transient_sources = g_list_delete_link (transient_sources, s);
          break;
        }
    }

  hrn_sidebar_update_sources (HRN_SIDEBAR (hrn_sidebar));
}

HrnSource *
hrn_get_local_source (void)
{
  return local_source;
}

GList *
hrn_get_transient_sources (void)
{
  return transient_sources;
}

HrnSource *
hrn_get_source_for_path (const char *path)
{
  GList *s;

  for (s = transient_sources; s; s = s->next)
    {
      HrnSource *source = s->data;

      if (g_str_equal (path, hrn_source_get_object_path (source)))
        {
          return source;
        }
    }

  return NULL;
}

BklItem *
hrn_get_item_for_uri (const char *uri)
{
  BklItem *item = NULL;
  GList   *s;

  /* Check local source first */
  item = hrn_source_get_item (local_source, uri);
  if (item)
    {
      return item;
    }

  /* Check transients */
  for (s = transient_sources; s; s = s->next)
    {
      HrnSource *source = s->data;

      item = hrn_source_get_item (source, uri);
      if (item)
        {
          return item;
        }
    }

  return NULL;
}

static void
local_source_ready (BklSourceClient *client)
{
  local_source = hrn_source_new (client);
  if (local_source)
    {
      hrn_sidebar_update_sources (HRN_SIDEBAR (hrn_sidebar));
    }
  else
    {
      g_warning ("Could not create a source for '%s'",
                 bkl_source_client_get_path (client));
    }

  /* HrnSource takes its own reference, we want to unref whether
     hrn_source_new() succeeded or not */
  g_object_unref (client);

  check_all_sources_ready ();
}

static void
get_sources_reply (BklSourceManagerClient *client, GList *sources,
                   GError                 *error,
                   gpointer userdata)
{
  BklSourceClient *ls;
  GList           *s;

  if (error != NULL)
    {
      g_warning ("Error getting sources: %s", error->message);
      g_error_free (error);
    }

  ls = bkl_source_client_new (BKL_LOCAL_SOURCE_PATH);
  g_signal_connect (ls, "ready", G_CALLBACK (local_source_ready), ls);

  for (s = sources; s; s = s->next)
    {
      source_count++;
      source_added (manager, s->data, NULL);
    }

  g_list_free (sources);
}

static void
setup_sources (BklSourceManagerClient *manager_client, gpointer userdata)
{
  bkl_source_manager_client_get_sources (manager, get_sources_reply, NULL);
}

void
hrn_play_uri_now (const gchar *uri)
{
  hrn_queue_play_uri_now (master_queue, uri);
}

void
hrn_play_now (BklItem *item)
{
  hrn_queue_play_now (master_queue, item);
}

void
hrn_stage_remove_decorations (ClutterStage *stage)
{
  struct
  {
    unsigned long flags;
    unsigned long functions;
    unsigned long decorations;
    long          inputMode;
    unsigned long status;
  }        MWMHints = { 2, 0, 0, 0, 0 };

  Display *xdisplay = clutter_x11_get_default_display ();
  Atom     wm_hints = XInternAtom (xdisplay, "_MOTIF_WM_HINTS", True);
  Window   xwindow  = clutter_x11_get_stage_window (CLUTTER_STAGE (stage));

  if (wm_hints != None)
    {
      XChangeProperty (xdisplay, xwindow, wm_hints, wm_hints, 32,
                       PropModeReplace, (guchar*) &MWMHints,
                       sizeof (MWMHints) / sizeof (long));
    }
}

static void
dbus_init (HrnStateManager *sm)
{
  DBusGConnection *connection;
  GError          *error = NULL;

  connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error);
  if (connection == NULL)
    {
      g_warning ("Error getting bus: %s", error->message);
      g_error_free (error);
      return;
    }

  /* We dont need to request the horney name as libunique has already
     done that for us */

  dbus_g_connection_register_g_object (connection, HORNSEY_DBUS_PATH,
                                       (GObject *) sm);
}

NbtkAdjustment *zoom_adjustment;

gint            DIM;
gboolean        hrn_auto_zoom = FALSE;

gint
main (gint argc, gchar **argv)
{
  UniqueApp   *app;
  ClutterColor stage_bg         = { HRN_STAGE_COLOR };
  gfloat       stage_width      = 0;
  gfloat       stage_height     = 0;
#ifdef HAVE_WINDOWMODE
  gboolean     stage_fullscreen = FALSE;
#else
  gboolean     stage_fullscreen = TRUE;
#endif
  gboolean     no_disclaimer    = TRUE; /* change this to FALSE
                                           and the annoying disclaimer
                                           will show on startup */
  SnLauncheeContext *context    = NULL;
  ClutterActor      *stage;

  DIM = 240;

  bindtextdomain (GETTEXT_PACKAGE, LOCALEDIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  textdomain (GETTEXT_PACKAGE);

  g_set_application_name ("Hornsey Media Player");

  clutter_init (&argc, &argv);

  stage = clutter_stage_get_default ();

  {
    SnDisplay *display;
    Display   *xdisplay;
    gint       screen;

    xdisplay = clutter_x11_get_default_display ();

    if (g_getenv ("LIBSN_SYNC") != NULL)
      XSynchronize (xdisplay, True);

    display = sn_display_new (xdisplay, NULL, NULL);
    screen  = clutter_x11_get_default_screen ();
    context = sn_launchee_context_new_from_environment (display, screen);
    if (context)
      {
        Window xwindow;
        xwindow = clutter_x11_get_stage_window ((void*) stage);
        sn_launchee_context_setup_window (context, xwindow);
        g_object_set_data (G_OBJECT (stage), "sn-context", context);
      }
  }

  g_thread_init (NULL);
  bkl_init ();
  notify_init ("Hornsey Media Player");
  clutter_gst_init (&argc, &argv);
  clutter_set_font_flags (CLUTTER_FONT_HINTING);

  gtk_init (&argc, &argv);
  app = unique_app_new ("org.moblin.Hornsey", NULL);

  if (unique_app_is_running (app))
    {
      if (argc < 2)
        {
          unique_app_send_message (app, UNIQUE_ACTIVATE, NULL);
        }
      else
        {
          UniqueMessageData *data = unique_message_data_new ();
          unique_message_data_set_text (data, argv[1], -1);
          unique_app_send_message (app, UNIQUE_OPEN, data);
          unique_message_data_free (data);
        }
      return 0;
    }

  state_manager = g_object_new (HRN_TYPE_STATE_MANAGER, NULL);
  dbus_init (state_manager);

  nbtk_style_load_from_file (nbtk_style_get_default (),
                             PKGDATADIR "hornsey.css", NULL);

  {
    gchar **arg = argv + 1;
    while (*arg)
      {
        if (g_str_equal (*arg, "-h") ||
            g_str_equal (*arg, "--help"))
          {
            g_print (
              "hornsey [-s WIDTHxHEIGHT] [-w] URI\n\n\n-w causes windowed mode (as opposed to default fullscreen)\npass -nd to disable disclaimer.");
            return 0;
          }
        else if (g_str_equal (*arg, "-s"))
          {
            arg++; g_assert (*arg);
            stage_width = atoi (*arg);
            if (strstr (*arg, "x"))
              stage_height = atoi (strstr (*arg, "x") + 1);
          }
        else if (g_str_equal (*arg, "-fs"))
          {
            stage_fullscreen = TRUE;
          }
        else if (g_str_equal (*arg, "-nd"))
          {
            no_disclaimer = TRUE;
          }
        else if (g_str_equal (*arg, "-w"))
          {
            stage_fullscreen = FALSE;
          }
        else if (g_str_equal (*arg, "-a"))
          {
            hrn_auto_zoom = TRUE;
          }
        else
          {
            load_path = *arg;
          }
        arg++;
      }
  }

  local_queue = g_object_new (BR_TYPE_QUEUE,
                              "object_path", BR_LOCAL_QUEUE_PATH,
                              NULL);

  if (!stage_fullscreen)
    {
      stage_width  = 800;
      stage_height = 480;
    }
  else if (stage_width == 0 &&
           stage_height == 0)
    {
      /* This causes us to read back the dimensions of the X desktop
       * using only Clutter APIs
       */
      stage_width  = gdk_screen_get_width (gdk_screen_get_default ());
      stage_height = gdk_screen_get_height (gdk_screen_get_default ());
    }

  g_object_set (G_OBJECT (stage),
                "width", stage_width,
                "height", stage_height,
                "color", &stage_bg,
                NULL);

  /* remove decorations */
  if (stage_fullscreen)
    {
      hrn_stage_remove_decorations (CLUTTER_STAGE (stage));
    }

  DIM = clutter_actor_get_width (CLUTTER_ACTOR (stage)) / 4.25;

  clutter_actor_show (stage);
  clutter_actor_realize (stage);

  zoom_adjustment = nbtk_adjustment_new (0.0, 0.0, 1.0, 0.2, 0.25, 0.0);
  g_signal_connect (zoom_adjustment, "notify::value", G_CALLBACK (
                      zoom_changed), hrn_view);

  pinned_searches = g_key_file_new ();

  {
    gchar *search_path = g_strdup_printf ("%s/.hrn-queries", g_get_home_dir ());

    g_key_file_load_from_file (pinned_searches, search_path,
                               G_KEY_FILE_KEEP_COMMENTS,
                               NULL);
    g_free (search_path);
  }
  hrn_view = hrn_view_new ();
  hrn_build_ui ();

  /* Setup the Bickley Source manager */
  manager = g_object_new (BKL_TYPE_SOURCE_MANAGER_CLIENT, NULL);
  g_signal_connect (manager, "ready", G_CALLBACK (setup_sources), NULL);

  g_signal_connect (manager, "source-added",
                    G_CALLBACK (source_added), NULL);
  g_signal_connect (manager, "source-removed",
                    G_CALLBACK (source_removed), NULL);
  hrn_sidebar_update_sources (HRN_SIDEBAR (hrn_sidebar));


#if HRN_DEBUG_OFF_STAGE
  {
    ClutterColor  fake_stage_color   = { 0x00, 0x00, 0x00, 0x00 };
    ClutterColor  stage_border_color = { 0xff, 0x66, 0x22, 0x22 };
    ClutterActor *fake_stage;
    fake_stage = clutter_rectangle_new ();
    clutter_actor_set_size (fake_stage, stage_width, stage_height);
    clutter_rectangle_set_border_color (CLUTTER_RECTANGLE (
                                          fake_stage), &stage_border_color);
    clutter_rectangle_set_border_width (CLUTTER_RECTANGLE (fake_stage), 20);
    clutter_rectangle_set_color (CLUTTER_RECTANGLE (
                                   fake_stage), &fake_stage_color);
    clutter_group_add (CLUTTER_GROUP (hrn_content_area), fake_stage);
    clutter_actor_set_depth (hrn_view, HRN_DEBUG_DEPTH);
    clutter_actor_set_depth (fake_stage, HRN_DEBUG_DEPTH);
  }
#endif

  clutter_group_add (CLUTTER_GROUP (hrn_view_parent), hrn_view);
  g_signal_connect (stage, "key-press-event", G_CALLBACK (
                      key_pressed), hrn_view);

  clutter_stage_set_title (CLUTTER_STAGE (stage), _ ("Media Player"));

  if (!no_disclaimer) hrn_disclaimer (
      "This application is in flux, and is included in the build for development testing purposes. Bugs filed against it will be ignored for now.");



  g_signal_connect (app, "message-received",
                    G_CALLBACK (hrn_message_received_cb), NULL);


  clutter_main ();

  /* Tell Bognor that we can't show visuals anymore */
  br_iface_player_emit_can_show_visual_changed (BR_IFACE_PLAYER (state_manager),
                                                FALSE);

  bkl_shutdown ();

  return 0;
}
