#include <nbtk/nbtk.h>

#include "hrn.h"

#include "hrn-cluster-node.h"
#include "hrn-cluster.h"
#include "hrn-cluster-thumbnail.h"
#include "hrn-album-cluster.h"
#include "hrn-month-cluster.h"
#include "hrn-tiler.h"
#include "hrn-tileable.h"
#include "hrn-tile-frame.h"
#include "hrn-image-tile.h"
#include "hrn-track-tile.h"
#include "hrn-texture-cache.h"
#include "hrn-up-tile.h"

enum {
    PROP_0,
    PROP_X_BASE,
    PROP_Y_BASE,
    PROP_PRIMARY,
    PROP_SECONDARY,
    PROP_NODE,
};

enum {
    LAST_SIGNAL,
};

struct _HrnClusterPrivate {
    HrnClusterNode *node;
    GHashTable *child_items;

    gboolean items_exist;
    guint items_per_row;

    NbtkWidget *table;

    HrnUpTile *up_tile;

    GPtrArray *children;
    guint visible_children;
    gboolean expanded;
    HrnTileable *shown_child;

    guint count;

    ClutterActor *frame;
    ClutterActor *thumbnail;
    NbtkWidget *primary;
    NbtkWidget *secondary;

    guint32 thumb_timeout_id;
    int base_x, base_y;
};

#define GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), HRN_TYPE_CLUSTER, HrnClusterPrivate))

static void tiler_interface_init (HrnTilerInterface *iface);
static void tileable_interface_init (HrnTileableInterface *iface);
G_DEFINE_TYPE_WITH_CODE (HrnCluster, hrn_cluster, CLUTTER_TYPE_GROUP,
                         G_IMPLEMENT_INTERFACE (HRN_TYPE_TILER,
                                                tiler_interface_init)
                         G_IMPLEMENT_INTERFACE (HRN_TYPE_TILEABLE,
                                                tileable_interface_init));

static void child_visibility_changed_cb (HrnTileable *tileable,
                                         gboolean     hidden,
                                         HrnCluster  *cluster);

static void
hrn_cluster_finalize (GObject *object)
{
    G_OBJECT_CLASS (hrn_cluster_parent_class)->finalize (object);
}

static void
hrn_cluster_dispose (GObject *object)
{
    G_OBJECT_CLASS (hrn_cluster_parent_class)->dispose (object);
}

static void
hrn_cluster_set_property (GObject      *object,
                          guint         prop_id,
                          const GValue *value,
                          GParamSpec   *pspec)
{
    HrnCluster *self = (HrnCluster *) object;
    HrnClusterPrivate *priv = self->priv;

    switch (prop_id) {
    case PROP_X_BASE:
        priv->base_x = g_value_get_uint (value);
        break;

    case PROP_Y_BASE:
        priv->base_y = g_value_get_uint (value);
        break;

    case PROP_PRIMARY:
        nbtk_label_set_text ((NbtkLabel *) priv->primary,
                             g_value_get_string (value));
        break;

    case PROP_SECONDARY:
        nbtk_label_set_text ((NbtkLabel *) priv->secondary,
                             g_value_get_string (value));
        break;

    case PROP_NODE:
        g_warning ("Subclasses should override Node property");
        break;

    default:
        break;
    }
}

static void
hrn_cluster_get_property (GObject    *object,
                          guint       prop_id,
                          GValue     *value,
                          GParamSpec *pspec)
{
    switch (prop_id) {

    default:
        break;
    }
}

static void
hrn_cluster_paint (ClutterActor *actor)
{
    HrnCluster *cluster = (HrnCluster *) actor;
    HrnClusterPrivate *priv = cluster->priv;

    if (priv->expanded && priv->node) {
        GSequenceIter *iter = g_sequence_get_begin_iter (priv->node->children);

        clutter_actor_paint ((ClutterActor *) priv->up_tile);

        while (g_sequence_iter_is_end (iter) == FALSE) {
            HrnClusterNode *child_node = g_sequence_get (iter);
            ClutterActor *child;

            if (child_node->hidden) {
                iter = g_sequence_iter_next (iter);
                continue;
            }

            child = g_hash_table_lookup (priv->child_items, child_node);
            if (child == NULL) {
                g_warning ("Could not find child item for %s",
                           child_node->name);
                iter = g_sequence_iter_next (iter);
                continue;
            }

            clutter_actor_paint (child);
            iter = g_sequence_iter_next (iter);
        }
    } else {
        clutter_actor_paint ((ClutterActor *) priv->table);
    }
}

static void
hrn_cluster_pick (ClutterActor       *actor,
                  const ClutterColor *color)
{
    HrnCluster *cluster = (HrnCluster *) actor;
    HrnClusterPrivate *priv = cluster->priv;

    if (priv->expanded && priv->node) {
        GSequenceIter *iter = g_sequence_get_begin_iter (priv->node->children);

        clutter_actor_paint ((ClutterActor *) priv->up_tile);

        while (g_sequence_iter_is_end (iter) == FALSE) {
            HrnClusterNode *child_node = g_sequence_get (iter);
            ClutterActor *child;

            if (child_node->hidden) {
                iter = g_sequence_iter_next (iter);
                continue;
            }

            child = g_hash_table_lookup (priv->child_items, child_node);
            if (child == NULL) {
                g_warning ("Could not find child item for %s",
                           child_node->name);
                iter = g_sequence_iter_next (iter);
                continue;
            }

            clutter_actor_paint (child);
            iter = g_sequence_iter_next (iter);
        }
    } else {
        clutter_actor_paint ((ClutterActor *) priv->table);
    }
}

static void
hrn_cluster_class_init (HrnClusterClass *klass)
{
    GObjectClass *o_class = (GObjectClass *) klass;
    ClutterActorClass *a_class = (ClutterActorClass *) klass;

    o_class->dispose = hrn_cluster_dispose;
    o_class->finalize = hrn_cluster_finalize;
    o_class->set_property = hrn_cluster_set_property;
    o_class->get_property = hrn_cluster_get_property;

    a_class->paint = hrn_cluster_paint;
    a_class->pick = hrn_cluster_pick;

    g_type_class_add_private (klass, sizeof (HrnClusterPrivate));

    g_object_class_install_property (o_class, PROP_X_BASE,
                                     g_param_spec_uint ("x-base", "", "",
                                                        0, G_MAXUINT, 0,
                                                        G_PARAM_READABLE |
                                                        G_PARAM_WRITABLE |
                                                        G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_Y_BASE,
                                     g_param_spec_uint ("y-base", "", "",
                                                        0, G_MAXUINT, 0,
                                                        G_PARAM_READABLE |
                                                        G_PARAM_WRITABLE |
                                                        G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_PRIMARY,
                                     g_param_spec_string ("primary", "", "", "",
                                                          G_PARAM_READABLE |
                                                          G_PARAM_WRITABLE |
                                                          G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_SECONDARY,
                                     g_param_spec_string ("secondary", "", "", "",
                                                          G_PARAM_READABLE |
                                                          G_PARAM_WRITABLE |
                                                          G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_NODE,
                                     g_param_spec_pointer ("node", "", "",
                                                           G_PARAM_READABLE |
                                                           G_PARAM_WRITABLE |
                                                           G_PARAM_STATIC_STRINGS));
}

/* Returns true if a movement from sy -> dy crosses the stage */
static gboolean
is_actor_on_stage (ClutterActor *actor,
                   float         sy,
                   float         dy)
{
    int sh = CLUTTER_STAGE_HEIGHT ();
    float ah;

    /* Not on stage if its not visible */
    if (CLUTTER_ACTOR_IS_VISIBLE (actor) == FALSE) {
        return FALSE;
    }

    ah = clutter_actor_get_height (actor);
    if (sy + ah < 0 && dy + ah < 0) {
        return FALSE;
    } else if (sy > sh && dy > sh) {
        return FALSE;
    } else {
        return TRUE;
    }
}

#define X_GRID_TO_COORDS(x) (x * (ITEM_WIDTH + COL_GAP))
#define Y_GRID_TO_COORDS(y) (y * (ITEM_HEIGHT + ROW_GAP))

static void
layout_cluster (HrnCluster *cluster)
{
    HrnClusterPrivate *priv = cluster->priv;
    int item_count;
    guint row_count;
    float ay;

    /* Plus 1 because we have the Up Tile at 0,0 */
    item_count = (priv->base_y * priv->items_per_row) + priv->base_x + 1;
    ay = clutter_actor_get_y ((ClutterActor *) cluster);

    if (priv->node) {
        GSequenceIter *iter = g_sequence_get_begin_iter (priv->node->children);

        while (g_sequence_iter_is_end (iter) == FALSE) {
            HrnClusterNode *child_node = g_sequence_get (iter);
            ClutterActor *child;
            float gx, gy, iy;
            int x, y;

            if (child_node->hidden) {
                iter = g_sequence_iter_next (iter);
                continue;
            }

            child = g_hash_table_lookup (priv->child_items, child_node);
            if (child == NULL) {
                g_warning ("No child found for %s", child_node->name);
                iter = g_sequence_iter_next (iter);
                continue;
            }

            x = (item_count % priv->items_per_row) - priv->base_x;
            y = (item_count / priv->items_per_row) - priv->base_y;

            gx = X_GRID_TO_COORDS (x);
            gy = Y_GRID_TO_COORDS (y);

            iy = clutter_actor_get_y (child);

            hrn_tileable_set_position ((HrnTileable *) child, x, y);

            if (is_actor_on_stage (child, ay + iy, ay + gy)) {
                clutter_actor_animate (child, CLUTTER_EASE_IN_OUT_CUBIC, 500,
                                       "x", gx,
                                       "y", gy,
                                       NULL);
            } else {
                clutter_actor_set_position (child, gx, gy);
            }

            item_count += hrn_tileable_get_count ((HrnTileable *) child);

            iter = g_sequence_iter_next (iter);
        }
    }

    /* How many rows do we cover, to set the size of the group correctly */
    row_count = 1 + ((item_count + (priv->children->len - 1)) / priv->items_per_row) - priv->base_y;
    clutter_actor_set_size ((ClutterActor *) cluster,
                            priv->items_per_row * (ITEM_WIDTH + COL_GAP),
                            row_count * (ITEM_HEIGHT + ROW_GAP));
}

static void
tiler_show_all (HrnTiler *tiler)
{
    HrnCluster *cluster = (HrnCluster *) tiler;
    HrnClusterPrivate *priv = cluster->priv;
    GSequenceIter *iter;

    if (priv->node == NULL) {
        return;
    }

    iter = g_sequence_get_begin_iter (priv->node->children);
    while (g_sequence_iter_is_end (iter) == FALSE) {
        HrnClusterNode *child_node = g_sequence_get (iter);
        ClutterActor *child = NULL;

        if (priv->expanded) {
            child = g_hash_table_lookup (priv->child_items, child_node);
            if (child == NULL) {
                g_warning ("%s No child found for %s",
                           G_STRLOC, child_node->name);
                iter = g_sequence_iter_next (iter);
                continue;
            }

            clutter_actor_show (child);
            clutter_actor_animate (child, CLUTTER_EASE_IN_OUT_CUBIC, 500,
                                   "opacity", 0xff,
                                   NULL);

            if (HRN_IS_TILER (child)) {
                hrn_tiler_show_all ((HrnTiler *) child);
            }
        }

        if (child) {
            g_signal_handlers_block_by_func (child,
                                             child_visibility_changed_cb,
                                             cluster);
        }

        hrn_cluster_node_set_hidden (child_node, FALSE);

        if (child) {
            g_signal_handlers_unblock_by_func (child,
                                               child_visibility_changed_cb,
                                               cluster);
        }

        iter = g_sequence_iter_next (iter);
    }
}

static void
tiler_show (HrnTiler    *tiler,
            HrnTileable *tile)
{
}

static void
add_item (HrnCluster     *cluster,
          HrnClusterNode *child)
{
    HrnClusterPrivate *priv = cluster->priv;
    ClutterActor *actor;

    switch (child->type) {
    case HRN_CLUSTER_NODE_TYPE_ALBUM:
        actor = g_object_new (HRN_TYPE_ALBUM_CLUSTER, NULL);
        hrn_tiler_add_items ((HrnTiler *) actor, child);
        hrn_tiler_set_items_per_row ((HrnTiler *) actor,
                                     priv->items_per_row);
        hrn_cluster_full ((HrnCluster *) actor);
        break;

    case HRN_CLUSTER_NODE_TYPE_TRACK:
        actor = g_object_new (HRN_TYPE_TRACK_TILE, "node", child, NULL);
        break;

    case HRN_CLUSTER_NODE_TYPE_MONTH:
        actor = g_object_new (HRN_TYPE_MONTH_CLUSTER, NULL);
        hrn_tiler_add_items ((HrnTiler *) actor, child);
        hrn_tiler_set_items_per_row ((HrnTiler *) actor,
                                     priv->items_per_row);
        hrn_cluster_full ((HrnCluster *) actor);
        break;

    case HRN_CLUSTER_NODE_TYPE_IMAGE:
        actor = g_object_new (HRN_TYPE_IMAGE_TILE, "node", child, NULL);
        break;

    default:
        return;
    }

    hrn_cluster_add_actor (cluster, (HrnTileable *) actor);
    g_hash_table_insert (priv->child_items, child, actor);
}

static void
create_items (HrnCluster *cluster)
{
    HrnClusterPrivate *priv = cluster->priv;
    HrnClusterNode *node = hrn_tileable_get_node ((HrnTileable *) cluster);
    GSequenceIter *iter = g_sequence_get_begin_iter (node->children);

    priv->up_tile = g_object_new (HRN_TYPE_UP_TILE, NULL);
    hrn_cluster_add_actor (cluster, (HrnTileable *) priv->up_tile);

    while (g_sequence_iter_is_end (iter) == FALSE) {
        HrnClusterNode *child = g_sequence_get (iter);

        add_item (cluster, child);

        iter = g_sequence_iter_next (iter);
    }
    priv->items_exist = TRUE;
}

static void
destroy_items (HrnCluster *cluster)
{
    HrnClusterPrivate *priv = cluster->priv;
    int i;

    /* set the up_tile to NULL, as it will be destroyed when all the children
     * are destroyed */
    priv->up_tile = NULL;

    for (i = 0; i < priv->children->len; i++) {
        ClutterActor *actor = priv->children->pdata[i];

        clutter_actor_destroy (actor);
    }
    g_hash_table_remove_all (priv->child_items);
    g_ptr_array_free (priv->children, TRUE);
    priv->children = g_ptr_array_new ();

    priv->items_exist = FALSE;
}

static void
tiler_set_expanded (HrnTiler *tiler,
                    gboolean  expanded)
{
    HrnCluster *cluster = HRN_CLUSTER (tiler);
    HrnClusterPrivate *priv = cluster->priv;
    int i, row_count, old_count;
    float ay;

    if (priv->expanded == expanded) {
        return;
    }

    if (priv->node == NULL) {
        return;
    }

    priv->expanded = expanded;

    ay = clutter_actor_get_y ((ClutterActor *) cluster);

    if (expanded) {
        GSequenceIter *iter;
        int item_count;

        if (priv->items_exist == FALSE) {
            create_items (cluster);
        }

        clutter_actor_animate ((ClutterActor *) priv->table,
                               CLUTTER_EASE_IN_OUT_CUBIC, 500,
                               "opacity", 0x00,
                               "signal-swapped::completed",
                               clutter_actor_hide, priv->table,
                               NULL);

        item_count = (priv->base_y * priv->items_per_row) + priv->base_x;
        old_count = priv->count;
        priv->count = 0;

        iter = g_sequence_get_begin_iter (priv->node->children);
        i = 1; /* 1 because the up tile is at 0,0 */
        while (g_sequence_iter_is_end (iter) == FALSE) {
            HrnClusterNode *child_node = g_sequence_get (iter);
            ClutterActor *child;
            int x, y;
            float gx, gy, iy;

            if (child_node->hidden) {
                iter = g_sequence_iter_next (iter);
                continue;
            }

            child = g_hash_table_lookup (priv->child_items, child_node);
            if (child == NULL) {
                g_warning ("%s No child found for %s",
                           G_STRLOC, child_node->name);
                iter = g_sequence_iter_next (iter);
                continue;
            }

            x = ((item_count + i) % priv->items_per_row) - priv->base_x;
            y = ((item_count + i) / priv->items_per_row) - priv->base_y;

            gx = (float) X_GRID_TO_COORDS (x);
            gy = (float) Y_GRID_TO_COORDS (y);

            iy = clutter_actor_get_y (child);

            clutter_actor_show (child);
            if (is_actor_on_stage (child, ay + iy, gy + ay)) {
                clutter_actor_animate (child, CLUTTER_EASE_IN_OUT_CUBIC, 500,
                                       "x", gx,
                                       "y", gy,
                                       "opacity", 0xff,
                                       NULL);
            } else {
                clutter_actor_set_position (child, gx, gy);
                clutter_actor_set_opacity (child, 0xff);
            }

            priv->count += hrn_tileable_get_count ((HrnTileable *) child);

            iter = g_sequence_iter_next (iter);
            i++;
        }

        row_count = 1 + ((item_count + (priv->children->len - 1)) / priv->items_per_row) - priv->base_y;

        /* FIXME: This sets the correct width for the group,
           but it starts at the wrong x co-ord... is this needed? */
        clutter_actor_set_size ((ClutterActor *) cluster,
                                priv->items_per_row * (ITEM_WIDTH + COL_GAP),
                                row_count * (ITEM_HEIGHT + ROW_GAP));
        clutter_actor_set_reactive ((ClutterActor *) cluster, FALSE);

        hrn_tileable_item_count_changed ((HrnTileable *) cluster,
                                         old_count, priv->count);
    } else {
        clutter_actor_show ((ClutterActor *) priv->table);
        clutter_actor_animate ((ClutterActor *) priv->table,
                               CLUTTER_EASE_IN_OUT_CUBIC, 500,
                               "opacity", 0xff,
                               NULL);

        /* Don't need to animate the child items here sucking in as they
           get destroyed right away */

        clutter_actor_set_size ((ClutterActor *) cluster,
                                ITEM_WIDTH, ITEM_HEIGHT);
        clutter_actor_set_reactive ((ClutterActor *) cluster, TRUE);
        hrn_tileable_item_count_changed ((HrnTileable *) cluster,
                                         priv->count, 1);

        if (priv->items_exist == TRUE) {
            destroy_items (cluster);
        }

        priv->shown_child = NULL;
    }
}

static void
tiler_set_items_per_row (HrnTiler *tiler,
                         guint     items_per_row)
{
    HrnCluster *cluster = HRN_CLUSTER (tiler);
    HrnClusterPrivate *priv = cluster->priv;
    int i;

    if (priv->items_per_row == items_per_row) {
        return;
    }

    priv->items_per_row = items_per_row;

    /* Inform each of the HrnTiler children */
    for (i = 0; i < priv->children->len; i++) {
        ClutterActor *actor = priv->children->pdata[i];

        if (HRN_IS_TILER (actor)) {
            hrn_tiler_set_items_per_row ((HrnTiler *) actor, items_per_row);
        }
    }

    if (priv->expanded) {
        layout_cluster (cluster);
    }
}

static void
thumbnail_secondary_cb (HrnTileFrame *frame,
                        HrnCluster   *cluster)
{
    hrn_tileable_activated ((HrnTileable *) cluster,
                            HRN_TILEABLE_ACTION_PLAY,
                            hrn_tileable_get_node ((HrnTileable *) cluster));
}

static void
thumbnail_primary_cb (HrnTileFrame *frame,
                      HrnCluster   *cluster)
{
    hrn_tileable_activated ((HrnTileable *) cluster,
                            HRN_TILEABLE_ACTION_EXPAND, cluster);
}

static void
set_thumbnail (HrnCluster     *cluster,
               HrnClusterNode *node)
{
    HrnClusterPrivate *priv = cluster->priv;
    HrnTextureCache *cache = hrn_texture_cache_get_default ();
    const char *uri = NULL;
    BklItemType type = BKL_ITEM_TYPE_AUDIO;

    switch (node->type) {
    case HRN_CLUSTER_NODE_TYPE_ARTIST:
        uri = ((ArtistCluster *) node->data)->thumbnail_uri;
        type = BKL_ITEM_TYPE_AUDIO;
        break;

    case HRN_CLUSTER_NODE_TYPE_ALBUM:
        uri = ((AlbumCluster *) node->data)->thumbnail_uri;
        type = BKL_ITEM_TYPE_AUDIO;
        break;

    case HRN_CLUSTER_NODE_TYPE_YEAR:
        uri = ((YearCluster *) node->data)->thumbnail_uri;
        type = BKL_ITEM_TYPE_IMAGE;
        break;

    case HRN_CLUSTER_NODE_TYPE_MONTH:
        uri = ((MonthCluster *) node->data)->thumbnail_uri;
        type = BKL_ITEM_TYPE_IMAGE;
        break;

    default:
        break;
    }

    if (uri) {
        priv->thumbnail = hrn_texture_cache_get_texture (cache, uri);
    } else {
        priv->thumbnail = hrn_texture_cache_get_default_texture (cache, type);
    }

    priv->frame = g_object_new (HRN_TYPE_TILE_FRAME,
                                "texture", priv->thumbnail,
                                "label", hrn_cluster_get_label (cluster),
                                NULL);
    clutter_actor_set_reactive (priv->frame, TRUE);
    g_signal_connect (priv->frame, "primary-action",
                      G_CALLBACK (thumbnail_primary_cb), cluster);
    g_signal_connect (priv->frame, "secondary-action",
                      G_CALLBACK (thumbnail_secondary_cb), cluster);

    nbtk_table_add_actor ((NbtkTable *) priv->table, priv->frame, 0, 0);
}

static void
child_added_cb (HrnClusterNode *node,
                HrnClusterNode *child,
                HrnCluster     *cluster)
{
    HrnClusterPrivate *priv = cluster->priv;

    if (priv->expanded == FALSE) {
        return;
    }

    /* FIXME: We could have a #insert_item function that would only shift the
       items after the newly added item */
    add_item (cluster, child);
    layout_cluster (cluster);
}

static void
child_removed_cb (HrnClusterNode *node,
                  HrnClusterNode *child_node,
                  HrnCluster     *cluster)
{
    HrnClusterPrivate *priv = cluster->priv;
    ClutterActor *child_actor;

    if (priv->expanded == FALSE) {
        return;
    }

    if (priv->node == NULL) {
        return;
    }

    child_actor = g_hash_table_lookup (priv->child_items, child_node);
    if (child_actor == NULL) {
        g_warning ("%s Could not find child item for %s",
                   G_STRLOC, child_node->name);
        return;
    }

    g_ptr_array_remove (priv->children, child_actor);
    g_hash_table_remove (priv->child_items, child_node);

    if (child_actor == (ClutterActor *) priv->shown_child) {
        hrn_tiler_show_all ((HrnTiler *) cluster);
    }

    clutter_actor_destroy (child_actor);

    g_print ("Laying out cluster\n");
    layout_cluster (cluster);
}

/* If HrnCluster implemented ClutterContainer then this could be put into
   HrnTiler */
static void
tiler_add_items (HrnTiler       *tiler,
                 HrnClusterNode *node)
{
    HrnCluster *cluster = (HrnCluster *) tiler;
    HrnClusterPrivate *priv = cluster->priv;

    priv->node = node;

    /* FIXME: Need to disconnect these signals somehow */
    g_signal_connect (node, "child-added",
                      G_CALLBACK (child_added_cb), tiler);
    g_signal_connect (node, "child-removed",
                      G_CALLBACK (child_removed_cb), tiler);

    g_object_set (cluster,
                  "node", node,
                  NULL);
    set_thumbnail (cluster, node);

    /* children items are created when the cluster is expanded */
}

static void
tiler_interface_init (HrnTilerInterface *iface)
{
    iface->show_all = tiler_show_all;
    iface->show = tiler_show;
    iface->set_expanded = tiler_set_expanded;
    iface->set_items_per_row = tiler_set_items_per_row;
    iface->add_items = tiler_add_items;
}

static guint
tileable_get_count (HrnTileable *tileable)
{
    HrnCluster *cluster = HRN_CLUSTER (tileable);
    HrnClusterPrivate *priv = cluster->priv;

    if (priv->expanded) {
        return priv->count + 1; /* + 1 for the up button */
    } else {
        return 1;
    }
}

static ClutterActor *
tileable_get_thumbnail (HrnTileable *tileable)
{
    HrnCluster *cluster = HRN_CLUSTER (tileable);
    HrnClusterPrivate *priv = cluster->priv;
    GSequenceIter *iter;
    HrnClusterNode *child_node;
    ClutterActor *child_actor;

    if (priv->node == NULL) {
        return NULL;
    }

    iter = g_sequence_get_begin_iter (priv->node->children);
    child_node = g_sequence_get (iter);
    child_actor = g_hash_table_lookup (priv->child_items, child_node);
    if (child_actor == NULL) {
        g_warning ("Could not find child for %s", child_node->name);
        return NULL;
    }

    return hrn_tileable_get_thumbnail ((HrnTileable *) child_actor);
}

static void
tileable_set_position (HrnTileable *tileable,
                       guint        x,
                       guint        y)
{
    HrnCluster *cluster = HRN_CLUSTER (tileable);
    HrnClusterPrivate *priv = cluster->priv;

    priv->base_x = x;
    priv->base_y = y;
}

static HrnClusterNode *
tileable_get_node (HrnTileable *tileable)
{
    HrnCluster *cluster = HRN_CLUSTER (tileable);

    return HRN_CLUSTER_GET_CLASS (tileable)->get_node (cluster);
}

static void
tileable_interface_init (HrnTileableInterface *iface)
{
    iface->get_count = tileable_get_count;
    iface->get_thumbnail = tileable_get_thumbnail;
    iface->set_position = tileable_set_position;
    iface->get_node = tileable_get_node;
}

static gboolean
label_clicked_cb (ClutterActor       *actor,
                  ClutterButtonEvent *event,
                  HrnCluster         *cluster)
{
    const char *text;

    if (event->button != 1) {
        return FALSE;
    }

    text = nbtk_label_get_text ((NbtkLabel *) actor);
    hrn_tileable_activated ((HrnTileable *) cluster,
                            HRN_TILEABLE_ACTION_SEARCH, (gpointer) text);
    return TRUE;
}

static gboolean
label_enter_cb (ClutterActor         *actor,
                ClutterCrossingEvent *event,
                HrnCluster           *cluster)
{
    nbtk_widget_set_style_pseudo_class ((NbtkWidget *) actor, "hover");
    return TRUE;
}

static gboolean
label_leave_cb (ClutterActor         *actor,
                ClutterCrossingEvent *event,
                HrnCluster           *cluster)
{
    nbtk_widget_set_style_pseudo_class ((NbtkWidget *) actor, NULL);
    return TRUE;
}

static void
hrn_cluster_init (HrnCluster *self)
{
    HrnClusterPrivate *priv = GET_PRIVATE (self);

    self->priv = priv;
    priv->children = g_ptr_array_new ();
    priv->child_items = g_hash_table_new (NULL, NULL);

    priv->expanded = FALSE;
    priv->base_x = 0;
    priv->base_y = 0;
    priv->items_per_row = 1; /* This should get overridden */
    priv->count = 1;

    /* The table displays the information when the cluster is collapsed */
    priv->table = nbtk_table_new ();
    clutter_actor_set_size ((ClutterActor *) priv->table, ITEM_WIDTH, ITEM_HEIGHT);
    clutter_container_add_actor ((ClutterContainer *) self,
                                 (ClutterActor *) priv->table);

#if 0
    priv->thumbnail = g_object_new (HRN_TYPE_CLUSTER_THUMBNAIL, NULL);
    clutter_actor_set_size (priv->thumbnail, /*THUMBNAIL_WIDTH, THUMBNAIL_HEIGHT*/ 104, 104);
    clutter_actor_set_reactive (priv->thumbnail, TRUE);
    g_signal_connect (priv->thumbnail, "button-release-event",
                      G_CALLBACK (thumbnail_clicked_cb), self);
#endif

    priv->primary = nbtk_label_new ("");
    nbtk_widget_set_style_class_name (priv->primary, "HrnTileablePrimary");
    clutter_actor_set_reactive ((ClutterActor *) priv->primary, TRUE);
    g_signal_connect (priv->primary, "button-release-event",
                      G_CALLBACK (label_clicked_cb), self);
    g_signal_connect (priv->primary, "enter-event",
                      G_CALLBACK (label_enter_cb), self);
    g_signal_connect (priv->primary, "leave-event",
                      G_CALLBACK (label_leave_cb), self);
    nbtk_table_add_actor_with_properties (NBTK_TABLE (priv->table),
                                          (ClutterActor *) priv->primary, 1, 0,
                                          "x-expand", FALSE,
                                          "x-fill", FALSE,
                                          "y-expand", FALSE,
                                          "y-fill", FALSE,
                                          "x-align", 0.0,
                                          NULL);

    priv->secondary = nbtk_label_new ("");
    nbtk_widget_set_style_class_name (priv->secondary, "HrnTileableSecondary");
    clutter_actor_set_reactive ((ClutterActor *) priv->secondary, TRUE);
    g_signal_connect (priv->secondary, "button-release-event",
                      G_CALLBACK (label_clicked_cb), self);
    g_signal_connect (priv->secondary, "enter-event",
                      G_CALLBACK (label_enter_cb), self);
    g_signal_connect (priv->secondary, "leave-event",
                      G_CALLBACK (label_leave_cb), self);
    nbtk_table_add_actor_with_properties (NBTK_TABLE (priv->table),
                                          (ClutterActor *) priv->secondary,
                                          2, 0,
                                          "x-expand", FALSE,
                                          "x-fill", FALSE,
                                          "y-expand", FALSE,
                                          "y-fill", FALSE,
                                          "x-align", 0.0,
                                          NULL);

    if (priv->expanded) {
        clutter_actor_hide ((ClutterActor *) priv->table);
        clutter_actor_set_opacity ((ClutterActor *) priv->table, 0x00);
    }
}

/* FIXME: These functions should maybe be shared between hrn-view.c and
   hrn-cluster.c */
static void
hide_child (HrnCluster   *cluster,
            ClutterActor *actor)
{
    HrnClusterPrivate *priv = cluster->priv;
    HrnClusterNode *node;

    /* FIXME: Should probably check that the child is onscreen */
    clutter_actor_animate (actor, CLUTTER_EASE_IN_OUT_CUBIC, 500,
                           "opacity", 0x00,
                           "signal-swapped::completed",
                           clutter_actor_hide, actor,
                           NULL);

    priv->visible_children--;

    node = hrn_tileable_get_node ((HrnTileable *) actor);
    /* Block this as we're already hiding it */
    g_signal_handlers_block_by_func (actor, child_visibility_changed_cb, cluster);
    hrn_cluster_node_set_children_hidden (node, TRUE);
    g_signal_handlers_unblock_by_func (actor, child_visibility_changed_cb, cluster);
}

static void
show_child (HrnCluster   *cluster,
            ClutterActor *actor)
{
    HrnClusterPrivate *priv = cluster->priv;
    HrnClusterNode *node;

    clutter_actor_show (actor);
    clutter_actor_animate (actor, CLUTTER_EASE_IN_OUT_CUBIC, 500,
                           "opacity", 0xff,
                           NULL);

    priv->visible_children++;

    node = hrn_tileable_get_node ((HrnTileable *) actor);

    g_signal_handlers_block_by_func (actor, child_visibility_changed_cb,
                                     cluster);
    hrn_cluster_node_set_children_hidden (node, FALSE);
    g_signal_handlers_unblock_by_func (actor, child_visibility_changed_cb,
                                       cluster);
}

static void
move_child (ClutterActor *actor,
            int           x,
            int           y)
{
    HrnClusterNode *node;

    node = hrn_tileable_get_node ((HrnTileable *) actor);
    g_print ("Moving child to %d, %d\n", x, y);

    hrn_tileable_set_position ((HrnTileable *) actor, x, y);

    /* FIXME: Check that we're on screen */
    clutter_actor_animate (actor, CLUTTER_EASE_IN_OUT_CUBIC, 500,
                           "x", X_GRID_TO_COORDS (x),
                           "y", Y_GRID_TO_COORDS (y),
                           NULL);
}

static void
child_activated_cb (HrnTileable      *tileable,
                    HrnTileableAction action,
                    gpointer          payload,
                    HrnCluster       *cluster)
{
    HrnClusterPrivate *priv = cluster->priv;
    HrnClusterNode *payload_node;
    GSequenceIter *iter;

    switch (action) {
    case HRN_TILEABLE_ACTION_EXPAND:
        payload_node = hrn_tileable_get_node ((HrnTileable *) payload);

        clutter_actor_animate ((ClutterActor *) priv->up_tile,
                               CLUTTER_EASE_IN_OUT_CUBIC, 500,
                               "opacity", 0x00,
                               "signal-swapped::completed",
                               clutter_actor_hide, priv->up_tile,
                               NULL);

        iter = g_sequence_get_begin_iter (priv->node->children);
        while (g_sequence_iter_is_end (iter) == FALSE) {
            HrnClusterNode *child_node = g_sequence_get (iter);

            if (child_node != payload_node) {
                ClutterActor *child;

                child = g_hash_table_lookup (priv->child_items, child_node);
                if (child == NULL) {
                    g_warning ("%s No child found for %s",
                               G_STRLOC, child_node->name);
                    iter = g_sequence_iter_next (iter);
                    continue;
                }
                hide_child (cluster, child);
            }

            iter = g_sequence_iter_next (iter);
        }

        priv->shown_child = tileable;

        priv->count = hrn_tileable_get_count (tileable);
        move_child ((ClutterActor *) payload, 0, 0);
        hrn_tiler_set_expanded ((HrnTiler *) payload, TRUE);

        hrn_tileable_activated ((HrnTileable *) cluster,
                                HRN_TILEABLE_ACTION_EXPANDED,
                                hrn_tileable_get_node ((HrnTileable *) payload));
        return;

    case HRN_TILEABLE_ACTION_UP_LEVEL:
        g_print ("Up level...\n");
        payload_node = hrn_tileable_get_node ((HrnTileable *) payload);

        hrn_tiler_set_expanded ((HrnTiler *) cluster, FALSE);
        hrn_tileable_activated ((HrnTileable *) cluster,
                                HRN_TILEABLE_ACTION_SHOW_ALL, cluster);
#if 0
        iter = g_sequence_get_begin_iter (priv->node->children);
        while (g_sequence_iter_is_end (iter) == FALSE) {
            HrnClusterNode *child_node = g_sequence_get (iter);
            ClutterActor *child;

            child = g_hash_table_lookup (priv->child_items, child_node);
            if (child == NULL) {
                g_warning ("%s No child found for %s",
                           G_STRLOC, child_node->name);
                iter = g_sequence_iter_next (iter);
                continue;
            }

            if (child_node != payload_node) {
            }

            show_child (cluster, child);
            iter = g_sequence_iter_next (iter);
        }
#endif

        priv->shown_child = NULL;
        return;

    case HRN_TILEABLE_ACTION_SHOW_ALL:
        g_print ("%s is showing all\n", priv->node->name);

        clutter_actor_show ((ClutterActor *) priv->up_tile);
        clutter_actor_animate ((ClutterActor *) priv->up_tile,
                               CLUTTER_EASE_IN_OUT_CUBIC, 500,
                               "opacity", 0xff,
                               NULL);

        payload_node = hrn_tileable_get_node ((HrnTileable *) payload);
        iter = g_sequence_get_begin_iter (priv->node->children);
        while (g_sequence_iter_is_end (iter) == FALSE) {
            HrnClusterNode *child_node = g_sequence_get (iter);
            ClutterActor *child;

            g_print ("showing %s\n", child_node->name);
            child = g_hash_table_lookup (priv->child_items, child_node);
            if (child == NULL) {
                g_warning ("%s No child found for %s",
                           G_STRLOC, child_node->name);
                iter = g_sequence_iter_next (iter);
                continue;
            }

            if (child_node != payload_node && HRN_IS_TILER (payload)) {
                hrn_tiler_set_expanded ((HrnTiler *) payload, FALSE);
            }

            show_child (cluster, child);
            iter = g_sequence_iter_next (iter);
        }

        priv->shown_child = NULL;
        if (priv->expanded) {
            layout_cluster (cluster);
        }

        hrn_tileable_activated ((HrnTileable *) cluster,
                                HRN_TILEABLE_ACTION_LEVEL_CHANGE, priv->node);
        return;

    default:
        break;
    }

    hrn_tileable_activated ((HrnTileable *) cluster, action, payload);
}

static void
child_count_changed_cb (HrnTileable *tileable,
                        int          old_count,
                        int          new_count,
                        HrnCluster  *cluster)
{
    HrnClusterPrivate *priv = cluster->priv;
    int old_total, row_count;

    old_total = priv->count;
    priv->count -= (old_count - new_count);

    row_count = 1 + ((priv->count + (priv->children->len - 1)) / priv->items_per_row) - priv->base_y;
    clutter_actor_set_size ((ClutterActor *) cluster,
                            priv->items_per_row * (ITEM_WIDTH + COL_GAP),
                            row_count * (ITEM_HEIGHT + ROW_GAP));

    hrn_tileable_item_count_changed ((HrnTileable *) cluster,
                                     old_total, priv->count);
}

static void
child_visibility_changed_cb (HrnTileable *tileable,
                             gboolean     hidden,
                             HrnCluster  *cluster)
{
    HrnClusterPrivate *priv = cluster->priv;

    if (hidden) {
        priv->visible_children--;
        clutter_actor_hide ((ClutterActor *) tileable);
        clutter_actor_set_position ((ClutterActor *) tileable, 0, 0);
    } else {
        priv->visible_children++;
        if (priv->expanded) {
            clutter_actor_show ((ClutterActor *) tileable);
            /* The position will be set correctly by the relayout */
        }
    }

    if (priv->expanded) {
        layout_cluster (cluster);
    }
}

/* FIXME: HrnCluster should implement ClutterContainer,
   well, it does by being a subclass of ClutterGroup, but it should be
   a ClutterActor and implement it itself. */
void
hrn_cluster_add_actor (HrnCluster  *cluster,
                       HrnTileable *tileable)
{
    HrnClusterPrivate *priv = cluster->priv;

    if (!HRN_IS_TILEABLE (tileable)) {
        g_warning ("Cannot add %s to HrnCluster as it does not implement HrnTileable", G_OBJECT_TYPE_NAME (tileable));
        return;
    }

    clutter_container_add_actor ((ClutterContainer *) cluster,
                                 (ClutterActor *) tileable);
    if (priv->expanded == FALSE) {
        clutter_actor_hide ((ClutterActor *) tileable);
    }

    g_signal_connect (tileable, "activated",
                      G_CALLBACK (child_activated_cb), cluster);
    g_signal_connect (tileable, "count-changed",
                      G_CALLBACK (child_count_changed_cb), cluster);
    g_signal_connect (tileable, "visibility-changed",
                      G_CALLBACK (child_visibility_changed_cb), cluster);

    g_ptr_array_add (priv->children, tileable);
}

void
hrn_cluster_full (HrnCluster *cluster)
{
    HrnClusterPrivate *priv = cluster->priv;

#if 0
    hrn_cluster_thumbnail_set_elements ((HrnClusterThumbnail *) priv->thumbnail, priv->children);
#endif

    if (priv->expanded) {
        layout_cluster (cluster);
    }
}

void
hrn_cluster_set_secondary_reactive (HrnCluster *cluster,
                                    gboolean    reactive)
{
    HrnClusterPrivate *priv = cluster->priv;

    clutter_actor_set_reactive ((ClutterActor *) priv->secondary, reactive);
}

const char *
hrn_cluster_get_label (HrnCluster *cluster)
{
    HrnClusterClass *klass = HRN_CLUSTER_GET_CLASS (cluster);

    return klass->get_label (cluster);
}
