#include "qfilesystemmodelqml.h"

#include <QDir>
#include <QTimer>

QFileSystemModelQML::QFileSystemModelQML(QObject* parent)
    : QAbstractListModel(parent)
    , m_nameFilterDisables(false)
{
    QHash<int, QByteArray> roles;
    roles[0] = "filePath";
    setRoleNames(roles);

    connect(&m_watcher, SIGNAL(directoryChanged(const QString&)),
            SLOT(onDirectoryChanged(const QString&)));
}

QString QFileSystemModelQML::rootPath() const
{
    return m_rootPath;
}

QStringList QFileSystemModelQML::nameFilters() const
{
    return m_nameFilters;
}

bool QFileSystemModelQML::nameFilterDisables() const
{
    return m_nameFilterDisables;
}

void QFileSystemModelQML::updateFiles()
{
    beginRemoveRows(QModelIndex(), 0, m_files.size());
    m_files.clear();
    endRemoveRows();

    QDir dir(m_rootPath);
    QFileInfoList files = dir.entryInfoList(m_nameFilters, QDir::Files, QDir::Name);
    Q_FOREACH(QFileInfo info, files) {
        if (info.isReadable() && (info.size() > 0)) {
            int index = m_files.size();
            beginInsertRows(QModelIndex(), index, index);
            m_files.append(info);
            endInsertRows();
        }
    }
}

void QFileSystemModelQML::setRootPath(const QString& newPath)
{
    if (newPath != m_rootPath) {
        if (!m_rootPath.isEmpty()) {
            m_watcher.removePath(m_rootPath);
        }
        m_rootPath = newPath;
        updateFiles();
        m_watcher.addPath(m_rootPath);
        emit rootPathChanged(m_rootPath);
    }
}

void QFileSystemModelQML::setNameFilters(const QStringList& filters)
{
    if (filters != m_nameFilters) {
        m_nameFilters = filters;
        if (!m_nameFilterDisables) {
            updateFiles();
        }
    }
}

void QFileSystemModelQML::setNameFilterDisables(bool disable)
{
    if (disable != m_nameFilterDisables) {
        m_nameFilterDisables = disable;
        if (!m_nameFilters.isEmpty()) {
            updateFiles();
        }
    }
}

void QFileSystemModelQML::onDirectoryChanged(const QString& path)
{
    if (path == m_rootPath) {
        /* Delay the update: when we get the change notification, new files may
           not have been written to the disk yet, and thus not be readable yet.
           Ref: https://bugs.launchpad.net/hedley/+bug/727327 */
        QTimer::singleShot(1000, this, SLOT(updateFiles()));
    }
}

int
QFileSystemModelQML::rowCount(const QModelIndex & parent) const
{
    return m_files.size();
}

QVariant QFileSystemModelQML::data(const QModelIndex& index, int role) const
{
    if (!index.isValid()) {
        return QVariant();
    }

    return m_files[index.row()].absoluteFilePath();
}

