/*
 * Copyright (C) 2005-2009 Freescale Semiconductor, Inc. All rights reserved.
 *
 */
 
/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
 
/*
 * Module Name:    mfw_gst_vpu_decoder.c
 *
 * Description:    Implementation of Hardware (VPU) Decoder Plugin for Gstreamer.
 *
 * Portability:    This code is written for Linux OS and Gstreamer
 */  
 
/*
 * Changelog: 
 *
 */


/*======================================================================================
                            INCLUDE FILES
=======================================================================================*/
#include <gst/gst.h>
#include <gst/base/gstadapter.h>
#include <string.h>
#include <fcntl.h>		/* fcntl */
#include <sys/mman.h>		/* mmap */
#include <sys/ioctl.h>		/* fopen/fread */		
#include "vpu_io.h"
#include "vpu_lib.h"
#include "mfw_gst_vpu_decoder.h"
#include "mfw_gst_utils.h"

//#define GST_DEBUG g_print

//#define GST_LATENCY g_print
#ifndef GST_LATENCY
#define GST_LATENCY 
#endif

//#define GST_MUTEX g_print
#ifndef GST_MUTEX
#define GST_MUTEX
#endif

//#define GST_FRAMEDBG g_print
#ifndef GST_FRAMEDBG
#define GST_FRAMEDBG
#endif

//#define GST_FRAMEDROP g_print
#ifndef GST_FRAMEDROP
#define GST_FRAMEDROP
#endif

//#define GST_TIMESTAMP g_print
#ifndef GST_TIMESTAMP
#define GST_TIMESTAMP
#endif


#define VPU_PARALLELIZATION 1

/*======================================================================================
                                        LOCAL MACROS
=======================================================================================*/

#define	GST_CAT_DEFAULT	mfw_gst_vpudec_debug
static MfwGstVPU_Dec *vpudec_global_ptr = NULL;
static gint mutex_cnt=0;

/*======================================================================================
                                      STATIC VARIABLES
=======================================================================================*/
extern int vpu_fd;  

#define VPU_PIC_TYPE  (vpu_dec->outputInfo->picType&3)
#ifdef VPU_MX27
#define STD_MPEG2 -1
#define STD_VC1   -1
#define STD_MJPG  -1
#define STD_RM    -1
#define STD_DIVX  -1
#define INTERLACED_FRAME 0
#else
#define INTERLACED_FRAME (vpu_dec->outputInfo->interlacedFrame)
#endif

/*======================================================================================
                                 STATIC FUNCTION PROTOTYPES
=======================================================================================*/
GST_DEBUG_CATEGORY_STATIC(mfw_gst_vpudec_debug);

static void mfw_gst_vpudec_class_init   (MfwGstVPU_DecClass *);
static void mfw_gst_vpudec_base_init    (MfwGstVPU_DecClass *);
static void mfw_gst_vpudec_init         (MfwGstVPU_Dec *, MfwGstVPU_DecClass *);
static GstFlowReturn mfw_gst_vpudec_chain(GstPad *, GstBuffer *);
static GstStateChangeReturn mfw_gst_vpudec_change_state(GstElement *, GstStateChange);
static void mfw_gst_vpudec_set_property (GObject *,guint,const GValue *,GParamSpec *);
static void mfw_gst_vpudec_get_property (GObject *,guint,GValue *,GParamSpec *);
static gboolean mfw_gst_vpudec_sink_event (GstPad *, GstEvent *);
static gboolean mfw_gst_vpudec_src_event (GstPad *, GstEvent *);
static gboolean mfw_gst_vpudec_setcaps  (GstPad * , GstCaps *);
static GstBuffer * mfw_gst_VC1_Create_RCVheader(MfwGstVPU_Dec *,GstBuffer *);
static gboolean mfw_gst_vpudec_preparevc1apheader(unsigned char *pDestBuffer, unsigned char *pDataBuffer, unsigned char *pAucHdrBuffer,int aucHdrLen, int * pLen);
/*======================================================================================
                                     GLOBAL VARIABLES
=======================================================================================*/
/*======================================================================================
                                     LOCAL FUNCTIONS
=======================================================================================*/
gboolean vpu_mutex_lock(GMutex *mutex, gboolean ftry);
void vpu_mutex_unlock(GMutex *mutex);
gint mfw_gst_vpudec_FrameBufferInit(MfwGstVPU_Dec *,FrameBuffer *,gint);

void mfw_gst_vpudec_FrameBufferClose(MfwGstVPU_Dec *);
gboolean mfw_gst_get_timestamp(MfwGstVPU_Dec *, GstClockTime * );
GstFlowReturn mfw_gst_vpudec_vpu_open(MfwGstVPU_Dec *vpu_dec);
GstFlowReturn mfw_gst_vpudec_copy_sink_input(MfwGstVPU_Dec *, GstBuffer *);
GstFlowReturn mfw_gst_vpudec_release_buff(MfwGstVPU_Dec *);
GstFlowReturn mfw_gst_vpudec_vpu_init(MfwGstVPU_Dec *);
GstFlowReturn mfw_gst_vpudec_render(MfwGstVPU_Dec *);
void  mfw_gst_vpudec_cleanup(MfwGstVPU_Dec *vpu_dec);

/*=============================================================================
FUNCTION:           mfw_gst_vpudec_set_property

DESCRIPTION:        Sets the property of the element

ARGUMENTS PASSED:
        object     - pointer to the elements object
        prop_id    - ID of the property;
        value      - value of the property set by the application
        pspec      - pointer to the attributes of the property

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static void mfw_gst_vpudec_set_property(GObject * object, guint prop_id,
					const GValue * value,
					GParamSpec * pspec)
{
    MfwGstVPU_Dec *vpu_dec = MFW_GST_VPU_DEC(object);
    if (vpudec_global_ptr == NULL)
        return;

    switch (prop_id) {
        case MFW_GST_VPU_PROF_ENABLE:
            vpu_dec->profiling = g_value_get_boolean(value);
            GST_DEBUG(">>VPU_DEC: profiling=%d\n", vpu_dec->profiling);
            break;

        case MFW_GST_VPU_CODEC_TYPE:
            vpu_dec->codec = g_value_get_enum(value);
            GST_DEBUG(">>VPU_DEC: codec=%d\n", vpu_dec->codec);
            break;

        case MFW_GST_VPU_LOOPBACK:
            vpu_dec->loopback = g_value_get_boolean(value);
            if (vpu_dec->loopback)
                 vpu_dec->allow_parallelization = FALSE;
            break;

        case MFW_GST_VPU_PASSTHRU:
             vpu_dec->passthru = g_value_get_boolean(value);
             break;

        case MFW_GST_VPU_LATENCY:
             vpu_dec->min_latency = g_value_get_boolean(value);
             break;

        case MFW_GST_VPU_PARSER:  // in case of no parser providing frame by frame - disable nal checking
             vpu_dec->parser_input = g_value_get_boolean(value);
             break;

        case MFW_GST_VPU_FRAMEDROP:
             vpu_dec->frame_drop_allowed = g_value_get_boolean(value);
             if (!vpu_dec->frame_drop_allowed)
                 vpu_dec->min_latency=TRUE;
             break;

#ifndef VPU_MX27
        case MFW_GST_VPU_DBK_ENABLE:
            vpu_dec->dbk_enabled= g_value_get_boolean(value);
            break;

        case MFW_GST_VPU_DBK_OFFSETA:
            vpu_dec->dbk_offset_a= g_value_get_int(value);
            break;

        case MFW_GST_VPU_DBK_OFFSETB:
            vpu_dec->dbk_offset_b= g_value_get_int(value);
            break;
#endif

        case MFW_GST_VPU_FORCE_NONDR:
            vpu_dec->force_nondr = g_value_get_boolean(value);
            break;

        case MFW_GST_VPU_MIRROR:
            vpu_dec->mirror_dir = g_value_get_enum(value);
            break;

        case MFW_GST_VPU_ROTATION:
            vpu_dec->rotation_angle = g_value_get_uint(value);
            switch (vpu_dec->rotation_angle)
            {
                case 0:
                case 90:
                case 180:
                case 270:
                    break;
                default:
                    vpu_dec->rotation_angle = 0;
                    G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
                break;
            }
            break;
#if (defined (VPU_MX37) || defined (VPU_MX51))
        case MFW_GST_VPU_OUTPUT_FMT:
            vpu_dec->fmt = g_value_get_int(value);
            break;
#endif

        default: // else rotation will fall through with invalid parameter
            G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
        break;
    }
    return;
}

/*=============================================================================
FUNCTION:           mfw_gst_vpudec_set_property

DESCRIPTION:        Gets the property of the element

ARGUMENTS PASSED:
        object     - pointer to the elements object
        prop_id    - ID of the property;
        value      - value of the property set by the application
        pspec      - pointer to the attributes of the property

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static void mfw_gst_vpudec_get_property(GObject * object, guint prop_id,
					GValue * value, GParamSpec * pspec)
{

    MfwGstVPU_Dec *vpu_dec = MFW_GST_VPU_DEC(object);
    if (vpudec_global_ptr == NULL)
        return;

    switch (prop_id) {
        case MFW_GST_VPU_PROF_ENABLE:
            g_value_set_boolean(value, vpu_dec->profiling);
            break;
        case MFW_GST_VPU_CODEC_TYPE:
            g_value_set_enum(value, vpu_dec->codec);
            break;
        case MFW_GST_VPU_LOOPBACK:
            g_value_set_boolean(value, vpu_dec->loopback);
            break;
        case MFW_GST_VPU_PASSTHRU:
            g_value_set_boolean(value, vpu_dec->passthru);
            break;
        case MFW_GST_VPU_LATENCY:
            g_value_set_boolean(value, vpu_dec->min_latency);
            break;
        case MFW_GST_VPU_PARSER:
            g_value_set_boolean(value, vpu_dec->parser_input);
            break;
        case MFW_GST_VPU_FRAMEDROP:
            g_value_set_boolean(value, vpu_dec->frame_drop_allowed);
            break;

#ifndef VPU_MX27
        case MFW_GST_VPU_DBK_ENABLE:
            g_value_set_boolean(value, vpu_dec->dbk_enabled);
            break;
        case MFW_GST_VPU_DBK_OFFSETA:
            g_value_set_int(value, vpu_dec->dbk_offset_a);
            break;
        case MFW_GST_VPU_DBK_OFFSETB:
            g_value_set_int(value, vpu_dec->dbk_offset_b);
            break;
#endif

        case MFW_GST_VPU_FORCE_NONDR:
            g_value_set_boolean(value, vpu_dec->force_nondr);
            break;
        case MFW_GST_VPU_MIRROR:
             g_value_set_enum(value, vpu_dec->mirror_dir);
            break;
        case MFW_GST_VPU_ROTATION:
            g_value_set_uint(value, vpu_dec->rotation_angle);
            break;

#if (defined (VPU_MX37) || defined (VPU_MX51))
        case MFW_GST_VPU_OUTPUT_FMT:
            g_value_set_int(value, vpu_dec->fmt);
            break;
#endif            
        default:
	        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
            break;
    }
    return;
}

/*=============================================================================
FUNCTION:           mfw_gst_vpudec_post_fatal_error_msg

DESCRIPTION:        This function is used to post a fatal error message and 
                    terminate the pipeline during an unrecoverable error.
                        
ARGUMENTS PASSED:   vpu_dec  - VPU decoder plugins context error_msg message to be posted 
        
RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static void mfw_gst_vpudec_post_fatal_error_msg(MfwGstVPU_Dec *vpu_dec,gchar *error_msg)
{
    GError *error = NULL;
    GQuark domain;
    domain = g_quark_from_string("mfw_vpudecoder");
    error = g_error_new(domain, 10, "fatal error");
    gst_element_post_message(GST_ELEMENT(vpu_dec),
        gst_message_new_error(GST_OBJECT
        (vpu_dec),error,error_msg));
    g_error_free(error);
}

/*=============================================================================
FUNCTION:           vpu_mutex_lock

DESCRIPTION:        Locks the VPU mutex
                        
ARGUMENTS PASSED:   mutex
                    fTry - if set means use trylock so it can fail if already locked
        
RETURN VALUE:       result - TRUE if lock succeeded, FALSE if failed
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
gboolean vpu_mutex_lock(GMutex *mutex, gboolean ftry)
{
    gboolean result=TRUE;
    if (!mutex || !vpudec_global_ptr)
    {
        GST_DEBUG(">>VPU_DEC: lock mutex is NULL cnt=%d\n",mutex_cnt);
        return FALSE;
    }    
    if (ftry)
    {
        result = g_mutex_trylock(mutex);      
        if (result == FALSE)
        {
            //GST_MUTEX(">>VPU_DEC: mutex lock failed cnt=%d\n");
        } else {
            mutex_cnt++;
            //GST_MUTEX(">>VPU_DEC: VPU mutex locked. +++ cnt=%d\n",mutex_cnt);
        }
    }
    else {
        g_mutex_lock(mutex);
        mutex_cnt++;
        //GST_MUTEX(">>VPU_DEC: VPU mutex locked. +++ cnt=%d\n",mutex_cnt);
    }
    return result;
}

/*=============================================================================
FUNCTION:           vpu_mutex_unlock

DESCRIPTION:        Unlocks the mutex
                        
ARGUMENTS PASSED:   mutex
        
RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
void vpu_mutex_unlock(GMutex *mutex)
{
    if (!mutex || !vpudec_global_ptr)
    {
        GST_DEBUG(">>VPU_DEC: unlock mutex is NULL cnt=%d\n",mutex_cnt);
        return;
    }
    mutex_cnt--;
    //GST_MUTEX(">>VPU_DEC: VPU mutex unlocked. --- cnt=%d\n", mutex_cnt);
    g_mutex_unlock(mutex);
    return;
}

/*=============================================================================
FUNCTION:           mfw_gst_vpudec_preparevc1apheader

DESCRIPTION:        prepare header for VC1 AP, reference from Zhujun's implementation
                        
ARGUMENTS PASSED:   mutex
        
RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static gboolean mfw_gst_vpudec_preparevc1apheader(unsigned char *pDestBuffer, unsigned char *pDataBuffer, unsigned char *pAucHdrBuffer,int aucHdrLen, int * pLen)
{
    int i;
    unsigned char* temp;

    if ( (pDestBuffer==NULL)||(pDataBuffer==NULL)||(pAucHdrBuffer==NULL)||(pLen ==NULL)||(aucHdrLen<4) )
        return FALSE;

    i = *pLen;
    temp = pDestBuffer;
    
    //Find SeqHdrStartCode in AucHdrBuffer
    unsigned char * pSeqHdrPtr = pAucHdrBuffer;
    int iSeqHdrHdrLen = aucHdrLen;

    gboolean bScanSeqHdr = TRUE;
    while ( bScanSeqHdr && (pSeqHdrPtr < pAucHdrBuffer + aucHdrLen) )
    {
        if ( (0x00 == pSeqHdrPtr[0]) && (0x00 == pSeqHdrPtr[1]) 
            && (0x01 == pSeqHdrPtr[2]) &&(0x0F == pSeqHdrPtr[3]) )
        {
            iSeqHdrHdrLen -= (pSeqHdrPtr - pAucHdrBuffer);
            bScanSeqHdr = FALSE;
            break;
        }
        else
        {
            pSeqHdrPtr ++;
        }
    }
    if (bScanSeqHdr)
    {
        return FALSE;
    }


    if ( (0x00 == pDataBuffer[0]) && (0x00 == pDataBuffer[1]) && (0x01 == pDataBuffer[2]) )
    {
        if ( 0x0E == pDataBuffer[3]) // Entry pointer
        {
            gboolean loop = TRUE;
            unsigned char * pHdrScan = pSeqHdrPtr;
            while (loop && ( pHdrScan < pSeqHdrPtr+iSeqHdrHdrLen) )
            {
                if ( (0x00 == pHdrScan[0]) && (0x00 == pHdrScan[1]) 
                    && (0x01 == pHdrScan[2]) &&(0x0E == pHdrScan[3]) )
                {
                    loop = FALSE;
                    break;
                }
                else
                {
                    temp[i++] = pHdrScan[0];
                    pHdrScan ++;
                }
            }
        }
        else if ( 0x0D == pDataBuffer[3]) // Frame StartCode
        {
            // copy Auc Hdr 
            memcpy(temp+i, pSeqHdrPtr, iSeqHdrHdrLen);
            i += iSeqHdrHdrLen;
        }
    }
    else
    {
        // copy Auc Hdr 
        memcpy(temp+i, pSeqHdrPtr, iSeqHdrHdrLen);
        i += iSeqHdrHdrLen;

        temp[i++] = 0x00;
        temp[i++] = 0x00;
        temp[i++] = 0x01;
        temp[i++] = 0x0D;
    }
    *pLen = i;

    return TRUE;
}

/*=============================================================================
FUNCTION:           mfw_gst_VC1_Create_RCVheader

DESCRIPTION:        This function is used to create the RCV header 
                    for integration with the ASF demuxer using the width,height and the 
                    Header Extension data recived through caps negotiation.
                        
ARGUMENTS PASSED:   vpu_dec  - VPU decoder plugins context
        
RETURN VALUE:       GstBuffer
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static GstBuffer * mfw_gst_VC1_Create_RCVheader(MfwGstVPU_Dec *vpu_dec,GstBuffer *inbuffer)
{
	GstBuffer *RCVHeader =NULL;
  unsigned char* RCVHeaderData;
  unsigned int value=0;
  int i =0;

#define RCV_HEADER_LEN  256
	RCVHeader = gst_buffer_new_and_alloc(RCV_HEADER_LEN);
	
	if (RCVHeader==NULL){
		g_print("RVCHeader not allocated\n");
		return NULL;
	}
	RCVHeaderData = GST_BUFFER_DATA(RCVHeader);
	if (RCVHeaderData==NULL){
		g_print("RVCHeader not allocated\n");
		return NULL;
	}
	
	if(vpu_dec->codec_subtype==1)
	{
		 GST_DEBUG("preparing header for AP!\n");
	   if(mfw_gst_vpudec_preparevc1apheader(RCVHeaderData,GST_BUFFER_DATA(inbuffer),
		                              GST_BUFFER_DATA(vpu_dec->codec_data),vpu_dec->codec_data_len,&i)==FALSE)
       {
         GST_DEBUG("Failed in prepare VC1 AP header!\n"); 
	   }
	   GST_DEBUG("preparing header for AP successful!\n");
	   
	}
	else
	{
    //Number of Frames, Header Extension Bit, Codec Version
    value = NUM_FRAMES | SET_HDR_EXT | CODEC_VERSION;
    RCVHeaderData[i++] = (unsigned char)value;
    RCVHeaderData[i++] = (unsigned char)(value >> 8);
    RCVHeaderData[i++] = (unsigned char)(value >> 16);
    RCVHeaderData[i++] = (unsigned char)(value >> 24);
    //Header Extension Size
    //ASF Parser gives 5 bytes whereas the VPU expects only 4 bytes, so limiting it
    if(vpu_dec->codec_data_len > 4)
        vpu_dec->codec_data_len = 4;
    RCVHeaderData[i++] = (unsigned char)vpu_dec->codec_data_len;
    RCVHeaderData[i++] = 0; //(unsigned char)(vpu_dec->codec_data_len >> 8);
    RCVHeaderData[i++] = 0; //(unsigned char)(vpu_dec->codec_data_len >> 16);
    RCVHeaderData[i++] = 0; //(unsigned char)(vpu_dec->codec_data_len >> 24);
 
    //Header Extension bytes obtained during negotiation
    memcpy(RCVHeaderData+i,GST_BUFFER_DATA(vpu_dec->codec_data),vpu_dec->codec_data_len);
    i += vpu_dec->codec_data_len;

    //Height
    RCVHeaderData[i++] = (unsigned char)vpu_dec->picHeight;
    RCVHeaderData[i++] = (unsigned char)(((vpu_dec->picHeight >> 8) & 0xff));
    RCVHeaderData[i++] = (unsigned char)(((vpu_dec->picHeight >> 16) & 0xff));
    RCVHeaderData[i++] = (unsigned char)(((vpu_dec->picHeight >> 24) & 0xff));
    //Width
    RCVHeaderData[i++] = (unsigned char)vpu_dec->picWidth;
    RCVHeaderData[i++] = (unsigned char)(((vpu_dec->picWidth >> 8) & 0xff));
    RCVHeaderData[i++] = (unsigned char)(((vpu_dec->picWidth >> 16) & 0xff)); 
    RCVHeaderData[i++] = (unsigned char)(((vpu_dec->picWidth >> 24) & 0xff));
    //Frame Size
    RCVHeaderData[i++] = (unsigned char)GST_BUFFER_SIZE(inbuffer);
    RCVHeaderData[i++] = (unsigned char)(GST_BUFFER_SIZE(inbuffer) >> 8);
    RCVHeaderData[i++] = (unsigned char)(GST_BUFFER_SIZE(inbuffer) >> 16);
    RCVHeaderData[i++] = (unsigned char)((GST_BUFFER_SIZE(inbuffer) >> 24) | 0x80);
	}
	
	  GST_BUFFER_SIZE(RCVHeader) = i; 
    return RCVHeader;
}

static gboolean mfw_gst_vpudec_prePareVC1Header(MfwGstVPU_Dec *vpu_dec, GstBuffer *buffer){
 // The Size of the input stream is appended with the input stream 
 // for integration with ASF
    unsigned char* tempChars;
    GstBuffer *tempBuf;
		if(vpu_dec->codec_subtype==0)
		{
			  tempBuf =gst_buffer_new_and_alloc(4);
			  tempChars = GST_BUFFER_DATA(tempBuf);
        tempChars[0] = (unsigned char) GST_BUFFER_SIZE(buffer);
        tempChars[1] = (unsigned char)(GST_BUFFER_SIZE(buffer) >> 8);
        tempChars[2] = (unsigned char)(GST_BUFFER_SIZE(buffer) >> 16);
        tempChars[3] = (unsigned char)(GST_BUFFER_SIZE(buffer) >> 24);
		 		vpu_dec->gst_buffer = gst_buffer_join(tempBuf,buffer);
		}
	  else
	  { 
	      unsigned char* pSourceBuffer = GST_BUFFER_DATA(buffer);
	      if ( (0x00 == pSourceBuffer[0])
                && (0x00 == pSourceBuffer[1])
                && (0x01 == pSourceBuffer[2])
                && (0x0D == pSourceBuffer[3])
                )
            {
            }
            else if ( (0x00 == pSourceBuffer[0])
                && (0x00 == pSourceBuffer[1])
                && (0x01 == pSourceBuffer[2])
                && (0x0E == pSourceBuffer[3])
                )
            {
            }
            else
            {
                //start code -- 0x0d010000;
                tempBuf =gst_buffer_new_and_alloc(4);
			          tempChars = GST_BUFFER_DATA(tempBuf);
                tempChars[0] = 0x00;
                tempChars[1] = 0x00;
                tempChars[2] = 0x01;
                tempChars[3] = 0x0D;
		            vpu_dec->gst_buffer = gst_buffer_join(tempBuf,buffer);
            } 
      }
 }
 
/*=============================================================================
FUNCTION:           mfw_gst_vpudec_FrameBufferClose

DESCRIPTION:        This function frees the allocated frame buffers
                        
ARGUMENTS PASSED:   vpu_dec  - VPU decoder plugins context
        
RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
void mfw_gst_vpudec_FrameBufferClose(MfwGstVPU_Dec *vpu_dec)
{
    gint i;
    for(i=0;i<vpu_dec->numframebufs;i++)
    {
        if (vpu_dec->frame_mem[i].phy_addr != 0)
        {
            IOFreeVirtMem(&vpu_dec->frame_mem[i]);
            IOFreePhyMem(&vpu_dec->frame_mem[i]);
            vpu_dec->frame_mem[i].phy_addr=0;
            vpu_dec->frame_virt[i]=NULL;
        }
        if (vpu_dec->outbuffers[i]) {
            gst_buffer_unref(vpu_dec->outbuffers[i]);
            vpu_dec->outbuffers[i] = NULL;
        }
    }
}

/*=============================================================================
FUNCTION:           mfw_gst_get_timestamp

DESCRIPTION:        Retrieves the time stamp 

ARGUMENTS PASSED:   vpu_dec  - VPU decoder plugins context
                    ptimestamp - time stamp pointer

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
gboolean mfw_gst_get_timestamp(MfwGstVPU_Dec *vpu_dec, GstClockTime * ptimestamp)
{
    GstClockTime current_time=vpu_dec->clock_base;
    GstClockTime ts=vpu_dec->timestamp_buffer[vpu_dec->ts_tx];

    *ptimestamp = GST_CLOCK_TIME_NONE; 

#if 0
    if (!vpu_dec->file_play_mode)
    {
        return FALSE;
    }
#endif
    if ((GST_ELEMENT (vpu_dec)->clock == NULL)) 
    {
        GST_TIMESTAMP (">>VPU_DEC: get_timestamp null clock clock_base=%d tx=%d \n", (guint) vpu_dec->clock_base, vpu_dec->ts_tx);
        if ( vpu_dec->ts_tx != vpu_dec->ts_rx)
            vpu_dec->ts_tx = (vpu_dec->ts_tx+1)%MAX_STREAM_BUF;
        return FALSE;
    } else {
        current_time = gst_clock_get_time (GST_ELEMENT (vpu_dec)->clock);
        current_time -= gst_element_get_base_time(GST_ELEMENT (vpu_dec));

        if ( vpu_dec->ts_tx == vpu_dec->ts_rx)
        {
            GST_TIMESTAMP (">>VPU_DEC: get_timestamp ts and rx wrapped tx=%d \n", (guint) vpu_dec->clock_base, vpu_dec->ts_tx);
            return FALSE;
        }
 
        GST_TIMESTAMP  (">>VPU_DEC: current_time at render % " GST_TIME_FORMAT " ts % " GST_TIME_FORMAT " ts_tx %d\n",
                         GST_TIME_ARGS(current_time), GST_TIME_ARGS(ts),vpu_dec->ts_tx);
      
        // if this is first frame to render after dropping then rest and drop flag
        if (vpu_dec->drop_before_decode )
        {
            vpu_dec->drop_before_decode = FALSE;
            GST_FRAMEDROP(">>VPU_DEC: Resuming after frame dropping numdropped=%d numdecoded=%d\n", 
                          vpu_dec->frames_dropped, vpu_dec->frames_decoded);
        }
  
        *ptimestamp = ts;
        vpu_dec->ts_tx = (vpu_dec->ts_tx+1)%MAX_STREAM_BUF; 
    
    }

    if (vpu_dec->set_ts_manually)
    {     
        *ptimestamp = (vpu_dec->frames_decoded * vpu_dec->time_per_frame) + vpu_dec->clock_base;
        GST_TIMESTAMP (">>VPU_DEC: get_timestamp ts=%d set manually current=%d \n", (guint) *ptimestamp, (guint) current_time);

    } else {
        // don't send 0 time stamps or if we were behind last time - be sure to adjust this time
        if (*ptimestamp == GST_CLOCK_TIME_NONE)
        {
            *ptimestamp = (vpu_dec->frames_decoded * vpu_dec->time_per_frame) + vpu_dec->clock_base;
            return FALSE;
        }
        if (*ptimestamp == 0)  
            *ptimestamp = GST_CLOCK_TIME_NONE;
    }

    if (*ptimestamp == GST_CLOCK_TIME_NONE)
        vpu_dec->last_ts_sent = current_time;
    else
        vpu_dec->last_ts_sent = *ptimestamp;

    // if we are already behind make sure that the next time stamp uses current time as its base instead of 
    // one from input so we won't stay behind forever
    if ((vpu_dec->frames_rendered > 2) && (*ptimestamp < current_time))
    {
        guint num_frames_late = (guint)((current_time - *ptimestamp)/vpu_dec->time_per_frame);
        GST_FRAMEDROP(">>VPU_DEC: Output ts % " GST_TIME_FORMAT " over current time % " GST_TIME_FORMAT " by %d frames rendered=%d ts_tx %d\n",GST_TIME_ARGS(*ptimestamp), GST_TIME_ARGS(current_time), num_frames_late, vpu_dec->frames_rendered, vpu_dec->ts_tx);
        vpu_dec->last_ts_sent = current_time;
    }  else {
        vpu_dec->frame_drop_allowed = FALSE;
    }
    
    return TRUE;
}



/*=============================================================================
FUNCTION:           mfw_gst_vpudec_FrameBufferInit

DESCRIPTION:        This function allocates the outbut buffer for the decoder

ARGUMENTS PASSED:   vpu_dec  - VPU decoder plugins context
                    frameBuf - VPU's Output Frame Buffer to be 
                                   allocated.
                        
                    num_buffers number of frame buffers to be allocated
        
RETURN VALUE:       0 (SUCCESS)/ -1 (FAILURE)
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
gint mfw_gst_vpudec_FrameBufferInit(MfwGstVPU_Dec *vpu_dec,
                                    FrameBuffer *frameBuf,
                                    gint num_buffers)
{    
    gint i=0;
    GstFlowReturn retval=GST_FLOW_OK;
    GstBuffer *outbuffer=NULL;
    guint strideY=vpu_dec->initialInfo->picWidth;
    guint height=vpu_dec->initialInfo->picHeight;
    gint mvsize=0;
    gint sw_cnt = 0, hw_cnt = 0;

#ifndef VPU_MX27  // mx27 does not support motion vector buffer
    mvsize = strideY * height/4;
#endif
    vpu_dec->direct_render=FALSE;

    /* try to allocate one frame from v4lsink */    
    {
        gint limit = 30;
        /* FixME: why the videosink is in the wrong state */
        retval = gst_pad_alloc_buffer_and_set_caps(vpu_dec->srcpad, 0,
                    vpu_dec->yuv_frame_size,GST_PAD_CAPS(vpu_dec->srcpad), &outbuffer);
        if (retval != GST_FLOW_OK) {
                GST_ERROR(">>VPU_DEC: Error %d in allocating the Framebuffer[%d]\n",retval,i);
                
        }

        /* FixME: why the videosink is in the wrong state */

        while( ((limit--)>0) 
            && ((retval != GST_FLOW_OK) 
                || (!(GST_BUFFER_FLAG_IS_SET(outbuffer,GST_BUFFER_FLAG_LAST))))){
            usleep(30000);
            if (retval == GST_FLOW_OK)
              gst_buffer_unref(outbuffer);
            outbuffer = NULL;
            retval = gst_pad_alloc_buffer_and_set_caps(vpu_dec->srcpad, 0,
            vpu_dec->yuv_frame_size,GST_PAD_CAPS(vpu_dec->srcpad), &outbuffer);
            if (retval != GST_FLOW_OK) {
                    GST_ERROR(">>VPU_DEC: Error %d in allocating the Framebuffer[%d]\n",retval,i);
                    continue;
            }

        }
        if (outbuffer)
            gst_buffer_unref(outbuffer);
        else {
            GST_ERROR("Could not allocate Framebuffer");
            return -1;
        }
    }


    for (i=0;i<num_buffers;i++) {
        retval = gst_pad_alloc_buffer_and_set_caps(vpu_dec->srcpad, 0,
            vpu_dec->yuv_frame_size,GST_PAD_CAPS(vpu_dec->srcpad), &outbuffer);

        if (retval != GST_FLOW_OK) {
            GST_ERROR(">>VPU_DEC: Error %d in allocating the Framebuffer[%d]\n",retval,i);
            return -1;
        }
        /* if the buffer allocated is the Hardware Buffer use it as it is */
        if (!vpu_dec->force_nondr && 
            (GST_BUFFER_FLAG_IS_SET(outbuffer,GST_BUFFER_FLAG_LAST)==TRUE) )
        {
#ifndef VPU_MX27
            vpu_dec->frame_mem[i].size = mvsize;
            IOGetPhyMem(&vpu_dec->frame_mem[i]);
            frameBuf[i].bufMvCol = vpu_dec->frame_mem[i].phy_addr;
#endif
            vpu_dec->outbuffers[i] = outbuffer;
            GST_BUFFER_SIZE(vpu_dec->outbuffers[i]) = vpu_dec->yuv_frame_size;
            GST_BUFFER_OFFSET_END(vpu_dec->outbuffers[i]) = i;
            vpu_dec->fb_state_plugin[i] = FB_STATE_ALLOCATED;
            vpu_dec->fb_type[i] = FB_TYPE_GST;
            sw_cnt++;
            frameBuf[i].bufY = GST_BUFFER_OFFSET(outbuffer);
            frameBuf[i].bufCb = frameBuf[i].bufY + (strideY * height);
            frameBuf[i].bufCr = frameBuf[i].bufCb + ((strideY/2) * (height/2));
            /* 
             * Set the DR mode to TRUE
             *  If the first buffer is GST_BUFFER_FLAG_IS_SET
             *
             */
            if (i == 0) 
                vpu_dec->direct_render=TRUE;
        } else {
            // else allocate The Hardware buffer through IOGetPhyMem
           // Note this to support writing the output to a file in case of 
           // File Sink 
            if (outbuffer!=NULL) {
                gst_buffer_unref(outbuffer);
                outbuffer=NULL;
            }
            vpu_dec->frame_mem[i].size = vpu_dec->yuv_frame_size + mvsize;
            IOGetPhyMem(&vpu_dec->frame_mem[i]);

            // if not enough physical memory free all we allocated and exit
            if (vpu_dec->frame_mem[i].phy_addr == 0) {
                gint j;
                /* Free all the buffers */
                for (j = 0; j < i; j++) {
                    IOFreeVirtMem(&vpu_dec->frame_mem[j]);
                    IOFreePhyMem(&vpu_dec->frame_mem[j]);
                    vpu_dec->frame_mem[j].phy_addr = 0;
                    vpu_dec->frame_virt[j]=NULL;
                    if (vpu_dec->fb_type[j] == FB_TYPE_GST) 
                    {
                        gst_buffer_unref(vpu_dec->outbuffers[j]);
                    }
                }
                GST_ERROR(">>VPU_DEC: Not enough memory for framebuffer!\n");
                return -1;
            }

            frameBuf[i].bufY = vpu_dec->frame_mem[i].phy_addr;
            frameBuf[i].bufCb = frameBuf[i].bufY + (strideY * height);
            frameBuf[i].bufCr = frameBuf[i].bufCb + ((strideY/2) * (height/2));

#ifndef VPU_MX27
            frameBuf[i].bufMvCol = frameBuf[i].bufCr + ((strideY/2) * (height/2));
#endif
            vpu_dec->frame_virt[i] = (guint8 *)IOGetVirtMem(&vpu_dec->frame_mem[i]);
            vpu_dec->fb_type[i] = FB_TYPE_HW;
            hw_cnt++;
        }
    }
    if (vpu_dec->direct_render)
        GST_DEBUG(">>VPU_DEC: Use DR mode, GST buffer count:%d, HW buffer count:%d.\n",sw_cnt, hw_cnt);
    else
        GST_DEBUG(">>VPU_DEC: Use Non-DR mode.\n");

    for(i=0;i<num_buffers;i++) {
        GST_DEBUG(">>VPU_DEC: buffer:%d, type:%s\n",i, (vpu_dec->fb_type[i]==FB_TYPE_HW)? "FB_TYPE_HW":"FB_TYPE_GST" );
    }    
    return 0;
}


/*======================================================================================
FUNCTION:           mfw_gst_vpudec_vpu_open

DESCRIPTION:        Open VPU

ARGUMENTS PASSED:   vpu_dec  - VPU decoder plugins context
                    buffer - pointer to the input buffer which has the video data.

RETURN VALUE:       GstFlowReturn - Success of Failure.
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=======================================================================================*/
GstFlowReturn mfw_gst_vpudec_vpu_open(MfwGstVPU_Dec *vpu_dec)
{
    RetCode vpu_ret=RETCODE_SUCCESS;
    guint8 *virt_bit_stream_buf=NULL;

    GST_DEBUG(">>VPU_DEC: codec=%d\n",vpu_dec->codec);
    vpu_dec->bit_stream_buf.size = BUFF_FILL_SIZE;
    IOGetPhyMem(&vpu_dec->bit_stream_buf);
    virt_bit_stream_buf = (guint8 *)IOGetVirtMem(&vpu_dec->bit_stream_buf);
    vpu_dec->start_addr = vpu_dec->base_addr = virt_bit_stream_buf;
    vpu_dec->end_addr = virt_bit_stream_buf + BUFF_FILL_SIZE;

    if (vpu_dec->codec == STD_AVC) {
        vpu_dec->ps_mem_desc.size = PS_SAVE_SIZE;
        IOGetPhyMem(&vpu_dec->ps_mem_desc);
        vpu_dec->decOP->psSaveBuffer = vpu_dec->ps_mem_desc.phy_addr;
        vpu_dec->decOP->psSaveBufferSize = PS_SAVE_SIZE;

        vpu_dec->slice_mem_desc.size = SLICE_SAVE_SIZE;
        IOGetPhyMem(&vpu_dec->slice_mem_desc);
    }

    vpu_dec->decOP->bitstreamBuffer = vpu_dec->bit_stream_buf.phy_addr;
    vpu_dec->decOP->bitstreamBufferSize = BUFF_FILL_SIZE;

#if (defined (VPU_MX37) || defined (VPU_MX51)) 
    if (vpu_dec->fmt==0){
        GST_DEBUG(">>VPU_DEC: set chromainterleave\n");
        vpu_dec->decOP->chromaInterleave = 1;
    }
#endif
    // For MX37 and beyond VPU handles file play mode well and is more efficient
    // however MPEG2 requires streaming mode on all chips
    // If you are using the VPU plugins outside of a parser then you must disable
    // file play mode on these chips also
    
    vpu_dec->file_play_mode = TRUE;
#if (defined (VPU_MX37) || defined (VPU_MX51))
    vpu_dec->decOP->reorderEnable = 0;

    if ((vpu_dec->codec == STD_AVC) || 
        (vpu_dec->codec == STD_VC1) ||
        (vpu_dec->codec == STD_RV)  ||
        (vpu_dec->codec == STD_DIV3))
    {
        vpu_dec->decOP->reorderEnable = 1;
    }
    if (vpu_dec->codec == STD_MPEG2)
    {
        vpu_dec->decOP->reorderEnable = 0;

        // note that some mpeg2 files can play in file play mode but it is hard to know
        // exactly which ones so for now assume all streaming mode (which does not play as well sometimes)
        vpu_dec->file_play_mode = FALSE;	
        //vpu_dec->set_ts_manually = TRUE;
    }
    vpu_dec->decOP->mp4Class = vpu_dec->mp4Class;
#else // MX27 does not support streaming mode and reordering - disable these
    vpu_dec->decOP->reorderEnable = 0;
    // file play mode might improve latency in loopback
    // but it does not work in all case so use it in specific cases like this
#endif

    // if caps weren't set time_per_frame was not set - so no parser input
    // if parser_input not set then assume streaming mode    
    if (!vpu_dec->parser_input || (vpu_dec->time_per_frame == 0))  
        vpu_dec->file_play_mode = FALSE;

    if (vpu_dec->time_per_frame == 0) 
    {
        vpu_dec->time_per_frame = gst_util_uint64_scale_int (GST_SECOND, 1, DEFAULT_FRAME_RATE_NUMERATOR);
        vpu_dec->set_ts_manually = TRUE;
        vpu_dec->nal_check = FALSE;
    }
    if (vpu_dec->min_latency)
        vpu_dec->set_ts_manually = TRUE;

    vpu_dec->decOP->filePlayEnable = (vpu_dec->file_play_mode) ? 1 : 0;
    GST_DEBUG(">>VPU_DEC: Setting file play mode to %d\n", vpu_dec->decOP->filePlayEnable);
    
    vpu_dec->decOP->bitstreamFormat = vpu_dec->codec;
    vpu_dec->decOP->picWidth = vpu_dec->picWidth;
    vpu_dec->decOP->picHeight = vpu_dec->picHeight;

    vpu_dec->base_write = vpu_dec->bit_stream_buf.phy_addr;
    vpu_dec->end_write = vpu_dec->bit_stream_buf.phy_addr + BUFF_FILL_SIZE;
    vpu_dec->last_ts_sent = 0;
    vpu_dec->last_ts_in = 0;

    /* open a VPU's decoder instance */
    vpu_ret = vpu_DecOpen(vpu_dec->handle,vpu_dec->decOP);
    if (vpu_ret != RETCODE_SUCCESS) {
        GST_ERROR(">>VPU_DEC: vpu_DecOpen failed. Error code is %d \n", vpu_ret);
        return GST_FLOW_ERROR;
    }
    vpu_dec->vpu_opened = TRUE;
    return GST_FLOW_OK;
}

/*======================================================================================
FUNCTION:           mfw_gst_vpudec_copy_data

DESCRIPTION:        Copies data to the VPU input 

ARGUMENTS PASSED:   vpu_dec  - VPU decoder plugins context

RETURN VALUE:       GstFlowReturn - Success of Failure.
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=======================================================================================*/
GstFlowReturn mfw_gst_vpudec_copy_data(MfwGstVPU_Dec *vpu_dec)
{
    RetCode vpu_ret=RETCODE_SUCCESS;
    guint  size_to_copy=GST_BUFFER_SIZE(vpu_dec->gst_buffer) - vpu_dec->buff_consumed;
    guint8 *buffer_to_copy = GST_BUFFER_DATA(vpu_dec->gst_buffer);
    PhysicalAddress p1, p2;
    Uint32 space=BUFF_FILL_SIZE;

    if (!vpu_dec->vpu_init  || !vpu_dec->file_play_mode)
    {
        vpu_ret = vpu_DecGetBitstreamBuffer(*(vpu_dec->handle), &p1, &p2, &space);
        if (vpu_ret != RETCODE_SUCCESS) {
            GST_ERROR(">>VPU_DEC: vpu_DecGetBitstreamBuffer failed. Error is %d \n", vpu_ret);
            return GST_FLOW_ERROR;
        } 
        if ((space <= 0) && (vpu_dec->num_timeouts == 0))
        {
            if (vpu_dec->vpu_init)
            {
                GST_ERROR (">>VPU_DEC: vpu_DecGetBitstreamBuffer returned zero space do a reset \n");
                //in this case our bitstream buffer is full - VPU is not able to decode into our buffers
                mfw_gst_vpudec_reset(vpu_dec);
                return GST_FLOW_OK; // can't do copy yet since we need that first frame which is done in copy_sink
            } else {
                GST_ERROR (">>VPU_DEC: vpu_DecGetBitstreamBuffer returned zero space flush buffer \n");
                vpu_ret = vpu_DecBitBufferFlush(*vpu_dec->handle);
                memset(&vpu_dec->timestamp_buffer[0],0,MAX_STREAM_BUF*sizeof(GstClockTime));
                vpu_dec->ts_rx=0;
                vpu_dec->ts_tx=0;
                vpu_ret = vpu_DecGetBitstreamBuffer(*(vpu_dec->handle), &p1, &p2, &space);
                if (space <= 0)
                   return GST_FLOW_ERROR;
            }
        }
        vpu_dec->data_in_vpu = BUFF_FILL_SIZE - space;
    } 

    // For File Play mode - copy over data from start of buffer and update VPU first time
    if (vpu_dec->file_play_mode == TRUE)
    {
        if ((vpu_dec->vpu_init == FALSE) && (space < size_to_copy))
        {
            GST_ERROR (">>VPU_DEC: copy_data failed space %d less then size_to_copy %d\n", space, size_to_copy);
            return GST_FLOW_ERROR;
        }

        memcpy(vpu_dec->start_addr, buffer_to_copy, size_to_copy);
        vpu_dec->decParam->chunkSize = size_to_copy;

        if (!vpu_dec->vpu_init)
        {
       	    /* Update Bitstream buffer just at init time */
            vpu_ret = vpu_DecUpdateBitstreamBuffer(*(vpu_dec->handle), (space <= 0)? BUFF_FILL_SIZE : space);
            if (vpu_ret != RETCODE_SUCCESS) {
           	    GST_ERROR(">>VPU_DEC: vpu_DecUpdateBitstreamBuffer failed. Error is %d \n", vpu_ret);
                return (GST_FLOW_ERROR);
            }
        }
        GST_FRAMEDBG(">>VPU_DEC: copy data size_to_copy %d  space %d\n", size_to_copy, space);

    } else {  // handle streaming mode
        guint8 *buffer_to_copy = GST_BUFFER_DATA(vpu_dec->gst_buffer)+vpu_dec->buff_consumed;

  	    // check if we can only copy part of the buffer if so copy what what we can this time
        if (space <= size_to_copy) {
            size_to_copy = space-4;
        }

		// check if we can do just one memcpy
        if ((vpu_dec->start_addr + size_to_copy) <= vpu_dec->end_addr) 
        {
            memcpy(vpu_dec->start_addr, buffer_to_copy, size_to_copy);
            vpu_dec->start_addr += size_to_copy;
        } 
		else  // otherwise split it into two copies - one at end and one from beginning
        {
            guint residue = (vpu_dec->end_addr - vpu_dec->start_addr);
            memcpy(vpu_dec->start_addr,buffer_to_copy,residue);
            memcpy(vpu_dec->base_addr,buffer_to_copy + residue, size_to_copy-residue);
            vpu_dec->start_addr = vpu_dec->base_addr + size_to_copy - residue;
        }

        // Now update the bitstream buffer with the amount we just copied
        vpu_ret = vpu_DecUpdateBitstreamBuffer(*(vpu_dec->handle), size_to_copy);
        if (vpu_ret != RETCODE_SUCCESS) {
            GST_ERROR("vpu_DecUpdateBitstreamBuffer failed. Error code is %d \n", vpu_ret);
            return (GST_FLOW_ERROR);
        }

        vpu_dec->data_in_vpu += size_to_copy;
        GST_FRAMEDBG(">>VPU_DEC: copy data data_in_vpu %d min_data_in_vpu %d space %d\n", vpu_dec->data_in_vpu,vpu_dec->min_data_in_vpu, space);
    }

    // add our totals and see if buffer is now consumed - if so unref the input buffer
    vpu_dec->buff_consumed += size_to_copy;
    if (vpu_dec->buff_consumed >= GST_BUFFER_SIZE(vpu_dec->gst_buffer))
    {
        gst_buffer_unref(vpu_dec->gst_buffer);
        vpu_dec->gst_buffer = NULL;
        vpu_dec->buff_consumed = 0;
        vpu_dec->must_copy_data = FALSE;
    } else {
        vpu_dec->must_copy_data = TRUE;
    }

    return GST_FLOW_OK;
}


/*======================================================================================
FUNCTION:           mfw_gst_vpudec_passthru

DESCRIPTION:        Copy undecoded data downstream

ARGUMENTS PASSED:   vpu_dec  - VPU decoder plugins context

RETURN VALUE:       GstFlowReturn - Success of Failure.
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=======================================================================================*/
GstFlowReturn mfw_gst_vpudec_passthru(MfwGstVPU_Dec *vpu_dec)
{
    GstFlowReturn retval=GST_FLOW_OK;
    if (!vpu_dec->vpu_init)
    {
        GstCaps *caps=gst_caps_new_simple("video/x-raw-yuv", NULL);

        if (!(gst_pad_set_caps(vpu_dec->srcpad, caps))) {
            GST_ERROR ("\n>>VPU_DEC: Passthru could not set the caps for src pad\n");
        }
        gst_caps_unref(caps);
        caps = NULL;
        vpu_dec->vpu_init=TRUE;
    } 
    GST_DEBUG (">>VPU_DEC: Passthru pushing %d downstream \n", GST_BUFFER_SIZE(vpu_dec->gst_buffer));  
    retval = gst_pad_alloc_buffer_and_set_caps(vpu_dec->srcpad, 0,
                                              GST_BUFFER_SIZE(vpu_dec->gst_buffer),
                                              GST_PAD_CAPS(vpu_dec->srcpad), 
                                              &vpu_dec->pushbuff);
    if (retval != GST_FLOW_OK) {
       GST_ERROR(">>VPU_DEC: Error %d allocating the pass thru buffer\n", retval);
       return retval;
    }
    memcpy(GST_BUFFER_DATA(vpu_dec->pushbuff),
          GST_BUFFER_DATA(vpu_dec->gst_buffer),
          GST_BUFFER_SIZE(vpu_dec->gst_buffer));
    gst_buffer_unref(vpu_dec->gst_buffer);
    vpu_dec->gst_buffer = NULL;
    retval = gst_pad_push(vpu_dec->srcpad,vpu_dec->pushbuff);
    if (retval != GST_FLOW_OK) {
        GST_ERROR(">>VPU_DEC: Error %d pushing buffer downstream \n", retval);
        return retval;
    }
    return GST_FLOW_OK;
}

/*======================================================================================
FUNCTION:           mfw_gst_scan_data_found_nal_header

DESCRIPTION:     * no nal header found, then we need to scan all the buffer
    		     * 1. find whether it have jump length greater than buffer length.
    		     *    if does not then replace the code with nal header until find it
    		     *    if found large length, then do nothing, and jump out
    		     ****************************************************************

ARGUMENTS PASSED:  
                    buffer  - input buffer
                    end - the end address of current buffer

RETURN VALUE:       GstFlowReturn - Success of Failure.

=======================================================================================*/

void mfw_gst_scan_data_found_nal_header(unsigned char *buf, int end)
{
    unsigned char *temp_buf;
    guint32 jump, data_size, unlegal_flag, counter;
	unlegal_flag = 0;
	data_size = 0;
	counter = 0;
	temp_buf = buf;
    while (((guint32)buf)<end) 
    {        
        jump = (buf[0]<<24)|(buf[1]<<16)|(buf[2]<<8)|(buf[3]);
        if (jump==0x00000001) 
		{
            //("nal found\n");
            break;
        }
		else
		{
			data_size = end - (guint32)buf;
			if(jump > data_size)
			{
			    unlegal_flag = 1;
				break;
			}
		}
		counter++;
        buf += (jump+4);
    }
    /* nal header replace procedure */
	if(!unlegal_flag)
	{
	    while(counter)
	    {
			jump = (temp_buf[0]<<24)|(temp_buf[1]<<16)|(temp_buf[2]<<8)|(temp_buf[3]);	    
            temp_buf[2] = temp_buf[1] = temp_buf[0] = 0;
            temp_buf[3] = 0x01;
			temp_buf += (jump + 4);
			counter--;
	    }
	}
}
/*======================================================================================
FUNCTION:           mfw_gst_AVC_Fix_NALheader

DESCRIPTION:        Modify the buffer so that the length 4 bytes is changed to start code
                    open source demuxers like qt and ffmpeg_mp4 generate length+payload
                    instead of startcode plus payload which VPU can't support

ARGUMENTS PASSED:   vpu_dec  - VPU decoder plugins context
                    buffer - pointer to the input buffer which has the video data.

RETURN VALUE:       GstFlowReturn - Success of Failure.
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=======================================================================================*/
GstFlowReturn mfw_gst_AVC_Fix_NALheader(MfwGstVPU_Dec *vpu_dec)
{
    GstBuffer *buffer = vpu_dec->gst_buffer;
    guint32 jump;
    unsigned char * buf = GST_BUFFER_DATA(buffer);
    guint32 end = (guint32)(GST_BUFFER_DATA(buffer)+GST_BUFFER_SIZE(buffer)-4);
    GstBuffer *hdrBuf=NULL;
	
    if (vpu_dec->firstFrameProcessed == FALSE)
    {
        if (vpu_dec->codec_data_len>=8)
        {
            unsigned char *buf_hdr = GST_BUFFER_DATA(vpu_dec->codec_data);
            unsigned char *bufout = NULL;
            gint length = (buf_hdr[6] << 8) | buf_hdr[7];
            if ((length+8)>vpu_dec->codec_data_len){
                 return GST_FLOW_OK;
            }
            hdrBuf = gst_buffer_new_and_alloc(vpu_dec->codec_data_len);
            bufout = GST_BUFFER_DATA(hdrBuf);
           
            {   // must be qtdemux input which must be parsed more carefully
                // if any codec data from qtdemux then add startcode and parse our paylose
                // usually there are 3 sections.  First section has 8 bytes with last 2 bytes
                // being length so put a startcode plus copy the length from codec data
                // next section is 3 bytes with 2nd two bytes being length (usually 4 bytes)
                // so put another start code and copy 4 bytes from codec data
                // last step is make sure input buffer is fixed with start codes instead of
                // length
                
                gint length2 = 0;
                gint total_length = length+4;
                GST_DEBUG (" AVC SPS header length=%d\n", length);
                bufout[2] = bufout[1] = bufout[0] = 0;
                bufout[3] = 0x01;
                memcpy(bufout+4,buf_hdr + 8,length);
                length2 = vpu_dec->codec_data_len - length - 8;
                if (length2 > 0)
                {
                    gint offset = length+4;  // start code plus length copied
                    gint offset_src = length+8+3;
                    length2 = (buf_hdr[length+8+1]<<8) | buf_hdr[length+8+2];
                    GST_DEBUG (" AVC PPS Header length=%d\n", length2);
                    bufout[offset] = bufout[offset+1] = bufout[offset+2] = 0;
                    bufout[offset+3] = 0x01;
                    offset += 4;        
                    memcpy(bufout + offset, buf_hdr + offset_src, length2);
                    total_length += length2+4;
                }
                GST_BUFFER_SIZE(hdrBuf) = total_length;
            }
        } else { 
            guint32 startcode =  (buf[0]<<24)|(buf[1]<<16)|(buf[2]<<8)|(buf[3]);           
            if ((startcode==0x00000001) && (vpu_dec->file_play_mode == FALSE)) {
                if ((buf[4] != 0x67) && (buf[4] != 0x27))                 
                    return GST_FLOW_ERROR;
                else {
                   return GST_FLOW_OK;
                }
            } 
        } 

    }
    if (vpu_dec->accumulate_hdr)
    {
        guint32 startcode =  (buf[0]<<24)|(buf[1]<<16)|(buf[2]<<8)|(buf[3]);           
        if (startcode==0x00000001) {
            if ((buf[4] == 0x67) || (buf[4] == 0x68))                 
            {   // no decoding of just headers
                vpu_dec->hdr_received = TRUE;
                return GST_FLOW_OK; 
            }  
        }

    }
    
    
   jump = (buf[0]<<24)|(buf[1]<<16)|(buf[2]<<8)|(buf[3]);
    	
   if (jump==0x00000001) 
   {
   	  GST_DEBUG("nal found in avc data\n");
   }
   else
   {
	    /* judge whether hdr received for the first time */
        if (((buf[4] == 0x67) || (buf[4] == 0x68)))
            vpu_dec->hdr_received = TRUE;

		
	    /* no nal header found, then we need to scan all the buffer
	     * 1. find whether it have jump length greater than buffer length.
	     *    if does not then replace the code with nal header until find it
	     *    if found large length, then do nothing, and jump out
	     ****************************************************************/
        mfw_gst_scan_data_found_nal_header(buf, end);
		
   }
   if (hdrBuf)
       vpu_dec->gst_buffer = gst_buffer_join(hdrBuf,vpu_dec->gst_buffer);

    return GST_FLOW_OK;
}


/*======================================================================================
FUNCTION:           mfw_gst_vpudec_check_time

DESCRIPTION:        Check timestamp and do frame dropping if enabled

ARGUMENTS PASSED:   vpu_dec  - VPU decoder plugins context
                    buffer - pointer to the input buffer which has the video data.

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=======================================================================================*/
void mfw_gst_vpudec_check_time(MfwGstVPU_Dec *vpu_dec, GstBuffer *buffer)
{
    // Time stamp Buffer is a circular buffer to store the timestamps which are later 
    // used while pushing the decoded frame onto the Sink element
    if (GST_CLOCK_TIME_IS_VALID(GST_BUFFER_TIMESTAMP(buffer))) 
    {
        GstClockTime ts = GST_BUFFER_TIMESTAMP(buffer);

        GST_TIMESTAMP(">>VPU_DEC: Input ts = % " GST_TIME_FORMAT " last ts in %"  GST_TIME_FORMAT " ts_rx=%d \n", GST_TIME_ARGS(ts), GST_TIME_ARGS(vpu_dec->last_ts_in), vpu_dec->ts_rx);

        if ( vpu_dec->last_ts_in && 
            (vpu_dec->last_ts_in != GST_CLOCK_TIME_NONE) &&
            (ts < vpu_dec->last_ts_in))
        {
             ts = GST_CLOCK_TIME_NONE;
        }
        if (ts != GST_CLOCK_TIME_NONE)
        {
            if (vpu_dec->last_ts_in == ts)
                 return;  // happens on some double loops
            vpu_dec->last_ts_in = ts;
        }

      
        vpu_dec->timestamp_buffer[vpu_dec->ts_rx] = ts;
        vpu_dec->ts_rx = (vpu_dec->ts_rx+1)%MAX_STREAM_BUF;

        // Check for need to do drop before decode - only do until we have at least 2 I frames received (1 gop)
        // if we are already dropping then just ignore check until next I frame
        // if we are having unpredictable gops then don't frame drop
        if (vpu_dec->frame_drop_allowed && GST_ELEMENT (vpu_dec)->clock && vpu_dec->last_ts_sent &&
           (ts > vpu_dec->last_ts_in) &&
           (vpu_dec->num_gops>1) && !vpu_dec->drop_before_decode)
        {
            // calculate the offset of this time stamp and where we are in the stream
            // this assumes that the timestamp coming in is close to the current time
            GstClockTime offset = gst_clock_get_time (GST_ELEMENT (vpu_dec)->clock);
            offset -= gst_element_get_base_time(GST_ELEMENT (vpu_dec));
            offset -= vpu_dec->clock_base;

            // Is our decoding behind
            if (offset > vpu_dec->last_ts_sent)
            {
                // calculate how many frames we are behind
                guint num_frames_late = (guint)(offset-vpu_dec->last_ts_sent)/(guint)(vpu_dec->time_per_frame);

                // are we more than 4 frames late? don't check if we can't predict gops reliably
                if ((num_frames_late > 4) && vpu_dec->predict_gop)
                {
                    GstClockTime next_gop_offset = (vpu_dec->gop_size - 
                                                   (vpu_dec->frames_decoded - vpu_dec->idx_last_gop));
                    next_gop_offset = next_gop_offset*vpu_dec->time_per_frame + vpu_dec->last_ts_sent;

                    // How far into our gop are we and if over 50% of gop and next gop will be late then start
                    // skipping until next I frame
                    guint idx_into_gop = vpu_dec->frames_decoded - vpu_dec->idx_last_gop;
                    if ((idx_into_gop > (vpu_dec->gop_size>>1)) /*&& 
                         ((offset+vpu_dec->time_per_frame) > next_gop_offset) */)
                    {
                        vpu_dec->drop_before_decode=TRUE;
                        vpu_dec->just_flushed = TRUE; //easier to check both below
                        if (vpu_dec->file_play_mode)
                            vpu_dec->decParam->iframeSearchEnable = 1;
                        else {
                            vpu_dec->decParam->skipframeMode = 1;
                            vpu_dec->decParam->skipframeNum = 1;
                            // unfortunately H.264 gets garbage in this unless you wait for next I frame - so keep gop sizes smaller!
                            // MPEG4 does not have this bug
                            //if (vpu_dec->codec == STD_AVC) 
                             //   vpu_dec->check_for_iframe = TRUE; // use this instead of below which causes garbage video
                            // below seems to cause VPU to get into hang state so avoid in frame dropping mode
                            //vpu_DecBitBufferFlush(*vpu_dec->handle);
                            //vpu_dec->start_addr = vpu_dec->base_addr;
                        }
                        GST_FRAMEDROP(">>VPU_DEC: Start Frame dropping at %d frames late=%d gop_size=%d\n", vpu_dec->frames_decoded, num_frames_late, vpu_dec->gop_size);
                        GST_FRAMEDROP(">>VPU_DEC: frames decoded %d idx_last_gop =%d\n", vpu_dec->frames_decoded, vpu_dec->idx_last_gop);
                    } 
                    //else GST_FRAMEDROP(">>VPU_DEC: No drop but %d frames late\n", num_frames_late);
                }
            } 
        } /* end frame drop before decode check */
    } else {
        // Some MPEG clips have NO time stamps so we must calculate it manually to slow down video
        // Some have timestamps on some buffers and not on others since it takes multiple buffers to generate a frame to render
        // in this case you skip the invalid timestamps
        // some will have one random invalid timestamp - usually where paste points are so it is best to ignore it
        if (vpu_dec->last_ts_in && (vpu_dec->last_ts_in != GST_CLOCK_TIME_NONE) && (vpu_dec->ts_rx != vpu_dec->ts_tx))
        {
            //GST_TIMESTAMP (">>VPU_DEC: Invalid timestamp once - skip now last_ts %d\n", (guint)vpu_dec->last_ts_in);
            //GST_TIMESTAMP (">>VPU_DEC: Invalid timestamp once - calculate manually last_ts %d\n", (guint)vpu_dec->last_ts_in);        
            //vpu_dec->timestamp_buffer[vpu_dec->ts_rx] = vpu_dec->last_ts_in + vpu_dec->time_per_frame;   
            //vpu_dec->last_ts_in = GST_CLOCK_TIME_NONE;     
        } else {  // some clips have no timestamps so later force it to be set via manually
            vpu_dec->timestamp_buffer[vpu_dec->ts_rx] = GST_CLOCK_TIME_NONE;
            vpu_dec->last_ts_in = 0;
            vpu_dec->last_ts_sent = 0;
            //GST_TIMESTAMP (">>VPU_DEC: Invalid timestamp previous was invalid \n");
            vpu_dec->ts_rx = (vpu_dec->ts_rx+1)%MAX_STREAM_BUF;
        }
    }
}

/*======================================================================================
FUNCTION:           mfw_gst_receive_mpeg2ts

DESCRIPTION:        Check timestamp and do frame dropping if enabled

ARGUMENTS PASSED:   vpu_dec  - VPU decoder plugins context
                    buffer - pointer to the input buffer which has the video data.

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=======================================================================================*/
void mfw_gst_receive_mpeg2ts(MfwGstVPU_Dec *vpu_dec, GstBuffer * buffer)
{
    GstClockTime timestamp = GST_BUFFER_TIMESTAMP(buffer);
    if (GST_CLOCK_TIME_IS_VALID(timestamp)){
        vpu_dec->timestamp_buffer[vpu_dec->ts_rx] = 
            timestamp;
        vpu_dec->ts_rx= ((vpu_dec->ts_rx+1) %MAX_STREAM_BUF);
    }else{
        vpu_dec->no_timestamp_count++;
    }
}

/*======================================================================================
FUNCTION:           mfw_gst_get_mpeg2ts

DESCRIPTION:        Check timestamp and do frame dropping if enabled

ARGUMENTS PASSED:   vpu_dec  - VPU decoder plugins context
                    ptimestamp - returned timestamp to use at render

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=======================================================================================*/
gboolean mfw_gst_get_mpeg2ts(MfwGstVPU_Dec *vpu_dec, GstClockTime * ptimestamp)
{
    gboolean found = FALSE;
    int i = vpu_dec->ts_tx;
    int index;
    GstClockTime timestamp = 0;
    GstClockTime half_interval = vpu_dec->time_per_frame/2;
    while(i!=vpu_dec->ts_rx){
        if (found){
            if (vpu_dec->timestamp_buffer[i]<timestamp){
                timestamp = vpu_dec->timestamp_buffer[i];
                index = i;
            }
        }else{
            timestamp = vpu_dec->timestamp_buffer[i];
            index = i;
            found = TRUE;
        }
        i = ((i+1) %MAX_STREAM_BUF);
    }
    if (found){
        if ((vpu_dec->no_timestamp_count) && (half_interval)){
            if (vpu_dec->timestamp_buffer[index]>=((*ptimestamp)+half_interval)){
                vpu_dec->no_timestamp_count--;
                return FALSE;
            }
        }
        if (index!=vpu_dec->ts_tx){
            vpu_dec->timestamp_buffer[index] = 
                vpu_dec->timestamp_buffer[vpu_dec->ts_tx];
        }
        vpu_dec->ts_tx= ((vpu_dec->ts_tx+1)%MAX_STREAM_BUF);
        //g_print("output %" GST_TIME_FORMAT "\n", GST_TIME_ARGS(timestamp));
        *ptimestamp = timestamp;
        return TRUE;
    }else{
        return FALSE;
    }
}

void decoder_close(MfwGstVPU_Dec *vpu_dec)
{
	DecOutputInfo outinfo = {0};
	RetCode ret;

	ret = vpu_DecClose(*vpu_dec->handle);
	if (ret == RETCODE_FRAME_NOT_COMPLETE) {
		vpu_DecGetOutputInfo(*vpu_dec->handle, &outinfo);
		ret = vpu_DecClose(*vpu_dec->handle);
		if (ret != RETCODE_SUCCESS)
			g_print("vpu_DecClose failed\n");
	}
}
/*======================================================================================
FUNCTION:           mfw_gst_vpudec_reset

DESCRIPTION:        Resets VPU

ARGUMENTS PASSED:   vpu_dec  - VPU decoder plugins context

RETURN VALUE:       GstFlowReturn - Success of Failure.
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=======================================================================================*/
GstFlowReturn mfw_gst_vpudec_reset(MfwGstVPU_Dec *vpu_dec)
{
    RetCode vpu_ret = RETCODE_SUCCESS;
    guint i=0;  
    GST_DEBUG(">>VPU_DEC: Reset close and reopen using same previously allocated memory \n");      
    for (i=0;i<vpu_dec->numframebufs;i++)
    {
        if (vpu_dec->fb_state_plugin[i]!=FB_STATE_DISPLAY)
            vpu_dec->fb_state_plugin[i] = FB_STATE_FREE;
    }
    mfw_gst_vpudec_release_buff(vpu_dec);
    
    // if VPU is busy you can't cleanup and close will hang so be sure to set
    // bitstream to 0 so reset can happen
    if (vpu_IsBusy())
        vpu_DecUpdateBitstreamBuffer(*(vpu_dec->handle), 0);

    i=0;
    while (vpu_IsBusy())
    {
        //GST_DEBUG (">>VPU_DEC: waiting for VPU to get out of busy state \n");
        usleep(500000);
        i++;
        if (i>100)
            return GST_FLOW_ERROR; // avoid infinite loop but VPU will not be closed
    }

    vpu_dec->data_in_vpu = 0; 
    if (vpu_dec->is_frame_started)
        vpu_ret = vpu_DecGetOutputInfo(*(vpu_dec->handle),vpu_dec->outputInfo);
    if (vpu_ret != RETCODE_SUCCESS) 
        GST_ERROR("vpu_DecGetOutputInfo failed. Error code is %d \n", vpu_ret);
#if 0    
    vpu_ret = vpu_DecClose(*vpu_dec->handle);
    if (vpu_ret != RETCODE_SUCCESS) 
        GST_ERROR("vpu_DecClose failed. Error code is %d \n", vpu_ret);
#else
    decoder_close(vpu_dec);
#endif
    vpu_dec->vpu_init=FALSE;
    vpu_dec->firstFrameProcessed=FALSE;
    memset(vpu_dec->handle ,0,sizeof(DecHandle));
    vpu_ret = vpu_DecOpen(vpu_dec->handle,vpu_dec->decOP);
    if (vpu_ret != RETCODE_SUCCESS) {
        GST_ERROR("vpu_DecOpen failed. Error code is %d \n", vpu_ret);
    }
    vpu_dec->is_frame_started = FALSE;
    vpu_dec->start_addr = vpu_dec->base_addr;
    vpu_dec->eos = FALSE;
    vpu_dec->frames_decoded = 0;     
    memset(&vpu_dec->timestamp_buffer[0],0,MAX_STREAM_BUF*sizeof(GstClockTime));
    vpu_dec->ts_rx=0;
    vpu_dec->ts_tx=0;
    vpu_dec->last_ts_sent = 0;      
    vpu_dec->last_ts_in = 0;   
    vpu_dec->num_timeouts = 0;

    if (vpu_dec->codec == STD_MPEG2)
    {
        vpu_dec->firstFrameProcessed=TRUE;
        memcpy(vpu_dec->start_addr, GST_BUFFER_DATA(vpu_dec->codec_data), vpu_dec->codec_data_len);
        vpu_dec->start_addr += vpu_dec->codec_data_len;
        // Now update the bitstream buffer with the amount we just copied
        vpu_ret = vpu_DecUpdateBitstreamBuffer(*(vpu_dec->handle), vpu_dec->codec_data_len);
        if (vpu_ret != RETCODE_SUCCESS) {
            GST_ERROR("vpu_DecUpdateBitstreamBuffer failed. Error code is %d \n", vpu_ret);
           return (GST_FLOW_ERROR);
        }
        vpu_dec->data_in_vpu += vpu_dec->codec_data_len;
        GST_DEBUG(">>VPU_DEC: Merging the codec data over size %d\n", vpu_dec->codec_data_len);
    }

    return GST_FLOW_OK;
}
/*======================================================================================
FUNCTION:           mfw_gst_vpudec_copy_sink_input

DESCRIPTION:        Update bitstream buffer in streaming mode which means we might not
                    be getting a full frame from upstream as we are in file play mode

ARGUMENTS PASSED:   vpu_dec  - VPU decoder plugins context
                    buffer - pointer to the input buffer which has the video data.

RETURN VALUE:       GstFlowReturn - Success of Failure.
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=======================================================================================*/
GstFlowReturn mfw_gst_vpudec_copy_sink_input(MfwGstVPU_Dec *vpu_dec, GstBuffer *buffer)
{
    RetCode vpu_ret = RETCODE_SUCCESS;

    vpu_dec->hdr_received = FALSE;

    // Check EOS and reset VPU accordingly
    if (vpu_dec->vpu_init && (G_UNLIKELY(buffer == NULL || vpu_dec->eos))) {
        // check if a non-null buffer was passed - in this case we were in EOS but are getting
        // more data so we need to close and reopen VPU to reset VPU before copying data

        vpu_ret = vpu_DecUpdateBitstreamBuffer(*(vpu_dec->handle), 0);
        if (vpu_ret != RETCODE_SUCCESS) {
             GST_ERROR(">>VPU_DEC: vpu_DecUpdateBitstreamBuffer failed. Error is %d \n", vpu_ret);
             return GST_FLOW_ERROR;
        }
        if (buffer && vpu_dec->eos)
        {    
            mfw_gst_vpudec_reset(vpu_dec);
        } else {  // normal EOS case - mark so we can decode/display remaining data in VPU
            if (vpu_dec->vpu_init && !vpu_dec->eos && (vpu_dec->decOP->reorderEnable == 1))
            {
                // Make display reorder buffer TRUE to read the decoded frames which are pending with VPU
                vpu_dec->decParam->dispReorderBuf = 1;
            }
            vpu_dec->eos = TRUE;
            vpu_dec->must_copy_data = FALSE;
            GST_DEBUG(">>VPU_DEC: After EOS set reorder enable\n");      
            return GST_FLOW_OK;  // nothing else left to do since buffer is null
        }
        vpu_dec->check_for_bframe = FALSE;
        vpu_dec->just_flushed = FALSE;
        vpu_dec->decParam->iframeSearchEnable = 0;
        vpu_dec->decParam->skipframeMode = 0;
        vpu_dec->decParam->skipframeNum = 0;
    }

    if (vpu_dec->is_frame_started && !vpu_dec->num_timeouts)
        return GST_FLOW_OK;

    if (buffer == NULL)
        return GST_FLOW_OK;

    // save the time stamp and see how late we are 
#if 1
   if ((vpu_dec->codec!=STD_MPEG2) && (vpu_dec->codec!=STD_RV))
        mfw_gst_vpudec_check_time(vpu_dec, buffer);
   else
      mfw_gst_receive_mpeg2ts(vpu_dec, buffer);
#else
   mfw_gst_vpudec_check_time(vpu_dec, buffer);
#endif


    // Check for first frame and copy codec specific data needed 
    // before copying input buffer to VPU
    if (G_UNLIKELY(vpu_dec->firstFrameProcessed==FALSE)){//first packet
        if ((vpu_dec->codec==STD_VC1)&&(vpu_dec->picWidth!=0))
        {
            // Creation of RCV Header is done in case of ASF Playback pf VC-1 streams
            //   from the parameters like width height and Header Extension Data */
            GstBuffer *tempBuf = mfw_gst_VC1_Create_RCVheader(vpu_dec,buffer);
            if (tempBuf==NULL){
                g_print("error in create rcv\n");
                return GST_FLOW_ERROR;
            }
            vpu_dec->gst_buffer = gst_buffer_join(tempBuf,vpu_dec->gst_buffer);
        }
        if ((vpu_dec->codec == STD_MPEG4) && vpu_dec->codec_data_len) 
        {
            // or for open source demux (qtdemux), it also needs the initial codec data
            GstBuffer *tempBuf = gst_buffer_new_and_alloc(vpu_dec->codec_data_len);
            memcpy(GST_BUFFER_DATA(tempBuf),GST_BUFFER_DATA(vpu_dec->codec_data),
                vpu_dec->codec_data_len);
            vpu_dec->gst_buffer = gst_buffer_join(tempBuf,buffer);
            if (vpu_dec->check_for_iframe)
            {
               vpu_dec->just_flushed = TRUE;
               if (vpu_dec->file_play_mode)
                   vpu_dec->decParam->iframeSearchEnable = 1;  
               else {
                   vpu_dec->decParam->skipframeMode = 1;
                   vpu_dec->decParam->skipframeNum = 1;
               }
               vpu_dec->check_for_iframe = FALSE; // not needed now with other settings
            }
        } 
        if (vpu_dec->codec == STD_AVC)
        {
            // this routine checks and makes sure we are starting on an I frame.  The VPU encoder
            // in streaming mode will always send headers with an I frame so checking for headers
            // helps us avoid starting VPU on a non-I frame which it does not like!
            if (vpu_dec->check_for_iframe)
            {
                vpu_dec->just_flushed = TRUE;
                if (vpu_dec->file_play_mode)
                    vpu_dec->decParam->iframeSearchEnable = 1;  
                else {
                    vpu_dec->decParam->skipframeMode = 1;
                    vpu_dec->decParam->skipframeNum = 1;
                }
            }

            if ((vpu_dec->nal_check || !vpu_dec->vpu_init) && mfw_gst_AVC_Fix_NALheader(vpu_dec) != GST_FLOW_OK)
            {
                if (vpu_dec->passthru)
                    goto passthru;
                return GST_FLOW_OK;  // ignore until we get an I frame
            }
        }
        if ((vpu_dec->codec == STD_MPEG2) && !vpu_dec->codec_data) 
        {
            // save this header in repeat mode since it is not always sent down again
            vpu_dec->codec_data_len = GST_BUFFER_SIZE(buffer);
            GST_FRAMEDBG(">>VPU_DEC: Initializing codec_data size=%d\n", vpu_dec->codec_data_len);
            vpu_dec->codec_data = gst_buffer_new_and_alloc(vpu_dec->codec_data_len);
            memcpy(GST_BUFFER_DATA(vpu_dec->codec_data),GST_BUFFER_DATA(buffer), 
                       vpu_dec->codec_data_len);
        }

        vpu_dec->firstFrameProcessed=TRUE;
       } 
       else if ((vpu_dec->codec==STD_VC1)&&(vpu_dec->picWidth!=0)) {
    	         mfw_gst_vpudec_prePareVC1Header(vpu_dec,buffer);
	     }
		   else  if (vpu_dec->codec==STD_AVC) {
        if (vpu_dec->nal_check)  // playing from file we might not have nal boundaries
        {
           mfw_gst_AVC_Fix_NALheader(vpu_dec);
        }
    }
   // vpu_dec->buff_consumed = 0;

passthru:
    // if passthru just pass it down to our pipeline - no decode 
    // Incase of the Filesink the output in the hardware buffer is copied onto the 
    // buffer allocated by filesink 
    if (vpu_dec->passthru)
    {
        return (mfw_gst_vpudec_passthru(vpu_dec));
    }

    return (mfw_gst_vpudec_copy_data(vpu_dec));
}

/*======================================================================================
FUNCTION:           mfw_gst_vpudec_release_buff

DESCRIPTION:        Release buffers that are already displayed

ARGUMENTS PASSED:   vpu_dec  - VPU decoder plugins context

RETURN VALUE:       GstFlowReturn - Success of Failure.
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=======================================================================================*/
GstFlowReturn mfw_gst_vpudec_release_buff(MfwGstVPU_Dec *vpu_dec)
{
    RetCode vpu_ret=RETCODE_SUCCESS;

    if (vpu_dec->codec == STD_MJPG || (vpu_dec->handle == NULL)) 
    {
        //GST_DEBUG (">>VPU_DEC: Release buf do nothing because of flush state or MJPEG \n");
        return GST_FLOW_OK;
    }

    // Loop through and release any buffers that are pending to be release or recently flipped

    // In this case we want to make sure that there is at least 1 buffer available before
    // we start decoding - preferably at least 2 but why start decoding when none are 
    // that should be rare.   This loop is not infinite as VPU will not decode if none
    // are free but the looping allows the downstream v4lsink to display and release buffers
    // so VPU can now use them for decoding.  It is a throttle to force this plugin to wait
    // for more display buffers to be released and used for decoding.
    gint i=0;
    int numFreeBufs = 0;
    int numBusyBufs = 0;
    for (i=0; i<vpu_dec->numframebufs; i++)
    {
        if (vpu_dec->fb_state_plugin[i] == FB_STATE_ALLOCATED) 
        {
            numFreeBufs++;
            //GST_FRAMEDBG (">>VPU_DEC: already free %d free buf %d busy bufs %d\n", i, numFreeBufs, numBusyBufs);
        } else if (vpu_dec->outbuffers[i] && 
                   gst_buffer_is_metadata_writable(vpu_dec->outbuffers[i]))
        {
            if ((vpu_dec->fb_state_plugin[i] == FB_STATE_PENDING) ||
                (vpu_dec->fb_state_plugin[i] == FB_STATE_FREE))
            {
                GST_FRAMEDBG (">>VPU_DEC: clearing PENDING %d \n", i);  // use for debugging VPU buffer flow
                vpu_ret = vpu_DecClrDispFlag (*(vpu_dec->handle), i);
                if (vpu_ret != RETCODE_SUCCESS) 
                {
                    GST_ERROR(">>VPU_DEC: vpu_DecClrDispFlag failed Error is %d pending buff idx=%d handle=0x%x\n",vpu_ret,i, vpu_dec->handle);
                    //return (GST_FLOW_ERROR);
                }
                vpu_dec->fb_state_plugin[i] = FB_STATE_ALLOCATED;
                numFreeBufs++;
                //GST_FRAMEDBG (">>VPU_DEC: pending release %d free buf %d busy bufs %d\n", i, numFreeBufs, numBusyBufs);
            } 
            if (vpu_dec->fb_state_plugin[i] == FB_STATE_DISPLAY)
            {
                // if VC-1 and last frame is display - do not release
                if (!((i == vpu_dec->outputInfo->indexFrameDisplay) && (vpu_dec->codec == STD_VC1)))
                {		     
                    GST_FRAMEDBG (">>VPU_DEC: clearing DISPLAY %d \n", i);  // use for debugging VPU buffer flow

                    vpu_ret = vpu_DecClrDispFlag (*(vpu_dec->handle), i);
                    if (vpu_ret != RETCODE_SUCCESS) 
                    {
                        GST_ERROR(">>VPU_DEC: vpu_DecClrDispFlag failed Error is %d disp buff idx=%d handle=0x%x\n",vpu_ret,i,vpu_dec->handle);
                        //return (GST_FLOW_ERROR);
                    }
                    vpu_dec->fb_state_plugin[i] = FB_STATE_ALLOCATED;
                    numFreeBufs++;
                    //GST_FRAMEDBG (">>VPU_DEC: release %d free buf %d busy bufs %d\n", i, numFreeBufs, numBusyBufs);
                } else {
                    numBusyBufs++;
                    //GST_FRAMEDBG (">>VPU_DEC: No release - previous displayed buffer %d free buf %d busy bufs %d\n", i, numFreeBufs, numBusyBufs);
                }
            } else if (vpu_dec->fb_state_plugin[i] == FB_STATE_DECODED) {
                numBusyBufs++;
                //GST_FRAMEDBG (">>VPU_DEC: No release - DECODED %d  free buf %d busy bufs %d\n", i, numFreeBufs, numBusyBufs);
            }
        } else {
            // This is a special case usually only needed after flushing/seeking where the downstream is not unrefing the buffers
            // after a seek so we will do it now - otherwise VPU will be stuck waiting for buffers to decode into
            if ((vpu_dec->fb_state_plugin[i] == FB_STATE_FREE) && vpu_dec->direct_render && 
                (vpu_dec->fb_type[i] == FB_TYPE_GST) ) 
            {
#if 0 //no need to unref, since vpu always hold the buffer reference.
			    gst_buffer_unref(vpu_dec->outbuffers[i]);
#endif
                vpu_ret = vpu_DecClrDispFlag (*(vpu_dec->handle), i);
                if (vpu_ret != RETCODE_SUCCESS) 
                {
                    GST_ERROR(">>VPU_DEC: vpu_DecClrDispFlag failed Error is %d disp buff idx=%d handle=0x%x\n",vpu_ret,i,vpu_dec->handle);
                }

                vpu_dec->fb_state_plugin[i] = FB_STATE_ALLOCATED;
                numFreeBufs++;
                GST_FRAMEDBG (">>VPU_DEC: Releasing and unreffing FLUSHED  %d \n", i);  // use for debugging VPU buffer flow

            } else {
                 numBusyBufs++;
                //GST_FRAMEDBG (">>VPU_DEC: No release - busy buffer %d  free buf %d busy bufs %d\n", i, numFreeBufs, numBusyBufs);
            }
        }
    }

    //GST_FRAMEDBG (">>VPU_DEC: can decode now - %d buffers are free %d are busy\n", numFreeBufs, numBusyBufs);
    return GST_FLOW_OK;
}

#define ROUNDUP8(data)\
    ((((data)+7)>>3)<<3)

/*======================================================================================
FUNCTION:           mfw_gst_vpudec_vpu_init

DESCRIPTION:        Initialize VPU and register allocated buffers

ARGUMENTS PASSED:   vpu_dec  - VPU decoder plugins context
 
RETURN VALUE:       GstFlowReturn - Success of Failure.
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=======================================================================================*/
GstFlowReturn mfw_gst_vpudec_vpu_init(MfwGstVPU_Dec *vpu_dec)
{

    RetCode vpu_ret=RETCODE_SUCCESS;
    gint fourcc;
    DecBufInfo bufinfo;
    guint needFrameBufCount=0;

#if (defined (VPU_MX37) || defined (VPU_MX51))
    if (vpu_dec->fmt==0)
        fourcc = GST_STR_FOURCC("NV12");
    else
#endif

        fourcc = GST_STR_FOURCC("I420");


    vpu_DecSetEscSeqInit(*(vpu_dec->handle), 1);

    memset (vpu_dec->initialInfo,0,sizeof (DecInitialInfo));
    vpu_ret = vpu_DecGetInitialInfo(*(vpu_dec->handle), vpu_dec->initialInfo);

    // Release the VPU for initialization regardless of error
    vpu_DecSetEscSeqInit(*(vpu_dec->handle), 0);

    // In some cases of streaming mode - might not be enough data so return OK
    // this is not an error - just signifies more input buffer is needed
    // should not happen in file play mode where a parser is giving a full frame 
    // each time
    if (vpu_ret == RETCODE_FRAME_NOT_COMPLETE) {
        GST_DEBUG(">>VPU_DEC: Exiting vpu_init for more data - init is not complete \n");
        return GST_FLOW_OK;
    }

    if (vpu_ret != RETCODE_SUCCESS) {
       GST_ERROR(">>VPU_DEC: vpu_DecGetInitialInfo failed. Error code is %d handle %d \n",vpu_ret, vpu_dec->handle);
       return(GST_FLOW_ERROR);
    }

    GST_DEBUG(">>VPU_DEC: min buffer count= %d\n", vpu_dec->initialInfo->minFrameBufferCount);
    GST_DEBUG(">>VPU_DEC: InitialInfo picWidth: %u, picHeight: %u, frameRate: %u\n",
        vpu_dec->initialInfo->picWidth, vpu_dec->initialInfo->picHeight,
        (unsigned int)vpu_dec->initialInfo->frameRateInfo);

    /* Check: Minimum resolution limitation */
    if (G_UNLIKELY(vpu_dec->initialInfo->picWidth  < MIN_WIDTH || 
   	               vpu_dec->initialInfo->picHeight < MIN_HEIGHT)) 
    {
        GstMessage *message = NULL;
        GError *gerror = NULL;
        gchar *text_msg = "unsupported video resolution.";
        gerror = g_error_new_literal(1, 0, text_msg);
        message = gst_message_new_error(GST_OBJECT(GST_ELEMENT(vpu_dec)), 
			   	  gerror,"debug none");
        gst_element_post_message(GST_ELEMENT(vpu_dec), message);
        g_error_free(gerror);
        return(GST_FLOW_ERROR);
    }

#if !defined(MPEG2) && !defined(VPU_MX27)
    if ((vpu_dec->codec==STD_MPEG2) && 
        (!((vpu_dec->initialInfo->profile==0) && (vpu_dec->initialInfo->level==0)))){
        #if 0
        GstMessage *message = NULL;
        GError *gerror = NULL;
        gchar *text_msg = "unsupported video codec.";
        gerror = g_error_new_literal(1, 0, text_msg);
        message = gst_message_new_error(GST_OBJECT(GST_ELEMENT(vpu_dec)), 
			   	  gerror,"debug none");
        gst_element_post_message(GST_ELEMENT(vpu_dec), message);
        g_error_free(gerror);
        #endif
        GST_ERROR(">>VPU_DEC: Exiting because no mpeg2 support and empty profile\n");
        return(GST_FLOW_ERROR);
    }
#endif


    // Is VPU asking for more buffers than our plugin can allocated? 
    if (G_UNLIKELY(vpu_dec->initialInfo->minFrameBufferCount > NUM_MAX_VPU_REQUIRED)) 
    {
        GST_ERROR(">>VPU_DEC: required frames number exceed max limitation, required %d.",
   	   	   	   	 vpu_dec->initialInfo->minFrameBufferCount);
        return(GST_FLOW_ERROR);
    }
        
#ifdef VPU_MX27
    needFrameBufCount = vpu_dec->initialInfo->minFrameBufferCount + 2;
#else
    needFrameBufCount = vpu_dec->initialInfo->minFrameBufferCount + 3;
#endif

    /* Padding the width and height to 16 */
    if (!vpu_dec->eos)
    {
        GstCaps *caps=NULL;
        gint crop_top_len, crop_left_len;
        gint crop_right_len, crop_bottom_len;
        gint orgPicW = vpu_dec->initialInfo->picWidth;
        gint orgPicH = vpu_dec->initialInfo->picHeight;
        guint numBufs = (vpu_dec->rotation_angle || vpu_dec->mirror_dir || (vpu_dec->codec == STD_MJPG)) ? 2 : needFrameBufCount;

        vpu_dec->initialInfo->picWidth = (vpu_dec->initialInfo->picWidth+15)/16*16;

#ifndef VPU_MX27
        if ((vpu_dec->initialInfo->interlace)
            &&((vpu_dec->codec==STD_MPEG2)||(vpu_dec->codec==STD_VC1)||(vpu_dec->codec==STD_AVC))) {
            vpu_dec->initialInfo->picHeight = (vpu_dec->initialInfo->picHeight+31)/32*32;
        }else
#endif
        {
            vpu_dec->initialInfo->picHeight = (vpu_dec->initialInfo->picHeight+15)/16*16;
        }

        if (vpu_dec->codec == STD_AVC &&
            (vpu_dec->initialInfo->picCropRect.right > 0 &&
             vpu_dec->initialInfo->picCropRect.bottom > 0)) {

            crop_top_len    = vpu_dec->initialInfo->picCropRect.top;                                
            crop_left_len   = vpu_dec->initialInfo->picCropRect.left;                              
            crop_right_len  = vpu_dec->initialInfo->picWidth - 
   	   	   	   	  vpu_dec->initialInfo->picCropRect.right;   
            crop_bottom_len = vpu_dec->initialInfo->picHeight - 
   	   	   	   	  vpu_dec->initialInfo->picCropRect.bottom;
        } else {                                                                                   
            crop_top_len = 0;                                                                    
            crop_left_len = 0;
            crop_right_len = vpu_dec->initialInfo->picWidth- orgPicW;                           
            crop_bottom_len = vpu_dec->initialInfo->picHeight-orgPicH;  
            
        }                                                               

        /* set the capabilites on the source pad */
        caps = gst_caps_new_simple("video/x-raw-yuv",
            "format", GST_TYPE_FOURCC, fourcc,
            "width", G_TYPE_INT,
                ((vpu_dec->rotation_angle == 90) || (vpu_dec->rotation_angle==270)) ? 
   	   	   	vpu_dec->initialInfo->picHeight : vpu_dec->initialInfo->picWidth, 
            "height", G_TYPE_INT,
               ((vpu_dec->rotation_angle == 90) || (vpu_dec->rotation_angle==270)) ? 
   	   	   	vpu_dec->initialInfo->picWidth : vpu_dec->initialInfo->picHeight, 
            "framerate", GST_TYPE_FRACTION, vpu_dec->frame_rate_nu, vpu_dec->frame_rate_de,
            "pixel-aspect-ratio",GST_TYPE_FRACTION,1,1,
            "crop-top-by-pixel", G_TYPE_INT,  
            (((vpu_dec->rotation_angle == 90) || (vpu_dec->rotation_angle==270)) ? 
   	   	   	ROUNDUP8(crop_left_len) : ROUNDUP8(crop_top_len)),
            
            "crop-left-by-pixel", G_TYPE_INT, 
            (((vpu_dec->rotation_angle == 90) || (vpu_dec->rotation_angle==270)) ? 
   	   	   	ROUNDUP8(crop_top_len) : ROUNDUP8(crop_left_len)),
   	   	   	
            "crop-right-by-pixel", G_TYPE_INT,
            (((vpu_dec->rotation_angle == 90) || (vpu_dec->rotation_angle==270)) ? 
   	   	   	ROUNDUP8(crop_bottom_len) : ROUNDUP8(crop_right_len)),
   	   	   	
            "crop-bottom-by-pixel", G_TYPE_INT,
            (((vpu_dec->rotation_angle == 90) || (vpu_dec->rotation_angle==270)) ? 
   	   	   	ROUNDUP8(crop_right_len) : ROUNDUP8(crop_bottom_len)),
   	   	   	
            "num-buffers-required",G_TYPE_INT, needFrameBufCount,   
            "field", G_TYPE_INT, 0,
            NULL);

        if (vpu_dec->min_latency)
        {   // set the max lateness to a higher number
            gst_caps_set_simple (caps,"sfd", G_TYPE_INT, 1,NULL);      
        }

        if (!(gst_pad_set_caps(vpu_dec->srcpad, caps))) {
            GST_ERROR ("\n>>VPU_DEC: Could not set the caps for src pad\n");
        }
        gst_caps_unref(caps);
        caps = NULL;
    }

    vpu_dec->yuv_frame_size = (vpu_dec->initialInfo->picWidth * 
                               vpu_dec->initialInfo->picHeight * 3)/2;

    vpu_dec->numframebufs = needFrameBufCount;
        
    /* Allocate the Frame buffers requested by the Decoder if not done already */
    if(vpu_dec->framebufinit_done==FALSE)
    {
        if((mfw_gst_vpudec_FrameBufferInit(vpu_dec,vpu_dec->frameBuf,
                                           needFrameBufCount))< 0)
        {
            GST_ERROR(">>VPU_DEC: Frame Buffer Init Memory system allocation failed!\n");
            mfw_gst_vpudec_post_fatal_error_msg(vpu_dec,
                                                "Allocation of the Frame Buffers Failed");
                 
            return(GST_FLOW_ERROR);
        }
        vpu_dec->framebufinit_done=TRUE;
    }

    // This is needed for H.264 only
    memset(&bufinfo, 0, sizeof (bufinfo));
    bufinfo.avcSliceBufInfo.sliceSaveBuffer = vpu_dec->slice_mem_desc.phy_addr;
    bufinfo.avcSliceBufInfo.sliceSaveBufferSize = SLICE_SAVE_SIZE;

    // Register the Allocated Frame buffers with the decoder
    // for rotation - only register minimum as extra two buffers 
    //                will be used for display separately outside of VPU

    vpu_ret = vpu_DecRegisterFrameBuffer(*(vpu_dec->handle),
        vpu_dec->frameBuf,
        (vpu_dec->rotation_angle || vpu_dec->mirror_dir || (vpu_dec->codec == STD_MJPG)) ? 
		   	vpu_dec->initialInfo->minFrameBufferCount : vpu_dec->numframebufs,
        vpu_dec->initialInfo->picWidth,
        &bufinfo);
    if (vpu_ret != RETCODE_SUCCESS) {
        GST_ERROR(">>VPU_DEC: vpu_DecRegisterFrameBuffer failed. Error code is %d \n",vpu_ret);
        mfw_gst_vpudec_post_fatal_error_msg(vpu_dec,
   	   	   	"Registration of the Allocated Frame Buffers Failed ");
        return(GST_FLOW_ERROR);
    }
	
    // Setup rotation or mirroring which will be output to separate buffers for display
    if (vpu_dec->rotation_angle || vpu_dec->mirror_dir || (vpu_dec->codec == STD_MJPG))
    {
        int rotStride = vpu_dec->initialInfo->picWidth;
        vpu_dec->allow_parallelization = FALSE;             

        if (vpu_dec->rotation_angle)
        {
            // must set angle before rotator stride since the stride uses angle 
            // or error checking
            vpu_ret = vpu_DecGiveCommand(*(vpu_dec->handle),
                            SET_ROTATION_ANGLE, &vpu_dec->rotation_angle);

            // Do a 90 degree rotation - buffer is allocated in FrameBufferInit at end
            // for rotation, set stride, rotation angle and initial buffer output
            if ((vpu_dec->rotation_angle == 90) || (vpu_dec->rotation_angle == 270))
                rotStride = vpu_dec->initialInfo->picHeight;
        } 
        if (vpu_dec->mirror_dir)
        {
            vpu_ret = vpu_DecGiveCommand(*(vpu_dec->handle),
                                          SET_MIRROR_DIRECTION, &vpu_dec->mirror_dir);
        }
        vpu_ret = vpu_DecGiveCommand(*(vpu_dec->handle),SET_ROTATOR_STRIDE, &rotStride);
	    if (vpu_ret != RETCODE_SUCCESS) {
            GST_ERROR(">>VPU_DEC: SET_ROTATOR_STRIDE failed. ret=%d \n",vpu_ret);
            mfw_gst_vpudec_post_fatal_error_msg(vpu_dec,
                                         "VPU SET_ROTATOR_STRIDE failed ");
   	   	   	return(GST_FLOW_ERROR);
        }
        vpu_dec->rot_buff_idx = vpu_dec->initialInfo->minFrameBufferCount;
        vpu_ret = vpu_DecGiveCommand(*(vpu_dec->handle),SET_ROTATOR_OUTPUT, 
                                     &vpu_dec->frameBuf[vpu_dec->rot_buff_idx]);
        if (vpu_dec->rotation_angle)
            vpu_ret = vpu_DecGiveCommand(*(vpu_dec->handle),ENABLE_ROTATION, 0);
        if (vpu_dec->mirror_dir)
            vpu_ret = vpu_DecGiveCommand(*(vpu_dec->handle),ENABLE_MIRRORING, 0);
    }  /* end rotation and mirroring setup */

    vpu_dec->decParam->prescanEnable = (vpu_dec->file_play_mode) ? 0 : 1;

    if (vpu_dec->initialInfo->picWidth >= 720) 
        vpu_dec->min_data_in_vpu = MIN_DATA_IN_VPU_720P;
    else if (vpu_dec->initialInfo->picWidth >= 640) 
        vpu_dec->min_data_in_vpu = MIN_DATA_IN_VPU_VGA;
    else if (vpu_dec->initialInfo->picWidth >= 320)
        vpu_dec->min_data_in_vpu = MIN_DATA_IN_VPU_QVGA;
    else
        vpu_dec->min_data_in_vpu = MIN_DATA_IN_VPU_QCIF;

#ifndef VPU_MX27  // not supported on MX27
    if (vpu_dec->dbk_enabled) {
        DbkOffset dbkoffset;
        dbkoffset.DbkOffsetEnable = 1;
        dbkoffset.DbkOffsetA = vpu_dec->dbk_offset_a;
        dbkoffset.DbkOffsetB = vpu_dec->dbk_offset_b;

        vpu_DecGiveCommand(*(vpu_dec->handle), SET_DBK_OFFSET, &dbkoffset);
    } else {
        DbkOffset dbkoffset;
        dbkoffset.DbkOffsetEnable = 0;
        dbkoffset.DbkOffsetA = 0;
        dbkoffset.DbkOffsetB = 0;

        vpu_DecGiveCommand(*(vpu_dec->handle), SET_DBK_OFFSET, &dbkoffset);
     }    
#endif

    GST_DEBUG(">>VPU_DEC: VPU Initialization complete \n");
    vpu_dec->vpu_init=TRUE;
    vpu_dec->flushing = FALSE;
    return GST_FLOW_OK;
}


/*======================================================================================
FUNCTION:           mfw_gst_vpudec_render

DESCRIPTION:        Render one frame if conditions allow

ARGUMENTS PASSED:   vpu_dec  - VPU decoder plugins context
 
RETURN VALUE:       GstFlowReturn - Success of Failure.
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=======================================================================================*/
GstFlowReturn mfw_gst_vpudec_render(MfwGstVPU_Dec *vpu_dec)
{
    RetCode vpu_ret=RETCODE_SUCCESS;
    GstFlowReturn retval=GST_FLOW_OK;

#if 0 // use this if you want to NOT display I frames good for testing if I frames have problems
    if (VPU_PIC_TYPE == 0) 
    {
        mfw_gst_vpudec_no_display(vpu_dec);
        return GST_FLOW_OK;
    }
#endif

    if (vpu_dec->outputInfo->indexFrameDisplay < 0)
        return GST_FLOW_OK;

    if (vpu_dec->flushing)
    {
        vpu_dec->fb_state_plugin[vpu_dec->outputInfo->indexFrameDisplay] = FB_STATE_PENDING;
        return GST_FLOW_OK;
    }

    if (G_LIKELY(vpu_dec->direct_render==TRUE))
    {
        if (vpu_dec->rotation_angle || vpu_dec->mirror_dir || (vpu_dec->codec == STD_MJPG))
        {
            vpu_dec->pushbuff = vpu_dec->outbuffers[vpu_dec->rot_buff_idx];
            // switch output buffer for every other frame so we don't overwrite display  
            // data in v4lsink this way VPU can still decode while v4l sink is displaying
            if (vpu_dec->rot_buff_idx == vpu_dec->initialInfo->minFrameBufferCount) 
                vpu_dec->rot_buff_idx = vpu_dec->initialInfo->minFrameBufferCount+1;
            else
                vpu_dec->rot_buff_idx = vpu_dec->initialInfo->minFrameBufferCount;
            vpu_DecGiveCommand(*(vpu_dec->handle), SET_ROTATOR_OUTPUT, 
                               &vpu_dec->frameBuf[vpu_dec->rot_buff_idx]);
            vpu_dec->fb_state_plugin[vpu_dec->outputInfo->indexFrameDisplay] = FB_STATE_PENDING;
        } else {
            // The HW case in DR mode 
            if (vpu_dec->fb_type[vpu_dec->outputInfo->indexFrameDisplay] == FB_TYPE_HW)
            {
                retval = gst_pad_alloc_buffer_and_set_caps(vpu_dec->srcpad, 0,
                                                           vpu_dec->yuv_frame_size,
                                                           GST_PAD_CAPS(vpu_dec->srcpad), 
                                                           &vpu_dec->pushbuff);
                if (retval != GST_FLOW_OK) {
                    GST_ERROR(">>VPU_DEC: Error %d in allocating the Framebuffer[%d]\n",
                               retval,vpu_dec->outputInfo->indexFrameDisplay);
                    return retval;
                }
                if (GST_BUFFER_FLAG_IS_SET(vpu_dec->pushbuff,GST_BUFFER_FLAG_LAST)==TRUE) {
                     GST_DEBUG(">>VPU_DEC: wrong here, should not be a HW buffer?\n");
                }
                else {
                    GST_DEBUG(">>VPU_DEC: Get a software buffer from V4L sink.\n");
                    memcpy(GST_BUFFER_DATA(vpu_dec->pushbuff),
                           vpu_dec->frame_virt[vpu_dec->outputInfo->indexFrameDisplay],
                           vpu_dec->yuv_frame_size);
                }
         	    vpu_dec->fb_state_plugin[vpu_dec->outputInfo->indexFrameDisplay] = FB_STATE_PENDING;
            } else {
                vpu_dec->pushbuff = vpu_dec->outbuffers[vpu_dec->outputInfo->indexFrameDisplay];
                gst_buffer_set_caps(vpu_dec->pushbuff, GST_PAD_CAPS(vpu_dec->srcpad));

                //GST_DEBUG(">>VPU_DEC: SW mode: will clear %d later.\n",vpu_dec->outputInfo->indexFrameDisplay);
                vpu_dec->fb_state_plugin[vpu_dec->outputInfo->indexFrameDisplay] = FB_STATE_DISPLAY;
            }
        }
    } else {
        // Incase of the Filesink the output in the hardware buffer is copied onto the 
        // buffer allocated by filesink 
        retval = gst_pad_alloc_buffer_and_set_caps(vpu_dec->srcpad, 0,
                                                   vpu_dec->yuv_frame_size,
                                                   GST_PAD_CAPS(vpu_dec->srcpad), 
                                                   &vpu_dec->pushbuff);
        if (retval != GST_FLOW_OK) {
            GST_ERROR(">>VPU_DEC: Error %d allocating the Framebuffer[%d]\n", 
                         retval,vpu_dec->outputInfo->indexFrameDisplay);
            return retval;
        }

        memcpy(GST_BUFFER_DATA(vpu_dec->pushbuff),
                               vpu_dec->frame_virt[vpu_dec->outputInfo->indexFrameDisplay],
                              vpu_dec->yuv_frame_size);
        vpu_ret = vpu_DecClrDispFlag (*(vpu_dec->handle), vpu_dec->outputInfo->indexFrameDisplay);
        vpu_dec->fb_state_plugin[vpu_dec->outputInfo->indexFrameDisplay] = FB_STATE_ALLOCATED;
    }

    // Update the time stamp base on the frame-rate 
    GST_BUFFER_SIZE(vpu_dec->pushbuff) = vpu_dec->yuv_frame_size;
    GST_BUFFER_DURATION(vpu_dec->pushbuff) = vpu_dec->time_per_frame;

    {
        GstClockTime ts;
        if ((vpu_dec->codec!=STD_MPEG2) && (vpu_dec->codec!=STD_RV)){
            mfw_gst_get_timestamp(vpu_dec, &ts);
        }else{
            ts = vpu_dec->last_ts_sent+vpu_dec->time_per_frame;
            mfw_gst_get_mpeg2ts(vpu_dec, &ts);
            vpu_dec->last_ts_sent = ts;
        }
        GST_BUFFER_TIMESTAMP(vpu_dec->pushbuff)= ts;
    }

    // only reference the gstreamer buffers that we allocated through GST
    if (vpu_dec->direct_render && 
        (vpu_dec->fb_type[vpu_dec->outputInfo->indexFrameDisplay] == FB_TYPE_GST) ) 
    {
        
        gst_buffer_ref(vpu_dec->pushbuff);
    }

    // Push the buffer downstream 
#ifndef VPU_MX27  
    mfw_vpudec_set_field(vpu_dec, vpu_dec->pushbuff);
#endif

    // always unlock before since pause might deadlock in this case
    vpu_mutex_unlock(vpu_dec->vpu_mutex);
    GST_MUTEX (">>VPU_DEC: unlock mutex before render cnt=%d\n", mutex_cnt);
    retval = gst_pad_push(vpu_dec->srcpad,vpu_dec->pushbuff);        
    if (!vpudec_global_ptr || vpu_dec->in_cleanup)
         return GST_FLOW_ERROR;

    if (vpu_mutex_lock(vpu_dec->vpu_mutex, TRUE) == FALSE)
    {
        vpu_dec->trymutex = FALSE;
        GST_MUTEX (">>VPU_DEC: after render - no mutex lock cnt=%d\n", mutex_cnt);
        return GST_FLOW_ERROR;  // no mutex so exit - must be flushing or cleaning up
    } else {
        GST_MUTEX (">>VPU_DEC: after render - mutex lock cnt=%d\n", mutex_cnt);
    }

    if (retval != GST_FLOW_OK)
    {
        GST_ERROR(">>VPU_DEC: Error %d Pushing Output onto the Source Pad, idx=%d \n",
		          retval, vpu_dec->outputInfo->indexFrameDisplay);
        //vpu_dec->fb_state_plugin[vpu_dec->outputInfo->indexFrameDisplay] = FB_STATE_FREE;        
        if (vpu_dec->direct_render && 
            (vpu_dec->fb_type[vpu_dec->outputInfo->indexFrameDisplay] == FB_TYPE_GST) ) 
        {
            //gst_buffer_unref(vpu_dec->pushbuff);
        } 
    } else 
    {
        //GST_DEBUG(">>VPU_DEC: Render buff %d\n", vpu_dec->outputInfo->indexFrameDisplay); 
        vpu_dec->frames_rendered++;
    }
    return retval;
}



/*======================================================================================
FUNCTION:           mfw_gst_vpudec_continue_looping

DESCRIPTION:        This routine checks various parameters to see if we should
                    keep looping

ARGUMENTS PASSED:   vpu_dec  - VPU decoder plugins context
                    frame_started - if chain was entered with a decode pending
                          then we might want to loop to start a new frame before exiting
                          since all we got was output of last frame and no new decode started

RETURN VALUE:       gboolean - continue looping is TRUE otherwise return FALSE .
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=======================================================================================*/
gboolean  mfw_gst_vpudec_continue_looping(MfwGstVPU_Dec *vpu_dec,  guint loop_cnt)
{
    gboolean fContinue = FALSE;
    RetCode vpu_ret=RETCODE_SUCCESS;

    if (vpudec_global_ptr == NULL)  
         return FALSE;

    vpu_dec->decParam->chunkSize = 0;

    if ((vpu_dec->file_play_mode == FALSE) && (loop_cnt < 3))
        fContinue = TRUE;

    // keep looping on EOS until VPU fails with EOS
#ifndef VPU_MX27  // mp4packedframes not supported on 27                   
    if (vpu_dec->outputInfo->mp4PackedPBframe==1) {
        fContinue = TRUE;
    } else 
#endif
    if (vpu_dec->flushing)
    {
        return FALSE;
    } else if (G_UNLIKELY(vpu_dec->eos))
    {
        if ((vpu_dec->outputInfo->indexFrameDisplay == -1) &&
            (vpu_dec->outputInfo->indexFrameDecoded == -2)) 
            return FALSE;

        // in some cases at eos VPU can get in an infinite loop
        if ((vpu_dec->outputInfo->indexFrameDisplay == -1) ||
            (vpu_dec->outputInfo->indexFrameDecoded == -1) &&
            (loop_cnt > vpu_dec->numframebufs))
            return FALSE;
        else
            fContinue = TRUE;

    }
	else if(vpu_dec->outputInfo->indexFrameDisplay == -3 && (vpu_dec->outputInfo->indexFrameDecoded >= 0) && (vpu_dec->codec == STD_VC1))
		    fContinue = FALSE;
	else {
       // for file play mode after seeking or interlaced sometimes need a 2nd decode to get 
        // positive display index to turn off iframesearchenable
        // happens most often with VC1 which will return -3 for disp and positive decode index after seek sometimes
        // add vc-1 advance profile support, if it is vc-1, indexFrameDisplay = -3  indexFrameDecoded = 0, return false
        if ((loop_cnt < 3) && // avoid infinite loop - usually only one loop is needed
             (vpu_dec->outputInfo->indexFrameDisplay < 0) &&
             ((vpu_dec->outputInfo->indexFrameDecoded >= 0) || (vpu_dec->outputInfo->indexFrameDecoded == -2)) )
            fContinue = TRUE;  
    }

    if (vpu_dec->loopback)
        fContinue = FALSE;

    // this is to avoid infinite loops where it might keep looping when it should not or flush is waiting
    if ( (vpu_dec->outputInfo->indexFrameDisplay < 0) && !vpu_dec->eos && (loop_cnt > vpu_dec->initialInfo->minFrameBufferCount))
        fContinue = FALSE;

    GST_FRAMEDBG(">>VPU_DEC: Continue Loop returns %d\n", fContinue);
    return fContinue;
}


/*======================================================================================
FUNCTION:           mfw_gst_vpudec_no_display

DESCRIPTION:        Releases a buffer back to VPU since it will not be displayed

ARGUMENTS PASSED:   vpu_dec  - VPU decoder plugins context
                    idx - index of buffer to be released

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=======================================================================================*/
void mfw_gst_vpudec_no_display(MfwGstVPU_Dec *vpu_dec)
{    
    if ( vpu_dec->ts_tx != vpu_dec->ts_rx)
        vpu_dec->ts_tx = (vpu_dec->ts_tx+1)%MAX_STREAM_BUF;

    if (vpu_dec->frames_rendered)
        vpu_dec->frames_dropped++;

    if (vpu_dec->just_flushed)
        vpu_dec->last_ts_sent += vpu_dec->time_per_frame;

    if ((vpu_dec->outputInfo->indexFrameDisplay >= 0) && 
        (vpu_dec->outputInfo->indexFrameDisplay < vpu_dec->numframebufs)) {
        vpu_dec->fb_state_plugin[vpu_dec->outputInfo->indexFrameDisplay] = FB_STATE_PENDING;
    }
}

/*======================================================================================
FUNCTION:           mfw_vpudec_set_field

DESCRIPTION:        Set the field info for interlaced content

ARGUMENTS PASSED:   vpu_dec  - VPU decoder plugins context

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=======================================================================================*/
void mfw_vpudec_get_field(MfwGstVPU_Dec *vpu_dec)
{
#ifndef VPU_MX27

    vpu_dec->field = FIELD_NONE;

    /* Set the field information */
    if (vpu_dec->codec == STD_VC1) {
    		GST_INFO("FRAME_INTERLACE:%d\n",vpu_dec->outputInfo->pictureStructure);
        
    	if (vpu_dec->outputInfo->pictureStructure == 2)
    		GST_INFO("FRAME_INTERLACE\n");
    	else if (vpu_dec->outputInfo->pictureStructure == 3) {
    		if (vpu_dec->outputInfo->topFieldFirst)
    			vpu_dec->field = FIELD_INTERLACED_TB;
    		else
    			vpu_dec->field = FIELD_INTERLACED_BT;
            
    	}
    	if (vpu_dec->outputInfo->vc1_repeatFrame)
    		GST_INFO("dec_idx %d : VC1 RPTFRM [%1d]\n", vpu_dec->frames_decoded, vpu_dec->outputInfo->vc1_repeatFrame);
    } else if (vpu_dec->codec == STD_AVC) {
    	if (vpu_dec->outputInfo->interlacedFrame) {
    		if (vpu_dec->outputInfo->topFieldFirst)
    			vpu_dec->field = FIELD_INTERLACED_TB;
    		else
    			vpu_dec->field = FIELD_INTERLACED_BT;
    		GST_INFO("Top Field First flag: %d, dec_idx %d\n",
    			  vpu_dec->outputInfo->topFieldFirst, vpu_dec->frames_decoded);
    	}
    } else if ((vpu_dec->codec != STD_MPEG4) && (vpu_dec->codec != STD_RV)){
    	if (vpu_dec->outputInfo->interlacedFrame || !vpu_dec->outputInfo->progressiveFrame) {
    		if (vpu_dec->outputInfo->pictureStructure == 1)
    			vpu_dec->field = FIELD_TOP;
    		else if (vpu_dec->outputInfo->pictureStructure == 2)
    			vpu_dec->field = FIELD_BOTTOM;
    		else if (vpu_dec->outputInfo->pictureStructure == 3) {
    			if (vpu_dec->outputInfo->topFieldFirst)
    				vpu_dec->field = FIELD_INTERLACED_TB;
    			else
    				vpu_dec->field = FIELD_INTERLACED_BT;
    		}
    	}
    	if (vpu_dec->outputInfo->repeatFirstField)
    		GST_INFO("frame_idx %d : Repeat First Field\n", vpu_dec->frames_decoded);
        
    }
    
#endif
}

/*======================================================================================
FUNCTION:           mfw_vpudec_set_field

DESCRIPTION:        Set the field info for interlaced content

ARGUMENTS PASSED:   vpu_dec  - VPU decoder plugins context

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=======================================================================================*/
void mfw_vpudec_set_field(MfwGstVPU_Dec *vpu_dec, GstBuffer *buf)
{
#ifndef VPU_MX27
    GstCaps *caps = GST_PAD_CAPS(vpu_dec->srcpad);

 
    caps = gst_caps_copy(caps);
    gst_caps_set_simple (caps,"field", G_TYPE_INT, vpu_dec->field,NULL);      
    //g_print(RED_STR("vpu caps:%s\n",gst_caps_to_string(caps)));
    gst_buffer_set_caps (buf, caps);
    gst_caps_unref(caps);
    return;
    
#endif
}

/*======================================================================================
FUNCTION:           mfw_gst_vpudec_chain

DESCRIPTION:        The main processing function where the data comes in as buffer. This 
                    data is decoded, and then pushed onto the next element for further 
                    processing.

ARGUMENTS PASSED:   pad - pointer to the sinkpad of this element
                    buffer - pointer to the input buffer which has the H.264 data.

RETURN VALUE:       GstFlowReturn - Success of Failure.
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=======================================================================================*/
static GstFlowReturn mfw_gst_vpudec_chain(GstPad *pad, GstBuffer *buffer)
{
    RetCode vpu_ret=RETCODE_SUCCESS;
    GstFlowReturn retval=GST_FLOW_OK;
    MfwGstVPU_Dec *vpu_dec  = MFW_GST_VPU_DEC(GST_PAD_PARENT(pad));
    gboolean frame_started = vpu_dec->is_frame_started;
    gboolean fContinue = FALSE;
    gfloat time_val = 0;
    struct timeval tv_prof, tv_prof1;
    struct timeval tv_prof2, tv_prof3;
    long time_before = 0, time_after = 0;
    guint gst_buffer_size = 0;
    guint loop_cnt = 0;

    // handle exit conditions
    if ((vpu_dec == NULL) || (vpudec_global_ptr == NULL))
        return GST_FLOW_OK;

    if (vpu_dec->flushing || vpu_dec->in_cleanup)
        return GST_FLOW_OK;    

    vpu_dec->trymutex = TRUE;

    // try lock works so that if it fails some other thread has it and in that case we should not
    // proceed with any decodes - either flushing or cleaning up
    if (vpu_mutex_lock(vpu_dec->vpu_mutex, TRUE) == FALSE)
    {
        GST_MUTEX (">>VPU_DEC: In chain but exiting since mutex is locked by other thread or nullmutex 0x%x\n",vpu_dec->vpu_mutex);
        return GST_FLOW_OK;
    } else {
        GST_MUTEX (">>VPU_DEC: In chain mutex acquired cnt=%d\n",mutex_cnt);
    }
    

    if (buffer && GST_BUFFER_FLAG_IS_SET(buffer,GST_BUFFER_FLAG_DELTA_UNIT))
    {
        if (!GST_BUFFER_SIZE(buffer) || 
            vpu_dec->check_for_iframe || 
            vpu_dec->drop_before_decode) 
        {
           GST_FRAMEDROP(">>VPU_DEC: Ignore delta before before I frame \n");
           vpu_dec->frames_decoded++;
           vpu_dec->frames_dropped++;
           goto done;
        }
    }

    if (vpu_dec->flushing)
        goto done;
    
    vpu_dec->num_timeouts = 0;
    
    if (buffer) {
        vpu_dec->must_copy_data = TRUE;
        gst_buffer_size = GST_BUFFER_SIZE(buffer);
        GST_FRAMEDBG(">>VPU_DEC: vpudec_chain input buffersize %d \n", gst_buffer_size);
    } else {
        GST_DEBUG(">>VPU_DEC: Null buffer enter as EOS \n");
        vpu_dec->must_copy_data = TRUE;
    }

    vpu_dec->gst_buffer = buffer;
    vpu_dec->buff_consumed = 0;

    // Update Profiling timestamps
    if (G_UNLIKELY(vpu_dec->profiling)) {
        gettimeofday(&tv_prof2, 0);
    }

    // Open VPU if not already opened
    if (G_UNLIKELY(!vpu_dec->vpu_opened)) {
        retval = mfw_gst_vpudec_vpu_open(vpu_dec);
        if (retval != GST_FLOW_OK)
        {
            GST_ERROR(">>VPU_DEC: mfw_gst_vpudec_vpu_open failed. Error code is %d \n",retval);
            goto done;
        }
    }

    // Initialize VPU and copy data 
    if(G_UNLIKELY(vpu_dec->vpu_init==FALSE)){

    // Write input bitstream to VPU - special for streaming mode
    retval = mfw_gst_vpudec_copy_sink_input(vpu_dec, buffer);
    if (retval != GST_FLOW_OK)
    {
        GST_ERROR(">>VPU_DEC: mfw_gst_vpudec_copy_sink_input failed - Error %d \n",retval);
        goto done;
    } 

    // if we have not processed our first frame than return
        // In networking mode we might check for headers before initializing VPU
        if (vpu_dec->firstFrameProcessed == FALSE) 
        {
            GST_DEBUG(">>VPU_DEC: Error with first frame - return for more data\n");
            retval = GST_FLOW_OK;
            goto done;
        }

        // this is used to dump what we send to VPU to a file to analyze if problems in bitstream.
        // it does not pass anything to VPU but exits after copy was done above in copy_sink_input
        if (vpu_dec->passthru)
        { 
            retval =  GST_FLOW_OK;
            goto done;
        }
        if (!vpu_dec->file_play_mode && 
            (vpu_dec->data_in_vpu < 2048)  && 
            (vpu_dec->num_timeouts == 0))
        {
            GST_DEBUG(">>VPU_DEC: init needs more data data_in_vpu=%d\n",vpu_dec->data_in_vpu);
            retval =  GST_FLOW_OK;
            goto done;
        }
        retval = mfw_gst_vpudec_vpu_init(vpu_dec);
        if (retval != GST_FLOW_OK)
        {
            // in the case of MPEG2 keep trying for at least 25 times
            if ((vpu_dec->codec == STD_MPEG2) && (vpu_dec->num_timeouts < 100))
            {
                GST_DEBUG(">>VPU_DEC: mfw_gst_vpudec_vpu_init failed but with MPEG2 keep asking for more data timeouts=%d\n",vpu_dec->num_timeouts);
                vpu_dec->num_timeouts++;
                retval = GST_FLOW_OK;
                goto done;
            }
            GST_ERROR(">>VPU_DEC: mfw_gst_vpudec_vpu_init failed \n");
            mfw_gst_vpudec_post_fatal_error_msg(vpu_dec,"VPU Decoder Initialization failed ");
            goto done;
        } 
        vpu_dec->num_timeouts=0;
        fContinue = TRUE;
    }

    // wait for headers
    if (vpu_dec->accumulate_hdr && vpu_dec->hdr_received)
    {
        //GST_DEBUG(">>VPU_DEC: Hdr received buff size %d\n", gst_buffer_size);
        if (vpu_dec->check_for_iframe || vpu_dec->just_flushed)
        {
            vpu_dec->check_for_iframe == FALSE;
            vpu_dec->just_flushed = FALSE;
            vpu_dec->decParam->iframeSearchEnable = 0; // turn off iframesearch - i frame found
            vpu_dec->decParam->skipframeMode = 0;  // turn off skipframe used in networking
            vpu_dec->decParam->skipframeNum = 0;  // turn off skipframe used in networking                       
        }
    }


    // Keep looping while there is enough in bitstream to decode
    do
    {
        loop_cnt++;

        if (vpudec_global_ptr == NULL)
            return GST_FLOW_OK;

        if (vpu_dec->flushing || vpu_dec->in_cleanup)
            goto done;

        // Start Decoding One frame in VPU if not already started
        if (!vpu_dec->is_frame_started)
        {
            // copy data if still there or process eos
            if (vpu_dec->must_copy_data)
            {
                retval = mfw_gst_vpudec_copy_sink_input(vpu_dec, buffer);
                if (retval != GST_FLOW_OK)
                {
                    GST_ERROR(">>VPU_DEC: mfw_gst_vpudec_copy_sink_input failed - Error %d \n",retval);
                    goto done;
                }           
				if (vpu_dec->passthru)
                { 
                   goto done;
                }
            } else if (!vpu_dec->eos) {  // check if it is okay to start another frame before exiting
                if (vpu_dec->file_play_mode) {
                    if (!fContinue)
                       goto done;  // exit if continue flag was turned off 
                } else { // if streaming mode check if enough data to start another decode
                    PhysicalAddress p1, p2;
                    Uint32 space=BUFF_FILL_SIZE;
                    vpu_ret = vpu_DecGetBitstreamBuffer(*(vpu_dec->handle), &p1, &p2, &space);
                    vpu_dec->data_in_vpu = BUFF_FILL_SIZE - space;
                    if (space == 0)
                    {  // sometimes the buffer goes empty - in that case make sure to exit out 
                        retval =  GST_FLOW_OK;
                        goto done;
                    }
                }
            }

            // Don't bother decoding in VPU if we don't have enough data and are not in eos
            if (!vpu_dec->file_play_mode && !vpu_dec->must_copy_data && !vpu_dec->loopback &&
                !vpu_dec->eos && (vpu_dec->data_in_vpu < vpu_dec->min_data_in_vpu) )  
            {
                //GST_FRAMEDBG(">>VPU_DEC: Exiting chain to get more data %d\n", vpu_dec->data_in_vpu);
                retval =  GST_FLOW_OK;
                goto done;
            }

            if (vpu_dec->vpu_init == FALSE)
            {
                retval = mfw_gst_vpudec_vpu_init(vpu_dec);
                if (retval != GST_FLOW_OK)
                   goto done;
            }


            // Release buffers back to VPU before starting next decode
            retval = mfw_gst_vpudec_release_buff(vpu_dec);
            if (retval != GST_FLOW_OK)
            {
                // Error in clearing VPU buffers
                retval =  GST_FLOW_OK;
                goto check_continue;
            }

            vpu_dec->outputInfo->indexFrameDisplay = -1;
            vpu_dec->outputInfo->indexFrameDecoded = -1;

            //GST_FRAMEDBG (">>VPU_DEC: Start frame decoding \n");

            vpu_ret = vpu_DecStartOneFrame(*(vpu_dec->handle), vpu_dec->decParam);

            if (vpu_ret == RETCODE_FRAME_NOT_COMPLETE) {
                retval =  GST_FLOW_OK;
                goto done;
            }


            if (vpu_ret != RETCODE_SUCCESS) {
                GST_ERROR(">>VPU_DEC: vpu_DecStartOneFrame failed. Error code is %d \n",vpu_ret);
                retval = GST_FLOW_ERROR;
                goto done;
            }

            vpu_dec->is_frame_started = TRUE;
            vpu_dec->num_timeouts = 0;

            // This is the parallelization hook - but it should not exit in the following cases
            // eos, unconsumed buffer, interlaced frame, packed frame
            //GST_FRAMEDBG(">>VPU_DEC: eos %d must_copy %d just_flushed %d\n", vpu_dec->eos, vpu_dec->must_copy_data,vpu_dec->just_flushed);
            if (vpu_dec->allow_parallelization && 
                !vpu_dec->eos && !fContinue && !vpu_dec->just_flushed &&
                !vpu_dec->must_copy_data &&
                vpu_dec->state_playing &&  // if paused we have to get the output
#ifndef VPU_MX27  // mp4packedframes not supported on 27                   
                (vpu_dec->outputInfo->mp4PackedPBframe == 0) &&
#endif
                !INTERLACED_FRAME)
            {
                //GST_FRAMEDBG(">>VPU_DEC: Parallelization exit \n");
                retval = GST_FLOW_OK;
                goto done;
            }
        }

        // Wait for decode to complete and pass any displayable buffers to next in pipeline
        if(G_LIKELY(vpu_dec->is_frame_started == TRUE))
        {
            if (G_UNLIKELY(vpu_dec->profiling)) 
            {
                gettimeofday(&tv_prof, 0);
            }

            while (vpu_IsBusy())
            {
                if ((loop_cnt > 1) && !vpu_dec->must_copy_data && !vpu_dec->file_play_mode && !vpu_dec->loopback && !vpu_dec->eos)
                    goto done;
                if (vpu_dec->in_cleanup || vpu_dec->flushing)
                    goto done;

                vpu_mutex_unlock(vpu_dec->vpu_mutex);
                GST_MUTEX (">>VPU_DEC: unlock mutex before vpu wait cnt=%d\n", mutex_cnt);
                vpu_ret = vpu_WaitForInt(33);

                if (!vpudec_global_ptr || vpu_dec->in_cleanup || !vpu_dec->vpu_init)
                    goto done;

                if (vpu_mutex_lock(vpu_dec->vpu_mutex, TRUE) == FALSE)
                {
                    vpu_dec->trymutex = FALSE;
                    GST_MUTEX (">>VPU_DEC: after VPU Wait  - no mutex lock cnt=%d\n", mutex_cnt);
                    goto done;  // no mutex so exit - must be flushing or cleaning up
                } else {
                    GST_MUTEX (">>VPU_DEC: after VPU Wait - mutex lock cnt=%d\n", mutex_cnt);
                    if (!vpu_dec->is_frame_started || !vpu_dec->vpu_init) // can hanppen after flush after mutex is reacquired
                        goto done;
                }
                // avoid an infinite loop and make sure that exit and try later
                if (vpu_IsBusy())
                {   
                    vpu_dec->num_timeouts++;
                    GST_FRAMEDBG(">>VPU_DEC: VPU is timing out past 1 frame time timeouts=%d loop_cnt% d vpu_ret=%d frame_started %d\n", vpu_dec->num_timeouts, loop_cnt, vpu_ret, frame_started);
                    if (vpu_dec->num_timeouts > 100) {
                        vpu_DecUpdateBitstreamBuffer(*(vpu_dec->handle), 0);
                        mfw_gst_vpudec_post_fatal_error_msg(vpu_dec,"VPU Decoder Failed - too many time outs ");
                        retval = GST_FLOW_ERROR;
                        goto done;
                    }
                    if (!vpu_dec->file_play_mode)            
                    {
                        if (vpu_dec->must_copy_data)
                        {
                            retval = mfw_gst_vpudec_copy_sink_input(vpu_dec, buffer);
                            if (retval != GST_FLOW_OK)
                            {
                                GST_ERROR(">>VPU_DEC: mfw_gst_vpudec_copy_sink_input failed - Error %d \n",retval);
                                goto done;
                            }
                         } 
                         if (!vpu_dec->must_copy_data && (loop_cnt > 1) && !vpu_dec->loopback)
                            goto done;
                    }
                    if ((vpu_dec->num_timeouts > 1) && !vpu_dec->loopback)
                        goto check_continue;
                }
            }

            vpu_dec->num_timeouts = 0;
            if (G_UNLIKELY(vpu_dec->profiling)) {
                gettimeofday(&tv_prof1, 0);
                time_before = (tv_prof.tv_sec * 1000000) + tv_prof.tv_usec;
                time_after = (tv_prof1.tv_sec * 1000000) + tv_prof1.tv_usec;
                vpu_dec->decode_wait_time += time_after - time_before;
            }

            // Get the VPU output from decoding
            vpu_ret = vpu_DecGetOutputInfo(*(vpu_dec->handle),vpu_dec->outputInfo);

            if (vpu_ret != RETCODE_SUCCESS) {
                // in some demuxers we get headers separately which will cause VPU to fail
                // so in those cases just reset our error and wait for more data
                if (vpu_ret == RETCODE_FAILURE)
                {
                    retval = GST_FLOW_OK;
                    vpu_dec->is_frame_started = FALSE;
                } else {
                    retval = GST_FLOW_ERROR;
                    GST_ERROR (">>VPU_DEC: vpu_DecGetOutputInfo failed. Error code is %d \n", vpu_ret);
                }
                if (vpu_dec->must_copy_data)
                    goto check_continue;
                else
                    goto done;
            }

            if ((vpu_dec->decParam->iframeSearchEnable) && (VPU_PIC_TYPE == 0))
            {   
                    vpu_dec->decParam->iframeSearchEnable = 0;
            }

            vpu_dec->is_frame_started = FALSE;

            //GST_DEBUG(">>VPU_DEC: output pictype=%d\n",vpu_dec->outputInfo->picType);

            if ((vpu_dec->decParam->prescanEnable == 1) && 
                (vpu_dec->outputInfo->prescanresult == 0))
            {
                //vpu_dec->allow_parallelization = FALSE;
                if ((loop_cnt == 1) && (vpu_dec->min_data_in_vpu  < (256*1024)))
                    vpu_dec->min_data_in_vpu += 2048;

                GST_FRAMEDBG(">>VPU_DEC: The prescan result is zero, VPU needs more input - data in vpu=%d\n", vpu_dec->data_in_vpu);
                // Return for more data as this is incomplete - but do not process as an error
                retval = GST_FLOW_OK;
                if (vpu_dec->must_copy_data)
                    goto check_continue;
                else
                    goto done;
            }

            // Use this for debugging VPU issues with buffer indexes
            GST_FRAMEDBG (">>VPU_DEC: --DEC disp=%d decode=%d pic_type %d loopcnt=%d datain vpu %d min %d\n",vpu_dec->outputInfo->indexFrameDisplay,
                 vpu_dec->outputInfo->indexFrameDecoded, VPU_PIC_TYPE , loop_cnt, vpu_dec->data_in_vpu, vpu_dec->min_data_in_vpu);

            if ((vpu_dec->outputInfo->indexFrameDisplay >= 0) || vpu_dec->drop_before_decode) 
                vpu_dec->frames_decoded++;


            if ((vpu_dec->outputInfo->indexFrameDisplay >= 0) &&
                (vpu_dec->fb_state_plugin[vpu_dec->outputInfo->indexFrameDisplay] == FB_STATE_DISPLAY))
            {
                GST_ERROR(">>VPU_DEC: Display frame is same as last so ignore \n");
                goto check_continue;
            }

            if (vpu_dec->outputInfo->indexFrameDecoded >= 0)
            {
                mfw_vpudec_get_field(vpu_dec);
#ifdef VPU_MX27
                // check in case VPU in error returns a buffer that was in decode state before
                if (vpu_dec->fb_state_plugin[vpu_dec->outputInfo->indexFrameDecoded] == FB_STATE_DISPLAY)
                {
                    GST_FRAMEDBG(">>VPU_DEC: Decoded returned was in display mode buff %d\n",vpu_dec->outputInfo->indexFrameDecoded);
                    vpu_DecClrDispFlag (*(vpu_dec->handle), vpu_dec->outputInfo->indexFrameDecoded);
                }
#endif
                vpu_dec->fb_state_plugin[vpu_dec->outputInfo->indexFrameDecoded] = FB_STATE_DECODED;



            }




            // interlaced not supported in MX27 
            if (INTERLACED_FRAME  && vpu_dec->lastframedropped)
            {
                 if ( vpu_dec->ts_tx != vpu_dec->ts_rx)
                     vpu_dec->ts_tx = (vpu_dec->ts_tx+1)%MAX_STREAM_BUF;                    
                 //GST_DEBUG(">>VPU_DEC: Interlaced - skipping one time stamp \n");
            }

            // Check for cases a frame not to display
            if (G_UNLIKELY(vpu_dec->outputInfo->indexFrameDisplay < 0))
            {
                // if eos marked by VPU then save the eos state
                if ((vpu_dec->outputInfo->indexFrameDisplay == -1) &&
                    (vpu_dec->outputInfo->indexFrameDecoded < 0))
                {
                    GST_DEBUG(">>VPU_DEC: VPU hits EOS must reset on next buffer \n");
                    vpu_dec->eos = TRUE;              
                }

                vpu_dec->lastframedropped = TRUE;
                 
#ifndef VPU_MX27 // mp4packed frames not supported on mx27 
                if (vpu_dec->outputInfo->mp4PackedPBframe == 1) 
                {
                    GST_FRAMEDBG(">>VPU_DEC: Packed Frame set with negative display idx \n");
                    vpu_dec->allow_parallelization = FALSE;             
                    goto check_continue;
                }
#endif

                // loop again unless our decoded was also negative
                if (vpu_dec->outputInfo->indexFrameDecoded < 0)
                {
                    // if i frame was not found then skip a timestamp
                    if (G_UNLIKELY(((vpu_dec->outputInfo->indexFrameDisplay == -3) && (vpu_dec->codec==STD_RV))
                        ||((loop_cnt == 1) && (vpu_dec->outputInfo->indexFrameDisplay == -3)))){
                        if ( vpu_dec->ts_tx != vpu_dec->ts_rx){
                            vpu_dec->ts_tx = (vpu_dec->ts_tx+1)%MAX_STREAM_BUF;  
                        }
                    }
                    retval = GST_FLOW_OK;
                    goto check_continue;
                }
                goto check_continue;
            }
            vpu_dec->lastframedropped = FALSE;


            // Check for discontinuity - set during flush that we not display until after i frame
            // this is important because on VC-1 there is sometimes b-frames after 
            if (vpu_dec->just_flushed || vpu_dec->drop_before_decode)
            {   // at least decoded or display has to be positive for us to reset iframesearch
                if (VPU_PIC_TYPE == 0) 
                {   
                    GST_FRAMEDROP(">>VPU_DEC: I frame received so resume play at %d \n", vpu_dec->frames_decoded);                                      
                    //vpu_dec->decParam->iframeSearchEnable = 0; // turn off iframesearch - i frame found, it must be off at this place
                    vpu_dec->decParam->skipframeMode = 0;  // turn off skipframe used in networking
                    vpu_dec->decParam->skipframeNum = 0;  // turn off skipframe used in networking                       
                    vpu_dec->just_flushed = FALSE;
    
                    if (vpu_dec->check_for_iframe)
                    {
                        //GST_FRAMEDROP (">>VPU_DEC: wait to display until next I frame \n");
                        mfw_gst_vpudec_no_display(vpu_dec);
                        goto check_continue;
                    }               
                } else {
                    GST_FRAMEDROP (">>VPU_DEC: No I frame don't display yet pictype= %d \n", VPU_PIC_TYPE );
                    mfw_gst_vpudec_no_display(vpu_dec);
                    goto check_continue;
                }            
            } else if (vpu_dec->check_for_iframe) { 
                if (VPU_PIC_TYPE == 0)
                {
                    GST_FRAMEDROP (">>VPU_DEC: I frame for H.264 now start displaying \n");
                    vpu_dec->check_for_iframe = FALSE;
                    vpu_dec->check_for_bframe = FALSE;
                } else                 {
                    GST_FRAMEDROP (">>VPU_DEC: wait to display until next I frame \n");
                    mfw_gst_vpudec_no_display(vpu_dec);
                    goto check_continue;
                }                
            }
            // the following check is important - after seeking there might be b frames that relate to 
            // earlier b-frames before seek.  We must skip b frames after a flush until either an I or P
            // frame arrives to avoid possible garbage that might be displayed from these b-frames
            // Note that this is done after 2nd decode after iframesearch turned off because we must wait
            // until next I or P before turning this off
            else if (vpu_dec->check_for_bframe)
            {
                if (VPU_PIC_TYPE == 2) 
                {
                    mfw_gst_vpudec_no_display(vpu_dec);
                    GST_DEBUG (">>VPU_DEC: B Frame still %d skip all B frames \n", VPU_PIC_TYPE );
                    goto check_continue;
                } else
                    vpu_dec->check_for_bframe= FALSE;
            }

            // To improve latency it helps to tweak our threshold in antipation of I frames especially with VBR input
            // can guess when an I frame is coming up and enable prescan for I frames and limit the minimim
            // data required in bitstream before starting a decode
            if (VPU_PIC_TYPE == 0)
            {
                vpu_dec->num_gops ++;
                if ((vpu_dec->frames_rendered > 0) && !vpu_dec->drop_before_decode)
                {
                    guint new_gop_size = (guint)(vpu_dec->frames_decoded - vpu_dec->idx_last_gop);
                    // if we are not having same gop sizes then turn off the predict gop flag
                    // we use for frame dropping
                    if (new_gop_size != vpu_dec->gop_size) {   
                        vpu_dec->predict_gop = FALSE;
                    } else {
                        vpu_dec->predict_gop = TRUE;
                    }
                } else  {
                    //GST_LATENCY (">>VPU_DEC: frames since last gop %d gop_size %d\n", 
                    //               (guint)(vpu_dec->frames_rendered - vpu_dec->idx_last_gop), vpu_dec->gop_size);
                }
                vpu_dec->gop_size = (guint)(vpu_dec->frames_decoded - vpu_dec->idx_last_gop);
                vpu_dec->idx_last_gop = vpu_dec->frames_decoded;
                vpu_dec->gop_is_next = FALSE;
                GST_LATENCY ("\n \n >>VPU_DEC: numgops %d gopsize %d \n", vpu_dec->num_gops, vpu_dec->gop_size);  
            } else {
                guint frames_since_gop = vpu_dec->frames_decoded - vpu_dec->idx_last_gop+1;
                if (vpu_dec->predict_gop && vpu_dec->gop_size && (frames_since_gop >= vpu_dec->gop_size))
                {
                    vpu_dec->gop_is_next = TRUE;
                    //GST_LATENCY (">>VPU_DEC: Delta -  frames since gop %d  INCREASING MINIMUM  to %d \n",
                    //             (guint)(vpu_dec->frames_rendered - vpu_dec->idx_last_gop), vpu_dec->min_data_in_vpu);
                } else {
                    //GST_LATENCY (">>VPU_DEC: Delta -  frames since gop %d  loop %d\n",
                    //       (guint)(vpu_dec->frames_decoded - vpu_dec->idx_last_gop), loop_cnt);
                }
            }
    
            // render a frame 
            retval = mfw_gst_vpudec_render(vpu_dec);
            // a pause could have locked the pipeline so could exit and vpu is dead so exit
            if (vpudec_global_ptr == NULL)
                return GST_FLOW_OK;

            if (retval != GST_FLOW_OK)
            {
                //GST_ERROR (">>VPU_DEC: Render frame failed \n");
                retval = GST_FLOW_OK;
                if (mutex_cnt && vpu_dec->must_copy_data && !vpu_dec->flushing && !vpu_dec->in_cleanup)
                    goto check_continue;
                else
                    goto done;
            }

        }  /* get output processing done */

check_continue:
        retval =  GST_FLOW_OK;
        fContinue = mfw_gst_vpudec_continue_looping(vpu_dec, loop_cnt);
    } while (fContinue || vpu_dec->must_copy_data );

done:
    if (!vpudec_global_ptr || vpu_dec->in_cleanup)
        return GST_FLOW_OK;

    if (vpu_dec->trymutex)
    {
        vpu_mutex_unlock(vpu_dec->vpu_mutex);
        GST_MUTEX (">>VPU_DEC: Unlocking mutex at end of chain cnt=%d\n",mutex_cnt);
    } else {
        GST_MUTEX (">>VPU_DEC: chain does not unlock mutex - might be cleaning up or flushing mutex_cnt=%d cleanup=%d\n",mutex_cnt, vpu_dec->in_cleanup);
    }

    if (G_UNLIKELY(vpu_dec->profiling)) {
        gettimeofday(&tv_prof3, 0);
        time_before = (tv_prof2.tv_sec * 1000000) + tv_prof2.tv_usec;
        time_after = (tv_prof3.tv_sec * 1000000) + tv_prof3.tv_usec;
        vpu_dec->chain_Time += time_after - time_before;
    }

    // vc1 buffer join does an unref so can't do it in this case
    if (vpu_dec->gst_buffer != NULL)
    {
        gst_buffer_unref(vpu_dec->gst_buffer);
        vpu_dec->gst_buffer = NULL;
    }

    return retval;
}

/*======================================================================================
FUNCTION:           mfw_gst_vpudec_sink_event

DESCRIPTION:        This function handles the events the occur on the sink pad
                    Like EOS

ARGUMENTS PASSED:   pad - pointer to the sinkpad of this element
                    event - event generated.

RETURN VALUE:       TRUE   event handled success fully.
                    FALSE .event not handled properly

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=======================================================================================*/
static gboolean mfw_gst_vpudec_sink_event(GstPad * pad, GstEvent * event)
{
    MfwGstVPU_Dec *vpu_dec = MFW_GST_VPU_DEC(GST_PAD_PARENT(pad));
    gboolean result = TRUE;
    gint i=0;
    guint height=0,width=0;
    RetCode vpu_ret=RETCODE_SUCCESS;
    gulong size=0;
    gint  flushsize=0;
    guint offset=0;

    if ((vpudec_global_ptr == NULL) || vpu_dec->in_cleanup)
        return GST_FLOW_OK;

    width = vpu_dec->initialInfo->picWidth;
    height = vpu_dec->initialInfo->picHeight;

    switch (GST_EVENT_TYPE(event)) {

    case GST_EVENT_NEWSEGMENT:
        {
            GstFormat format;
            gint64 start, stop, position;
            gdouble rate;
            gint i;

            gst_event_parse_new_segment(event, NULL, &rate, &format,
                &start, &stop, &position);
            GST_DEBUG(">>VPU_DEC: Receiving new seg \n");
            GST_DEBUG(">>VPU_DEC: start = %" GST_TIME_FORMAT, GST_TIME_ARGS(start));
            GST_DEBUG(" stop = %" GST_TIME_FORMAT, GST_TIME_ARGS(stop));
            GST_DEBUG(" position =% " GST_TIME_FORMAT, GST_TIME_ARGS(position));
            GST_DEBUG("\n");
            if (vpu_dec->codec==STD_MPEG2){
                vpu_dec->last_ts_sent = start;
            }
            vpu_dec->drop_before_decode = FALSE;
            vpu_dec->frames_decoded = 0; 
            vpu_dec->frames_rendered = 0; 
            vpu_dec->just_flushed = FALSE;
            vpu_dec->check_for_bframe = FALSE;

            if (GST_FORMAT_TIME == format) {
                result = gst_pad_push_event(vpu_dec->srcpad, event);
                if (TRUE != result) {
                    GST_ERROR("\n>>VPU_DEC: Error in pushing the event %d\n", event);
                }
             } else GST_DEBUG (">>VPU_DEC: new seg event not pushed format = %d\n", (guint)format);
            GST_DEBUG (">>VPU_DEC: End New Seg event \n");

            break;
        }

    case GST_EVENT_FLUSH_START:
        {
            GST_DEBUG (">>VPU_DEC: Receiving Flush start event \n");
            if (vpu_dec->in_cleanup)
                return GST_FLOW_OK;

            GST_MUTEX(">>VPU_DEC: flush start before mutex_lock cnt=%d\n", mutex_cnt);
            vpu_mutex_lock(vpu_dec->vpu_mutex, FALSE);
            vpu_dec->flushing = TRUE;
            vpu_mutex_unlock(vpu_dec->vpu_mutex);
            GST_MUTEX(">>VPU_DEC: flush start after mutex_lock cnt=%d\n", mutex_cnt);

            if (!vpu_dec->vpu_init)
            {
                GST_DEBUG (">>VPU_DEC: Ignore flush since vpu not init \n");
                vpu_mutex_lock(vpu_dec->vpu_mutex, FALSE);
                vpu_dec->flushing = FALSE;
                vpu_mutex_unlock(vpu_dec->vpu_mutex);
                result = gst_pad_push_event(vpu_dec->srcpad, event);
                return GST_FLOW_OK;
            }


            result = gst_pad_push_event(vpu_dec->srcpad, event);
            if (TRUE != result) {
                GST_ERROR("\n>>VPU_DEC: Error in pushing the event %d\n", event);
            }
            GST_DEBUG (">>VPU_DEC: End Flush start event \n");
        }
        break;

    case GST_EVENT_FLUSH_STOP:
        {
            GST_DEBUG (">>VPU_DEC: Receiving Flush stop event \n");
            gboolean fdoReset = FALSE;

            vpu_dec->data_in_vpu = 0;
            memset(&vpu_dec->timestamp_buffer[0],0,MAX_STREAM_BUF*sizeof(GstClockTime));
            vpu_dec->ts_rx=0;
            vpu_dec->ts_tx=0;
            vpu_dec->last_ts_sent = 0;      
            vpu_dec->last_ts_in = 0;      
            vpu_dec->num_timeouts = 0;

            if (!vpu_dec->vpu_init)
            {
                GST_DEBUG (">>VPU_DEC: Ignore flush stop since vpu not init \n");
                vpu_dec->flushing = FALSE;
                result = gst_pad_push_event(vpu_dec->srcpad, event);
                return GST_FLOW_OK;
            }

            if (vpu_dec->is_frame_started) {
                vpu_ret=vpu_DecGetOutputInfo(*vpu_dec->handle, vpu_dec->outputInfo);
                if (vpu_ret==RETCODE_SUCCESS)
                    vpu_dec->is_frame_started = FALSE;
                else  {
                    vpu_ret = vpu_DecUpdateBitstreamBuffer(*(vpu_dec->handle), 0); 
                    i=0;
                    fdoReset = TRUE;
                    while (vpu_IsBusy())
                    {
                        //GST_DEBUG (">>VPU_DEC: waiting for VPU to get out of busy state \n");
                         usleep(500000);
                         i++;
                         if (i>10)
                             break;
                    }
                    vpu_dec->is_frame_started = FALSE;
                    vpu_DecGetOutputInfo(*vpu_dec->handle, vpu_dec->outputInfo);
                    GST_ERROR(">>VPU_DEC: Error in Flush getting output from vpu so flush bitstream buffer\n");
                }
            }
            
            // turn on i frame search so VPU will not return display buffers until i frame
            // this is very slow so we only use it on flushes and turn it off as soon as i frame apears
            // this helps us prevent garbage from being displayed after seeking
            //vpu_dec->check_for_bframe = TRUE;  // this is only used after flushing
            
            // put all buffers in pending free state to free later

            for (i=0;i<vpu_dec->numframebufs;i++)
            {
                // this is because for display cases, in some cases the buffers are not unreffed
                // later we must unref if these show up as locked in release_buf - if we do not do
                // this eventually after a bunch of seeks VPU just stays locked up constantly returning
                // -3 for disp and -1 for decode (usually for VC1).
                if (vpu_dec->fb_state_plugin[i] == FB_STATE_DISPLAY)
                   ;//vpu_dec->fb_state_plugin[i] = FB_STATE_FREE;
                if (vpu_dec->fb_state_plugin[i] == FB_STATE_DECODED)
                   vpu_dec->fb_state_plugin[i] = FB_STATE_PENDING;
            }

            if (!vpu_dec->eos)
            {
                vpu_dec->check_for_bframe = TRUE;  // this is only used after flushing
                vpu_dec->just_flushed = TRUE; 
            }

            if (vpu_dec->file_play_mode == FALSE) {
                //GST_DEBUG(">>VPU_DEC: FLUSHSTOP streaming mode\n");
                mfw_gst_vpudec_reset(vpu_dec);
#if 0
                if (vpu_dec->eos || vpu_IsBusy())
                {
                    mfw_gst_vpudec_reset(vpu_dec);
                } else {
                    vpu_ret = vpu_DecBitBufferFlush(*vpu_dec->handle);
                    vpu_dec->start_addr = vpu_dec->base_addr;
                }
#endif
            } else {
                //if (fdoReset){
                if((vpu_dec->codec == STD_MPEG4) || (vpu_dec->codec == STD_RV))
                    mfw_gst_vpudec_reset(vpu_dec);  
                //}
				vpu_dec->decParam->iframeSearchEnable = 1;
            }

            result = gst_pad_push_event(vpu_dec->srcpad, event);
            if (TRUE != result) {
                GST_ERROR("\n>>VPU_DEC: Error in pushing the event %d\n", event);
            }
            vpu_mutex_lock(vpu_dec->vpu_mutex, FALSE);
            vpu_dec->flushing = FALSE;
            vpu_mutex_unlock(vpu_dec->vpu_mutex);
            GST_DEBUG (">>VPU_DEC: End Flush stop event \n");

            break;
        }
    case GST_EVENT_EOS:
        {
            GST_DEBUG (">>VPU_DEC: receiving EOS event \n");

            if (!vpu_dec->flushing && 
                (vpu_dec->codec != STD_MJPG))
            {
                mfw_gst_vpudec_chain(vpu_dec->sinkpad, NULL);
            } 
            result = gst_pad_push_event(vpu_dec->srcpad, event);
            if (result != TRUE) {
               GST_ERROR("\n>>VPU_DEC: Error in pushing EOS event %d\n", event);
            }
            break;
        }
    default:
        {
              
            GST_DEBUG ("\n>>VPU_DEC: Event unhandled %d\n", event);

            result = gst_pad_event_default(pad, event);
            break;
        }

    }
    return result;
        
}

/*======================================================================================
FUNCTION:           mfw_gst_vpudec_cleanup

DESCRIPTION:        cleans up allocated memory and resources

ARGUMENTS PASSED:   vpu_dec  - VPU decoder plugins context

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=======================================================================================*/
void  mfw_gst_vpudec_cleanup(MfwGstVPU_Dec *vpu_dec)
{
    RetCode vpu_ret=RETCODE_SUCCESS;

    if (vpudec_global_ptr == NULL)
        return;

    vpu_dec->in_cleanup = TRUE;
    vpu_dec->start_addr=NULL;
    vpu_dec->end_addr=NULL;
    vpu_dec->base_addr=NULL;

    GST_DEBUG(">>>>VPU_DEC: Cleanup frame started %d\n", vpu_dec->is_frame_started);

    if (vpu_dec->is_frame_started)
    {
        // need this to avoid hangs on vpu close which will not close if outstanding
        // decode - best way to kill an oustanding decode is flush the bitstream buffer
        vpu_ret = vpu_DecUpdateBitstreamBuffer(*(vpu_dec->handle), 0);

        // if VPU is busy you can't cleanup and close will hang so be sure to set
        // bitstream to 0 so reset can happen
        while (vpu_IsBusy())
        {
             //GST_DEBUG (">>VPU_DEC: waiting for VPU to get out of busy state \n");
             usleep(500000);
        }

        vpu_DecGetOutputInfo(*vpu_dec->handle, vpu_dec->outputInfo);
        if (vpu_ret == RETCODE_SUCCESS) {
            vpu_dec->is_frame_started = FALSE;        
        }
    }

    mfw_gst_vpudec_FrameBufferClose(vpu_dec);

    if (vpu_dec->direct_render)
    {
        /* release framebuffers hold by vpu */
        int cnt;
        for (cnt=0; cnt<vpu_dec->numframebufs; cnt++) {
#if 0 //no need to unref, since vpu always hold the buffer reference.
            if (vpu_dec->outbuffers[cnt])
				gst_buffer_unref(vpu_dec->outbuffers[cnt]);
#endif
        }
    }

    vpu_dec->firstFrameProcessed=FALSE;
    memset(&vpu_dec->timestamp_buffer[0],0,MAX_STREAM_BUF*sizeof(GstClockTime));
    vpu_dec->yuv_frame_size=0;
    vpu_dec->direct_render=FALSE;
    vpu_dec->framebufinit_done=FALSE;
    vpu_dec->last_ts_sent = 0;
    vpu_dec->last_ts_in = 0;      

    if ((vpu_dec->codec == STD_MPEG2) && vpu_dec->codec_data)
    {
        gst_buffer_unref(vpu_dec->codec_data);
        vpu_dec->codec_data_len = 0;
        vpu_dec->codec_data = 0;
    }
    if (vpu_dec->vpu_opened)
    {                
        GST_DEBUG(">>>>VPU_DEC: Before close frame_started=%d vpu busy %d\n", vpu_dec->is_frame_started, vpu_IsBusy() );
        // if vpu is still in busy state this might hang but if you don't do close then can't open another file after this.
#if 0
        vpu_ret = vpu_DecClose(*vpu_dec->handle);
#else
        decoder_close(vpu_dec);
#endif
        vpu_dec->vpu_opened = FALSE;
        GST_DEBUG(">>>>VPU_DEC: After close\n");
    }

    if (vpu_dec->bit_stream_buf.phy_addr)
    {
        IOFreeVirtMem(&(vpu_dec->bit_stream_buf));
        IOFreePhyMem(&(vpu_dec->bit_stream_buf));
        
        vpu_dec->bit_stream_buf.phy_addr = 0;
    } 
    if (vpu_dec->ps_mem_desc.phy_addr) 
    {

        IOFreeVirtMem(&(vpu_dec->ps_mem_desc));
        IOFreePhyMem(&(vpu_dec->ps_mem_desc));
        vpu_dec->ps_mem_desc.phy_addr = 0;
    }
    if (vpu_dec->slice_mem_desc.phy_addr) 
    {
        IOFreeVirtMem(&(vpu_dec->slice_mem_desc));
        IOFreePhyMem(&(vpu_dec->slice_mem_desc));
        vpu_dec->ps_mem_desc.phy_addr = 0;
    }
    if (vpu_dec->decOP !=NULL)
    {
        MM_FREE(vpu_dec->decOP);  
        vpu_dec->decOP = NULL;
    }

    if (vpu_dec->initialInfo !=NULL)
    {                
        MM_FREE(vpu_dec->initialInfo);
        vpu_dec->initialInfo = NULL;               
    }
    if (vpu_dec->decParam !=NULL)
    {
        MM_FREE(vpu_dec->decParam);
        vpu_dec->decParam = NULL;
    }
    if (vpu_dec->outputInfo !=NULL)
    {
        MM_FREE(vpu_dec->outputInfo);
        vpu_dec->outputInfo = NULL;
    }
    if (vpu_dec->handle !=NULL)
    {
        MM_FREE(vpu_dec->handle);
        vpu_dec->handle = NULL;
    }

#if 0
    /* Unlock the mutex to free the mutex 
    * in case of date terminated.
    */
    if (vpu_dec->vpu_mutex != NULL)
    {
        GST_MUTEX(">>VPU_DEC: Before mutex free \n");
        vpu_mutex_unlock(vpu_dec->vpu_mutex);
        g_mutex_free(vpu_dec->vpu_mutex);
        vpu_dec->vpu_mutex = NULL;    
        mutex_cnt = 0;       
    }
#endif    
    vpu_dec->last_ts_sent = 0;
    vpu_dec->last_ts_in = 0;
    vpu_dec->clock_base = 0;
    GST_DEBUG(">>>>VPU_DEC: End Cleanup \n");

}

/*======================================================================================
FUNCTION:           mfw_gst_vpudec_change_state

DESCRIPTION:        This function keeps track of different states of pipeline.

ARGUMENTS PASSED:
                element     -   pointer to element
                transition  -   state of the pipeline

RETURN VALUE:
                GST_STATE_CHANGE_FAILURE    - the state change failed
                GST_STATE_CHANGE_SUCCESS    - the state change succeeded
                GST_STATE_CHANGE_ASYNC      - the state change will happen asynchronously
                GST_STATE_CHANGE_NO_PREROLL - the state change cannot be prerolled

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=======================================================================================*/
static GstStateChangeReturn mfw_gst_vpudec_change_state
                            (GstElement *element, GstStateChange transition)
{
    GstStateChangeReturn ret  = GST_STATE_CHANGE_SUCCESS;
    MfwGstVPU_Dec *vpu_dec    = MFW_GST_VPU_DEC(element);
    gint vpu_ret=RETCODE_SUCCESS;
    
    switch (transition) 
    {
        case GST_STATE_CHANGE_NULL_TO_READY:
	    {
            vpu_versioninfo ver;
            MM_INIT_DBG_MEM("vpu_dec");
            GST_DEBUG("\n>>VPU_DEC: State: Null to Ready\n");
            if (vpu_dec->vpu_chipinit == FALSE)
            {
                vpu_ret = vpu_Init(NULL);
                if(vpu_ret < 0){
                    GST_DEBUG(">>VPU_DEC: Error in initializing the VPU\n,error is %d",vpu_ret);
                    return GST_STATE_CHANGE_FAILURE;
                }
            }

            vpu_dec->vpu_chipinit = TRUE;

            vpu_ret = vpu_GetVersionInfo(&ver);
            if (vpu_ret) {
                GST_DEBUG(">>VPU_DEC: Error in geting the VPU version\n,error is %d",vpu_ret);
                vpu_UnInit();
                vpu_dec->vpu_chipinit = FALSE;
                return GST_STATE_CHANGE_FAILURE;
            }

            g_print(YELLOW_STR("VPU Version: firmware %d.%d.%d; libvpu: %d.%d.%d \n", 
                ver.fw_major, ver.fw_minor, ver.fw_release,
                ver.lib_major, ver.lib_minor, ver.lib_release));

            #define MFW_GST_VPU_DECODER_PLUGIN VERSION
            PRINT_PLUGIN_VERSION(MFW_GST_VPU_DECODER_PLUGIN); 
            break;
	    }
        case GST_STATE_CHANGE_READY_TO_PAUSED:
	    {
            GST_DEBUG("\n>>VPU_DEC: State: Ready to Paused\n\n");
            if (vpudec_global_ptr == NULL)
                return GST_FLOW_OK;

            vpu_dec->vpu_init=FALSE;
            vpu_dec->vpu_opened = FALSE;
            vpu_dec->just_flushed = FALSE;
            vpu_dec->flushing = FALSE;
            vpu_dec->data_in_vpu = 0;
            vpu_dec->start_addr=NULL;
            vpu_dec->end_addr=NULL;
            vpu_dec->base_addr=NULL;
            vpu_dec->yuv_frame_size=0;
            vpu_dec->decode_wait_time = 0;
            vpu_dec->chain_Time = 0;
            vpu_dec->num_timeouts = 0;
            vpu_dec->codec_data = 0;
            vpu_dec->codec_data_len = 0;

            // used for latency
            vpu_dec->num_gops = 0;          
            vpu_dec->gop_size = 0;      
            vpu_dec->idx_last_gop = 0;      
            vpu_dec->gop_is_next = FALSE;


            vpu_dec->frames_rendered = 0;
            vpu_dec->frames_decoded = 0;
            vpu_dec->avg_fps_decoding = 0.0;
            vpu_dec->frames_dropped = 0;
            vpu_dec->drop_before_decode = FALSE;
            vpu_dec->direct_render=FALSE;
            vpu_dec->firstFrameProcessed=FALSE;
            vpu_dec->ts_rx=0;
            vpu_dec->ts_tx=0;
            vpu_dec->last_ts_sent = 0;
            vpu_dec->last_ts_in = 0;   
            vpu_dec->clock_base = 0;
            vpu_dec->state_playing = FALSE;;
            vpu_dec->framebufinit_done=FALSE;
            vpu_dec->file_play_mode = FALSE;
            vpu_dec->eos = FALSE;
            vpu_dec->just_flushed = FALSE;
            vpu_dec->mp4Class = MP4_MPEG4;
            vpu_dec->frame_rate_de = 1;
            vpu_dec->accumulate_hdr = FALSE;
            vpu_dec->nal_check = FALSE;
            vpu_dec->frame_rate_nu = DEFAULT_FRAME_RATE_NUMERATOR;
            {
                int cnt=0;
                for (cnt=0;cnt<NUM_FRAME_BUF;cnt++)
                    vpu_dec->outbuffers[cnt]=NULL;
            }
            memset(&vpu_dec->bit_stream_buf, 0, sizeof(vpu_mem_desc));
            memset(&vpu_dec->timestamp_buffer[0],0,MAX_STREAM_BUF * sizeof(GstClockTime));
            memset(&vpu_dec->frameBuf[0],0,NUM_FRAME_BUF * sizeof(FrameBuffer));
            memset(&vpu_dec->frame_mem[0],0,NUM_FRAME_BUF * sizeof(vpu_mem_desc));
            /* Handle the decoder Initialization over here. */
            vpu_dec->decOP          = MM_MALLOC(sizeof(DecOpenParam));
            vpu_dec->initialInfo    = MM_MALLOC(sizeof(DecInitialInfo));
            vpu_dec->decParam       = MM_MALLOC(sizeof(DecParam));
            vpu_dec->handle         = MM_MALLOC(sizeof(DecHandle));
            vpu_dec->outputInfo     = MM_MALLOC(sizeof(DecOutputInfo));
            memset(vpu_dec->decOP,0,sizeof(DecOpenParam));
            memset(vpu_dec->handle ,0,sizeof(DecHandle));
            memset(vpu_dec->decParam,0,sizeof(DecParam));
            memset(vpu_dec->outputInfo,0,sizeof(DecOutputInfo));
            memset(&vpu_dec->ps_mem_desc, 0, sizeof(vpu_mem_desc));
            memset(&vpu_dec->slice_mem_desc, 0, sizeof(vpu_mem_desc));
            memset(vpu_dec->initialInfo, 0, sizeof(DecInitialInfo));
            memset(vpu_dec->outputInfo,0, sizeof(DecOutputInfo));
            break;
	    }
        case GST_STATE_CHANGE_PAUSED_TO_PLAYING:
        case 0x24:  /* playing to playing - get this after first frame pushed downstream */
        {
            if (vpudec_global_ptr == NULL)
                return GST_FLOW_OK;

            if (GST_ELEMENT (vpu_dec)->clock)
            {
                vpu_dec->clock_base = gst_clock_get_time (GST_ELEMENT (vpu_dec)->clock);
                vpu_dec->clock_base -= gst_element_get_base_time(GST_ELEMENT (vpu_dec));
                GST_DEBUG("\n>>VPU_DEC: State: Transition to Playing new clock %d\n",(guint) vpu_dec->clock_base);
            } 
            vpu_dec->state_playing = TRUE;
            break;
        }
        default:
            GST_DEBUG ("\n>>VPU_DEC: State unhandled 0x%x\n", transition);
	    break;
    }

    ret = vpu_dec->parent_class->change_state(element, transition);
    GST_DEBUG("\n>>VPU_DEC: State Change 0x%x for VPU returned %d\n", transition, ret);

    switch (transition) 
    {
        case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
        {
            if (vpudec_global_ptr == NULL)
                return GST_FLOW_OK;

            GST_DEBUG("\n>>VPU_DEC: State: Playing to Paused\n");
            vpu_dec->state_playing = FALSE;;
	    break;
        }
        case GST_STATE_CHANGE_PAUSED_TO_READY:
	    {
            gfloat avg_mcps = 0, avg_plugin_time = 0, avg_dec_time = 0;

            GST_DEBUG("\n>>VPU_DEC: State: Paused to Ready\n");
 
            if (vpu_dec->profiling) {
                g_print(">>VPU_DEC: PROFILE FIGURES OF VPU DECODER PLUGIN");
                g_print("\nTotal decode wait time is            %fus",
                    (gfloat)vpu_dec->decode_wait_time);
                g_print("\nTotal plugin time is                 %ldus",
                    vpu_dec->chain_Time);
                g_print("\nTotal number of frames decoded is    %ld",
                    vpu_dec->frames_decoded);
                g_print("\nTotal number of frames dropped is    %ld\n",
                    vpu_dec->frames_dropped);
                if (vpu_dec->frame_rate != 0) {
                    avg_mcps = ((float) vpu_dec->decode_wait_time * PROCESSOR_CLOCK /
                        (1000000 *
                        (vpu_dec->frames_rendered -
                        vpu_dec->frames_dropped)))
                        * vpu_dec->frame_rate;
                    g_print("\nAverage decode WAIT MCPS is          %f", avg_mcps);
                    
                    avg_mcps = ((float) vpu_dec->chain_Time * PROCESSOR_CLOCK /
                        (1000000 *
                        (vpu_dec->frames_decoded -
                        vpu_dec->frames_dropped)))
                        * vpu_dec->frame_rate;
                    g_print("\nAverage plug-in MCPS is              %f",
                        avg_mcps);
                }
                else
                {
                    GST_DEBUG("enable the Frame Rate property of the decoder to get the MCPS \
                        ... \n ! mfw_vpudecoder framerate=value ! .... \
                        \n Note: value denotes the framerate to be set");
                }
                avg_dec_time =
                    ((float) vpu_dec->decode_wait_time) / vpu_dec->frames_decoded;
                g_print("\nAverage decoding Wait time is        %fus",
                    avg_dec_time);
                avg_plugin_time =
                    ((float) vpu_dec->chain_Time) / vpu_dec->frames_decoded;
                g_print("\nAverage plugin time is               %fus\n",
                    avg_plugin_time);
                
                vpu_dec->decode_wait_time = 0;
                vpu_dec->chain_Time = 0;
                vpu_dec->avg_fps_decoding = 0.0;
                vpu_dec->frames_decoded = 0;
                vpu_dec->frames_rendered = 0;
                vpu_dec->frames_dropped = 0;
            }
            if (vpudec_global_ptr == NULL || vpu_dec->in_cleanup)
                return GST_FLOW_OK;

            GST_MUTEX (">>VPU_DEC: Before mutex lock in Paused to ready cnt=%d\n", mutex_cnt);
            if (vpu_dec->vpu_mutex) {
                vpu_mutex_lock(vpu_dec->vpu_mutex, FALSE);

                // sometimes there was error in opening and/or init so still need cleanup here since pipeline was never started
                mfw_gst_vpudec_cleanup(vpu_dec);
                /* Unlock the mutex to free the mutex 
                * in case of date terminated.
                */
                vpu_mutex_unlock(vpu_dec->vpu_mutex);
                mutex_cnt = 0;         
                g_mutex_free(vpu_dec->vpu_mutex);
                vpu_dec->vpu_mutex = NULL;    
            }
            break;
        }   
        case GST_STATE_CHANGE_READY_TO_NULL:
	    {
            g_print("\n>>VPU_DEC: State: Ready to Null\n");
            if (vpudec_global_ptr == NULL)
                return GST_FLOW_OK;

            GST_MUTEX (">>VPU_DEC: Before mutex lock in Ready to NULL cnt=%d\n", mutex_cnt);
            if (vpu_dec->vpu_mutex) {
                vpu_mutex_lock(vpu_dec->vpu_mutex, FALSE);

                // sometimes there was error in opening and/or init so still need cleanup here since pipeline was never started
                mfw_gst_vpudec_cleanup(vpu_dec);
                /* Unlock the mutex to free the mutex 
                * in case of date terminated.
                */
                vpu_mutex_unlock(vpu_dec->vpu_mutex);
                mutex_cnt = 0;         
                g_mutex_free(vpu_dec->vpu_mutex);
                vpu_dec->vpu_mutex = NULL;    
            }
            if (vpu_dec->vpu_chipinit){
                g_print("vpu_uninit\n");
                
                vpu_UnInit();
            }
            vpu_dec->vpu_chipinit = FALSE;

            MM_DEINIT_DBG_MEM();
            break;
        }
        default:
            GST_DEBUG ("\n>>VPU_DEC: State unhandled next 0x%x\n", transition);
            break;
    }

    return ret;

}



/*=============================================================================
FUNCTION:           src_templ

DESCRIPTION:        Template to create a srcpad for the decoder

ARGUMENTS PASSED:   None

RETURN VALUE:       GstPadTemplate
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
/* table with framerates expressed as fractions */
static const gint fpss[][2] = { {24000, 1001},
{24, 1}, {25, 1}, {30000, 1001},
{30, 1}, {50, 1}, {60000, 1001},
{60, 1}, {0, 1}
};
static GstPadTemplate *src_templ(void)
{
    GstCaps *caps;
	GstStructure *structure;
	GValue list = {0}, fps = {0}	, fmt = {0};
	char *fmts[] = { "YV12", "I420", "Y42B", "NV12",  NULL };

	guint n;
	caps = gst_caps_new_simple("video/x-raw-yuv",
				   "format", GST_TYPE_FOURCC,
				   GST_MAKE_FOURCC('I', '4', '2', '0'),
				   "width", GST_TYPE_INT_RANGE, 16, 4096,
				   "height", GST_TYPE_INT_RANGE, 16, 4096,
				   NULL);

	structure = gst_caps_get_structure(caps, 0);
#if 0
	g_value_init(&list, GST_TYPE_LIST);
	g_value_init(&fps, GST_TYPE_FRACTION);
	for (n = 0; fpss[n][0] != 0; n++) {
	    gst_value_set_fraction(&fps, fpss[n][0], fpss[n][1]);
	    gst_value_list_append_value(&list, &fps);
	}
	gst_structure_set_value(structure, "framerate", &list);
	g_value_unset(&list);
	g_value_unset(&fps);
#endif
	g_value_init(&list, GST_TYPE_LIST);
	g_value_init(&fmt, GST_TYPE_FOURCC);
	for (n = 0; fmts[n] != NULL; n++) {
	    gst_value_set_fourcc(&fmt, GST_STR_FOURCC(fmts[n]));
	    gst_value_list_append_value(&list, &fmt);
	}
	gst_structure_set_value(structure, "format", &list);
	g_value_unset(&list);
	g_value_unset(&fmt);

	return(gst_pad_template_new("src", GST_PAD_SRC, GST_PAD_ALWAYS, caps));
}

/*======================================================================================
FUNCTION:           mfw_gst_vpudec_setcaps

DESCRIPTION:        This function negoatiates the caps set on the sink pad

ARGUMENTS PASSED:
                pad   -   pointer to the sinkpad of this element
                caps  -     pointer to the caps set

RETURN VALUE:
               TRUE         negotiation success full
               FALSE        negotiation Failed

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=======================================================================================*/
static gboolean mfw_gst_vpudec_setcaps(GstPad *pad, GstCaps *caps)
{
    MfwGstVPU_Dec *vpu_dec = MFW_GST_VPU_DEC(gst_pad_get_parent(pad));
    GstStructure *structure = gst_caps_get_structure(caps, 0);
    const gchar *mime = gst_structure_get_name(structure);
    GValue *codec_data_buf = NULL;
    gint i=0;
    gint wmvversion,wmvprofile;

    vpu_dec->mp4Class = MP4_MPEG4; /* Set the default value. */ 

    if(strcmp(mime,"video/x-h264") == 0)
        vpu_dec->codec = STD_AVC;
    else if(strcmp(mime,"video/mpeg") == 0)
    {
        gint mpegversion;
        gst_structure_get_int(structure, "mpegversion", &mpegversion);
        if(mpegversion == 4)            
            vpu_dec->codec = STD_MPEG4;        
        else            
            vpu_dec->codec = STD_MPEG2;
        
        vpu_dec->mp4Class = MP4_MPEG4;
    } else if (strcmp(mime,"video/mpegts") == 0) 
        vpu_dec->codec = STD_MPEG2;
    else if(strcmp(mime,"video/x-h263") == 0)
        vpu_dec->codec = STD_H263;
    else if(strcmp(mime,"video/x-wmv") == 0)
        vpu_dec->codec = STD_VC1;
    else if(strcmp(mime,"video/mp2v") == 0)
        vpu_dec->codec = STD_MPEG2;

#if defined (REALMEDIA) && defined (VPU_MX51)
    else if(strcmp(mime,"video/x-pn-realvideo") == 0)
        vpu_dec->codec = STD_RV;
#endif
#ifdef DIVX
    else if ((strcmp(mime,"video/x-divx") == 0) || 
             (strcmp(mime,"video/x-xvid") == 0))
    {        
        gint divx_version;        
        gst_structure_get_int(structure, "divxversion", &divx_version);
        if(divx_version == 3)            
            vpu_dec->codec = STD_DIV3;
                    
        else            
            vpu_dec->codec = STD_MPEG4;
        if(divx_version >= 5)
            vpu_dec->mp4Class = MP4_DIVX5_HIGHER;
        else
            vpu_dec->mp4Class = MP4_DIVX4;            
    }
#endif
#if defined (VPU_MX51)
    else if(strcmp(mime,"image/jpeg") == 0)
        vpu_dec->codec = STD_MJPG;
#endif
    else {
        GST_ERROR(">>VPU_DEC: Codec Standard not supporded \n");
        return FALSE;
    }

    gst_structure_get_fraction(structure, "framerate", &vpu_dec->frame_rate_nu,
			                   &vpu_dec->frame_rate_de);
    if (vpu_dec->frame_rate_de == 0) {
	    vpu_dec->frame_rate_nu =  DEFAULT_FRAME_RATE_NUMERATOR;
        vpu_dec->frame_rate_de =  1;
        vpu_dec->frame_rate = DEFAULT_FRAME_RATE_NUMERATOR;
        GST_DEBUG(">>VPU_DEC: not set framerate nu %d de %d\n", vpu_dec->frame_rate_nu, vpu_dec->frame_rate_de);
    }
    else {
        vpu_dec->frame_rate = (gfloat) (vpu_dec->frame_rate_nu) / vpu_dec->frame_rate_de;
        GST_DEBUG(">>VPU_DEC: set framerate nu %d de %d\n", vpu_dec->frame_rate_nu, vpu_dec->frame_rate_de);
    }
    if (vpu_dec->frame_rate_de > 0)
         vpu_dec->time_per_frame = gst_util_uint64_scale_int (GST_SECOND, vpu_dec->frame_rate_de, vpu_dec->frame_rate_nu);

    vpu_dec->set_ts_manually = FALSE;
    vpu_dec->nal_check = TRUE;

    GST_DEBUG(">>VPU_DEC: Time per frame %d\n", (guint) vpu_dec->time_per_frame);


    GST_DEBUG(">>VPU_DEC: Frame Rate = %d \n", (guint)vpu_dec->frame_rate);
    gst_structure_get_int(structure, "width", &vpu_dec->picWidth);
    GST_DEBUG(">>VPU_DEC: Input Width is %d\n", vpu_dec->picWidth);
    gst_structure_get_int(structure, "height", &vpu_dec->picHeight);
    GST_DEBUG(">>VPU_DEC: Input Height is %d\n", vpu_dec->picHeight);

    // if resolution is not set then most likely we are getting input from an RTP
    // depayloader which does not give our first frame as I frame.  Even if we wait
    // for first I frame we might display garbage video
    if ((vpu_dec->picHeight == 0) && (vpu_dec->picWidth == 0))
    {
        vpu_dec->accumulate_hdr = TRUE;
        vpu_dec->check_for_iframe = TRUE;

        //vpu_dec->min_latency = TRUE; // rtp depayers give strange timestamps        
        if (vpu_dec->loopback == FALSE)
            vpu_dec->parser_input=FALSE; // no parser input
        vpu_dec->set_ts_manually = TRUE;
        GST_DEBUG(">>VPU_DEC: Assuming no parser input so using streaming mode \n");
    } else {
        vpu_dec->check_for_iframe = FALSE;
        if (vpu_dec->codec != STD_MPEG2)
        {
            GST_DEBUG(">>VPU_DEC: Assuming parser input using file play mode \n");
        }
    }

    if(vpu_dec->codec==STD_VC1)
    {
        gst_structure_get_int(structure, "wmvversion", &wmvversion);
        if(wmvversion !=3)
        {
            mfw_gst_vpudec_post_fatal_error_msg(vpu_dec,
                "WMV Version error:This is a VC1 decoder supports "
                "only WMV 9 Simple,Main and Advance Profile decode (WMV3 or WVC1)");
            gst_object_unref(vpu_dec);   
            return FALSE;
        }
        gst_structure_get_int(structure, "wmvprofile", &wmvprofile);
		    vpu_dec->codec_subtype = 0;
		    if(wmvprofile==2)
			      vpu_dec->codec_subtype = 1;
  
        codec_data_buf = (GValue *) gst_structure_get_value(structure, "codec_data");
        
        if (NULL != codec_data_buf) {
            vpu_dec->codec_data = gst_value_get_buffer(codec_data_buf);
            vpu_dec->codec_data_len = GST_BUFFER_SIZE(vpu_dec->codec_data);
            GST_DEBUG("\n>>VPU_DEC: VC1 Codec specific data length is %d\n",
                        vpu_dec->codec_data_len);
        }
        else
        {
            GST_ERROR(">>VPU_DEC: No Header Extension Data found during Caps Negotiation \n");
            mfw_gst_vpudec_post_fatal_error_msg(vpu_dec,
                "No Extension Header Data Received from the Demuxer");
            gst_object_unref(vpu_dec);   
            return FALSE;
        }
    }
    if (vpu_dec->codec==STD_MPEG4)
    {
        codec_data_buf = (GValue *) gst_structure_get_value(structure, "codec_data");        
        if (NULL != codec_data_buf) {
            vpu_dec->codec_data = gst_value_get_buffer(codec_data_buf);
            vpu_dec->codec_data_len = GST_BUFFER_SIZE(vpu_dec->codec_data);
            GST_DEBUG("\n>>VPU_DEC: MPEG4 Codec specific data length is %d\n",vpu_dec->codec_data_len);
        }
    } 
    if (vpu_dec->codec==STD_AVC)
    {

        codec_data_buf = (GValue *) gst_structure_get_value(structure, "codec_data");
        if (NULL != codec_data_buf) {
            guint8 *hdrextdata;
            vpu_dec->codec_data = gst_value_get_buffer(codec_data_buf);
            GST_DEBUG ("H.264 SET CAPS check for codec data \n");
            vpu_dec->codec_data_len = GST_BUFFER_SIZE(vpu_dec->codec_data);
            GST_DEBUG("\n>>VPU_DEC: AVC Codec specific data length is %d\n",vpu_dec->codec_data_len);
            GST_DEBUG("AVC codec data is \n");
            hdrextdata = GST_BUFFER_DATA(vpu_dec->codec_data);
            for(i=0;i<vpu_dec->codec_data_len;i++)
                GST_DEBUG("%x ",hdrextdata[i]);
            GST_DEBUG("\n");

        }

    }
    gst_object_unref(vpu_dec);
    return TRUE;
    //return gst_pad_set_caps(pad, caps);
}

/*======================================================================================
FUNCTION:           mfw_gst_vpudec_base_init

DESCRIPTION:        Element details are registered with the plugin during
                    _base_init ,This function will initialise the class and child
                    class properties during each new child class creation

ARGUMENTS PASSED:   klass - void pointer

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=======================================================================================*/
#ifdef VPU_MX51
#define MFW_GST_VPUDEC_VIDEO_CAPS \
    "video/mpeg, " \
    "width = (int) [16, 1920], " \
    "height = (int) [16, 1080], " \
    "mpegversion = (int) 4; " \
    \
    "video/mpeg, " \
    "width = (int) [16, 1920], " \
    "height = (int) [16, 1080], " \
    "mpegversion = (int) {1,2}, " \
    "systemstream = (boolean) false;" \
    \
    "video/mpegts, " \
    "width = (int) [16, 1920], " \
    "height = (int) [16, 1080], " \
    "systemstream = (boolean) false;" \
    \
    "video/x-h263, " \
    "width = (int) [16, 1920], " \
    "height = (int)[16, 1080]; " \
    \
    "video/x-h264, " \
    "width = (int) [16, 1920], " \
    "height = (int)[16, 1080]; " \
     \
    "video/x-wmv, " \
    "wmvversion = (int) 3, " \
    "width = (int) [16, 1920], " \
    "height = (int)[16, 1080]; " \
    \
    "video/mp2v, "  \
    "mpegversion = (int) [1,2] , "  \
    "systemstream = (boolean) false;"\
    \
    "image/jpeg, "  \
    "width = (int) [16, 1920], " \
    "height = (int)[16, 1080]" 

#ifdef DIVX
#define DIVXCAPS \
    "video/x-divx, " \
    "divxversion = (int) {3, 4, 5, 6}, " \
    "width = (int) [16, 1920], " \
    "height = (int)[16, 1080] ;" \
    \
    "video/x-xvid, " \
    "divxversion = (int) {3, 4, 5, 6}, " \
    "width = (int) [16, 1920], " \
    "height = (int)[16, 1080] ;"

#else
#define DIVXCAPS ""
#endif

#if defined (REALMEDIA) && defined (VPU_MX51)
#define RVCAPS \
    "video/x-pn-realvideo;"
#else
#define RVCAPS ""
#endif

#elif defined(VPU_MX37)
#define MFW_GST_VPUDEC_VIDEO_CAPS \
    "video/mpeg, " \
    "width = (int) [16, 1280], " \
    "height = (int) [16, 720], " \
    "mpegversion = (int) 4; " \
    \
    "video/mpeg, " \
    "width = (int) [16, 1280], " \
    "height = (int) [16, 720], " \
    "mpegversion = (int) {1,2}, " \
    "systemstream = (boolean) false;" \
    \
    "video/mpegts, " \
    "width = (int) [16, 1280], " \
    "height = (int) [16, 720], " \
    "systemstream = (boolean) false;"   \
    \
    "video/x-h263, " \
    "width = (int) [16, 1280], " \
    "height = (int)[16, 720]; " \
    \
    "video/x-h264, " \
    "width = (int) [16, 1280], " \
    "height = (int)[16, 720]; " \
     \
    "video/x-wmv, " \
    "wmvversion = (int) 3, " \
    "width = (int) [16, 1280], " \
    "height = (int)[16, 720]; " \
    \
    "video/mp2v, "  \
    "mpegversion = (int) [1,2] , "  \
    "systemstream = (boolean) false"

#ifdef DIVX
#define DIVXCAPS \
    "video/x-divx, " \
    "divxversion = (int) {3, 4, 5, 6}, " \
    "width = (int) [16, 1280], " \
    "height = (int)[16, 720] ;" \
    \
    "video/x-xvid, " \
    "divxversion = (int) {3, 4, 5, 6}, " \
    "width = (int) [16, 1280], " \
    "height = (int)[16, 720] ;"
#else
#define DIVXCAPS ""
#endif
#define RVCAPS ""

#else  // everything like mx27

#define DIVXCAPS ""  // no divx support
#define RVCAPS ""  // no real media support

#define MFW_GST_VPUDEC_VIDEO_CAPS \
    "video/mpeg, " \
    "width = (int) [16, 720], " \
    "height = (int) [16, 576], " \
    "mpegversion = (int) 4; " \
    \
    "video/x-h263, " \
    "width = (int) [16, 720]," \
    "height = (int)[16, 576]; " \
    \
    "video/x-h264, " \
    "width = (int) [16, 720], " \
    "height = (int)[16, 576]" 
#endif

/* defines sink pad  properties of the VPU Decoder element */
static GstStaticPadTemplate mfw_gst_vpudec_sink_factory =
    GST_STATIC_PAD_TEMPLATE("sink",
			GST_PAD_SINK,
			GST_PAD_ALWAYS,
			GST_STATIC_CAPS(DIVXCAPS RVCAPS MFW_GST_VPUDEC_VIDEO_CAPS));

/* get the element details */
#ifdef VPU_MX27
static GstElementDetails mfw_gst_vpudec_details =
    GST_ELEMENT_DETAILS("Freescale: Hardware (VPU) Decoder",
		    "Codec/Decoder/Video",
		    "Decodes H.264, MPEG4, H263 "
            "Elementary data into YUV 4:2:0 data",
		    FSL_GST_MM_PLUGIN_AUTHOR);
#else
static GstElementDetails mfw_gst_vpudec_details =
    GST_ELEMENT_DETAILS("Freescale: Hardware (VPU) Decoder",
		    "Codec/Decoder/Video",
		    "Decodes H.264, MPEG4, H263, and VC-1 "
            "Elementary data into YUV 4:2:0 data",
		    FSL_GST_MM_PLUGIN_AUTHOR);
#endif

static void mfw_gst_vpudec_base_init(MfwGstVPU_DecClass *klass)
{

    GstElementClass *element_class = GST_ELEMENT_CLASS(klass);

    gst_element_class_add_pad_template(element_class, src_templ());

    gst_element_class_add_pad_template(element_class,
				       gst_static_pad_template_get
				       (&mfw_gst_vpudec_sink_factory));

    gst_element_class_set_details(element_class, &mfw_gst_vpudec_details);

}

/*======================================================================================
FUNCTION:           mfw_gst_vpudec_codec_get_type

DESCRIPTION:        Gets an enumeration for the different 
                    codec standars supported by the decoder

ARGUMENTS PASSED:   None

RETURN VALUE:       enumerated type of the codec standards 
                    supported by the decoder

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
========================================================================================*/
GType mfw_gst_vpudec_codec_get_type(void)
{
#if defined (VPU_MX51)  
  static GEnumValue vpudec_codecs[] = {
    {STD_MPEG4, STR(STD_MPEG4), "std_mpeg4"},
    {STD_H263,  STR(STD_H263), "std_h263"},
    {STD_AVC,   STR(STD_AVC), "std_avc" },
    {STD_VC1,   STR(STD_VC1), "std_vc1"},
    {STD_MPEG2,  STR(STD_MPEG2), "std_mpeg2"},
    {STD_MJPG,  STR(STD_MJPG), "std_mjpg"},
    {0, NULL, NULL},
  };
#elif defined (VPU_MX37)
  static GEnumValue vpudec_codecs[] = {
    {STD_MPEG4, STR(STD_MPEG4), "std_mpeg4"},
    {STD_H263,  STR(STD_H263), "std_h263"},
    {STD_AVC,   STR(STD_AVC), "std_avc" },
    {STD_MPEG2,  STR(STD_MPEG2), "std_mpeg2"},
    {STD_VC1,   STR(STD_VC1), "std_vc1"},
    {0, NULL, NULL},
  };
#else
    static GEnumValue vpudec_codecs[] = {
    {STD_MPEG4, STR(STD_MPEG4), "std_mpeg4"},
    {STD_H263,  STR(STD_H263), "std_h263"},
    {STD_AVC,   STR(STD_AVC), "std_avc" },
    {0, NULL, NULL},
  };
#endif
  return (g_enum_register_static ("MfwGstVpuDecCodecs", vpudec_codecs));
}

/*======================================================================================
FUNCTION:           mfw_gst_vpudec_base_finalize

DESCRIPTION:        Base finalized

ARGUMENTS PASSED:   object     - pointer to the elements object

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
========================================================================================*/
static void mfw_gst_vpudec_base_finalize(MfwGstVPU_DecClass *klass)
{
   GST_DEBUG(">>VPU_DEC: base finalized.\n");
}

/*======================================================================================
FUNCTION:           mfw_gst_vpudec_finalize

DESCRIPTION:        Class finalized

ARGUMENTS PASSED:   object     - pointer to the elements object

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
========================================================================================*/
static void mfw_gst_vpudec_finalize (GObject *object)
{
    GST_DEBUG(">>VPU_DEC: class finalized.\n");
}

/*======================================================================================
FUNCTION:           mfw_gst_vpudec_mirror_get_type

DESCRIPTION:        Gets an enumeration for mirror directions

ARGUMENTS PASSED:   None

RETURN VALUE:       Enumerated type of the mirror directions supported by VPU

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
========================================================================================*/
GType mfw_gst_vpudec_mirror_get_type(void)
{
  static GEnumValue vpudec_mirror[] = {
    {MIRDIR_NONE,    STR(MIRDIR_NONE),   "none"},
    {MIRDIR_VER,     STR(MIRDIR_VER),    "ver"},
    {MIRDIR_HOR,     STR(MIRDIR_HOR),    "hor" },
    {MIRDIR_HOR_VER, STR(MIRDIR_HOR_VER),"hor_ver"},
    {0, NULL, NULL},
  };
  return (g_enum_register_static ("MfwGstVpuDecMirror", vpudec_mirror));
}

/*======================================================================================
FUNCTION:           mfw_gst_vpudec_class_init

DESCRIPTION:        Initialise the class.(specifying what signals,
                    arguments and virtual functions the class has and setting up
                    global states)

ARGUMENTS PASSED:
                klass - pointer to H.264Decoder element class

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=======================================================================================*/
static void mfw_gst_vpudec_class_init(MfwGstVPU_DecClass *klass)
{
    GObjectClass *gobject_class         = NULL;
    GstElementClass *gstelement_class   = NULL;
    gobject_class                       = (GObjectClass *) klass;
    gstelement_class                    = (GstElementClass *) klass;
    gstelement_class->change_state      = mfw_gst_vpudec_change_state;
    gobject_class->set_property = mfw_gst_vpudec_set_property;
    gobject_class->get_property = mfw_gst_vpudec_get_property;
    gobject_class->finalize     = mfw_gst_vpudec_finalize;
    gobject_class->dispose      = mfw_gst_vpudec_finalize;

    g_object_class_install_property(gobject_class, MFW_GST_VPU_PROF_ENABLE,
				    g_param_spec_boolean("profiling", "Profiling", 
                    "enable time profiling of the vpu decoder plug-in",
                    FALSE, G_PARAM_READWRITE));

    g_object_class_install_property(gobject_class, MFW_GST_VPU_CODEC_TYPE,
				    g_param_spec_enum("codec-type", "codec_type", 
                    "selects the codec type for decoding",
                    MFW_GST_TYPE_VPU_DEC_CODEC, STD_AVC, G_PARAM_READWRITE));

    g_object_class_install_property(gobject_class, MFW_GST_VPU_LOOPBACK,
				    g_param_spec_boolean("loopback", "LoopBack", 
                    "enables the decoder plug-in to operate"
                    "in loopback mode with encoder ",
                    FALSE, G_PARAM_READWRITE));

    g_object_class_install_property(gobject_class, MFW_GST_VPU_ROTATION,
				    g_param_spec_uint("rotation", "Rotation",
                    "Rotation Angle should be 0, 90, 180 or 270.",
                    0, 270, 0, G_PARAM_READWRITE));

    g_object_class_install_property(gobject_class, MFW_GST_VPU_MIRROR,
				    g_param_spec_enum("mirror-dir", "mirror_dir", 
                    "specifies mirror direction",
                    MFW_GST_TYPE_VPU_DEC_MIRROR, MIRDIR_NONE, G_PARAM_READWRITE));

#ifndef VPU_MX27
    g_object_class_install_property(gobject_class, MFW_GST_VPU_DBK_ENABLE,
				    g_param_spec_boolean("dbkenable", "dbkenable", 
                    "enables the decoder plug-in deblock",
                    FALSE, G_PARAM_READWRITE));
					
    g_object_class_install_property(gobject_class, MFW_GST_VPU_DBK_OFFSETA,
				    g_param_spec_int("dbk-offseta", "dbk_offseta", 
                    "set the deblock offset a",
					G_MININT, G_MAXINT, 5, 
                    G_PARAM_READWRITE));
					
    g_object_class_install_property(gobject_class, MFW_GST_VPU_DBK_OFFSETB,
				    g_param_spec_int("dbk-offsetb", "dbk_offsetb", 
                    "set the deblock offset b",
					G_MININT, G_MAXINT, 5, 
                    G_PARAM_READWRITE));
#endif
	
    g_object_class_install_property(gobject_class, MFW_GST_VPU_FORCE_NONDR,
				    g_param_spec_boolean("force-nondr", "force_nondr", 
                    "Force to use the direct render mode or not."
                    "true:Non-DR mode, false: auto mode",
                    FALSE, G_PARAM_READWRITE));	

    g_object_class_install_property(gobject_class, MFW_GST_VPU_PASSTHRU,
				    g_param_spec_boolean("passthru", "passthru", 
                    "No decode but passes data to be sent to VPU."
                    "true:passthru, false: no pass thru - full decode",
                    FALSE, G_PARAM_READWRITE));	

    g_object_class_install_property(gobject_class, MFW_GST_VPU_LATENCY,
				    g_param_spec_boolean("min_latency", "minimum latency", 
                    "Minimizes latency through plugin"
                    "true: minimize latency, false: full path",
                    FALSE, G_PARAM_READWRITE));	

    g_object_class_install_property(gobject_class, MFW_GST_VPU_PARSER,
				    g_param_spec_boolean("parser", "parser providing input in frame boundaries", 
                    "parser on is default - used to turn off when playing elementary video files or streaming "
                    "true: parser input , false: no parser - playing from elementary ",
                    FALSE, G_PARAM_READWRITE));	

    g_object_class_install_property(gobject_class, MFW_GST_VPU_FRAMEDROP,
				    g_param_spec_boolean("framedrop", "enable frame dropping before decode default is on", 
                    "Used to disable frame dropping before decode which might affect a/v sync "
                    "true: frame drop , false: disable frame dropping",
                    FALSE, G_PARAM_READWRITE));	

#if (defined (VPU_MX37) || defined (VPU_MX51))
    g_object_class_install_property(gobject_class, MFW_GST_VPU_OUTPUT_FMT,
				    g_param_spec_int("fmt", "output_fmt", 
                    "set the format of output(0 for NV12, 1 for I420",
					0, 1, 0, 
                    G_PARAM_READWRITE));
#endif
}


/*======================================================================================
FUNCTION:           mfw_gst_vpudec_init

DESCRIPTION:        Create the pad template that has been registered with the
                    element class in the _base_init

ARGUMENTS PASSED:
                vpu_dec -    pointer to vpu_decoder element structure

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=======================================================================================*/
static void mfw_gst_vpudec_init(MfwGstVPU_Dec *vpu_dec, MfwGstVPU_DecClass *gclass)
{
    GstElementClass *klass = GST_ELEMENT_GET_CLASS(vpu_dec);

    vpudec_global_ptr = vpu_dec;

    /* create the sink and src pads */
    vpu_dec->sinkpad = gst_pad_new_from_template(gst_element_class_get_pad_template
        (klass, "sink"), "sink");
    vpu_dec->srcpad = gst_pad_new_from_template(src_templ(), "src");
    gst_element_add_pad(GST_ELEMENT(vpu_dec), vpu_dec->sinkpad);
    gst_element_add_pad(GST_ELEMENT(vpu_dec), vpu_dec->srcpad);
    vpu_dec->parent_class = g_type_class_peek_parent(gclass);
    gst_pad_set_chain_function(vpu_dec->sinkpad, mfw_gst_vpudec_chain);
    
    gst_pad_set_setcaps_function(vpu_dec->sinkpad,mfw_gst_vpudec_setcaps);
    gst_pad_set_event_function(vpu_dec->sinkpad, GST_DEBUG_FUNCPTR(mfw_gst_vpudec_sink_event));

    vpu_dec->rotation_angle = 0;
    vpu_dec->mirror_dir = MIRDIR_NONE;
    vpu_dec->codec = STD_AVC;
    vpu_dec->loopback = FALSE;
    vpu_dec->vpu_chipinit = FALSE;

    vpu_dec->vpu_mutex = g_mutex_new();
    vpu_dec->lastframedropped = FALSE;
    vpu_dec->frame_drop_allowed = FALSE;
    vpu_dec->parser_input = TRUE;
    vpu_dec->predict_gop = FALSE;
    vpu_dec->min_latency=FALSE;
    vpu_dec->is_frame_started = FALSE;
    vpu_dec->dbk_enabled = FALSE;
#if (defined (VPU_MX37) || defined (VPU_MX51))
    vpu_dec->fmt = 1; /* Default I420 */
#endif

#ifdef VPU_PARALLELIZATION
    vpu_dec->allow_parallelization = TRUE;
#else
    vpu_dec->allow_parallelization = FALSE;
#endif
    vpu_dec->dbk_offset_a = vpu_dec->dbk_offset_b = DEFAULT_DBK_OFFSET_VALUE;
    vpu_dec->field = FIELD_NONE;
    vpu_dec->in_cleanup = FALSE;
}

/*======================================================================================
FUNCTION:           plugin_init

DESCRIPTION:        Special function , which is called as soon as the plugin or
                    element is loaded and information returned by this function
                    will be cached in central registry

ARGUMENTS PASSED:
		        plugin - pointer to container that contains features loaded
                        from shared object module

RETURN VALUE:
		        return TRUE or FALSE depending on whether it loaded initialized any
                dependency correctly

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=======================================================================================*/
static gboolean plugin_init(GstPlugin *plugin)
{
    return gst_element_register(plugin, "mfw_vpudecoder",
				FSL_GST_RANK_HIGH, MFW_GST_TYPE_VPU_DEC);
}

GST_PLUGIN_DEFINE (
          GST_VERSION_MAJOR,	                    /* major version of gstreamer */
		  GST_VERSION_MINOR,	                    /* minor version of gstreamer */
		  "mfw_vpudecoder",	                        /* name of our  plugin */
		  "Decodes H264, MPEG4 and H263 Encoded"
                  "data to Raw YUV Data ",                   /* what our plugin actually does */
		  plugin_init,	                            /* first function to be called */
		  VERSION,
		  GST_LICENSE_UNKNOWN,
		  FSL_GST_MM_PLUGIN_PACKAGE_NAME, FSL_GST_MM_PLUGIN_PACKAGE_ORIG)

/*======================================================================================
FUNCTION:           mfw_gst_type_vpu_dec_get_type

DESCRIPTION:        Interfaces are initiated in this function.you can register one
                    or more interfaces after having registered the type itself.

ARGUMENTS PASSED:   None

RETURN VALUE:       A numerical value ,which represents the unique identifier 
                    of this element.

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
========================================================================================*/
GType mfw_gst_type_vpu_dec_get_type(void)
{
    static GType vpu_dec_type = 0;
    if (!vpu_dec_type) 
    {
	static const GTypeInfo vpu_dec_info = 
        {
	        sizeof(MfwGstVPU_DecClass),
	        (GBaseInitFunc) mfw_gst_vpudec_base_init,
	        (GBaseFinalizeFunc) mfw_gst_vpudec_base_finalize,
	        (GClassInitFunc) mfw_gst_vpudec_class_init,
	        NULL,
	        NULL,
	        sizeof(MfwGstVPU_Dec),
	        0,
	        (GInstanceInitFunc) mfw_gst_vpudec_init,
	    };
	    vpu_dec_type = g_type_register_static(GST_TYPE_ELEMENT,
		    	                    "MfwGstVPU_Dec",
			                        &vpu_dec_info, 0);
    }
    GST_DEBUG_CATEGORY_INIT(mfw_gst_vpudec_debug,
			    "mfw_vpudecoder", 0,
			    "FreeScale's VPU  Decoder's Log");
    return vpu_dec_type;
}


/*======================================================================================
FUNCTION:           mfw_gst_vpudec_vpu_finalize

DESCRIPTION:        Handles cleanup of any unreleased memory if player is closed

ARGUMENTS PASSED:   None
RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
========================================================================================*/
void __attribute__ ((destructor)) mfw_gst_vpudec_vpu_finalize(void);

void mfw_gst_vpudec_vpu_finalize(void)
{
    RetCode vpu_ret=RETCODE_SUCCESS;
    MfwGstVPU_Dec *vpu_dec = vpudec_global_ptr;
    if (vpu_dec == NULL) {
       GST_DEBUG(">>VPU_DEC: vpu_dec is null,exit no clean up needed\n");
       return;
    }
    GST_DEBUG(">>VPU_DEC: Destructor - final cleanup \n");
    
    if (vpu_dec->vpu_mutex)
    {
        GST_MUTEX (">>VPU_DEC: Before cleanup mutex lock cnt=%d\n", mutex_cnt);
        vpu_mutex_lock(vpu_dec->vpu_mutex, FALSE);
        vpu_dec->in_cleanup = TRUE;
        mfw_gst_vpudec_cleanup(vpu_dec);
        /* Unlock the mutex to free the mutex 
        * in case of date terminated.
        */
         vpu_mutex_unlock(vpu_dec->vpu_mutex);
         mutex_cnt = 0;       
         g_mutex_free(vpu_dec->vpu_mutex);
         vpu_dec->vpu_mutex = NULL;    
    } else {
        GST_DEBUG(">>VPU_DEC: Cleanup already done before destructor time \n");
    }

    if (vpu_dec->vpu_chipinit)
         vpu_UnInit();

    vpu_dec->vpu_chipinit = FALSE;
    
    GST_DEBUG(">>VPU_DEC: vpu instance 0x%x destroyed.\n", vpudec_global_ptr);
    vpudec_global_ptr = NULL;
    return;

}
