/*
 * Copyright (C) 2008-2009 Freescale Semiconductor, Inc. All rights reserved.
 *
 */
 
/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
 
/*
 * Module Name:    mfw_gst_radec.c
 *
 * Description:    RealMedia audio decoder plugin for Gstreamer.
 *
 * Portability:    This code is written for Linux OS and Gstreamer
 */  
 
/*
 * Changelog: 
 * Sep 04 2008 Sario HU <b01138@freescale.com>
 * - Initial version
 *
 */


/*=============================================================================
                            INCLUDE FILES
=============================================================================*/




#include <gst/gst.h>
#include <string.h>

#include "gecko2codec.h"

#ifdef MEMORY_DEBUG
#include "mfw_gst_debug.h"
#endif

#include "mfw_gst_radec.h"

#include "mfw_gst_utils.h"

/*=============================================================================
                            LOCAL CONSTANTS
=============================================================================*/
#define RA_DECODER_FATAL_ERROR(...) g_print(RED_STR(__VA_ARGS__))

#define RA_DECODER_FLOW(...) g_print(BLUE_STR(__VA_ARGS__))
    
#ifndef RA_DECODER_FLOW
#define RA_DECODER_FLOW(...)
#endif

#ifdef MEMORY_DEBUG
#define RA_MALLOC(ra_dec, size)\
    dbg_malloc((&((ra_dec)->memmgr)),(size), "line" STR(__LINE__) "of" STR(__FUNCTION__) )
#define RA_FREE(ra_dec, ptr)\
    dbg_free(&((ra_dec)->memmgr), (ptr), "line" STR(__LINE__) "of" STR(__FUNCTION__) )
#else
#define RA_MALLOC(ra_dec, size)\
    g_malloc((size))
#define RA_FREE(ra_dec, ptr)\
    g_free((ptr))
#endif


#define MAX_FLAVOR 30

/*=============================================================================
                LOCAL TYPEDEFS (STRUCTURES, UNIONS, ENUMS)
=============================================================================*/

static GstStaticPadTemplate sink_factory = GST_STATIC_PAD_TEMPLATE("sink",
								   GST_PAD_SINK,
								   GST_PAD_ALWAYS,
								   GST_STATIC_CAPS
								   ("audio/x-pn-realaudio"));

static GstStaticPadTemplate src_factory = GST_STATIC_PAD_TEMPLATE("src",
								  GST_PAD_SRC,
								  GST_PAD_ALWAYS,
								  GST_STATIC_CAPS("audio/x-raw-int"));
								  

static FlavorParameter flavor_para[MAX_FLAVOR] = 
    {{0,256,1,12,256,8000,0,0},           
    {1,256,1,12,256,11025,0,0},           
    {2,512,1,18,376,22050,0,0},           
    {3,512,1,23,480,22050,0,0},           
    {4,1024,1,37,744,44100,0,0},       
    {5,1024,1,47,1024,44100,0,0},       
    {6,1024,1,47,1488,44100,0,0},       
    {7,512,1,24,744,22050,0,0},       
    {8,256,1,9,192,8000,0,0},       
    {9,256,2,11,464,11025,0,0},       
    {10,512,2,18,752,22050,0,0},   
    {11,512,2,24,1024,22050,0,0},   
    {12,1024,2,37,1488,44100,0,0},      
    {13,1024,2,47,2224,44100,0,0},      
    {14,1024,1,47,1488,44100,0,0},      
    {15,1024,1,47,480,44100,0,0},      
    {16,1024,1,47,744,44100,0,0},       
    {17,512,2,16,384,22050,1,3},       
    {18,512,2,20,480,22050,1,3},       
    {19,512,2,23,480,22050,1,3},       
    {20,512,2,24,744,22050,2,4},        
    {21,1024,2,32,744,44100,2,4},       
    {22,1024,2,32,1024,44100,5,5},      
    {23,1024,2,37,1024,44100,2,4},      
    {24,1024,2,37,1488,44100,6,5},      
    {25,1024,2,37,2240,44100,8,5},      
    {26,256,2,9,288,11025,1,2},         
    {27,1024,2,30,1488,44100,17,5},     
    {28,1024,2,34,2240,44100,19,5},     
    {29,512,2,23,1024,22050,17,5}};      


/*=============================================================================
                                LOCAL MACROS
=============================================================================*/
#define RA_ALIGHMENT 4
#define	GST_CAT_DEFAULT    mfw_gst_radec_debug

/*=============================================================================
                        STATIC FUNCTION PROTOTYPES
=============================================================================*/
GST_DEBUG_CATEGORY_STATIC(mfw_gst_radec_debug);
static void mfw_gst_radec_class_init(MFW_GST_RADEC_CLASS_T *
					  klass);
static void mfw_gst_radec_base_init(MFW_GST_RADEC_CLASS_T *
					 klass);
static void mfw_gst_radec_init(MFW_GST_RADEC_INFO_T *
				    radec_info);
static void mfw_gst_radec_set_property(GObject * object,
					    guint prop_id,
					    const GValue * value,
					    GParamSpec * pspec);
static void mfw_gst_radec_get_property(GObject * object,
					    guint prop_id, GValue * value,
					    GParamSpec * pspec);
static gboolean mfw_gst_radec_set_caps(GstPad * pad, GstCaps * caps);
static GstFlowReturn mfw_gst_radec_chain(GstPad * pad,
					      GstBuffer * buf);
static gboolean mfw_gst_radec_sink_event(GstPad *, GstEvent *);
static gboolean mfw_gst_radec_src_event(GstPad *, GstEvent *);
static gboolean plugin_init(GstPlugin * plugin);
static gboolean mfw_gst_radec_seek(MFW_GST_RADEC_INFO_T *,
					GstPad *, GstEvent *);
static gboolean mfw_gst_radec_convert_src(GstPad * pad,
					       GstFormat src_format,
					       gint64 src_value,
					       GstFormat * dest_format,
					       gint64 * dest_value);
static gboolean mfw_gst_radec_convert_sink(GstPad * pad,
						GstFormat src_format,
						gint64 src_value,
						GstFormat * dest_format,
						gint64 * dest_value);
static gboolean mfw_gst_radec_src_query(GstPad *, GstQuery *);
static const GstQueryType *mfw_gst_radec_get_query_types(GstPad *
							      pad);
static gboolean mfw_gst_radec_mem_flush(MFW_GST_RADEC_INFO_T *);

/*=============================================================================
                            STATIC VARIABLES
=============================================================================*/
static GstElementClass *parent_class_ra = NULL;
static MFW_GST_RADEC_INFO_T *radec_global_ptr = NULL;

/*=============================================================================
                            LOCAL FUNCTIONS
=============================================================================*/

/*=============================================================================
FUNCTION: mfw_gst_radec_set_property

DESCRIPTION: sets the property of the element

ARGUMENTS PASSED:
        object     - pointer to the elements object
        prop_id    - ID of the property;
        value      - value of the property set by the application
        pspec      - pointer to the attributes of the property

RETURN VALUE:
        None

PRE-CONDITIONS:
        None

POST-CONDITIONS:


IMPORTANT NOTES:
        None
=============================================================================*/
static void
mfw_gst_radec_set_property(GObject * object, guint prop_id,
				const GValue * value, GParamSpec * pspec)
{

    GST_DEBUG(" in mfw_gst_radec_set_property routine \n");
    GST_DEBUG(" out of mfw_gst_radec_set_property routine \n");
}

/*=============================================================================
FUNCTION: mfw_gst_radec_set_property

DESCRIPTION: gets the property of the element

ARGUMENTS PASSED:
        object     - pointer to the elements object
        prop_id    - ID of the property;
        value      - value of the property got from the application
        pspec      - pointer to the attributes of the property

RETURN VALUE:
        None

PRE-CONDITIONS:
        None

POST-CONDITIONS:


IMPORTANT NOTES:
        None
=============================================================================*/
static void
mfw_gst_radec_get_property(GObject * object, guint prop_id,
				GValue * value, GParamSpec * pspec)
{
    GST_DEBUG(" in mfw_gst_radec_get_property routine \n");
    GST_DEBUG(" out of mfw_gst_radec_get_property routine \n");

}


static void * malloc_nbyte_aligned(MFW_GST_RADEC_INFO_T *radec_info, guint size, guint alignbytes)
{
    char * buf, *bufaligned;
    size += alignbytes;
    buf = RA_MALLOC(radec_info, size);
    if (buf==NULL) return buf;
    bufaligned = (char *)(((guint32)buf + alignbytes) & (~(alignbytes-1)));
    *(bufaligned-1)=(guint32)bufaligned-(guint32)buf;
    return bufaligned;
}


static void free_nbyte_aligned(MFW_GST_RADEC_INFO_T *radec_info, char * buf)
{
    char * buforiginal = (guint32)buf - *(buf-1);
    RA_FREE(radec_info, buforiginal);
}

static int
mfw_ra_dec_allocate_core_memory(MFW_GST_RADEC_INFO_T *radec_info)
{
    int i;
    Gecko2_Mem_Alloc_Info * meminfo = &radec_info->decoder_config->gecko2_mem_info;
    for (i=0;i<meminfo->num_reqs;i++){
        meminfo->mem_info_sub[i].app_base_ptr = 
            malloc_nbyte_aligned(radec_info, meminfo->mem_info_sub[i].size, RA_ALIGHMENT);
    }
}
static int
mfw_ra_dec_free_core_memory(MFW_GST_RADEC_INFO_T *radec_info)
{
    int i;
    Gecko2_Mem_Alloc_Info * meminfo = &radec_info->decoder_config->gecko2_mem_info;
    for (i=0;i<meminfo->num_reqs;i++){
        if (meminfo->mem_info_sub[i].app_base_ptr){
            free_nbyte_aligned(radec_info, meminfo->mem_info_sub[i].app_base_ptr);
            meminfo->mem_info_sub[i].app_base_ptr = NULL;
        }
    }
}

static void
mfw_ra_dec_core_cleanup(MFW_GST_RADEC_INFO_T *radec_info)
{
    if (radec_info->decoder_config){
        mfw_ra_dec_free_core_memory(radec_info);
        RA_FREE(radec_info, radec_info->decoder_config);
        radec_info->decoder_config = NULL;
    }
}


static GstFlowReturn
mfw_ra_dec_core_init(MFW_GST_RADEC_INFO_T *radec_info)
{
    GstFlowReturn ret = GST_FLOW_OK;
    
    int core_ret;

    radec_info->decoder_config = RA_MALLOC(radec_info, sizeof(Gecko2_Decoder_Config));
    
    if (radec_info->decoder_config==NULL){
        ret = GST_FLOW_ERROR;
        GST_ERROR("Error in mfw_ra_dec_core_init, can not allocate memory\n");
        return ret;
    } 

    radec_info->opt = &flavor_para[radec_info->flavorindex];
    
    core_ret = Gecko2QueryDecMem(
                radec_info->opt->nSamples,
                radec_info->opt->nChannels,
                radec_info->opt->nRegions,
                radec_info->opt->nFrameBits,
                radec_info->opt->sampRate,
                radec_info->opt->cplStart, 
                radec_info->opt->cplQbits, 
                radec_info->decoder_config); 
    
    if (core_ret){
        ret = GST_FLOW_ERROR;
        RA_FREE(radec_info, radec_info->decoder_config);
        radec_info->decoder_config = NULL;
        GST_ERROR("Error in Gecko2QueryDecMem, ret = %d\n", core_ret);
        return ret;
    }

    mfw_ra_dec_allocate_core_memory(radec_info);
    
    core_ret = Gecko2InitDecoder(
                radec_info->decoder_config,
                radec_info->opt->nSamples,
                radec_info->opt->nChannels,
                radec_info->opt->nRegions,
                radec_info->opt->nFrameBits,
                radec_info->opt->sampRate,
                radec_info->opt->cplStart, 
                radec_info->opt->cplQbits, 
                &radec_info->codedelay); 
    
    if (core_ret){
        ret = GST_FLOW_ERROR;
        RA_FREE(radec_info, radec_info->decoder_config);
        radec_info->decoder_config = NULL;
        GST_ERROR("Error in Gecko2InitDecoder, ret = %d\n", core_ret);
    }
    return ret;
}

/***************************************************************************
*
*   FUNCTION NAME - mfw_gst_radec_data
*
*   DESCRIPTION
*                   This function decodes data in the input buffer and 
*                   pushes the decode pcm output to the next element in the 
*                   pipeline
*   ARGUMENTS
*       radec_info    - pointer to the plugin context
*       inbuffsize         - pointer to the input buffer size
*
*   RETURN VALUE
*       TRUE               - decoding is succesful
*       FALSE              - error in decoding
***************************************************************************/
static GstFlowReturn
mfw_gst_radec_data(MFW_GST_RADEC_INFO_T *radec_info, GstBuffer * gstbuf)
{
    GstFlowReturn ret = GST_FLOW_OK;
    GstCaps *caps = NULL;
    GstBuffer *outbuffer;
    short * outbuf;
    char * inbuf;
    int result;
    guint64 time_duration;
    gint frame_num;
    gint frame_bytes;
    
    if (!radec_info->caps_set) {
            radec_info->caps_set = TRUE;
            caps = gst_caps_new_simple("audio/x-raw-int",
                        "endianness", G_TYPE_INT,
                        G_BYTE_ORDER, "signed",
                        G_TYPE_BOOLEAN, TRUE, "width",
                        G_TYPE_INT, 16, "depth", G_TYPE_INT,
                        16, "rate", G_TYPE_INT,
                        radec_info->opt->sampRate,
                        "channels", G_TYPE_INT, 
                        radec_info->opt->nChannels, NULL);
            gst_pad_set_caps(radec_info->srcpad, caps);
            gst_caps_unref(caps);
    }

    frame_bytes = radec_info->opt->nFrameBits>>3;
    frame_num = GST_BUFFER_SIZE(gstbuf)/frame_bytes;
    inbuf = GST_BUFFER_DATA(gstbuf);
   // RA_DECODER_FLOW("audio tm %" GST_TIME_FORMAT "\n", 
  //                          GST_TIME_ARGS(GST_BUFFER_TIMESTAMP(gstbuf)));
    while (frame_num-->0){
        caps = GST_PAD_CAPS(radec_info->srcpad);

        ret = gst_pad_alloc_buffer_and_set_caps(radec_info->srcpad,
                                                0,
                                                radec_info->opt->nSamples * radec_info->opt->nChannels * sizeof(short),
                                                caps, &outbuffer);

        if (ret != GST_FLOW_OK){
            break;
        }

        outbuf = GST_BUFFER_DATA(outbuffer);

        result = Gecko2Decode(radec_info->decoder_config, inbuf,0, outbuf);

        if (radec_info->codedelay>0){
            radec_info->codedelay--;
            gst_buffer_unref(outbuffer);
            continue;
        }
        
        time_duration = gst_util_uint64_scale_int(radec_info->opt->nSamples , GST_SECOND, radec_info->opt->sampRate);

        GST_BUFFER_DURATION(outbuffer) = time_duration;
        GST_BUFFER_TIMESTAMP(outbuffer) = radec_info->time_offset;
        
        radec_info->time_offset += time_duration; 

        ret = gst_pad_push(radec_info->srcpad, outbuffer);
        if (ret != GST_FLOW_OK){
            break;
        }
       
        inbuf += frame_bytes;
    }
    return ret;
}


/*=============================================================================
FUNCTION: mfw_gst_radec_chain

DESCRIPTION: Initializing the decoder and calling the actual decoding function

ARGUMENTS PASSED:
        pad     - pointer to pad
        buffer  - pointer to received buffer

RETURN VALUE:
        GST_FLOW_OK		- Frame decoded successfully
		GST_FLOW_ERROR	- Failure

PRE-CONDITIONS:
        None

POST-CONDITIONS:


IMPORTANT NOTES:
        None
=============================================================================*/
static GstFlowReturn
mfw_gst_radec_chain(GstPad * pad, GstBuffer * buf)
{

    GstFlowReturn ret;
    MFW_GST_RADEC_INFO_T *radec_info;
    
    radec_info = MFW_GST_RADEC(GST_OBJECT_PARENT(pad));

    if (radec_info->demo_mode == 2)
        return GST_FLOW_ERROR;

    if (radec_global_ptr == NULL){
	    radec_global_ptr = radec_info;
    }else {
    	if (radec_global_ptr != radec_info) {
    	    ret = gst_pad_push(radec_info->srcpad, buf);
    	    if (ret != GST_FLOW_OK) {
    		    GST_ERROR("ra dec:could not push onto next element %d\n", ret);
    	    }
    	    return GST_FLOW_OK;
    	}
    }

    if (!radec_info->init_done) {
        ret = mfw_ra_dec_core_init(radec_info);
        if (ret!=GST_FLOW_OK){
            RA_DECODER_FATAL_ERROR("mfw_ra_dec_core_init failed with result = %d\n", ret);
            gst_buffer_unref(buf);
            return ret;
        }
        radec_info->init_done = TRUE;
    }

    ret = mfw_gst_radec_data(radec_info, buf);
    if (ret!=GST_FLOW_OK){
        RA_DECODER_FATAL_ERROR("mfw_gst_radec_data failed with result = %d\n", ret);
    }
    gst_buffer_unref(buf);
    
    return ret;
}

/*=============================================================================
FUNCTION:   mfw_gst_radec_change_state

DESCRIPTION: this function keeps track of different states of pipeline.

ARGUMENTS PASSED:
        element     -   pointer to element
        transition  -   state of the pipeline

RETURN VALUE:
        GST_STATE_CHANGE_FAILURE    - the state change failed
        GST_STATE_CHANGE_SUCCESS    - the state change succeeded
        GST_STATE_CHANGE_ASYNC      - the state change will happen
                                        asynchronously
        GST_STATE_CHANGE_NO_PREROLL - the state change cannot be prerolled

PRE-CONDITIONS:
        None

POST-CONDITIONS:
        None

IMPORTANT NOTES:
        None
=============================================================================*/
static GstStateChangeReturn
mfw_gst_radec_change_state(GstElement * element,
				GstStateChange transition)
{
    MFW_GST_RADEC_INFO_T *radec_info;
    GstStateChangeReturn ret = GST_STATE_CHANGE_SUCCESS;
    gint rec_no = 0;
    gint nr=0,retval=0;
    gboolean res;
    radec_info = MFW_GST_RADEC(element);

    GST_DEBUG(" in mfw_gst_radec_change_state routine \n");
    switch (transition) {
    case GST_STATE_CHANGE_NULL_TO_READY:
    	radec_info->caps_set = FALSE;
    	radec_info->init_done = FALSE;
    	radec_info->eos = FALSE;
        radec_info->flow_error = FALSE;
    	radec_info->time_offset = 0;
    	radec_info->inbuffer1 = NULL;
    	radec_info->inbuffer2 = NULL;
        radec_info->corrupt_bs = FALSE;
#ifdef MEMORY_DEBUG
        init_memmanager(&radec_info->memmgr, "RA");
#endif 

	    break;

    case GST_STATE_CHANGE_READY_TO_PAUSED:
	    radec_info->bitsPerFrame = 0;
	    radec_info->bit_rate = 0;
	    radec_info->nFramesReceived = 0;
	    radec_info->bitstream_count = 0;
	    radec_info->bitstream_buf_index = 0;
	    radec_info->in_buf_done = 0;
	    radec_info->nBitsReceived = 0;
	    radec_info->total_time = 0;
	    radec_info->seek_flag = FALSE;
        
#ifdef PUSH_MODE        
        radec_info->pAdapter = gst_adapter_new();
        init_tsmanager(&radec_info->tsMgr);
#endif
#if 0
	    if (gst_pad_check_pull_range(radec_info->sinkpad)) {
    		gst_pad_set_query_function(radec_info->srcpad,
    			                        GST_DEBUG_FUNCPTR
    			                        (mfw_gst_radec_src_query));
    		radec_info->seek_flag = TRUE;
	    }
	    radec_info->total_frames = 0;
        radec_info->time_offset = 0;
#endif        
	    break;

    case GST_STATE_CHANGE_PAUSED_TO_PLAYING:
	    break;
        
    default:
	    break;
    }

    ret = parent_class_ra->change_state(element, transition);

    switch (transition) {
    case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
	    break;
    
    case GST_STATE_CHANGE_PAUSED_TO_READY:
    	GST_DEBUG("GST_STATE_CHANGE_PAUSED_TO_READY \n");
#ifndef PUSH_MODE        
            /* Following is memory leak fix */   
        if ((radec_info->inbuffer1) && (radec_info->inbuffer1 != radec_info->inbuffer2))
        {

    	    gst_buffer_unref(radec_info->inbuffer1);
    	    radec_info->inbuffer1 = NULL;
    	}  
        if (radec_info->inbuffer2) {

    	    gst_buffer_unref(radec_info->inbuffer2);
    	    radec_info->inbuffer1 = NULL;
    	}
#else
        radec_info->inbuffer1 = NULL;
	    radec_info->inbuffer2 = NULL;
#endif
            
    	radec_info->total_frames = 0;
        
        mfw_ra_dec_core_cleanup(radec_info);
	    break;

    case GST_STATE_CHANGE_READY_TO_NULL:
    	GST_DEBUG("GST_STATE_CHANGE_READY_TO_NULL \n");
#ifdef MEMORY_DEBUG
        deinit_memmanager(&radec_info->memmgr);
#endif 
    	radec_global_ptr = NULL;
    	break;
        
    default:
	    break;
    }
    GST_DEBUG(" out of mfw_gst_radec_change_state routine \n");
    return ret;

}

/*=============================================================================
FUNCTION: mfw_gst_radec_get_query_types

DESCRIPTION: gets the different types of query supported by the plugin

ARGUMENTS PASSED:
        pad     - pad on which the function is registered 

RETURN VALUE:
        query types ssupported by the plugin

PRE-CONDITIONS:
        None

POST-CONDITIONS:


IMPORTANT NOTES:
        None
=============================================================================*/
static const GstQueryType *mfw_gst_radec_get_query_types(GstPad * pad)
{
    static const GstQueryType src_query_types[] = {
	GST_QUERY_POSITION,
	GST_QUERY_DURATION,
	GST_QUERY_CONVERT,
	0
    };

    return src_query_types;
}

/*==================================================================================================

FUNCTION:   mfw_gst_radec_src_query   

DESCRIPTION:    performs query on src pad.    

ARGUMENTS PASSED:
        pad     -   pointer to GstPad
        query   -   pointer to GstQuery        
            
RETURN VALUE:
        TRUE    -   success
        FALSE   -   failure

PRE-CONDITIONS:
        None

POST-CONDITIONS:
		None

IMPORTANT NOTES:
        None

==================================================================================================*/
static gboolean mfw_gst_radec_src_query(GstPad * pad,
					     GstQuery * query)
{
    gboolean res = TRUE;
    GstPad *peer;

    MFW_GST_RADEC_INFO_T *radec_info;
    radec_info = MFW_GST_RADEC(GST_OBJECT_PARENT(pad));

    peer = gst_pad_get_peer(radec_info->sinkpad);
    switch (GST_QUERY_TYPE(query)) {

    case GST_QUERY_DURATION:
	{

	    GST_DEBUG("coming in GST_QUERY_DURATION \n");
	    GstFormat format;
	    GstFormat rformat;
	    gint64 total, total_bytes;
	    GstPad *peer;

	    /* save requested format */
	    gst_query_parse_duration(query, &format, NULL);
	    if ((peer =
		 gst_pad_get_peer(radec_info->sinkpad)) == NULL)
		goto error;

	    if (format == GST_FORMAT_TIME && gst_pad_query(peer, query)) {
		gst_query_parse_duration(query, NULL, &total);
		GST_DEBUG_OBJECT(radec_info,
				 "peer returned duration %"
				 GST_TIME_FORMAT, GST_TIME_ARGS(total));

	    }
	    /* query peer for total length in bytes */
	    gst_query_set_duration(query, GST_FORMAT_BYTES, -1);


	    if (!gst_pad_query(peer, query)) {
		goto error;
	    }
	    gst_object_unref(peer);

	    /* get the returned format */
	    gst_query_parse_duration(query, &rformat, &total_bytes);

	    if (rformat == GST_FORMAT_BYTES) {
		GST_DEBUG("peer pad returned total bytes=%d", total_bytes);
	    } else if (rformat == GST_FORMAT_TIME) {
		GST_DEBUG("peer pad returned total time=%",
			  GST_TIME_FORMAT, GST_TIME_ARGS(total_bytes));
	    }

	    /* Check if requested format is returned format */
	    if (format == rformat)
		return TRUE;


	    if (total_bytes != -1) {
		if (format != GST_FORMAT_BYTES) {
		    if (!mfw_gst_radec_convert_sink
			(pad, GST_FORMAT_BYTES, total_bytes, &format,
			 &total))
			goto error;
		} else {
		    total = total_bytes;
		}
	    } else {
		total = -1;
	    }
	    radec_info->total_time = total;
	    gst_query_set_duration(query, format, total);

	    if (format == GST_FORMAT_TIME) {
		GST_DEBUG("duration=%" GST_TIME_FORMAT,
			  GST_TIME_ARGS(total));
	    } else {
		GST_DEBUG("duration=%" G_GINT64_FORMAT ",format=%u", total,
			  format);
	    }
	    break;
	}
    case GST_QUERY_CONVERT:
	{
	    GstFormat src_fmt, dest_fmt;
	    gint64 src_val, dest_val;
	    gst_query_parse_convert(query, &src_fmt, &src_val, &dest_fmt,
				    &dest_val);
	    if (!
		(res =
		 mfw_gst_radec_convert_src(pad, src_fmt, src_val,
						&dest_fmt, &dest_val)))
		goto error;

	    gst_query_set_convert(query, src_fmt, src_val, dest_fmt,
				  dest_val);
	    break;
	}
    default:
	res = FALSE;
	break;
    }
    return res;

  error:
    GST_ERROR("error handling query");
    return FALSE;
}

/*==================================================================================================
FUNCTION:   mfw_gst_radec_convert_src   

DESCRIPTION:    converts the format of value from src format to destination format on src pad .    

ARGUMENTS PASSED:
        pad         -   pointer to GstPad   
        src_format  -   format of source value
        src_value   -   value of soure 
        dest_format -   format of destination value
        dest_value  -   value of destination         

RETURN VALUE:
        TRUE    -   sucess
        FALSE   -   failure  

PRE-CONDITIONS:
        None

POST-CONDITIONS:
		None

IMPORTANT NOTES:
        None

==================================================================================================*/
static gboolean
mfw_gst_radec_convert_src(GstPad * pad, GstFormat src_format,
			       gint64 src_value, GstFormat * dest_format,
			       gint64 * dest_value)
{
    gboolean res = TRUE;
    guint scale = 1;
    gint bytes_per_sample;

    MFW_GST_RADEC_INFO_T *radec_info;
    radec_info = MFW_GST_RADEC(GST_OBJECT_PARENT(pad));

    bytes_per_sample = radec_info->number_of_channels * 4;

    switch (src_format) {
    case GST_FORMAT_BYTES:
	switch (*dest_format) {
	case GST_FORMAT_DEFAULT:
	    if (bytes_per_sample == 0)
		return FALSE;
	    *dest_value = src_value / bytes_per_sample;
	    break;
	case GST_FORMAT_TIME:
	    {
		gint byterate = bytes_per_sample *
		    radec_info->sampling_freq;
		if (byterate == 0)
		    return FALSE;
		*dest_value = src_value * GST_SECOND / byterate;
		break;
	    }
	default:
	    res = FALSE;
	}
	break;
    case GST_FORMAT_DEFAULT:
	switch (*dest_format) {
	case GST_FORMAT_BYTES:
	    *dest_value = src_value * bytes_per_sample;
	    break;
	case GST_FORMAT_TIME:
	    if (radec_info->sampling_freq == 0)
		return FALSE;
	    *dest_value = src_value * GST_SECOND /
		radec_info->sampling_freq;
	    break;
	default:
	    res = FALSE;
	}
	break;
    case GST_FORMAT_TIME:
	switch (*dest_format) {
	case GST_FORMAT_BYTES:
	    scale = bytes_per_sample;
	    /* fallthrough */
	case GST_FORMAT_DEFAULT:
	    *dest_value = src_value * scale *
		radec_info->sampling_freq / GST_SECOND;
	    break;
	default:
	    res = FALSE;
	}
	break;
    default:
	res = FALSE;
    }

    return res;
}

/*==================================================================================================

FUNCTION:   mfw_gst_radec_convert_sink    

DESCRIPTION:    converts the format of value from src format to destination format on sink pad .  
   

ARGUMENTS PASSED:
        pad         -   pointer to GstPad   
        src_format  -   format of source value
        src_value   -   value of soure 
        dest_format -   format of destination value
        dest_value  -   value of destination 

RETURN VALUE:
        TRUE    -   sucess
        FALSE   -   failure

PRE-CONDITIONS:
        None

POST-CONDITIONS:
		None

IMPORTANT NOTES:
        None

==================================================================================================*/
static gboolean
mfw_gst_radec_convert_sink(GstPad * pad, GstFormat src_format,
				gint64 src_value, GstFormat * dest_format,
				gint64 * dest_value)
{
    gboolean res = TRUE;
    float avg_bitrate = 0;
    MFW_GST_RADEC_INFO_T *radec_info;
    GST_DEBUG(" in mfw_gst_radec_convert_sink \n");
    radec_info = MFW_GST_RADEC(GST_OBJECT_PARENT(pad));


    switch (src_format) {
    case GST_FORMAT_BYTES:
	switch (*dest_format) {
	case GST_FORMAT_TIME:
	    if (avg_bitrate) {

		*dest_value =
		    gst_util_uint64_scale(src_value, 8 * GST_SECOND,
					  avg_bitrate);
	    } else {
		*dest_value = GST_CLOCK_TIME_NONE;
	    }
	    break;
	default:
	    res = FALSE;
	}
	break;
    case GST_FORMAT_TIME:
	switch (*dest_format) {
	case GST_FORMAT_BYTES:
	    if (avg_bitrate) {

		*dest_value = gst_util_uint64_scale(src_value, avg_bitrate,
						    8 * GST_SECOND);

	    } else {
		*dest_value = 0;
	    }

	    break;

	default:
	    res = FALSE;
	}
	break;
    default:
	res = FALSE;
    }

    GST_DEBUG(" out of mfw_gst_radec_convert_sink \n");
    return res;
}

/*==================================================================================================

FUNCTION:   mfw_gst_radec_seek  

DESCRIPTION:    performs seek operation    

ARGUMENTS PASSED:
        radec_info -   pointer to decoder element
        pad         -   pointer to GstPad
        event       -   pointer to GstEvent

RETURN VALUE:
        TRUE    -   sucess
        FALSE   -   failure        

PRE-CONDITIONS:
        None

POST-CONDITIONS:
		None

IMPORTANT NOTES:
        None

==================================================================================================*/
static gboolean
mfw_gst_radec_seek(MFW_GST_RADEC_INFO_T * radec_info,
			GstPad * pad, GstEvent * event)
{
    gdouble rate;
    GstFormat format, conv;
    GstSeekFlags flags;
    GstSeekType cur_type, stop_type;
    gint64 cur = 0, stop = 0;
    gint64 time_cur = 0, time_stop = 0;
    gint64 bytes_cur = 0, bytes_stop = 0;
    gboolean flush;
    gboolean res;
    guint bytesavailable;
    gst_event_parse_seek(event, &rate, &format, &flags, &cur_type, &cur,
			 &stop_type, &stop);

    GST_DEBUG("\nseek from  %" GST_TIME_FORMAT "--------------- to %"
	      GST_TIME_FORMAT, GST_TIME_ARGS(cur), GST_TIME_ARGS(stop));

    if (format != GST_FORMAT_TIME) {
	conv = GST_FORMAT_TIME;
	if (!mfw_gst_radec_convert_src
	    (pad, format, cur, &conv, &time_cur))
	    goto convert_error;
	if (!mfw_gst_radec_convert_src
	    (pad, format, stop, &conv, &time_stop))
	    goto convert_error;
    } else {
	time_cur = cur;
	time_stop = stop;
    }
    GST_DEBUG("\nseek from  %" GST_TIME_FORMAT "--------------- to %"
	      GST_TIME_FORMAT, GST_TIME_ARGS(time_cur),
	      GST_TIME_ARGS(time_stop));

    /* shave off the flush flag, we'll need it later */
    flush = ((flags & GST_SEEK_FLAG_FLUSH) != 0);
    res = FALSE;
    conv = GST_FORMAT_BYTES;



    if (!mfw_gst_radec_convert_sink
	(pad, GST_FORMAT_TIME, time_cur, &conv, &bytes_cur))
	goto convert_error;

    if (!mfw_gst_radec_convert_sink
	(pad, GST_FORMAT_TIME, time_stop, &conv, &bytes_stop))
	goto convert_error;

    {
	GstEvent *seek_event;


	seek_event =
	    gst_event_new_seek(rate, GST_FORMAT_BYTES, flags, cur_type,
			       bytes_cur, stop_type, bytes_stop);

	/* do the seek */
	res = gst_pad_push_event(radec_info->sinkpad, seek_event);

    }


    return TRUE;

    /* ERRORS */
  convert_error:
    {
	/* probably unsupported seek format */
	GST_ERROR("failed to convert format %u into GST_FORMAT_TIME",
		  format);
	return FALSE;
    }
}

/*=============================================================================
FUNCTION:   mfw_gst_radec_src_event

DESCRIPTION: This functions handles the events that triggers the
			 source pad of the mpeg4 decoder element.

ARGUMENTS PASSED:
        pad        -    pointer to pad
        event      -    pointer to event
RETURN VALUE:
        TRUE       -	if event is sent to src properly
	    FALSE	   -	if event is not sent to src properly

PRE-CONDITIONS:
        None

POST-CONDITIONS:
        None

IMPORTANT NOTES:
        None
=============================================================================*/
static gboolean
mfw_gst_radec_src_event(GstPad * pad, GstEvent * event)
{

    gboolean res;
    MFW_GST_RADEC_INFO_T *radec_info;
    GST_DEBUG(" in mfw_gst_radec_src_event routine \n");
    radec_info = MFW_GST_RADEC(GST_OBJECT_PARENT(pad));

    switch (GST_EVENT_TYPE(event)) {
    case GST_EVENT_SEEK:
	gst_event_ref(event);
	if (radec_info->seek_flag == TRUE) {
	    res = mfw_gst_radec_seek(radec_info, pad, event);

	} else {
	    res = gst_pad_push_event(radec_info->sinkpad, event);
	}
	break;

    default:
	{
	    res = gst_pad_event_default(pad, event);
	    break;
	}


    }
    GST_DEBUG(" out of mfw_gst_radec_src_event routine \n");

    gst_event_unref(event);
    return res;
}

/*=============================================================================
FUNCTION:   mfw_gst_radec_sink_event 

DESCRIPTION:

ARGUMENTS PASSED:
        pad        -    pointer to pad
        event      -    pointer to event
RETURN VALUE:
        TRUE       -	if event is sent to sink properly
	    FALSE	   -	if event is not sent to sink properly

PRE-CONDITIONS:
        None

POST-CONDITIONS:
        None

IMPORTANT NOTES:
        None
=============================================================================*/
static gboolean
mfw_gst_radec_sink_event(GstPad * pad, GstEvent * event)
{
    gboolean result = TRUE;
    MFW_GST_RADEC_INFO_T *radec_info;
    GstCaps *src_caps = NULL, *caps = NULL;
    GstBuffer *outbuffer = NULL;
    GstFlowReturn res = GST_FLOW_OK;
    guint8 *inbuffer;
    gint inbuffsize;
    guint64 time_duration = 0;



    radec_info = MFW_GST_RADEC(GST_OBJECT_PARENT(pad));
    GST_DEBUG(" in mfw_gst_radec_sink_event function \n");
    switch (GST_EVENT_TYPE(event)) {
    case GST_EVENT_NEWSEGMENT:
	{
	    GstFormat format;
	    gint64 start, stop, position;
	    gint64 nstart, nstop;
	    GstEvent *nevent;

	    GST_DEBUG(" in GST_EVENT_NEWSEGMENT \n");
	    gst_event_parse_new_segment(event, NULL, NULL, &format, &start,
					&stop, &position);

	    if (format == GST_FORMAT_BYTES) {
		format = GST_FORMAT_TIME;
		if (start != 0)
		    result =
			mfw_gst_radec_convert_sink(pad,
							GST_FORMAT_BYTES,
							start, &format,
							&nstart);
		else
		    nstart = start;
		if (stop != 0)
		    result =
			mfw_gst_radec_convert_sink(pad,
							GST_FORMAT_BYTES,
							stop, &format,
							&nstop);
		else
		    nstop = stop;

		nevent =
		    gst_event_new_new_segment(FALSE, 1.0, GST_FORMAT_TIME,
					      nstart, nstop, nstart);
		gst_event_unref(event);
		radec_info->time_offset = (guint64) nstart;

		result =
		    gst_pad_push_event(radec_info->srcpad, nevent);
		if (TRUE != result) {
		    GST_ERROR
			("\n Error in pushing the event,result	is %d\n",
			 result);

		}
	    } else if (format == GST_FORMAT_TIME) {
		radec_info->time_offset = (guint64) start;

		result =
		    gst_pad_push_event(radec_info->srcpad, event);
		if (TRUE != result) {
		    GST_ERROR
			("\n Error in pushing the event,result	is %d\n",
			 result);
		   
		}
	    }
	    break;
	}
    case GST_EVENT_EOS:
	{

	    RA_DECODER_FLOW("\nDecoder: Got an EOS from Demuxer\n", 0);
	    if (radec_global_ptr == NULL)
		radec_global_ptr = radec_info;
	    else {
		if (radec_global_ptr != radec_info) {
		    gboolean res;
		    res =
			gst_pad_push_event(radec_info->srcpad, event);
		    if (res != TRUE) {
			GST_ERROR("ra dec:could not push onto \
                            next element %d\n", res);
		    }
		    return TRUE;
		}
	    }

	    result = gst_pad_push_event(radec_info->srcpad, event);
	    if (TRUE != result) {
		GST_ERROR
		    ("\n Error in pushing the event,result	is %d\n",
		     result);
	
	    }

            return TRUE;
	    break;
	}
    case GST_EVENT_FLUSH_STOP:
	{

	    GST_DEBUG(" GST_EVENT_FLUSH_STOP \n");

	    result = gst_pad_push_event(radec_info->srcpad, event);
	    if (TRUE != result) {
		GST_ERROR("\n Error in pushing the event,result	is %d\n",
			  result);
	
	    }
	    break;
	}

    case GST_EVENT_FLUSH_START:
        default:
        {
            result = gst_pad_event_default(pad, event);
            break;
        }

    }

    GST_DEBUG(" out of mfw_gst_radec_sink_event \n");
    return result;
}


/*=============================================================================
FUNCTION:   mfw_gst_radec_set_caps

DESCRIPTION:    this function handles the link with other plug-ins and used for
                capability negotiation  between pads

ARGUMENTS PASSED:
        pad        -    pointer to GstPad
        caps       -    pointer to GstCaps

RETURN VALUE:
        TRUE       -    if capabilities are set properly
        FALSE      -    if capabilities are not set properly
PRE-CONDITIONS:
        None

POST-CONDITIONS:
        None

IMPORTANT NOTES:
        None
=============================================================================*/
static gboolean mfw_gst_radec_set_caps(GstPad * pad, GstCaps * caps)
{



    MFW_GST_RADEC_INFO_T *radec_info;
    const gchar *mime;
    gint mpeg_version;
    GstStructure *structure = gst_caps_get_structure(caps, 0);
    radec_info = MFW_GST_RADEC(gst_pad_get_parent(pad));

    GST_DEBUG(" in mfw_gst_radec_set_caps routine \n");
    mime = gst_structure_get_name(structure);

    if (strcmp(mime, "audio/x-pn-realaudio") != 0) {
	GST_WARNING
	    ("Wrong	mimetype %s	provided, we only support %s",
	     mime, "audio/x-pn-realaudio");
        gst_object_unref(radec_info);
	return FALSE;
    }

    gst_structure_get_int(structure, "bitrate",
			  &radec_info->bit_rate);

    radec_info->flavorindex = 0;
    gst_structure_get_int(structure, "flavorindex",
			  &radec_info->flavorindex);
    

    if (!gst_pad_set_caps(pad, caps)) {
        gst_object_unref(radec_info);
	return FALSE;
    }

    GST_DEBUG(" out of mfw_gst_radec_set_caps routine \n");
    gst_object_unref(radec_info);
    return TRUE;
}

/*=============================================================================
FUNCTION:   mfw_gst_radec_init

DESCRIPTION:This function creates the pads on the elements and register the
			function pointers which operate on these pads.

ARGUMENTS PASSED:
        pointer the ra decoder element handle.

RETURN VALUE:
        None

PRE-CONDITIONS:
        _base_init and _class_init are called

POST-CONDITIONS:
        None

IMPORTANT NOTES:
        None
=============================================================================*/

static void
mfw_gst_radec_init(MFW_GST_RADEC_INFO_T * radec_info)
{

    GstElementClass *klass = GST_ELEMENT_GET_CLASS(radec_info);
    GST_DEBUG(" \n in mfw_gst_radec_init routine \n");
    radec_info->sinkpad =
	gst_pad_new_from_template(gst_element_class_get_pad_template
				  (klass, "sink"), "sink");

    radec_info->srcpad =
	gst_pad_new_from_template(gst_element_class_get_pad_template
				  (klass, "src"), "src");

    gst_element_add_pad(GST_ELEMENT(radec_info),
			radec_info->sinkpad);
    gst_element_add_pad(GST_ELEMENT(radec_info),
			radec_info->srcpad);

    gst_pad_set_setcaps_function(radec_info->sinkpad,
				 mfw_gst_radec_set_caps);
    gst_pad_set_chain_function(radec_info->sinkpad,
			       mfw_gst_radec_chain);

    gst_pad_set_event_function(radec_info->sinkpad,
			       GST_DEBUG_FUNCPTR
			       (mfw_gst_radec_sink_event));

    gst_pad_set_query_type_function(radec_info->srcpad,
				    GST_DEBUG_FUNCPTR
				    (mfw_gst_radec_get_query_types));
    gst_pad_set_event_function(radec_info->srcpad,
			       GST_DEBUG_FUNCPTR
			       (mfw_gst_radec_src_event));


    GST_DEBUG("\n out of mfw_gst_radec_init \n");


    //

    #define MFW_GST_RA_DECODER_PLUGIN VERSION
    PRINT_CORE_VERSION(RealaudioDCodecVersionInfo());
    PRINT_PLUGIN_VERSION(MFW_GST_RA_DECODER_PLUGIN);    

    INIT_DEMO_MODE(RealaudioDCodecVersionInfo(), radec_info->demo_mode);
}

/*=============================================================================
FUNCTION:   mfw_gst_radec_class_init

DESCRIPTION:Initialise the class only once (specifying what signals,
            arguments and virtual functions the class has and setting up
            global state)
ARGUMENTS PASSED:
       	klass   - pointer to ra decoder's element class

RETURN VALUE:
        None

PRE-CONDITIONS:
        None

POST-CONDITIONS:
        None

IMPORTANT NOTES:
        None
=============================================================================*/
static void
mfw_gst_radec_class_init(MFW_GST_RADEC_CLASS_T * klass)
{
    GObjectClass *gobject_class = NULL;
    GstElementClass *gstelement_class = NULL;
    gobject_class = (GObjectClass *) klass;
    gstelement_class = (GstElementClass *) klass;
    parent_class_ra =
	(GstElementClass *) g_type_class_ref(GST_TYPE_ELEMENT);
    gobject_class->set_property = mfw_gst_radec_set_property;
    gobject_class->get_property = mfw_gst_radec_get_property;
    gstelement_class->change_state = mfw_gst_radec_change_state;
}

/*=============================================================================
FUNCTION:  mfw_gst_radec_base_init

DESCRIPTION:
            ra decoder element details are registered with the plugin during
            _base_init ,This function will initialise the class and child
            class properties during each new child class creation


ARGUMENTS PASSED:
        Klass   -   pointer to ra decoder plug-in class

RETURN VALUE:
        None

PRE-CONDITIONS:
        None

POST-CONDITIONS:
        None

IMPORTANT NOTES:
        None
=============================================================================*/

static void
mfw_gst_radec_base_init(MFW_GST_RADEC_CLASS_T * klass)
{
    static GstElementDetails element_details = {
	"Freescale RealMedia audio decoder",
	"Codec/Decoder/Audio",
	"Decodes RealVideo Bitstreams",
	FSL_GST_MM_PLUGIN_AUTHOR
    };
    GstElementClass *element_class = GST_ELEMENT_CLASS(klass);
    gst_element_class_add_pad_template(element_class,
				       gst_static_pad_template_get
				       (&src_factory));
    gst_element_class_add_pad_template(element_class,
				       gst_static_pad_template_get
				       (&sink_factory));
    gst_element_class_set_details(element_class, &element_details);
}

/*=============================================================================
FUNCTION: mfw_gst_radec_get_type

DESCRIPTION:    intefaces are initiated in this function.you can register one
                or more interfaces  after having registered the type itself.

ARGUMENTS PASSED:
            None

RETURN VALUE:
                 A numerical value ,which represents the unique identifier of this
            element(radecoder)

PRE-CONDITIONS:
            None

POST-CONDITIONS:
            None

IMPORTANT NOTES:
            None
=============================================================================*/
GType mfw_gst_radec_get_type(void)
{
    static GType radec_type = 0;

    if (!radec_type) {
	static const GTypeInfo radec_info = {
	    sizeof(MFW_GST_RADEC_CLASS_T),
	    (GBaseInitFunc) mfw_gst_radec_base_init,
	    NULL,
	    (GClassInitFunc) mfw_gst_radec_class_init,
	    NULL,
	    NULL,
	    sizeof(MFW_GST_RADEC_INFO_T),
	    0,
	    (GInstanceInitFunc) mfw_gst_radec_init,
	};
	radec_type = g_type_register_static(GST_TYPE_ELEMENT,
						 "MFW_GST_RADEC_INFO_T",
						 &radec_info,
						 (GTypeFlags) 0);
    }
    GST_DEBUG_CATEGORY_INIT(mfw_gst_radec_debug, "mfw_radecoder",
			    0, "FreeScale's RA Decoder's Log");

    return radec_type;
}


/*****************************************************************************/
/*    This is used to define the entry point and meta data of plugin         */
/*****************************************************************************/
GST_PLUGIN_DEFINE(GST_VERSION_MAJOR,	/* major version of Gstreamer    */
		  GST_VERSION_MINOR,	/* minor version of Gstreamer    */
		  "mfw_radecoder",	/* name of the plugin            */
		  "decodes RealAudio streams ",	/* what plugin actually does     */
		  plugin_init,	/* first function to be called   */
		  VERSION,
		  GST_LICENSE_UNKNOWN,
		  FSL_GST_MM_PLUGIN_PACKAGE_NAME, FSL_GST_MM_PLUGIN_PACKAGE_ORIG)
/*=============================================================================
FUNCTION:   plugin_init

DESCRIPTION:    special function , which is called as soon as the plugin or
                element is loaded and information returned by this function
                will be cached in central registry

ARGUMENTS PASSED:
        plugin     -    pointer to container that contains features loaded
                        from shared object module

RETURN VALUE:
        return TRUE or FALSE depending on whether it loaded initialized any
        dependency correctly

PRE-CONDITIONS:
        None

POST-CONDITIONS:
        None

IMPORTANT NOTES:
        None
=============================================================================*/
static gboolean plugin_init(GstPlugin * plugin)
{
    return gst_element_register(plugin, "mfw_radecoder",
				GST_RANK_PRIMARY, MFW_GST_TYPE_RADEC);
}
