/*
 * $Id: conn_eth.c,v 1.1 2009-01-30 15:21:08 vrsieh Exp $
 *
 * Copyright (C) 2003-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <stdio.h>

#include "glue-shm.h"
#include "random100.h"

#include "conn_eth.h"

#define CHIP_(x) conn_eth_ ## x
#define CHIP "conn_eth"

struct cpssp {
	struct sig_eth *port_board;
	struct sig_eth *port_eth;
	struct sig_fault *port_recv_loss;
	struct sig_fault *port_send_loss;

	/* State */
	unsigned int recv_loss;
	unsigned int send_loss;
};

static void
CHIP_(out_recv)(void *_cpssp, const void *buf, unsigned int buflen)
{
	struct cpssp *cpssp = _cpssp;

	if (random100() < cpssp->send_loss) {
		/* Packet lost due to fault injection. */
		return;
	}

	sig_eth_send(cpssp->port_eth, cpssp, buf, buflen);
}

static void
CHIP_(in_recv)(void *_cpssp, const void *buf, unsigned int buflen)
{
	struct cpssp *cpssp = _cpssp;

	if (random100() < cpssp->recv_loss) {
		/* Packet lost due to fault injection. */
		return;
	}

	sig_eth_send(cpssp->port_board, cpssp, buf, buflen);
}

static void
CHIP_(recv_loss_set)(
	void *_cpssp,
	unsigned long long loc0, 
	unsigned long long loc1, 
	unsigned int val
)
{
	struct cpssp *cpssp = _cpssp;

	cpssp->recv_loss = val;
}

static void
CHIP_(send_loss_set)(
	void *_cpssp,
	unsigned long long loc0, 
	unsigned long long loc1, 
	unsigned int val
)
{
	struct cpssp *cpssp = _cpssp;

	cpssp->send_loss = val;
}

void
CHIP_(init)(
	unsigned int nr,
	struct sig_eth *port_board,
	struct sig_eth *port_eth,
	struct sig_fault *port_recv_loss,
	struct sig_fault *port_send_loss
)
{
	static const struct sig_eth_funcs board_funcs = {
		.recv = CHIP_(out_recv),
	};
	static const struct sig_eth_funcs eth_funcs = {
		.recv = CHIP_(in_recv),
	};
	static const struct sig_fault_funcs recv_loss_funcs = {
		.set = CHIP_(recv_loss_set),
	};
	static const struct sig_fault_funcs send_loss_funcs = {
		.set = CHIP_(send_loss_set),
	};
	struct cpssp *cpssp;

	cpssp = shm_map(CHIP, nr, sizeof(*cpssp), 0);

	/* Call */
	cpssp->port_board = port_board;
	sig_eth_connect(port_board, cpssp, &board_funcs);

	cpssp->port_eth = port_eth;
	sig_eth_connect(port_eth, cpssp, &eth_funcs);

	cpssp->port_recv_loss = port_recv_loss;
	sig_fault_connect(port_recv_loss, cpssp, &recv_loss_funcs);

	cpssp->port_send_loss = port_send_loss;
	sig_fault_connect(port_send_loss, cpssp, &send_loss_funcs);

	/* Out */
	/* In */
}

unsigned int
CHIP_(create)(void)
{
	static unsigned int nr = 0;
	struct cpssp *cpssp;

	shm_create(CHIP, nr, sizeof(*cpssp));
	cpssp = shm_map(CHIP, nr, sizeof(*cpssp), 0);

	cpssp->recv_loss = 0;
	cpssp->send_loss = 0;

	shm_unmap(cpssp, sizeof(*cpssp));

	return nr++;
}

void
CHIP_(destroy)(unsigned int nr)
{
	struct cpssp *cpssp;

	cpssp = shm_map(CHIP, nr, sizeof(*cpssp), 0);

	shm_unmap(cpssp, sizeof(*cpssp));
	shm_destroy(CHIP, nr);
}
