/* $Id: rtc.c,v 1.48 2009-01-28 14:38:02 potyra Exp $ 
 *
 * Copyright (C) 2004-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "build_config.h"
#include "compiler.h"

/* ==================== RUNTIME_RM ==================== */
#ifdef RUNTIME_RM

CODE16;

#include "ptrace.h"
#include "debug.h"

#include "io.h"
#include "cmos.h"
#include "const.h"
#include "var.h"
#include "assert.h"

#include "disk.h"
#include "rtc.h"


void
rtc_tick(void)
{
	unsigned long t;
	unsigned char d;

	t = var_get(ticks_high);
	t = (t << 16) | var_get(ticks_low);
	t++;
	/* 1573064 = 60 * 60 * 24 * 1193180 / 65535 */
	if (1573064 <= t) {
		/* One day is over... */
		t = 0;
		d = var_get(time_24h);
		d++;
		var_put(time_24h, d);
	}
	var_put(ticks_high, (unsigned short) (t >> 16));
	var_put(ticks_low, (unsigned short) (t >> 0));
}

/*
 * Wait for CX:DX microseconds
 */
void
bios_15_86xx(struct regs *regs)
{
	/*
	 * Use RAM refresh counter (PIT counter 1).
	 * It toggles every 15usecs.
	 */
	unsigned long usecs;
	unsigned long ticks;

	usecs = ((unsigned long) CX << 16) | DX;

	for (ticks = (usecs + 7) / 15; 0 < ticks; ticks--) {
		/* Wait for refresh timer high. */
		while (! ((inb(0x61) >> 4) & 1)) {
		}
		/* Wait for refresh timer low. */
		while (! ((inb(0x61) >> 4) & 1)) {
		}
	}

	F &= ~(1 << 0);	/* Clear carry. */
}

static int
rtc_updating(void)
{
	unsigned short count;

	count = 25000;
	while (--count != 0) {
		if ((cmos_get(stat_reg_a) & RTC_UIP) == 0) {
			/* Clock currently not updating. */
			return 0;
		}
	}

	return 1; /* update-in-progress never transitioned to 0. */
}

/*
 * Set timer ticks.
 *
 * In:	AH	= 1Ah
 *	AL	= 01h
 *	CH:DX	= timer ticks since midnight
 *
 * Out:	-
 */
void
bios_1a_01xx(struct regs *regs)
{
	var_put(ticks_high, CX);
	var_put(ticks_low, DX);
	var_put(time_24h, 0);

	AH = 0;
	F &= ~(1 << 0);	/* Clear carry. */
}

/*
 * Read CMOS time.
 *
 * In:	AH	= 1Ah
 * 	AL	= 02h
 *
 * Out:	F
 *	 Carry	= 0: Ok, 1: clock currently updating
 *	CL	= hours (BCD)
 *	CH	= minutes (BCD)
 *	DH	= seconds (BCD)
 *	DL	= daylight savings time (00h std, 01h daylight)
 */
void
bios_1a_02xx(struct regs *regs)
{
	unsigned char hours;	/* BCD */
	unsigned char minutes;	/* BCD */
	unsigned char seconds;	/* BCD */
	unsigned char dst;

	if (rtc_updating()) {
		F |= (1 << 0);	/* Set carry. */
		return;
	}

	seconds = cmos_get(seconds);
	assert(/* 0x00 <= seconds && */ seconds < 0x60);

	minutes = cmos_get(minutes);
	assert(/* 0x00 <= minutes && */ minutes < 0x60);

	hours = cmos_get(hours);
	assert(/* 0x00 <= hours && */ hours < 0x24);

	dst = (cmos_get(stat_reg_b) & RTC_DST_EN) ? 1 : 0;
	assert(/* 0x00 <= dst && */ dst < 2);

	DEBUGPRINT(10, "return CMOS time: %02x:%02x:%02x\n",
			hours, minutes, seconds);

	CH = hours;
	CL = minutes;
	DH = seconds;
	DL = dst;

	AL = hours;

	F &= ~(1 << 0);	/* Clear carry. */
}

/*
 * Write CMOS time.
 *
 * In:	AH	= 1Ah
 * 	AL	= 03h
 *
 * 	CH	= hours (BCD)
 * 	CL	= minutes (BCD)
 * 	DH	= seconds (BCD)
 *	DL	= daylight savings time (00h std, 01h daylight)
 * Out:	F
 *	 Carry	= 0: Ok, 1: clock currently updating
 */
void
bios_1a_03xx(struct regs *regs)
{
	unsigned char val8;

	if (rtc_updating()) {
		/* Init RTC. */
		assert(0); /* FIXME VOSSI */
	}

	DL &= 1;

	cmos_put(seconds, DH);
	cmos_put(minutes, CL);
	cmos_put(hours, CH);
	val8 = cmos_get(stat_reg_b);
	val8 &= ~RTC_DST_EN;
	val8 |= DL ? RTC_DST_EN : 0;
	val8 |= RTC_24H;
	cmos_put(stat_reg_b, val8);

	F &= ~(1 << 0);	/* Clear carry. */
}

/*
 * Read CMOS date.
 *
 * In:	AH	= 1Ah
 * 	AL	= 04h
 *
 * Out:	CH	= century (BCD)
 * 	CL	= year (BCD)
 *	DH	= month (BCD)
 *	DL	= day of month (BCD)
 *	F
 *	 Carry	= 0: Ok, 1: clock currently updating
 */
void
bios_1a_04xx(struct regs *regs)
{
	unsigned char century;
	unsigned char year;
	unsigned char month;
	unsigned char day;

	if (rtc_updating()) {
		F |= (1 << 0);	/* Set carry. */
		return;
	}

	century = cmos_get(century);
	year = cmos_get(year);
	month = cmos_get(month);
	day = cmos_get(day_of_month);

	DEBUGPRINT(10, "return CMOS date: %04x/%02x/%02x\n",
			century * 0x100 + year, month, day);
	CH = century;
	CL = year;
	DH = month;
	DL = day;

	AL = century;

	F &= ~(1 << 0);	/* Clear carry. */
}

/*
 * Write CMOS date.
 *
 * In:	AH	= 1Ah
 * 	CH	= Century
 * 	CL	= Year
 * 	DH	= Month
 * 	DL	= Day of month
 *
 * Out:	F
 *	 Carry	= 0: Ok, 1: clock currently updating
 */
void
bios_1a_05xx(struct regs *regs)
{
	uint8_t val8;

	if (rtc_updating()) {
		/* Init RTC. */
		assert(0); /* FIXME VOSSI */
	}

	cmos_put(century, CH);
	cmos_put(year, CL);
	cmos_put(month, DH);
	cmos_put(day_of_month, DL);
	cmos_put(day_of_week, 0);

	/* Start clock. */
	val8 = cmos_get(stat_reg_b);
	val8 &= ~RTC_SET;
	cmos_put(stat_reg_b, val8);

	AH = 0x00;
	AL = val8;

	F &= ~(1 << 0);	/* Clear carry. */
}

/*
 * Set Alarm.
 *
 * In:	AH	= 1Ah
 *	AL	= 06h
 * Out:	F
 *	 Carry	= 0: Ok, 1: alarm already set
 */
void
bios_1a_06xx(struct regs *regs)
{
	uint8_t val8;

	if (cmos_get(stat_reg_b) & RTC_AIE) {
		/* Alarm already set. */
		F |= (1 << 0); /* Set carry. */
		return;
	}

	cmos_put(hours_alarm, CH);
	cmos_put(minutes_alarm, CL);
	cmos_put(seconds_alarm, DH);

	/* Enable RTC Interrupt. */
	val8 = inb(0xa1);
	val8 &= 1 << (8 - 8);
	outb(val8, 0xa1);

	/* Enable AIE. */
	val8 = cmos_get(stat_reg_b);
	val8 |= RTC_AIE;
	cmos_put(stat_reg_b, val8);

	F &= ~(1 << 0);	/* Clear carry. */
}

/*
 * Reset Alarm.
 *
 * In:	AH	= 1Ah
 *	AL	= 07h
 * Out:	F
 *	 Carry	= 0
 */
void
bios_1a_07xx(struct regs *regs)
{
	uint8_t val8;

	/* Disable AIE. */
	val8 = cmos_get(stat_reg_b);
	val8 &= ~RTC_AIE;
	cmos_put(stat_reg_b, val8);

	F &= ~(1 << 0);	/* Clear carry. */
}

#endif /* RUNTIME_RM */
/* ==================== REAL-MODE INIT ==================== */
#ifdef INIT_RM

CODE16;

#include "io.h"
#include "cmos.h"
#include "var.h"
#include "rtc.h"
#include "assert.h"

void
rtc_init(void)
{
	uint8_t year;

	/*
	 * Initialize 18Hz clock.
	 */
	var_put(ticks_low, 0);
	var_put(ticks_high, 0);
	var_put(time_24h, 0);

	/*
	 * Set clock control to 32kHz.
	 * Windows XP needs this!
	 */
	cmos_put(stat_reg_a, RTC_CLCK_32KHZ);

	/*
	 * Set mode to BCD, 24h.
	 */
	cmos_put(stat_reg_b, RTC_24H);

	/*
	 * Clear pending interrupts.
	 */
	(void) cmos_get(stat_reg_c);

	/*
	 * Fix RTC century if not set properly.
	 */
	year = cmos_get(year);
	if (0x80 <= year) {
		/* We assume 1980...1999. */
		cmos_put(century, 0x19);
	} else {
		/* We assume 2000...2079. */
		cmos_put(century, 0x20);
	}
}

#endif /* INIT_RM */
