/*
 * $Id: sig_ide_bus.c,v 1.10 2009-02-24 15:12:23 vrsieh Exp $
 *
 * Copyright (C) 2005-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"
#include <assert.h>
#include <inttypes.h>
#include <stdio.h>

#include "glue-shm.h"

#include "sig_ide_bus.h"

void
sig_ide_bus_inw(
	struct sig_ide_bus *bus,
	unsigned short port,
	unsigned short *valp
)
{
	unsigned int nr;
	int (*func)(void *, unsigned short, unsigned short *);

	assert(/* 0 <= port && */ port <= 8);

	for (nr = 0; ; nr++) {
		if (nr == bus->ndevices) {
			/* Host has pull-down resistor on bit 7. */
			*valp = (uint16_t) ~(1 << 7);
			break;
		}
		func = bus->device[nr].funcs->inw;
		if (func(bus->device[nr].s, port, valp) == 0) {
			break;
		}
	}
}

void
sig_ide_bus_outw(
	struct sig_ide_bus *bus,
	unsigned short port,
	unsigned short val
)
{
	unsigned int nr;
	void (*func)(void *, unsigned short, unsigned short);

	assert(/* 0 <= port && */ port <= 8);

	for (nr = 0; nr < bus->ndevices; nr++) {
		func = bus->device[nr].funcs->outw;
		func(bus->device[nr].s, port, val);
	}
}

void
sig_ide_bus_irq(
	struct sig_ide_bus *bus,
	void *s,
	unsigned int val
)
{
	unsigned int oldresval;
	unsigned int newresval;
	unsigned int nr;

	assert(/* 0 <= val && */ val <= 1);

	oldresval = 0;
	newresval = 0;

	/*
	 * Compute old and new bus value.
	 */
	for (nr = 0; nr < bus->ndevices; nr++) {
		oldresval |= bus->device[nr].irq_state;
		if (bus->device[nr].s == s) {
			bus->device[nr].irq_state = val;
		}
		newresval |= bus->device[nr].irq_state;
	}

	/*
	 * If old value equal new value do nothing.
	 */
	if (oldresval == newresval) {
		return;
	}

	/*
	 * Call callback function.
	 */
	bus->controller.funcs->irq(bus->controller.s, newresval);
}

void
sig_ide_bus_dmarq_set(
	struct sig_ide_bus *bus,
	unsigned int val
)
{
	bus->controller.funcs->dmarq_set(bus->controller.s, val);
}

void
sig_ide_bus_connect_device(
	struct sig_ide_bus *bus,
	void *s,
	const struct sig_ide_bus_device_funcs *funcs
)
{
	assert(bus->ndevices < 2);
	assert(bus->type == SIG_GEN_IDE_BUS);

	bus->device[bus->ndevices].s = s;
	bus->device[bus->ndevices].funcs = funcs;
	bus->device[bus->ndevices].irq_state = 0;
	bus->ndevices++;
}

void
sig_ide_bus_connect_controller(
	struct sig_ide_bus *bus,
	void *s,
	const struct sig_ide_bus_controller_funcs *funcs
)
{
	assert(bus->type == SIG_GEN_IDE_BUS);
	bus->controller.s = s;
	bus->controller.funcs = funcs;
}

struct sig_ide_bus *
sig_ide_bus_init(
	const char *name,
	int nr
)
{
	struct sig_ide_bus *bus;

	bus = shm_map(name, nr, sizeof(struct sig_ide_bus), 0);

	bus->ndevices = 0;
	bus->type = SIG_GEN_IDE_BUS;

	return bus;
}

void
sig_ide_bus_create(const char *name, int nr)
{
	shm_create(name, nr, sizeof(struct sig_ide_bus));
}

void
sig_ide_bus_destroy(const char *name, int nr)
{
	shm_destroy(name, nr);
}
