# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Authors: Philippe Normand <philippe@fluendo.com>
#          Benjamin Kampmann <benjamin@fluendo.com>

"""
Miscellaneaous utilities that don't need their own module because they are
reasonnably small.
"""

import mimetypes
import sys, platform
import os, re
from twisted.internet import defer
from twisted.trial.unittest import SkipTest

import pkg_resources

"""
The name of the environment variable that has to be set to 'True' to run also
functional tests
"""
FUNCTIONAL_TESTS_STR = 'RUN_FUNCTIONAL_TESTS'

def env_var_expand(var_name):
    """
    Expand the given environment variable content. If it contains
    other references to environment variables, they are expanded too.

    Supported platforms are win32 and linux.
    
    Example of use::

       >>> env_var_expand('$HOME')
       >>> '/home/phil'

    @raises ValueError: if current system's platform is not windows or linux
    @param var_name:    environment variable
    @type var_name:     string
    @rtype:             string
    """
    platform_type = platform.system().lower()
    if platform_type == 'windows':
        re_env = re.compile(r'%(\w+)%')
        def expander(mo):
            return os.environ.get(mo.group()[1:-1], 'UNKNOWN')
        
    elif platform_type == 'linux':
        re_env = re.compile(r'\$(\w+)')
        
        def expander(mo):
            xpanded = os.environ.get(mo.group()[1:], 'UNKNOWN')
            if xpanded.find('$') > -1:
                xpanded = re_env.sub(expander, xpanded)
            return xpanded
        
    else:
        raise ValueError("Unsupported platform")
    
    expanded = re_env.sub(expander, var_name)
    return expanded

def env_var_explode_list(var_name, default=''):
    """
    Explode a list of values stored in an environment variable as a
    single string. On win32 the item separator is ';' and on other
    platforms it is ':'.

    Example of use::

       >>> env_var_explode_list('$PATH')
       >>> ['/usr/bin','/bin']
    
    @param var_name:    environment variable
    @type var_name:     string
    @keyword default:   value to use if environment variable not found
    @type default:      string
    @rtype:             list of strings
    """
    value = os.environ.get(var_name, default)
    platform_type = platform.system().lower()
    if platform_type == 'windows':
        separator = ';'
    else:
        separator = ':'
    exploded = value.split(separator)
    if exploded == ['']:
        exploded = []
    return exploded
    

def un_camelify(camel_string):
    """
    Convert CamelCase styled strings to lower_cased style.

    @param camel_string: CamelStyled string to convert
    @type camel_string:  string
    @rtype:              string
    """
    if len(camel_string) < 2:
        un_cameled = camel_string.lower()
    else:
        camel_string = camel_string.replace(' ', '')
        un_cameled = camel_string[0].lower()
        for letter in camel_string[1:]:
            if letter.isupper():
                un_cameled += '_%s' % letter.lower()
            else:
                un_cameled += letter
    return un_cameled

# cache the result of the first is_hildon_desktop_running call
_hildon_desktop_running = None

def is_hildon_desktop_running():
    global _hildon_desktop_running

    try:
        import dbus
    except ImportError:
        _hildon_desktop_running = False
        return False

    if _hildon_desktop_running is not None:
        # return the cached value
        return _hildon_desktop_running
    try: 
        bus = dbus.SystemBus()
        driver = bus.get_object(dbus.BUS_DAEMON_NAME,
            dbus.BUS_DAEMON_PATH, dbus.BUS_DAEMON_IFACE)
        iface = dbus.Interface(driver, dbus.BUS_DAEMON_IFACE)
        _hildon_desktop_running = \
            bool(iface.NameHasOwner('org.moblin.SystemDaemon'))
    except dbus.DBusException:
        # it fails so we assume it is not hildon
        _hildon_desktop_running = False
        return False

    return _hildon_desktop_running

def get_distro():
    return "UME HP 1.0"

    DISTROS = {
        "Ubuntu" : "/etc/lsb-release",
        "Debian" : "/etc/debian_version",
        #"RedHat" : "/etc/redhat-release",
        "SUSE" : "/etc/SUSE-release",
        "Fedora" : "/etc/fedora-release",
        "Gentoo" : "/etc/gentoo-release",
        "Slackware" : "/etc/slackware-version",
        "Mandriva" : "/etc/mandriva-release",
        "Mandrake" : "/etc/mandrake-release",
        "YellowDog" : "/etc/yellowdog-release",
        "SUN JDS" : "/etc/sun-release",
        "UnitedLinux" : "/etc/UnitedLinux-release"
    }

    for name, location in DISTROS.iteritems():
        if os.path.exists(location):
            return name
    return 'Unknown'


def get_os_name():
    distro = get_distro()
    if distro != "Unknown":
        return distro
    else:
        return platform.system().lower()


def pkg_resources_copy_dir(resource_spec, resource_dir, dest_dir):
    """
    Copy a directory recursively using pkg_resources.
    """
    assert pkg_resources.resource_isdir(resource_spec, resource_dir)

    # add '' at the end so that working_dir ends with a slash
    working_dir = os.path.join(dest_dir, '')

    pkg_resources.ensure_directory(working_dir)

    for rel_entry in pkg_resources.resource_listdir(resource_spec, resource_dir):
        # rel_entry is relative to resource_dir
        entry = os.path.join(resource_dir, rel_entry)

        if pkg_resources.resource_isdir(resource_spec, entry):
            dest_entry = os.path.join(dest_dir, rel_entry)
            pkg_resources_copy_dir(resource_spec, entry, dest_entry)
            continue

        working_file = os.path.join(working_dir, rel_entry)
        resource_stream = pkg_resources.resource_stream(resource_spec, entry)
        
        out = file(working_file, 'w')
        out.write(resource_stream.read())
        out.close()
        resource_stream.close()

def run_functional_tests_check():
    """
    Check whether functional tests should be run or not. This is done by
    checking whether the L{FUNCTIONAL_TESTS_STR} environment variable is set to
    'True'. If it is not explictly set to True, this method raises a
    L{twisted.trial.unittest.SkipTest} (saying that the functional test is not
    run).

    This method allows functional tests to have an easy way to check if they
    should be run. This simple example checks in the setup (but it could also
    be done inside the method itself)::

        from elisa.core.utils.misc import run_functional_tests_check
        [...]
        class MyFunctionalTestCase(TestCase):

            def setUp(self):
                run_functional_tests_check()
                ...
    """
    msg = "Functional Test skipped. If you want to run it set the environment"\
            " variable '%s' to True" % FUNCTIONAL_TESTS_STR

    if os.environ.get(FUNCTIONAL_TESTS_STR, 'False').lower() != 'true':
        raise SkipTest(msg)

def chainDeferredIgnoringResult(first, chained):
    def cb(result):
        chained.callback(result)
        return result

    def eb(failure):
        chained.errback(failure)
        return failure

    first.addCallbacks(cb, eb)

def callAfterDeferredCalled(dfr, method, *args, **kw):
    retry_callback = kw.get('retry_callback', None)
    if retry_callback is None:
        def retry(result_or_failure):
            return method(*args, **kw)

        retry_callback = retry

    new_dfr = defer.Deferred()
    new_dfr.addBoth(retry_callback)

    chainDeferredIgnoringResult(dfr, new_dfr)

    return new_dfr
