# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

"""
Access resources on the local file system
"""


from elisa.core.components.resource_provider import ResourceProvider

from elisa.core.utils import defer, locale_helper

from elisa.plugins.base.models.file import FileModel, DirectoryModel

from twisted.internet import threads, task

import os, platform

if platform.system() == 'Windows':
    from elisa.core.utils import locale_helper
    from win32com.shell import shell
    import pythoncom

class LocalResource(ResourceProvider):
    """
    Access resources on the local file system
    """

    supported_uri = "^file://"
    def __init__(self):
        super(LocalResource, self).__init__()
        if platform.system() == 'Windows':
            self._init_com()

    def _listdir(self, path):
        """
        wrap os.listdir in a private method that can be overriden in
        the tests if needed
        """
        return os.listdir(path)

    def get(self, uri, model=None):
        """
        This reacts different in the case you give it a directory. If you want
        to have the C{Files} of a directory, make sure you have a ending '/' in
        your request. Otherwise the implementation assumes that you just want to
        have informations about that object (like the size and the modification
        time). The context model is reused if given. But in case it is a
        directory listing the C{Files} attribute is cleared before.

        Some example request would be: uri-> model with data you get

        C{file:///tmp}  -> empty DirectoryModel with informations about the
                           directory it self
        C{file://tmp/}  -> Directory with all the files (and directories) in it
        """

        shortcut_name = None

        if not os.path.exists(uri.path):
            msg = "[Errno 2] No such file or directory: '%s'" % uri.path
            return (None, defer.fail(IOError(msg)))

        # is it a win32 shortcut?
        if platform.system() == 'Windows' and uri.extension.lower() == 'lnk':
            shortcut_name = uri.filename[:-4]
            uri.path = self._resolve_shortcut(uri.path).replace('\\', '/')
            if not os.path.exists(uri.path):
                msg = "[Errno 2] No such file or directory: '%s'" % uri.path
                return (None, defer.fail(IOError(msg)))

        path = uri.path

        if model is None:
            # no model given, return just simple models
            if os.path.isdir(path):
                model = DirectoryModel()
            else:
                model = FileModel()

            model.uri = uri
            if not shortcut_name:
                model.name = uri.filename
            else:
                model.name = shortcut_name

        # read the stat and set it to the model
        stat = os.stat(path)
        model.size = stat.st_size
        model.mtime = stat.st_mtime
        model.atime = stat.st_atime

        dfr = defer.succeed(model)

        if uri.path[-1] == '/':
            # someone asks us for the content of the directory

            def wrong_file(error, file):
                self.warning("Skipping %s: %s" % (file, error.value))
                return None

            # read the files and append them
            def iterate_files(list, dir_model, uri):
                for filename in list:
                    try:
                        new_uri = uri.join(filename)
                        if not os.path.exists(new_uri.path):
                            filename = filename.decode(locale_helper.system_encoding())
                            new_uri = uri.join(filename)
                    except UnicodeError, exc:
                        self.warning("Skipping %s: %s" % (filename, exc))
                        continue

                    model, dfr = self.get(new_uri)
                    dfr.addCallbacks(dir_model.files.append, wrong_file,
                        errbackArgs=(filename,))
                    yield dfr

            def set_files(list, model, uri):
                dfr = task.coiterate(iterate_files(list,model,uri))
                dfr.addCallback(lambda x: model)
                return dfr

            dfr = threads.deferToThread(self._listdir, path)
            dfr.addCallback(set_files, model, uri)

            # act like the cancellable defer. PFUI!
            dfr.cancel = lambda x: x

        return (model, dfr)

    def post(self, uri, time=None):
        """
        change the access and modifcation time with given time=value.
        """
        path = uri.path
        if not os.path.exists(path):
            msg = "[Errno 2] No such file or directory: '%s'" % path
            return defer.fail(IOError(msg))

        if time:
            os.utime(path, (time, time))
        return defer.succeed('Done')

    def delete(self, uri):
        """
        delete a file from the file system
        """

        path = uri.path
        if not os.path.exists(path):
            msg = "[Errno 2] No such file or directory: '%s'" % path
            return defer.fail(IOError(msg))

        if os.path.isdir(path):
            os.rmdir(path)
        else:
            os.remove(path)

        return defer.succeed(uri)

    def _init_com(self):
        """
        [win32] init the stuff needed for using COM Objects
        """
        pythoncom.CoInitialize()

        self._shell_link = pythoncom.CoCreateInstance(shell.CLSID_ShellLink, None,
                                               pythoncom.CLSCTX_INPROC_SERVER, shell.IID_IShellLink)
        self._persistant_file = self._shell_link.QueryInterface(pythoncom.IID_IPersistFile)

    def _resolve_shortcut(self, filename):
        """
        [win32] Returns the path refered to by a windows shortcut (.lnk) file.
        """
        # FIXME: this is blocking code
        self._persistant_file.Load(filename)
        # Remark: SLR_NO_UI is used to not display a message box when link
        # cannot be resolved. The second parameter type is DWORD, high order
        # word set to timeout value (ms) for resolving the link.
        self._shell_link.Resolve(0, (500 << 16) | shell.SLR_NO_UI)
        linked_to_file = self._shell_link.GetPath(shell.SLGP_UNCPRIORITY)[0]

        return linked_to_file.decode(locale_helper.system_encoding())

