# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from twisted.trial.unittest import TestCase
from twisted.internet import defer, reactor
from twisted.internet.protocol import Protocol
from twisted.web2.http import Response, StatusResponse
from twisted.web2.channel.http import HTTPFactory, HTTPChannel
from twisted.web2 import iweb, server
from twisted.web2 import responsecode

from zope.interface import implements
from elisa.plugins.weather.report_provider import WeatherReportProvider, \
        WeatherReportError
from elisa.core.media_uri import MediaUri
from elisa.core.config import Config

BASE_URL = 'http://localhost:1234'
# uncomment to test against the real server
#BASE_URL = None

LEBL_DATA =  """\
Barcelona / Aeropuerto, Spain (LEBL) 41-17N 002-04E 6M
May 05, 2008 - 12:30 PM EDT / 2008.05.05 1630 UTC
Wind: from the SSE (160 degrees) at 9 MPH (8 KT) (direction variable):0
Visibility: greater than 7 mile(s):0
Sky conditions: mostly clear
Temperature: 66 F (19 C)
Dew Point: 59 F (15 C)
Relative Humidity: 77%
Pressure (altimeter): 30.00 in. Hg (1016 hPa)
ob: LEBL 051630Z 16008KT 140V200 9999 FEW020 19/15 Q1016 NOSIG
cycle: 16
"""

class TestLeblResource(object):
    implements(iweb.IResource)

    def renderHTTP(self, request):
        # Render the HTTP response. The body of the response simply contains
        # the name of the resource repeated ten times.
        response = Response(responsecode.OK, stream=LEBL_DATA)

        return response

class TestWeatherServerRootResource(object):
    implements(iweb.IResource)

    def __init__(self):
        # Children resources for our test server
        self.my_children = {'': self,
                'LEBL.TXT': TestLeblResource()
            }

    def locateChild(self, request, segments):
        res = self.my_children.get(segments[0])
        if res == self:
            return (self, server.StopTraversal)
        elif res == None:
            return (None, segments)
        else:
            return (res, segments[1:])

    def renderHTTP(self, request):
        response = StatusResponse(responsecode.OK,
                                  'Server\'s root resource: /',
                                  'Elisa test HTTP Server BETA')
        return response

class TestWeatherServerRequest(server.Request):
    site = server.Site(TestWeatherServerRootResource())

class TestWeatherReportProvider(TestCase):
    def setUpClass(self):
        if BASE_URL is None:
            return

        uri = MediaUri(BASE_URL)
        if uri.host != 'localhost':
            return

        self.server_factory = HTTPFactory(TestWeatherServerRequest)
        self.port = reactor.listenTCP(uri.port, self.server_factory)

    def tearDownClass(self):
        if BASE_URL is None:
            return
        
        uri = MediaUri(BASE_URL)
        if uri.host != 'localhost':
            return
        
        return defer.maybeDeferred(self.port.stopListening)

    def setUp(self):
        def set_provider(provider):
            self.provider = provider

            return provider

        config = dict()
        if BASE_URL is not None:
            config['base_url'] = BASE_URL

        dfr = WeatherReportProvider.create(config)
        dfr.addCallback(set_provider)

        return dfr

    def tearDown(self):
        return self.provider.clean()

    def test_get(self):
        def get_callback(report):
            for attribute in ('temperature', 'humidity',
                    'sky', 'country', 'city'):
                self.failUnless(hasattr(report, attribute),
                        'Weather report doesn\'t contain %s' % attribute)

            self.failUnlessEqual(report.country, 'Spain')
            self.failUnlessEqual(report.city, 'Barcelona / Aeropuerto')

        station = MediaUri('metar://LEBL')
        model, dfr = self.provider.get(station)
        dfr.addCallback(get_callback)

        return dfr
    
    def test_get_not_found(self):
        station = MediaUri('metar://i_dont_exist_(in_fact_im_not_even_here)')
        model, dfr = self.provider.get(station)
        self.failUnlessFailure(dfr, WeatherReportError)

        return dfr
        
