# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Olivier Tilloy <olivier@fluendo.com>

"""
Custom button widgets.
"""

from elisa.plugins.pigment.widgets.widget import Widget
from elisa.plugins.pigment.widgets.theme import Theme
from elisa.plugins.pigment.widgets.box import HBox, WidgetNotBoxedError

from elisa.plugins.pigment.graph.image import Image
from elisa.plugins.pigment.graph.text import Text
from elisa.plugins.pigment.graph.group import NodeNotInGroup

import pgm

import logging


class TextButton(Widget):

    """
    A simple button widget with only a text label.
    """

    style_prefix = 'text-button'

    def __init__(self):
        super(TextButton, self).__init__()

        # Invisible background to receive mouse events
        self._background = Image()
        self.add(self._background)
        self._background.size = (1.0, 1.0)
        self._background.position = (0.0, 0.0, 0.0)
        self._background.bg_a = 0
        self._background.visible = True

        self._text = Text()
        self.add(self._text)
        self._text.visible = True

        self._update_style_properties(self._style.get_items())

    def _update_style_properties(self, props=None):
        super(TextButton, self)._update_style_properties(props)

        if props is None:
            return

        for key, value in props.iteritems():
            if key == '%s-width' % self.style_prefix:
                self._text.width = value
            elif key == '%s-height' % self.style_prefix:
                self._text.height = value
            elif key == '%s-x' % self.style_prefix:
                self._text.x = value
            elif key == '%s-y' % self.style_prefix:
                self._text.y = value
            elif key == '%s-z' % self.style_prefix:
                self._text.z = value
            elif key == '%s-alignment' % self.style_prefix:
                self._text.alignment = eval('pgm.TEXT_ALIGN_%s' % \
                                            value.upper())
            elif key == '%s-ellipsize' % self.style_prefix:
                self._text.ellipsize = eval('pgm.TEXT_ELLIPSIZE_%s' % \
                                            value.upper())
            elif key == '%s-weight' % self.style_prefix:
                self._text.weight = eval('pgm.TEXT_WEIGHT_%s' % value.upper())
            elif key == '%s-bg-a' % self.style_prefix:
                self._text.bg_a = value
            elif key == '%s-fg-color' % self.style_prefix:
                self._text.fg_color = value

    def clean(self):
        self._text.clean()
        self._text = None
        self._background.clean()
        self._background = None
        return super(TextButton, self).clean()

    def label__get(self):
        return self._text.label

    def label__set(self, label):
        self._text.label = label

    label = property(label__get, label__set)


class StateButton(Widget):

    """
    A button widget with a background image and a text that has two states:
    when focused, the background image is different than when unfocused.
    Hovering over it with the mouse pointer will change its state.
    """

    def __init__(self, image_resource_unfocused, image_resource_focused, text=''):
        """
        Constructor.

        @param image_resource_unfocused: the image resource to display as
                                         background when unfocused
        @type image_resource_unfocused:  C{str}
        @param image_resource_focused:   the image resource to display as
                                         background when focused
        @type image_resource_focused:    C{str}
        @param text:                     the text label of the button
        @type text:                      C{str}
        """
        super(StateButton, self).__init__()

        theme = Theme.get_default()

        self.bg_unfocused = Image()
        self.add(self.bg_unfocused)
        self.bg_unfocused.layout = pgm.IMAGE_FILLED
        image_file = theme.get_resource(image_resource_unfocused)
        self.bg_unfocused.set_from_file(image_file)
        self.bg_unfocused.size = self.size
        self.bg_unfocused.position = (0.0, 0.0, 0.0)
        self._unfocused_entered_id = self.bg_unfocused.connect('entered',
                                                               self._hover_cb,
                                                               True)
        self._unfocused_left_id = self.bg_unfocused.connect('left',
                                                            self._hover_cb,
                                                            False)

        self.bg_focused = Image()
        self.add(self.bg_focused)
        self.bg_focused.layout = pgm.IMAGE_FILLED
        image_file = theme.get_resource(image_resource_focused)
        self.bg_focused.set_from_file(image_file)
        self.bg_focused.size = self.size
        self.bg_focused.position = (0.0, 0.0, 0.0)
        self._focused_entered_id = self.bg_focused.connect('entered',
                                                           self._hover_cb,
                                                           True)
        self._focused_left_id = self.bg_focused.connect('left',
                                                        self._hover_cb,
                                                        False)

        self.text = Text()
        self.add(self.text)
        self.text.ellipsize = pgm.TEXT_ELLIPSIZE_MIDDLE
        self.text.alignment = pgm.TEXT_ALIGN_CENTER
        self.text.bg_a = 0
        self.text.label = text
        self.text.size = (self.width, self.height * 0.6)
        self.text.position = (0.0, self.height * 0.2, 0.0)
        self.text.visible = True

        self.highlight(False)

    def clean(self):
        self.bg_unfocused.disconnect(self._unfocused_entered_id)
        self.bg_unfocused.disconnect(self._unfocused_left_id)
        self.bg_unfocused.clean()
        self.bg_unfocused = None

        self.bg_focused.disconnect(self._focused_entered_id)
        self.bg_focused.disconnect(self._focused_left_id)
        self.bg_focused.clean()
        self.bg_focused = None

        self.text.clean()
        self.text = None

        return super(StateButton, self).clean()

    def label__get(self):
        return self.text.label

    def label__set(self, value):
        self.text.label = value

    label = property(label__get, label__set)

    def highlight(self, highlight):
        """
        Change the background image depending on the focus.

        @param highlight: whether the button should appear highlighted
        @type highlight:  C{bool}
        """
        self.bg_unfocused.visible = not highlight
        self.bg_focused.visible = highlight

    def do_focus(self, focus):
        self.highlight(focus)

    def _hover_cb(self, button, x, y, z, time, hover):
        if not self.focus:
            self.highlight(hover)


class IconButton(Widget):

    """
    A button widget with an icon and an optional text.
    """

    style_prefix = 'icon-button'

    def __init__(self):
        super(IconButton, self).__init__()

        # Invisible background to receive mouse events
        self._background = Image()
        self.add(self._background)
        self._background.size = (1.0, 1.0)
        self._background.position = (0.0, 0.0, 0.0)
        self._background.bg_a = 0
        self._background.visible = True

        self._container = HBox()
        self.add(self._container, forward_signals=False)
        self._container.visible = True

        self.icon = Image()
        self._container.pack_start(self.icon)
        self.icon.layout = pgm.IMAGE_SCALED
        self.icon.bg_a = 0
        self.icon.visible = True

        self.text = Text()
        self.text.alignment = pgm.TEXT_ALIGN_LEFT
        self.text.ellipsize = pgm.TEXT_ELLIPSIZE_END
        self.text.bg_a = 0
        self.text.visible = True

        self._update_style_properties(self._style.get_items())

    def _update_style_properties(self, props=None):
        super(IconButton, self)._update_style_properties(props)

        if props is None:
            return

        for key, value in props.iteritems():
            if key == '%s-container-x' % self.style_prefix:
                self._container.x = value
            elif key == '%s-container-y' % self.style_prefix:
                self._container.y = value
            elif key == '%s-container-width' % self.style_prefix:
                self._container.width = value
            elif key == '%s-container-height' % self.style_prefix:
                self._container.height = value
            elif key == '%s-container-spacing' % self.style_prefix:
                self._container.spacing = value
            elif key == '%s-icon-only-width' % self.style_prefix:
                self.icon.width = value
            elif key == '%s-icon-with-text-width' % self.style_prefix:
                self.icon.width = value
            elif key == '%s-icon-height' % self.style_prefix:
                self.icon.height = value
            elif key == '%s-text-width' % self.style_prefix:
                self.text.width = value

    def clean(self):
        try:
            self._container.remove(self.text)
        except NodeNotInGroup:
            pass
        self._container.remove(self.icon)
        self.text.clean()
        self.text = None
        self.icon.clean()
        self.icon = None
        self._container.clean()
        self._container = None
        self._background.clean()
        self._background = None
        return super(IconButton, self).clean()

    def _layout_text(self):
        style = self._style.get_items()
        if len(self.label) == 0:
            try:
                self._container.remove(self.text)
            except WidgetNotBoxedError:
                pass
            self.icon.width = style['%s-icon-only-width' % self.style_prefix]
        else:
            if self.text not in self._container:
                self._container.pack_start(self.text)
            self.icon.width = style['%s-icon-with-text-width' % self.style_prefix]
            self.text.width = style['%s-text-width' % self.style_prefix]

    def label__get(self):
        return self.text.label

    def label__set(self, label):
        self.text.label = label
        self._layout_text()

    label = property(label__get, label__set)

    @classmethod
    def _demo_widget(cls, *args, **kwargs):
        theme = Theme.load_from_module('elisa.plugins.poblesec')

        widget = cls()

        resource = theme.get_resource('elisa.plugins.poblesec.directory_settings_add')
        widget.icon.set_from_file(resource)

        widget.label = 'Add'

        widget.size = (0.8, 0.4)
        widget.position = (0.1, 0.1, 0.0)
        widget.visible = True

        def _button_clicked_cb(button, x, y, z, mbutton, time, data):
            logging.debug('button clicked')
            if len(button.label) == 0:
                label = 'Add'
                resource = theme.get_resource('elisa.plugins.poblesec.directory_settings_add')
            else:
                label = ''
                resource = theme.get_resource('elisa.plugins.poblesec.directory_settings_added')
            button.label = label
            button.icon.set_from_file(resource)

        button_clicked_id = widget.connect('clicked', _button_clicked_cb)

        return widget


if __name__ == '__main__':
    logger = logging.getLogger()
    logger.setLevel(logging.DEBUG)

    button = IconButton.demo()
    try:
        __IPYTHON__
    except NameError:
        pgm.main()
