# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Olivier Tilloy <olivier@fluendo.com>

"""
Controller combining an input entry, an on-screen keyboard and a search results
widget to perform generic search queries.
"""

from elisa.core.utils.i18n import install_translation
from elisa.core.input_event import EventValue, EventSource
from elisa.core.media_uri import MediaUri
from elisa.core.common import application
from elisa.core.utils import defer

from elisa.plugins.pigment.pigment_controller import PigmentController
from elisa.plugins.pigment.graph.text import Text
from elisa.plugins.pigment.widgets.const import *

from elisa.plugins.poblesec.widgets.osk import OSK
from elisa.plugins.poblesec.widgets.entry import EntryBar, SearchEntryField
from elisa.plugins.poblesec.widgets.button import StateButton
from elisa.plugins.poblesec.widgets.search_results import SearchResultsWidget

import pgm
from pgm.timing import implicit

import string

import gobject

from twisted.internet import reactor


_ = install_translation('poblesec')


class SearcherEntry(object):

    """
    A searcher entry that should be used by external plugins that implement
    searchers to describe how they should be displayed.
    """

    def __init__(self, searcher, title, icon_resource):
        """
        Constructor.

        @param searcher:      the searcher's entry point
        @type searcher:       C{str}
        @param title:         the title to display for the search results
        @type title:          C{str}
        @param icon_resource: the icon to display for the search results
        @type icon_resource:  C{str}
        """
        self.searcher = searcher
        self.title = title
        self.icon_resource = icon_resource
        self.deferred = None


class SearchController(PigmentController):

    """
    Controller combining an input entry and an on-screen keyboard to perform
    search queries.
    """

    # The 'text-changed' signal is sent when the contents of the entry is
    # updated.
    __gsignals__ = {'text-changed': (gobject.SIGNAL_RUN_LAST,
                                     gobject.TYPE_BOOLEAN,
                                     (str,)),
                   }

    def initialize(self, mediatype):
        """
        Initialize the controller.

        @param mediatype: the type of media searched
        @type mediatype:  C{str}
        """
        self.mediatype = mediatype
        # The list of searchers is going to be populated by the decorators of
        # the plugins that provide them.
        self.searchers = []
        return super(SearchController, self).initialize()

    def set_frontend(self, frontend):
        super(SearchController, self).set_frontend(frontend)

        self.title = Text()
        self.widget.add(self.title)
        self.title.label = _('Enter Your Search Query')
        self.title.fg_color = (255, 255, 255, 255)
        self.title.bg_color = (0, 0, 0, 0)
        self.title.alignment = pgm.TEXT_ALIGN_CENTER
        self.title.size = (1.0, 0.05)
        self.title.position = (0.0, 0.15, 0.0)
        self.title.visible = True
        self._animated_title = implicit.AnimatedObject(self.title)
        self._animated_title.setup_next_animations(duration=300)

        self.result_widget = SearchResultsWidget(frontend, self.mediatype)
        self.widget.add(self.result_widget)
        self.result_widget.size = (0.8, 0.33)
        self.result_widget.position = (0.1, 0.04, 0.0)
        self.result_widget.opacity = 0
        self.result_widget.visible = False
        self._animated_result_widget = \
            implicit.AnimatedObject(self.result_widget)
        self._animated_result_widget.setup_next_animations(duration=300)

        self.entrybar = EntryBar()
        self.widget.add(self.entrybar)
        self.entrybar.size = (1.0, 0.14)
        self.entrybar.position = (0.0, 0.4, 0.0)
        self.entrybar.visible = True

        self.entry = SearchEntryField()
        self.widget.add(self.entry, forward_signals=False)
        self.entry.size = (0.5, 0.08)
        self.entry.position = (0.2, 0.43, 0.0)
        self.entry.visible = True

        unfocused = 'elisa.plugins.poblesec.search_button_unfocused'
        focused = 'elisa.plugins.poblesec.search_button_focused'
        self.button = StateButton(unfocused, focused)
        self.widget.add(self.button, forward_signals=False)
        self.button.label = _('Enter')
        self.button.size = (0.08, 0.06)
        self.button.position = (0.71, 0.44, 0.0)
        self.button.visible = True

        self._signal_handler_ids = []

        theme = frontend.get_theme()
        alphabetic = 'elisa.plugins.poblesec.osk_alphabetic'
        special_chars = 'elisa.plugins.poblesec.osk_numeric_special_chars'
        self.keyboards = [OSK(theme.get_resource(alphabetic)),
                          OSK(theme.get_resource(special_chars))]
        self._animated_keyboards = []
        for keyboard in self.keyboards:
            self.widget.add(keyboard)
            keyboard.size = (0.6, 0.45)
            keyboard.position = (0.2, 0.55, 0.0)
            keyboard.opacity = 0
            keyboard.visible = True
            char_id = keyboard.connect('key-press-char', self._key_pressed_cb)
            self._signal_handler_ids.append((keyboard, char_id))
            special_id = keyboard.connect('key-press-special',
                                          self._special_key_pressed_cb)
            self._signal_handler_ids.append((keyboard, special_id))
            animated_keyboard = implicit.AnimatedObject(keyboard)
            animated_keyboard.setup_next_animations(duration=300)
            self._animated_keyboards.append(animated_keyboard)
        self.current_keyboard = 0
        self._animated_keyboards[self.current_keyboard].opacity = 255

        # Select the switch key; this works because we know it is the last one
        # for each keyboard.
        for i in xrange(1, len(self.keyboards)):
            keyboard = self.keyboards[i]
            keyboard.select_key(keyboard.keys[-1][-1])

        result_clicked_id = self.result_widget.connect('clicked',
                                                     self._result_clicked_cb)
        self._signal_handler_ids.append((self.result_widget,
                                         result_clicked_id))

        result_focus_id = self.result_widget.connect('focus',
                                                     self._on_result_focus)
        self._signal_handler_ids.append((self.result_widget, result_focus_id))

        entry_focus_id = self.entry.connect('focus', self._on_entry_focus)
        self._signal_handler_ids.append((self.entry, entry_focus_id))

        entrybar_clicked_id = self.entrybar.connect('clicked',
                                                    self._entrybar_clicked_cb)
        self._signal_handler_ids.append((self.entrybar, entrybar_clicked_id))

        button_clicked_id = self.button.connect('clicked',
                                                self._button_clicked_cb)
        self._signal_handler_ids.append((self.button, button_clicked_id))

        button_focus_id = self.button.connect('focus', self._on_button_focus)
        self._signal_handler_ids.append((self.button, button_focus_id))

        widget_focus_id = self.widget.connect('focus', self._on_focus)
        self._signal_handler_ids.append((self.widget, widget_focus_id))

        live_search_id = self.connect('text-changed', self._live_search_cb)
        self._signal_handler_ids.append((self, live_search_id))
        self._live_search = None

    def clean(self):
        for widget, signal_id in self._signal_handler_ids:
            widget.disconnect(signal_id)

        self._animated_title.stop_animations()
        self._animated_title = None
        self.title.clean()
        self.title = None
        
        self._animated_result_widget.stop_animations()
        self._animated_result_widget = None
        self.result_widget.clean()
        self.result_widget = None

        self.entrybar.clean()
        self.entrybar = None

        self.entry.clean()
        self.entry = None

        self.button.clean()
        self.button = None

        for animated_keyboard in self._animated_keyboards:
            animated_keyboard.stop_animations()
        self._animated_keyboards = []
        for keyboard in self.keyboards:
            keyboard.clean()
        self.keyboards = []

        if (self._live_search is not None) and \
            not self._live_search.called and \
            not self._live_search.cancelled:
            self._live_search.cancel()

        self.searchers = []

        return super(SearchController, self).clean()

    def _on_focus(self, widget, focus):
        if focus:
            if self.result_widget.opacity > 0:
                self.result_widget.focus = True
            else:
                self.entry.focus = True

    def _result_clicked_cb(self, button, x, y, z, mbutton, time, data):
        self.result_widget.focus = True

    def _on_result_focus(self, widget, focus):
        if focus:
            self._animated_keyboards[self.current_keyboard].opacity = 100
        elif self.keyboards[self.current_keyboard].focus or \
            self.entrybar.focus or self.entry.focus or self.button.focus:
            self._animated_keyboards[self.current_keyboard].opacity = 255

    def _on_entry_focus(self, widget, focus):
        if focus:
            self.entrybar.highlight(focus)
        elif not self.button.focus:
            self.entrybar.highlight(focus)

    def _entrybar_clicked_cb(self, button, x, y, z, mbutton, time, data):
        if not self.button.focus:
            self.entry.focus = True

    def _button_clicked_cb(self, button, x, y, z, mbutton, time, data):
        self.button.focus = True
        self._search(focus=True)

    def _on_button_focus(self, widget, focus):
        if focus:
            self.entrybar.highlight(focus)
        elif not self.entry.focus:
            self.entrybar.highlight(focus)

    def _switch_clicked_cb(self, button, x, y, z, mbutton, time, data):
        self._switch_keyboard()

    def _switch_keyboard(self):
        self._animated_keyboards[self.current_keyboard].opacity = 0
        self.current_keyboard = \
            (self.current_keyboard + 1) % len(self.keyboards)
        self.keyboards[self.current_keyboard].focus = True
        self._animated_keyboards[self.current_keyboard].opacity = 255

    def _key_pressed_cb(self, keyboard, value):
        self.entry.text += value
        self.emit('text-changed', self.entry.text)

    def _special_key_pressed_cb(self, keyboard, value):
        if value == 'delete':
            self.entry.text = self.entry.text[:-1]
            self.emit('text-changed', self.entry.text)
        elif value == 'switch_keyboard':
            self._switch_keyboard()

    def _search(self, focus=False):
        if (self._live_search is not None) and \
            not self._live_search.called and \
            not self._live_search.cancelled:
            self._live_search.cancel()

        search_string = self.entry.text.strip()
        dfrs = []

        def got_search_result(result_model, searcher_entry):
            if focus:
                self.result_widget.focus = True
            self.result_widget.update(searcher_entry, result_model)
            self.result_widget.set_loading(searcher_entry, False)

        first_search = not self.result_widget.visible
        if first_search:
            self.result_widget.opacity = 0
            self.result_widget.visible = True
            self.result_widget.set_searchers(self.searchers)

        for searcher_entry in self.searchers:
            deferred = searcher_entry.deferred
            # TODO: Cancel the current request if there is one
            #if deferred is not None:
            #    try:
            #        deferred.cancel()
            #    except defer.AlreadyCalledError:
            #        pass
            self.result_widget.set_loading(searcher_entry, True)
            searcher = searcher_entry.searcher.lower()
            uri = MediaUri('elisa://search/%s/%s?only_searchers=%s' % \
                           (self.mediatype, search_string, searcher))
            model, dfr = application.resource_manager.get(uri)
            if first_search:
                self.result_widget.update(searcher_entry, model)
            dfr.addCallback(got_search_result, searcher_entry)
            dfrs.append(dfr)
            searcher_entry.deferred = dfr

        if first_search:
            self._animated_title.opacity = 0
            self._animated_result_widget.opacity = 255

        return defer.DeferredList(dfrs)

    def _live_search_cb(self, search_controller, text):
        if (self._live_search is not None) and \
            not self._live_search.called and \
            not self._live_search.cancelled:
            self._live_search.cancel()
        if len(text.strip()) >= 3:
            self._live_search = reactor.callLater(1, self._search)

    def handle_input(self, manager, input_event):
        keyboard = self.keyboards[self.current_keyboard]
        su = super(SearchController, self)
        if keyboard.focus:
            if input_event.value in (EventValue.KEY_OK, EventValue.KEY_RETURN):
                keyboard.activate_key(keyboard.current_key)
            elif input_event.value == EventValue.KEY_GO_LEFT:
                keyboard.move_selector(LEFT)
            elif input_event.value == EventValue.KEY_GO_RIGHT:
                keyboard.move_selector(RIGHT)
            elif input_event.value == EventValue.KEY_GO_UP:
                top_row = keyboard.keys[0]
                if keyboard.current_key in top_row:
                    self.entry.focus = True
                else:
                    keyboard.move_selector(TOP)
            elif input_event.value == EventValue.KEY_GO_DOWN:
                keyboard.move_selector(BOTTOM)
            else:
                return su.handle_input(manager, input_event)
            return True
        elif self.entry.focus:
            if input_event.value in (EventValue.KEY_OK, EventValue.KEY_RETURN):
                self._search(focus=True)
            elif str(input_event.value)[4:] in \
                 (string.ascii_letters + string.digits):
                self._key_pressed_cb(keyboard, str(input_event.value)[4:])
            elif input_event.value == EventValue.KEY_SPACE:
                self._key_pressed_cb(keyboard, ' ')
            elif (input_event.value == EventValue.KEY_MENU) and \
                 (input_event.source == EventSource.KEYBOARD):
                self._special_key_pressed_cb(keyboard, 'delete')
            elif input_event.value == EventValue.KEY_GO_DOWN:
                keyboard.focus = True
            elif input_event.value == EventValue.KEY_GO_UP:
                if self.result_widget.opacity > 0:
                    self.result_widget.focus = True
                else:
                    return su.handle_input(manager, input_event)
            elif input_event.value == EventValue.KEY_GO_RIGHT:
                self.button.focus = True
            return True
        elif self.button.focus:
            if input_event.value in (EventValue.KEY_OK, EventValue.KEY_RETURN):
                self._search(focus=True)
            elif input_event.value == EventValue.KEY_GO_DOWN:
                keyboard.focus = True
            elif input_event.value == EventValue.KEY_GO_UP:
                if self.result_widget.opacity > 0:
                    self.result_widget.focus = True
                else:
                    return su.handle_input(manager, input_event)
            elif input_event.value == EventValue.KEY_GO_LEFT:
                self.entry.focus = True
            else:
                return su.handle_input(manager, input_event)
            return True
        elif self.result_widget.focus or \
            (self.result_widget.focus_child is not None):
            if not self.result_widget.handle_event(input_event.value):
                if input_event.value == EventValue.KEY_GO_DOWN:
                    self.entry.focus = True
                else:
                    return su.handle_input(manager, input_event)
            return True
