# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Benjamin Kampmann <benjamin@fluendo.com>

"""
test the media scanning
"""

from twisted.trial.unittest import TestCase

from elisa.plugins.database.media_scanner import MediaScanner
from elisa.plugins.base.models.file import FileModel, DirectoryModel

from elisa.plugins.database.scanner_models import ScanResource 

from twisted.internet import defer, reactor

from elisa.core import common

from elisa.core.media_uri import MediaUri

from collections import deque

import os

class FakeResourceManager(object):
    tree = {"root": { 'usr' : '',
                   'home' : {'meeko' : '',
                             'chubaka' : '',
                             'ben' : '',
                            },
                   'lib': '',
                },
            "usr" : {'lib' : {'doc' : '',
                              'share' : '',
                              'python' : ''},
                     'share' : {'python' : '',
                                'usr' : '',
                                'test' : '42',
                                },
                    'log' : '',
                    'lock' : '',
                },
            "var" : {'log' : {'x11': '',
                              'gdm': '',
                              'test': '' },
                     'empty' : {},
                     'file1' : '',
                     'file3' : '',
                },   

            }

    def get(self, uri, context=None):
        if not context:
            context = DirectoryModel()
            context.uri = uri

        path = uri.path.split('/')[1:]
        if uri.filename == '':
            path.pop(-1)

        result_dict = self.tree
        for key in path:
            result_dict = result_dict[key]

        for key, children in result_dict.iteritems():
            cur_uri = MediaUri("%s%s" % (uri, key))
            if isinstance(children, dict):
                d = DirectoryModel()
                d.uri = cur_uri
                context.files.append(d)
            else:
                f = FileModel()
                f.uri = cur_uri
                context.files.append(f)
        return (context, defer.succeed(context))

class Application(object):
    pass

class DummyParser(object):
    def query_model(self, *args):
        return defer.succeed(None)
    def mark_deleted(self, *args):
        return defer.succeed(None)
    def delete_files(self, *args):
        return defer.succeed(None)

class DummyResult(object):
    result = None
    def get_all(self):
        return defer.succeed(self.result)

class DummyStore(object):
    def execute(self, *args):
        result = True
        if args[0].startswith('PRAGMA'):
            result = DummyResult()
            result.result = [(0,'playcount'),(1,'last_played'),\
                              (3,'shot_time'),(2,'release_date')]
        return defer.succeed(result)
    def commit(self, *args):
        return defer.succeed(True)

class DummyConfig(object):
    def get_section(self, name):
        return {}

class TestMediaScanner(TestCase):
    """
    Test the media scanning part
    """
    stat_uri = MediaUri('media_scanner://localhost/statistic')
    queue_uri = MediaUri('media_scanner://localhost/queue')
    # rescan after ~ 4 secs
    config = {'delay' : 0.2, 'scan_every' : 0.001}

    def setUp(self):
        self._dbus_init = MediaScanner._initialize_dbus
        self._dbus_clean = MediaScanner._clean_dbus

        MediaScanner._initialize_dbus = lambda x,y :x
        MediaScanner._clean_dbus = lambda x :x

        def set_scanner(result):
            self.scanner = result
            # overwrite the Parser and stop the old one
            parser = self.scanner.parser
            self.scanner.parser = DummyParser()
            self.scanner._scan_directories = lambda: None
            self.scanner.local_resource = FakeResourceManager()
            return parser.clean()

        def create(result):
            return MediaScanner.create(self.config).addCallback(set_scanner)
    
        return self.patch_application().addCallback(create)

    def patch_application(self):
        self.app = common.application
        common.application = Application()
#        common.application.resource_manager = FakeResourceManager()
        common.application.store = DummyStore()
        common.application.config = DummyConfig()
        return defer.succeed(self)

    def tearDown(self):
        def reset_dbus(result):
            MediaScanner._initialize_dbus = self._dbus_init
            MediaScanner._clean_dbus = self._dbus_clean
            return result

        common.application = self.app

        dfr  = self.scanner.clean()
        dfr.addCallback(reset_dbus)
        return dfr

    def test_get_stats(self):
        model, dfr = self.scanner.get(MediaUri(self.stat_uri))
        self.assertEquals(model, self.scanner.scan_stat)

    def test_recursive_scan(self):
        s = ScanResource()
        s.root_uri = MediaUri("file:///root/")


        def check_result(result):
            stat, dfr = self.scanner.get(self.stat_uri)
            self.assertFalse(self.scanner.running)
            self.assertFalse(stat.running)

            self.assertEquals(s.files_scanned, 5)
            self.assertEquals(s.files_failed, [])
            return dfr

        return self.scanner._scan_recursive(s).addCallback(check_result)

    def test_scan_auto_restart(self):

        # we don't really want to scan. we just emulate it
        wait_dfr = defer.Deferred()
        self.scanner._scan_recursive = lambda x: wait_dfr

        # fake the _rescan method
        self._rescan_called = False
        def rescan(*args):
            self._rescan_called = True

        self.scanner._rescan = rescan

        # let's assume, we are the last one to scan
        while not self.scanner.scan_stat.queue.empty():
            self.scanner.scan_stat.queue.get()

        s = ScanResource()
        s.root_uri = MediaUri("file:///root/")
        
        self.scanner.scan_stat.queue.put(s)
            
        # start the 'scanning' process
        self.scanner._scan()

        scan_dfr = self.scanner._current_scan_deferred
        wait_dfr.callback(s)

        def check(result):
            self.assertTrue(self._rescan_called)

        def wait_for_second_scan(result):
            # rescan was not called yet!
            self.assertFalse(self._rescan_called)

            dfr = defer.Deferred()
            dfr.addCallback(check)

            reactor.callLater(5, dfr.callback, None)

            return dfr

        scan_dfr.addCallback(wait_for_second_scan)
        return scan_dfr
    
    test_scan_auto_restart.timeout = 10

    def test_reschedule(self):
        scan_stat = self.scanner.scan_stat
        for i in xrange(5):
            scan_stat.scanned.append('Source %s' % i)

        self.assertTrue(scan_stat.queue.empty())

        self.scanner._reschedule_scanned()

        # scanned is cleared
        self.assertEquals(scan_stat.scanned, [])

        for i in xrange(5):
            self.assertEquals(scan_stat.queue.get_nowait(), "Source %s" % i)
        
        self.assertTrue(scan_stat.queue.empty())
        

    def test_put_deleting(self):
        """
        Try to put some requests into the media scanner and check if it works
        """
        marked = []

        # we don't want them really start to scan, so overwrite the later
        # processes
        self.scanner._file_found = lambda model, stat: defer.succeed(model)
        self.scanner._count_files = lambda path: defer.succeed(19)
        self.scanner._scan = lambda: None

        stat, dfr = self.scanner.get(self.stat_uri)

        # overwrite the mark_deleted method
        self.scanner.parser.mark_deleted = lambda x: defer.succeed(marked.append(x))
        dfrs = []
        for name in ('root', 'usr', 'var'):
            dfrs.append(self.scanner.put(MediaUri('file:///%s' % name),
                    self.queue_uri))

        # as soon as they are put, they should be marked as deleted
        self.assertEquals(marked, [u'/root/', u'/usr/', u'/var/'])

    test_put_deleting.timeout = 4


class TestFileCounting(TestCase):

    tree = {'test' : {'test' : None,
                      'drei' : {'vier' : None,
                                'fünf' : None,
                                'sechs': None,
                                },
                     'sieben' : None,
                     'acht' : { 'neun' : None,
                                'one' : None,
                                'two' : None,
                                'three' : None},
                    },
            'fuffzig' : {'four' : None,
                         'five' : None,
                         'six' : None,
                         'seven' : None,
            },
    }


    
    def setUp(self):
        self.counter_path = os.path.abspath('counter_dir')
        self._create_structure(self.counter_path, self.tree)

        # we only test on certain method, so it is enough to initiate
        self.scanner = MediaScanner()

    def _create_structure(self, path, values):
        if not isinstance(values, dict):
            # nothing. it is a file
            open(path,'w').close()
        else:
            os.makedirs(path)
            for key, value in values.iteritems():
                new_path = os.path.join(path, key)
                self._create_structure(new_path, value)

    def test_counter(self):
        dfr = self.scanner._count_files(self.counter_path)
        dfr.addCallback(self.assertEquals, 13)
        return dfr
        

