/*
 * Easy Network
 * Copyright (c) <2009>, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */
#include "ProfileManagementControl.h"

#include <wx/dir.h>
#include <wx/file.h>
#include "ENMUtil.h"
#include "DataCenter.h"
#include "ENMTaskTray.h"

#include "wlanacmevent.h"
#include "ProfileEvent.h"
#include <fstream>

#if defined(__UNIX__)
#include <sys/stat.h>
#endif

using namespace std;

#if defined(__WXMSW__)
static wxString logFile(_T("C:\\ProfileController.log"));
#else
static wxString logFile(_T("/tmp/ProfileController.log"));
#endif

#if defined(__WXMSW__)
//const wxString ProfileManagementControl::SYSTEM_PROFOLE_DIR = ENMUtil::GetAllUserAppDataFolder() + _T("\\Network Neighborhood");
const wxString ProfileManagementControl::SYSTEM_PROFOLE_DIR = _T("C:\\Documents and Settings\\Default User\\Application Data\\Network Neighborhood");
const wxString ProfileManagementControl::USER_PROFILE_DIR = ENMUtil::GetCurrentUserAppDataFolder() + _T("\\Network Neighborhood");
const wxChar ProfileManagementControl::PATH_SEPARATOR(_T('\\'));
#endif

#if defined(__UNIX__)
const wxString ProfileManagementControl::SYSTEM_PROFOLE_DIR(_T("/etc/Network Neighborhood"));
const wxString ProfileManagementControl::USER_PROFILE_DIR = ENMUtil::GetCurrentUserAppDataFolder() + _T("/Network Neighborhood");
const wxChar ProfileManagementControl::PATH_SEPARATOR(_T('/'));
#endif

const int ProfileManagementControl::LAN_CONNECT_TIMER_ID = ::wxNewId();
const int ProfileManagementControl::APPLY_TIMER_ID = ::wxNewId();
const int ProfileManagementControl::PREPARE_TIMER_ID = ::wxNewId();
const long ProfileManagementControl::ID_TIMER_SETSTATICIP = wxNewId();
#if defined(__WXMSW__)
const int ProfileManagementControl::ADDRESS_TIMER_ID = ::wxNewId();
#endif

BEGIN_EVENT_TABLE(ProfileManagementControl, wxEvtHandler)
    EVT_TIMER(LAN_CONNECT_TIMER_ID, ProfileManagementControl::OnTimer)
    EVT_TIMER(APPLY_TIMER_ID, ProfileManagementControl::OnTimer)
    EVT_TIMER(PREPARE_TIMER_ID, ProfileManagementControl::OnTimerPrepare)
    EVT_TIMER(ID_TIMER_SETSTATICIP, ProfileManagementControl::OnStaticIpTimer)
#if defined(__WXMSW__)
    EVT_TIMER(ADDRESS_TIMER_ID, ProfileManagementControl::OnTimer)
#endif
    EVT_WLAN_ACM(ID_WLAN_ACM_CONNECT, ProfileManagementControl::OnWlanConnect)
    EVT_SWITCH_ADAPTER(ID_SWITCH_ADAPTER, ProfileManagementControl::OnAdapterSwitch)
    EVT_WIRELESS(ID_WIRELESS_NETWORK_SWITCH, ProfileManagementControl::OnWirelessNetworkChanged)
END_EVENT_TABLE()

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
ProfileManagementControl::ProfileManagementControl() : wxEvtHandler()
        , m_timerApply(this, APPLY_TIMER_ID)
        , m_timerLanConnect(this, LAN_CONNECT_TIMER_ID)
        , m_timerPrepare(this, PREPARE_TIMER_ID)
        , m_SetStaticIp(this, ID_TIMER_SETSTATICIP)
#if defined(__WXMSW__)
        , m_timerAddress(this, ADDRESS_TIMER_ID)
#endif
{
    m_iApplyDelay = 0;
    m_iCurrentProfileIndex = -1;
    m_setIpCounter = 0;
    m_pToConnect = NULL;
    m_isBusy = false;
    m_bIfStaticIp = false;
    m_status = WAITING;
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
ProfileManagementControl::~ProfileManagementControl()
{
    m_systemProfiles.clear();
    m_userProfiles.clear();
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
wxString ProfileManagementControl::GetCurrentProfile() const
{
    return m_strCurrentProfile;
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
const ProfileList_t& ProfileManagementControl::GetCurrentUsedProfile() const
{
    ENMUtil::Log(logFile, _T("Enter ProfileManagementControl::GetCurrentUsedProfile"));
    #if defined(__WXMSW__)
    ENMUtil::Log(logFile, wxString::Format(_T("Return ProfileManagementControl::GetCurrentUsedProfile : %s\n"), wxString::FromUTF8(m_profileToApply.sProfileName.c_str())));
    #endif
    return m_profileToApply;
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
int ProfileManagementControl::GetCurrentProfileIndex() const
{
    return m_iCurrentProfileIndex;
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
int ProfileManagementControl::GetProfilePriority(const ProfileList_t& profile)
{
	long lPriority = -1;
	profile.sPriority.ToLong(&lPriority);
	return (int)lPriority;
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
#if defined(__UNIX__)
int ProfileManagementControl::ContinueApplyProfile(ProfileList_t& profile)
{
    ENMUtil::Log(wxT("/tmp/ENMLog.txt"), wxT("ContinueApplyProfile--begin"));
    int result = 0;

    if ( profile.stOtherSet.NetworkSecurity == _T("1") )
    {
        if ( profile.stOtherSet.stNetSecurity.sForbidShareInternet == _T("1") )
        {
            ENMUtil::Log(wxT("/tmp/ENMLog.txt"), wxT("here---------->false"));
            DCS_EnableInternetSharing(false);
        }
        else
        {
            ENMUtil::Log(wxT("/tmp/ENMLog.txt"), wxT("here---------->end"));
            DCS_EnableInternetSharing(true);
        }
    }

    if ( profile.stOtherSet.AutoRunAppFlag == _T("1") )
    {
        vector<wxString> appList = profile.stOtherSet.AutoRunApp;
        for ( vector<string>::size_type i = 0; i < profile.stOtherSet.AutoRunApp.size(); i++ )
        {
            wxString app = profile.stOtherSet.AutoRunApp.at(i);
            ENMUtil::AutoRunApp(app);
        }
    }

    int adapterType = ADAPTER_NONE;
    if ( profile.stGenernalSet.ConnecType == _T("lan") )
        adapterType = ADAPTER_WIRED;
    else
        adapterType = ADAPTER_WIREDLESS;
    wxString ip = _T("");
    wxString mask = _T("");
    wxString gateway = _T("");
    wxString dns = _T("");
    if ( profile.stOtherSet.IPDNSFlag == _T("1") )
    {
        ip = profile.stOtherSet.stIPDNSSet.IPvalue;
        mask = profile.stOtherSet.stIPDNSSet.SubnetAddress;
        gateway = profile.stOtherSet.stIPDNSSet.NetGateway;
        dns = profile.stOtherSet.stIPDNSSet.DNSvalue;
        if (!ip.IsEmpty())
        {
            ENM_String* sKill = new ENM_String();
            memset(sKill, 0, sizeof(ENM_String));
            strcpy((char*)sKill->dcsString, "killall");
            DCS_WriteFile(sKill);
            delete sKill;
            ENMUtil::Log(logFile, wxT("begin ChangeIPAddress"));
            m_bIfStaticIp = true;
            ENMUtil::ChangeIPAddress(ip, mask, gateway, dns, adapterType, _T(""));
            m_SetStaticIp.Start(2000, wxTIMER_CONTINUOUS);
            ENMUtil::Log(logFile, wxT("end ChangeIPAddress"));
        }
    }
    if ( profile.stOtherSet.IPDNSFlag == _T("0") && profile.stGenernalSet.ConnecType == _T("lan") )
    {
        ENMUtil::Log(logFile, wxT("begin dhcp"));
        ENM_String* sDhclient = new ENM_String();
        memset(sDhclient, 0, sizeof(ENM_String));
        strcpy((char*)sDhclient->dcsString, "dhcp");
        DCS_WriteFile(sDhclient);
        delete sDhclient;
        ENMUtil::Log(logFile, wxT("end dhcp"));
    }
    if ( profile.stOtherSet.IsHomepage != _T("1") && profile.stOtherSet.ProxyFlag != _T("1") )
        return result;

    int confirm = wxID_YES;
    while ( ENMUtil::IsFirefoxRunning() && confirm == wxID_YES )
    {
        wxMessageDialog msg(NULL, _("Firefox is running. \nPlease shut down Firefox and press Yes."), wxEmptyString, wxYES_NO | wxICON_QUESTION);
        confirm = msg.ShowModal();
    }
    if ( confirm == wxID_YES )
    {
        if ( profile.stOtherSet.IsHomepage == _T("1") )
        {
            ENMUtil::OverwriteHomePage(profile.stOtherSet.Homepage);
        }
        if ( profile.stOtherSet.ProxyFlag == _T("1") )
        {
            ENMUtil::OverwriteProxy(profile.stOtherSet.stProxySet.Proxy, profile.stOtherSet.stProxySet.Port);
        }
    }
    ENMUtil::Log(wxT("/tmp/ENMLog.txt"), wxT("ContinueApplyProfile--end"));
    return result;
}
#endif // __UNIX__

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
#if defined(__WXMSW__)
int ProfileManagementControl::ContinueApplyProfile(ProfileList_t& profile)
{
    int result = 0;
    if ( profile.stOtherSet.NetworkSecurity == _T("1") )
    {
        if ( profile.stOtherSet.stNetSecurity.sForbidShareInternet == _T("1") )
        {
            ENMUtil::DisableInternetSharing();
        }
        else
        {
            ENMUtil::EnableInternetSharing();
        }

        if ( profile.stOtherSet.stNetSecurity.sForbidShareFileAndPrint == _T("1") )
        {
            ENMUtil::DisableFileAndPrint();
        }
        else
        {
            ENMUtil::EnableFileAndPrint();
        }

        if ( profile.stOtherSet.stNetSecurity.sStartWindowFirewall == _T("1") )
        {
            ENMUtil::StartupWindowsFirewall();
        }
        else
        {
            ENMUtil::StopWindowsFirewall();
        }
    }

    if ( profile.stOtherSet.AutoRunAppFlag == _T("1") )
    {
        vector<wxString> appList = profile.stOtherSet.AutoRunApp;
        for ( vector<wxString>::size_type i = 0; i < profile.stOtherSet.AutoRunApp.size(); i++ )
        {
            wxString app = profile.stOtherSet.AutoRunApp.at(i);
            ENMUtil::AutoRunApp(app);
        }
    }

    if ( profile.stOtherSet.IsHomepage == _T("1") )
    {
        ENMUtil::OverwriteHomePage(profile.stOtherSet.Homepage);
    }
    if ( profile.stOtherSet.ProxyFlag == _T("1") )
    {
        ENMUtil::OverwriteProxy(profile.stOtherSet.stProxySet.Proxy, profile.stOtherSet.stProxySet.Port);
    }

    int adapterType = ADAPTER_NONE;
    if ( profile.stGenernalSet.ConnecType == _T("lan") )
        adapterType = ADAPTER_WIRED;
    else
        adapterType = ADAPTER_WIREDLESS;
    wxString ip = _T("");
    wxString mask = _T("");
    wxString gateway = _T("");
    wxString dns = _T("");
    if ( profile.stOtherSet.IPDNSFlag == _T("1") )
    {
        ip = profile.stOtherSet.stIPDNSSet.IPvalue;
        mask = profile.stOtherSet.stIPDNSSet.SubnetAddress;
        gateway = profile.stOtherSet.stIPDNSSet.NetGateway;
        dns = profile.stOtherSet.stIPDNSSet.DNSvalue;
    }
    ENMUtil::ChangeIPAddress(ip, mask, gateway, dns, adapterType, _T("ip"));
    m_timerAddress.Start(25000, wxTIMER_ONE_SHOT);
    return result;
}
#endif // __WXMSW__

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
#if defined(__WXMSW__)
bool ProfileManagementControl::ContinueSetAddress(ProfileList_t& profile)
{
    int adapterType = ADAPTER_NONE;
    if ( profile.stGenernalSet.ConnecType == _T("lan") )
        adapterType = ADAPTER_WIRED;
    else
        adapterType = ADAPTER_WIREDLESS;
    wxString ip = _T("");
    wxString mask = _T("");
    wxString gateway = _T("");
    wxString dns = _T("");
    if ( profile.stOtherSet.IPDNSFlag == _T("1") )
    {
        ip = profile.stOtherSet.stIPDNSSet.IPvalue;
        mask = profile.stOtherSet.stIPDNSSet.SubnetAddress;
        gateway = profile.stOtherSet.stIPDNSSet.NetGateway;
        dns = profile.stOtherSet.stIPDNSSet.DNSvalue;
    }
    ENMUtil::ChangeIPAddress(ip, mask, gateway, dns, adapterType, _T("dns"));
    return true;
}
#endif


/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
bool ProfileManagementControl::GetUserProfiles(vector<ProfileList_t>& newVector, bool isAdminUser, bool reload)
{
    if ( reload && !LoadProfiles(isAdminUser) )
        return false;
    newVector.clear();
    newVector.insert(newVector.begin(), m_systemProfiles.begin(), m_systemProfiles.end());
    newVector.insert(newVector.end(), m_userProfiles.begin(), m_userProfiles.end());
    return true;
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
bool ProfileManagementControl::GetUserProfiles(vector<ProfileList_t>& newVector, vector<ProfileList_t>& CurrentUserVector, bool isAdminUser, bool reload)
{
    if ( reload && !LoadProfiles(isAdminUser) )
        return false;
    newVector.clear();
    newVector.insert(newVector.begin(), m_systemProfiles.begin(), m_systemProfiles.end());
    newVector.insert(newVector.end(), m_userProfiles.begin(), m_userProfiles.end());

    CurrentUserVector.clear();
    if ( isAdminUser )
    {
        CurrentUserVector.insert(CurrentUserVector.begin(), m_systemProfiles.begin(), m_systemProfiles.end());
    }
    else
    {
        CurrentUserVector.insert(CurrentUserVector.end(), m_userProfiles.begin(), m_userProfiles.end());
    }
    return true;
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
int ProfileManagementControl::GetAdminProfileCount()
{
    return (int)m_systemProfiles.size();
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
bool ProfileManagementControl::ChangePriority(int oldPriority, bool increase)
{
    size_t index = oldPriority - 1;
    vector<ProfileList_t>* pVector = &m_systemProfiles;
    if ( index >= m_systemProfiles.size() )
    {
        index = oldPriority - m_systemProfiles.size();
        pVector = &m_userProfiles;
    }

    if ( increase && index <= 0 )
        return false;

    if ( !increase && index >= pVector->size() - 1 )
        return false;

    vector<ProfileList_t>::iterator iter = pVector->begin();
    while ( index > 0 )
    {
        iter++;
        index--;
    }
    if ( increase )
    {
        vector<ProfileList_t>::iterator other = iter - 1;
        other = pVector->insert(other, *iter);
        pVector->erase(other + 2);
    }
    else
    {
        vector<ProfileList_t>::iterator other = iter + 1;
        iter = pVector->insert(iter, *other);
        pVector->erase(iter + 2);
    }
    return true;
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
bool ProfileManagementControl::ReadProfile(ProfileList_t& profile, const wxString& fileName)
{
    if ( !wxFile::Exists(fileName) )
        return false;

    bool result = m_editor.ReadProfile(profile, fileName) == 1;
    if ( result )
    {
        wxString profileName = fileName.SubString(fileName.Find(PATH_SEPARATOR, true) + 1, fileName.Find('.', true) - 1);
        profile.sProfileName = profileName;
    }
    return result;
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
int ProfileManagementControl::CheckWirelessProfileExist(const APStatus& network)
{
    bool isAdminUser = ENMUtil::IsAdministrator();
    if ( !LoadProfiles(isAdminUser) )
        return 0;

    vector<ProfileList_t>* pVector = NULL;
    if ( isAdminUser )
    {
        pVector = &m_systemProfiles;
    }
    else
    {
        pVector = &m_userProfiles;
    }
    bool ssidExist = false;
    bool fileExist = false;
    int maxPriority = 0;
	wxString ssid = wxString::FromUTF8((char*)network.ssid.ucSsid, network.ssid.length);
    vector<ProfileList_t>::iterator iter = pVector->begin();
    while ( iter != pVector->end() )
    {
        ProfileList_t profile = *iter;
        ssidExist |= profile.stWirelessSet.SSID == ssid;
        wxString wxstrProfileName = profile.sProfileName;
        wxString wxstrSsid = wxString::FromUTF8((char*)network.ssid.ucSsid, network.ssid.length);
        fileExist |= ENMUtil::FilenameEquals(wxstrProfileName, wxstrSsid);
        long p = 0;
		profile.sPriority.ToLong(&p);
        maxPriority = p > maxPriority ? p : maxPriority;
        iter++;
    }
    int result = 0x01000000;
    if ( ssidExist ) result |= 0x00100000;
    if ( fileExist ) result |= 0x00010000;
    result |= (maxPriority & 0x0000FFFF);
    return result;
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
bool ProfileManagementControl::InsertNewWirelessProfile(ProfileList_t& profile)
{
    wxString strFileName;
    if ( ENMUtil::IsAdministrator() )
    {
        strFileName = SYSTEM_PROFOLE_DIR + PATH_SEPARATOR + profile.sProfileName + _T(".profile");
    }
    else
    {
        strFileName = USER_PROFILE_DIR + PATH_SEPARATOR + profile.sProfileName + _T(".profile");
    }
	long p = 0;
	profile.sPriority.ToLong(&p);
    if ( m_editor.WriteProfile(profile, strFileName, p) != 1 )
        return false;

    ProfileEvent profileEvent(ID_PROFILE_CHANGED, PEVT_PROFILE_ADD);
    wxEvtHandler* pEvtHandler = NULL;
    pEvtHandler = (wxEvtHandler*) DataCenter::GetInstance()->GetNetworkManagerDlgHandle();
    ::wxPostEvent(pEvtHandler, profileEvent);
    pEvtHandler = (wxEvtHandler*) DataCenter::GetInstance()->GetProfileManagerDlgHandle();
    ::wxPostEvent(pEvtHandler, profileEvent);
    return true;
    return true;
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
bool ProfileManagementControl::ImportProfile(ProfileList_t& profile)
{
    wxString strFileName;
    int priority = 0;
    if ( ENMUtil::IsAdministrator() )
    {
        strFileName = SYSTEM_PROFOLE_DIR + PATH_SEPARATOR + profile.sProfileName + _T(".profile");
        priority = m_systemProfiles.size() + 1;
        profile.sPermission = _T("admin");
    }
    else
    {
        strFileName = USER_PROFILE_DIR + PATH_SEPARATOR + profile.sProfileName + _T(".profile");
        priority = m_userProfiles.size() + 1;
        profile.sPermission = _T("user");
    }

    if ( m_editor.WriteProfile(profile, strFileName, priority) != 1 )
        return false;
    ProfileEvent profileEvent(ID_PROFILE_CHANGED, PEVT_PROFILE_ADD);
    wxEvtHandler* pEvtHandler = NULL;
    pEvtHandler = (wxEvtHandler*) DataCenter::GetInstance()->GetNetworkManagerDlgHandle();
    ::wxPostEvent(pEvtHandler, profileEvent);
    pEvtHandler = (wxEvtHandler*) DataCenter::GetInstance()->GetProfileManagerDlgHandle();
    ::wxPostEvent(pEvtHandler, profileEvent);
    return true;
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
bool ProfileManagementControl::ExportProfile(ProfileList_t& profile, const wxString& path)
{
    if ( !wxDir::Exists(path) )
        return false;

    wxString fileName = path + PATH_SEPARATOR + profile.sProfileName + _T(".profile");
    return m_editor.WriteProfile(profile, fileName) == 1;
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
bool ProfileManagementControl::RemoveProfile(size_t index)
{
    bool isAdminUser = ENMUtil::IsAdministrator();
    vector<ProfileList_t>* pVector = NULL;
    wxString dir;
    if ( isAdminUser )
    {
        pVector = &m_systemProfiles;
        dir = SYSTEM_PROFOLE_DIR;
    }
    else
    {
        pVector = &m_userProfiles;
        dir = USER_PROFILE_DIR;
        index -= m_systemProfiles.size();
    }
    if ( index < 0 || index >= pVector->size() )
        return false;

    vector<ProfileList_t>::iterator iter = pVector->begin();
    size_t count = 0;
    while ( count < index )
    {
        iter ++;
        count ++;
    }
    ProfileList_t profile = *iter;
    wxString fileName = dir + PATH_SEPARATOR + profile.sProfileName + _T(".profile");
    if ( !m_editor.DeleteProfile(fileName) )
        return false;
    pVector->erase(iter);
    ProfileEvent profileEvent(ID_PROFILE_CHANGED, PEVT_PROFILE_DELETE);
    ::wxPostEvent(DataCenter::GetInstance()->GetNetworkManagerDlgHandle(), profileEvent);
    return true;
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
bool ProfileManagementControl::DisconnectCurrentProfile()
{
//    ProfileList_t* pProfile = GetProfile(m_strCurrentProfile);
//    if ( pProfile == NULL )
//        return false;
    m_isBusy = true;
    m_status = DISCONNECT_PROFILE;

    if ( m_profileToApply.stGenernalSet.ConnecType == _T("wireless") )
    {
        DataCenter::GetInstance()->CloseWlanConnection();
    }
    else
    {
//        DataCenter::GetInstance()->EnableNetworkAdapter(ADAPTER_WIRED, false);
        DataCenter::GetInstance()->SwitchAdapterParameter(ADAPTER_WIRED, (ADAPTER_TYPE)0, this);
    }

//    m_profileToApply.sProfileName.clear();
//    m_profileToApply.sPermission.clear();
//    m_strCurrentProfile.Clear();
    ProfileEvent event(ID_PROFILE_APPLY, PEVT_PROFILE_APPLY_DISCONNECT);
    ::wxPostEvent(DataCenter::GetInstance()->GetNetworkManagerDlgHandle(), event);
    m_isBusy = false;
    return true;
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
bool ProfileManagementControl::ApplyProfile(size_t index)
{
    if ( index < 0 || index >= m_systemProfiles.size() + m_userProfiles.size() )
        return false;

    m_iCurrentProfileIndex = (int)index;
    if ( index < m_systemProfiles.size() )
        return ApplyProfile(m_systemProfiles.at(index));
    else
        return ApplyProfile(m_userProfiles.at(index - m_systemProfiles.size()));
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
bool ProfileManagementControl::ApplyProfile(const ProfileList_t& profile)
{
    ENMUtil::Log(logFile, _T("Enter ProfileManagementControl::ApplyProfile(const ProfileList_t&)"));
    m_isBusy = true;
    m_isStopped = false;
    m_status = APPLY_PROFILE;

    m_strCurrentProfile.Clear();
    m_profileToApply = profile;
    wxString type = profile.stGenernalSet.ConnecType;
    if ( type == _T("wireless") )
    {
        WirelessSettings_t wlanSet = profile.stWirelessSet;
        m_pToConnect = new APStatus();
        m_strKey.Clear();
        memset(m_pToConnect, 0, sizeof(APStatus));
        m_pToConnect->auth = WLAN_SECURITY_UNKNOWN;
        wxString ssid = wlanSet.SSID;
        m_pToConnect->ssid.length = (int)ssid.Length();
        memcpy(&m_pToConnect->ssid.ucSsid, ssid.mb_str(wxConvUTF8), ssid.length());
        mbstowcs((wchar_t*)&m_pToConnect->name, (const char*)&m_pToConnect->ssid.ucSsid, m_pToConnect->ssid.length);
        if ( wlanSet.WlanSecurity == _T("none") )
        {
            m_pToConnect->auth = WLAN_SECURITY_NONE;
        }
        else if ( wlanSet.WlanSecurity == _T("wep") )
        {
            m_pToConnect->auth = WLAN_SECURITY_WEP;
            m_strKey = wlanSet.stWEPSecurity.value;
        }
        else if ( wlanSet.WlanSecurity == _T("wpa-psk") )
        {
            m_pToConnect->auth = WLAN_SECURITY_WPA_PSK;
            m_strKey = wlanSet.stWPASecurity.value;
        }
        m_pToConnect->bssType = BSS_TYPE_ANY;
    }
    m_timerPrepare.Start(200, wxTIMER_ONE_SHOT);
    ENMUtil::Log(logFile, _T("Return ProfileManagementControl::ApplyProfile\n"));
    return true;
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
void ProfileManagementControl::InsertProfile(const ProfileList_t& profile, vector<ProfileList_t>* pVector)
{
    long priority = 0;
	profile.sPriority.ToLong(&priority);
    vector<ProfileList_t>::iterator iter = pVector->begin();
    while ( iter != pVector->end() && priority > GetProfilePriority(*iter) )
    {
        iter ++;
    }
    pVector->insert(iter, profile);
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
bool ProfileManagementControl::LoadUserProfiles()
{
    if ( !wxDir::Exists(USER_PROFILE_DIR) && !::wxMkdir(USER_PROFILE_DIR) )
        return false;

    wxArrayString fileNameArray;
    size_t fileCount = wxDir::GetAllFiles(USER_PROFILE_DIR, &fileNameArray, _T("*.profile"), wxDIR_FILES | wxDIR_HIDDEN);
    if ( fileCount == NO_FILES )
        return false;

    m_userProfiles.clear();
    for ( size_t i = 0; i < fileCount; i ++ )
    {
        wxString fileName = fileNameArray.Item(i);
        ProfileList_t profile;
        if ( m_editor.ReadProfile(profile, fileName) && profile.sPermission == _T("user") )
        {
            wxString profileName = fileName.SubString(fileName.Find(PATH_SEPARATOR, true) + 1, fileName.Find('.', true) - 1);
            profile.sProfileName = profileName;
            InsertProfile(profile, &m_userProfiles);
        }
    }
    return true;
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
bool ProfileManagementControl::LoadSystemProfiles()
{
    if ( !wxDir::Exists(SYSTEM_PROFOLE_DIR) && !::wxMkdir(SYSTEM_PROFOLE_DIR) )
        return false;

    wxArrayString fileNameArray;
    size_t fileCount = wxDir::GetAllFiles(SYSTEM_PROFOLE_DIR, &fileNameArray, _T("*.profile"), wxDIR_FILES | wxDIR_HIDDEN);
    if ( fileCount == NO_FILES )
        return false;

    m_systemProfiles.clear();
    for ( size_t i = 0; i < fileCount; i ++ )
    {
        wxString fileName = fileNameArray.Item(i);
        ProfileList_t profile;
        if ( m_editor.ReadProfile(profile, fileName) )
        {
            wxString profileName = fileName.SubString(fileName.Find(PATH_SEPARATOR, true) + 1, fileName.Find('.', true) - 1);
            profile.sProfileName = profileName;
            profile.sPermission = _T("admin");
            InsertProfile(profile, &m_systemProfiles);
        }
    }
    return true;
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
bool ProfileManagementControl::LoadProfiles(bool isAdminUser)
{
    bool result = LoadSystemProfiles();
    if ( !isAdminUser )
    {
        result &= LoadUserProfiles();
    }
    return result;
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
bool ProfileManagementControl::SetFileWritable(const wxString& fileName)
{
#if defined(__WXMSW__)
    return _wchmod(fileName.fn_str(), _S_IWRITE) == 0;
#endif
#if defined(__UNIX__)
    return chmod(fileName.mbc_str(), S_IWUSR) == 0;
#endif
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
bool ProfileManagementControl::SaveProfile(ProfileList_t& profile, int priority, bool isAdminUser)
{
    wxString dir;
    if ( isAdminUser )
    {
        if ( !wxDir::Exists(SYSTEM_PROFOLE_DIR) && !::wxMkdir(SYSTEM_PROFOLE_DIR) )
            return false;
        dir = SYSTEM_PROFOLE_DIR;
    }
    else
    {
        if ( !wxDir::Exists(USER_PROFILE_DIR) && !::wxMkdir(USER_PROFILE_DIR) )
            return false;
        dir = USER_PROFILE_DIR;
    }

    wxString fileName = dir + PATH_SEPARATOR + profile.sProfileName + _T(".profile");
    return m_editor.WriteProfile(profile, fileName, priority) == 1;
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
ProfileList_t* ProfileManagementControl::GetProfile(size_t index)
{
    if ( index >= 0 && index < m_systemProfiles.size() )
        return &m_systemProfiles.at(index);
    else
    {
        index -= m_systemProfiles.size();
        if ( index >= 0 && index < m_userProfiles.size() )
            return &m_userProfiles.at(index);
        else
            return NULL;
    }
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
ProfileList_t* ProfileManagementControl::GetProfile(const wxString& name, bool admin)
{
    vector<ProfileList_t>* pVector;
    if ( admin )
        pVector = &m_systemProfiles;
    else
        pVector = &m_userProfiles;
//    string strName = ENMUtil::WxStringToString(name);
    ProfileList_t* pResult = NULL;
    for ( size_t i = 0; i < pVector->size(); i++ )
    {
        ProfileList_t profile =pVector->at(i);
        if (profile.sProfileName == name)
        {
            pResult = &pVector->at(i);
            break;
        }
    }
    return pResult;
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
bool ProfileManagementControl::Contains(const wxString& name)
{
    for ( size_t i = 0; i < m_systemProfiles.size(); i ++ )
    {
        ProfileList_t pe = m_systemProfiles.at(i);
        if ( pe.sProfileName == name )
            return true;
    }
    return false;
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
void ProfileManagementControl::OnTimerPrepare(wxTimerEvent& event)
{
    ENMUtil::Log(logFile, _T("Enter ProfileManagementControl::OnTimerPrepare(wxTimerEvent&)"));
    DataCenter* pDataCenter = DataCenter::GetInstance();
    if ( pDataCenter == NULL )
    {
        PostConnectEvent(false, WLAN_ACM_ERROR_UNKNOWN);
        PostProfileEvent(false);
        return;
    }

    ENMUtil::Log(logFile, _T("\t Begin switch adapter"));
    if ( m_profileToApply.stGenernalSet.ConnecType == _T("wireless") )
    {
        #if defined(__WXMSW__)
        ENMUtil::Log(logFile, wxString::Format(_T("\t Begin switch adapter, from %d to %d, time: %d"), ADAPTER_WIRED, ADAPTER_WIREDLESS, ::wxGetLocalTimeMillis()));
        #endif
        DataCenter::GetInstance()->SwitchAdapterParameter(ADAPTER_WIRED, ADAPTER_WIREDLESS, this);
    }
    else
    {
        #if defined(__WXMSW__)
        ENMUtil::Log(logFile, wxString::Format(_T("\t Begin switch adapter, from %d to %d, time: %d"), ADAPTER_WIREDLESS, ADAPTER_WIRED, ::wxGetLocalTimeMillis()));
        #endif
        DataCenter::GetInstance()->SwitchAdapterParameter(ADAPTER_WIREDLESS, ADAPTER_WIRED, this);
    }
//    ENMUtil::Log(logFile, wxString::Format(_T("\t Finished switch adapter, time: %d"), ::wxGetLocalTimeMillis()));
    #if defined(__WXMSW__)
    ENMUtil::Log(logFile, wxString::Format(_T("Return ProfileManagementControl::OnTimerPrepare, time: %d\n"), ::wxGetLocalTimeMillis()));
    #endif
    ENMUtil::Log(logFile, _T("Leave ProfileManagementControl::OnTimerPrepare(wxTimerEvent&)"));
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
void ProfileManagementControl::OnTimer(wxTimerEvent& event)
{
    int id = event.GetId();
    if ( id == LAN_CONNECT_TIMER_ID )
    {
        static int iLanConnectTimes = 0;
        DataCenter* pDataCenter = DataCenter::GetInstance();
        if( iLanConnectTimes < 20 )
        {
            if( pDataCenter->IsWiredInterfaceConnected() )
            {
                m_timerLanConnect.Stop();
                iLanConnectTimes = 0 ;
                WlanAcmEvent connectEvent(ID_WLAN_ACM_CONNECT, WLAN_ACM_EVENT_CONNECT_SUCCESS);
                OnWlanConnect(connectEvent);
//                ::wxPostEvent(pDataCenter->GetProfileController(), connectEvent);
            }
            else
            {
                iLanConnectTimes++;
            }
        }
        else
        {
            iLanConnectTimes = 0 ;
            m_timerLanConnect.Stop();
            WlanAcmEvent connectEvent(ID_WLAN_ACM_CONNECT, WLAN_ACM_EVENT_CONNECT_FAILED, WLAN_ACM_ERROR_NO_RESPONSE);
            OnWlanConnect(connectEvent);
//            ::wxPostEvent(pDataCenter->GetProfileController(), connectEvent);
        }
    }
#if defined(__WXMSW__)
    else if ( id == ADDRESS_TIMER_ID )
    {
        ContinueSetAddress(m_profileToApply);
        PostProfileEvent(true);
    }
#endif
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
void ProfileManagementControl::OnWlanConnect(WlanAcmEvent& event)
{
    ENMUtil::Log(logFile, _T("Enter ProfileManagementControl::OnWlanConnect(WlanAcmEvent& event)"));
    if ( m_isStopped )
    {
        m_isBusy = false;
        m_status = WAITING;
        ENMUtil::Log(logFile, _T("Return ProfileManagementControl::OnWlanConnect, stopped.\n"));
        return;
    }

    if ( m_pToConnect != NULL )
    {
        delete m_pToConnect;
        m_pToConnect = NULL;
    }
    unsigned long eventCode = event.GetEventCode();
    if ( eventCode == WLAN_ACM_EVENT_CONNECT_SUCCESS && IsCorrectNetwork() )
    {
        PostConnectEvent(true);
        ContinueApplyProfile(m_profileToApply);
#if defined(__UNIX__)
        PostProfileEvent(true);
#endif
    }
    else
    {
        PostConnectEvent(false);
        PostProfileEvent(false);
    }
    ENMUtil::Log(logFile, _T("Return ProfileManagementControl::OnWlanConnect\n"));
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
void ProfileManagementControl::OnAdapterSwitch(SwitchAdapterEvent& event)
{
    ENMUtil::Log(logFile, _T("Enter ProfileManagementControl::OnAdapterSwitch(SwitchAdapterEvent&)"));
    if ( m_isStopped )
    {
        m_isBusy = false;
        m_status = WAITING;
        ENMUtil::Log(logFile, _T("Return ProfileManagementControl::OnAdapterSwitch, stopped.\n"));
        return;
    }

    DataCenter* pDataCenter = DataCenter::GetInstance();
    if ( event.IsSuccess() )
    {
        ENMUtil::Log(logFile, _T("\t Switch adapter success"));
        if ( m_status == DISCONNECT_PROFILE && m_profileToApply.stGenernalSet.ConnecType == _T("lan") )
        {
            ENMUtil::Log(logFile, _T("\t Disconnecting profile..."));
            ProfileEvent event(ID_PROFILE_APPLY, PEVT_PROFILE_APPLY_DISCONNECT);
            ::wxPostEvent(pDataCenter->GetNetworkManagerDlgHandle(), event);
        }
        if ( m_status == APPLY_PROFILE )
        {
            if ( m_profileToApply.stGenernalSet.ConnecType == _T("wireless") )
            {
                ENMUtil::Log(logFile, _T("\t Connecting wireless..."));
#if defined(__UNIX__)
                ENMUtil::Log(wxT("/tmp/ENMLog.txt"), wxString(wxT("switch adapter successful, now sleep for 6secs")));
//                wxSleep(6);
                ENMUtil::Log(wxT("/tmp/ENMLog.txt"), wxString(wxT("wake up, now connect AP")));
#endif
				long _pt = 0;
				m_profileToApply.stWirelessSet.stWEPSecurity.keyIndex.ToLong(&_pt);
                pDataCenter->ConnectWlanNetwork(*m_pToConnect, m_strKey, (int)_pt, true);
#if defined(__UNIX__)
                ENMUtil::Log(wxT("/tmp/ENMLog.txt"), wxString(wxT("connect AP end")));
#endif
            }
            else
            {
//                ENM_String* sKill = new ENM_String();
//                memset(sKill, 0, sizeof(ENM_String));
//                strcpy((char*)sKill->dcsString, "killall");
//                DCS_WriteFile(sKill);
//                delete sKill;
                ENMUtil::Log(logFile, _T("\t Connecting wired..."));
                m_timerLanConnect.Start(1000);
            }
        }
//        ENMUtil::Log(logFile, _T("\t Start connecting..."));
    }
    else
    {
        ENMUtil::Log(logFile, _T("\t Switch adapter failed"));
        ClearCurrentProfile();
//        ProfileEvent event(ID_PROFILE_APPLY, PEVT_PROFILE_APPLY_FAILED);
//        ::wxPostEvent(pDataCenter->GetNetworkManagerDlgHandle(), event);
        PostConnectEvent(false);
        PostProfileEvent(false);
        ENMUtil::Log(logFile, _T("\t Apply profile failed"));
    }
    ENMUtil::Log(logFile, _T("Return ProfileManagementControl::OnAdapterSwitch\n"));
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
void ProfileManagementControl::OnWirelessNetworkChanged(const WirelessSwitchEvent& event)
{
    ENMUtil::Log(logFile, wxString::Format(_T("Enter ProfileManagementControl::OnWirelessNetworkChanged(%08X)"), &event));
    if ( m_isBusy )
    {
        ENMUtil::Log(logFile, _T("Return ProfileManagementControl::OnWirelessNetworkChanged, busy\n"));
        return;
    }
    WlanSsid oldSsid = event.GetOldSsid();
    WlanSsid newSsid = event.GetNewSsid();
    #if defined(__WXMSW__)
    ENMUtil::Log(logFile, wxString::Format(_T(" \t Switched from \"%s\" to \"%s\""), wxString::FromUTF8((char*)oldSsid.ucSsid, oldSsid.length), wxString::FromUTF8((char*)newSsid.ucSsid, newSsid.length)));
    #endif
    if ( oldSsid.length != 0 && newSsid != oldSsid )
    {
        #if defined(__WXMSW__)
        ENMUtil::Log(logFile, wxString::Format(_T(" \t Clear current profile: %s"), wxString::FromUTF8(m_profileToApply.sProfileName.c_str())));
        #endif
        ClearCurrentProfile();
        ENMUtil::Log(logFile, _T("\t Clear current profile OK"));
    }
    ENMUtil::Log(logFile, _T("Return ProfileManagementControl::OnWirelessNetworkChanged\n"));
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
bool ProfileManagementControl::PostConnectEvent(bool success, unsigned long errorCode)
{
    #if defined(__WXMSW__)
    ENMUtil::Log(logFile, wxString::Format(_T("Enter ProfileManagementControl::PostConnectEvent(%d, %d), time: %d"), success, errorCode, ::wxGetLocalTimeMillis()));
    #endif
    DataCenter* pDataCenter = DataCenter::GetInstance();
    unsigned long eventCode;
    if ( success )
    {
        eventCode = WLAN_ACM_EVENT_CONNECT_SUCCESS;
        m_strCurrentProfile = m_profileToApply.sProfileName;
    }
    else
    {
        eventCode = WLAN_ACM_EVENT_CONNECT_FAILED;
        ENMUtil::Log(logFile, wxString(wxT("\t Clear profile name, because connecting failed")));
        ClearCurrentProfile();
    }
    WlanAcmEvent connectEvent(ID_WLAN_ACM_CONNECT, eventCode, errorCode);
    ::wxPostEvent(ENMTaskTray::GetInstance(), connectEvent);
    ::wxPostEvent(pDataCenter->GetNetworkManagerDlgHandle(), connectEvent);
    #if defined(__WXMSW__)
    ENMUtil::Log(logFile, wxString::Format(_T("Return ProfileManagementControl::PostConnectEvent : %d, time: %d\n"), success, ::wxGetLocalTimeMillis()));
    #endif
    return success;
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
bool ProfileManagementControl::PostProfileEvent(bool applySuccess, bool toTaskTray)
{
    #if defined(__WXMSW__)
    ENMUtil::Log(logFile, wxString::Format(_T("Enter ProfileManagementControl::PostProfileEvent(%d), time: %d"), applySuccess, ::wxGetLocalTimeMillis()));
    #endif
    DataCenter* pDataCenter = DataCenter::GetInstance();
    unsigned long eventCode = PEVT_PROFILE_APPLY_FAILED;
    if ( applySuccess )
    {
        eventCode = PEVT_PROFILE_APPLY_SUCCESS;
    }
    else
    {
        eventCode = PEVT_PROFILE_APPLY_FAILED;
    }
    ProfileEvent event(ID_PROFILE_APPLY, eventCode);
    ::wxPostEvent(pDataCenter->GetNetworkManagerDlgHandle(), event);
    if ( toTaskTray )
    {
    ::wxPostEvent(ENMTaskTray::GetInstance(), event);
    }
    m_isBusy = false;
    m_status = WAITING;
    #if defined(__WXMSW__)
    ENMUtil::Log(logFile, wxString::Format(_T("Return ProfileManagementControl::PostProfileEvent : %d, time: %d\n"), applySuccess, ::wxGetLocalTimeMillis()));
    #endif
    return applySuccess;
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
bool ProfileManagementControl::WriteProfiles(vector<ProfileList_t>* pVector, const wxString& path, wxArrayString& failedProfiles)
{
    if ( NULL == pVector )
        return false;
    if ( !wxDir::Exists(path) )
        return false;
    int priority = 1;
    bool success = true;
    vector<ProfileList_t>::iterator iter;
    for ( iter = pVector->begin(); iter != pVector->end(); iter++ )
    {
        ProfileList_t profile = *iter;
//        wxString strFileName = wxString::Format(_T("%s%s%s.profile"), path, PATH_SEPARATOR, profile.sProfileName.c_str());
        wxString strFileName = path + PATH_SEPARATOR + profile.sProfileName + _T(".profile");
        if ( !m_editor.WriteProfile(profile, strFileName, priority) )
        {
            failedProfiles.Add(profile.sProfileName);
            success = false;
        }
        priority++;
    }
    return success;
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
bool ProfileManagementControl::ConfirmModify(wxArrayString& failedProfiles)
{
    if ( ENMUtil::IsAdministrator() )
    {
        return WriteProfiles(&m_systemProfiles, SYSTEM_PROFOLE_DIR, failedProfiles);
    }
    else
    {
        return WriteProfiles(&m_userProfiles, USER_PROFILE_DIR, failedProfiles);
    }
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
bool ProfileManagementControl::ConfirmModify(const vector<ProfileList_t>* pVector, wxArrayString& failedProfiles)
{
    if ( NULL == pVector )
        return false;

    size_t startPos = 0;
    wxString dir = SYSTEM_PROFOLE_DIR;
    if ( !ENMUtil::IsAdministrator() )
    {
        startPos = m_systemProfiles.size();
        dir = USER_PROFILE_DIR;
    }

    wxArrayString fileNameArray;
    size_t fileCount = wxDir::GetAllFiles(dir, &fileNameArray, _T("*.profile"), wxDIR_FILES | wxDIR_HIDDEN);
    if (fileCount != NO_FILES)
    {
        for ( size_t i = 0; i < fileCount; i++ )
        {
            wxString fileName = fileNameArray.Item(i);
            SetFileWritable(fileName);
            ::wxRemoveFile(fileName);
        }
    }

    bool success = true;
    int priority = 10001;
    for ( size_t i = startPos; i < pVector->size(); i++ )
    {
        ProfileList_t profile = pVector->at(i);
        wxString fileName = dir + PATH_SEPARATOR + profile.sProfileName + _T(".profile");
        if ( !m_editor.WriteProfile(profile, fileName, priority++) )
        {
            failedProfiles.Add(profile.sProfileName);
            success = false;
        }
    }
    ProfileEvent profileEvent(ID_PROFILE_CHANGED);
    ::wxPostEvent(DataCenter::GetInstance()->GetNetworkManagerDlgHandle(), profileEvent);
    return success;
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
void ProfileManagementControl::ClearCurrentProfile()
{
#if defined(__WXMSW__)
    ENMUtil::Log(logFile, _T("Enter ProfileManagementControl::ClearCurrentProfile"));
#endif
    m_profileToApply.sProfileName.clear();
    m_profileToApply.sPermission.clear();
    m_strCurrentProfile.Clear();
    m_iCurrentProfileIndex = -1;
#if defined(__WXMSW__)
    ENMUtil::Log(logFile, _T("Return ProfileManagementControl::ClearCurrentProfile\n"));
#endif
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
void ProfileManagementControl::NotifyModeChanged(NETWORK_MODE networkMode)
{
    ENMUtil::Log(logFile, wxString::Format(_T("Enter ProfileManagementControl::NotifyModeChanged(%d)"), networkMode));

    if ( m_isBusy )
    {
        ENMUtil::Log(logFile, _T("Return ProfileManagementControl::NotifyModeChanged, because it is busy\n"));
        return;
    }
    // no current profile
    if ( m_profileToApply.sProfileName.length() == 0 )
    {
        ENMUtil::Log(logFile, _T("Return ProfileManagementControl::NotifyModeChanged, because current profile is empty\n"));
        return;
    }

//    if ( networkMode == NETWORK_MODE_WIRELESS_LAN_CONNECT || networkMode == NETWORK_MODE_WIRELESS_LAN_DISCONNECT )
    if ( networkMode == NETWORK_MODE_DISCONNECTED || networkMode == NETWORK_MODE_WIRELESS_LAN_DISCONNECT || networkMode == NETWORK_MODE_WIRED_LAN_DISCONNECT )
    {
        ENMUtil::Log(logFile, wxString(wxT("\t Clear current profile, because of disconnect")));
        ClearCurrentProfile();
    }
    else if ( networkMode == NETWORK_MODE_WIRED_LAN_CONNECT )
    {
        if ( m_profileToApply.stGenernalSet.ConnecType == _T("wireless") )
        {
            ENMUtil::Log(logFile, wxString(wxT("\t Clear current profile, because switch wireless to wired")));
            ClearCurrentProfile();
        }
    }
    else if ( networkMode == NETWORK_MODE_WIRELESS_LAN_CONNECT )
    {
        if ( m_profileToApply.stGenernalSet.ConnecType == _T("lan") )
        {
            ENMUtil::Log(logFile, wxString(wxT("\t Clear current profile, because switch wired to wireless")));
            ClearCurrentProfile();
        }
    }
    ENMUtil::Log(logFile, _T("Return ProfileManagementControl::NotifyModeChanged\n"));
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
bool ProfileManagementControl::IsCorrectNetwork()
{
#if defined(__WXMSW__)
    if ( m_profileToApply.stGenernalSet.ConnecType == "lan" )
        return true;

    APStatus* pAP = DataCenter::GetInstance()->GetCurrentAPStatus();
    if ( NULL == pAP )
        return true;

    string ssid((char*)pAP->ssid.ucSsid);
    return m_profileToApply.stWirelessSet.SSID == ssid;
#else
    bool result = true;
#endif
    return result;
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
void ProfileManagementControl::StopApplying()
{
    m_isStopped = true;
    ClearCurrentProfile();
}

bool ProfileManagementControl::GetStaticFlag()
{
    return m_bIfStaticIp;
}

void ProfileManagementControl::OnStaticIpTimer(wxTimerEvent& event)
{
    if ( m_setIpCounter < 5 )
    {
        ENM_String* IP = new ENM_String();
        memset(IP, 0, sizeof(ENM_String));
        DCS_GetIPAddressLAN(IP);
        string strIP((const char*)IP->dcsString);
        delete IP;
        wxString desIp = m_profileToApply.stOtherSet.stIPDNSSet.IPvalue;

        if ( ENMUtil::StringToWxString(strIP) == desIp )
        {
            m_SetStaticIp.Stop();
            m_bIfStaticIp = false;
            m_setIpCounter = 0;
        }
        else
        {
            m_setIpCounter++;
        }
    }
    else
    {
        m_SetStaticIp.Stop();
        m_bIfStaticIp = false;
        m_setIpCounter = 0;
    }

}
