/****************************************************************************

Copyright (c) 2008 Intel Corporation.  All rights reserved. 

 

DISCLAIMER OF WARRANTY

NEITHER INTEL NOR ITS SUPPLIERS MAKE ANY REPRESENTATION OR WARRANTY OR

CONDITION OF ANY KIND WHETHER EXPRESS OR IMPLIED (EITHER IN FACT OR BY

OPERATION OF LAW) WITH RESPECT TO THE SOURCE CODE.  INTEL AND ITS SUPPLIERS

EXPRESSLY DISCLAIM ALL WARRANTIES OR CONDITIONS OF MERCHANTABILITY OR

FITNESS FOR A PARTICULAR PURPOSE.  INTEL AND ITS SUPPLIERS DO NOT WARRANT

THAT THE SOURCE CODE IS ERROR-FREE OR THAT OPERATION OF THE SOURCE CODE WILL

BE SECURE OR UNINTERRUPTED AND HEREBY DISCLAIM ANY AND ALL LIABILITY ON

ACCOUNT THEREOF.  THERE IS ALSO NO IMPLIED WARRANTY OF NON-INFRINGEMENT.

SOURCE CODE IS LICENSED TO LICENSEE ON AN "AS IS" BASIS AND NEITHER INTEL

NOR ITS SUPPLIERS WILL PROVIDE ANY SUPPORT, ASSISTANCE, INSTALLATION,

TRAINING OR OTHER SERVICES.  INTEL AND ITS SUPPLIERS WILL NOT PROVIDE ANY

UPDATES, ENHANCEMENTS OR EXTENSIONS. 

****************************************************************************/


/**
 * 
 * @file TabletHelper.h
 *
 * @version 0.1
 * @author Christina
 * @date 2008/07/17
 */

#ifndef TABLET_HELPER_H
#define TABLET_HELPER_H

#ifdef WIN32
#include <windows.h>
#else
#include <pthread.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#endif /** WIN32 */

#include "plugins/DcsPlugin.h"
#include "plugins/tablet/TabletDef.h"
#include "plugins/tablet/TabletPlugin.h"

/**
 * @class TabletHelper
 * @brief TabletHelper provides help on operating Tablet driver.
 *
 * It is implemented as a singleton.
 */

class TabletHelper
{
public:
	static TabletHelper &GetInstance()
	{
		static TabletHelper helper;
		return helper;
	};

	/** public functions which really implement DCS Plugin interfface. */
	int Initialize(DCS_Funcs *pDispatcher);
	DCS_Return_Code HandleRequest(DCS_RequestData *pRequest);
	
	int Cleanup();

private:

#ifdef WIN32	
	TabletHelper(void)
		: m_pFuncs(NULL), m_hTabletEvent(NULL), m_hExitEvent(NULL),
		m_hThread(NULL), m_hDevice(NULL)
	{
	};
	
	~TabletHelper(void)
	{
	};
#else
	TabletHelper(void)
		: m_Thread(0), m_Device(-1)
	{
	};

	~TabletHelper(void)
	{
	};
#endif /** WIN32 */

	/** tablet related functions */
	BOOL StartTablet();
	BOOL StopTablet();
	BOOL GetTabletData(DCS_Tablet_Sensor_Data &data);
	
	/** declare copy constructor and equal operator to prevent
	 *  copied instance
	 */
	TabletHelper(const TabletHelper &);
	TabletHelper &operator =(const TabletHelper &);

	/** pointer to DCS dispatch function */
	DCS_Funcs *m_pFuncs;

	static const char * const MODULE_NAME; /** tablet chip module name */

#ifdef WIN32
	/** WIN32 working thread function */
	friend DWORD WINAPI TabletEventHandler(LPVOID lpParameter);

	void CloseHandles();

	static const wchar_t * const EVENT_NAME; /** event shared with driver */
	static const wchar_t * const DEVICE_PATH; /** path to open driver */

	HANDLE m_hTabletEvent; /** event to get notification from driver */
	HANDLE m_hExitEvent; /** event to exit working thread */
	HANDLE m_hThread; /** working thread */
	HANDLE m_hDevice; /** tablet device */

#else
	/** Linux working thread function */
	friend void *TabletEventHandler(void *arg);	

	static const char * const DEVICE_PATH; /** path to open driver */

	pthread_t m_Thread;
	int m_Device;
#endif /** WIN32 */

	
	/**
	 * @class LockHelper
	 * @brief LcokHelper provides auto lock and unlock approach by
	 *        constructor and destructor.
	 */

#ifdef WIN32
	class LockHelper
	{
	public:
		LockHelper(CRITICAL_SECTION &lock)
			: m_Lock(lock)
		{
			EnterCriticalSection(&m_Lock);
		};

		~LockHelper()
		{
			LeaveCriticalSection(&m_Lock);
		};

	private:
		CRITICAL_SECTION &m_Lock;
	};
#else
	class LockHelper
	{
	public:
		LockHelper(pthread_mutex_t &lock)
			:m_Lock(lock)
		{
			pthread_mutex_lock(&m_Lock);
		};

		~LockHelper()
		{
			pthread_mutex_unlock(&m_Lock);
		};

	private:
		pthread_mutex_t m_Lock;
	};
#endif /** WIN32 */
};
#endif /** TABLET_HELPER_H */

