/*****************************************************************************

 * Copyright (C) 2008  Intel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License, version 2.1, as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA

File Name:          PMBatteryMgmt.cpp

Description:

    This file implements class PMBatteryMgmt.h.

Environment (opt):

    OS: Ubuntu
    SE: Code::Block

Notes (opt):

  =====================================================================

  Revision   Revision History               Author     Date

  =====================================================================

  0.1        Create                         Zhang Hui 2008-7-20

  =====================================================================

****************************************************************************/

#include "PMBatteryMgmt.h"

static int g_isACPI = NOT_SUP;
static int gCurrentSnapshot = 0;
static Capacity_Snapshot remCapacityArray[10] = {{0, 0}, };

int CheckACPIAVAI()
{
	if (g_isACPI != NOT_SUP)
	{
		return g_isACPI;
	}


	if (access(ACPI_DIR, R_OK))
	{
		g_isACPI = NOT_SUP;
	}
	else
	{
	    g_isACPI = ACPI_S;
	}

	return g_isACPI;

}

int CheckACPIDirExit(const char* dir)
{
    int iCount;
	DIR *acpi_dir;
	struct dirent *dirent;


	acpi_dir = opendir(dir);
	if (!acpi_dir)
	{
		return MODULE_UNAVIA;
	}

	for (iCount = 0; iCount < 3; iCount++)
	{
		dirent = readdir(acpi_dir);
		if (dirent <= 0)
		{
			closedir(acpi_dir);
			return DEVICE_UNAVIA;
		}
		if (dirent->d_name[0] == '.')
		{
			continue;
		}
	}

	closedir(acpi_dir);

	if (iCount == 3)
	{
		return 1;
	}
	else
	{
		return DEVICE_UNAVIA;
	}
}

int GetBatteriesInformation(Battery_General *bat_general)
{
	int iRet = 0;

	if (bat_general == NULL)
	{
		errno = EINVAL;
		return -2;
	}

    bat_general->remMins = -1;
	bat_general->remPerc = -1;
    bat_general->remChargingMins = -1;
	bat_general->chargingState=CHARG_UNKNOWN;

	iRet = CheckACPIAVAI();

	if (iRet == ACPI_S)
	{
		if (GetBatteryInfoofGeneral(bat_general) < 0)
		{
			return PMUNKOWN;
		}
		else
		{
			return 1;
		}
	}
	else
	{
		return PMUNKOWN;
	}
}

int GetBatteryInfoByDetail(const int index, Battery_Detailed *batInfo)
{
    int temp = 0;
	int iRet = 0;
	int dirSum = 0;
	DIR *dir = NULL;
	struct dirent *dirent = NULL;
	char batInfoFilePath[MAX_FILE_PATH_LENGTH + 1] = "";
    char direntName[MAX_FILE_PATH_LENGTH + 1] = "";
	char value[MAX_LINE_LENGTH + 1] = "";
	char temp_value[MAX_LINE_LENGTH + 1] = "";
	char batTypeUnit[10] = "";

	FILE *batInfoFile = NULL;


    if (index > MAX_BAT_COUNT - 1 || index < 0)
	{
		return -2;
	}


	if (batInfo == NULL)
	{
		errno = EINVAL;
		return -3;
	}

	if (CheckACPIAVAI() != ACPI_S)
	{
		return ACPI_UNAVAI;
	}


	iRet = CheckACPIDirExit(ACPI_BAT_DIR);
	if (iRet < 0)
	{
		return iRet;
	}

	dir = opendir(ACPI_BAT_DIR);

	while (1)
	{
		dirent = readdir(dir);
		if (dirent == NULL)
		{
			closedir(dir);
			return -2;
		}

		if (dirent->d_name[0] == '.')
		{
			continue;
		}

		if (dirSum == index)
		{
			break;
		}

		dirSum++;
	}

    strncpy(direntName, dirent->d_name, MAX_FILE_PATH_LENGTH);
    closedir(dir);

	iRet = snprintf(batInfoFilePath, MAX_FILE_PATH_LENGTH, "%s/%s/state", ACPI_BAT_DIR, direntName);
	if (iRet >= MAX_FILE_PATH_LENGTH)
	{
	  return -1;
	}

	batInfoFile = fopen(batInfoFilePath, "r");
	if (!batInfoFile)
	{
		return -2;
	}

    if (GetValueByColon(batInfoFile, value, sizeof (value), temp_value, sizeof (temp_value)) < 0)
    {
		batInfo->present = PMUNKOWN;
	}
	else
	{
		if (!strcmp(value, "yes"))
		{
			batInfo->present = PRESENT_YES;
		}
		else if (!strcmp(value, "no"))
		{
			batInfo->present = PRESENT_NO;
			batInfo->capacityState = PMUNKOWN;
			batInfo->chargingState = CHARG_UNKNOWN;
			batInfo->designCapacityLow = PMUNKOWN;
			batInfo->designVoltage = PMUNKOWN;
			batInfo->batteryTechnology = PMUNKOWN;
			batInfo->fullCapacity = PMUNKOWN;
			batInfo->designCapacity = PMUNKOWN;
			batInfo->presentVoltage = PMUNKOWN;
			batInfo->remainingCapacity = PMUNKOWN;
			batInfo->presentRate = PMUNKOWN;
			batInfo->capGran2 = PMUNKOWN;
			batInfo->capGran1 = PMUNKOWN;
			batInfo->designCapacityWarning = PMUNKOWN;
			strcpy(batInfo->OEMInfo, "PMUNKOWN");
			strcpy(batInfo->serialNumber, "PMUNKOWN");
			strcpy(batInfo->modelNumber, "PMUNKOWN");
			strcpy (batInfo->batteryType, "PMUNKOWN");
			fclose(batInfoFile);
			return 0;
		}
		else
		{
			batInfo->present = PMUNKOWN;
		}
	}

    if (GetValueByColon(batInfoFile, value, sizeof (value), temp_value, sizeof(temp_value)) < 0)
    {

		 batInfo->capacityState = PMUNKOWN;
	}
	else
	{
		if (!strcmp(value, "ok"))
		{
			batInfo->capacityState = BAT_OK;
		}
		else if (!strcmp(value, "critical"))
		{
			batInfo->capacityState = BAT_CRIT;
		}
		else
		{
			batInfo->capacityState = PMUNKOWN;
		}
	}

    if (GetValueByColon(batInfoFile, value, sizeof(value), temp_value, sizeof(temp_value)) < 0)
    {
		batInfo->chargingState = CHARG_UNKNOWN;
    }
	else
	{
		if (!strcmp(value, "unknown"))
		{
			batInfo->chargingState = CHARG_UNKNOWN;
		}
		else if (!strcmp(value, "discharging"))
		{
			batInfo->chargingState = CHARG_DISCHARGING;
		}
		else if (!strcmp(value, "charging"))
		{
			batInfo->chargingState = CHARG_CHARGING;
		}
		else if (!strcmp(value, "charging/discharging"))
		{
			batInfo->chargingState = CHARG_CHARG_DISCHARG;
		}
		else
		{
			batInfo->chargingState = CHARG_UNKNOWN;
		}
	}


    if (GetValueByColon(batInfoFile, value, sizeof(value), temp_value, sizeof (temp_value)) < 0)
    {
		batInfo->presentRate = PMUNKOWN;
    }
	else
	{
		if (*value == '\0' || !strcmp(value, "unknown"))
		{
			batInfo->presentRate = PMUNKOWN;
		}
		else
		{
			batInfo->presentRate = (int)strtol(value, NULL, 10);
			//must have a value above 0!
			if (batInfo->presentRate == 0)
			{
				batInfo->presentRate = PMUNKOWN;
			}
		}
	}

    if (GetValueByColon(batInfoFile, value, sizeof (value), temp_value, sizeof (temp_value)) < 0)
    {
		batInfo->remainingCapacity = PMUNKOWN;
    }
	else
	{
		temp = sscanf(value, "%s%s", temp_value, batTypeUnit);
		if (*value == '0')
		{
			batInfo->remainingCapacity = 0;
		}
		else if (temp == 0 || temp == EOF
                ||*value == '\0' || !strcmp(value, "unknown"))
		{
			batInfo->remainingCapacity = PMUNKOWN;
		}
		else
		{
			batInfo->remainingCapacity = (int)strtol(value, NULL, 10);
			if (!batInfo->remainingCapacity)
			{
				batInfo->remainingCapacity = PMUNKOWN;
			}
		}
	}

    if (GetValueByColon(batInfoFile, value, sizeof (value), temp_value, sizeof (temp_value)) < 0)
    {
		batInfo->presentVoltage = PMUNKOWN;
    }
	else
	{
		if (*value == '\0' || !strcmp(value, "unknown"))
		{
			batInfo->presentVoltage = PMUNKOWN;
		}
		else
		{
			batInfo->presentVoltage = (int)strtol(value, NULL, 10);
			if (!batInfo->presentVoltage)
			{
				batInfo->presentVoltage = PMUNKOWN;
			}
		}
	}

	fclose(batInfoFile);

	snprintf(batInfoFilePath, MAX_FILE_PATH_LENGTH, "%s/%s/info", ACPI_BAT_DIR, direntName);
	batInfoFile = fopen(batInfoFilePath, "r");
	if (!batInfoFile)
	{
		return -2;
	}

	if (!fgets(value, sizeof(value), batInfoFile))
	{
		fclose (batInfoFile);
		return -1;
	}


    if (GetValueByColon(batInfoFile, value, sizeof(value), temp_value, sizeof (temp_value)) < 0)
    {
		batInfo->designCapacity = PMUNKOWN;
    }
	else
	{
		if (*value == '\0' || !strcmp(value, "unknown"))
		{
			batInfo->designCapacity = PMUNKOWN;
		}
		else
		{
			batInfo->designCapacity = (int)strtol(value, NULL, 10);
			if (!batInfo->designCapacity)
			{
				batInfo->designCapacity = PMUNKOWN;
			}
		}
	}

    if (GetValueByColon(batInfoFile, value, sizeof (value), temp_value, sizeof (temp_value)) < 0)
    {
		batInfo->fullCapacity = PMUNKOWN;
    }
	else
	{
		if (*value == '\0' || !strcmp(value, "unknown"))
		{
			batInfo->fullCapacity = PMUNKOWN;
		}
		else
		{
			batInfo->fullCapacity = (int)strtol(value, NULL, 10);
			if (!batInfo->fullCapacity)
			{
				batInfo->fullCapacity = PMUNKOWN;
			}
		}
	}

    if (GetValueByColon(batInfoFile, value, sizeof (value), temp_value, sizeof (temp_value)) < 0)
    {
		batInfo->batteryTechnology = PMUNKOWN;
    }
	else
	{
		if (*value == '\0' || !strcmp(value, "unknown"))
		{
			batInfo->batteryTechnology = PMUNKOWN;
		}
		else if (!strcmp(value, "rechargeable"))
		{
			batInfo->batteryTechnology = RECHARGEABLE;
		}
		else if (!strcmp(value, "non-rechargeable"))
		{
			batInfo->batteryTechnology = NON_RECHARGEABLE;
		}
		else
		{
			batInfo->batteryTechnology = PMUNKOWN;
		}
	}


    if (GetValueByColon(batInfoFile, value, sizeof (value), temp_value, sizeof (temp_value)) < 0)
    {
		batInfo->designVoltage = PMUNKOWN;
    }
	else
	{
		if (*value == '\0' || !strcmp(value, "unknown"))
		{
			batInfo->designVoltage = PMUNKOWN;
		}
		else
		{
			batInfo->designVoltage = (int)strtol(value, NULL, 10);
			if (!batInfo->designVoltage)
			{
				batInfo->designVoltage = PMUNKOWN;
			}
		}
	}


    if (GetValueByColon(batInfoFile, value, sizeof (value), temp_value, sizeof (temp_value)) < 0)
    {
		batInfo->designCapacityWarning = PMUNKOWN;
    }
	else
	{
		if (*value == '\0' || !strcmp(value, "unknown"))
		{
			batInfo->designCapacityWarning = PMUNKOWN;
		}
		else
		{
			batInfo->designCapacityWarning = (int)strtol(value, NULL, 10);
			if (!batInfo->designCapacityWarning)
			{
				batInfo->designCapacityWarning = PMUNKOWN;
			}
		}
	}


    if (GetValueByColon(batInfoFile, value, sizeof (value), temp_value, sizeof (temp_value)) < 0)
    {
		batInfo->designCapacityLow = PMUNKOWN;
    }
	else
	{
		if (*value == '\0' || !strcmp(value, "unknown"))
		{
			batInfo->designCapacityLow = PMUNKOWN;
		}
		else
		{
			batInfo->designCapacityLow = (int)strtol(value, NULL, 10);
			if (!batInfo->designCapacityLow)
			{
				batInfo->designCapacityLow = PMUNKOWN;
			}
		}
	}


    if (GetValueByColon(batInfoFile, value, sizeof (value), temp_value, sizeof (temp_value)) < 0)
    {
		batInfo->capGran1 = PMUNKOWN;
    }
	else
	{
		if (*value == '\0' || !strcmp(value, "unknown"))
		{
			batInfo->capGran1 = PMUNKOWN;
		}
		else
		{
			batInfo->capGran1 = (int)strtol(value, NULL, 10);

			if (!batInfo->capGran1)
			{
				batInfo->capGran1 = PMUNKOWN;
			}
		}
	}


    if (GetValueByColon(batInfoFile, value, sizeof (value), temp_value, sizeof (temp_value)) < 0)
    {
		batInfo->capGran2 = PMUNKOWN;
    }
	else
	{
		if (*value == '\0' || !strcmp(value, "unknown"))
		{
			batInfo->capGran2 = PMUNKOWN;
		}
		else
		{
			batInfo->capGran2 = (int)strtol(value, NULL, 10);
			if (!batInfo->capGran2)
			{
				batInfo->capGran2 = PMUNKOWN;
			}
		}
	}

    if (GetValueByColon(batInfoFile, batInfo->modelNumber, sizeof(batInfo->modelNumber), temp_value, sizeof (temp_value)) < 0)
    {
		strcpy(batInfo->modelNumber, "PMUNKOWN");
    }


	if (GetValueByColon(batInfoFile, batInfo->serialNumber, sizeof(batInfo->serialNumber), temp_value, sizeof (temp_value)) < 0)
	{
		strcpy(batInfo->serialNumber, "PMUNKOWN");
	}


	if (GetValueByColon(batInfoFile, batInfo->batteryType, sizeof (batInfo->batteryType), temp_value, sizeof (temp_value)) < 0)
	{
		strcpy (batInfo->batteryType, "PMUNKOWN");
	}


	if (GetValueByColon(batInfoFile, batInfo->OEMInfo, sizeof (batInfo->OEMInfo), temp_value, sizeof (temp_value)) < 0)
	{
		strcpy(batInfo->OEMInfo, "PMUNKOWN");
	}


	fclose(batInfoFile);


    if ( (batInfo->fullCapacity > 0) && (batInfo->remainingCapacity > batInfo->fullCapacity))
    {
        batInfo->remainingCapacity = batInfo->fullCapacity;
    }
	return 0;
}

int GetACAdapterStatus(void)
{
    int iRet = 0;
	int testRet = 0;

	FILE *acFile = NULL;

	char ac_state_file[MAX_FILE_PATH_LENGTH + 1] = "";
 	char ac_file[MAX_FILE_PATH_LENGTH + 1] = "";

    char value[MAX_LINE_LENGTH + 1] = "";
 	char temp_str[MAX_LINE_LENGTH + 1] = "";


	testRet = CheckACPIAVAI();

	if (ACPI_S == testRet)
	{
		iRet = CheckACPIDirExit(ACPI_ACADAPTER_DIR);

		if (iRet < 0)
		{
            return iRet;
		}

		if (GetDir(0, ac_file, sizeof (ac_file), ACPI_ACADAPTER_DIR) < 0)
		{
			return DEVICE_UNAVIA;
		}

		snprintf(ac_state_file, MAX_FILE_PATH_LENGTH, "%s/%s/state", ACPI_ACADAPTER_DIR, ac_file);
		acFile = fopen(ac_state_file, "r");

		if (!acFile)
		{
			return AC_UNKNOWN;
		}

		if (GetValueByColon(acFile, value, sizeof (value), temp_str, sizeof (temp_str)) < 0)
		{
			iRet = AC_UNKNOWN;
		}
		else
		{
		    if (!strcmp(value, "off-line"))
			{
				iRet = AC_OFF;
			}
			else if (!strcmp(value, "on-line"))
			{
				iRet = AC_ON;
			}

			else
			{
				iRet = AC_UNKNOWN;
			}
		}

		fclose(acFile);
	}
	else
	{
		return AC_UNKNOWN;
	}
	return iRet;
}

int GetValueByColon(FILE *bat_file, char *value, const size_t value_size,
                        char *key,  const size_t key_size)
{
	char *pRet = NULL;
	char line[MAX_LINE_LENGTH + 1] = "";
	int sLength = 0;
	int index = 0;
	int index_end = 0;
	int index_mid = 0;

	pRet = fgets(line, sizeof(line), bat_file);
	if (!pRet)
	{
		strcpy(value, "unknown");
		return -1;
	}

	sLength = strlen(line);

	for (index = 0;
        index < MAX_LINE_LENGTH && line[index] != ':' && line[index] != '\0' && line[index] != '\n';
        index++);


    if (index >= key_size)
	{
		strcpy(value, "unknown");
		return -1;
	}

	if (index >= MAX_LINE_LENGTH || line[index] != ':')
	{
		strcpy(value, "unknown");
		return -1;
	}


	if (isspace(line[0]) || index < 2)
	{
		return -1;
	}

	/*skip  all spaces */
	for (index_end = index - 1; index_end >= 0 && isspace(line[index_end]); index_end--);

	index_end++;
	memcpy (key, line, index_end);

	*(key + index_end) = '\0';
	index++;

	/*skip all spaces*/
	for (index_mid = index;
        index_mid < sLength && index_mid < MAX_LINE_LENGTH && isspace(line[index_mid]);
        index_mid++);

	for (index = index_mid;
        index < sLength && index < MAX_LINE_LENGTH && *(line + index) != '\n' && *(line + index) != '\0';
        index++);

	/*skip all spaces*/
	for (index_end = sLength-1; index_end >= index && isspace(line[index_end]) ; index_end--);

	index_end++;
	if (index_end - index_mid >= value_size)
	{
		strcpy(value, "unknown");
		return -1;
	}

	memcpy (value, line + index_mid, index_end - index_mid);

	*(value + (index_end - index_mid)) = '\0';
	return 1;
}

int GetDir(const int number, char *file_dest, const size_t file_size, const char *dir)
{

    int dirSum = 0;
	DIR *dir_dest;
	struct dirent *dirent;

	dir_dest = opendir(dir);

	if (!dir_dest)
	{
		return -1;
	}

	while ((dirent = readdir(dir_dest)))
	{
		if (dirent->d_name[0] == '.')
		{
			continue;
		}
        if (number == dirSum)
        {
			break;
        }

		dirSum++;
	}

	if (!dirent || !file_dest || !dirent->d_name)
	{
		closedir(dir_dest);
		return -1;
	}

	strcpy(file_dest, dirent->d_name);
	closedir(dir_dest);
	return 1;
}

int GetDevicesNum(const char* dir)
{
    int deviceCount = 0;
	DIR *dirParent;
	struct dirent *dirent;

	dirParent = opendir(dir);
	if (!dirParent)
	{
		return -1;
	}

	while (NULL != (dirent = readdir(dirParent)))
	{
		if (dirent->d_name[0] == '.')
		{
			continue;
		}
		deviceCount++;
	}

	closedir(dirParent);
	return deviceCount;
}

int GetBatteryInfoofGeneral(Battery_General *bginfo)
{
    int acpi_avia = 0;
    int iCount = 0;
	Battery_Detailed batDetail;
	int remCapacity = 0;
	int fullCapacity = 0;
	int presentRate = 0;
	int comparer = 0;
	float remBatSum = 0;
	int divCount = 0;

	acpi_avia = CheckACPIDirExit(ACPI_BAT_DIR);
	if (acpi_avia < 0)
		return acpi_avia;

	for (iCount = 0; (GetBatteryInfoByDetail(iCount, &batDetail) == 0) && iCount < MAX_BAT_COUNT; iCount++)
	{
		if (batDetail.present == PRESENT_YES)
		{
			acpi_avia = 1;
			if (batDetail.fullCapacity > 0)
			{
				fullCapacity += (batDetail.remainingCapacity > batDetail.fullCapacity) ? batDetail.remainingCapacity : batDetail.fullCapacity;
				remCapacity += batDetail.remainingCapacity;
				presentRate += batDetail.presentRate;
			}
			bginfo->chargingState = (bginfo->chargingState | batDetail.chargingState);
		}
	}

	if (!acpi_avia)
	{
		return 0;
	}

	if (remCapacity <= 0)
	{
		return 1;
	}
	else
	{
		if (GetACAdapterStatus() == AC_OFF)
		{
			if (presentRate > 0)
			{
				bginfo->remMins =  (int)roundf(((float)(remCapacity) / presentRate) * 60.0);
			}
			else
			{
				static int pre_remMins = -1;
				if (remCapacityArray[gCurrentSnapshot].remCapacity == remCapacity)
				{
					bginfo->remMins = pre_remMins;
				}
				else
				{
					gCurrentSnapshot++;
					if (gCurrentSnapshot > 9)
					{
						gCurrentSnapshot = 0;
					}
					remCapacityArray[gCurrentSnapshot].measurePoint = time(NULL);
					remCapacityArray[gCurrentSnapshot].remCapacity = remCapacity;

					int snapshot = gCurrentSnapshot - 1;
					int temp = 0;
					for (; temp < 9; snapshot--, temp++)
					{
						if (snapshot < 0)
						{
							comparer = snapshot + 10;
						}
						else
						{
							comparer = snapshot;
						}

						if (remCapacityArray[comparer].measurePoint == 0
						    || remCapacityArray[gCurrentSnapshot].measurePoint < remCapacityArray[comparer].measurePoint
						    || remCapacityArray[gCurrentSnapshot].remCapacity > remCapacityArray[comparer].remCapacity)
							break;

						remBatSum += (float) remCapacityArray[gCurrentSnapshot].remCapacity
							* (remCapacityArray[gCurrentSnapshot].measurePoint -
							   remCapacityArray[comparer].measurePoint)
							/ ((remCapacityArray[comparer].remCapacity -
							    remCapacityArray[gCurrentSnapshot].remCapacity) * 60);
						divCount++;
					}

					if (divCount > 1)
					{
						bginfo->remMins = remBatSum / divCount;
						pre_remMins = bginfo->remMins;
					}
				}
			}
		}
		else if (fullCapacity > 0)
		{
			if (presentRate > 0)
			{
				bginfo->remChargingMins = (int)roundf(((float)(fullCapacity - remCapacity)
								      / presentRate) * 60.0);
			}
			else
			{
				if (remCapacityArray[gCurrentSnapshot].remCapacity == remCapacity)
				{
				}
				else
				{
					gCurrentSnapshot++;
					if (gCurrentSnapshot > 9)
					{
						gCurrentSnapshot = 0;
					}

					remCapacityArray[gCurrentSnapshot].measurePoint = time(NULL);
					remCapacityArray[gCurrentSnapshot].remCapacity = remCapacity;

					int snapshot = gCurrentSnapshot - 1;
					int temp = 0;

					for (; temp < 9; snapshot--, temp++)
					{
						if (snapshot < 0)
                        {
							comparer = snapshot + 10;
                        }
						else
						{
							comparer = snapshot;
						}

						if (remCapacityArray[comparer].measurePoint == 0
						    || remCapacityArray[gCurrentSnapshot].measurePoint < remCapacityArray[comparer].measurePoint
						    || remCapacityArray[gCurrentSnapshot].remCapacity > remCapacityArray[comparer].remCapacity)
                        {
							break;
                        }

						remBatSum += (float) (remCapacityArray[gCurrentSnapshot].remCapacity - fullCapacity)
							* (remCapacityArray[gCurrentSnapshot].measurePoint - remCapacityArray[comparer].measurePoint)
							/ ((remCapacityArray[gCurrentSnapshot].remCapacity - remCapacityArray[comparer].remCapacity) * 60);
						divCount++;
					}

					if (divCount > 1)
					{
						bginfo->remMins = remBatSum / divCount;
					}
				}
			}
		}

		if (fullCapacity > 0)
		{
            bginfo->remPerc = (int)roundf(((float)remCapacity / (float)fullCapacity) * 100.0);

            if (bginfo->remPerc > 100 || bginfo->remPerc < 0)
            {
                bginfo->remPerc = 100;
            }
		}

	}
	return 1;
}
