/*
 * Copyright (C) 2008  Intel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License, version 2.1, as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "PMApi.h"
#include "PMPlugin.h"
#include "ReqMgmt.h"
#include "ConnMgmt.h"

extern int DCS_WaitResponse(DCS_Response* pResponse, int, int);
extern DCS_Return_Code DCS_DispatchEvent();
extern int DCS_Lock();
extern int DCS_Unlock();
extern void WINAPI DebugString(const char* format, ...);
static int GetModuleID()
{
	return DCS_POWER_MANAGER;
}

DCS_Return_Code DCS_GetCPUSpeed(DCS_CPUSpeed* pSpeed)
{
    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();
	pReqMgmt->GenFuncReq(GetModuleID(),PM_GET_CPU_SPEED, 0, NULL);
	
	DCS_Response response;
	
	memset(&response, 0, sizeof(response));
	DCS_DispatchEvent();
	
	DCS_WaitResponse(&response,  GetModuleID(), PM_GET_CPU_SPEED);

	if (response.data != NULL)
		*pSpeed = *((DCS_CPUSpeed* )response.data);
    DCS_Unlock();

	if (response.data != NULL)
	{
		delete[]response.data;
		response.data = NULL;
	}

	return response.returnCode;
}

DCS_Return_Code DCS_SetCPUSpeed(DCS_CPUSpeed speed)
{
    DCS_Lock();
	DCS_Return_Code ret = DCS_SUCCESS;
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();
	ret = pReqMgmt->GenFuncReq(GetModuleID(), PM_CPU_SPEED_CHANGED, sizeof(DCS_CPUSpeed),&speed);
    if (ret){    DCS_Unlock(); return ret;}
	DCS_Response response;
	
	memset(&response, 0, sizeof(response));
	DCS_DispatchEvent();
	
	DCS_WaitResponse(&response,  GetModuleID(), PM_CPU_SPEED_CHANGED);
    DCS_Unlock();

	return response.returnCode;
}

DCS_Return_Code DCS_GetLCDMaxBrightness(int* bright)
{
    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();	
	pReqMgmt->GenFuncReq(GetModuleID(),PM_GET_MAX_LCD_BRIGHTNESS, 0, NULL);

	DCS_Response response;
	
	memset(&response, 0, sizeof(response));
	DCS_DispatchEvent();
	
	DCS_WaitResponse(&response,  GetModuleID(), PM_GET_MAX_LCD_BRIGHTNESS);

	if (response.data != NULL)
		*bright = *((int* )response.data);
    DCS_Unlock();

	if (response.data != NULL)
	{
		delete[]response.data;
		response.data = NULL;
	}

	return response.returnCode;
}

DCS_Return_Code DCS_GetLCDBrightness(int* bright)
{
    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();	
	pReqMgmt->GenFuncReq(GetModuleID(),PM_GET_LCD_BRIGHTNESS, 0, NULL);

	DCS_Response response;
	
	*bright = 0;

	memset(&response, 0, sizeof(response));
	DCS_DispatchEvent();
	
	DCS_WaitResponse(&response,  GetModuleID(), PM_GET_LCD_BRIGHTNESS);

	if (response.data != NULL)
		*bright = *((int* )response.data);

	if (*bright >= 8)   //this is just one error exception catch, this means error occurs
	{
		*bright = 7;
		DebugString("error occurs... GetLCDBrightness get bad value.");
	}
	DCS_Unlock();

	if (response.data != NULL)
	{
		delete[]response.data;
		response.data = NULL;
	}

	return response.returnCode;
}

DCS_Return_Code DCS_SetLCDBrightness(int bright)
{
	if (bright >= 8) return DCS_FAIL_OPERATION;  //temporary error handling.

    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();
	pReqMgmt->GenFuncReq(GetModuleID(),PM_SET_LCD_BRIGHTNESS, sizeof(int), &bright);
	
	DCS_Response response;
	
	memset(&response, 0, sizeof(response));
	DCS_DispatchEvent();
	
	DCS_WaitResponse(&response,  GetModuleID(), PM_SET_LCD_BRIGHTNESS);
	DCS_Unlock();
	return response.returnCode;
}

DCS_Return_Code DCS_GetWirelessStatus(BOOL* pEnabled)
{
    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();
	pReqMgmt->GenFuncReq(GetModuleID(),PM_GET_WIRELESS_STATUS, 0, NULL);
	DCS_Response response;
	
	memset(&response, 0, sizeof(response));
	DCS_DispatchEvent();
	
	DCS_WaitResponse(&response,  GetModuleID(), PM_GET_WIRELESS_STATUS);

	if (response.data != NULL)
		*pEnabled = *((BOOL* )response.data);
	DCS_Unlock();

	if (response.data != NULL)
	{
		delete[]response.data;
		response.data = NULL;
	}

	return response.returnCode;
}

DCSAPI DCS_Return_Code DCS_GetWirelessEnableStatus(BOOL* pEnabled)
{
    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();
	pReqMgmt->GenFuncReq(GetModuleID(),PM_GET_WIRELESS_ENABLE_STATUS, 0, NULL);
	DCS_Response response;
	
	memset(&response, 0, sizeof(response));
	DCS_DispatchEvent();
	
	DCS_WaitResponse(&response,  GetModuleID(), PM_GET_WIRELESS_ENABLE_STATUS);

	if (response.data != NULL)
		*pEnabled = *((BOOL* )response.data);
	DCS_Unlock();

	if (response.data != NULL)
	{
		delete[]response.data;
		response.data = NULL;
	}

	return response.returnCode;
}

DCS_Return_Code DCS_SetWirelessStatus(BOOL enable)
{
    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();
	pReqMgmt->GenFuncReq(GetModuleID(),PM_SET_WIRELESS_STATUS, sizeof(BOOL), &enable);
	DCS_Response response;
	
	memset(&response, 0, sizeof(response));
	DCS_DispatchEvent();
	
	DCS_WaitResponse(&response,  GetModuleID(), PM_SET_WIRELESS_STATUS);
	DCS_Unlock();

	return response.returnCode;
}

DCS_Return_Code DCS_GetLANStatus(BOOL* pEnabled)
{
    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();
	pReqMgmt->GenFuncReq(GetModuleID(),PM_GET_LAN_STATUS, 0, NULL);
	DCS_Response response;
	
	memset(&response, 0, sizeof(response));
	DCS_DispatchEvent();
	
	DCS_WaitResponse(&response,  GetModuleID(), PM_GET_LAN_STATUS);

	if (response.data != NULL)
		*pEnabled = *((BOOL* )response.data);
	DCS_Unlock();

	if (response.data != NULL)
	{
		delete[]response.data;
		response.data = NULL;
	}

	return response.returnCode;
}

DCSAPI DCS_Return_Code DCS_GetLANEnableStatus(BOOL* pEnabled)
{
    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();
	pReqMgmt->GenFuncReq(GetModuleID(),PM_GET_LAN_ENABLE_STATUS, 0, NULL);
	DCS_Response response;
	
	memset(&response, 0, sizeof(response));
	DCS_DispatchEvent();
	
	DCS_WaitResponse(&response,  GetModuleID(), PM_GET_LAN_ENABLE_STATUS);

	if (response.data != NULL)
		*pEnabled = *((BOOL* )response.data);
	DCS_Unlock();

	if (response.data != NULL)
	{
		delete[]response.data;
		response.data = NULL;
	}

	return response.returnCode;
}

DCS_Return_Code DCS_SetLANStatus(BOOL enable)
{
    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();
	pReqMgmt->GenFuncReq(GetModuleID(),PM_SET_LAN_STATUS, sizeof(BOOL), &enable);
	DCS_Response response;
	
	memset(&response, 0, sizeof(response));
	DCS_DispatchEvent();
	
	DCS_WaitResponse(&response,  GetModuleID(), PM_SET_LAN_STATUS);
	DCS_Unlock();

	return response.returnCode;
}

DCS_Return_Code DCS_GetLANSpeed(DCS_Lan_Speed* pSpeed)
{
    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();
	pReqMgmt->GenFuncReq(GetModuleID(),PM_GET_LAN_SPEED, 0, NULL);
	DCS_Response response;
	
	memset(&response, 0, sizeof(response));
	DCS_DispatchEvent();
	
	DCS_WaitResponse(&response,  GetModuleID(), PM_GET_LAN_SPEED);

	
	if (response.data != NULL)
		*pSpeed = *((DCS_Lan_Speed* )response.data);
	DCS_Unlock();

	if (response.data != NULL)
	{
		delete[]response.data;
		response.data = NULL;
	}

    return response.returnCode;	
}

DCS_Return_Code DCS_SetLANSpeed(DCS_Lan_Speed speed)
{
    DCS_Lock();
	DCS_Return_Code ret = DCS_SUCCESS;
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();
	ret = pReqMgmt->GenFuncReq(GetModuleID(), PM_LAN_SPEED_CHANGED, sizeof(DCS_Lan_Speed),&speed);
    if (ret) {    DCS_Unlock(); return ret;}
	DCS_Response response;
	
	memset(&response, 0, sizeof(response));
	DCS_DispatchEvent();
	
	DCS_WaitResponse(&response,  GetModuleID(), PM_LAN_SPEED_CHANGED);
	DCS_Unlock();
	return response.returnCode;
}

DCS_Return_Code DCS_GetDPSTLevel(DCS_DPST* pLevel)
{
    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();
	pReqMgmt->GenFuncReq(GetModuleID(),PM_GET_DPST_LEVEL, 0, NULL);
	DCS_Response response;
	
	memset(&response, 0, sizeof(response));
	DCS_DispatchEvent();
	
	DCS_WaitResponse(&response,  GetModuleID(), PM_GET_DPST_LEVEL);

	if (response.data != NULL)
		*pLevel = *((DCS_DPST* )response.data);
	DCS_Unlock();

	if (response.data != NULL)
	{
		delete[]response.data;
		response.data = NULL;
	}

	return response.returnCode;
}

DCS_Return_Code DCS_SetDPSTLevel(DCS_DPST level)
{
    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();
	pReqMgmt->GenFuncReq(GetModuleID(),PM_SET_DPST_LEVEL, sizeof(DCS_DPST), &level);
	DCS_Response response;
	
	memset(&response, 0, sizeof(response));
	DCS_DispatchEvent();
	
	DCS_WaitResponse(&response,  GetModuleID(), PM_SET_DPST_LEVEL);
	DCS_Unlock();
	return response.returnCode;
}

/** Gets the battery count. */
DCS_Return_Code DCS_GetBatteryCount(ULONG * batCount)
{
    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();	
	pReqMgmt->GenFuncReq(GetModuleID(),PM_GET_BATTERY_COUNT, 0, NULL);
	DCS_Response response;
	
	memset(&response, 0, sizeof(response));
	DCS_DispatchEvent();
	
	DCS_WaitResponse(&response,  GetModuleID(), PM_GET_BATTERY_COUNT);

	if (response.data != NULL)
		*batCount = *((ULONG* )response.data);
	DCS_Unlock();

	if (response.data != NULL)
	{
		delete[]response.data;
		response.data = NULL;
	}

	return response.returnCode;
}

DCS_Return_Code DCS_GetBatteryInfo(PDCS_Battery_Info battery,int index)
{
    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();	
	pReqMgmt->GenFuncReq(GetModuleID(),PM_GET_BATTERY_INFO, sizeof(int), &index);
	DCS_Response response;
	
	memset(&response, 0, sizeof(response));
	DCS_DispatchEvent();
	
	DCS_WaitResponse(&response,  GetModuleID(), PM_GET_BATTERY_INFO);

	if (response.data != NULL)
	{
		*battery = *((PDCS_Battery_Info )response.data);

		battery->Chemistry[0] = ((PDCS_Battery_Info )response.data)->Chemistry[0];
		battery->Chemistry[1] = ((PDCS_Battery_Info )response.data)->Chemistry[1];
		battery->Chemistry[2] = ((PDCS_Battery_Info )response.data)->Chemistry[2];
		battery->Chemistry[3] = ((PDCS_Battery_Info )response.data)->Chemistry[3];

		battery->Reserved[0] = ((PDCS_Battery_Info )response.data)->Reserved[0];
		battery->Reserved[1] = ((PDCS_Battery_Info )response.data)->Reserved[1];
		battery->Reserved[2] = ((PDCS_Battery_Info )response.data)->Reserved[2];
	}

	DCS_Unlock();

	if (response.data != NULL)
	{
		delete[]response.data;
		response.data = NULL;
	}

	return response.returnCode;
}

DCSAPI DCS_Return_Code DCS_GetPowerStatus(PDCS_Power_Status battery)
{
    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();	
	pReqMgmt->GenFuncReq(GetModuleID(),PM_GET_POWER_INFO, 0, NULL);
	DCS_Response response;
	
	memset(&response, 0, sizeof(response));
	DCS_DispatchEvent();
	
	DCS_WaitResponse(&response,  GetModuleID(), PM_GET_POWER_INFO);

	if (response.data != NULL)
	{
		*battery = *((PDCS_Power_Status )response.data);
	}

	DCS_Unlock();

	if (response.data != NULL)
	{
		delete[]response.data;
		response.data = NULL;
	}

	return response.returnCode;
}

DCS_Return_Code DCS_GetPwrSchemeNameDesc(int schemeId, char * pSchemeName,int nameSize, char * pSchemeDesc, int descSize)
{
	if (nameSize<1 || descSize<1) return DCS_FAIL_OPERATION;
    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();	
	pReqMgmt->GenFuncReq(GetModuleID(),PM_GET_SCHEME_NAME, sizeof(int), &schemeId);
	DCS_Response response;
	
	memset(&response, 0, sizeof(response));
	DCS_DispatchEvent();
	
	DCS_WaitResponse(&response,  GetModuleID(), PM_GET_SCHEME_NAME);

	DebugString("DCS_GetPwrSchemeNameDesc:%s, %s", ((PM_Power_Scheme_Data*)response.data)->pName, ((PM_Power_Scheme_Data*)response.data)->pDesc);
	if (response.data != NULL)
	{
		strncpy(pSchemeName, ((PM_Power_Scheme_Data*)response.data)->pName, nameSize-1);
		pSchemeName[nameSize-1] = '\0';
		strncpy(pSchemeDesc , ((PM_Power_Scheme_Data*)response.data)->pDesc,descSize-1 );
		pSchemeDesc[descSize-1] = '\0';
		
	}

	DCS_Unlock();

	if (response.data != NULL)
	{
		delete[]response.data;
		response.data = NULL;
	}
	return response.returnCode;
}

DCS_Return_Code DCS_SetActivePwrScheme(int schemeId)
{
    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();
	pReqMgmt->GenFuncReq(GetModuleID(),PM_APPLY_SCHEME, sizeof(int), &schemeId);
	DCS_Response response;
	
	memset(&response, 0, sizeof(response));
	DCS_DispatchEvent();
	
	DCS_WaitResponse(&response,  GetModuleID(), PM_APPLY_SCHEME);
	DCS_Unlock();
	return response.returnCode;
}

DCS_Return_Code DCS_AddPwrScheme(char *pName, char *pDesc, int copyFrom, int *schemeId)
{
    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();
	PM_Power_Scheme_Data PwrSchemeData;
	memcpy(PwrSchemeData.pName,pName,MAX_STRING_LEN);
	memcpy(PwrSchemeData.pDesc,pDesc,MAX_STRING_LEN);
	PwrSchemeData.copyFrom=copyFrom;
	PwrSchemeData.schemeId=*schemeId;

	pReqMgmt->GenFuncReq(GetModuleID(),PM_ADD_SCHEME, sizeof(PM_Power_Scheme_Data), &PwrSchemeData);
	DCS_Response response;
	
	memset(&response, 0, sizeof(response));
	DCS_DispatchEvent();
	
	DCS_WaitResponse(&response,  GetModuleID(), PM_ADD_SCHEME);
	if (response.data != NULL)
	{
		*schemeId = *((int* )response.data);
	}

	DCS_Unlock();

	if (response.data != NULL)
	{
		delete[]response.data;
		response.data = NULL;
	}
	return response.returnCode;
}

DCSAPI DCS_Return_Code DCS_AddPwrScheme_All(PPM_Power_Scheme_Data_ALL pSchemeDataAll, int *schemeId)
{
    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();

	PM_Power_Scheme_Data_ALL PwrSchemeDataAll;
	memcpy(&PwrSchemeDataAll, pSchemeDataAll, sizeof(PM_Power_Scheme_Data_ALL));

	pReqMgmt->GenFuncReq(GetModuleID(),PM_ADD_SCHEME_ALL, sizeof(PM_Power_Scheme_Data_ALL), &PwrSchemeDataAll);
	DCS_Response response;
	
	memset(&response, 0, sizeof(response));
	DCS_DispatchEvent();
	
	DCS_WaitResponse(&response,  GetModuleID(), PM_ADD_SCHEME_ALL);
	if (response.data != NULL)
	{
		*schemeId = *((int* )response.data);
	}

	DCS_Unlock();

	if (response.data != NULL)
	{
		delete[]response.data;
		response.data = NULL;
	}
	return response.returnCode;
}

DCS_Return_Code DCS_DeletePwrScheme(int schemeId)
{
    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();

	pReqMgmt->GenFuncReq(GetModuleID(),PM_DELETE_SCHEME, sizeof(int), &schemeId);
	DCS_Response response;
	
	memset(&response, 0, sizeof(response));
	DCS_DispatchEvent();
	
	DCS_WaitResponse(&response,  GetModuleID(), PM_DELETE_SCHEME);
	DCS_Unlock();
	
	return response.returnCode;
}

DCS_Return_Code DCS_SetPwrVariable(int schemeId, BOOL isAC, const char *pVarName, int value)
{
	if (pVarName == NULL) return DCS_FAIL_OPERATION;
    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();
	
	PM_Variable_Data pm_Varible_Data;
	pm_Varible_Data.schemeId = schemeId;
	pm_Varible_Data.isAC = isAC;
	memcpy(pm_Varible_Data.pVarName , pVarName,MAX_STRING_LEN);
	pm_Varible_Data.value=value;

	pReqMgmt->GenFuncReq(GetModuleID(),PM_EDIT_SCHEME, sizeof(pm_Varible_Data), &pm_Varible_Data);
	DCS_Response response;
	
	memset(&response, 0, sizeof(response));
	DCS_DispatchEvent();
	
	DCS_WaitResponse(&response,  GetModuleID(), PM_EDIT_SCHEME);
	DCS_Unlock();
    return response.returnCode;
	
}

DCSAPI DCS_Return_Code DCS_SetPwrVariable_All(PPM_Power_Scheme_Data_ALL pSchemeDataAll)
{
    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();
	
	PM_Power_Scheme_Data_ALL PwrSchemeDataAll;
	memcpy(&PwrSchemeDataAll, pSchemeDataAll, sizeof(PM_Power_Scheme_Data_ALL));

	pReqMgmt->GenFuncReq(GetModuleID(),PM_EDIT_SCHEME_ALL, sizeof(PM_Power_Scheme_Data_ALL), &PwrSchemeDataAll);
	DCS_Response response;
	
	memset(&response, 0, sizeof(response));
	DCS_DispatchEvent();
	
	DCS_WaitResponse(&response,  GetModuleID(), PM_EDIT_SCHEME_ALL);
	DCS_Unlock();
    return response.returnCode;
}

DCS_Return_Code DCS_GetPwrVariable(int schemeId, BOOL isAC, const char *pVarName, int *pValue)
{
	if (pVarName == NULL) return DCS_FAIL_OPERATION;
	if (pValue == NULL)
		return DCS_FAIL_OPERATION;

    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();
	PM_Variable_Data pm_Varible_Data;
	pm_Varible_Data.schemeId = schemeId;
	pm_Varible_Data.isAC = isAC;
	memcpy(pm_Varible_Data.pVarName , pVarName,MAX_STRING_LEN);

	if (pVarName == NULL)
    {DCS_Unlock();	return DCS_FAIL_OPERATION;}

	pReqMgmt->GenFuncReq(GetModuleID(),PM_QUERY_SCHEME, sizeof(PM_Variable_Data), &pm_Varible_Data);
	DCS_Response response;
	
	memset(&response, 0, sizeof(response));
	DCS_DispatchEvent();
	
	DCS_WaitResponse(&response,  GetModuleID(), PM_QUERY_SCHEME);
		if (response.data != NULL)
		{
			*pValue = *((int* )response.data);
		}
    DCS_Unlock();

	if (response.data != NULL)
	{
		delete[]response.data;
		response.data = NULL;
	}

    return response.returnCode;
}

DCSAPI DCS_Return_Code DCS_GetPwrVariable_All(PPM_Power_Scheme_Data_ALL pSchemeDataAll, int schemeId)
{
	if (pSchemeDataAll == NULL)
		return DCS_FAIL_OPERATION;

    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();

	pReqMgmt->GenFuncReq(GetModuleID(),PM_QUERY_SCHEME_ALL, sizeof(int), &schemeId);
	DCS_Response response;
	
	memset(&response, 0, sizeof(response));
	DCS_DispatchEvent();
	
	DCS_WaitResponse(&response,  GetModuleID(), PM_QUERY_SCHEME_ALL);
		if (response.data != NULL)
		{
			*pSchemeDataAll = *((PPM_Power_Scheme_Data_ALL )response.data);
		}
    DCS_Unlock();

	if (response.data != NULL)
	{
		delete[]response.data;
		response.data = NULL;
	}

    return response.returnCode;
}

DCSAPI DCS_Return_Code DCS_GetPwrVariableOptCount(const char *pVarName, int *pValue)
{
	if (pVarName == NULL) return DCS_FAIL_OPERATION;
	if (pValue == NULL)
		return DCS_FAIL_OPERATION;

    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();

	PM_Variable_Data pm_Varible_Data;
	pm_Varible_Data.schemeId = 0;
	pm_Varible_Data.isAC = 0;
	pm_Varible_Data.value = 0;
	memcpy(pm_Varible_Data.pVarName, pVarName, MAX_STRING_LEN);

	if (pVarName == NULL)
    {DCS_Unlock();	return DCS_FAIL_OPERATION;}

	pReqMgmt->GenFuncReq(GetModuleID(),PM_GET_SCHEME_VAR_COUNT, sizeof(PM_Variable_Data), &pm_Varible_Data);
	DCS_Response response;
	
	memset(&response, 0, sizeof(response));
	DCS_DispatchEvent();
	
	DCS_WaitResponse(&response,  GetModuleID(), PM_GET_SCHEME_VAR_COUNT);
		if (response.data != NULL)
		{
			*pValue = *((int* )response.data);
		}
    DCS_Unlock();

	if (response.data != NULL)
	{
		delete[]response.data;
		response.data = NULL;
	}

    return response.returnCode;
}

DCSAPI DCS_Return_Code DCS_GetPwrVariableOptValue(const char *pVarName, int index, int *pValue)
{
	if (pVarName == NULL) return DCS_FAIL_OPERATION;
	if (pValue == NULL)
		return DCS_FAIL_OPERATION;

    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();

	PM_Variable_Data pm_Varible_Data;
	pm_Varible_Data.schemeId = 0;
	pm_Varible_Data.isAC = 0;
	pm_Varible_Data.value = index;
	memcpy(pm_Varible_Data.pVarName , pVarName,MAX_STRING_LEN);

	if (pVarName == NULL)
    {DCS_Unlock();	return DCS_FAIL_OPERATION;}

	pReqMgmt->GenFuncReq(GetModuleID(),PM_GET_SCHEME_VAR_OPT_VALUE, sizeof(PM_Variable_Data), &pm_Varible_Data);
	DCS_Response response;
	
	memset(&response, 0, sizeof(response));
	DCS_DispatchEvent();
	
	DCS_WaitResponse(&response,  GetModuleID(), PM_GET_SCHEME_VAR_OPT_VALUE);
		if (response.data != NULL)
		{
			*pValue = *((int* )response.data);
		}
    DCS_Unlock();

	if (response.data != NULL)
	{
		delete[]response.data;
		response.data = NULL;
	}

    return response.returnCode;
}

DCSAPI DCS_Return_Code DCS_GetUserID(char * pUserID, int bufSize)
{
    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();
	pReqMgmt->GenFuncReq(GetModuleID(),PM_GET_USERID, 0, NULL);
	DCS_Response response;
	
	memset(&response, 0, sizeof(response));
	DCS_DispatchEvent();
	
	DCS_WaitResponse(&response,  GetModuleID(), PM_GET_USERID);
		if (response.data != NULL)
		{
			strncpy(pUserID, response.data, bufSize-1);
			pUserID[bufSize-1] = '\0';
		}
    DCS_Unlock();

	if (response.data != NULL)
	{
		delete[]response.data;
		response.data = NULL;
	}

    return response.returnCode;    
}

DCSAPI DCS_Return_Code DCS_GetActivePwrScheme(int * pSchemeId)
{
    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();
	pReqMgmt->GenFuncReq(GetModuleID(),PM_GET_ACTIVE_SCHEME, 0, NULL);
	DCS_Response response;
	
	memset(&response, 0, sizeof(response));
	DCS_DispatchEvent();
	
	DCS_WaitResponse(&response,  GetModuleID(), PM_GET_ACTIVE_SCHEME);
		if (response.data != NULL)
		{
			*pSchemeId = *((int* )response.data);
		}
    DCS_Unlock();

	if (response.data != NULL)
	{
		delete[]response.data;
		response.data = NULL;
	}

    return response.returnCode;    
}

DCSAPI DCS_Return_Code DCS_InitCfgFile()
{
    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();
	pReqMgmt->GenFuncReq(GetModuleID(),PM_INIT_CFG_FILE, 0, NULL);
	DCS_Response response;
	
	memset(&response, 0, sizeof(response));
	DCS_DispatchEvent();
	
	DCS_WaitResponse(&response,  GetModuleID(), PM_INIT_CFG_FILE);
	DCS_Unlock();
	return response.returnCode;
}

DCSAPI DCS_Return_Code DCS_GetIntFromCfg(int nFlag, 
										 const char * pVarFile, 
										 const char * pVarSec, 
										 const char * pVarItem,
										 int nDefault, 
										 int * pValue)
 {
 	if (pVarFile == NULL) return DCS_FAIL_OPERATION;
	if (pVarSec == NULL)  return DCS_FAIL_OPERATION;
	if (pVarItem == NULL)  return DCS_FAIL_OPERATION;

    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();

	PM_Configure_Data pm_Config_Data;
	pm_Config_Data.nFlag = nFlag;
	pm_Config_Data.defValue = nDefault;
	pm_Config_Data.value = 0;
	memcpy(pm_Config_Data.pFile, pVarFile,MAX_STRING_LEN);
	memcpy(pm_Config_Data.pSec, pVarSec,MAX_STRING_LEN);
	memcpy(pm_Config_Data.pItem, pVarItem,MAX_STRING_LEN);
	memset(pm_Config_Data.pValue,0,MAX_STRING_LEN);

	if (pVarFile == NULL)
    {DCS_Unlock();	return DCS_FAIL_OPERATION;}

	pReqMgmt->GenFuncReq(GetModuleID(),PM_GET_INT_FROM_CONFIG, sizeof(PM_Configure_Data), &pm_Config_Data);
	DCS_Response response;
	
	memset(&response, 0, sizeof(response));
	DCS_DispatchEvent();
	
	DCS_WaitResponse(&response,  GetModuleID(), PM_GET_INT_FROM_CONFIG);
		if (response.data != NULL)
		{
			*pValue = *((int* )response.data);
		}
    DCS_Unlock();

	if (response.data != NULL)
	{
		delete[]response.data;
		response.data = NULL;
	}

    return response.returnCode;
 }

 DCSAPI DCS_Return_Code DCS_PutIntIntoCfg(int nFlag, 
										 const char * pVarFile, 
										 const char * pVarSec, 
										 const char * pVarItem,
										 int value)
 {
 	if (pVarFile == NULL) return DCS_FAIL_OPERATION;
 	if (pVarSec == NULL) return DCS_FAIL_OPERATION;
 	if (pVarItem == NULL) return DCS_FAIL_OPERATION;

    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();
	
	PM_Configure_Data pm_Config_Data;
	pm_Config_Data.nFlag = nFlag;
	pm_Config_Data.defValue = 0;
	pm_Config_Data.value = value;
	memcpy(pm_Config_Data.pFile, pVarFile,MAX_STRING_LEN);
	memcpy(pm_Config_Data.pSec, pVarSec,MAX_STRING_LEN);
	memcpy(pm_Config_Data.pItem, pVarItem,MAX_STRING_LEN);
	memset(pm_Config_Data.pValue,0,MAX_STRING_LEN);

	pReqMgmt->GenFuncReq(GetModuleID(),PM_PUT_INT_INTO_CONFIG, sizeof(PM_Configure_Data), &pm_Config_Data);
	DCS_Response response;
	
	memset(&response, 0, sizeof(response));
	DCS_DispatchEvent();
	
	DCS_WaitResponse(&response,  GetModuleID(), PM_PUT_INT_INTO_CONFIG);
	DCS_Unlock();
    return response.returnCode;
}

DCSAPI DCS_Return_Code DCS_PutStringIntoCfg(int nFlag, 
										 const char * pVarFile, 
										 const char * pVarSec, 
										 const char * pVarItem,
										 const char * pVarValue)
{
 	if (pVarFile == NULL) return DCS_FAIL_OPERATION;
 	if (pVarSec == NULL) return DCS_FAIL_OPERATION;
 	if (pVarItem == NULL) return DCS_FAIL_OPERATION;
 	if (pVarValue == NULL) return DCS_FAIL_OPERATION;

    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();
	
	PM_Configure_Data pm_Config_Data;
	pm_Config_Data.nFlag = nFlag;
	pm_Config_Data.defValue = 0;
	pm_Config_Data.value = 0;
	memcpy(pm_Config_Data.pFile, pVarFile,MAX_STRING_LEN);
	memcpy(pm_Config_Data.pSec, pVarSec,MAX_STRING_LEN);
	memcpy(pm_Config_Data.pItem, pVarItem,MAX_STRING_LEN);
	memcpy(pm_Config_Data.pValue, pVarValue,MAX_STRING_LEN);

	pReqMgmt->GenFuncReq(GetModuleID(),PM_PUT_STR_INTO_CONFIG, sizeof(PM_Configure_Data), &pm_Config_Data);
	DCS_Response response;
	
	memset(&response, 0, sizeof(response));
	DCS_DispatchEvent();
	
	DCS_WaitResponse(&response,  GetModuleID(), PM_PUT_STR_INTO_CONFIG);
	DCS_Unlock();
    return response.returnCode;
}

DCSAPI DCS_Return_Code DCS_GetStringFromCfg(int nFlag, 
										 const char * pVarFile, 
										 const char * pVarSec, 
										 const char * pVarItem,
										 char * pVarValue,
										 int )
{
 	if (pVarFile == NULL) return DCS_FAIL_OPERATION;
 	if (pVarSec == NULL) return DCS_FAIL_OPERATION;
 	if (pVarItem == NULL) return DCS_FAIL_OPERATION;
 	if (pVarValue == NULL) return DCS_FAIL_OPERATION;

	DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();	

	PM_Configure_Data pm_Config_Data;
	pm_Config_Data.nFlag = nFlag;
	pm_Config_Data.defValue = 0;
	pm_Config_Data.value = 0;
	memcpy(pm_Config_Data.pFile, pVarFile,MAX_STRING_LEN);
	memcpy(pm_Config_Data.pSec, pVarSec,MAX_STRING_LEN);
	memcpy(pm_Config_Data.pItem, pVarItem,MAX_STRING_LEN);
	memset(pm_Config_Data.pValue, 0, MAX_STRING_LEN);

	pReqMgmt->GenFuncReq(GetModuleID(),PM_GET_STR_FROM_CONFIG, sizeof(PM_Configure_Data), &pm_Config_Data);
	DCS_Response response;
	
	memset(&response, 0, sizeof(response));
	DCS_DispatchEvent();
	
	DCS_WaitResponse(&response,  GetModuleID(), PM_GET_STR_FROM_CONFIG);

	if (response.data != NULL)
	{
		strncpy(pVarValue, response.data, response.dataLen);
		//pVarValue[valueSize-1] = '\0';
	}

	DCS_Unlock();

	if (response.data != NULL)
	{
		delete[]response.data;
		response.data = NULL;
	}
	return response.returnCode;
}

/* Registers the callback function to be notified  when the classmate PC changes the LCD brightness. */
DCSAPI DCS_Return_Code DCS_RegisterLCDBrightnessChangedCallback (DCS_PM_LCDBrightness_Callback pCallback)
{
    DCS_Return_Code ret = DCS_SUCCESS;
    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();
	ret = pReqMgmt->GenEventReq(GetModuleID(), PM_LCD_BRIGHTNESS_CHANGED, (DCS_Callback)pCallback);
    if (ret)   {  DCS_Unlock(); return ret;}
	DCS_DispatchEvent();
    DCS_Unlock();
	return DCS_SUCCESS;		
}

/* Unregisters the callback function for LCD brightness changed notifications. */
DCSAPI DCS_Return_Code DCS_UnregisterLCDBrightnessChangedCallback()
{
    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();
	DCS_Return_Code ret =  pReqMgmt->GenUnregisterEventReq(GetModuleID(),PM_LCD_BRIGHTNESS_CHANGED);	
    //DCS_Return_Code ret =  pReqMgmt->GenEventReq(GetModuleID(),PM_LCDBRIGHTNESS_UNREGISTER_CALLBACK, NULL);	
    DCS_Unlock();
    return ret;
}


/* Registers the callback function to be notified  when the classmate PC changes the enabled/disabled status of WLAN. */
DCSAPI DCS_Return_Code DCS_RegisterWLanChangedCallback (DCS_PM_WLan_Callback pCallback)
{
    DCS_Return_Code ret = DCS_SUCCESS;
    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();
	ret = pReqMgmt->GenEventReq(GetModuleID(), PM_WLAN_CHANGED, (DCS_Callback)pCallback);
    if (ret)   {  DCS_Unlock(); return ret;}
	DCS_DispatchEvent();
    DCS_Unlock();
	return DCS_SUCCESS;		
}


/* Unregisters the callback function for WLAN status changed notifications. */
DCSAPI DCS_Return_Code DCS_UnregisterWLanChangedCallback()
{
    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();
	DCS_Return_Code ret =  pReqMgmt->GenUnregisterEventReq(GetModuleID(),PM_WLAN_CHANGED);	
//	DCS_Return_Code ret =  pReqMgmt->GenEventReq(GetModuleID(),PM_WLAN_UNREGISTER_CALLBACK, NULL);	
    DCS_Unlock();
    return ret;
}

/* Registers the callback function to be notified  when the classmate PC changes the enabled/disabled status of LAN. */
DCSAPI DCS_Return_Code DCS_RegisterLanChangedCallback (DCS_PM_Lan_Callback pCallback)
{
    DCS_Return_Code ret = DCS_SUCCESS;
    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();
	ret = pReqMgmt->GenEventReq(GetModuleID(), PM_LAN_CHANGED, (DCS_Callback)pCallback);
    if (ret)   {  DCS_Unlock(); return ret;}
	DCS_DispatchEvent();
    DCS_Unlock();
	return DCS_SUCCESS;		
}


/* Unregisters the callback function for LAN status changed notifications. */
DCSAPI DCS_Return_Code DCS_UnregisterLanChangedCallback()
{
    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();
	DCS_Return_Code ret =  pReqMgmt->GenUnregisterEventReq(GetModuleID(),PM_LAN_CHANGED);	
//	DCS_Return_Code ret =  pReqMgmt->GenEventReq(GetModuleID(),PM_WLAN_UNREGISTER_CALLBACK, NULL);	
    DCS_Unlock();
    return ret;
}

/* Registers the callback function to be notified  when the classmate PC changes the speed of LAN. */
DCSAPI DCS_Return_Code DCS_RegisterLanSpeedChangedCallback (DCS_PM_Lan_Speed_Callback pCallback)
{
    DCS_Return_Code ret = DCS_SUCCESS;
    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();
	ret = pReqMgmt->GenEventReq(GetModuleID(), PM_LAN_SPEED_CHANGED, (DCS_Callback)pCallback);
    if (ret)   {  DCS_Unlock(); return ret;}
	DCS_DispatchEvent();
    DCS_Unlock();
	return DCS_SUCCESS;		
}


/* Unregisters the callback function for LAN speed changed notifications. */
DCSAPI DCS_Return_Code DCS_UnregisterLanSpeedChangedCallback()
{
    DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();
	DCS_Return_Code ret =  pReqMgmt->GenUnregisterEventReq(GetModuleID(),PM_LAN_SPEED_CHANGED);	
//	DCS_Return_Code ret =  pReqMgmt->GenEventReq(GetModuleID(),PM_WLAN_UNREGISTER_CALLBACK, NULL);	
    DCS_Unlock();
    return ret;
}

/*
 *post the event of down net interface to ENM
 *
 *@param pNetType the type of network adapter type
 */
DCSAPI DCS_Return_Code DCS_Down_Interface_Request(BOOL pNetType)
{
    	DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();
	
	pReqMgmt->GenFuncReq(GetModuleID(),PM_POST_DOWN_INTERFACE, sizeof(pNetType), &pNetType);
	DCS_Response response;
	
	memset(&response, 0, sizeof(response));
	DCS_DispatchEvent();
	
	DCS_WaitResponse(&response,  GetModuleID(), PM_POST_DOWN_INTERFACE);
	DCS_Unlock();
    	return response.returnCode;
}

/*
 * Register the callback function to be notified  when PM 
 *requests to shut down the network interface.
 *
 * @param pCallback		the pointer to the callback function of wireless lan change
 */
DCSAPI DCS_Return_Code DCS_RegisterDownInterfaceCallback (DCS_PM_Down_Interface_Callback pCallback)
{
	DCS_Return_Code ret = DCS_SUCCESS;
    	DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();
	ret = pReqMgmt->GenEventReq(GetModuleID(), PM_DOWN_INTERFACE_REQ, (DCS_Callback)pCallback);
    	if (ret)   {  DCS_Unlock(); return ret;}
	DCS_DispatchEvent();
    	DCS_Unlock();
	return DCS_SUCCESS;
}

/*
 * Unregister the callback function for down network interface notifications. 
 */
DCSAPI DCS_Return_Code DCS_UnregisterDownInterfaceCallback()
{
    	DCS_Lock();
	ReqMgmt* pReqMgmt = ReqMgmt::Instance();
	DCS_Return_Code ret =  pReqMgmt->GenUnregisterEventReq(GetModuleID(),PM_DOWN_INTERFACE_REQ);	
    	DCS_Unlock();
    	return ret;
}
