#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

import os
import logging

import evas.decorators
import ecore.x as X

from terra.ui.base import EdjeWidget
from terra.core.manager import Manager


mger = Manager()
db = mger.canola_db
OptionsController = mger.get_class("OptionsController")
OptionsControllerMixin = mger.get_class("OptionsControllerMixin")
PlayerController = mger.get_class("Controller/Media")
VideoPlayerOptionsModel = mger.get_class("Model/Options/Folder/Player/Video")
VideoListOptionsModelFolder = mger.get_class("Model/Options/Folder/Video/Local")

log = logging.getLogger("plugins.canola-core.video.ui")

class VideoWindow(object):
    def __init__(self, parent_xid):
        self.parent_xid = parent_xid
        self.video_window = X.Window(self.parent_xid)
        self.input_window = X.Window(self.parent_xid, input=True)
        self.video_window.background_color_set(0, 0, 0)
        self.xid = self.video_window.xid
        self.input_window.show()

    def hide(self):
        self.video_window.hide()
        self.input_window.hide()

    def show(self):
        self.video_window.show()
        self.input_window.show()

    def move_resize(self, x=0, y=0, w=0, h=0):
        self.video_window.move_resize(x, y, w, h)
        self.input_window.move_resize(x, y, w, h)

    def delete(self):
        self.video_window.delete()
        self.input_window.delete()
        self.video_window = None
        self.input_window = None


class VideoPlayerScreen(EdjeWidget):
    def __init__(self, model, parent, xid, theme=None):
        EdjeWidget.__init__(self, parent.evas, "player_video",
                            parent, theme)

        self.parent_view = parent
        self.parent_view.signal_emit("media_area,video", "")
        self.parent_view.part_swallow("player_view/video_throbber",
                                      self.parent_view.throbber)
        self.video = None
        self.video_w = None
        self.video_h = None
        self.callback_stop = None
        self.callback_prev = None
        self.callback_next = None
        self.callback_pause = None
        self.callback_fullscreen = None

        self._setup_x_window(xid)
        self.parent_view.signal_callback_add("resize", "*", self.resize_cb)
        self.parent_view.signal_callback_add("transition,out", "*",
                                             self.transition_out_cb)
        self.setup_model(model)

    def _setup_x_window(self, xid):
        X.init()
        self.fullscreen = False
        self.parent_xid = X.Window_from_xid(xid)
        self._setup_background()
        self.video = VideoWindow(self.parent_xid)
        self.mouse_cb = X.on_mouse_button_up_add(self._on_video_area_clicked)
        self.key_cb = X.on_key_down_add(self._key_down_x_wrapper)
        self.parent_view.video_window = self.video.video_window

    def _setup_background(self):
        self.background = self.parent_view.canvas.Rectangle()
        self.background.color_set(0, 0, 0, 255)
        self.background.geometry_set(*self.parent_xid.geometry_get())
        self.background.hide()

    def setup_model(self, model=None):
        if model:
            self.model = model
            self.parent_view.title_set(self.model.title)

    def _clear_damage_area(self):
        video_area = \
            self.parent_view.part_object_get("player_view/media_area")
        x, y, w, h = video_area.geometry_get()
        self.parent_view.canvas.damage_rectangle_add(x, y, w, h)

    def set_aspect(self, width, height):
        self.video.hide()
        self.video_w = float(width)
        self.video_h = float(height)
        self._resize_keep_aspect()
        self._clear_damage_area()
        X.sync()
        self.video.show()

    def _resize_keep_aspect(self):
        if self.video_w is None or self.video_h is None:
            return

        x, y, w, h = self._get_window_state()
        w2 = int(h * (self.video_w / self.video_h))
        h2 = int(w / (self.video_w / self.video_h))

        if w2 > w:
            y2 = y + (h - h2) / 2
            self.video.move_resize(x, int(y2), w, int(h2))
        else:
            x2 = x + (w - w2) / 2
            self.video.move_resize(int(x2), y, int(w2), h)

    def _get_window_state(self):
        if self.fullscreen:
            self.background.show()
            return self.parent_xid.geometry_get()
        else:
            self.background.hide()
            video_area = \
                self.parent_view.part_object_get("player_view/media_area")
            return video_area.geometry_get()

    def _toggle_fullscreen(self):
        self.video.hide()
        self.fullscreen = not self.fullscreen
        self.video.move_resize(*self._get_window_state())
        self._resize_keep_aspect()
        self.video.show()
        X.sync()

        if self.callback_fullscreen:
            self.callback_fullscreen(self.fullscreen)
        return True

    def _on_video_area_clicked(self, event):
        if self.video and event.window == self.video.input_window:
            self._toggle_fullscreen()
        return True

    def transition_out_cb(self, obj, emission, source):
        if self.callback_stop:
            self.callback_stop()
        self.video.hide()
        self.video.delete()
        self.video = None
        self.parent_xid = None
        X.shutdown()

        self.parent_view.video_window = None
        self._clear_damage_area()
        self.parent_view.signal_callback_del("resize", "*", self.resize_cb)
        self.parent_view.signal_callback_del("transition,out", "*",
                                             self.transition_out_cb)
        self.parent_view.throbber.stop()
        self.parent_view.part_unswallow(self.parent_view.throbber)
        self.parent_view = None
        self.callback_stop = None
        self.callback_prev = None
        self.callback_next = None
        self.callback_pause = None
        self.callback_fullscreen = None

    def _key_down_x_wrapper(self, event):
        if self.video and event.window == self.video.input_window:
            self.handle_key_down(event)
        return True

    def handle_key_down(self, event):
        # Handle keys while focus is on video area
        if event.keyname in ("F6", "f"):
            self._toggle_fullscreen()
        elif event.keyname == "Escape" and self.fullscreen:
            self._toggle_fullscreen()
        elif event.keyname == "F7":
            self.parent_view.raise_volume(5)
        elif event.keyname == "F8":
            self.parent_view.lower_volume(5)
        elif event.keyname == "Left":
            if self.callback_prev:
                self.callback_prev()
        elif event.keyname == "Right":
            if self.callback_next:
                self.callback_next()
        elif event.keyname == "Return":
            if self.callback_pause:
                self.callback_pause()
        else:
            self._parent_widget.handle_key_down(event)
        return True

    def resize_cb(self, obj, emission, source):
        if self.video:
            self.video.move_resize(*self._get_window_state())
            X.sync()

    def delete(self):
        self.parent_xid = None
        self.model = None
        self.key_cb.delete()
        self.mouse_cb.delete()
        EdjeWidget.delete(self)


class VideoPlayerController(PlayerController):
    terra_type = "Controller/Media/Video"
    player_type = "Player/Video"
    hooks_type = "Hook/Player/Video"
    name = "video"
    # safe delta to restore video
    safe_delta = 3

    def __init__(self, model, canvas, parent):
        PlayerController.__init__(self, model, canvas, parent)
        # disable repeat and shuffle for video
        self._repeat = False
        self._shuffle = False
        self.first_load = True
        self.view.callback_theme_changed = self.theme_changed
        self.log = logging.getLogger("canola.plugins.video.player")
        self._setup_video_screen()
        self.setup_view()
        self.view.throbber_start()
        self.view.signal_callback_add("transition,in,finished", "*",
                                      self.transition_in_finished_cb)

    def _setup_video_screen(self):
        ee = self.parent.parent.ecore_evas # XXX
        self.video_screen = VideoPlayerScreen(self.model, self.view, ee.window)
        self.video_screen.callback_stop = self.stop
        self.video_screen.callback_prev = self.prev
        self.video_screen.callback_next = self.next
        self.video_screen.callback_pause = self.handle_play_pause
        self.video_screen.callback_fullscreen = self.fullscreen_cb

    def _commit_pos(self, pos):
        try:
            db.execute("INSERT INTO video_pos (id,pos) "
                       "values (?,?)", (self.model.id, pos))
        except Exception, e:
            try:
                db.execute("UPDATE video_pos SET pos = ? "
                           "WHERE id = ?", (pos, self.model.id))
            except Exception, e:
                self.log.error("Error commiting position: %s", e)
        mger.canola_db.commit()

    def _get_pos(self):
        pos = 0
        try:
            rows = db.execute("SELECT pos FROM video_pos WHERE "
                              "id == ?", (self.model.id,))
            pos = rows[0][0]
        except Exception, e:
            self.log.error("Error getting initial position: %s", e)
        self.log.info("Returning pos: %s", pos)
        return pos

    saved_pos = property(_get_pos, _commit_pos)

    def options_model_get(self):
        return VideoPlayerOptionsModel(None, self)

    def transition_in_finished_cb(self, obj, emission, source):
        self.view.evas.render() # XXX: load_atabake() may take too long
                                # XXX: force redraw to clear last frame
                                # XXX: and avoid idler_add()
        self.load_atabake()
        self.setup_interface()
        self.set_video_window(self.video_screen.video.xid)
        self.setup_model()
        self.view.signal_callback_del("transition,in,finished", "*",
                                      self.transition_in_finished_cb)

    def callback_model_state_changed(self, model):
        PlayerController.callback_model_state_changed(self, model)
        if not model.is_valid and self.video_screen.video:
            self.video_screen.video.hide()

    def setup_model(self, view=True):
        self.setup_view()
        self.current = self.model.parent.current
        self.set_uri(self.model.uri)
        self.view.setup_volume(self._volume)
        self.play()

    def setup_view(self):
        self.view.setup_view()
        self.video_screen.setup_model(self.model)
        self.view.part_swallow("player_view/media_area", self.video_screen)

    def fullscreen_cb(self, fs):
        self.set_fullscreen(fs)
        # hack for mplayer!!
        # force play because mplayer is handling our keys
        # and can generate inconsistent states
        self.play()

    def theme_changed(self):
        self.video_screen.theme_changed()
        self.setup_view(self._volume)
        self.video_screen.setup_model(self.model)

    def back(self):
        self.saved_pos = self.pos
        PlayerController.back(self)

    def home(self):
        self.saved_pos = self.pos
        PlayerController.home(self)

    def _handle_media_info(self, info):
        self.video_screen.set_aspect(info["width"], info["height"])

    def get_media_details(self):
        try:
            self.pl_iface.get_media_details(reply_handler=self._handle_media_info,
                                            error_handler=self._error_handler)
        except Exception, e:
            self.log.debug_error("Get Media Details error: %s", e)

    def _player_buffering(self, value):
        PlayerController._player_buffering(self, value)
        if value == 100.0:
            self.get_media_details()
            if self.first_load:
                # seek to last position.
                self.seek(self.saved_pos - self.safe_delta)
                self.first_load = False

    def _player_eos(self):
        self.saved_pos = 0
        PlayerController._player_eos(self)

    def delete(self):
        self.video_screen.delete()
        self.view.callback_theme_changed = None
        PlayerController.delete(self)

    def options(self):
        self.view.parent.signal_emit("events,block", "")

        self.options_model.position = self.get_position()

        # Define what to do when options close
        if self.state == self.STATE_PLAYING:
            self.pause()
            def leave_cb():
                self.video_screen.video.show()
                self.pause()
        else:
            def leave_cb():
                self.video_screen.video.show()
        self.video_screen.video.hide()

        def cb(controller, *ignored):
            def cb_finished(*ignored):
                self.view.parent.signal_emit("events,unblock", "")
            self.parent.use_options(controller, cb_finished)

        oc = OptionsController(self.options_model, self.evas, self.parent,
                               self, end_callback=cb, leave_callback=leave_cb)


BaseRowRendererWidget = mger.get_class("Widget/BaseRowRenderer")
class BaseVideoRowRendererWidget(BaseRowRendererWidget):
    def __init__(self, parent, theme=None):
        BaseRowRendererWidget.__init__(self, parent, theme)
        self._img = self.evas.FilledImage()
        self.part_swallow("contents", self._img)
        self.signal_emit("thumb,hide", "")

    @evas.decorators.del_callback
    def on_del(self):
        self._img.delete()

    def value_set(self, v):
        if v is not None:
            self.do_value_set(v)
            self._model = v

    def do_value_set(self, v):
        self.part_text_set("title", v.name)
        if v.artist:
            info = "%s, %s" % (v.artist, self._bytes_to_units(v.size))
        else:
            info = self._bytes_to_units(v.size)
        self.part_text_set("info", info)

        if v.thumb is None:
            self.signal_emit("thumb,hide", "")
        else:
            if not os.path.exists(v.thumb):
                v.thumb = None
                self.signal_emit("thumb,hide", "")
            else:
                try:
                    self._img.file_set(v.thumb)
                    self.signal_emit("thumb,show", "")
                except Exception, e:
                    log.error("could not load image %r: %s", v.thumb, e)
                    self.signal_emit("thumb,hide", "")

    @staticmethod
    def _bytes_to_units(bytes):
        if bytes >= 1000000:
            return "%0.1fmb" % (float(bytes) / 1000000)
        elif bytes >= 1000:
            return "%0.1fkb" % (float(bytes) / 1000)
        else:
            return "%db" % bytes


ResizableRowRendererWidget = mger.get_class("Widget/ResizableRowRenderer")
class ResizableVideoRowRendererWidget(BaseVideoRowRendererWidget, ResizableRowRendererWidget):
    row_group = "list_item_video_resizable"


RowRendererWidget = mger.get_class("Widget/RowRenderer")
class VideoRowRendererWidget(BaseVideoRowRendererWidget, RowRendererWidget):
    row_group = "list_item_video"
    resizable_row_renderer = ResizableVideoRowRendererWidget


GenericListController = mger.get_class("Controller/Folder")
class VideoListController(GenericListController, OptionsControllerMixin):
    terra_type = "Controller/Folder/Video/Local"
    row_renderer = VideoRowRendererWidget
    list_group = "list_video"

    def __init__(self, model, canvas, parent):
        GenericListController.__init__(self, model, canvas, parent)
        OptionsControllerMixin.__init__(self)

    def options_model_get(self):
        return VideoListOptionsModelFolder(None, self)
