#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

import logging

from ConfigParser import ConfigParser, NoOptionError

log = logging.getLogger("plugins.canola-core.iradio.extfeedparser")

class attrdict(dict):
    def __getattr__(self, attr):
        try:
            return self[attr]
        except KeyError:
            raise AttributeError(attr)


class Entry(attrdict):
    def __setitem__(self, key, value):
        dict.__setitem__(self, key, value)

        if key == "file":
            self['link'] = value


class Dummy(attrdict):
    def __init__(self, mimetype):
        self['feed'] = attrdict()
        self['headers'] = attrdict()
        self['entries'] = []
        self.headers['content-type'] = mimetype


def parse(filename, mimetype):
    exts = ['pls', 'm3u', 'asx', 'ram', 'rpm']

    ext = filename.split(".")[-1]

    if ext in exts:
        mimetype = "application/" + ext

    if mimetype in ['application/pls', 'audio/x-scpls']:
        return _parse_pls(filename, mimetype)
    elif mimetype in ['application/m3u', 'audio/x-mpegurl']:
        return _parse_m3u(filename, mimetype)
    elif mimetype in ['application/asx', 'video/x-ms-asf']:
        return _parse_asx(filename, mimetype)
    elif mimetype in ['application/ram', 'application/rpm', 'audio/x-pn-realaudio']:
        return _parse_ram(filename, mimetype)
    elif mimetype in ['text/html']:
        return None
    else:
        return _parse_feedparser(filename, mimetype)

def _parse_feedparser(filename, mimetype):
    from feedparser import parse as parsefeed
    contents = parsefeed(filename)

    if not 'headers' in contents:
        contents['headers'] = attrdict()
        contents.headers['content-type'] = mimetype

    return contents

def _parse_pls(filename, mimetype):
    p = ConfigParser()
    p.readfp(open(filename, "r"))

    contents = Dummy(mimetype)

    try:
        num = int(p.get("playlist", "NumberOfEntries")) + 1
    except:
        return contents

    for i in xrange(1, num):
        if i >= len(contents.entries):
            contents.entries.append(Entry())
        data = contents.entries[i - 1]

        try:
            data['file'] = p.get("playlist", "File%d" % i)
        except NoOptionError:
            contents.entries.pop()
            break
        try:
            data['title'] = p.get("playlist", "Title%d" % i)
        except NoOptionError:
            data['title'] = data['file']
        try:
            data['length'] = int(p.get("playlist", "Length%d" % i))
        except NoOptionError:
            data['length'] = -1

    return contents

def _parse_m3u(filename, mimetype):
    contents = Dummy(mimetype)

    fp = open(filename, "r")
    lines = fp.readlines()
    fp.close()

    if lines[0].startswith("#EXTM3U"):
        lines.pop(0)

    i = 0
    for line in lines:
        line = line.strip()

        if i > len(contents.entries) - 1:
            contents.entries.append(Entry())
        data = contents.entries[i]

        if line.startswith("#EXTINF:"):
            data['length'], data['title'] = line.split(":")[1].split(",")
        elif not line:
            continue
        else:
            data['link'] = line
            if 'title' not in data:
                title = line.split("/")[-1].replace("_", " ")
                try: # remove extension if exists
                     title = title[:title.rindex(".")]
                except: pass
                data['title'] = title
            i += 1

    if not data:
        del contents.entries[-1]

    return contents

def _fix_xml(filename):
    import re

    fp = open(filename, "r")
    text = fp.read()
    fp.close()

    fp = open(filename, "w")
    text = text.replace(" & ", " &amp; ")
    text = re.sub("<.+?>", lambda x: x.group(0).lower(), text)
    fp.write(text)
    fp.close()

def _parse_asx(filename, mimetype):
    try:
        from xml.etree import cElementTree as ElementTree
    except:
        log.warning("Could not import C version of ElementTree")
        from xml.etree import ElementTree

    for try_to_fix in [False, True]:
        try:
            try_to_fix and _fix_xml(filename)
            fp = open(filename, "r")
            tree = ElementTree.parse(fp)
            fp.close()
            return _parse_asx_xml(filename, mimetype, tree)
        except:
            fp.close()
    else:
        return _parse_asx_references(filename, mimetype)

def _parse_asx_xml(filename, mimetype, tree):
    contents = Dummy(mimetype)
    root = tree.getroot()

    if not root.tag.upper() == 'ASX':
        return contents

    def find(tag, parent):
        res = parent.find(tag)
        if res is None:
            res = parent.find(tag.upper())
        if res is None:
            res = parent.find(tag.lower())
        return res

    def findall(tag, parent):
        res = parent.findall(tag)
        if len(res) == 0:
            res = parent.findall(tag.lower())
        if len(res) == 0:
            res = parent.findall(tag.upper())
        return res

    contents.feed['title'] = find("Title", root).text

    try:
        contents.feed['author'] = find("Author", root).text
    except:
        contents.feed['author'] = None

    try:
        contents.feed['link'] = find("MoreInfo", root).attrib['href']
    except:
        contents.feed['link'] = filename

    i = 0

    entries = [root]
    for elem in findall("Repeat", root) + [root]:
        entries += findall("Entry", elem)

    for entry in entries:
        for ref in entry.findall("ref"):
            if i > len(contents.entries) - 1:
                contents.entries.append(Entry())
            data = contents.entries[i]

            data["link"] = ref.attrib['href']
            try:
                data['title'] = find("Title", entry).text
            except:
                data["title"] = find("Abstract", entry).text
            i += 1

    return contents

def _parse_asx_references(filename, mimetype):
    contents = Dummy(mimetype)

    contents.feed['link'] = filename
    contents.feed['title'] = "New ASX feed"

    fp = open(filename, "r")

    for line in fp:
        if line.strip() == "[Reference]":
            break
    else:
        return contents

    i = 1
    for line in fp:
        lsplit = line.strip().split("=", 1)
        if len(lsplit) == 2 and lsplit[0].startswith("Ref"):
            new_entry = Entry()
            new_entry['link'] = lsplit[1]
            new_entry['title'] = "Stream #%d" % i
            contents.entries.append(new_entry)
            i += 1

    fp.close()

    return contents

def _parse_ram(filename, mimetype):
    contents = Dummy(mimetype)

    fp = open(filename, "r")

    i = 1
    for line in fp:
        line = line.strip()
        if not line:
            continue
        new_entry = Entry()
        new_entry['link'] = line
        new_entry['title'] = "Stream #%d" % i
        contents.entries.append(new_entry)
        i += 1

    fp.close()

    return contents
