#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

import edje
import logging

from terra.ui.base import EdjeWidget
from terra.core.singleton import Singleton
from terra.core.controller import Controller
from terra.core.terra_object import TerraObject

__all__ = ("RemotePlayer", "RemotePlayerWindow", "RemotePlayerController",)

log = logging.getLogger("canola.plugins.audio.remote")


class RemotePlayer(Singleton, TerraObject):
    terra_type = "Controller/RemotePlayer"

    def __init__(self):
        Singleton.__init__(self)
        self._volume = 0
        self._state = None
        self._model = None
        self.callback_prev = None
        self.callback_next = None
        self.callback_get_volume = None
        self.callback_set_volume = None
        self.callback_model_changed = None
        self.callback_state_changed = None
        self.callback_handle_play_pause = None

    def _set_state(self, value):
        self._state = value
        self.callback_state_changed and self.callback_state_changed(value)

    state = property(lambda self: self._state, _set_state)

    def _set_model(self, value):
        self._model = value
        self.callback_model_changed and self.callback_model_changed(value)

    model = property(lambda self: self._model, _set_model)

    def _get_volume(self):
        if self.callback_get_volume:
            self._volume = self.callback_get_volume()
        return self._volume

    def _set_volume(self, value):
        self.callback_set_volume and self.callback_set_volume(value)
        self._volume = value

    volume = property(_get_volume, _set_volume)

    def raise_volume(self):
        self.volume += 5

    def lower_volume(self):
        self.volume -= 5

    def next(self):
        self.callback_next and self.callback_next()

    def prev(self):
        self.callback_prev and self.callback_prev()

    def handle_play_pause(self):
        self.callback_handle_play_pause and self.callback_handle_play_pause()

    def delete(self):
        self._volume = 0
        self._state = None
        self._volume = None
        self._model = None
        self.callback_prev = None
        self.callback_next = None
        self.callback_get_volume = None
        self.callback_set_volume = None
        self.callback_model_changed = None
        self.callback_state_changed = None
        self.callback_handle_play_pause = None


class RemotePlayerWindow(EdjeWidget):
    def __init__(self, canvas, parent, theme=None):
        EdjeWidget.__init__(self, canvas, "notify/remoteplayer", parent, theme)
        self.cb_prev = None
        self.cb_play_pause = None
        self.cb_next = None

    def set_playing(self, playing):
        if playing:
            signal = "media,state,play"
        else:
            signal = "media,state,pause"
        self.signal_emit(signal, "")

    @edje.decorators.signal_callback("action,clicked", "previous")
    def prev(obj, emission, source):
        obj.cb_prev()

    @edje.decorators.signal_callback("action,clicked", "play_pause")
    def play_pause(obj, emission, source):
        obj.cb_play_pause()

    @edje.decorators.signal_callback("action,clicked", "next")
    def next(obj, emission, source):
        obj.cb_next()

    def delete(self):
        EdjeWidget.delete(self)
        self.cb_prev = None
        self.cb_play_pause = None
        self.cb_next = None


class RemotePlayerController(Controller):
    terra_type = "Controller/Notify/RemotePlayer"

    # Player state type
    (STATE_NONE, STATE_PLAYING,
     STATE_PAUSED, STATE_ERROR, STATE_HOLD) = range(5)

    def __init__(self, model, canvas, parent):
        Controller.__init__(self, model, canvas, parent)
        self._remote_player = RemotePlayer()
        self._remote_player.callback_state_changed = self.state_changed
        self._remote_player.callback_model_changed = self.model_changed
        self._setup_view()

    def _setup_view(self):
        self.view = RemotePlayerWindow(self.evas, self.parent.view)
        self.view.on_key_down_add(self._on_key_down)
        self._old_focus = self.view.evas.focus
        self.view.focus = True
        self.view.cb_prev = self._remote_player.prev
        self.view.cb_next = self._remote_player.next
        self.view.cb_play_pause = self._remote_player.handle_play_pause
        state = self._remote_player.state
        self.view.set_playing(state == self.STATE_PLAYING)

    def _hide_notify(self, *ignored):
        log.debug("Hiding remote player window")
        self.parent.hide_notify(self, True)

    def _on_key_down(self, obj, event):
        if event.keyname == "Escape":
            self._hide_notify()
        elif event.keyname == "F7":
            self._remote_player.raise_volume()
            log.debug("Raising volume")
        elif event.keyname == "F8":
            self._remote_player.lower_volume()
            log.debug("Lowering volume")
        elif event.keyname == "Return":
            self._remote_player.handle_play_pause()
            log.debug("Starting/Pausing player")
        elif event.keyname == "Left":
            self._remote_player.prev()
            log.debug("Previous song")
        elif event.keyname == "Right":
            self._remote_player.next()
            log.debug("Next song")

    def state_changed(self, state):
        self.view.set_playing(state == self.STATE_PLAYING)
        log.debug("Changed state: %s", state)

    def model_changed(self, model):
        ## future needs
        pass

    def delete(self):
        if self.view.evas.focus is self.view:
            self._old_focus.focus = True
        self.view.delete()
