# -*- coding: utf-8 -*-
# Canola2 Tuning Plugin
# Copyright (C) 2008 Instituto Nokia de Tecnologia
# Authors: Adriano Rezende <adriano.rezende@openbossa.org>
#          Renato Chencarek <renato.chencarek@gmail.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Additional permission under GNU GPL version 3 section 7
#
# If you modify this Program, or any covered work, by linking or combining it
# with Canola2 and its core components (or a modified version of any of those),
# containing parts covered by the terms of Instituto Nokia de Tecnologia End
# User Software Agreement, the licensors of this Program grant you additional
# permission to convey the resulting work.

import os
import logging
from terra.core.plugin_prefs import PluginPrefs
from terra.core.threaded_func import ThreadedFunction


from terra.core.manager import Manager
from tuning_manager import TuningManager

log = logging.getLogger("plugins.canola-tuning.backend")


manager = Manager()


class CanolaTuningBackend(object):
    skip_processed_videos = False

    def __init__(self):
        self.prefs = PluginPrefs("settings")
        self.cancelled = False
        self.cover_path = self.prefs["cover_path"]
        self.artist_album_list = []

    def scan_audio(self, sources, artist, album, end_callback):
        self.scan_audios(
            sources, end_callback, artist=artist, album=album, skip=False)

    def scan_audios(self, sources, end_callback, set_progress=None,
                    artist=None, album=None, skip=True):
        self.cancelled = False

        def cb_progress(fname, progress, total, status):
            if set_progress:
                set_progress(fname, progress, total, status)
            log.debug("[%d] scan album: %s (%s)" % (progress, fname, status))

        manager = TuningManager(self._get_audio_data(),
                                lambda: not self.cancelled, cb_progress)

        ThreadedFunction(end_callback, manager.process_albums,
                         self.cover_path, sources, skip, artist, album).start()

    def clean(self, artist=None, album=None, end_callback=None):
        self.cancelled = False

        if artist or album:
            self._clean_unique(artist, album, end_callback)
            return

        def cb_progress(fname, progress, total, status):
            log.debug("[%d] album cleanup: %s (%s)" % (progress, fname, status))

        manager = TuningManager(self._get_audio_data(),
                                lambda: not self.cancelled, cb_progress)

        ThreadedFunction(end_callback, manager.cleanup_covers, self.cover_path).start()

    def _clean_unique(self, artist=None, album=None, end_callback=None):
        path = os.path.join(self.cover_path, artist, album)
        ThreadedFunction(end_callback, TuningManager.remove_unused_covers, path).start()

    def scan_video(self, filename, time, end_callback=None):
        self.cancelled = False

        def cb_progress(fname, progress, total, status):
            log.debug("[%d] video scan: %s (%s)" % (progress, fname, status))

        manager = TuningManager(None, lambda: not self.cancelled, cb_progress)

        ThreadedFunction(end_callback, manager.process_videos, [filename],
                         self.prefs["video_thumb_path"],
                         time, 150, 90, False).start()

    def scan_videos(self, time, end_callback=None, set_progress=None):
        self.cancelled = False
        files = self._get_video_data()

        def cb_progress(fname, progress, total, status):
            if set_progress:
                set_progress(fname, progress, total, status)
            log.debug("[%d] video scan: %s (%s)" % (progress, fname, status))

        cover_manager = TuningManager(None, lambda: not self.cancelled, cb_progress)

        ThreadedFunction(end_callback, cover_manager.process_videos, files,
                         self.prefs["video_thumb_path"],
                         time, 150, 90, self.skip_processed_videos).start()

    def cancel(self):
        self.cancelled = True

    def _get_video_data(self):
        try:
            cursor = manager.canola_db.get_cursor()
            cursor.execute("SELECT DISTINCT files.path "
                           "FROM videos, files WHERE files.id = videos.id")
            return [str(a[0]) for a in cursor.fetchall()]
        except Exception, e:
            log.error("Error while trying to get video data: %s" % e.message)
            return []

    def _get_audio_data(self):
        data = {}
        try:
            cursor = manager.canola_db.get_cursor()
            cursor.execute("SELECT DISTINCT audio_albums.name,audio_artists.name,files.path \
                     FROM audios,files,audio_albums,audio_artists WHERE audios.id = files.id AND files.dtime = 0 \
                     AND audios.album_id = audio_albums.id AND audio_albums.artist_id = audio_artists.id")
            for p in cursor.fetchall():
                data.setdefault((p[1], p[0]), []).append(str(p[2]))
        except Exception, e:
            log.error("Error while trying to get album data: %s" % e.message)

        return data
