/*
 * Bickley - a meta data management framework.
 * Copyright © 2008, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */
#include <config.h>
#include <errno.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/syscall.h>

#include <glib.h>
#include <gio/gio.h>

#if ENABLE_GST
#include <gst/gst.h>
#endif

#include <dbus/dbus-protocol.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-bindings.h>

#include "audio-metadata.h"
#include "exif-metadata.h"
#include "pixbuf-metadata.h"
#include "pixbuf-thumbnail.h"
#include "video-gst-thumbnailer.h"
#include "video-metadata.h"

#include "bkl-investigator.h"
#include "bkl-investigator-dbus.h"
#include "bkl-marshal.h"

#include "metadata-defines.h"

typedef struct _InvestigateUri {
    char *source;
    char *uri;
} InvestigateUri;

struct _BklInvestigator {
    DBusGConnection *connection;
    BklInvestigatorDBus *dbus;

    InvestigateUri *current_iu;

    GQueue *pending;
    guint32 worker_id;
    guint32 death_id;

    GCancellable *cancellable;
    GMainLoop *mainloop;
};

static BklInvestigator *investigator = NULL;

#undef HAVE_IO_PRIO
#if defined (__i386__)
#  define HAVE_IO_PRIO
#  define __NR_ioprio_set 289
#elif defined (__x86_64__)
#  define HAVE_IO_PRIO
#  define __NR_ioprio_set 251
#else
#  warning "Architecture does not support ioprio modification"
#endif
#define IOPRIO_WHO_PROCESS 1
#define IOPRIO_CLASS_IDLE 3
#define IOPRIO_CLASS_SHIFT 13
#define IOPRIO_IDLE_LOWEST (7 | (IOPRIO_CLASS_IDLE << IOPRIO_CLASS_SHIFT))

#define ORBITER_NAME "org.moblin.Bickley.Orbiter"
#define ORBITER_PATH "/org/moblin/Bickley/Orbiter"
#define ORBITER_IFACE "org.moblin.Bickley.Orbiter"

#define INVESTIGATOR_NAME "org.moblin.Bickley.Investigator"
#define INVESTIGATOR_PATH "/org/moblin/Bickley/Investigator"
#define INVESTIGATOR_IFACE "org.moblin.Bickley.Investigator"

#define START_DEATH_TIMEOUT 60 /* seconds: time allowed from start to first
                                  file to be processed */
#define DEATH_TIMEOUT 10 /* seconds from the last file is processed to death */

/* FIXME: Whats the difference between time::modified and time::changed? */
#define REQUIRED_FIELDS "standard::size,standard::content-type,time::modified"

static gboolean no_death = FALSE;
static gboolean show_process = FALSE;
static GOptionEntry bkl_entries[] = {
    { "no-death", 0, 0, G_OPTION_ARG_NONE, &no_death,
      "The investigator should never die", NULL },
    { "show-processing", 0, 0, G_OPTION_ARG_NONE, &show_process,
      "The investigator should print uris and timing info", NULL},
    { NULL }
};

static GTimer *process_timer = NULL;

static gboolean
get_audio_metadata (GFile        *file,
                    GFileInfo    *info,
                    const char   *mimetype,
                    GHashTable   *metadata,
                    GCancellable *cancellable)
{
    if (bkl_task_audio_get_metadata (file, info,
                                     mimetype, metadata,
                                     cancellable) == FALSE) {
        return FALSE;
    }

    return TRUE;
}

static gboolean
get_image_metadata (GFile      *file,
                    GFileInfo  *info,
                    const char *mimetype,
                    GHashTable *metadata)
{
    gboolean care = FALSE;

    care |= bkl_task_pixbuf_get_metadata (file, info, mimetype, metadata);
    if (care) {
        bkl_task_pixbuf_thumbnail (file, info, mimetype, metadata);
    }
    care |= bkl_task_exif_get_metadata (file, info, mimetype, metadata);

    return care;
}

static gboolean
get_video_metadata (GFile        *file,
                    GFileInfo    *info,
                    const char   *mimetype,
                    GHashTable   *metadata,
                    GCancellable *cancellable)
{
    gboolean care = FALSE;

    care = bkl_task_video_get_metadata (file, info, mimetype, metadata);
    if (care) {
        bkl_task_video_thumbnail (file, info, mimetype, metadata, cancellable);
    }

    return care;
}

static gboolean
get_rm_metadata (GFile        *file,
                 GFileInfo    *info,
                 const char   *mimetype,
                 GHashTable   *metadata,
                 GCancellable *cancellable)
{
    gboolean care = FALSE;
    const char * data = NULL;

    if (bkl_task_audio_get_metadata (file, info,
                                     mimetype, metadata,
                                     cancellable) == FALSE) {
        return FALSE;
    }

    data = g_hash_table_lookup (metadata, METADATA_FILE_MIMETYPE);

    if (data) {
        if (g_str_has_prefix (data, "video")) {
            care = get_video_metadata (file, info,
                                       mimetype, metadata,
                                       cancellable);
            g_hash_table_insert (metadata, METADATA_TYPE,
                                 g_strdup (VIDEO_TYPE));
        } else if (g_str_has_prefix (data, "audio")) {
            g_hash_table_insert (metadata, METADATA_TYPE,
                                 g_strdup (AUDIO_TYPE));
        }
    }

    return care;
}

#if __WORDSIZE == 64
#define G_GINT64_MODIFIER "l"
# else
#define G_GINT64_MODIFIER "ll"
#endif

static GHashTable *
process_uri (const char   *uri,
             GCancellable *cancellable)
{
    GFile *file;
    GFileInfo *info;
    GHashTable *metadata;
    GError *error = NULL;
    const char *mimetype;
    GTimeVal result;
    gboolean care = FALSE;

    g_cancellable_reset (cancellable);

    metadata = g_hash_table_new_full (g_str_hash, g_str_equal, NULL, g_free);
    file = g_file_new_for_uri (uri);
    info = g_file_query_info (file, REQUIRED_FIELDS, G_FILE_QUERY_INFO_NONE,
                              NULL, &error);
    if (info == NULL) {
        g_warning ("Error getting %s info: %s", uri, error->message);
        g_error_free (error);

        return metadata;
    }

    mimetype = g_file_info_get_content_type (info);

    g_print ("%s - %s\n", mimetype, uri);
    /* Go through the various tasks */
    /* g_file_info_get_content_type can't determine rm content is audio only
       or video content, we have to handle it specially */
    if (g_strrstr (mimetype, "vnd.rn-realmedia")) {
        if (get_rm_metadata (file, info, mimetype, metadata, cancellable)) {
            care = TRUE;
        }
    } else if (get_audio_metadata (file, info, mimetype,
                                   metadata, cancellable)) {
        g_hash_table_insert (metadata, METADATA_TYPE, g_strdup (AUDIO_TYPE));
        care = TRUE;
    } else if (get_image_metadata (file, info, mimetype, metadata)) {
        g_hash_table_insert (metadata, METADATA_TYPE, g_strdup (IMAGE_TYPE));
        care = TRUE;
    } else if (get_video_metadata (file, info, mimetype,
                                   metadata, cancellable)) {
        g_hash_table_insert (metadata, METADATA_TYPE, g_strdup (VIDEO_TYPE));
        care = TRUE;
    }

    if (care == FALSE) {
        if (g_cancellable_is_cancelled (cancellable)) {
            g_object_unref (info);
            g_object_unref (file);

            return NULL;
        }

        g_hash_table_insert (metadata, METADATA_TYPE, g_strdup (BROKEN_TYPE));
    }

    g_hash_table_insert (metadata, METADATA_FILE_URI, g_strdup (uri));

    g_hash_table_insert (metadata, METADATA_FILE_SIZE,
                         g_strdup_printf ("%" G_GINT64_MODIFIER "d",
                                          g_file_info_get_size (info)));

    g_file_info_get_modification_time (info, &result);
    g_hash_table_insert (metadata, METADATA_FILE_MTIME,
                         g_strdup_printf ("%ld", result.tv_sec));

    if (g_hash_table_lookup (metadata, METADATA_FILE_MIMETYPE) == NULL) {
        g_hash_table_insert (metadata, METADATA_FILE_MIMETYPE,
                             g_strdup (mimetype));
    }

    g_hash_table_insert (metadata, METADATA_EXTENDED_DATE_INDEXED,
                         g_strdup_printf ("%ld", time (NULL)));

    g_object_unref (info);
    g_object_unref (file);

    return metadata;
}

static gboolean
kill_investigator (gpointer userdata)
{
    g_print ("Killing...\n");
    g_main_loop_quit (investigator->mainloop);
    return FALSE;
}

static void G_GNUC_UNUSED
dump_metadata (gpointer key,
               gpointer value,
               gpointer userdata)
{
    g_print ("%s: %s\n", (char *) key, (char *) value);
}

static gboolean
worker (gpointer userdata)
{
    InvestigateUri *iu;
    GHashTable *metadata;

    iu = g_queue_pop_head (investigator->pending);
    if (G_UNLIKELY (iu == NULL)) {
        /* This shouldn't happen, but just in case */
        investigator->worker_id = 0;
        return FALSE;
    }

    investigator->current_iu = iu;

    /* Let the orbiter know we're doing something */
    bkl_investigator_dbus_processing (investigator->dbus, iu->source, iu->uri);

    if (show_process) {
        g_print ("Processing %s\n", iu->uri);
        g_timer_start (process_timer);
        metadata = process_uri (iu->uri, investigator->cancellable);
        g_timer_stop (process_timer);
        g_print ("Processed in %.2fs\n",
                 g_timer_elapsed (process_timer, NULL));
    } else {
        metadata = process_uri (iu->uri, investigator->cancellable);
    }

    if (metadata) {
        /* g_hash_table_foreach (metadata, dump_metadata, NULL); */

        bkl_investigator_dbus_completed (investigator->dbus, iu->source,
                                         iu->uri, metadata);
    }

    investigator->current_iu = NULL;

    g_hash_table_destroy (metadata);
    g_free (iu->uri);
    g_free (iu->source);
    g_free (iu);

    if (g_queue_peek_head (investigator->pending)) {
        return TRUE;
    } else {
        if (no_death == FALSE) {
            /* When we've run out of things to investigate, we stop the worker
               and start a timeout til shutdown. If new uris are added before
               DEATH_TIMEOUT expires, it is stopped and the worker is started
               again */
            g_print ("Death approaches\n");
            investigator->death_id = g_timeout_add_seconds (DEATH_TIMEOUT,
                                                            kill_investigator,
                                                            NULL);
        }
        investigator->worker_id = 0;
        return FALSE;
    }
}

static void
source_removed_cb (DBusGProxy *proxy,
                   const char *source_name,
                   const char *uri,
                   gpointer    userdata)
{
    GList *head, *h;

    /* When the source is removed, we need to remove all the uris owned
       by the source */
    head = g_queue_peek_head_link (investigator->pending);
    for (h = head; h;) {
        InvestigateUri *iu = (InvestigateUri *) h->data;
        GList *old;

        if (g_str_equal (iu->source, source_name)) {
            g_queue_unlink (investigator->pending, h);
            old = h;
            h = h->next;

            g_print ("(%s) Cancelling %s\n", iu->source, iu->uri);
            g_free (iu->source);
            g_free (iu->uri);
            g_free (iu);

            g_list_free (old);
        } else {
            h = h->next;
        }
    }
}

static gboolean
dbus_init (void)
{
    DBusGProxy *proxy, *orbiter;
    guint32 request_name_ret;
    GError *error = NULL;

    dbus_g_object_register_marshaller (bkl_marshal_VOID__STRING_STRING,
                                       G_TYPE_NONE, G_TYPE_STRING,
                                       G_TYPE_STRING, G_TYPE_INVALID);

    investigator->connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error);
    if (!investigator->connection) {
        return FALSE;
    }

    proxy = dbus_g_proxy_new_for_name (investigator->connection,
                                       DBUS_SERVICE_DBUS,
                                       DBUS_PATH_DBUS,
                                       DBUS_INTERFACE_DBUS);
    if (!org_freedesktop_DBus_request_name (proxy, INVESTIGATOR_NAME,
                                            0, &request_name_ret, &error)) {
        return FALSE;
    }

    if (request_name_ret != DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER) {
        return FALSE;
    }

    investigator->dbus = bkl_investigator_dbus_new ();
    dbus_g_connection_register_g_object (investigator->connection,
                                         INVESTIGATOR_PATH,
                                         (GObject *) investigator->dbus);

    /* Hook up to the orbiter to listen for sources being removed */
    orbiter = dbus_g_proxy_new_for_name (investigator->connection,
                                         ORBITER_NAME,
                                         ORBITER_PATH,
                                         ORBITER_IFACE);
    dbus_g_proxy_add_signal (orbiter, "SourceRemoved",
                             G_TYPE_STRING, G_TYPE_STRING, G_TYPE_INVALID);
    dbus_g_proxy_connect_signal (orbiter, "SourceRemoved",
                                 G_CALLBACK (source_removed_cb), NULL, NULL);

    return TRUE;
}

static void
ensure_thumbnail_dir (void)
{
    char *thumbdir;

    thumbdir = g_build_filename (g_get_home_dir (), ".bkl-thumbnails", NULL);
    g_mkdir_with_parents (thumbdir, 0777);
    g_free (thumbdir);
}

int
main (int    argc,
      char **argv)
{
    GError *error = NULL;
    GOptionContext *context;
    int ret;

    g_type_init();
    g_thread_init (NULL);

    context = g_option_context_new ("- Bickley Investigator");
    g_option_context_add_main_entries (context, bkl_entries, "");
#if ENABLE_GST
    g_option_context_add_group (context, gst_init_get_option_group ());
#endif
    g_option_context_set_ignore_unknown_options (context, TRUE);

    g_option_context_parse (context, &argc, &argv, &error);
    if (error != NULL) {
        g_warning ("Error parsing options: %s", error->message);
        g_error_free (error);
        error = NULL;
    }

    g_option_context_free (context);

    if (show_process) {
        g_print ("Showing processing info\n");
        process_timer = g_timer_new ();
    }

#if ENABLE_GST
#else
    if(!g_getenv("HELIX_THUMB_LIBS")) {
        g_setenv("HELIX_THUMB_LIBS" , "/opt/real/RealPlayer", FALSE);
    }
#endif

    ensure_thumbnail_dir ();

    /* We're a background service, make sure we run on a nice level */
    ret = nice (19);
#ifdef HAVE_IO_PRIO
    if (syscall(__NR_ioprio_set,
                IOPRIO_WHO_PROCESS, 0, IOPRIO_IDLE_LOWEST) == -1) {
        g_warning ("Cannot set IO priority to idle class: %s (%d)",
                   g_strerror (errno), errno);
    }
#endif

    investigator = g_new0 (BklInvestigator, 1);
    if (dbus_init () == FALSE) {
        /* Investigator is already running...quit */
        return 0;
    }

    investigator->pending = g_queue_new ();

    if (no_death == FALSE) {
        /* Start a longer timeout for the initial start. Allow orbiter 1minute
           to tell us to do something before killing ourselves */
        investigator->death_id = g_timeout_add_seconds (START_DEATH_TIMEOUT,
                                                        kill_investigator,
                                                        NULL);
    }

    investigator->cancellable = g_cancellable_new ();
    investigator->mainloop = g_main_loop_new (NULL, TRUE);
    g_main_loop_run (investigator->mainloop);

    return 0;
}

void
bkl_investigator_add_uris (const char  *source,
                           const char **uris,
                           int          n_uris)
{
    int i;

    if (n_uris == 0) {
        return;
    }

    /* Stop the killer */
    if (investigator->death_id) {
        g_source_remove (investigator->death_id);
        investigator->death_id = 0;
    }

    for (i = 0; uris[i]; i++) {
        InvestigateUri *iu = g_new (InvestigateUri, 1);
        iu->source = g_strdup (source);
        iu->uri = g_strdup (uris[i]);

        g_queue_push_tail (investigator->pending, iu);
    }

    /* Start the worker if its not running */
    if (investigator->worker_id == 0) {
        investigator->worker_id = g_idle_add_full (G_PRIORITY_LOW, worker,
                                                   NULL, NULL);
    }
}

void
bkl_investigator_shutdown (void)
{
    g_main_loop_quit (investigator->mainloop);
}

void
_bkl_investigator_cancel_source (const char *source)
{
    GList *l;

    if (investigator->current_iu &&
        g_str_equal (investigator->current_iu->source, source)) {

        g_print ("Cancelling %s\n", investigator->current_iu->uri);
        g_cancellable_cancel (investigator->cancellable);
    }

    for (l = investigator->pending->head; l;) {
        InvestigateUri *iu = (InvestigateUri *) l->data;

        if (g_str_equal (source, iu->source)) {
            GList *old;

            old = l;
            l = l->next;
            g_queue_unlink (investigator->pending, old);

            g_free (iu->source);
            g_free (iu->uri);
            g_free (iu);

            g_list_free (old);
        } else {
            l = l->next;
        }
    }
}

void
_bkl_investigator_cancel_uris (const char **uris)
{
    GList *l;
    int i;

    /* FIXME: Would this be quicker by flipping the two loops? */
    for (i = 0; uris[i]; i++) {
        if (investigator->current_iu &&
            g_str_equal (uris[i], investigator->current_iu->uri)) {
            g_print ("Cancelling %s\n", investigator->current_iu->uri);
            g_cancellable_cancel (investigator->cancellable);

            continue;
        }

        for (l = investigator->pending->head; l;) {
            InvestigateUri *iu = (InvestigateUri *) l->data;

            if (g_str_equal (uris[i], iu->uri)) {
                GList *old;

                g_queue_unlink (investigator->pending, l);

                g_free (iu->source);
                g_free (iu->uri);
                g_free (iu);

                old = l;
                g_list_free (old);

                break;
            } else {
                l = l->next;
            }
        }
    }
}
