/*
 * Bickley - a meta data management framework.
 * Copyright © 2008, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <gtk/gtk.h>

#include <bickley/bkl-db.h>
#include <bickley/bkl-utils.h>
#include <bickley/bkl-item-audio.h>
#include <bickley/bkl-item-image.h>
#include <bickley/bkl-item-video.h>
#include <bickley/bkl-source-client.h>
#include <bickley/bkl-source-manager-client.h>

enum {
    SOURCE_NAME,
    SOURCE_POINTER,
    SOURCE_COLS
};

enum {
    ITEM_POINTER,
    ITEM_PIXBUF,
    ITEM_COLS
};

typedef struct _Item {
    BklItem *item;
    GtkTreeRowReference *row_ref;
} Item;

typedef struct _Source {
    BklSourceClient *client;
    BklDB *db;
    char *source_name;

    GPtrArray *items;
    GHashTable *uri_to_item;
    GtkListStore *store;
    GtkTreeRowReference *row_ref;
} Source;

static GHashTable *source_by_path = NULL;
static GTimer *timer = NULL;
static GtkListStore *source_store;
static GtkWidget *source_list, *item_list;

static void
add_item (Source       *source,
          BklItem      *item)
{
    Item *i;
    GtkTreeIter iter;
    GtkTreePath *path;
    GdkPixbuf *thumb = NULL;
    const char *uri;
    GError *error = NULL;

    i = g_slice_new0 (Item);

    uri = bkl_item_extended_get_thumbnail ((BklItemExtended *) item);
    if (uri) {
        char *path;
        path = g_filename_from_uri (uri, NULL, NULL);

        thumb = gdk_pixbuf_new_from_file_at_size (path, 64, 64, &error);
        if (thumb == NULL) {
            g_warning ("Error loading %s: %s", path, error->message);
            g_error_free (error);
        }

        g_free (path);

    }

    gtk_list_store_append (source->store, &iter);
    gtk_list_store_set (source->store, &iter,
                        ITEM_POINTER, item,
                        ITEM_PIXBUF, thumb,
                        -1);
    i->item = item;

    path = gtk_tree_model_get_path ((GtkTreeModel *) source->store, &iter);
    i->row_ref = gtk_tree_row_reference_new ((GtkTreeModel *) source->store,
                                             path);
    gtk_tree_path_free (path);

    g_hash_table_insert (source->uri_to_item,
                         (char *) bkl_item_get_uri (item), i);
}

static GtkListStore *
create_item_store (Source *source)
{
    int i;

    source->store = gtk_list_store_new (ITEM_COLS, G_TYPE_POINTER, GDK_TYPE_PIXBUF);
    for (i = 0; i < source->items->len; i++) {
        BklItem *item = source->items->pdata[i];

        add_item (source, item);
    }

    return source->store;
}

static void
add_source (Source *source)
{
    GtkTreeIter iter;
    GtkTreePath *path;

    gtk_list_store_append (source_store, &iter);
    gtk_list_store_set (source_store, &iter,
                        SOURCE_NAME, kozo_db_get_name (source->db->db),
                        SOURCE_POINTER, source,
                        -1);

    path = gtk_tree_model_get_path ((GtkTreeModel *) source_store, &iter);
    source->row_ref = gtk_tree_row_reference_new ((GtkTreeModel *) source_store,
                                                  path);
    gtk_tree_path_free (path);
}

static void
remove_source (Source *source)
{
    GtkTreeIter iter;
    GtkTreePath *path;

    path = gtk_tree_row_reference_get_path (source->row_ref);
    gtk_tree_model_get_iter ((GtkTreeModel *) source_store, &iter, path);
    gtk_list_store_remove (source_store, &iter);

    gtk_tree_path_free (path);
    gtk_tree_row_reference_free (source->row_ref);
}

static void
uri_added (BklSourceClient *client,
           const char      *uri,
           Source          *source)
{
    BklItem *item;
    GError *error = NULL;

    g_print ("(%s): Item added - %s\n", source->source_name, uri);

    item = bkl_db_get_item (source->db, uri, &error);
    if (error != NULL) {
        g_warning ("Error getting %s: %s", uri, error->message);
        g_error_free (error);
        return;
    }

    add_item (source, item);
}

static void
uri_deleted (BklSourceClient *client,
             const char      *uri,
             Source          *source)
{
    GtkTreePath *path;
    GtkTreeIter iter;
    Item *item;

    g_print ("(%s): Item removed - %s\n", source->source_name, uri);
    item = g_hash_table_lookup (source->uri_to_item, uri);
    g_assert (item);

    path = gtk_tree_row_reference_get_path (item->row_ref);
    if (gtk_tree_model_get_iter ((GtkTreeModel *) source->store, &iter, path)) {
        gtk_list_store_remove (source->store, &iter);
    }
    gtk_tree_path_free (path);

    gtk_tree_row_reference_free (item->row_ref);
    g_object_unref (item->item);
    g_hash_table_remove (source->uri_to_item, uri);
    g_slice_free (Item, item);
}

static void
uri_changed (BklSourceClient *client,
             const char      *uri,
             Source          *source)
{
    GError *error = NULL;
    GtkTreePath *path;
    GtkTreeIter iter;
    Item *item;

    g_print ("(%s): Item changed - %s\n", source->source_name, uri);
    item = g_hash_table_lookup (source->uri_to_item, uri);
    g_assert (item);

    g_object_unref (item->item);
    item->item = bkl_db_get_item (source->db, uri, &error);
    if (item->item == NULL) {
        g_warning ("(%s): Error getting %s from DB: %s",
                   source->source_name, uri,
                   error->message);
        return;
    }

    path = gtk_tree_row_reference_get_path (item->row_ref);
    if (gtk_tree_model_get_iter ((GtkTreeModel *) source->store, &iter, path)) {
        const char *thumb_uri;
        GdkPixbuf *thumb = NULL;

        thumb_uri = bkl_item_extended_get_thumbnail
            ((BklItemExtended *) item->item);
        if (thumb_uri) {
            char *path;

            path = g_filename_from_uri (uri, NULL, NULL);

            thumb = gdk_pixbuf_new_from_file_at_size (path, 64, 64, &error);
            if (thumb == NULL) {
                g_warning ("Error loading %s: %s", path, error->message);
                g_error_free (error);
            }

            g_free (path);
        }

        gtk_list_store_set (source->store, &iter,
                            ITEM_POINTER, item->item,
                            ITEM_PIXBUF, thumb,
                            -1);
        gtk_tree_model_row_changed ((GtkTreeModel *) source->store,
                                    path, &iter);
    }
    gtk_tree_path_free (path);
}

static void
index_changed (BklSourceClient *client,
               const char     **added,
               const char     **removed,
               Source          *source)
{
    int i;

    g_print ("Added:\n");
    for (i = 0; added[i]; i++) {
        g_print ("   [%d] %s\n", i, added[i]);
    }

    g_print ("\nRemoved:\n");
    for (i = 0; removed[i]; i++) {
        g_print ("   [%d] %s\n", i, removed[i]);
    }
}

static Source *
create_source (BklSourceClient *client)
{
    Source *source;
    GError *error = NULL;

    source = g_new (Source, 1);
    source->client = client;

    g_signal_connect (client, "uri-added",
                      G_CALLBACK (uri_added), source);
    g_signal_connect (client, "uri-changed",
                      G_CALLBACK (uri_changed), source);
    g_signal_connect (client, "uri-deleted",
                      G_CALLBACK (uri_deleted), source);
    g_signal_connect (client, "index-changed",
                      G_CALLBACK (index_changed), source);

    source->db = bkl_source_client_get_db (client);
    if (source->db == NULL) {
        g_error ("Error loading database");
    }

    source->source_name = g_strdup (kozo_db_get_name (source->db->db));
    source->uri_to_item = g_hash_table_new (g_str_hash, g_str_equal);
    g_timer_start (timer);
    source->items = bkl_db_get_items (source->db, FALSE, &error);
    if (source->items == NULL) {
        g_error ("(%s): Error getting items: %s", source->source_name,
                 error->message);
    }
    g_timer_stop (timer);
    g_print ("(%s): Loaded %d items in %.3fseconds\n",
             source->source_name,
             source->items->len,
             g_timer_elapsed (timer, NULL));

    create_item_store (source);

    return source;
}

static void
delete_source (Source *source)
{
    int i;

    g_object_unref (source->client);
    bkl_db_free (source->db);
    g_free (source->source_name);

    g_hash_table_destroy (source->uri_to_item);
    for (i = 0; i < source->items->len; i++) {
        g_object_unref ((GObject *) source->items->pdata[i]);
    }
    g_ptr_array_free (source->items, TRUE);

    gtk_tree_view_set_model (GTK_TREE_VIEW (item_list), NULL);

    g_object_unref (source->store);
}

static void
get_item_text_func (GtkTreeViewColumn *col,
                    GtkCellRenderer   *renderer,
                    GtkTreeModel      *model,
                    GtkTreeIter       *iter,
                    gpointer           userdata)
{
    BklItem *item;
    char *markup;

    gtk_tree_model_get (model, iter, ITEM_POINTER, &item, -1);
    g_assert (item);

    switch (bkl_item_get_item_type (item)) {
    case BKL_ITEM_TYPE_AUDIO: {
        BklItemAudio *audio = (BklItemAudio *) item;
        char *title = NULL, *album = NULL, *artist = NULL;
        GPtrArray *artists;
        guint track;

        artists = bkl_item_audio_get_artists (audio);
        if (artists) {
            if (artists->pdata[0]) {
                artist = g_markup_escape_text (artists->pdata[0], -1);
            }
        }

        if (bkl_item_audio_get_title (audio)) {
            title = g_markup_escape_text (bkl_item_audio_get_title (audio), -1);
        } else {
            title = g_markup_escape_text (bkl_item_get_uri (item), -1);
        }

        if (bkl_item_audio_get_album (audio)) {
            album = g_markup_escape_text (bkl_item_audio_get_album (audio), -1);
        }

        track = bkl_item_audio_get_track (audio);

        markup = g_strdup_printf ("<b>%d - %s</b>\n<small><i>%s by %s</i></small>",
                                  track, title,
                                  album ? album : "Unknown",
                                  artist ? artist : "Unknown");
        g_free (title);
        g_free (album);
        g_free (artist);

        break;
    }

    case BKL_ITEM_TYPE_IMAGE: {
        BklItemImage *image = (BklItemImage *) item;
        char *title = NULL;

        if (bkl_item_image_get_title (image)) {
            title = g_markup_escape_text (bkl_item_image_get_title (image), -1);
        } else {
            title = g_markup_escape_text (bkl_item_get_uri (item), -1);
        }

        markup = g_strdup_printf ("<b>%s</b>\n<small><i>%dx%d</i></small>",
                                  title,
                                  bkl_item_image_get_width (image),
                                  bkl_item_image_get_height (image));

        break;
    }
    case BKL_ITEM_TYPE_VIDEO: {
        BklItemVideo *video = (BklItemVideo *) item;
        char *title = NULL;

        if (bkl_item_video_get_series_name (video)) {
            title = g_markup_escape_text (bkl_item_video_get_series_name (video), -1);
        } else {
            title = g_markup_escape_text (bkl_item_get_uri (item), -1);
        }

        markup = g_strdup_printf ("<b>%s</b> - %s\n<small><i>%dx%d</i></small>",
                                  title,
                                  bkl_item_video_get_title (video) ?
                                  bkl_item_video_get_title (video) : "",
                                  bkl_item_video_get_season (video),
                                  bkl_item_video_get_episode (video));
        break;
    }

    case BKL_ITEM_TYPE_BROKEN:
        markup = g_strdup ("Broken");
        break;

    default:
        markup = g_strdup ("TBA");

        break;
    }

    g_object_set (renderer, "markup", markup, NULL);
    g_free (markup);
}

static void
source_changed (GtkTreeSelection *selection,
                gpointer          userdata)
{
    GtkTreeModel *model;
    GtkTreeIter iter;

    if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
        Source *source;

        gtk_tree_model_get (model, &iter, SOURCE_POINTER, &source, -1);
        if (source == NULL) {
            g_warning ("No source found");
            return;
        }

        gtk_tree_view_set_model (GTK_TREE_VIEW (item_list),
                                 GTK_TREE_MODEL (source->store));
    }
}

static void
create_ui (void)
{
    GtkWidget *window, *hbox, *vbox, *scroller;
    GtkTreeViewColumn *col;
    GtkCellRenderer *renderer;
    GtkTreeSelection *selection;

    window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title (GTK_WINDOW (window), "Bickley Source Tester");
    gtk_window_set_default_size (GTK_WINDOW (window), 600, 400);
    g_signal_connect (window, "destroy",
                      G_CALLBACK (gtk_main_quit), NULL);

    hbox = gtk_hbox_new (FALSE, 6);
    gtk_container_add (GTK_CONTAINER (window), hbox);

    vbox = gtk_vbox_new (FALSE, 6);
    gtk_box_pack_start (GTK_BOX (hbox), vbox, FALSE, FALSE, 0);

    scroller = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroller),
                                    GTK_POLICY_AUTOMATIC,
                                    GTK_POLICY_AUTOMATIC);
    gtk_box_pack_start (GTK_BOX (vbox), scroller, TRUE, TRUE, 0);

    source_list = gtk_tree_view_new_with_model (GTK_TREE_MODEL (source_store));
    gtk_container_add (GTK_CONTAINER (scroller), source_list);

    col = gtk_tree_view_column_new ();
    gtk_tree_view_column_set_title (col, "Source");
    gtk_tree_view_append_column (GTK_TREE_VIEW (source_list), col);

    renderer = gtk_cell_renderer_text_new ();
    gtk_tree_view_column_pack_start (col, renderer, TRUE);
    gtk_tree_view_column_add_attribute (col, renderer, "text", SOURCE_NAME);

    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (source_list));
    gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);
    g_signal_connect (selection, "changed",
                      G_CALLBACK (source_changed), NULL);

    scroller = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroller),
                                    GTK_POLICY_AUTOMATIC,
                                    GTK_POLICY_AUTOMATIC);
    gtk_box_pack_start (GTK_BOX (hbox), scroller, TRUE, TRUE, 0);

    item_list = gtk_tree_view_new ();
    gtk_container_add (GTK_CONTAINER (scroller), item_list);

    col = gtk_tree_view_column_new ();
    gtk_tree_view_append_column (GTK_TREE_VIEW (item_list), col);

    renderer = gtk_cell_renderer_pixbuf_new ();
    gtk_tree_view_column_pack_start (col, renderer, TRUE);
    gtk_tree_view_column_add_attribute (col, renderer, "pixbuf", ITEM_PIXBUF);

    renderer = gtk_cell_renderer_text_new ();
    gtk_tree_view_column_pack_start (col, renderer, TRUE);
    gtk_tree_view_column_set_cell_data_func (col, renderer,
                                             get_item_text_func, NULL, NULL);

    gtk_widget_show_all (window);
}

static void
source_added (BklSourceManagerClient *manager,
              const char             *path,
              gpointer                userdata)
{
    Source *source;
    BklSourceClient *client;

    client = bkl_source_client_new (path);
    if (client == NULL) {
        g_error ("Unable to get client %s", path);
    }

    g_print ("Added %s\n", path);

    source = create_source (client);
    if (source == NULL) {
        return;
    }

    add_source (source);
    g_hash_table_insert (source_by_path, g_strdup (path), source);

    gtk_tree_view_set_model (GTK_TREE_VIEW (item_list),
                             GTK_TREE_MODEL (source->store));
}

static void
source_removed (BklSourceManagerClient *manager,
                const char             *path,
                gpointer                userdata)
{
    Source *source;

    g_print ("Removed %s\n", path);

    source = g_hash_table_lookup (source_by_path, path);
    if (source == NULL) {
        g_warning ("No source for %s", path);
        return;
    }

    g_hash_table_remove (source_by_path, path);
    remove_source (source);
    delete_source (source);
}

static void
source_ready (BklSourceClient *client,
              char            *path)
{
    Source *source;

    source = create_source (client);
    add_source (source);
    g_hash_table_insert (source_by_path, path, source);
}

static void
get_sources_reply (BklSourceManagerClient *manager,
                   GList                  *sources,
                   GError                 *error,
                   gpointer                userdata)
{
    BklSourceClient *client;
    GList *s;

    if (error != NULL) {
        g_warning ("Error getting sources: %s", error->message);
    }

    for (s = sources; s; s = s->next) {
        char *path = s->data;

        client = bkl_source_client_new (path);
        g_signal_connect (client, "ready",
                          G_CALLBACK (source_ready), path);
    }

    g_list_free (sources);
}

static void
manager_ready (BklSourceManagerClient *manager,
               gpointer                userdata)
{
    bkl_source_manager_client_get_sources (manager,
                                           get_sources_reply, userdata);
}

int
main (int    argc,
      char **argv)
{
    BklSourceClient *client;
    BklSourceManagerClient *manager;

    g_thread_init (NULL);
    bkl_init ();
    gtk_init (&argc, &argv);

    timer = g_timer_new ();

    manager = g_object_new (BKL_TYPE_SOURCE_MANAGER_CLIENT, NULL);
    g_signal_connect (manager, "ready",
                      G_CALLBACK (manager_ready), NULL);
    g_signal_connect (manager, "source-added",
                      G_CALLBACK (source_added), NULL);
    g_signal_connect (manager, "source-removed",
                      G_CALLBACK (source_removed), NULL);

    source_by_path = g_hash_table_new_full (g_str_hash, g_str_equal,
                                            g_free, NULL);
    source_store = gtk_list_store_new (SOURCE_COLS, G_TYPE_STRING, G_TYPE_POINTER);

    client = bkl_source_client_new (BKL_LOCAL_SOURCE_PATH);
    if (client == NULL) {
        g_error ("Unable to get local client");
    }
    g_signal_connect (client, "ready",
                      G_CALLBACK (source_ready),
                      g_strdup (BKL_LOCAL_SOURCE_PATH));

    create_ui ();

    gtk_main ();

    return 0;
}
