/*
 * Bickley - a meta data management framework.
 * Copyright © 2008, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#include <glib.h>

#include <kozo.h>

enum {
    FIELD_STRING,
    FIELD_INT,
    FIELD_TIME,
    FIELD_STRING_LIST,
    FIELD_INT_LIST,
    FIELD_TIME_LIST,
    LAST_FIELD
};

static KozoDB *db;

#define TEST_INT (gint32) 12345
#define TEST_STRING "string"
static time_t testtime;

static int fields[LAST_FIELD] = {0, };
static char *fieldnames[LAST_FIELD] = {
    "string", "int", "time", "string-list", "int-list", "time-list"
};
static char *test_string_list[] = {
    "test1", "test2", "test3", "test4", "test5",
    "test6", "test7", "test8", "test9", "test10",
    NULL
};

static void
test_db ()
{
    GError *error = NULL;
    char *dirname, *filename;

    dirname = g_build_filename (g_get_home_dir (), ".kozo", "databases", NULL);
    g_mkdir_with_parents (dirname, 0777);

    filename = g_build_filename (dirname, "kozo-test-db", NULL);
    unlink (filename);

    db = kozo_db_get_for_path (filename, "kozo-test-db", 1, &error);
    if (error != NULL) {
        g_print ("error: %s", error->message);
    }

    g_assert (db);

    g_free (dirname);
    g_free (filename);
}

static void
test_register_fields ()
{
    GError *error = NULL;
    int i;

    /* Add fields to the database */
    for (i = 0; i < LAST_FIELD; i++) {
        fields[i] = kozo_db_register_field (db, fieldnames[i], &error);
        g_assert (fields[i] >= 0);
    }
}

static GSList *
make_string_list (void)
{
    GSList *l = NULL;
    int i;
    for (i = 0; test_string_list[i]; i++) {
        l = g_slist_append (l, g_strdup (test_string_list[i]));
    }
    return l;
}

static GSList *
make_int_list (void)
{
    GSList *l = NULL;
    int i;
    for (i = 0; i < 10; i++) {
        l = g_slist_append (l, GINT_TO_POINTER (i));
    }
    return l;
}

static GSList *
make_time_list (void)
{
    GSList *l = NULL;
    int i;
    for (i = 0; i < 10; i++) {
        l = g_slist_append (l, GINT_TO_POINTER (testtime + i));
    }
    return l;
}

static void
test_add_row ()
{
    GSList *f = NULL;
    GError *error = NULL;
    gboolean ret;

    testtime = time (NULL);

    f = g_slist_append (f, kozo_field_new_string (FIELD_STRING, TEST_STRING));
    f = g_slist_append (f, kozo_field_new_int (FIELD_INT, TEST_INT));
    f = g_slist_append (f, kozo_field_new_time (FIELD_TIME, testtime));
    f = g_slist_append (f, kozo_field_new_string_list (FIELD_STRING_LIST,
                                                       make_string_list ()));
    f = g_slist_append (f, kozo_field_new_int_list (FIELD_INT_LIST,
                                                    make_int_list ()));
    f = g_slist_append (f, kozo_field_new_time_list (FIELD_TIME_LIST,
                                                     make_time_list ()));

    ret = kozo_db_add (db, "test-key", f, &error);
    g_assert (ret);

    kozo_field_list_free (f);
}

static void
test_add_duplicate_row ()
{
    GSList *f = NULL;
    GError *error = NULL;
    gboolean ret;

    testtime = time (NULL);

    f = g_slist_append (f, kozo_field_new_string (FIELD_STRING, TEST_STRING));
    f = g_slist_append (f, kozo_field_new_int (FIELD_INT, TEST_INT));
    f = g_slist_append (f, kozo_field_new_time (FIELD_TIME, testtime));
    f = g_slist_append (f, kozo_field_new_string_list (FIELD_STRING_LIST,
                                                       make_string_list ()));
    f = g_slist_append (f, kozo_field_new_int_list (FIELD_INT_LIST,
                                                    make_int_list ()));
    f = g_slist_append (f, kozo_field_new_time_list (FIELD_TIME_LIST,
                                                     make_time_list ()));

    ret = kozo_db_add (db, "test-key", f, &error);
    g_assert (ret == FALSE);

    g_assert (error->code == KOZO_DB_ERROR_KEY_EXISTS);

    kozo_field_list_free (f);
}

static void
test_lookup_row ()
{
    GSList *ids = NULL;
    GError *error = NULL;
    KozoEntry *entry;
    KozoField *field;
    const char *data;
    int number;
    time_t t;
    int i;

    ids = g_slist_append (ids, GINT_TO_POINTER (fields[FIELD_STRING]));
    ids = g_slist_append (ids, GINT_TO_POINTER (fields[FIELD_INT]));
    ids = g_slist_append (ids, GINT_TO_POINTER (fields[FIELD_TIME]));
    ids = g_slist_append (ids, GINT_TO_POINTER (fields[FIELD_STRING_LIST]));
    ids = g_slist_append (ids, GINT_TO_POINTER (fields[FIELD_INT_LIST]));
    ids = g_slist_append (ids, GINT_TO_POINTER (fields[FIELD_TIME_LIST]));

    entry = kozo_db_lookup (db, "test-key", ids, &error);
    if (entry == NULL) {
        g_print ("\nError finding test-key: %s\n", error->message);

        g_assert (entry);
    }

    /* Test string */
    field = kozo_entry_get_field (entry, FIELD_STRING);
    g_assert (field);

    data = kozo_field_get_value_string (field, 0);
    g_assert (data);

    g_assert (strcmp (data, TEST_STRING) == 0);


    /* Test integer */
    field = kozo_entry_get_field (entry, FIELD_INT);
    g_assert (field);

    number = kozo_field_get_value_int (field, 0);

    g_assert (number == TEST_INT);

    /* Test time */
    field = kozo_entry_get_field (entry, FIELD_TIME);
    g_assert (field);

    t = kozo_field_get_value_time (field, 0);
    g_assert (t == testtime);


    /* Test string list */
    field = kozo_entry_get_field (entry, FIELD_STRING_LIST);
    g_assert (field);

    for (i = 0; test_string_list[i]; i++) {
        const char *str = kozo_field_get_value_string (field, i);

        g_assert (strcmp (str, test_string_list[i]) == 0);
    }

    /* Test int list */
    field = kozo_entry_get_field (entry, FIELD_INT_LIST);
    g_assert (field);

    for (i = 0; i < 10; i++) {
        number = kozo_field_get_value_int (field, i);

        g_assert (number == i);
    }

    field = kozo_entry_get_field (entry, FIELD_TIME_LIST);
    g_assert (field);

    for (i = 0; i < 10; i++) {
        t = kozo_field_get_value_time (field, i);
        g_assert (t == testtime + i);
    }
}

static void
test_remove_row ()
{
    GError *error = NULL;
    gboolean ret;

    ret = kozo_db_remove (db, "test-key", &error);
    if (ret == FALSE) {
        g_print ("\nError removing row: %s\n", error->message);
        g_assert (ret);
    }
}

int
main (int    argc,
      char **argv)
{
    int ret;
    char *fn;

    g_thread_init (NULL);
    g_test_init (&argc, &argv, NULL);

    kozo_init ();

    g_test_add_func ("/Create DB", test_db);
    g_test_add_func ("/Register Fields", test_register_fields);
    g_test_add_func ("/Add Row", test_add_row);
    g_test_add_func ("/Add Duplicate Row", test_add_duplicate_row);
    g_test_add_func ("/Lookup Row", test_lookup_row);
    g_test_add_func ("/Remove Row", test_remove_row);

    ret = g_test_run ();

    kozo_db_unref (db);

    return ret;
}
