/*
 * Bickley - a meta data management framework.
 * Copyright © 2008, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <string.h>

#include <glib/gprintf.h>
#include <gio/gio.h>
#include <libexif/exif-loader.h>
#include <libexif/exif-data.h>

#include "exif-metadata.h"
#include "metadata-defines.h"

typedef struct ExifMapping {
    ExifTag  tag;
    gchar   *string;
} ExifMapping;

static ExifMapping mappings[] =
{
    {EXIF_TAG_MAKE,                METADATA_IMAGE_MAKE},
    {EXIF_TAG_MODEL,               METADATA_IMAGE_MODEL},
    {EXIF_TAG_DATE_TIME,           METADATA_IMAGE_DATE_TIME},
    {EXIF_TAG_DATE_TIME_ORIGINAL,  METADATA_IMAGE_DATE_TIME_ORIGINAL},
    {EXIF_TAG_DATE_TIME_DIGITIZED, METADATA_IMAGE_DATE_TIME_DIGITIZED},

    {EXIF_TAG_ORIENTATION,         METADATA_IMAGE_ORIENTATION},
    {EXIF_TAG_EXPOSURE_TIME,       METADATA_IMAGE_EXPOSURE_TIME},
    {EXIF_TAG_APERTURE_VALUE,      METADATA_IMAGE_APERTURE_VALUE},
    {EXIF_TAG_METERING_MODE,       METADATA_IMAGE_METERING_MODE},
    {EXIF_TAG_FLASH,               METADATA_IMAGE_FLASH},
    {EXIF_TAG_FOCAL_LENGTH,        METADATA_IMAGE_FOCAL_LENGTH},
    {EXIF_TAG_SHUTTER_SPEED_VALUE, METADATA_IMAGE_SHUTTER_SPEED},
    {EXIF_TAG_ISO_SPEED_RATINGS,   METADATA_IMAGE_ISO},
    {EXIF_TAG_SOFTWARE,            METADATA_IMAGE_SOFTWARE},

    {EXIF_TAG_XP_TITLE,            METADATA_IMAGE_TITLE},
    {EXIF_TAG_XP_COMMENT,          METADATA_IMAGE_COMMENT},
    {EXIF_TAG_USER_COMMENT,        METADATA_IMAGE_USER_COMMENT},
    {EXIF_TAG_XP_AUTHOR,           METADATA_IMAGE_AUTHOR},
    {EXIF_TAG_COPYRIGHT,           METADATA_IMAGE_COPYRIGHT},
    {EXIF_TAG_XP_KEYWORDS,         METADATA_IMAGE_KEYWORDS},
    {EXIF_TAG_XP_SUBJECT,          METADATA_IMAGE_SUBJECT},

    {0, NULL}
};

void each_entry (ExifEntry *entry,
                 gpointer   userdata)
{
    GHashTable *metadata = userdata;
    gchar value[512];
    gint i;

    if (entry == NULL)
        return;

    exif_entry_get_value(entry, value, 512);
    value[511] = '\0';

    if (value[0] == '\0')
        return;

    for (i=0;mappings[i].string;i++) {
        if (mappings[i].tag == entry->tag) {
            g_hash_table_insert (metadata, mappings[i].string,
                                 g_strdup (value));
            break;
        }
    }
}

static void
each_content (ExifContent *content,
              gpointer     userdata)
{
    if (content) {
        exif_content_foreach_entry(content, each_entry, userdata);
    }
}


gboolean
bkl_task_exif_get_metadata (GFile      *file,
                            GFileInfo  *info,
                            const char *mimetype,
                            GHashTable *metadata)
{
    ExifLoader *exif_loader;
    ExifData   *exif_data;
    gchar *path;

    if (strncmp (mimetype, "image/", 6) != 0) {
        return FALSE;
    }

    path = g_file_get_path (file);
    if (!path) {
        return FALSE;
    }

    exif_loader = exif_loader_new ();

    /* XXX: make use of GIO instead */
    exif_loader_write_file (exif_loader, path);
    exif_data = exif_loader_get_data (exif_loader);

    if (exif_data == NULL) {
        g_free (path);
        return FALSE;
    }

    exif_data_foreach_content (exif_data, each_content, metadata);
    exif_loader_unref (exif_loader);

    exif_data_free (exif_data);
    g_free (path);

    return TRUE;
}
