/*
 * Bickley - a meta data management framework.
 * Copyright © 2008, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <errno.h>
#include <sys/types.h>
#include <dirent.h>
#include <string.h>

#include <gio/gio.h>
#include <gst/gst.h>

#include <gdk-pixbuf/gdk-pixbuf.h>

#include "audio-metadata.h"
#include "gst-helper.h"
#include "metadata-defines.h"

struct _metametadata {
    GHashTable *metadata;
    GFile *file;
};

/* FIXME: 2 minutes might be too long, it might be helpful to do some
   timing experiments over cache hit/misses, but the amount of memory
   used is probably not worth it. */
#define COVERART_CACHE_LIFESPAN 120 /* 2 minutes */
static GHashTable *coverart_cache = NULL;

#define THUMBNAIL_SIZE 256
static char *
write_thumbnail (GFile     *file,
                 GdkPixbuf *pixbuf)
{
    GdkPixbuf *pb;
    char *uri, *filename, *fullname;
    GError *error = NULL;
    int width, height;

    uri = g_file_get_uri (file);
    filename = g_compute_checksum_for_string (G_CHECKSUM_MD5, uri, -1);

    fullname = g_build_filename (g_get_home_dir (), ".bkl-thumbnails",
                                 filename, NULL);
    g_free (filename);

    width = gdk_pixbuf_get_width (pixbuf);
    height = gdk_pixbuf_get_height (pixbuf);

    if (width > THUMBNAIL_SIZE || height > THUMBNAIL_SIZE) {
        int nw, nh;
        float ratio;

        if (width > height) {
            nw = THUMBNAIL_SIZE;
            ratio = (float) width / (float) THUMBNAIL_SIZE;
            nh = (int) ((float) height / ratio);
        } else {
            nh = THUMBNAIL_SIZE;
            ratio = (float) height / (float) THUMBNAIL_SIZE;
            nw = (int) ((float) width / ratio);
        }

        pb = gdk_pixbuf_scale_simple (pixbuf, nw, nh, GDK_INTERP_BILINEAR);
    } else {
        pb = g_object_ref (pixbuf);
    }

    if (gdk_pixbuf_save (pb, fullname, "png", &error, NULL) == FALSE) {
        g_warning ("Error writing file %s for %s: %s", fullname, uri,
                   error->message);
        g_error_free (error);

        g_object_unref (pb);
        g_free (uri);
        g_free (fullname);

        return NULL;
    }

    g_object_unref (pb);
    g_free (uri);

    uri = g_filename_to_uri (fullname, NULL, &error);
    if (uri == NULL) {
        g_warning ("Error converting %s to uri: %s", fullname, error->message);
        g_error_free (error);
        g_free (fullname);

        return NULL;
    }

    g_free (fullname);
    return uri;
}

static void
parse_tags (const GstTagList *tags,
            const char       *tag,
            gpointer          userdata)
{
    struct _metametadata *mmd = (struct _metametadata *) userdata;
    GHashTable *metadata = (GHashTable *) mmd->metadata;
    char *value;

    /* FIXME: Deal with replay gain? */

    if (g_str_equal (tag, GST_TAG_TITLE)) {
        gst_tag_list_get_string (tags, tag, &value);
        g_hash_table_insert (metadata, METADATA_AUDIO_TITLE, value);
    } else if (g_str_equal (tag, GST_TAG_ARTIST)) {
        gst_tag_list_get_string (tags, tag, &value);
        g_hash_table_insert (metadata, METADATA_AUDIO_ARTIST, value);
    } else if (g_str_equal (tag, GST_TAG_ALBUM)) {
        gst_tag_list_get_string (tags, tag, &value);
        g_hash_table_insert (metadata, METADATA_AUDIO_ALBUM, value);
    } else if (g_str_equal (tag, GST_TAG_GENRE)) {
        gst_tag_list_get_string (tags, tag, &value);
        g_hash_table_insert (metadata, METADATA_AUDIO_GENRE, value);
    } else if (g_str_equal (tag, GST_TAG_DATE)) {
        GDate *d;

        gst_tag_list_get_date (tags, tag, &d);
        g_hash_table_insert (metadata, METADATA_AUDIO_YEAR,
                             g_strdup_printf ("%u", g_date_get_year (d)));
        /* FIXME: Free d? */
    } else if (g_str_equal (tag, GST_TAG_COMMENT)) {
        gst_tag_list_get_string (tags, tag, &value);
        g_hash_table_insert (metadata, METADATA_AUDIO_COMMENT, value);
    } else if (g_str_equal (tag, GST_TAG_TRACK_NUMBER)) {
        guint track;

        gst_tag_list_get_uint (tags, tag, &track);
        g_hash_table_insert (metadata, METADATA_AUDIO_TRACK,
                             g_strdup_printf ("%u", track));
    } else if (g_str_equal (tag, GST_TAG_TRACK_COUNT)) {
        guint count;

        gst_tag_list_get_uint (tags, tag, &count);
        g_hash_table_insert (metadata, METADATA_AUDIO_TOTAL_TRACKS,
                             g_strdup_printf ("%u", count));
    } else if (g_str_equal (tag, GST_TAG_ALBUM_VOLUME_NUMBER)) {
        guint disc;

        gst_tag_list_get_uint (tags, tag, &disc);
        g_hash_table_insert (metadata, METADATA_AUDIO_DISC,
                             g_strdup_printf ("%u", disc));
    } else if (g_str_equal (tag, GST_TAG_ALBUM_VOLUME_COUNT)) {
        guint count;

        gst_tag_list_get_uint (tags, tag, &count);
        g_hash_table_insert (metadata, METADATA_AUDIO_TOTAL_DISCS,
                             g_strdup_printf ("%u", count));
    } else if (g_str_equal (tag, GST_TAG_PERFORMER)) {
        gst_tag_list_get_string (tags, tag, &value);
        g_hash_table_insert (metadata, METADATA_AUDIO_PERFORMERS, value);
    } else if (g_str_equal (tag, GST_TAG_COPYRIGHT)) {
        gst_tag_list_get_string (tags, tag, &value);
        g_hash_table_insert (metadata, METADATA_AUDIO_COPYRIGHT, value);
    } else if (g_str_equal (tag, GST_TAG_LICENSE)) {
        gst_tag_list_get_string (tags, tag, &value);
        g_hash_table_insert (metadata, METADATA_AUDIO_LICENSE, value);
    } else if (g_str_equal (tag, GST_TAG_ORGANIZATION)) {
        gst_tag_list_get_string (tags, tag, &value);
        g_hash_table_insert (metadata, METADATA_AUDIO_ORGANISATION, value);
    } else if (g_str_equal (tag, GST_TAG_CONTACT)) {
        gst_tag_list_get_string (tags, tag, &value);
        g_hash_table_insert (metadata, METADATA_AUDIO_CONTACT, value);
    } else if (g_str_equal (tag, GST_TAG_ISRC)) {
        gst_tag_list_get_string (tags, tag, &value);
        g_hash_table_insert (metadata, METADATA_AUDIO_ISRC, value);
    } else if (g_str_equal (tag, GST_TAG_COMPOSER)) {
        gst_tag_list_get_string (tags, tag, &value);
        g_hash_table_insert (metadata, METADATA_AUDIO_COMPOSER, value);
    } else if (g_str_equal (tag, GST_TAG_IMAGE)) {
        const GValue *value;
        GstBuffer *buffer;
        GdkPixbuf *thumb;

        value = gst_tag_list_get_value_index (tags, tag, 0);
        buffer = gst_value_get_buffer (value);

        if (buffer) {
            GdkPixbufLoader *loader;
            GError *error = NULL;

            loader = gdk_pixbuf_loader_new ();
            if (!gdk_pixbuf_loader_write (loader, GST_BUFFER_DATA (buffer),
                                          GST_BUFFER_SIZE (buffer), &error)) {
                char *uri = g_file_get_uri (mmd->file);

                g_warning ("Error loading data for %s: %s", uri,
                           error->message);
                g_free (uri);
                g_error_free (error);

                g_object_unref (loader);
                return;
            }

            gdk_pixbuf_loader_close (loader, NULL);
            thumb = gdk_pixbuf_loader_get_pixbuf (loader);

            if (thumb) {
                char *art = write_thumbnail (mmd->file, thumb);

                if (art) {
                    g_hash_table_insert (metadata, METADATA_EXTENDED_THUMBNAIL,
                                         art);
                }
            }

            /* Unrefs the pixbuf as well */
            g_object_unref (loader);
        }
    }

    /* Doesn't handle CONDUCTOR */
}

static guint
get_track_from_file (GFile *file)
{
    char *basename = g_file_get_basename (file);
    guint track;

    /* This will break on tracks that have a number as their name:
       "10,000 Hurts" by Tori Amos for example, that don't have
       the track number before it... */
    track = atoi (basename);
    g_free (basename);

    /* Just check we have a slightly sensible number...
       Some albums do have 99 tracks: Broken by NIN and
       The Second Coming by Stone Roses for example */
    if (track > 99) {
        track = 0;
    }

    return track;
}

static char *
get_title_from_file (GFile *file)
{
    return g_strdup ("");
}

static gboolean
clean_cache (gpointer data)
{
    g_hash_table_destroy (coverart_cache);
    coverart_cache = NULL;
    return FALSE;
}

static GdkPixbuf *
find_art_in_directory (GFile *file)
{
    GdkPixbuf *pixbuf;
    char *path;
    char *parent_dir;
    DIR *parent;
    struct dirent *d_ent;
    char *found_art = NULL;
    char *album_art = NULL;
    char *small = NULL, *small2 = NULL, *large = NULL;
    GError *error = NULL;

    path = g_file_get_path (file);
    parent_dir = g_path_get_dirname (path);
    g_free (path);

    if (G_UNLIKELY (coverart_cache == NULL)) {
        coverart_cache = g_hash_table_new_full (g_str_hash, g_str_equal,
                                                g_free, g_free);

        /* Clean the cache after 2 minutes so memory isn't wasted too much
           for directories we don't really care about anymore */
        g_timeout_add_seconds (COVERART_CACHE_LIFESPAN, clean_cache, NULL);
    }

    /* Check the cache first */
    found_art = g_hash_table_lookup (coverart_cache, parent_dir);
    if (found_art != NULL) {
        pixbuf = gdk_pixbuf_new_from_file (found_art, &error);
        if (error != NULL) {
            g_warning ("Error loading %s: %s", found_art, error->message);
            g_error_free (error);
        }

        g_free (parent_dir);
        return pixbuf;
    }

    parent = opendir (parent_dir);
    if (parent == NULL) {
        g_warning ("Error opening %s: %s (%d)",
                   parent_dir, g_strerror (errno), errno);
        g_free (parent_dir);
        return NULL;
    }

    d_ent = readdir (parent);
    while (d_ent) {
        if (d_ent->d_name[0] == '.') {
            d_ent = readdir (parent);
            continue;
        }

        if (g_ascii_strncasecmp (d_ent->d_name, "folder", 6) == 0 ||
            g_ascii_strncasecmp (d_ent->d_name, "cover", 5) == 0 ||
            g_ascii_strncasecmp (d_ent->d_name, "front", 5) == 0) {
            if (g_str_has_suffix (d_ent->d_name, ".jpg") ||
                g_str_has_suffix (d_ent->d_name, ".gif") ||
                g_str_has_suffix (d_ent->d_name, ".png") ||
                g_str_has_suffix (d_ent->d_name, ".JPG")) {
                album_art = g_strdup (d_ent->d_name);
                break;
            }
        }

        if (strstr (d_ent->d_name, "folder") ||
            strstr (d_ent->d_name, "cover") ||
            strstr (d_ent->d_name, "front")) {
            if (g_str_has_suffix (d_ent->d_name, ".jpg") ||
                g_str_has_suffix (d_ent->d_name, ".gif") ||
                g_str_has_suffix (d_ent->d_name, ".png") ||
                g_str_has_suffix (d_ent->d_name, ".JPG")) {
                album_art = g_strdup (d_ent->d_name);
                break;
            }
        }

        if (g_ascii_strncasecmp (d_ent->d_name, "AlbumArt", 8) == 0) {
            if (g_ascii_strcasecmp (d_ent->d_name + 8, "Small.jpg") == 0) {
                /* We've found AlbumArtSmall.jpg */
                small = g_strdup (d_ent->d_name);
            } else {
                /* FIXME: Would be nice to have a case free version */
                if (g_str_has_suffix (d_ent->d_name, "_Small.jpg") ||
                    g_str_has_suffix (d_ent->d_name, "_small.jpg")) {
                    /* We've found AlbumArt_{<GUID>}_Small.jpg */
                    small2 = g_strdup (d_ent->d_name);
                } else if (g_str_has_suffix (d_ent->d_name, "_Large.jpg") ||
                           g_str_has_suffix (d_ent->d_name, "_large.jpg")) {
                    /* We've found AlbumArt_{<GUID>}_Large.jpg */
                    large = g_strdup (d_ent->d_name);
                }
            }
        }

        d_ent = readdir (parent);
    }

    closedir (parent);

    /* folder.jpg, cover.jpg and front.jpg take priority over the
       AlbumArt* ones. */
    if (album_art == NULL) {
        /* large > small2 > small */
        album_art = g_strdup (large ? large : (small2 ? small2 : small));
    }

    g_free (large);
    g_free (small2);
    g_free (small);

    if (album_art == NULL) {
        g_free (parent_dir);
        return NULL;
    }

    found_art = g_build_filename (parent_dir, album_art, NULL);
    g_hash_table_insert (coverart_cache, parent_dir, found_art);

    pixbuf = gdk_pixbuf_new_from_file (found_art, &error);
    if (error != NULL) {
        g_warning ("Error loading %s: %s", found_art, error->message);
        g_error_free (error);
    }

    /* found_art and parent_dir will be destroyed when the cache is destroyed */
    g_free (album_art);

    return pixbuf;
}

/* Check the values for metadata make sense
    - Is there a track number?
    - Is track number > 0 < trackcount?
    - Is there album art?
    - Is there a track title?

    Try to get track name, track number from the filename
    Try to find album art in the current directory
*/
static void
sanitise_metadata (GFile      *file,
                   GHashTable *metadata)
{
    char *data;
    int track = 0;

    if ((data = g_hash_table_lookup (metadata, METADATA_AUDIO_TRACK))) {
        gboolean track_changed = FALSE;

        track = atoi (data);

        if (track < 0) {
            track = 0;
            track_changed = TRUE;
        }

        if ((data = g_hash_table_lookup (metadata, METADATA_AUDIO_TOTAL_TRACKS))) {
            int tracks = atoi (data);

            if (track > tracks) {
                track = 0;
                track_changed = TRUE;
            }
        }

        if (track == 0) {
            track = get_track_from_file (file);
            track_changed = TRUE;
        }

        if (track_changed) {
            g_hash_table_replace (metadata, METADATA_AUDIO_TRACK,
                                  g_strdup_printf ("%u", track));
        }
    }

    if (g_hash_table_lookup (metadata, METADATA_AUDIO_TITLE) == NULL) {
        char *title = get_title_from_file (file);
        g_hash_table_insert (metadata, METADATA_AUDIO_TITLE, title);
    }

    /* Find album art */
    if (g_hash_table_lookup (metadata, METADATA_EXTENDED_THUMBNAIL) == NULL) {
        GdkPixbuf *pixbuf = find_art_in_directory (file);

        if (pixbuf) {
            char *art = write_thumbnail (file, pixbuf);

            g_object_unref (pixbuf);

            if (art) {
                g_hash_table_insert (metadata, METADATA_EXTENDED_THUMBNAIL, art);
            }
        }
    }
}

/* Return TRUE if we have hit EOS or error */
static gboolean
parse_message (GstMessage *msg,
               GFile      *file,
               GHashTable *metadata)
{
    struct _metametadata mmd;
    GError *error;
    char *debug, *src;
    GstTagList *tags;

    switch (GST_MESSAGE_TYPE (msg)) {
    case GST_MESSAGE_EOS:
        return TRUE;

    case GST_MESSAGE_ERROR:
        gst_message_parse_error (msg, &error, &debug);
        src = gst_element_get_name (GST_MESSAGE_SRC (msg));

        g_warning ("Error from %s: %s (%s)", src, error->message, debug);
        g_free (src);
        g_error_free (error);
        g_free (debug);

        /* Return TRUE to give up on error */
        return TRUE;

    case GST_MESSAGE_TAG:
        mmd.metadata = metadata;
        mmd.file = file;

        gst_message_parse_tag (msg, &tags);
        gst_tag_list_foreach (tags, parse_tags, &mmd);
        gst_tag_list_free (tags);
        break;

    default:
        break;

    }

    return FALSE;
}

#if __WORDSIZE == 64
#define G_GINT64_MODIFIER "l"
# else
#define G_GINT64_MODIFIER "ll"
#endif
gboolean
bkl_task_audio_get_metadata (GFile      *file,
                             GFileInfo  *info,
                             const char *mimetype,
                             GHashTable *metadata)
{
    GstElement *playbin, *audio_sink, *video_sink;
    GstStateChangeReturn state;
    char *uri;
    int count = 0;

    if (!g_str_has_prefix (mimetype, "audio/") ||
        g_str_equal (mimetype, "audio/x-mpegurl")) {
        return FALSE;
    }

    uri = g_file_get_uri (file);

    playbin = gst_element_factory_make ("playbin", "playbin");
    audio_sink = gst_element_factory_make ("fakesink", "audiosink");
    video_sink = gst_element_factory_make ("fakesink", "videosink");

    g_object_set (playbin,
                  "uri", uri,
                  "audio-sink", audio_sink,
                  "video-sink", video_sink,
                  NULL);

    g_free (uri);

    state = gst_element_set_state (playbin, GST_STATE_PAUSED);
    while (state == GST_STATE_CHANGE_ASYNC && count < 5) {
        g_print ("Audio loop count %d\n", count);
        state = gst_element_get_state (playbin, NULL, 0, 1 * GST_SECOND);
        count++;
    }

    if (state != GST_STATE_CHANGE_FAILURE &&
        state != GST_STATE_CHANGE_ASYNC) {
        GstFormat format = GST_FORMAT_TIME;
        GstMessage *msg;
        GstBus *bus;
        gint64 duration;
        gboolean eos = FALSE;

        if (gst_element_query_duration (playbin, &format, &duration)) {
            g_hash_table_insert (metadata, METADATA_AUDIO_DURATION,
                                 g_strdup_printf ("%" G_GINT64_MODIFIER "d", duration / GST_SECOND));
        }

        bus = gst_element_get_bus (playbin);
        while (!eos) {
            msg = gst_bus_poll (bus,
                                GST_MESSAGE_TAG
                                | GST_MESSAGE_EOS
                                | GST_MESSAGE_ERROR,
                                1 * GST_SECOND);

            if (msg == NULL) {
                g_object_unref (bus);

                goto clean_up;
            }

            eos = parse_message (msg, file, metadata);
            gst_message_unref (msg);
        }

        g_object_unref (bus);
    }

  clean_up:
    gst_element_set_state (playbin, GST_STATE_NULL);
    g_object_unref (playbin);

    sanitise_metadata (file, metadata);
    return TRUE;
}
