#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <glib.h>

#include <kozo/kozo-db-private.h>
#include <bickley/bkl.h>

#define KOZO_INDEX_KEY_PREFIX "kozo:index:"

char * XDG_USER_DIRS[3];

static char *
fix_uri (const char *uri,
         const char *username)
{
    char *new_uri;

    if (uri == NULL) {
        return NULL;
    }

    if (g_str_has_prefix (uri, "file:///home/") == FALSE) {
        return g_strdup (uri);
    }

    /* Replace the dir in uris with XDG_USER_DIRS  */
    if ((g_str_has_prefix (uri, "file:///home/moblin/Music") == TRUE) 
        && XDG_USER_DIRS[0]) {
        new_uri = g_strdup_printf ("%s%s", XDG_USER_DIRS[0],
                                   uri + strlen ("file:///home/moblin/Music"));
    }
    else if ((g_str_has_prefix (uri, "file:///home/moblin/Pictures") == TRUE) 
             && XDG_USER_DIRS[1]) {
        new_uri = g_strdup_printf ("%s%s", XDG_USER_DIRS[1],
                                   uri + strlen ("file:///home/moblin/Pictures"));
    }
    else if ((g_str_has_prefix (uri, "file:///home/moblin/Videos") == TRUE) 
             && XDG_USER_DIRS[2]) {
        new_uri = g_strdup_printf ("%s%s", XDG_USER_DIRS[2],
                                   uri + strlen ("file:///home/moblin/Videos"));
    }
    /* Replace the word moblin in uris with @username */
    else {
        new_uri = g_strdup_printf ("file:///home/%s%s", username,
                                   uri + strlen ("file:///home/moblin"));
    }

    g_print ("Replaced %s with %s\n", uri, new_uri);
    return new_uri;
}

struct _index_item {
    char *key;
    char *data;
};

static int
traverse_read_func (TDB_CONTEXT *tdb,
                    TDB_DATA     key,
                    TDB_DATA     data,
                    gpointer     userdata)
{
    GPtrArray *index = (GPtrArray *) userdata;
    struct _index_item *item;

    if (strncmp ((char *) key.dptr, KOZO_INDEX_KEY_PREFIX,
                 strlen (KOZO_INDEX_KEY_PREFIX)) != 0) {
        return 0;
    }

    item = g_slice_new (struct _index_item);
    item->key = g_strndup ((char *) key.dptr, key.dsize);
    item->data = g_strndup ((char *) data.dptr, data.dsize);

    g_ptr_array_add (index, item);
    return 0;
}

static char *
fix_index_key (const char *key,
               const char *username)
{
    char *new_key;

    if (g_str_has_prefix (key, "kozo:index:file:///home/") == FALSE) {
        return g_strdup (key);
    }

    new_key = g_strdup_printf ("kozo:index:%s", 
                               fix_uri (key + strlen ("kozo:index:"), username));
    g_print ("Replaced %s with %s\n", key, new_key);
    return new_key;
}

static char *
fix_index_data (const char *data,
                const char *username)
{
    GString *new_data;
    char **str_v, *ret;
    int i;

    str_v = g_strsplit (data, "\x01", 0);

    new_data = g_string_new (fix_uri(str_v[0], username));
    for (i = 1; str_v[i]; i++) {
        g_string_append_c (new_data, 0x01);
        g_string_append (new_data, fix_uri (str_v[i], username));
    }

    g_strfreev (str_v);
    ret = new_data->str;
    g_string_free (new_data, FALSE);

    return ret;
}

static gboolean
fix_index (BklDB      *db,
           const char *username)
{
    KozoDB *kdb = db->db;
    TDB_CONTEXT *tdb;
    GPtrArray *index;
    int i;

    index = g_ptr_array_new ();

    tdb = kozo_db_get_db (kdb);

    /* Extract all the index strings */
    tdb_traverse_read (tdb, traverse_read_func, index);

    /* Go through each of the index entries,
       removing them, fixing them up and putting them back in */
    for (i = 0; i < index->len; i++) {
        struct _index_item *item = index->pdata[i];
        GError *error = NULL;
        char *new_key, *new_data;
        TDB_DATA k, d;
        int err;

        kozo_db_remove (kdb, item->key, &error);
        if (error != NULL) {
            g_warning ("Error removing %s: %s", item->key, error->message);
            return FALSE;
        }

        new_key = fix_index_key (item->key, username);
        new_data = fix_index_data (item->data, username);

        k.dptr = (guchar *) new_key;
        k.dsize = strlen (new_key);

        d.dptr = (guchar *) new_data;
        d.dsize = strlen (new_data);

        tdb_transaction_start (tdb);
        err = tdb_store (tdb, k, d, TDB_INSERT);
        if (G_UNLIKELY (err)) {
            g_warning ("Error putting %s into DB: %d", new_key, err);

            tdb_transaction_cancel (tdb);
            return FALSE;
        }

        tdb_transaction_commit (tdb);

        g_free (new_key);
        g_free (new_data);
    }

    return TRUE;
}

static gboolean
fix_items (BklDB      *db,
           GPtrArray  *items,
           const char *username)
{
    int i;

    for (i = 0; i < items->len; i++) {
        BklItem *item = items->pdata[i];
        GError *error = NULL;
        char *new_uri;
        char *thumb_uri, *new_thumb_file;
        int ret;

        /* Remove the item from the database */
        bkl_db_remove (db, bkl_item_get_uri (item), &error);
        if (error != NULL) {
            g_warning ("Error removing %s from db: %s", bkl_item_get_uri (item),
                       error->message);
            return FALSE;
        }

        new_uri = fix_uri (bkl_item_get_uri (item), username);
        bkl_item_set_uri (item, new_uri);

        thumb_uri = fix_uri (bkl_item_extended_get_thumbnail
                             ((BklItemExtended *) item), username);

        if (g_str_has_prefix (thumb_uri, "file:///home/") == FALSE) {
            g_free (new_uri);
            g_free (thumb_uri);
        } else {
            /* Generate new thumb uri*/
            new_thumb_file = g_build_filename
                ("/home", username, ".bkl-thumbnails",
                 g_compute_checksum_for_string (G_CHECKSUM_MD5, new_uri, -1),
                 NULL);
            g_free (new_uri);
            new_uri = g_filename_to_uri (new_thumb_file, NULL, &error);
            if (new_uri == NULL) {
                g_warning ("Error converting %s to uri: %s",
                           new_thumb_file, error->message);
                return FALSE;
            }

            /* Rename thumb file*/
            ret = rename (thumb_uri + strlen ("file://"), new_thumb_file);
            if (ret != 0)  {
                 g_warning("Error renaming thumbnail %s to %s",
                           thumb_uri + strlen ("file://"), new_thumb_file);
                 return FALSE;
            }

            g_print ("Rename thumb file %s to %s\n",
                     thumb_uri + strlen ("file://"), new_thumb_file);

            g_free (thumb_uri);
            g_free (new_thumb_file);

            bkl_item_extended_set_thumbnail ((BklItemExtended *) item, new_uri);
            g_free (new_uri);
        }

        /* Put the item back into the database */
        bkl_db_add_item (db, bkl_item_get_uri (item), item, &error);
        if (error != NULL) {
            g_warning ("Error adding %s to db: %s", bkl_item_get_uri (item),
                       error->message);
            return FALSE;
        }

        g_object_unref (item);
    }

    return TRUE;
}

static gboolean
get_xdg_user_dirs (const char *username)
{
    char *uri;
    GUserDirectory special_dirs[] = { G_USER_DIRECTORY_MUSIC,
                                      G_USER_DIRECTORY_PICTURES,
                                      G_USER_DIRECTORY_VIDEOS, 
                                      G_USER_N_DIRECTORIES
                                     };
    int i;
    for (i = 0; special_dirs[i] != G_USER_N_DIRECTORIES; i++) {
        const char *filename;
        
        filename = g_get_user_special_dir (special_dirs[i]);
        if (filename == NULL) {            
            XDG_USER_DIRS[i] = NULL;
            g_print ("Failed to get XDG_USER_DIR[%d]\n", i);
            continue;
        }

        uri = g_filename_to_uri (filename, NULL, NULL);
        if (g_str_has_prefix (uri, "file:///root") == TRUE) {
            /* Replace the word root in uris with @username */
            XDG_USER_DIRS[i] = g_strdup_printf ("file:///home/%s%s", username,
                                                uri + strlen ("file:///root"));
        }
        else  {
            XDG_USER_DIRS[i] = g_strdup (uri);
        }

        g_print ("Get XDG_USER_DIR[%d] = %s\n", i , XDG_USER_DIRS[i]);
        g_free (uri);
   }
   return TRUE;
}


int
main (int    argc,
      char **argv)
{
    GError *error = NULL;
    BklDB *src_db;
    GPtrArray *items;

    if (argc != 3) {
        g_print ("A tool for replacing the initial username in a Bickley DB\n");
        g_print ("Usage: %s <username> <src-db>\n", argv[0]);
        return 1;
    }

    g_thread_init (NULL);
    bkl_init ();

    src_db = bkl_db_get_for_path (argv[2], "", &error);
    if (error != NULL) {
        g_print ("Error loading %s: %s", argv[2], error->message);
        return 1;
    }

    items = bkl_db_get_items (src_db, TRUE, &error);
    if (error != NULL) {
        g_print ("Error getting items: %s", error->message);
        return 1;
    }

    get_xdg_user_dirs (argv[1]);

    if (fix_items (src_db, items, argv[1]) == FALSE) {
        return 1;
    }

    if (fix_index (src_db, argv[1]) == FALSE) {
        return 1;
    }

    return 0;
}
