/*
 * Bickley - a meta data management framework.
 * Copyright © 2008, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <string.h>
#include <bickley/bkl-db.h>

#include "bkl-indexer.h"

gboolean
bkl_indexer_index_item (BklSource *source,
                        BklItem   *item,
                        KozoDB    *db,
                        GError   **error)
{
    char *contents, *p;
    int len, count;
    GPtrArray *added, *removed;

    if (item == NULL) {
        return TRUE;
    }

    contents = bkl_item_to_string (item);
    if (contents == NULL) {
        return TRUE;
    }

    added = g_ptr_array_new ();
    removed = g_ptr_array_new ();

    p = contents;
    len = strlen (contents);
    count = 0;
    while (count < len) {
        gboolean new_word = FALSE;
        char *space = p;
        char *word;

        while (*space != '\0' && g_ascii_isgraph (*space)) {
            space++;
            count++;
        }

        *space = 0;

        if (space - p == 0) {
            p = space + 1;
            count++;
            continue;
        }

        word = g_ascii_strup (p, -1);

        if (kozo_db_index_add_word (db, word,
                                    bkl_item_get_uri (item),
                                    &new_word, error) == FALSE) {
            g_free (contents);
            return FALSE;
        }

        if (new_word == TRUE) {
            g_ptr_array_add (added, word);
        }

        p = space + 1;
        count++;
    }

    bkl_source_index_changed (source, added, removed);

    g_ptr_array_free (added, TRUE);
    g_ptr_array_free (removed, TRUE);
    g_free (contents);

    return TRUE;
}

static void
parse_words (char      *contents,
             GPtrArray *words)
{
    char *p;
    int len, count;

    if (contents == NULL) {
        return;
    }

    p = contents;
    len = strlen (contents);
    count = 0;
    while (count < len) {
        char *space = p;

        while (*space != '\0' && g_ascii_isgraph (*space)) {
            space++;
            count++;
        }

        *space = 0;

        if (space - p == 0) {
            p = space + 1;
            count++;
            continue;
        }

        g_ptr_array_add (words, g_ascii_strup (p, -1));

        p = space + 1;
        count++;
    }
}

static void
compare_words (GPtrArray *old_words,
               GPtrArray *new_words,
               GList    **added,
               GList    **removed)
{
    GHashTableIter iter;
    GHashTable *old_hash;
    char *key;
    int i;

    old_hash = g_hash_table_new (g_str_hash, g_str_equal);
    /* Put the old words in a hashtable */
    for (i = 0; i < old_words->len; i++) {
        g_hash_table_insert (old_hash, old_words->pdata[i],
                             old_words->pdata[i]);
    }

    /* Go through the new words, checking for them in the hashtable */
    for (i = 0; i < new_words->len; i++) {
        if (g_hash_table_lookup (old_hash, new_words->pdata[i]) == NULL) {
            /* The new word isn't in the hash table, therefore its been
               added */
            *added = g_list_prepend (*added, new_words->pdata[i]);
        } else {
            /* Remove it from the hash table as its in both arrays */
            g_hash_table_remove (old_hash, new_words->pdata[i]);
        }
    }

    /* The only words left in the hash table should be the words that need to
       be removed */
    g_hash_table_iter_init (&iter, old_hash);
    while (g_hash_table_iter_next (&iter, (gpointer *) &key, NULL)) {
        *removed = g_list_prepend (*removed, key);
    }

    g_hash_table_destroy (old_hash);
}

gboolean
bkl_indexer_update_item (BklSource *source,
                         BklItem   *new_item,
                         BklItem   *old_item,
                         KozoDB    *db)
{
    char *old_contents, *new_contents;
    GPtrArray *old_words, *new_words;
    GPtrArray *r_added, *r_removed;
    GList *added = NULL, *removed = NULL, *l;
    int i;

    if (old_item == NULL || new_item == NULL) {
        return TRUE;
    }

    old_contents = bkl_item_to_string (old_item);
    old_words = g_ptr_array_new ();
    parse_words (old_contents, old_words);
    g_free (old_contents);

    new_contents = bkl_item_to_string (new_item);
    new_words = g_ptr_array_new ();
    parse_words (new_contents, new_words);
    g_free (new_contents);

    compare_words (old_words, new_words, &added, &removed);

    r_added = g_ptr_array_new ();
    for (l = added; l; l = l->next) {
        gboolean new_word = FALSE;
        char *word = l->data;
        GError *error = NULL;

        kozo_db_index_add_word (db, word,
                                bkl_item_get_uri (new_item),
                                &new_word, &error);
        if (error != NULL) {
            g_warning ("Error adding %s to %s: %s", bkl_item_get_uri (new_item),
                       word, error->message);
            g_error_free (error);
        }

        if (new_word == TRUE) {
            g_ptr_array_add (r_added, word);
        }
    }
    g_list_free (added);

    r_removed = g_ptr_array_new ();
    for (l = removed; l; l = l->next) {
        gboolean deleted = FALSE;
        char *word = l->data;
        GError *error = NULL;

        kozo_db_index_remove_word (db, word,
                                   bkl_item_get_uri (new_item),
                                   &deleted, &error);
        if (error != NULL) {
            g_warning ("Error removing %s from %s: %s", bkl_item_get_uri (old_item),
                       word, error->message);
            g_error_free (error);
        }

        if (deleted == TRUE) {
            g_ptr_array_add (r_removed, word);
        }
    }
    g_list_free (removed);

    bkl_source_index_changed (source, r_added, r_removed);

    g_ptr_array_free (r_added, TRUE);
    g_ptr_array_free (r_removed, TRUE);

    for (i = 0; i < old_words->len; i++) {
        g_free (old_words->pdata[i]);
    }
    g_ptr_array_free (old_words, TRUE);

    for (i = 0; i < new_words->len; i++) {
        g_free (new_words->pdata[i]);
    }
    g_ptr_array_free (new_words, TRUE);

    return TRUE;
}

gboolean
bkl_indexer_remove_item (BklSource *source,
                         BklItem   *item,
                         KozoDB    *db)
{
    char *contents, *p;
    GError *error = NULL;
    int len, count;
    GPtrArray *added, *removed;

    if (item == NULL) {
        return TRUE;
    }

    contents = bkl_item_to_string (item);
    if (contents == NULL) {
        return TRUE;
    }

    added = g_ptr_array_new ();
    removed = g_ptr_array_new ();

    /* Find the words that are in the index for this item */
    p = contents;
    len = strlen (contents);
    count = 0;
    while (count < len) {
        gboolean deleted = FALSE;
        char *space = p;
        char *word;

        while (*space != '\0' && g_ascii_isgraph (*space)) {
            count++;
            space++;
        }

        *space = 0;

        if (space - p == 0) {
            p = space + 1;
            count++;
            continue;
        }

        word = g_ascii_strup (p, -1);

        if (kozo_db_index_remove_word (db, word,
                                       bkl_item_get_uri (item),
                                       &deleted, &error) == FALSE) {
            g_warning ("Error removing %s from %s: %s",
                       bkl_item_get_uri (item), word, error->message);
            g_error_free (error);
            g_free (contents);
            return FALSE;
        }

        if (deleted == TRUE) {
            g_ptr_array_add (removed, word);
        }

        p = space + 1;
        count++;
    }

    bkl_source_index_changed (source, added, removed);

    g_ptr_array_free (added, TRUE);
    g_ptr_array_free (removed, TRUE);
    g_free (contents);

    return TRUE;
}
